/* -*- linux-c -*- --------------------------------------------------------- *  
 * This module is an interface to the ZORAN zr367x0 dvd decoder chip.
 *
 * Copyright (C) 1999 David Barth <dbarth@besancon.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#ifndef __KERNEL__
#define __KERNEL__
#endif
#ifndef MODULE
#define MODULE
#endif

#include <linux/module.h>
#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/poll.h>
#include <asm/io.h>
#include <linux/sched.h>
#include <linux/videodev2.h>
#include <linux/proc_fs.h>
#include "mgavideo.h"
#include "i34.h"

#define dprintk     if (debug) printk

static int debug = 1; 	 	/* insmod parameter */
static int reset = 1;		/* insmod parameter */

#if LINUX_VERSION_CODE >= 0x020100
MODULE_PARM(debug,"i");
MODULE_PARM(reset,"i");
#endif

static struct mga_dev *host;


#define I34_STATE_NO_UCODE	0	/* indicates that no microcode has been downloaded so far */

static int state = 0;			/* internal state */

static u8 ucode_buffer[16384];		/* buffer holding ucode to download into the chipset */
static u16 ucode_length = 0;		/* length of the ucode inside the buffer */
static u16 ucode_wmark = 0;		/* number of bytes written to the chipset
					   needed so that we can zero-pad ucode when downloading
					   a new ucode */

/****************************************************************************
 * DVD/MPEG Add-on board						    *
 ****************************************************************************/

/* main registers */
#define ZR_PARAM_ADDR	0x0
#define ZR_PARAM_DATA	0x1
#define ZR_ISR		0x2
#define ZR_STATUS0	0x3
#define ZR_HOSTCMD	0x3
#define ZR_STATUS1	0x4
#define ZR_STATUS2	0x5
#define ZR_DEVICE_ID	0x7
#define ZR_ADP_DATA	0x8
#define ZR_ADP_STATUS	0x8
#define ZR_SYS_CLOCK	0x9
#define ZR_NAV_DATA	0xA
#define ZR_RESERVED1	0xB
#define ZR_SUBPICBUF	0xC
#define ZR_VIDEOBUF	0xD
#define ZR_AUDIOBUF	0xE
#define ZR_RESERVED2	0xF

/* indirect registers */
#define ZR_SYSCONFIG		0x00
#define ZR_VIDCONFIG		0x01
#define ZR_SDCONFIG		0x02
#define ZR_DVPGEN1		0x03
#define ZR_DVPGEN2		0x04
#define ZR_DVPGEN3		0x05
#define ZR_COLORV		0x08
#define ZR_COLORY		0x09
#define ZR_COLORU		0x0A
#define ZR_VIDOUT		0x10
#define ZR_HTOTAL		0x11
#define ZR_VTOTAL		0x12
#define ZR_HSYNCSIZE		0x13
#define ZR_VSYNCSIZE		0x14
#define ZR_CBHSTART		0x15
#define ZR_CBVSTART		0x16
#define ZR_CBHEND		0x17
#define ZR_CBVEND		0x18
#define ZR_ACTIVESTARTX		0x19
#define ZR_ACTIVESTARTY		0x1A
#define ZR_ACTIVEENDX		0x1B
#define ZR_ACTIVEENDY		0x1C
#define ZR_ACTIVESIZEX		0x1D
#define ZR_ACTIVESIZEY		0x1E

#define ZR_IMAGESTARTX		0x19
#define ZR_IMAGESTARTY		0x1A
#define ZR_IMAGEENDX		0x1B
#define ZR_IMAGEENDY		0x1C
#define ZR_IMAGESIZEX		0x1D
#define ZR_IMAGESIZEY		0x1E
#define ZR_CAPTIONOFFSET	0x1F
#define ZR_STARTDISPLAY 	0x20
#define ZR_OSDSWITCH		0x2D
#define ZR_PLAYBACKMODE		0x41
#define ZR_AUDSID		0x42
#define ZR_VIDSID		0x43
#define ZR_SPSID		0x44
#define ZR_AUDPORTDELAY		0x45
#define ZR_VIDPORTDELAY		0x46
#define ZR_VIDTOLERANCE		0x47
#define ZR_VIDSYNCMODE		0x48
#define ZR_AUTOSCALING		0x50
#define ZR_AUTOPANSCAN		0x68
#define ZR_BACKGROUNDSWITCH	0x6B
#define ZR_INITFILE_1		0xF0
#define ZR_INITFILE_2		0xF1
#define ZR_INITFILE_3		0xF2
#define ZR_INITFILE_4		0xF3
#define ZR_MICROCODE		0xFD

static u16 shadow_register[0xff];	// Warning: not all registers are 2 bytes long
// but it's enough for those we're interested in

u16 I34_Read(struct mga_dev *mga, u8 reg) {
	codec_reset(mga);
	codec_read(mga, reg);
	codec_exec(mga);
	return __swab16(codec_getw(mga));
}

void I34_Write(struct mga_dev *mga, u8 reg, u16 data) {
	codec_reset(mga);
	codec_write(mga, reg, __swab16(data));
	codec_exec(mga);
}

/* see table 32 p. 79 for the protocol to access the indirect address space */
u16 I34_ParameterRead(struct mga_dev *mga, u8 addr) {
	codec_reset(mga);
	/* addr. in the l.s. bits, m.s. bits must be 0 */
	codec_write(mga, ZR_PARAM_ADDR, __swab16(addr));
	codec_read(mga, ZR_PARAM_DATA);
	codec_exec(mga);
	shadow_register[addr] = __swab16(codec_getw(mga));
	return shadow_register[addr];
}

void I34_ParameterWrite(struct mga_dev *mga, u8 addr, u16 data) {
	shadow_register[addr] = data;
	codec_reset(mga);
	codec_write(mga, ZR_PARAM_ADDR, __swab16(addr));
	codec_write(mga, ZR_PARAM_DATA, __swab16(data));
	codec_exec(mga);
}

void I34_ParameterWriteSeq(struct mga_dev *mga, u8 addr, u8* buf, u32 len) {
	u32 i;

	codec_reset(mga);
	codec_write(mga, ZR_PARAM_ADDR, __swab16(addr));
	/* multiple read/writes witout modifying the Parameter Address Register */
	if (buf == 0) {
		for (i=0; i<len; i+=2)	// we count the buffer size in bytes, but write a _word_ at a time
			codec_write(mga, ZR_PARAM_DATA, 0);
  
	} else {
		for (i=0; i<len; i+=2)
			codec_write(mga, ZR_PARAM_DATA, __swab16(*(((u16 *)buf)++)));

	}
	codec_exec(mga);
}

int I34_State(struct mga_dev *mga, int *s) {
	*s = (I34_Read(mga, ZR_STATUS0) >> 10);
	return 0;
}

int I34_Command(struct mga_dev *mga, u16 cmd) {
	/* p. 100 ZR HB: check that the host is ready to receive a new command */
	u16 status1 = I34_Read(mga, ZR_STATUS1);
	u16 hcready = (status1 >> 8) & 0x80;
	if (hcready) {
		I34_Write(mga, ZR_HOSTCMD, cmd);
		return 0;
	} else {
		I34_Write(mga, ZR_HOSTCMD, cmd);
		dprintk("I34: host is not ready to receive commands (%#4.4x)\n", status1);
		return -EBUSY;
	}
}


/* new form: returns an error status and passes the actual result via a pointer */
int I34_ReadInterruptStatus(struct mga_dev *mga, u16 *status) {
	*status = I34_Read(mga, ZR_ISR);
	return 1;
}

void I34_SetPar_bits(struct mga_dev *mga, u16 addr, u16 or, u16 and) {
	// the windows driver uses a shadow register that recalls
	// the last read value of the register...
	u16 value = (shadow_register[addr] & ~and) | or;
	I34_ParameterWrite(mga, addr, value);
}
	
#define I34_WaitMicroSecond(a)	udelay(a)

void I34_ResetPin(struct mga_dev *mga) {

	if (reset == 0) return;

	codec_miscctl_func1(mga, reset);
	I34_WaitMicroSecond(160);
	codec_I34_reset(mga);
	codec_miscctl(mga, 0x0f, 0xf9);
}

int I34_Reset(struct mga_dev *mga) {
	u16 i, isr;

	dprintk("I34: Reset\n");

	I34_ResetPin(mga);

	for (i=0; i<5000; i++) {
		if (!I34_ReadInterruptStatus(mga, &isr)) {
			return 0;
		}
#ifdef BYTEORDERHACK
		if ((isr & 0x1800) == 0x1800) {

			mga->little_endian_codec = 0;
			dprintk("I34_Reset ISR = 0x%4.4x Codec big-endian after reset\n", isr);	
			I34_ParameterWrite(mga, 0x0F3, 0xA0);
			I34_ParameterWrite(mga, 0x0F3, 0x88);
			return 1;
		} else if ((isr & 0x18) == 0x18) {
			mga->little_endian_codec = 1;
			dprintk("I34_Reset ISR = 0x%4.4x Codec little-endian after reset\n", isr);	
			I34_ParameterWrite(mga, 0x0F3, 0xA0);
			I34_ParameterWrite(mga, 0x0F3, 0x88);
			return 1;
		}
#else
		if ((isr & 0x1800) == 0x1800) {
			I34_ParameterWrite(mga, 0x0F3, 0xA0);
			I34_ParameterWrite(mga, 0x0F3, 0x88);
			return 1;
		}
#endif
		I34_WaitMicroSecond(100);
	}
	dprintk("PLL's NOT locked ---> 0x%4.4x\n", isr);
	return 0;
}

int I34_Detect(struct mga_dev *mga) {
	u32 id;

#ifdef BYTEORDERHACK
	mga->little_endian_codec = 0; /* assume the codec is bigendian */
#endif

	I34_Reset(mga);
	id = I34_Read(mga, ZR_DEVICE_ID);
	/* we should obtain: 0x34 m.s. bits, rev in l.s. bits p. 112 ZR HB */
	/* warning: this test is arch. dependent ! */
#ifndef __LITTLE_ENDIAN
#error FIX ME
#else
	if (((id>>8) & 0xFF) == 0x34) {
		dprintk("I34: found chipset rev 0x%2.2x)\n", id & 0xFF);
		return 1;
	} else {
		dprintk("I34: too bad, codec said 0x%4.4x\n", id);
		return 0;
	}
#endif
}

int I34_ColorKey(struct mga_dev* mga, struct ColorKey *ck)
{
	I34_ParameterWrite(mga, ZR_COLORV, ck->v);
	I34_ParameterWrite(mga, ZR_COLORU, ck->y);
	I34_ParameterWrite(mga, ZR_COLORY, ck->u);
	return 0;
}

static struct VideoStandard pal1 = { // was unk_28E0
	1,	// VidFPS => PAL
	0,	// ??
	0,	// FieldSel
	0,	// ??
	864,	// HTotal
	625,	// VTotal
	64,	// HSyncSize
	6,	// VSyncSize
	142,	// ActiveStartX
	14,	// ActiveStartY
	860,	// ActiveEndX
	310,	// ActiveEndY
	720,  // ActiveSizeX
	288,	// ActiveSizeY
	0x4006 // VidConfig
};

static struct VideoStandard ntsc1 = { // was unk_2900
	0,	// VidFPS => NTSC
	0,	// ??
	0,	// FieldSel
	0,	// ??
	858,	// HTotal
	525,	// VTotal
	64,	// HSyncSize
	6,	// VSyncSize
	136,	// ActiveStartX
	14,	// ActiveStartY
	856,	// ActiveEndX
	260,	// ActiveEndY
	720,  // ActiveSizeX
	240,	// ActiveSizeY
	0x4006 // VidConfig <=> Video8, Sync8, PixLim
};

static struct VideoStandard pal2 = { // was unk_3930
	1,	// VidFPS => PAL
	0,	// ??
	0,	// FieldSel
	0,	// ??
	864,	// HTotal
	625,	// VTotal
	64,	// HSyncSize
	6,	// VSyncSize
	142,	// ActiveStartX
	16,	// ActiveStartY
	862,	// ActiveEndX
	310,	// ActiveEndY
	720,  // ActiveSizeX
	288,	// ActiveSizeY
	0x4006 // VidConfig
};

static struct VideoStandard ntsc2 = { // was unk_2900
	0,	// VidFPS => NTSC
	0,	// ??
	0,	// FieldSel
	0,	// ??
	858,	// HTotal
	525,	// VTotal
	64,	// HSyncSize
	6,	// VSyncSize
	136,	// ActiveStartX
	16,	// ActiveStartY
	856,	// ActiveEndX
	352,	// ActiveEndY
	720,  // ActiveSizeX
	240,	// ActiveSizeY
	0x4006 // VidConfig <=> Video8, Sync8, PixLim
};


/* this is pure guess (but works somehow...) */
static struct CompositeBlankSync cbs1 = {
	6, 	// HStart
	2, 	// VStart;
	857,	// HEnd;
	261	// VEnd;
};

static struct ColorKey ckBlank = {
	0,	
	0x80,
	0x80
};


int I34_LoadVideoStandard(struct mga_dev *mga, struct VideoStandard *vs) {
	I34_ParameterWrite(mga, ZR_VIDOUT, vs->FieldSel | vs->VidFPS);
	I34_ParameterWrite(mga, ZR_HTOTAL, vs->HTotal);
	I34_ParameterWrite(mga, ZR_VTOTAL, vs->VTotal);
	I34_ParameterWrite(mga, ZR_HSYNCSIZE, vs->HSyncSize);
	I34_ParameterWrite(mga, ZR_VSYNCSIZE, vs->VSyncSize);
	I34_ParameterWrite(mga, ZR_ACTIVESTARTX, vs->ActiveStartX);
	I34_ParameterWrite(mga, ZR_ACTIVESTARTY, vs->ActiveStartY);
	I34_ParameterWrite(mga, ZR_ACTIVEENDX, vs->ActiveEndX);
	I34_ParameterWrite(mga, ZR_ACTIVEENDY, vs->ActiveEndY);
	I34_ParameterWrite(mga, ZR_ACTIVESIZEX, vs->ActiveSizeX);
	I34_ParameterWrite(mga, ZR_ACTIVESIZEY, vs->ActiveSizeY);
	I34_ParameterWrite(mga, ZR_VIDCONFIG, vs->VidConfig);

	return 0;
}

int I34_OutputVideoStandard(struct mga_dev *mga, int standard) {
	struct VideoStandard *vs;

	switch(standard) {
	case I34_NTSC:
		vs = &ntsc1;
		break;
	case I34_NTSC_ALT:
		vs = &ntsc2;
		break;
	case I34_PAL:
		vs = &pal1;
		break;
	case I34_PAL_ALT:
	default:
		vs = &pal2;
		break;
    
	}

	/*  if (standard == 0 || standard == -1) {
	    vs = &ntsc2;
	    } else if (standard == 1) {
	    vs = &pal1;
	    } else 
	    return 0;
	*/

	return I34_LoadVideoStandard(mga, vs);
}

int I34_CompositeBlankSync(struct mga_dev *mga, struct CompositeBlankSync *cb) {
	I34_ParameterWrite(mga, ZR_CBHSTART, cb->CBHStart);
	I34_ParameterWrite(mga, ZR_CBVSTART, cb->CBVStart);
	I34_ParameterWrite(mga, ZR_CBHEND, cb->CBHEnd);
	I34_ParameterWrite(mga, ZR_CBVEND, cb->CBVEnd);

	return 0;
}

int I34_StartDisplay(struct mga_dev *mga) {
	I34_ParameterWrite(mga, ZR_PLAYBACKMODE, 0);
	I34_SetPar_bits(mga, ZR_BACKGROUNDSWITCH, 1, 1);
	I34_ParameterWrite(mga, ZR_AUTOSCALING, 1<<15);
	I34_ParameterWrite(mga, ZR_AUTOPANSCAN, 1);
	I34_ParameterWrite(mga, ZR_OSDSWITCH, 0);	       
	I34_ParameterWrite(mga, ZR_AUDPORTDELAY, 0);
	I34_ParameterWrite(mga, ZR_VIDPORTDELAY, 0);
	I34_ParameterWrite(mga, ZR_VIDTOLERANCE, 0);
	/* validate the new video parameters */
	I34_ParameterWrite(mga, ZR_STARTDISPLAY, 1);
	return 0;
}

/* this function progressively fills the ucode buffer prior to downloading */
/* length = size of the new chunk to add in the buffer
   or 0 to reset to the start of the ucode buffer
*/
int I34_MicrocodeRead(struct mga_dev *mga, struct i34_ucode_chunk *chunk) {

	if ((ucode_length + chunk->length) >= 16384) return -EINVAL;

	memcpy(chunk->buffer, ucode_buffer + ucode_length, chunk->length);
	ucode_length += chunk->length;
	return 0;
}


/* example calls :

     in ADPBoot
     	I34_MicrocodeRead(1, 2, offset unk_210, dword ptr [ebp+8])
     	I34_MicrocodeRead(edi, 0x3c, offset unk_4c8, dword ptr [ebp+8])

     in DVPMicrocodeDownload
	result = I34_MicrocodeRead(1, 2, offset unk_2888, dword ptr [ebp+8])
	result = I34_MicrocodeRead(0, 0x3c, result, dword ptr [ebp+8])
	I34_ParameterWriteSeq(0xFD, 0, buf);

     in LoadSlotMachine
	I34_MicrocodeRead(1, 2, offset unk_4C8, 0x0103)
	I34_MicrocodeRead(0, 0x3c, esi, edi)

     in LoadDRAMMapping
	I34_MicrocodeRead(1, 0x3c, offset unk_4C8, 0x0104)

   */  

int I34_DVPMicrocodeDownload(struct mga_dev *mga) {

	if (ucode_length == 0) return 0;
 
	/* In the general case, we should store the size of the last microcode
	   and zero-pad it to 16384, and then upload the new code */
	if (ucode_wmark != 0)
		I34_ParameterWriteSeq(mga, 0xfd, 0, 16384 - ucode_wmark);

	dprintk("I34: trying to upload %d bytes into the chipset\n", ucode_length);
	I34_ParameterWriteSeq(mga, ZR_MICROCODE, ucode_buffer, ucode_length);

	return 1;
	/* I need to down/upload : 340EDVD.DVP 340EAC3.ADP & \ZI34SLT1.PRG */
}

int I34_Init(struct mga_dev *mga) {

	/* According to Annex C p. 248 of the ZR36710 Handbook */

	dprintk("I34_init\n");
	/* Stage 1 : Reset */
	if (!I34_Detect(mga)) return -ENODEV;

	/* Stage 2 : I/O Port Configuration */
	/* SDRAM, Host Bus Port */
	I34_ParameterWrite(mga, ZR_SYSCONFIG, 0x1408);	// CodeSource = HostBus; Decryption = enable
	I34_SetPar_bits(mga, 0xA0, 0x80, 0x80);		// ??
	I34_SetPar_bits(mga, 0xA0, 0x00, 0x80);
	I34_SetPar_bits(mga, 0xA0, 0x0F, 0x0F);
	I34_SetPar_bits(mga, 0xA4, 0x00, 0x40);
	I34_SetPar_bits(mga, 0xA4, 0x01, 0x07);
	I34_ParameterWrite(mga, ZR_SDCONFIG, 0);

	return 0;

	/* Audio Port */
	/* tricky, so I leave it at defaults value for now... */

	/* Video Port */
	/* mga video-in is an 8-bits bus (Appendix A1.5) */
	I34_OutputVideoStandard(mga, I34_NTSC);

	I34_CompositeBlankSync(mga, &cbs1);
	I34_ColorKey(mga, &ckBlank);
	I34_ParameterWrite(mga, ZR_STARTDISPLAY, 1);
	I34_SetPar_bits(mga, ZR_BACKGROUNDSWITCH, 1, 1);

	/* Stage 3 :  OSD and Preparation for Decoding*/
	/* OSD */
	I34_ParameterWrite(mga, ZR_OSDSWITCH, 0);	       

	/* Microcode uploading - DVP and ADP*/
	/* we first need to zero-pad the 16384 bytes of the microcode prior to loading a new one */
	// I34_ParameterWriteClean(mga, ZR_MICROCODE, 0x0000, 16384);
	//	dprintk("I34: uploading %d bytes of microcode\n", sizeof(microcode));
	//	I34_ParameterWrite_buffer(mga, ZR_MICROCODE, (u16*)microcode, sizeof(microcode));

	I34_StartDisplay(mga);

	/* TEST */
	I34_Command(mga, I34_CMD_START);	/* START */
	I34_Command(mga, I34_CMD_END | 1);	/* END and display bgnd color */



	/* Stage  :  */
	/* Stage  :  */

	/* Stage  :  */
	/* 0 since no bistream is decoded yet (p. 91 ZR-HB) */
	I34_ParameterWrite(mga, ZR_AUDPORTDELAY, 0);
	I34_ParameterWrite(mga, ZR_VIDPORTDELAY, 0);
	I34_ParameterWrite(mga, ZR_VIDTOLERANCE, 0);

	/* Stage  :  */
	/* Stage  :  */
	/* Stage  :  */
	/* Stage  :  */
	/* Stage  :  */
	/* Stage  :  */
	/* Stage  :  */
	/* Stage  :  */
	/* Stage  :  */
	/* Stage  :  */

	return 0;
}

int codec_read_info(char *buf, char **start, off_t offset, int len, int unused) {
	u16 data;
        struct mga_dev* mga = host;

	if (host == NULL) return 0;
	
	len = 0;
	len += sprintf(buf + len, "Zoran codec interface\n");

	data = I34_ParameterRead(mga, ZR_VIDCONFIG);
	len += sprintf(buf + len, "VidConfig: 0x%4.4x\n\n", data);

	len += sprintf(buf + len, "PARAM_ADDR %#4.4x\n", I34_Read(mga, ZR_PARAM_ADDR));
	len += sprintf(buf + len, "PARAM_DATA %#4.4x\n", I34_Read(mga, ZR_PARAM_DATA));
	len += sprintf(buf + len, "ISR        %#4.4x\n", I34_Read(mga, ZR_ISR));
	data = I34_Read(mga, ZR_STATUS0);
	len += sprintf(buf + len, "STATUS0    %#4.4x (c-state 0x%x)\n",
		       data, data >> 10);
	len += sprintf(buf + len, "STATUS1    %#4.4x\n", I34_Read(mga, ZR_STATUS1));
	len += sprintf(buf + len, "STATUS2    %#4.4x\n", I34_Read(mga, ZR_STATUS2));
	len += sprintf(buf + len, "DEV_ID     %#4.4x\n", I34_Read(mga, ZR_DEVICE_ID));
	len += sprintf(buf + len, "ADP_STATUS %#4.4x\n", I34_Read(mga, ZR_ADP_STATUS));
	len += sprintf(buf + len, "SYS_CLOCK  %#4.4x\n", I34_Read(mga, ZR_SYS_CLOCK));
	len += sprintf(buf + len, "NAV_DATA   %#4.4x\n", I34_Read(mga, ZR_NAV_DATA));
	len += sprintf(buf + len, "RESERVED1  %#4.4x\n", I34_Read(mga, ZR_RESERVED1));
	len += sprintf(buf + len, "SUBPICBUF  %#4.4x\n", I34_Read(mga, ZR_SUBPICBUF));
	len += sprintf(buf + len, "VIDEOBUF   %#4.4x\n", I34_Read(mga, ZR_VIDEOBUF));
	len += sprintf(buf + len, "AUDIOBUF   %#4.4x\n", I34_Read(mga, ZR_AUDIOBUF));
	len += sprintf(buf + len, "RESERVED2  %#4.4x\n", I34_Read(mga, ZR_RESERVED2));

	
	return len;
}

struct proc_dir_entry codec_proc_entry = {
        0,                 /* low_ino: the inode -- dynamic */
        8, "mgacodec",     /* len of name and name */
        S_IFREG | S_IRUGO, /* mode */
        1, 0, 0,           /* nlinks, owner, group */
        0, NULL,           /* size - unused; operations -- use default */
        &codec_read_info,  /* function used to read data */
        /* nothing more */
};
	

int I34_ioctl(struct mga_dev *mga, unsigned int cmd, void *arg)
{
	// dprintk("I34: ioctl %d, arg=%p\n", _IOC_NR(cmd), arg);
	switch(cmd) {
	case I34_IOCSUCODE:
	{
		return I34_MicrocodeRead(mga, (struct i34_ucode_chunk *)arg);
	}
	case I34_IOCSVIDEOSTD:
	{
		return I34_OutputVideoStandard(mga, (int)arg);
	}
	case I34_IOCSVDSTDCST:
	{
		return I34_LoadVideoStandard(mga, (struct VideoStandard *)arg);
	}
	case I34_IOCSCOLORKEY:
	{
		return I34_ColorKey(mga, (struct ColorKey *)arg);
	}
	case I34_IOCSCBSYNC:
	{
		return I34_CompositeBlankSync(mga, (struct CompositeBlankSync *)arg);
	}
	case I34_IOCGSTATE:
	{
	        return I34_State(mga, (int *)arg);
	}
	case I34_IOCTSTRTDSPL:
	{
		return I34_StartDisplay(mga);
	}
	case I34_IOCTCMD:
	{
		return I34_Command(mga, (int)arg);
	}
	case I34_IOCTINIT:
	{
		return I34_Init(mga);
	}
	default:    
		return -ENOIOCTLCMD;
	}
}


/****************************************************************************
* linux kernel module api
****************************************************************************/
#ifdef MODULE
int init_module(void)
#else
int i34_init(void)
#endif
{
	/* Be sure that the mgavideo module is loaded */
	/* I should take a look at get_module_symbol ... */
	if ((host = mgavideo_get_codec()) == NULL)
		return -ENODEV;

  //  if (!I34_Detect(host))
  //    return -ENODEV;

  //  I34_Init(host);
	proc_register(&proc_root, &codec_proc_entry);

	return 0;
}

#ifdef MODULE
void cleanup_module(void)
{
	proc_unregister(&proc_root, codec_proc_entry.low_ino);
}
#endif


/******************************************************************************/

/*int I34_ResetOrig(struct mga_dev *mga, u16 param1, u16 param2) {
  u16 i, isr;

  dprintk("I34: Reset\n");

  if (!(I34_ResetPin(mga, param1))) {
  dprintk("I34: 23 FALSE reg\n");
  return 0;
  }  

  for (i = 1000; i>0; i--)
  I34_WaitMicroSecond(500);

  if (param1 == 1) {
  for (i=0; i<0x1388; i++) {
  if (!I34_ReadInterruptStatus(mga, &isr)) {
  dprintk("I34: 24 FALSE reg\n");
  return 0;
  }
  if ((isr & 0x1800) != 0x1800) {
  I34_ParameterWrite(mga, 0xA0, 0x0F3);
  I34_ParameterWrite(mga, 0x88, 0x0F3);
  return 1;
  }
  I34_WaitMicroSecond(100);
  }
  dprintk("PLL's NOT locked ---> 0x%4.4x", isr);
  return 0;

  } else {
  if (param2 == 1)
  // I34_Init(mga);
  return 1;
  }
  return 0;
  }
*/

/* ******************************************************************
   Explanation: 
   I frankly don't know how to reliably reset this ^*&#@ !
   This 2-phase sequence _seems_ to succeed in unlocking the card
   and placing it in a state suitable for further operations.
   But who knows if it will work for you.
   _Please_ report any success or failure with the corresponding
   rev number or the answer of the codec.
   ****************************************************************** */

/* taken verbatim from mgallx64 */
/* writeb(codecen, mga->ctrl + CODECCTL);
   writeb(readb(mga->ctrl + CODECCTL + 3) & 0xfb, mga->ctrl + CODECCTL + 3);
   writeb(~codecen, mga->ctrl + CODECCTL);
   mdelay(160); */

/* this one is from Mike's zr36060 driver */
/* writeb(codecen, mga->ctrl + CODECCTL);
   writeb(readb(mga->ctrl + CODECCTL + 3) | 0x40, mga->ctrl + CODECCTL + 3);
   writeb(readb(mga->ctrl + CODECCTL + 3) & ~0x40, mga->ctrl + CODECCTL + 3);
   writeb(~codecen, mga->ctrl + CODECCTL);
   mdelay(160); */

/* writeb(codecen, mga->ctrl + CODECCTL);
   u8 misc;
   u8 value = 0x98;
   misc = ~((value & 0x9f) | (~value & 0x60)) & 0xff;
   writeb(misc, mga->ctrl + CODECCTL + 3);
   writeb(~codecen, mga->ctrl + CODECCTL);
   mdelay(160); */

/* od -vtx1 340edvd.dvp | cut -d" " -f2- | sed -e 's/ /\,0x/g' | sed -e's/^/\,0x/g' | sed -e 's/$/,/g | sed -e 's/^,//g' > 340edvd.inc '
   do not forget to suppress the last line of the resulting file
*/
