// ListVwEx.cpp : implementation of the CListViewEx class
//
// This is a part of the Microsoft Foundation Classes C++ library.
// Copyright (C) 1992-1997 Microsoft Corporation
// All rights reserved.
//
// This source code is only intended as a supplement to the
// Microsoft Foundation Classes Reference and related
// electronic documentation provided with the library.
// See these sources for detailed information regarding the
// Microsoft Foundation Classes product.

#include "stdafx.h"
#include "ListVwEx.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CListViewEx

IMPLEMENT_DYNCREATE(CListViewEx, CListView)

BEGIN_MESSAGE_MAP(CListViewEx, CListView)
	//{{AFX_MSG_MAP(CListViewEx)
	ON_WM_SIZE()
	ON_WM_PAINT()
	ON_WM_SETFOCUS()
	ON_WM_KILLFOCUS()
	ON_COMMAND(ID_EDIT_COPY, OnEditCopy)
	ON_UPDATE_COMMAND_UI(ID_EDIT_COPY, OnUpdateEditCopy)
	//}}AFX_MSG_MAP
	ON_MESSAGE(LVM_SETIMAGELIST, OnSetImageList)
	ON_MESSAGE(LVM_SETTEXTCOLOR, OnSetTextColor)
	ON_MESSAGE(LVM_SETTEXTBKCOLOR, OnSetTextBkColor)
	ON_MESSAGE(LVM_SETBKCOLOR, OnSetBkColor)
	ON_COMMAND(ID_FILE_PRINT, CListView::OnFilePrint)
	ON_COMMAND(ID_FILE_PRINT_PREVIEW, CListView::OnFilePrintPreview)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CListViewEx construction/destruction

CListViewEx::CListViewEx()
{
	m_bFullRowSel = FALSE;
	m_bClientWidthSel = TRUE;
	m_bFocus=FALSE;

	m_cxClient = 0;
	m_cxStateImageOffset = 0;

	m_clrText = ::GetSysColor(COLOR_WINDOWTEXT);
	m_clrTextBk = ::GetSysColor(COLOR_WINDOW);
	m_clrBkgnd = ::GetSysColor(COLOR_WINDOW);

	m_hPrinterFont=NULL;
	dlgSetupPage.top=0;
	dlgSetupPage.bottom=0;
	dlgSetupPage.left=0;
	dlgSetupPage.right=0;
	m_fontPrinter=NULL;
	m_Separator=" | ";
}

CListViewEx::~CListViewEx()
{
}

BOOL CListViewEx::PreCreateWindow(CREATESTRUCT& cs)
{
	// default is report view and full row selection
	cs.style &= ~LVS_TYPEMASK;
	cs.style |= LVS_REPORT | LVS_OWNERDRAWFIXED;
	m_bFullRowSel = TRUE;

	return(CListView::PreCreateWindow(cs));
}

BOOL CListViewEx::SetFullRowSel(BOOL bFullRowSel)
{
	// no painting during change
	LockWindowUpdate();

	m_bFullRowSel = bFullRowSel;

	BOOL bRet;

	if (m_bFullRowSel)
		bRet = ModifyStyle(0L, LVS_OWNERDRAWFIXED);
	else
		bRet = ModifyStyle(LVS_OWNERDRAWFIXED, 0L);

	// repaint window if we are not changing view type
	if (bRet && (GetStyle() & LVS_TYPEMASK) == LVS_REPORT)
		Invalidate();

	// repaint changes
	UnlockWindowUpdate();

	return(bRet);
}

BOOL CListViewEx::GetFullRowSel()
{
	return(m_bFullRowSel);
}

/////////////////////////////////////////////////////////////////////////////
// CListViewEx drawing

// offsets for first and other columns
#define OFFSET_FIRST	6
#define OFFSET_BORDER	2
#define OFFSET_OTHER	6

void CListViewEx::DrawItem(LPDRAWITEMSTRUCT lpDrawItemStruct)
{
	CListCtrl& ListCtrl=GetListCtrl();
	CDC* pDC = CDC::FromHandle(lpDrawItemStruct->hDC);
	CRect rcItem(lpDrawItemStruct->rcItem);
	UINT uiFlags = ILD_TRANSPARENT;
	CImageList* pImageList;
	int nItem = lpDrawItemStruct->itemID;
	BOOL bFocus = (GetFocus() == this);
	COLORREF clrTextSave, clrBkSave;
	COLORREF clrImage = m_clrBkgnd;
	static _TCHAR szBuff[MAX_PATH];
	LPCTSTR pszText;

// get item data

	LV_ITEM lvi;
	lvi.mask = LVIF_TEXT | LVIF_IMAGE | LVIF_STATE;
	lvi.iItem = nItem;
	lvi.iSubItem = 0;
	lvi.pszText = szBuff;
	lvi.cchTextMax = sizeof(szBuff);
	lvi.stateMask = 0xFFFF;		// get all state flags
	ListCtrl.GetItem(&lvi);

	BOOL bSelected = (bFocus || (GetStyle() & LVS_SHOWSELALWAYS)) && lvi.state & LVIS_SELECTED;
	bSelected = bSelected || (lvi.state & LVIS_DROPHILITED);

// set colors if item is selected

	CRect rcAllLabels;
	ListCtrl.GetItemRect(nItem, rcAllLabels, LVIR_BOUNDS);

	CRect rcLabel;
	ListCtrl.GetItemRect(nItem, rcLabel, LVIR_LABEL);

	rcAllLabels.left = rcLabel.left;
	rcAllLabels.left += OFFSET_BORDER;
	if (m_bClientWidthSel && rcAllLabels.right<m_cxClient)
		rcAllLabels.right = m_cxClient;

	if (bSelected)
	{
		if (m_bFocus==0)
		{
			clrBkSave = pDC->SetBkColor(::GetSysColor(COLOR_MENU));
			pDC->FillRect(rcAllLabels, &CBrush(::GetSysColor(COLOR_MENU)));
			clrTextSave = pDC->GetTextColor();
		}
		else
		{
			clrBkSave = pDC->SetBkColor(::GetSysColor(COLOR_HIGHLIGHT));
			pDC->FillRect(rcAllLabels, &CBrush(::GetSysColor(COLOR_HIGHLIGHT)));
			clrTextSave = pDC->SetTextColor(::GetSysColor(COLOR_HIGHLIGHTTEXT));
		}
	}
	else
	{
		clrBkSave = pDC->GetBkColor();
		pDC->FillRect(rcAllLabels, &CBrush(m_clrTextBk));
	}

// set color and mask for the icon

	if (lvi.state & LVIS_CUT)
	{
		clrImage = m_clrBkgnd;
		uiFlags |= ILD_BLEND50;
	}

// draw state icon

	UINT nStateImageMask = lvi.state & LVIS_STATEIMAGEMASK;
	if (nStateImageMask)
	{
		int nImage = (nStateImageMask>>12) - 1;
		pImageList = ListCtrl.GetImageList(LVSIL_STATE);
		if (pImageList)
		{
			pImageList->Draw(pDC, nImage,
				CPoint(rcItem.left, rcItem.top), ILD_TRANSPARENT);
		}
	}

// draw normal and overlay icon

	CRect rcIcon;
	ListCtrl.GetItemRect(nItem, rcIcon, LVIR_ICON);

	pImageList = ListCtrl.GetImageList(LVSIL_SMALL);
	if (pImageList)
	{
		UINT nOvlImageMask=lvi.state & LVIS_OVERLAYMASK;
		if (rcItem.left<rcItem.right-1)
		{
			RECT r;
			r.top=rcAllLabels.top;
			r.bottom=rcAllLabels.bottom;
			r.left=0;
			r.right=rcAllLabels.left;
			pDC->FillRect(&r, &CBrush(clrBkSave));
			ImageList_DrawEx(pImageList->m_hImageList, lvi.iImage,
					pDC->m_hDC,rcIcon.left,rcIcon.top, 16, 16,
					m_clrBkgnd, clrImage, uiFlags | nOvlImageMask);
		}
	}

// draw item label

	ListCtrl.GetItemRect(nItem, rcItem, LVIR_LABEL);
	rcItem.right -= m_cxStateImageOffset;

	pszText = MakeShortString(pDC, szBuff,
				rcItem.right-rcItem.left, 2*(OFFSET_FIRST));

	rcLabel = rcItem;
	rcLabel.left += OFFSET_FIRST;
	rcLabel.right -= OFFSET_FIRST;

	pDC->DrawText(pszText,-1,rcLabel,DT_LEFT | DT_SINGLELINE | DT_NOPREFIX | DT_NOCLIP | DT_VCENTER);

// draw labels for extra columns

	LV_COLUMN lvc;
	lvc.mask = LVCF_FMT | LVCF_WIDTH;
				   
	for(int nColumn = 1; ListCtrl.GetColumn(nColumn, &lvc); nColumn++)
	{
		rcItem.left = rcItem.right;
		rcItem.right += lvc.cx;

		int nRetLen = ListCtrl.GetItemText(nItem, nColumn,
						szBuff, sizeof(szBuff));
		rcLabel = rcItem;
		pDC->MoveTo(rcLabel.left-1,rcLabel.top);
		pDC->LineTo(rcLabel.left-1,rcLabel.bottom);
		RECT r2;
		r2.top=rcLabel.top;
		r2.bottom=rcLabel.bottom;
		r2.left=rcLabel.left;
		r2.right=rcLabel.left + OFFSET_OTHER;
		pDC->FillRect(&r2, &((CBrush)pDC->GetBkColor()));

		if (nRetLen == 0)
			continue;

		pszText = MakeShortString(pDC, szBuff,
			rcItem.right - rcItem.left, 2*(OFFSET_OTHER));

		UINT nJustify = DT_LEFT;

		if(pszText == szBuff)
		{
			switch(lvc.fmt & LVCFMT_JUSTIFYMASK)
			{
			case LVCFMT_RIGHT:
				nJustify = DT_RIGHT;
				break;
			case LVCFMT_CENTER:
				nJustify = DT_CENTER;
				break;
			default:
				break;
			}
		}

		rcLabel.left += OFFSET_OTHER;
		rcLabel.right -= OFFSET_OTHER;

		pDC->DrawText(pszText, -1, rcLabel,
			nJustify | DT_SINGLELINE | DT_NOPREFIX | DT_NOCLIP | DT_VCENTER);
	}

// draw focus rectangle if item has focus

	if (lvi.state & LVIS_FOCUSED && bFocus)
		pDC->DrawFocusRect(rcAllLabels);

// set original colors if item was selected

	if (bSelected)
	{
        pDC->SetTextColor(clrTextSave);
		pDC->SetBkColor(clrBkSave);
	}
}

LPCTSTR CListViewEx::MakeShortString(CDC* pDC, LPCTSTR lpszLong, int nColumnLen, int nOffset)
{
	static const _TCHAR szThreeDots[] = _T("...");

	int nStringLen = lstrlen(lpszLong);

	if(nStringLen == 0 ||
		(pDC->GetTextExtent(lpszLong, nStringLen).cx + nOffset) <= nColumnLen)
	{
		return(lpszLong);
	}

	static _TCHAR szShort[MAX_PATH];

	lstrcpy(szShort,lpszLong);
	int nAddLen = pDC->GetTextExtent(szThreeDots,sizeof(szThreeDots)).cx;

	for(int i = nStringLen-1; i > 0; i--)
	{
		szShort[i] = 0;
		if((pDC->GetTextExtent(szShort, i).cx + nOffset + nAddLen)
			<= nColumnLen)
		{
			break;
		}
	}

	lstrcat(szShort, szThreeDots);
	return(szShort);
}

void CListViewEx::RepaintSelectedItems()
{
	CListCtrl& ListCtrl = GetListCtrl();
	CRect rcItem, rcLabel;

// invalidate focused item so it can repaint properly

	int nItem = ListCtrl.GetNextItem(-1, LVNI_FOCUSED);

	if(nItem != -1)
	{
		ListCtrl.GetItemRect(nItem, rcItem, LVIR_BOUNDS);

		InvalidateRect(rcItem, FALSE);
	}

// if selected items should not be preserved, invalidate them

	//if(!(GetStyle() & LVS_SHOWSELALWAYS))
	{
		for(nItem = ListCtrl.GetNextItem(-1, LVNI_SELECTED);
			nItem != -1; nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED))
		{
			ListCtrl.GetItemRect(nItem, rcItem, LVIR_BOUNDS);
			//ListCtrl.GetItemRect(nItem, rcLabel, LVIR_LABEL);
			//rcItem.left = rcLabel.left;
			rcItem.right=m_cxClient;

			InvalidateRect(rcItem, FALSE);
		}
	}

// update changes 

	UpdateWindow();
}

/////////////////////////////////////////////////////////////////////////////
// CListViewEx diagnostics

#ifdef _DEBUG

void CListViewEx::Dump(CDumpContext& dc) const
{
	CListView::Dump(dc);

	dc << "m_bFullRowSel = " << (UINT)m_bFullRowSel;
	dc << "\n";
	dc << "m_cxStateImageOffset = " << m_cxStateImageOffset;
	dc << "\n";
}

#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CListViewEx message handlers

LRESULT CListViewEx::OnSetImageList(WPARAM wParam, LPARAM lParam)
{
	if( (int) wParam == LVSIL_STATE)
	{
		int cx, cy;

		if(::ImageList_GetIconSize((HIMAGELIST)lParam, &cx, &cy))
			m_cxStateImageOffset = cx;
		else
			m_cxStateImageOffset = 0;
	}

	return(Default());
}

LRESULT CListViewEx::OnSetTextColor(WPARAM wParam, LPARAM lParam)
{
	m_clrText = (COLORREF)lParam;
	return(Default());
}

LRESULT CListViewEx::OnSetTextBkColor(WPARAM wParam, LPARAM lParam)
{
	m_clrTextBk = (COLORREF)lParam;
	return(Default());
}

LRESULT CListViewEx::OnSetBkColor(WPARAM wParam, LPARAM lParam)
{
	m_clrBkgnd = (COLORREF)lParam;
	return(Default());
}

void CListViewEx::OnSize(UINT nType, int cx, int cy) 
{
	m_cxClient = cx;
	CListView::OnSize(nType, cx, cy);
}

void CListViewEx::OnPaint() 
{
	// in full row select mode, we need to extend the clipping region
	// so we can paint a selection all the way to the right
	if (m_bClientWidthSel &&
		(GetStyle() & LVS_TYPEMASK) == LVS_REPORT &&
		GetFullRowSel())
	{
		CRect rcAllLabels;
		GetListCtrl().GetItemRect(0, rcAllLabels, LVIR_BOUNDS);

		if(rcAllLabels.right < m_cxClient)
		{
			// need to call BeginPaint (in CPaintDC c-tor)
			// to get correct clipping rect
			CPaintDC dc(this);

			CRect rcClip;
			dc.GetClipBox(rcClip);

			rcClip.left = min(rcAllLabels.right-1, rcClip.left);
			rcClip.right = m_cxClient;

			InvalidateRect(rcClip, FALSE);
			// EndPaint will be called in CPaintDC d-tor
		}
	}

	CListView::OnPaint();
}

void CListViewEx::OnSetFocus(CWnd* pOldWnd) 
{
	CListView::OnSetFocus(pOldWnd);
	m_bFocus=TRUE;

	// check if we are getting focus from label edit box
	if(pOldWnd!=NULL && pOldWnd->GetParent()==this)
		return;

	// repaint items that should change appearance
	if(m_bFullRowSel && (GetStyle() & LVS_TYPEMASK)==LVS_REPORT)
		RepaintSelectedItems();
}

void CListViewEx::OnKillFocus(CWnd* pNewWnd) 
{
	CListView::OnKillFocus(pNewWnd);
	m_bFocus=FALSE;

	// check if we are losing focus to label edit box
	if(pNewWnd != NULL && pNewWnd->GetParent() == this)
		return;

	// repaint items that should change appearance
	if(m_bFullRowSel && (GetStyle() & LVS_TYPEMASK) == LVS_REPORT)
		RepaintSelectedItems();
}

BOOL CListViewEx::OnPreparePrinting(CPrintInfo* pInfo)
{
	BOOL b=DoPreparePrinting (pInfo);
    return b;
}

void CListViewEx::OnBeginPrinting(CDC* pDC, CPrintInfo* pInfo)
{
	CView::OnBeginPrinting(pDC, pInfo);

	if (!pInfo->m_bPreview)
		return;

}

void CListViewEx::OnPrint(CDC* pDC, CPrintInfo* pInfo)
{
	// get string to show as "filename" in header/footer
	LPCTSTR pszFileName = GetDocument()->GetPathName();
	CFont* pOldFont=NULL;
	CFont font;
    if(m_fontPrinter) pOldFont = pDC->SelectObject (m_fontPrinter);

	if (pszFileName[0] == 0)
		pszFileName = GetDocument()->GetTitle();

	// go thru global CSetupPageDlg to format the header and footer
	SetStrHeader(m_strHeader);
	SetStrFooter(m_strFooter);
	CString strHeader=m_strHeader;
	CString strFooter=m_strFooter;
	RECT RightBorder;
	RightBorder.top=0;
	RightBorder.bottom=pInfo->m_rectDraw.bottom;
	RightBorder.left=pInfo->m_rectDraw.right-dlgSetupPage.right;
	RightBorder.right=pInfo->m_rectDraw.right;
	SIZE size;

	size.cy=dlgSetupPage.top*100;
	pDC->HIMETRICtoDP(&size);
	pInfo->m_rectDraw.top+=size.cy;
	size.cy=dlgSetupPage.bottom*100;
	pDC->HIMETRICtoDP(&size);
	pInfo->m_rectDraw.bottom-=size.cy;
	size.cx=dlgSetupPage.left*100;
	pDC->HIMETRICtoDP(&size);
	pInfo->m_rectDraw.left+=size.cx;
	size.cx=dlgSetupPage.right*100;
	pDC->HIMETRICtoDP(&size);
	pInfo->m_rectDraw.right-=size.cx;

	// Added
	TEXTMETRIC tm;
	pDC->GetTextMetrics(&tm);
	int cyChar = tm.tmHeight+tm.tmHeight/4;
	int nl=GetListCtrl().GetItemCount();
	int heigth=pInfo->m_rectDraw.bottom-pInfo->m_rectDraw.top;
	if (!strHeader.IsEmpty()) heigth-=(tm.tmHeight+tm.tmHeight/2);
	if (!strFooter.IsEmpty()) heigth-=(tm.tmHeight+tm.tmHeight/2);
	int nls=nl*cyChar;
	int np=nls/heigth;
	if (nls%heigth) np++;
	int nlpp=heigth/cyChar;
	pInfo->SetMaxPage(np);
	if (pInfo->m_nCurPage>=(unsigned)np) 
		pInfo->m_bContinuePrinting=FALSE;
	// End Added
	
	pDC->GetTextMetrics(&tm);
	cyChar = tm.tmHeight;
	CRect rectPage = pInfo->m_rectDraw;
	int a,b,c,d,e;

	// draw and exclude space for header
	if (!strHeader.IsEmpty())
	{
		a=rectPage.left;
		b=rectPage.top;
		rectPage.top += cyChar + cyChar / 4;
		c=rectPage.left;
		d=rectPage.top;
		e=rectPage.right;
		rectPage.top += cyChar / 4;
	}

	// allow space for footer
	pInfo->m_rectDraw = rectPage;
	if (!strFooter.IsEmpty())
		pInfo->m_rectDraw.bottom -= cyChar + cyChar/4 + cyChar/4;

	// draw body text
	Print(pDC, pInfo, (pInfo->m_nCurPage-1)*nlpp,pInfo->m_nCurPage*nlpp-1,cyChar);

	if (!strHeader.IsEmpty())
	{
		pDC->TextOut(a,b, strHeader);
		pDC->MoveTo(c,d);
		pDC->LineTo(e,d);
	}
	// draw footer
	if (!strFooter.IsEmpty())
	{
		rectPage.bottom -= cyChar+cyChar/2;
		pDC->TextOut(rectPage.left, rectPage.bottom, strFooter);
		rectPage.bottom -= cyChar / 4;
		pDC->MoveTo(rectPage.left, rectPage.bottom);
		pDC->LineTo(rectPage.right, rectPage.bottom);
		rectPage.bottom -= cyChar / 4;
	}
    if (pOldFont) pDC->SelectObject (pOldFont);
	pDC->FillRect(&RightBorder, &CBrush(pDC->GetBkColor()));
}

void CListViewEx::OnPrepareDC(CDC* pDC, CPrintInfo* pInfo)
{
	ASSERT_VALID(pDC);
	UNUSED(pDC); // unused in release builds

	// Default to one page printing if doc length not known
	if (pInfo != NULL)
		pInfo->m_bContinuePrinting =
			(pInfo->m_nCurPage<=pInfo->GetMaxPage() || (pInfo->m_nCurPage == 1));
}

void CListViewEx::Print(CDC* pDC, CPrintInfo* pInfo, int nLStart, int nLEnd, int cyChar)
{
	int i,j,k,h;
	LV_COLUMN lvc;
	lvc.mask = LVCF_FMT | LVCF_WIDTH;
	int nColumn;			   
	for(nColumn = 0; GetListCtrl().GetColumn(nColumn, &lvc); nColumn++);
	int *s;
	s=new int[nColumn];
	CListCtrl &ListCtrl=GetListCtrl();
	CRect rectPage = pInfo->m_rectDraw;
	CString str;
	k=0;
	LV_COLUMN Column;
	Column.mask=LVCF_WIDTH;
	for (i=0;i<nColumn;i++)
	{
		ListCtrl.GetColumn(i, &Column );
		s[i]=Column.cx;
		k+=s[i];
	}
	h=rectPage.right-rectPage.left;
	for (i=0;i<nColumn;i++)
	{
		s[i]=s[i]*h/k;
	}

	int l;
	for(i=nLStart;i<=nLEnd;i++)
	{
		k=0;
		for(j=0;j<nColumn;j++)
		{
			CString str;
			str=ListCtrl.GetItemText(i,j);
			if (str!="")
			{
			 if (rectPage.left+k>=rectPage.right) break;
			 while (rectPage.left+k+(pDC->GetTextExtent(str)).cx>=rectPage.left+k+s[j] && 
					str.GetLength()>0)
			 {
				 str=str.Left(str.GetLength()-1);
				 l=str.GetLength()-1;
				 if (l>0) str.SetAt (l,'.');
				 l--;
				 if (l>0) str.SetAt (l,'.');
				 l--;
				 if (l>0) str.SetAt (l,'.');
			 }
			 if (k>0 && i<=nLEnd)
			 {
			  pDC->MoveTo(rectPage.left+k-3, rectPage.top);
			  pDC->LineTo(rectPage.left+k-3, rectPage.top+cyChar + cyChar / 4);
			 }
			 pDC->TextOut(rectPage.left+k, rectPage.top, str);
			 k+=s[j];
			}
		}
		rectPage.top += cyChar + cyChar / 4;
	}
	delete[]s;
}


void CListViewEx::OnUpdateFilePrint(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(TRUE);
}

void CListViewEx::OnPrintPreview()
{
	CView::OnFilePrintPreview();
}

// Clipboard support
void CListViewEx::OnEditCopy() 
{
  LPTSTR  lpDataBuffer;
  HGLOBAL hDataBuffer;
  DWORD dwMemSize;

  int ni = (DWORD) GetListCtrl().GetItemCount();
  LV_COLUMN lvc;
  lvc.mask = LVCF_FMT | LVCF_WIDTH;
  int nColumn,i;			   
  for (nColumn = 0; GetListCtrl().GetColumn(nColumn, &lvc); nColumn++);
  CString string,str;
  for(i=0; i<(signed)ni; i++)
	  {
		if (GetListCtrl().GetItemState(i,LVIS_SELECTED)==0) continue;
		for(int j=0;j<nColumn;j++)
		{
			str=GetListCtrl().GetItemText(i,j);
			string+=str;
			if (j<nColumn-1) string+=m_Separator; 
			 else string+="\r\n"; 
		}
	  }
  string+="\0";
  dwMemSize=string.GetLength()+1;

  if( !(hDataBuffer = GlobalAlloc( GMEM_DDESHARE, dwMemSize ) ) ) {
    AfxMessageBox("Out of memory");
    return;
  }
  if( !(lpDataBuffer = (LPTSTR) GlobalLock( hDataBuffer ) ) ) {
    GlobalFree( hDataBuffer );
    AfxMessageBox("Out of memory");
    return;
  }

  *lpDataBuffer = '\0';
  memcpy(lpDataBuffer,(const char*)string,dwMemSize);

  GlobalUnlock( hDataBuffer );

  if( !OpenClipboard() ) {
    Sleep( 250 );  // wait a quarter second and try again.
    if( !OpenClipboard() ) {
      AfxMessageBox("Could not open the Clipboard!");
      GlobalFree( hDataBuffer );
      return;
    }
  }
  if( !EmptyClipboard() ) {
    AfxMessageBox("Could not empty the Clipboard!");
    GlobalFree( hDataBuffer );
    return;
  }
  if( !SetClipboardData( CF_TEXT, hDataBuffer ) ) {
    AfxMessageBox("Could not copy data to the Clipboard!");
    GlobalFree( hDataBuffer );
    return;
  }
  CloseClipboard();

  return;
}


void CListViewEx::SetBorders(RECT * r)
{
	if (!r) return;
	dlgSetupPage.top=r->top;
	dlgSetupPage.bottom=r->bottom;
	dlgSetupPage.left=r->left;
	dlgSetupPage.right=r->right;
}

void CListViewEx::OnUpdateEditCopy(CCmdUI* pCmdUI) 
{
  int ni = (DWORD) GetListCtrl().GetSelectedCount();
  if (ni>0) pCmdUI->Enable(TRUE); 
  else pCmdUI->Enable(FALSE); 
}
