/************************************************************
MODULO: Packet32.c 
	Rappresenta la DLL di interfacciamento al VXD catturatore
	di pacchetti.
************************************************************/
#include <windows.h>
#include <windowsx.h>
#include <crtdbg.h>
#include "..\Inc\ntddndis.h"
#include "..\Inc\ntddpack.h"
#include "..\Inc\packet32.h"

#ifdef __cplusplus
extern "C"
{		/* Impostiamo le chiamate C se il compilatore  C++*/
#endif				

TCHAR szWindowTitle[] = TEXT ("PACKET32.DLL");
LPADAPTER lpTheAdapter = NULL;
ULONG ulInstanceCount = 0;

#if DBG
#define ODS(_x) OutputDebugString(TEXT(_x))
#else
#define ODS(_x)
#endif

BOOLEAN StartPacketDriver (LPTSTR ServiceName);
BOOLEAN StopPacketDriver (void);

/************************************************************
Questa routine  il punto di ingresso della DLL di 
interfacciamento al VXD.
INPUT:	hModule
		dwReason
		lpReserved
OUTPUT:	TRUE o FALSE in base all'esito della inizializ-
		zazione.
************************************************************/
BOOL APIENTRY DllMain (HANDLE hModule,
					DWORD dwReason,
					LPVOID lpReserved)
{
    BOOL Status;

	ODS ("Packet32: DllEntry\n");
    switch (dwReason)
	{
		case DLL_PROCESS_ATTACH:
		case DLL_THREAD_ATTACH:
			Status = StartPacketDriver (TEXT ("VPACKET"));
			break;
		case DLL_PROCESS_DETACH:
		case DLL_THREAD_DETACH:
			Status = StopPacketDriver ();
			break;
		default:
			Status = TRUE;
			break;
	}
	return Status;
}

/************************************************************
Questa routine  quella che si preoccupa di spedire i messaggi
al VXD mediante la DeviceIOControl. 
INPUT:	lpAdapterObject - Ritornata da PacketOpenAdapter
		lpPacket		- Ritornata da PacketAllocatePacket
		ulIoctl			- Funzione da invocare
		bSync			- Indica se il servizio  sincrono o no
OUTPUT:	TRUE o FALSE in base all'esito della operazione.
************************************************************/
BOOL PacketDeviceIoControl (LPADAPTER lpAdapterObject,
			      LPPACKET lpPacket,
			      ULONG ulIoctl,
			      BOOLEAN bSync)
{
	BOOLEAN Result;
	DWORD Error;
 
	ODS ("Packet32: PacketDeviceIoControl\n");
	_ASSERTE (lpAdapterObject != NULL);
	_ASSERTE (lpPacket != NULL);
	lpPacket->OverLapped.Offset = 0;
	lpPacket->OverLapped.OffsetHigh = 0;
	lpPacket->ulBytesReceived		= 0;
	if (!ResetEvent (lpPacket->OverLapped.hEvent))
	{
		lpPacket->bIoComplete = FALSE;
		return FALSE;
	}
    Result = DeviceIoControl (lpAdapterObject->hFile,
				ulIoctl,
				lpPacket->Buffer,
				lpPacket->Length,
				lpPacket->Buffer,
				lpPacket->Length,
				&(lpPacket->ulBytesReceived), 
				&(lpPacket->OverLapped));
	Error=GetLastError () ;
    
	if (!Result && bSync)
	{
		if (Error == ERROR_IO_PENDING)
		{
			Result = GetOverlappedResult (lpAdapterObject->hFile,
					&(lpPacket->OverLapped),
					&(lpPacket->ulBytesReceived), 
					TRUE);
		}
		else
			ODS ("Packet32: unsupported API call return error!\n");
	}

	lpPacket->bIoComplete = Result;

	return Result;
}


/************************************************************
Routine per aprire una istanza dell'adattore di cui  fornito
il nome come parametro. //Ancora non implementato
INPUT:	AdapterName	- Nome dell'adattore su cui fare il bind.
OUTPUT:	Ritorna un puntatore ad un oggetto adattore, o NULL se
		c' stato un errore.
************************************************************/
LPADAPTER PacketOpenAdapter (LPTSTR AdapterName)
{
    LPPACKET lpSupport;	

    ODS ("Packet32: PacketOpenAdapter\n");
    lpSupport=PacketAllocatePacket();
    PacketInitPacket(lpSupport,AdapterName,strlen(AdapterName));
    	if (lpTheAdapter && (lpTheAdapter->hFile != INVALID_HANDLE_VALUE))
	  {
	    PacketDeviceIoControl(lpTheAdapter,
				  lpSupport,
				  (ULONG) IOCTL_OPEN,
				  TRUE);
		if (((char*)lpSupport->Buffer)[0]=='\0') goto err;
	    PacketFreePacket(lpSupport);
	    return lpTheAdapter;
	  }
err:
    ODS ("Packet32: PacketOpenAdapter Could not open adapter\n");
    return NULL;
}


/************************************************************
Routine per chiudere una istanza di un adattore di cui  fornito
il nome come parametro. //Ancora non implementato
INPUT:	lpAdapter - Adattatore da chiudere.
OUTPUT:	
************************************************************/
VOID PacketCloseAdapter (LPADAPTER lpAdapter)
{
	ODS ("Packet32: PacketCloseAdapter\n");
    CloseHandle (lpAdapter->hFile);
    GlobalFreePtr (lpAdapter);
}

/************************************************************
Routine per allocare lo spazio in memoria per contenere un
pacchetto.
INPUT:
OUTPUT:	Un puntatore al pacchetto o NULL.
************************************************************/
LPPACKET PacketAllocatePacket (void)
{
	LPPACKET lpPacket;

    lpPacket = (LPPACKET) GlobalAllocPtr (
					   GMEM_MOVEABLE | GMEM_ZEROINIT,
					   sizeof (PACKET));
    if (lpPacket == NULL)
	{
		ODS ("Packet32: PacketAllocatePacket: GlobalAlloc Failed\n");
		return NULL;
	}
    lpPacket->OverLapped.hEvent = CreateEvent (NULL,
						FALSE,
						FALSE,
						NULL);
    if (lpPacket->OverLapped.hEvent == NULL)
    {
		ODS ("Packet32: PacketAllocateSendPacket: CreateEvent Failed\n");
		GlobalFreePtr (lpPacket);
		return NULL;
    }
	lpPacket->Buffer=NULL;
	lpPacket->Length=0;
	lpPacket->ulBytesReceived	= 0;
	lpPacket->bIoComplete		= FALSE;

    return lpPacket;
}

/************************************************************
Routine per disallocare lo spazio in memoria occupata da un
pacchetto.
INPUT:	lpPacket - Pacchetto da liberare.
OUTPUT:	
************************************************************/
VOID PacketFreePacket (LPPACKET lpPacket)
{
	CloseHandle (lpPacket->OverLapped.hEvent);
    GlobalFreePtr (lpPacket);
}

/************************************************************
Routine per inizializzate lo spazio in memoria per contenere 
un pacchetto.
INPUT:	lpPacket - Adattatore cui  legato il pacchetto.
		Buffer - Buffer da usare per inizializzare lpPacket
		Length - Lunghezza del buffer di inizializzazione.
OUTPUT:	
************************************************************/
VOID PacketInitPacket (LPPACKET lpPacket,
				       PVOID Buffer,
				       UINT Length)
{
	lpPacket->Buffer = Buffer;
    lpPacket->Length = Length;
}

/************************************************************
Routine per spedire un pacchetto su un adattore.
INPUT:	AdapterObject - Adattatore su cui spedire il pacchetto
		lpPacket - Pacchetto da spedire.
		Sync - TRUE se il servizio deve essere sincrono.
OUTPUT:	TRUE se ben eseguito e Sync==TRUE
************************************************************/
BOOLEAN PacketSendPacket (LPADAPTER AdapterObject,
						LPPACKET lpPacket,
						BOOLEAN Sync)
{
	return PacketDeviceIoControl (AdapterObject,
				    lpPacket,
				    (ULONG) IOCTL_PROTOCOL_WRITE,
				    Sync);
}

/************************************************************
Routine per ricevere un pacchetto da un adattore.
INPUT:	AdapterObject - Adattatore su cui ricevere il pacchetto
		lpPacket - Buffer in cui termina il pacchetto ricevuto.
		Sync - TRUE se il servizio deve essere sincrono.
		BytesReceived - Puntatore al numero di byte letti.
OUTPUT:	TRUE se ben eseguito e Sync==TRUE
************************************************************/
BOOLEAN PacketReceivePacket(LPADAPTER AdapterObject,
		       LPPACKET lpPacket,
		       BOOLEAN Sync)
{
	return PacketDeviceIoControl(AdapterObject,
								lpPacket,
								(ULONG) IOCTL_PROTOCOL_READ,
								Sync);

}


/************************************************************
Routine per aspettare il completamento di una operazione.
INPUT:	AdapterObject - Adattatore su cui ricevere il pacchetto
		lpPacket - Buffer in cui termina il pacchetto ricevuto.
		BytesReceived - Puntatore al numero di byte letti.
OUTPUT:	TRUE se ben eseguito e Sync==TRUE
************************************************************/
BOOLEAN PacketWaitPacket (LPADAPTER AdapterObject,
		       LPPACKET lpPacket)
{
	lpPacket->bIoComplete =  GetOverlappedResult( AdapterObject->hFile,
													&lpPacket->OverLapped,
													&lpPacket->ulBytesReceived,
												    TRUE );

	return lpPacket->bIoComplete;

}

/************************************************************
Routine per resettare un adattore.
INPUT:	AdapterObject - Adattatore da resettare.
OUTPUT:	TRUE
************************************************************/
BOOLEAN PacketResetAdapter (LPADAPTER AdapterObject)
{
    UINT BytesReturned;
    DeviceIoControl (
		      AdapterObject->hFile,
		      (DWORD) IOCTL_PROTOCOL_RESET,
		      NULL,
		      0,
		      NULL,
		      0,
		      &BytesReturned,
		      NULL);
    return TRUE;
}

/************************************************************
Routine per richiedere informazioni ad un adattatore.
INPUT:	AdapterObject - Adattatore cui richiedere le inf.
		Set - Se TRUE allora setta i parametri altrimenti li 
			  richiede.
		OidData - Struttura contenente i dettagli dell'OID.
OUTPUT:	TRUE o FALSE.
************************************************************/
BOOLEAN PacketRequest (LPADAPTER AdapterObject,
		    BOOLEAN Set,
		    PPACKET_OID_DATA OidData)
{
    UINT BytesReturned;
    BOOLEAN Result;
	OVERLAPPED Overlap;

	ODS ("Packet32: PacketRequest\n");
	_ASSERTE (AdapterObject != NULL);
	_ASSERTE (OidData != NULL);
	_ASSERTE (OidData->Data != NULL);
	Overlap.Offset = 0;
	Overlap.OffsetHigh = 0;
    Overlap.hEvent = CreateEvent (NULL,
						FALSE,
						FALSE,
						NULL);
    if (Overlap.hEvent == NULL)
    {
		ODS ("Packet32: PacketRequestPacket: CreateEvent Failed\n");
		return FALSE;
    }
	if (!ResetEvent(Overlap.hEvent))
	{
		ODS ("Packet32: PacketRequestPacket: ResetEvent Failed\n");
		return FALSE;
	}
    Result = DeviceIoControl (
			       AdapterObject->hFile,
	    (DWORD) Set ? IOCTL_PROTOCOL_SET_OID : IOCTL_PROTOCOL_QUERY_OID,
			       OidData,
			     sizeof (PACKET_OID_DATA) - 1 + OidData->Length,
			       OidData,
			     sizeof (PACKET_OID_DATA) - 1 + OidData->Length,
			       &BytesReturned,
			       &Overlap);
	if (!Result)
	{
		if (GetLastError() == ERROR_IO_PENDING)
		{
			Result = GetOverlappedResult(AdapterObject->hFile,
										&Overlap,
										&BytesReturned,
										TRUE);
		}
		else
		{
			ODS("Packet32: Unssupported API call return error!\n");
		}
	}
	if (BytesReturned == 0)
	{
		// There was an ndis error
		ODS ("Packet32: Ndis returned error to OID\n");
		Result = FALSE;
	}
	return Result;
}

/************************************************************
Routine per impostare un filtro su un adattatore.
INPUT:	AdapterObject - Adattatore cui impostare il filtro.
		Filter - Tipo di filtro da impostare.
OUTPUT:	TRUE o FALSE.
************************************************************/
BOOLEAN PacketSetFilter (LPADAPTER AdapterObject,
				      ULONG Filter)
{
	BOOLEAN Status;
    ULONG IoCtlBufferLength = (sizeof (PACKET_OID_DATA) + sizeof (ULONG) - 1);
    PPACKET_OID_DATA OidData;

	ODS ("Packet32: PacketSetFilter\n");
	_ASSERTE (AdapterObject != NULL);
    OidData = GlobalAllocPtr (
			       GMEM_MOVEABLE | GMEM_ZEROINIT,
			       IoCtlBufferLength
      );
    if (OidData == NULL)
	{
		return FALSE;
	}
    OidData->Oid = OID_GEN_CURRENT_PACKET_FILTER;
    OidData->Length = sizeof (ULONG);
    *((PULONG) OidData->Data) = Filter;
    Status = PacketRequest (
			     AdapterObject,
			     TRUE,
			     OidData
      );
    GlobalFreePtr (OidData);
    return Status;
}

/************************************************************
Routine per avviare il collegamento con il VXD.
INPUT:	lpstrServiceName - VXD da avviare.
OUTPUT:	TRUE o FALSE.
************************************************************/
BOOLEAN StartPacketDriver (LPTSTR lpstrServiceName)
{
    BOOLEAN Result;
    ODS ("Packet32: StartPacketDriver\n");

    if (lpTheAdapter)
	{
		ulInstanceCount++;
		Result = TRUE;
	}
    else
	{
		lpTheAdapter = (LPADAPTER) GlobalAllocPtr (GMEM_MOVEABLE | GMEM_ZEROINIT,
							   sizeof (ADAPTER));
		if (lpTheAdapter == NULL)
		{
			ODS ("Packet32: StartPacketDriver GlobalAlloc Failed\n");
			return FALSE;
		}
		wsprintf (lpTheAdapter->SymbolicLink,
			TEXT ("\\\\.\\%s"),
			lpstrServiceName);
		lpTheAdapter->hFile = CreateFile (lpTheAdapter->SymbolicLink,
					  GENERIC_WRITE | GENERIC_READ,
					  0,
					  NULL,
					  OPEN_EXISTING,
					  FILE_FLAG_OVERLAPPED,
					  0);
		if (lpTheAdapter->hFile != INVALID_HANDLE_VALUE)
		{
			ulInstanceCount++;
			Result = TRUE;
		}
		else
		{
			ODS ("Packet32: StartPacketDriver Could not open adapter\n");
			GlobalFreePtr (lpTheAdapter);
			lpTheAdapter = NULL;
			Result = FALSE;
		}
	}
	return Result;
}

/************************************************************
Routine per stoppare il collegamento con il VXD.
INPUT:	
OUTPUT:	TRUE o FALSE.
************************************************************/
BOOLEAN StopPacketDriver(void)
{
	ODS ("Packet32: StopPacketDriver\n");
    if (lpTheAdapter)
	{
		if (--ulInstanceCount == 0)
		{
			CloseHandle (lpTheAdapter->hFile);
			GlobalFreePtr (lpTheAdapter);
			lpTheAdapter = NULL;
		}
	}
    return TRUE;
}


/************************************************************
Routine per stoppare il collegamento con il VXD.
INPUT:	pStr - Puntatore al buffer di stringe contenente il
				nome degli adattori possibili.
		BufferSize - Puntatore alle dimensioni del buffer 
					 delle stringhe, serve per comunicare in
					 ingresso la dim. del buffer definito;
					 in uscita serve invece a comunicare la 
					 quantit di caratteri letti.
OUTPUT:	TRUE o FALSE.
************************************************************/
ULONG PacketGetAdapterNames (PTSTR pStr,
				PULONG BufferSize)
{
    LPPACKET lpPacket;
    ULONG Result,i;

    lpPacket = PacketAllocatePacket();
	_ASSERTE (lpPacket != NULL);
	
	for (i=0;i<*BufferSize;i++) pStr[i]=0;
	PacketInitPacket(lpPacket,
					pStr,
					*BufferSize);
	ODS ("Packet32: PacketGetAdapterNames\n");
	Result = PacketDeviceIoControl(lpTheAdapter,
								  lpPacket,
								  (ULONG)IOCTL_PROTOCOL_MACNAME,
								  TRUE );
	if (Result)
		*BufferSize = lpPacket->ulBytesReceived;
	else 
		*BufferSize = 0;

	PacketFreePacket(lpPacket);
	return Result;
}

#ifdef __cplusplus
}				/* Chiude la definizione di extern C */
#endif				
