//========================================================
//
// Physmem
//
// Mark Russinovich
// Systems Internals
// http://www.sysinternals.com
//
// This program demonstrates how you can open and
// map physical memory. This is essentially the NT 
// equivalent of the \dev\kmem device in UNIX.
//
//========================================================
#include <windows.h>
#include <stdio.h>
#include "native.h"

//
// Number of bytes to print per line
//
#define BYTESPERLINE	16

//
// Lines to print before pause
//
#define LINESPERSCREEN	25


//
// Functions in NTDLL that we dynamically locate
//

NTSTATUS (__stdcall *NtUnmapViewOfSection)(
		IN HANDLE  ProcessHandle,
		IN PVOID  BaseAddress
		);

NTSTATUS (__stdcall *NtOpenSection)(
		OUT PHANDLE  SectionHandle,
		IN ACCESS_MASK  DesiredAccess,
		IN POBJECT_ATTRIBUTES  ObjectAttributes
		);

NTSTATUS (__stdcall *NtMapViewOfSection)(
		IN HANDLE  SectionHandle,
		IN HANDLE  ProcessHandle,
		IN OUT PVOID  *BaseAddress,
		IN ULONG  ZeroBits,
		IN ULONG  CommitSize,
		IN OUT PLARGE_INTEGER  SectionOffset,	/* optional */
		IN OUT PULONG  ViewSize,
		IN SECTION_INHERIT  InheritDisposition,
		IN ULONG  AllocationType,
		IN ULONG  Protect
		);

VOID (__stdcall *RtlInitUnicodeString)(
		IN OUT PUNICODE_STRING  DestinationString,
		IN PCWSTR  SourceString
		);

ULONG (__stdcall *RtlNtStatusToDosError) (
		IN NTSTATUS Status
		);

//----------------------------------------------------------------------
//
// PrintError
//
// Formats an error message for the last error
//
//----------------------------------------------------------------------
void PrintError( char *message, NTSTATUS status )
{
	char *errMsg;

	FormatMessage( FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
			NULL, RtlNtStatusToDosError( status ), 
			MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), 
			(LPTSTR) &errMsg, 0, NULL );
	printf("%s: %s\n", message, errMsg );
	LocalFree( errMsg );
}


//--------------------------------------------------------
//
// UnmapPhysicalMemory
//
// Maps a view of a section.
//
//--------------------------------------------------------
VOID UnmapPhysicalMemory( DWORD Address )
{
	NTSTATUS		status;

	status = NtUnmapViewOfSection( (HANDLE) -1, (PVOID) Address );
	if( !NT_SUCCESS(status)) {

		PrintError("Unable to unmap view", status );
	}
}


//--------------------------------------------------------
//
// MapPhysicalMemory
//
// Maps a view of a section.
//
//--------------------------------------------------------
BOOLEAN MapPhysicalMemory( HANDLE PhysicalMemory,
							PDWORD Address, PDWORD Length,
							PDWORD VirtualAddress )
{
	NTSTATUS			ntStatus;
	PHYSICAL_ADDRESS	viewBase;
	char				error[256];

	*VirtualAddress = 0;
	viewBase.QuadPart = (ULONGLONG) (*Address);
	ntStatus = NtMapViewOfSection (PhysicalMemory,
                               (HANDLE) -1,
                               (PVOID) VirtualAddress,
                               0L,
                               *Length,
                               &viewBase,
                               Length,
                               ViewShare,
                               0,
                               PAGE_READONLY );

	if( !NT_SUCCESS( ntStatus )) {

		sprintf( error, "Could not map view of %X length %X",
				*Address, *Length );
		PrintError( error, ntStatus );
		return FALSE;					
	}

	*Address = viewBase.LowPart;
	return TRUE;
}


//--------------------------------------------------------
//
// OpensPhysicalMemory
//
// This function opens the physical memory device. It
// uses the native API since 
//
//--------------------------------------------------------
HANDLE OpenPhysicalMemory()
{
	NTSTATUS		status;
	HANDLE			physmem;
	UNICODE_STRING	physmemString;
	OBJECT_ATTRIBUTES attributes;
	WCHAR			physmemName[] = L"\\device\\physicalmemory";

	RtlInitUnicodeString( &physmemString, physmemName );	

	InitializeObjectAttributes( &attributes, &physmemString,
								OBJ_CASE_INSENSITIVE, NULL, NULL );			
	status = NtOpenSection( &physmem, SECTION_MAP_READ, &attributes );

	if( !NT_SUCCESS( status )) {

		PrintError( "Could not open \\device\\physicalmemory", status );
		return NULL;
	}

	return physmem;
}



//--------------------------------------------------------
//
// LocateNtdllEntryPoints
//
// Finds the entry points for all the functions we 
// need within NTDLL.DLL.
//
//--------------------------------------------------------
BOOLEAN LocateNtdllEntryPoints()
{
	if( !(RtlInitUnicodeString = (void *) GetProcAddress( GetModuleHandle("ntdll.dll"),
			"RtlInitUnicodeString" )) ) {

		return FALSE;
	}
	if( !(NtUnmapViewOfSection = (void *) GetProcAddress( GetModuleHandle("ntdll.dll"),
			"NtUnmapViewOfSection" )) ) {

		return FALSE;
	}
	if( !(NtOpenSection = (void *) GetProcAddress( GetModuleHandle("ntdll.dll"),
			"NtOpenSection" )) ) {

		return FALSE;
	}
	if( !(NtMapViewOfSection = (void *) GetProcAddress( GetModuleHandle("ntdll.dll"),
			"NtMapViewOfSection" )) ) {

		return FALSE;
	}
	if( !(RtlNtStatusToDosError = (void *) GetProcAddress( GetModuleHandle("ntdll.dll"),
			"RtlNtStatusToDosError" )) ) {

		return FALSE;
	}
	return TRUE;
}


//--------------------------------------------------------
//
// Main
// 
// This program drives the command loop
//
//--------------------------------------------------------
int main( int argc, char *argv[] )
{
	HANDLE		physmem;
	DWORD		vaddress, paddress, length;
	char		input[256];
	DWORD		lines;
	char		ch;
	DWORD		i, j;

	printf("\nPhysmem v1.0: physical memory viewer\n"
		   "By Mark Russinovich\n"
		   "Systems Internals - http://www.sysinternals.com\n\n");

	//
	// Load NTDLL entry points
	//
	if( !LocateNtdllEntryPoints() ) {

		printf("Unable to locate NTDLL entry points.\n\n");
		return -1;
	}

	//
	// Open physical memory
	//
	if( !(physmem = OpenPhysicalMemory())) {

		return -1;
	}

	//
	// Enter the command loop
	//
	printf("Enter values in hexadecimal. Enter 'q' to quit.\n");
	while( 1 ) {

		printf("\nAddress: " ); fflush( stdout );
		gets( input );
		if( input[0] == 'q' || input[0] == 'Q' ) break;
		sscanf( input, "%x", &paddress );

		printf("Bytes: "); fflush( stdout );
		gets( input );
		if( input[0] == 'q' || input[0] == 'Q' ) break;
		sscanf( input, "%x", &length );

		//
		// Map it
		//
		if( !MapPhysicalMemory( physmem, &paddress, &length,
								&vaddress )) 
			continue;

		//
		// Dump it
		//
		lines = 0;
		for( i = 0; i < length; i += BYTESPERLINE ) {

			printf("%08X: ", paddress + i );

			for( j = 0; j < BYTESPERLINE; j++ ) {

				if( i+j == length ) break;
				if( j == BYTESPERLINE/2 ) printf("-" );
				printf("%02X ", *(PUCHAR) (vaddress + i +j ));
			}

			for( j = 0; j < BYTESPERLINE; j++ ) {

				if( i+j == length ) break;
				ch = *(PUCHAR) (vaddress + i +j );

				if( __iscsym( ch ) || 
					isalnum( ch ) ||
					ch == ' ') {

					printf("%c", ch);

				} else {

					printf("." );
				}
			}

			printf("\n");

			if( lines++ == LINESPERSCREEN ) {

				printf("-- more -- ('q' to abort)" ); fflush(stdout);
				ch = getchar();
				if( ch == 'q' || ch == 'Q' ) {
					fflush( stdin );
					break;
				}
				lines = 0;
			}
		}

		//
		// Unmap the view
		//
		UnmapPhysicalMemory( vaddress );
	}

	//
	// Close physical memory section
	//
	CloseHandle( physmem );

	return 0;
}