/* File: <sys/wait.h> */
/* Copyright (C) 1991, 1992 Free Software Foundation, Inc.

This file is part of the GNU C Library.

The GNU C Library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public License as
published by the Free Software Foundation; either version 2 of the
License, or (at your option) any later version.

The GNU C Library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with the GNU C Library; see the file COPYING.LIB. If
not, write to the, 1992 Free Software Foundation, Inc., 675 Mass Ave,
Cambridge, MA 02139, USA.  */

/*
 *	POSIX Standard: 3.2.1 Wait for Process Termination	<sys/wait.h>
 */

/* 29-Jun-1998	J. Malmberg	Adapted for OpenVMS/VAX
** 28-Dec-1998	J. Malmberg	Updated to comply with X/OPEN
*/

#ifndef	_SYS_WAIT_H

#define	_SYS_WAIT_H	1

#ifndef _SYS_TYPES_H
#include <sys/types.h>
#endif

#include <sys/signal.h>
#include <sys/resource.h>

# ifdef __cplusplus
extern "C" {
# endif

/* This will define the `W*' macros for the flag
   bits to `waitpid', `wait3', and `wait4'.  */

#define WNOHANG		(1)
#define WUNTRACED	(2)

/* Options for waitid() */
#define WEXITED		(3)
#define WSTOPPED	(4)
#define WCONTINUED	(4)

#define WNOWAIT		(8)

/* The type idtype_t is defined as an enumeration type */
typedef enum {P_ALL, P_PID, P_PGID} idtype_t;


#ifdef	__USE_BSD

/* Lots of hair to allow traditional BSD use of `union wait'
   as well as POSIX.1 use of `int' for the status word.  */

#ifdef	__GNUC__
#define	__WAIT_INT(status)						     \
  (__extension__ ({ union { __typeof(status) __in; int __i; } __u;	      \
		 __u.__in = (status); __u.__i; }))
#else
#define	__WAIT_INT(status)	(*(int*) &(status))
#endif

/* This is the type of the argument to `wait'.
   With GCC 2, the funky union causes redeclarations with either `int *' or
   `union wait *' to be allowed. */

#if	!defined (__GNUC__) || __GNUC__ < 2 || 1 /* GCC broken as of 2.3.1 */
#define	__WAIT_STATUS	int *
#else
#define	__WAIT_STATUS
union { union wait *__unionptr; int *__intptr; }
#endif

#else /* Don't use BSD. */

#define	__WAIT_INT(status)	(status)
#define	__WAIT_STATUS		int *

#endif /* Use BSD.  */


/* If WIFEXITED(STATUS), the low-order 8 bits of the status.  */
#define	__WEXITSTATUS(status)	(((status) & 0xff00) >> 8)

/* If WIFSIGNALED(STATUS), the terminating signal.  */
#define	__WTERMSIG(status)	((status) & 0x7f)

/* If WIFSTOPPED(STATUS), the signal that stopped the child. */
#define	__WSTOPSIG(status)	__WEXITSTATUS(status)

/* Nonzero if STATUS indicates normal termination.  */
#define	__WIFEXITED(status)	(__WTERMSIG(status) == 0)

/* DEC C header says that WIFCONTINUED is not implemented */
#define __WIFCONTINUED(status) (((status) &0xFF) == 0xff)

/* Nonzero if STATUS indicates termination by a signal.  */
#ifdef	__GNUC__
#define	__WIFSIGNALED(status)					     \
  		   (__extension__ ({ int __stat = (status);		   \
		   !__WIFSTOPPED(__stat) && !__WIFEXITED(__stat); }))

#else	/* Not GCC.  */

#define	__WIFSIGNALED(status)	(!__WIFSTOPPED(status) && !__WIFEXITED(status))

#endif	/* GCC.  */

/* Nonzero if STATUS indicates the child is stopped.  */

#define	__WIFSTOPPED(status)	(((status) & 0xff) == 0x7f)

/* Nonzero if STATUS indicates the child dumped core.  */
#define	__WCOREDUMP(status)	((status) & 0200)

/* Macros for constructing status values.  */
#define	__W_EXITCODE(ret, sig)	((ret) << 8 | (sig))
#define	__W_STOPCODE(sig)	((sig) << 8 | 0x7f)



#define	WEXITSTATUS(status)	__WEXITSTATUS(__WAIT_INT(status))
#define	WTERMSIG(status)	__WTERMSIG(__WAIT_INT(status))
#define	WSTOPSIG(status)	__WSTOPSIG(__WAIT_INT(status))
#define	WIFEXITED(status)	__WIFEXITED(__WAIT_INT(status))
#define	WIFSIGNALED(status)	__WIFSIGNALED(__WAIT_INT(status))
#define	WIFSTOPPED(status)	__WIFSTOPPED(__WAIT_INT(status))
#define WIFCONTINUED(status)	__WIFCONTINUED(__WAIT_INT(status))

#ifdef	__USE_BSD
#define	WCOREDUMP(status)	__WCOREDUMP(__WAIT_INT(status))
#define	W_EXITCODE(ret,sig)	__W_EXITCODE(ret, sig)
#define	W_STOPCODE(sig)		__W_STOPCODE(sig)
#endif


/* Wait for a child to die.  When one does, put its status in *STAT_LOC
   and return its process ID.  For errors, return (pid_t) -1.  */

pid_t wait (__WAIT_STATUS __stat_loc);


#ifdef	__USE_BSD
/* Special values forthe PID argument to `waitpid' and `wait4'.  */
#define	WAIT_ANY	(-1)	/* Any process.  */
#define	WAIT_MYPGRP	0	/* Any process in myprocess group.  */
#endif

/* Wait for a child matching PID to die.
   If PID is greater than 0, match any process whose process ID is PID.
   If PID is (pid_t) -1, match any process.
   If PID is (pid_t) 0, match any process with the
   same process group as the current process.
   If PID is less than -1, match any process whose
   process group is the absolute value of PID.
   If the WNOHANG bit is set in OPTIONS, and that child
   is not already dead, return (pid_t) 0.  If successful,
   return PID and store the dead child's status in STAT_LOC.
   Return (pid_t) -1 for errors.  If the WUNTRACED bit is
   set in OPTIONS, return status for stopped children; otherwise don't.
*/
extern pid_t waitpid (pid_t __pid, int *__stat_loc,
			    int __options);

/* This being here makes the prototypes valid whether or not
   we have already included <sys/resource.h> to define `struct rusage'.  */

/* Wait for a child to exit.  When one does, put its status in *STAT_LOC and
   return its process ID.  For errors return (pid_t) -1.  If USAGE is not
   nil, store information about the child's resource usage there.  If the
   WUNTRACED bit is set in OPTIONS, return status for stopped children;
   otherwise don't.  */

pid_t wait3
       (__WAIT_STATUS __stat_loc,
	int __options,
	struct rusage * __usage);

int wait_id(idtype_t, id_t, siginfo_t *, int);

pid_t wait_pid(pid_t, int *, int);

/* PID is like waitpid.  Other args are like wait3.  */
pid_t wait4
       (pid_t __pid,
	__WAIT_STATUS __stat_loc,
	int __options,
	struct rusage *__usage);


# ifdef __cplusplus
}
# endif


#endif /* sys/wait.h  */
