/* File: <time.h>
 *
 *	VAXCRTL compatible time definitions
 *
**
** Time types.  X/OPEN specifies that there is a <sys/time.h> and a <time.h>
** with different definitions in them.
**
** On the Compaq Digital compilers, there is only one time.h file.
**
** Existing C programs written for the DEC C or VAX C compiler will not expect
** a second <time.h> file, the only way not to break these programs is to
** have each <time.h> file call the other.
**
** To simplify maintenance, both files are merged and presented here.
** Including the <sys/time.h> file only pulls the <time.h> file in.
 *
 *
 * 27-Jun-1998	J. Malmberg	Changed to include <sys/timeb.h> instead of
 *				duplicating the contents.
 *
 *		    *		Added timeval definition.  Older GNU C, DEC C,
 *				and VAC C programs may expect it here.
 *				ftime() moved to <sys/timeb.h> per X/OPEN.
 *
 * 22-Dec-1998	J. Malmberg	Added new routines for DEC C RTL
 *
 * 28-Dec-1998	J. Malmberg	Changed to include <sys/times.h> instead of
 *				duplicating the contents.
 */
#ifndef _TIME_H
#define _TIME_H

# ifdef __cplusplus
extern "C" {
# endif

#if defined(__STDC__) && !defined(_STDDEF_H)
#include <stddef.h>	/* get size_t, NULL */
#endif	/*_SIZE_T*/

#include <signal.h>	/* Just for the struct sigevent */


 /* Compatability with DEC C */
/*--------------------------*/
#include <sys/timeb.h>
#include <sys/times.h>


    /* The value of CLK_TCK is currently the same as the value of	     */
   /*  sysconf(_SC_CLK_TCK); however new applications should call	    */
  /*   sysconf() because the CLK_TCK macro may be withdrawn in the future. */
#define CLK_TCK		(100)		/* Legacy */

     /*	Although the value of CLOCKS_PER_SECOND is required to be 1 million */
    /*	on all XSI-conformant systems, it may be variable on other systems */
   /*	and it should not be assumed that CLOCKS_PER_SEC is a 		  */
  /*	compile-time constant.  Implies that the future may map this to  */
 /*	system function.						*/
#define CLOCKS_PER_SEC	(100)

#ifndef _CLOCK_T
#define _CLOCK_T
typedef long clock_t;
#endif	/*_CLOCK_T*/

#ifndef _TIME_T
#define _TIME_T
typedef unsigned long time_t;
#endif

#if !defined(_SIZE_T) && !defined(_SIZE_T_) && !defined(__SIZE_T)
typedef unsigned int size_t;
#endif				/* Used for sizes of objects */
#ifndef _SIZE_T
#define _SIZE_T
#endif
#ifndef _SIZE_T_
#define _SIZE_T_
#endif
#ifndef __SIZE_T
#define __SIZE_T 1
#endif

#ifndef _TIMER_T
#define _TIMER_T
typedef long	timer_t;	/* Used for timer ID returned by */
				/* timer_create()		 */
#endif

#ifndef _TM_T
#define _TM_T
typedef struct tm {
	int	tm_sec;
	int	tm_min;
	int	tm_hour;
	int	tm_mday;
	int	tm_mon;
	int	tm_year;
	int	tm_wday;
	int	tm_yday;
	int	tm_isdst;
} tm_t;
#endif	/*_TM_T*/

#ifndef _TIMESPEC_T_
#define _TIMESPEC_T_
typedef struct timespec {
	time_t	tv_sec;		/* seconds */
	long	tv_nsec;	/* nanoseconds */
} timespec_t;
#endif

#ifndef _ITIMERSPEC_S
#define _ITIMERSPEC_S
struct itimerspec {
	struct timespec	it_interval;	/* Timer period */
	struct timespec	it_value;	/* Timer expiration */
};
#endif


 /* The identifier of the systemwide realtime clock */
#define CLOCK_REALTIME	(1)

  /* Flag indicating time is absolute with respect to the clock associated */
 /*  with a timer							  */
#define TIMER_ABSTIME	(1)


  /*  X/OPEN requires that getdate_err be defined, guess an int	 */
 /*   DEC and VAX C rtls are not documented to use it		*/
extern int getdate_err;




/* check for alternate representation of doubles */
#ifdef __GFLOAT__
# define difftime vaxc$gdifftime
#endif

#ifdef __STDC__

char	*asctime(const struct tm *);

char	*asctime_r(const struct tm *, char *);

clock_t  clock(void);

int	 clock_getres(clockid_t, struct timespec *);	/* Not in DEC C RTL */
int	 clock_gettime(clockid_t, struct timespec *);	/* Not in DEC C RTL */
int	 clock_settime					/* Not in DEC C RTL */
	       (clockid_t, const struct timespec *);

char	*ctime(const time_t *);
char	*ctime_r(const time_t *);

double	 difftime(time_t,time_t);

struct tm *getdate(const char *);

struct tm *gmtime(const time_t *);
struct tm *gmtime_r(const time_t *);

struct tm *localtime(const time_t *);
struct tm *localtime_r(const time_t *);

time_t	 mktime(struct tm *);

int	 nanosleep					/* Not in DEC C RTL */
	       (const struct timespec *, struct timespec *);

size_t	 strftime(char *, size_t, const char *, const struct tm *);
char	 *strptime
	       (const char * __buf,
		const char * __format,
		struct tm * __timeptr);

time_t	 time(time_t *);

 /* The timer_*() functions are not documented as implemented in DEC C RTL */
int	 timer_create(clockid_t, struct sigevent *, timer_t);
int	 timer_delete(timer_t);
int	 timer_gettime(timer_t, struct itimerspec *);
int	 timer_getoverrun(timer_t);
int	 timer_settime(timer_t, int, const struct itimerspec *);


void tzset(void);

extern char *tzname[2];
extern int daylight;
extern long timezone;

#else	/*!__STDC__ => -traditional */

char	*asctime(), *ctime();
clock_t  clock();
double	 difftime();
void	 ftime(), times();
tm_t	*gmtime(), *localtime();
time_t	 mktime(), time();
unsigned strftime();

#endif	/*?__STDC__*/

 /* This is the X/OPEN <sys/times.h section */
/*=========================================*/

#ifndef __TIMERVAL
#define __TIMERVAL
struct timeval {
	time_t	tv_sec;
	long	tv_usec;
};
#endif

struct itimerval {
	struct timeval it_interval;
	struct timeval it_value;
};


#define ITIMER_REAL	(0)	/* Real */
#define ITIMER_VIRTUAL	(1)	/* process */
#define ITIMER_PROF	(2)	/* process user mode */

  /* This from the LINUX <types.h>, but X/OPEN says that this <sys/types.h> */
 /*  is the real home for the FD_ macros				   */
/*------------------------------------------------------------------------*/
#if !defined(__FD_SET) && !defined(_XOPEN_SOURCE_EXTENDED)
#define __FD_SET 1
#ifndef FD_SETSIZE
#define FD_SETSIZE 32
#endif
    typedef long fd_mask;

#define NFDBITS (sizeof(fd_mask) * (NBBY))
#define FD_SET(__n, __p)  \
	((__p)->fds_bits[(__n)/NFDBITS] |= (1 << ((__n) % NFDBITS)))
#define FD_CLR(__n, __p) \
	((__p)->fds_bits[(__n)/NFDBITS] &= ~(1 << ((__n) % NFDBITS)))
#define FD_ISSET(__n, __p) \
	((__p)->fds_bits[(__n)/NFDBITS] & (1 << ((__n) % NFDBITS)))

#   define __howmany(__x, __y)   (((__x)+((__y)-1))/(__y))

    typedef struct fd_set {                            /* descriptor set */
       fd_mask fds_bits[__howmany(FD_SETSIZE, NFDBITS)];
    } fd_set;

#    define FD_ZERO(__p)	bzero((char *)(__p),sizeof(*(__p)))

#endif /* FD_SET */

int	 getitimer(int __which, struct itimerval * __value);
int	 gettimeofday(struct timeval * __tp, void * __tzp);
int	 setitimer
	       (int __which,
		const struct itimerval * __value,
		struct itimerval * __ovalue);

int	 select
	       (int __nfds,
		int * __readfds,
		int * __writefds,
		int * __exceptfds,
		struct timeval * __timeout);

 /* utimes() is not documented to be in the DEC C RTL */
/*---------------------------------------------------*/
int	 utimes(const char *, const struct timeval [2]);


# ifdef __cplusplus
}
# endif

#endif	/*_TIME_H*/
