-----------------------------------------------------------------------------
--                                                                         --
--                             P L O O K U P                               --
--                                                                         --
--                         P L I P S   B o d y                             --
--                                                                         --
--                        $ReleaseVersion: -----  $                         --
--                                                                         --
--  Copyright (C) 1999   Mr G. A. Craig Carey, Auckland, New Zealand.      --
--                                                                         --
--   Plookup is free software; you can  redistribute  it  and/or  modify   --
--   it  under terms of the GNU  General  Public License as published by   --
--   the Free Software Foundation; either version 0, or (at your option)   --
--   any later version.   Plookup is distributed  in the  hope  that  it   --
--   will be useful, but WITHOUT ANY  WARRANTY; without even the implied   --
--   warranty of MERCHANTABILITY  or FITNESS FOR  A PARTICULAR PURPOSE.    --
--   See the GNU General Public  License  for more details.  You  should   --
--   have received a copy of the  GNU General Public License distributed   --
--   with Plookup; see   file COPYING.  If  not,  write  to  the Free      --
--   Software  Foundation, 59   Temple Place -   Suite  330,  Boston, MA   --
--   02111-1307, USA.                                                      --
--                                                                         --
--   As a special exception, if  other  files instantiate generics  from   --
--   this unit, or  you link this  unit with other  files to produce  an   --
--   executable,  this  unit does  not  by  itself cause  the  resulting   --
--   executable to be  covered by the  GNU General Public License.  This   --
--   exception does  not  however invalidate any  other reasons  why the   --
--   executable file might be covered by the GNU Public License.           --
--                                                                         --
--   The author of the software is Mr Craig Carey. The homepage of the     --
--   author is located at:                                                 --
--       http://home.clear.net.nz/pages/research/                          --
--   Dated 8 March 1999                                                    --
--                                                                         --
-----------------------------------------------------------------------------

   --  Program written by Mr Craig Carey, Auckland city, research@clear.net.nz
   --   http://home.clear.net.nz/pages/research/
   --  6 March 1999

--  pragma Source_File_Name (UNIT_NAME => PlIPs, BODY_FILE_NAME => "plips.adb");

--  pragma Normalize_Scalars;   --  Set uninitialized scalar objects out of range



with Ada.Unchecked_Deallocation;
with Ada.Exceptions;
with Ada.Strings;
with Ada.Strings.Fixed;
with Ada.Strings.Maps.Constants;

with Interfaces;
with Interfaces.C;
with Interfaces.C.Strings;

with Sockets;   --  in Sockets.Thin in Win95/NT: pragma Import (Convention=>StdCall, ..
with Sockets.Naming;

with PlPre;         use PlPre;


package body PlIPs is

   use type Ustr;

   procedure Initialize (Object : in out NameIPs_AFtype) is
   begin
      Object.Reference := null;
   end Initialize;

   procedure Adjust (Object : in out NameIPs_AFtype) is
   begin
      if Object.Reference /= null then
         Object.Reference := new NameIPs_array'(Object.Reference.all);
      end if;
   end Adjust;

   procedure Finalize (Object : in out NameIPs_AFtype) is
      procedure Deallocate is
         new Ada.Unchecked_Deallocation (NameIPs_array, NameIPs_array_Access);
   begin
      Deallocate (Object.Reference);
   end Finalize;

   --  procedure Free_IDs (X : in out NameIPs_AFtype) is
   --        new Ada.Unchecked_Deallocation (Object=>NameIPs_array, Name=>NameIPs_AFtype);



   pragma Linker_Options ("-lwsock32");


   function Clean_IP_Number (IPNum : Ustr) return Ustr is
      K     : Natural := 1;        --  Clean up 099.00.000.0
      Res   : Ustr := IPNum;
   begin
      loop
         exit when K >= SU.Length (Res);
         if ("0" = SU.Slice (Res, K, K)) and then (1=K or else "."=SU.Slice(Res,K-1,K-1))
               and then ("." /= SU.Slice(Res,K+1,K+1)) then
            SU.Delete (Res, K, K);
         else
            K := K + 1;
         end if;
      end loop;
      return Res;
   end Clean_IP_Number;



   function Is_List_of_Numbers (
               Source, Divider : String;
               UB_Nums         : Long_Integer)
             return Boolean
         -- Maximum of nine digits if Long_Integer'Last = 2**31 (=~= 2.1E9)
   is
      package S renames Ada.Strings;
      package SMC renames Ada.Strings.Maps.Constants;
      package SF renames Ada.Strings.Fixed;
      subtype Nums_Type is Long_Integer range 0 .. UB_Nums;
      Max_W     : constant Integer := Nums_Type'Width - 1;
      Max_Width : constant Integer := Integer'Min (Max_W, Long_Integer'Width - 2);
               --  S'Width is maximum length of SImage over all values of subtype S.
               --  In Win 95, max long integer = 2,147,483,648;
               --   Long_Integer'Width = 11, and that includes the leading space.
      -- pragma Assert (Max_Width <= Long_Integer'Width - 2);
      K         : Natural;
      Res       : Boolean;
   begin
      if Source = "" then
         Res := false;
      else
         K := SF.Index (Source, Divider);
         if 0 = K then
            Res := (Source'Length <= Max_Width)
                  and then (0 = SF.Index (Source, SMC.Decimal_Digit_Set, S.Outside));
            if Res then Res := (Nums_Type'Value(Source) <= UB_Nums); end if;
         else
            Res := Is_List_of_Numbers (Source (Source'First .. K-1), Divider, UB_Nums) and then
                     Is_List_of_Numbers (Source (
                           Natural'Min (K+Divider'Length, 1+Source'Last)
                           .. Source'Last), Divider, UB_Nums);
         end if;
      end if;
      return Res;
   end Is_List_of_Numbers;



   function Is_IP_Address_Old (ID : String) return Boolean is
            --  Will be deleted
      S      : Ustr := us (ID & '.');
      R1, R2 : Ustr;
      Is_IP  : Boolean := False;
      K      : Positive := 1;
   begin
      loop
         Split (S, ".", R1, R2);
         exit when S /= (R1 & "." & R2)
                  or else ("" = R1) or else (SU.Length (R1) > 3)
                  or else 0 < SU.Index (R1, SMC.Decimal_Digit_Set, Outside);
         exit when Integer'Value(fs (R1)) > 255;
         if 4 = K then  Is_IP := ("" = R2);  exit;  end if;
         K := K + 1;
         S := R2;
      end loop;
      return Is_IP;
   end Is_IP_Address_Old;



   function Is_IP_Address (Source : String) return Boolean is
      Is_IP    : Boolean;
   begin
      Is_IP := (3 = Ada.Strings.Fixed.Count (Source, ".")) and then
               Is_List_of_Numbers (Source, ".", 255);

      -------  assertions
      if Is_IP /= Is_IP_Address_Old (Source) then
         Put_Line ("! Bug alert: determined that ('" & Source & "' is an IP number) is " &
                  Boolean'Image (Is_IP));
      else
         PutL ("Is_IP(" & Source & ") = " & Boolean'Image (Is_IP));
      end if;
      -------

      return Is_IP;
   end Is_IP_Address;



   function Is_IP_Address (Source : Ustr) return Boolean is
   begin
      return Is_IP_Address (fs (Source));
   end Is_IP_Address;



   function is_No_Bad_Hostname_Chars (Source : Ustr) return Boolean is
      S    : String := fs (Source);
         --  Crashing was occurring when Chinese text was fed in as hostnames
         --  Should use RFC
   begin
      for K in S'Range loop
         if S (K) > Character'Val(126) or else S (K) <= ' ' then
            return false;
         end if;
      end loop;
	   return true;
   end is_No_Bad_Hostname_Chars;



   ---------------------------------------------------------------------------



   procedure IP_of_ID (
               Host                   : Ustr;
               Convert_Direction      : IP_Name_Convert_Direction_etype;
               pIDs                   : out NameIPs_AFtype;
               Find_Only_One_Hostname : Boolean;
               Was_Already_Converted_No_F_msg  : out Boolean)
   is

      Host_IP   : Ustr renames Host;
      Host_Name : Ustr renames Host;
      dat         : String := "0000";



      Procedure Improve_Name_of_IP (HNs : in out NameIPs_array) is


         SortTmp   : Ustr;

         procedure Move (From, To : NameIPs_Natural) is
         begin
            if 0 = From then HNs (To).id := SortTmp;
            elsif 0 = To then SortTmp := HNs (From).id;
            else HNs (To).id := HNs (From).id;
            end if;
         end Move;

         function Name_Badness (J : NameIPs_Positive) return Measure_type is
            Q       : Integer := Integer (HNs (J).quality);
            E       : String := toLowerCase (fs (HNs (J).id));
            LastDot : Natural := SF.Index (E, ".", Backward);
            Len     : Natural := E'Length;

         begin
                  --  don't want word proxy to show up in usage logs.

            Q := Q -  5 * SF.Count (E, "dial");
            if 1 = SF.Index (E, "dial") then Q := Q + 30; end if;
            Q := Q - (4 * SF.Count (E, "user") - 3 * SF.Count (E, "users"));
                                 --  E.g. users.phillynews.com, users.carle.com
            if 1 = SF.Index (E, "usr") then Q := Q + 30; end if;

            Q := Q - 50 * SF.Count (E, "prox");
            Q := Q - 40 * SF.Count (E, "cach");
            Q := Q -  7 * SF.Count (E, "ns"  );
            Q := Q -  7 * SF.Count (E, "dns" );

            Q := Q + 20 * SF.Count (E, "www.");
            Q := Q + 18 * SF.Count (E, "www ");
            if 1 = SF.Index (E, "www.") then Q := Q + 90; end if;

            Q := Q -  2 * SF.Count (E, "-"   );
            Q := Q - SF.Count (E, Set => SMC.Decimal_Digit_Set);

            if Len - 3 <= LastDot and then LastDot <= Len - 2 then Q := Q + 50; end if;
            if "" = E or 0 = SF.Index (E, ".") then Q := Q - 100_000; end if;

            return Measure_type (-Q);
         end Name_Badness;


         function Lt (J1, J2 : NameIPs_Positive) return Boolean is
         begin
            return Name_Badness(J1) < Name_Badness(J2);
         end Lt;

         package Sort_Names is new PlPre.Sort (NameIPs_Positive, Move, Lt);


         type IP_2_HN_2_nIPs_etype is (Multiple_IPs, Matching_IP, No_IP, No_Matching_IP);


         function Lookup_IP_from_Name (HostName, Match_IP : Ustr) return IP_2_HN_2_nIPs_etype is
            pIDs  : NameIPs_AFtype;
            Res   : IP_2_HN_2_nIPs_etype := No_Matching_IP;
            WACO  : Boolean;
         begin
            IP_of_ID (HostName, From_Name_to_IP, pIDs, Find_Only_One_Hostname,
                     Was_Already_Converted_No_F_msg => WACO);
            if pIDs.Reference.all'Last > 1 then
               Res := Multiple_IPs;
            elsif (pIDs.Reference.all'Last = 0)
                     or else (Null_Ustr /= pIDs.Reference.all (1).failed_msg) then
               Res := No_IP;
            elsif pIDs.Reference.all (1).id = Match_IP then
               Res := Matching_IP;
            end if;
            return Res;
         end Lookup_IP_from_Name;


         HN2IP        : IP_2_HN_2_nIPs_etype;
         Hostname_Found : Ustr;
         Null_HName   : constant Ustr := us ('"' & '"');

      begin
         for K in 1 .. HNs'Last loop
            PutL ("IP->HN: Improve_Name: IP='" & Host_IP & "', Name='" & HNs (K).id &
                     "', msg='" & HNs (K).failed_msg & "'");
            if HNs (K).id = Null_Ustr then
               HNs (K).id := Null_HName;
            end if;
         end loop;
         Sort_Names.Sort_Up(1, HNs'Last);
         for K in 1 .. HNs'Last loop
            Hostname_Found := toLowerCase (HNs (K).id);
            if Hostname_Found /= HNs (K).id then    --  uppercase characters
               HN2IP := Lookup_IP_from_Name (Hostname_Found, Host_IP);
            end if;
            if HN2IP /= Matching_IP then
               Hostname_Found := HNs (K).id;
               HN2IP := Lookup_IP_from_Name (Hostname_Found, Host_IP);
            end if;
            if HN2IP = Matching_IP then
               HNs (K).id := Hostname_Found;
               HNs (K).quality := 100_000;
               if K > 1 then  Sort_Names.Sort_Up(1, HNs'Last);  end if;
               exit;
            else
               HNs (K).id := Host_IP;

               if HN2IP = Multiple_IPs then
                     --  E.g. access.mbnet.mb.ca, algosoft.ru, cache.bora.net,
                     --   cathy.cs.keele.ac.uk, gw.nstcorp.com, ipcfs.ws.ipc.fit.ac.jp,
                     --   kummer.mathematik.hu-berlin.de, mail3.paonline.com,
                     --   proxy.nchu.edu.tw, wwwproxy.westgroup.com
                  HNs (K).failed_msg := pqIFq ("Name->(IP1,IP2..): ", Hostname_Found);

               elsif HN2IP = No_Matching_IP then
                     --  includes cases of IP->HN ok but can't HN->IP
                     --   E.g. copenet.copetel.net.ar
                  HNs (K).failed_msg := pqIFq ("IP1->HN->IP2: ", Hostname_Found);

               elsif HN2IP = No_IP then
                     --  includes cases of IP->HN ok but can't HN->IP
                     --   E.g. copenet.copetel.net.ar
                  HNs (K).failed_msg := pqIFq ("Can't Name->IP: ", Hostname_Found);
               end if;
            end if;
         end loop;
      end Improve_Name_of_IP;


      package SN renames Sockets.Naming;
      use type SN.String_Access;


      procedure PutL_Hostent (Hostent : SN.Host_Entry) is
      begin
         if not dbg then return; end if;
         if Hostent.Name = null then
            PutL ("Hostent of '" & Host_IP & "': Name=NULL !");
         else
            PutL ("Hostent of '" & Host_IP & "': Name='" & Hostent.Name.all & "'");
         end if;
         for J in Hostent.Aliases'Range loop
            if Hostent.Aliases (J) = null then
                  PutL ("Hostent of '" & Host_IP & "': Aliases (" & Natural'Image (J) & ")=NULL !");
            else
                  PutL ("Hostent of '" & Host_IP & "': Aliases (" & Natural'Image (J) & ")='" &
                           Hostent.Aliases (J).all & "'");
            end if;
         end loop;
         for J in Hostent.Addresses'Range loop
               PutL ("Hostent of '" & Host_IP & "': Addresses (" & Natural'Image (J) & ")='" &
                        SN.Image (Hostent.Addresses (J)) & "'");
         end loop;
      exception
         when others => PutL("!! PutL_Hostent debug routine died !!!!!!");
      end PutL_Hostent;


      procedure One_ID (ID : Ustr; failed_msg : String) is
      begin
         pIDs.Reference := new NameIPs_array'(1 => (
               id => ID, quality => No_quality, failed_msg => us (failed_msg)));
      end One_ID;


      Cant_Die   : Boolean := false;

   begin
      Was_Already_Converted_No_F_msg := false;

      if Convert_Direction = From_Name_to_IP then  --  to IP or Proxy Hunter format
         if "" = Host_Name or Is_IP_Address (Host_Name) then
            One_ID (Host_Name, "");   --  No hostnames are IP#s so must be an IP# already
            Was_Already_Converted_No_F_msg := true;

         elsif not is_No_Bad_Hostname_Chars (Host_Name) then
            One_ID (Host_Name, "Bad character(s)");

         else
            begin
               declare
                  Hostent : SN.Host_Entry := SN.Info_Of (Name => fs (Host_Name));
               begin
                  PutL_Hostent (Hostent);
                  if Hostent.Addresses'Length < 1 then
                     One_ID (Host_Name, "Can't Name->IP");
                  else
                     pIDs.Reference := new NameIPs_array (1 .. Hostent.Addresses'Length);
                     for J in Hostent.Addresses'Range loop
                        pIDs.Reference.all (NameIPs_Natural (J)) :=
                                 (id => us (SN.Image (Hostent.Addresses (J))),
                                 quality => No_quality, failed_msg => Null_Ustr);
                     end loop;
                  end if;
               end;
            exception
               when X : others =>
                  declare
                     EM  : String := Ada.Exceptions.Exception_Message (X);
                                 --  E.g. EM =  "Host not found: jkl.com"
                  begin
                     PutL ("HN->IP exception: dat= " & dat & "Msg = '" & EM & "'");
                     One_ID (host, ("Failed: " * (1 /= SF.Index(EM, "Host not found"))) & EM);
                  end;
            end;
         end if;

      elsif Convert_Direction = From_IP then  --  to Name
         if "" = Host_Name or not Is_IP_Address (Host_IP) then
            One_ID (Host_IP, "");   --  Not an IP#, don't check it
            Was_Already_Converted_No_F_msg := true;

         elsif not is_No_Bad_Hostname_Chars (Host_Name) then
            One_ID (Host_IP, "Bad character(s)");

         else
            begin
               declare
                  Hostent : SN.Host_Entry := SN.Info_Of (Addr => SN.Value (fs (Host_IP)));
                  Aliases : SN.String_Array renames Hostent.Aliases;
               begin
                  PutL_Hostent (Hostent);
                  if Hostent.Name = null or else "" = Hostent.Name.all then
                     One_ID (Host_IP, "NSLookup failed");
                  else
                     pIDs.Reference := new NameIPs_array (1 .. 1+Aliases'Length);
                     pIDs.Reference.all (1).id := us (Hostent.Name.all);
                     for J in Aliases'Range loop
                        pIDs.Reference.all (NameIPs_Natural (2+J-Aliases'First)).id :=
                                 us (Aliases (J).all);
                        SN.Free (Aliases (J));
                     end loop;
                     Cant_Die := true;
                     Improve_Name_of_IP(pIDs.Reference.all);  -- sorts best to first
                  end if;
               end;
            exception
               when X : others =>
                  declare
                     EM   : String := Ada.Exceptions.Exception_Message (X);
                  begin
                     PutL ("IP->HN Exception: '" & EM & "', dat= " & dat);
                     if Cant_Die then
                        One_ID (Host_IP, "Error when doing Name->IP: " & EM);
                     elsif 1 <= SF.Index (EM, "Unknown error 0:") or else "" = EM then
                        One_ID (Host_IP, "NSLookup failed");
                     else
                        One_ID (Host_IP, EM);
                     end if;
                  end;
            end;
         end if;
      end if;
   end IP_of_ID;


--
--     Cache_Size   : Positive := 2;
--
--     type Cache_rec is record
--        Valid          : Boolean := false;
--        Host           : Ustr;
--        Convert_Direction : IP_Name_Convert_Direction_etype;
--        pIDs           : NameIPs_AFtype;
--     end record;
--     type Data_Positive is new Positive range 1 .. Cache_Size;
--     type Data_array is array (Data_Positive) of Cache_rec;
--
--
--
--     protected Cache is
--
--        procedure Query_Cache (
--                    Host           : Ustr;
--                    Convert_Direction : IP_Name_Convert_Direction_etype;
--                    Found          : out Boolean;
--                    pIDs           : out NameIPs_AFtype);
--        procedure Put_Into_Cache (
--                    Host           : Ustr;
--                    Convert_Direction : IP_Name_Convert_Direction_etype;
--                    pIDs           : NameIPs_AFtype);
--
--     private
--
--        Data         : Data_array;
--        Beginning    : Data_Positive := 1;
--
--     end Cache;
--
--
--     protected body Cache is
--
--        procedure Query_Cache (
--                    Host           : Ustr;
--                    Convert_Direction : IP_Name_Convert_Direction_etype;
--                    Found          : out Boolean;
--                    pIDs           : out NameIPs_AFtype) is
--        begin
--           Found := False;
--           for K in Data'Range loop
--              declare
--                 D : Cache_rec renames Data (K);
--              begin
--                 if D.Valid
--                       and then (D.Host = Host)
--                       and then (D.Convert_Direction = Convert_Direction)
--                 then
--                    Found := True;
--                    pIDs := D.pIDs;
--                    goto DoneL;
--                 end if;
--              end;
--           end loop;
--           <<DoneL>> null;
--        end Query_Cache;
--
--        procedure Put_Into_Cache (
--                    Host           : Ustr;
--                    Convert_Direction : IP_Name_Convert_Direction_etype;
--                    pIDs           : NameIPs_AFtype) is
--        begin
--           for H in 0 .. (Data'Last - 1) loop
--              declare
--                 D : Cache_rec renames
--                       Data (Data_Positive (1 + (Beginning+H) mod Data'Last ));
--              begin
--                 D.Valid := true;
--                 D.Host := Host;
--                 D.Convert_Direction := Convert_Direction;
--                 D.pIDs := pIDs;       --  pointers
--              end;
--           end loop;
--           Beginning := 1 + Beginning mod Data'Last;
--        end Put_Into_Cache;
--
--     end Cache;
--
--
--     Random_Num  : Integer := 1;
--
--
--     procedure IP_of_ID (
--                 Host              : Ustr;
--                 Convert_Direction : IP_Name_Convert_Direction_etype;
--                 pIDs              : out NameIPs_AFtype;
--                 Find_Only_One_Hostname : Boolean)
--     is
--        Found       : Boolean;
--        Check       : Boolean;
--        Z_pIDs      : NameIPs_AFtype;
--
--     begin
--        Cache.Query_Cache (Host, Convert_Direction, Found, pIDs);
--        Random_Num := (Random_Num + 17) mod 63;
--        Check := (Random_Num < 19) and Found;
--        if not Found or Check then
--           if Check then
--              Z_pIDs := pIDs;
--           end if;
--           IP_of_ID_2 (Host, Convert_Direction, pIDs, Find_Only_One_Hostname);
--           if Check then
--              if pIDs.Reference.all'Last /= Z_pIDs.Reference.all'Last then  goto Fail; end if;
--              for K in pIDs.Reference.all'Range loop
--                 if pIDs.Reference.all (K).id /= Z_pIDs.Reference.all (K).id then  goto Fail; end if;
--                 if pIDs.Reference.all (K).quality /= Z_pIDs.Reference.all (K).quality then  goto Fail; end if;
--                 if pIDs.Reference.all (K).failed_msg /= Z_pIDs.Reference.all (K).failed_msg then  goto Fail; end if;
--              end loop;
--              goto Pass;
--              <<Fail>>
--              Put_Line (PH_LHS_comment & "! Cached data mismatched: host/id = '" & Host & "'");
--              <<Pass>> null;
--           end if;
--           Cache.Put_Into_Cache (Host, Convert_Direction, pIDs);
--        end if;
--     end IP_of_ID;
--
--


   ---------------------------------------------------------------------------

   package body PHSocket is

      --  use Interfaces;
      --  use Interfaces.C;
      --  use Interfaces.C.Strings;

      ----------------------------------------------------------------
      Sockets_Version : constant Interfaces.unsigned_16 := 16#0101#;

      ----------------------------------------------------------------
      WSAEINTR_VALUE           : constant Integer := 10004;
      WSAEFAULT_VALUE          : constant Integer := 10014;
      WSAEINVAL_VALUE          : constant Integer := 10022;
      WSANO_DATA_VALUE         : constant Integer := 11004;
      WSAENETDOWN_VALUE        : constant Integer := 10050;
      WSATRY_AGAIN_VALUE       : constant Integer := 11002;
      WSAEINPROGRESS_VALUE     : constant Integer := 10036;
      WSASYSNOTREADY_VALUE     : constant Integer := 10091;
      WSANO_RECOVERY_VALUE     : constant Integer := 11003;
      WSANOTINITIALISED_VALUE  : constant Integer := 10093;
      WSAHOST_NOT_FOUND_VALUE  : constant Integer := 11001;
      WSAVERNOTSUPPORTED_VALUE : constant Integer := 10092;

      function WSAGetLastError return Integer;
      pragma Import (StdCall, WSAGetLastError, "WSAGetLastError");

      procedure Check_Result (Result : in Integer) is
      begin
         if Result /= 0 then
            case Result is
               when WSAEINTR_VALUE            => raise WSAEINTR;
               when WSAEINVAL_VALUE           => raise WSAEINVAL;
               when WSAEFAULT_VALUE           => raise WSAEFAULT;
               when WSANO_DATA_VALUE          => raise WSANO_DATA;
               when WSAENETDOWN_VALUE         => raise WSAENETDOWN;
               when WSATRY_AGAIN_VALUE        => raise WSATRY_AGAIN;
               when WSAEINPROGRESS_VALUE      => raise WSAEINPROGRESS;
               when WSASYSNOTREADY_VALUE      => raise WSASYSNOTREADY;
               when WSANO_RECOVERY_VALUE      => raise WSANO_RECOVERY;
               when WSANOTINITIALISED_VALUE   => raise WSANOTINITIALISED;
               when WSAHOST_NOT_FOUND_VALUE   => raise WSAHOST_NOT_FOUND;
               when WSAVERNOTSUPPORTED_VALUE  => raise WSAVERNOTSUPPORTED;
               when others                    => raise UNKNOWN_SOCKETS_ERROR;
            end case;
         end if;
      end Check_Result;
      pragma Inline (Check_Result);

      ----------------------------------------------------------------
      WSASYS_STATUS_LEN  : constant := 128;
      WSADESCRIPTION_LEN : constant := 256;

      use type Interfaces.C.size_t;

      type WSADATA is
         record
            wVersion       : Interfaces.unsigned_16;
            wHighVersion   : Interfaces.unsigned_16;
            szDescription  : Interfaces.C.char_array (1 .. WSADESCRIPTION_LEN+1);
            szSystemStatus : Interfaces.C.char_array (1 .. WSASYS_STATUS_LEN+1);
            iMaxSockets    : Interfaces.unsigned_16;
            iMaxUdpDg      : Interfaces.unsigned_16;
            lpVendorInfo   : Interfaces.C.Strings.chars_ptr;
         end record;
      pragma Convention (C, WSADATA);

      type LPWSADATA is access all WSADATA;
      pragma Convention (C, LPWSADATA);

      procedure Free is new Ada.Unchecked_Deallocation (WSADATA, LPWSADATA);

      function WSAStartup (VersionRequested : Interfaces.unsigned_16;
                      WSAData          : LPWSADATA) return Integer;
      pragma Import (StdCall, WSAStartup, "WSAStartup");

      function WSACleanup return Integer;
      pragma Import (StdCall, WSACleanup, "WSACleanup");

   --          ----------------------------------------------------------------
   --          function gethostname (name : Interfaces.C.char_array; length : size_t) return Integer;
   --          pragma Import (StdCall, gethostname, "gethostname");
   --
   --          ----------------------------------------------------------------
   --          type IN_ADDR is new unsigned_32;
   --
   --          type IN_ADDR_Ptr is access all IN_ADDR;
   --          pragma Convention (C, IN_ADDR_Ptr);
   --
   --          type IN_ADDR_Ptr_Ptr is access all IN_ADDR_Ptr;
   --          pragma Convention (C, IN_ADDR_Ptr_Ptr);
   --
   --          type chars_ptr_ptr is access all Interfaces.C.Strings.chars_ptr;
   --          pragma Convention (C, Interfaces.C.Strings.chars_ptr_ptr);
   --
   --          type HOSTENT is
   --              record
   --                  h_name      : Interfaces.C.Strings.chars_ptr;
   --                  h_aliases   : Interfaces.C.Strings.chars_ptr_ptr;
   --                  h_addrtype  : Interfaces.unsigned_16;
   --                  h_length    : Interfaces.unsigned_16;
   --                  h_addr_list : IN_ADDR_Ptr_Ptr;
   --              end record;
   --          pragma Convention (C, HOSTENT);
   --
   --          type LPHOSTENT is access all HOSTENT;
   --          pragma Convention (C, LPHOSTENT);
   --
   --          function gethostbyname (Name : Interfaces.C.char_array) return LPHOSTENT;
   --          pragma Import (StdCall, gethostbyname, "gethostbyname");
   --
   --          function inet_ntoa (Addr : IN_ADDR) return Interfaces.C.Strings.chars_ptr;
   --          pragma Import (StdCall, inet_ntoa, "inet_ntoa");

      ----------------------------------------------------------------
      procedure Start_Sockets is
         Result   : Integer;
         Data_Ptr : LPWSADATA    := new WSADATA;
         Version  : Interfaces.unsigned_16 := Sockets_Version;
         use type Interfaces.Unsigned_16;
      begin
         Result := WSAStartup (Version, Data_Ptr);
         Free (Data_Ptr);
         Check_Result (Result);
         if Version /= Sockets_Version then
            Result := WSACleanup;
            raise WSAVERNOTSUPPORTED;
         end if;
      end Start_Sockets;

      ----------------------------------------------------------------
      procedure Finishup_Sockets is
         Result : Integer;
      begin
         Result := WSACleanup;
         Check_Result (WSAGetLastError);
      end Finishup_Sockets;

      ----------------------------------------------------------------

   --          function Get_Host_Name return String is
   --              Result        : Integer;
   --              Temp_Length : constant size_t := 256;
   --              Temp_Name    : Interfaces.C.char_array (1 .. Temp_Length) := (others => ' ');
   --          begin
   --              Result := gethostname (Temp_Name, Temp_Length);
   --              Check_Result (WSAGetLastError);
   --              return To_Ada (Temp_Name);
   --          end Get_Host_Name;
   --
   --          ----------------------------------------------------------------
   --          function Get_Host_Address return String is
   --              Data : LPHOSTENT;
   --          begin
   --              Data := gethostbyname (To_C (Get_Host_Name));
   --              return Value (inet_ntoa (Data.h_addr_list.all.all));
   --          end Get_Host_Address;

   end PHSocket;



end PlIPs;