-----------------------------------------------------------------------------
--                                                                         --
--                             P L O O K U P                               --
--                                                                         --
--                                B o d y                                  --
--                                                                         --
--          Release: Code 11-Apr-99.    Comments updated: 21-Apr-99        --
--     Special exceptions to deactivate licence are quite obtainable       --
--                                                                         --
--  Copyright (C) 1999   Mr G. A. Craig Carey, Auckland, New Zealand.      --
--                                                                         --
--   Plookup is free software; you can  redistribute  it  and/or  modify   --
--   it  under terms of the GNU  General  Public License as published by   --
--   the Free Software Foundation; either version 0, or (at your option)   --
--   any later version.   Plookup is distributed  in the  hope  that  it   --
--   will be useful, but WITHOUT ANY  WARRANTY; without even the implied   --
--   warranty of MERCHANTABILITY  or FITNESS FOR  A PARTICULAR PURPOSE.    --
--   See the GNU General Public  License  for more details.  You  should   --
--   have received a copy of the  GNU General Public License distributed   --
--   with Plookup; see   file COPYING.  If  not,  write  to  the Free      --
--   Software  Foundation, 59   Temple Place -   Suite  330,  Boston, MA   --
--   02111-1307, USA. (Mar 1999, gnu@gnu.org)                              --
--                                                                         --
--   As a special exception, if  other  files instantiate generics  from   --
--   this unit, or  you link this  unit with other  files to produce  an   --
--   executable,  this  unit does  not  by  itself cause  the  resulting   --
--   executable to be  covered by the  GNU General Public License.  This   --
--   exception does  not  however invalidate any  other reasons  why the   --
--   executable file might be covered by the GNU Public License.           --
--                                                                         --
--   The author of this software is Mr Craig Carey.                        --
--   The homepage of the author is at:                                     --
--       http://home.clear.net.nz/pages/research/info/snooz.htm            --
--   Dated: 11 April 1999                                                   --
--   Auckland, New Zealand: phone NZ-09-8283959 (April 1999)               --
--                                                                         --
--   The program was designed to help persons who search or and publish    --
--    lists of free public proxies. They might use the Proxy Hunter        --
--    program.                                                             --
--    That program only knows about IP numbers and not about hostnames.    --
--    The homepage of Proxy Hunter (for Win 95) is at                      --
--        http://member.netease.com/~windzh/software/proxyht/intro.htm )      --
--    This program was written after to eliminate the problem of           --
--     calling a single lookup program using Cygwin xargs and transferring --
--     port numbers & comments & ":X"s in the presence of ill              --
--     formatted output & aliases (& lookup failures), and GNU's join.     --
--   It does ordinary NS lookups, IP#<->Name, also.                        --
--                                                                         --
--   Although no enhancement requests seem likely ever presumably, the     --
--    author expects that any not too difficult requests for improvement   --
--    would be done. Some bugs could be Ada Core Technology bugs.          --
--                                                                         --
--   The program can be made to run in Unix/Linux/etc.                     --
--   All lines of the form                                                 --
--       'pragma Import (StdCall, *, "*");'                                --
--    would be altered to                                                  --
--       'pragma Import (C, *, "*");'                                      --
--                                                                         --
-----------------------------------------------------------------------------


--  This program shows how to process in parallel lines read from a file,
--   with the results for each line being output being in the same order
--   present in the input file, and with the printing (to STDOUT) having
--   background priority (unless the quantity of results queued to be
--   printed became too large).

--  The program reads in lines from a file, and puts them into a linked list.
--  It creates an array of the correct length (Lines_pkg.Lines) and it
--  copies the input data into that. An array of tasks does DNS lookups.
--  A protected object is used since:
--   (1) Ada's 'when' blocking is required. That construct is only allowed
--       inside protected objects. The alternative of slowing down execution
--       with select and delay statements was being avoided as the program
--       was not intended to make assumptions about the speed of the machine
--       it was running on, and nor was a complex algorithm desired just to
--       correct an initial bad assumption about the machine's speed.
--   (2) The variable Lines_pkg.Lines_Server.Read_UpTo had to be protected.
--       That variable holds holds information on how much of the array of
--       names (or IP numbers) had alredy been processed.
--  A printer task does printing in the background.
--  Ada 95 priorities (a pragma) were tried. A negative priority stalls the
--   program and a positive priority led to the program taking definitely
--   too much of the CPU (in Windows 95), and anyway, setting the priority
--   of DNS lookup tasks to be higher than the priority of the printer task
--   resulted in the printer task still using 100% of the CPU without any
--   DNS lookup work occurring.
--  So prioritization is explicity done with the 'when' blocking expressions.
--   with conditions on entry routines that tend to alternate from
--   (true,false) to (false,true). An aim is to not have the program do too
--   many lookups without printing if it might be about to be stopped with a
--   Ctrl-C.

--  Sockets in Windows 95 must be initialised (unlike in Unix).

--  Use of Pragma Inline (on PutL) with a generic procedure & block declare
--   of an array caused a not-simple GNAT bug to show up: the code at
--   "Inline bug 1" was not being executed.

-----------------------------------------------------------------------------
-- Compiled with Gnat 3.11p:
-- gnatmake <name> -O3 -gnato -m -v -gnatl -gnatf -gnatv -gnata -gnatq
--    -gnatr -gnatwl -gnatu -largs -v >@.lst

--  Possible bugs, matters to investigate

-- Bug 22 Jul 99:
--
--    C:\1>plookup 417137774
--    417137774 { Host not found
--
--    C:\1>lookup 417137774
--    417137774's official name is h-004-110.phoenix.speedchoice.com.
--    |PK: Is_IP(417137774) = FALSE
--    |PK: HN->IP exception: dat= 0000Msg = 'Host not found: 417137774'
--
--
--
-- 11-Jul-99 Fixes for not transferring the ":A" notes, and "-aa" added.
-- 20June99: Bug1:
-- plookup -i -a on this: 209.151.160.41:8080 { Name->(IP1,IP2..): fastweb.clover.net
--  causes the name to be lost.
-- BUG2: plookup -p 1.2.3.4:80XY vs -p 1.2.3.4:8081XY

-- AdaPower: http://www.adapower.com/adacode.html

-- Bug? : On this data:
--  209.20.42.6:80@HTTP
--  209.128.34.1:80@HTTP
--  142.217.192.3:80@HTTP
-- Plookup [ ploo-100 -dbg -t 3 -n1 -f !a-$.txt ] produced ":80" on the 3rd
--    line (missing hostname). Might be because an old version was used.





--  1.0.0.0:80@HTTP [ lpsa.usr.ofijaen.com          -- Not reproduced
--  1.0.0.0:80@HTTP [ seisinfo.usr.ciberia.es
--  1.0.0.0:80@HTTP [ vistamar.usr.axarnet.com
-- E:plookup -dbg -n -a 203.137.6.194:3128@HTTP     -- Fixed. Inlines disabled.
--  "196.27.0.30:80", ph->name     -- Nameserver returned null str

------------------------------------------------------------------------------
--  Program By Mr Craig Carey, Auckland city, research@ijs.co.nz

--  Reference manual: ftp://ftp.cdrom.com/pub/ada/ajpo/standards/95lrm/LRMzip/

--  pragma Source_File_Name (UNIT_NAME => plookup, BODY_FILE_NAME => "plookup.adb");

--  pragma Normalize_Scalars;   --  Set uninitialized scalar objects out of range

--  In Gnat 3.11p, pragma Normalize_Scalars interacted with tasking
--  to cause the link time error message
--     "undefined reference to `system__tasking___init_proc$13'

with Ada.Unchecked_Deallocation;
with Ada.Command_Line;
-- with System;            --  for setting priorities

   --  with System.Assertions;
   --  Assert_Failure : exception;
   --        --  Exception raised when assertion fails
   --  procedure Raise_Assert_Failure (Msg : String);
   --        --  Called to raise Assert_Failure with given message.
   --        --   like pragma Assert (Test : Boolean[; Msg : String]);

with PlPre;    use PlPre;
with PlIPs;    use PlIPs;




procedure plookup is
--  =============


   --  pragma Linker_Options ("-Xlinker --stack=0x20000,0x1000"); -- GNAT
   --  pragma Linker_Options ("-Xlinker --stack=0x200000,0x10000"); -- GNAT (for big arrays)
   pragma Linker_Options ("-Xlinker --stack=0x100"); -- GNAT (for >60-ish tasks)

      --  Must increase number of threads/tasks


   use type Ustr;     --  Make visible operators "=", "&" (L:String; R:Ustr).

   type Results_array is array (Natural range <>) of Ustr;  --  E.g. (1=>"102.2.3.4")
   type Results_array_access is access Results_array;

   procedure Free_Results_array is new Ada.Unchecked_Deallocation (
               Results_array, Results_array_access);

   type Task_Number is new Natural;
   type Line_Number is new L_Natural;

   function LImg (U : Line_Number) return String renames Line_Number'Image;
   function TImg (U : Task_Number) return String renames Task_Number'Image;

   type Line_rec is record
      Text            : Ustr := Null_Ustr;
      Contains_Result : Boolean := False;
      Results         : Results_array_access;
   end record;

   type Lines_array is array (Line_Number range <>) of Line_rec;

   type Lines_array_access is access Lines_array;

   procedure Free_Lines_array is new Ada.Unchecked_Deallocation (
                  Lines_array, Lines_array_access);


   -------------------------------------------------------------------


   generic                --  'generic' to get the Lines array inside the protected object
      Num_Lines   : Line_Number;

   package Lines_pkg is


         --  This package does only just this: It prints the lines found by the
         --   array of DNS-lookup worker tasks; and these aspects hold:
         --   (A) the findings are printed in the same order that occurs in the
         --   input file. [The output is buffered into an array].
         --   (B) Printing doesn't delay worker tasks from getting the next task.
         --   [The load (in KB/sec) on the DNS server is kept at about the
         --   maximum; and why not?.]


      protected Lines_Server is

         procedure Assign (
                     Line_Num     : Line_Number;
                     Text         : Ustr);

         entry Get_Next (
                     Line_Num     : out Line_Number; --  = 0 if end
                     Line         : out Ustr);

         entry Put_Data (
                     Task_Num     : Task_Number;
                     Line_Num     : Line_Number;
                     Results      : Results_array);

         entry Get_Slice_LB (Can_Print_UpTo_Copy_2 : out Line_Number);

         entry Block_Printing_and_Force_ReCheck_of_Get_Next_Barrier (
                     Printed_UpTo_Copy_val : Line_Number);

      private
                                 --  "_UpTo" indicates 'done up to and including'
         Read_UpTo      : Line_Number := 0;
         Can_Print_UpTo : Line_Number := 0;
         Printed_UpTo_Copy : Line_Number := 0;
      end Lines_Server;


      task Printer_Task is

         entry Start;

      end Printer_Task;


      procedure Finalize;


   end Lines_pkg;



   package body Lines_pkg is


      Lines          : Lines_array_access := new Lines_array (1 .. Num_Lines);
                                 --  Allows outputting in the input-file order.

               --  Put_Data () must be called for every element of the
               --   Lines array. If it is not called for the first element, then
               --   Printing will not occur or will not complete (14-Aug-99).

      procedure Finalize is
      begin
         null;
         --  Free_Lines_Array (Lines);
         --    Not called. Need code to ensure Printer_Task does not attempt to access Lines.
      end Finalize;


      type Lines_Bool_array is array (1 .. Num_Lines) of Boolean;
      Was_Printed    : Lines_Bool_array := (others => False); --  check no bugs

      Printed_UpTo   : Line_Number := 0;

      protected body Lines_Server is

         function Must_Slow_Down return Boolean is
            Max_Printing_Lag : constant Line_Number := 500;
            type R is digits 9;
            Print_Lag_F      : constant R := 0.75;
            Print_Lag_G      : constant R := 1.0 - Print_Lag_F;
         begin
            return (R (Max_Printing_Lag + Printed_UpTo_Copy)
                     <= Print_Lag_F * R (Can_Print_UpTo) + Print_Lag_G * R (Read_UpTo));
         end Must_Slow_Down;


         procedure Assign (
                     Line_Num     : Line_Number;
                     Text         : Ustr) is
         begin
            Lines (Line_Num) := (Text => Text, Contains_Result => false,
                     Results => null);
         end Assign;


         entry Get_Next (
                     Line_Num     : out Line_Number; --  = 0 if end
                     Line         : out Ustr)
            when not Must_Slow_Down
                              --  Without an entry barrier, DNS lookups may get too
                              --   too far ahead & load on CPU too high.
         is
         begin
            if Read_UpTo >= Num_Lines then
               Line_Num := 0;             --  finished
               Line := Null_Ustr;
            else
               Read_UpTo := Read_UpTo + 1;
               Line_Num := Read_UpTo;
               Line := Lines (Line_Num).Text;
            end if;

         end Get_Next;


         entry Put_Data (
                     Task_Num     : Task_Number;
                     Line_Num     : Line_Number;
                     Results      : Results_array)
            when true
         is

            procedure PutL2 (S : String) is
            begin
               PlPre.PutL ("--PD" & SF.Trim (TImg (Task_Num), Left) & S);
            end PutL2;
            cp    : constant Line_Number := Can_Print_UpTo;

         begin
            if 0 <= Line_Num and Line_Num <= Num_Lines then   --  jic
               Lines (Line_Num).Results := new Results_array (1 .. Results'Last);
               Lines (Line_Num).Results.all := Results;
               Lines (Line_Num).Contains_Result := True;

               for K in Can_Print_UpTo + 1 .. Num_Lines loop
                  exit when not Lines (K).Contains_Result;
                  Can_Print_UpTo := K;            --  allow Printer task to print
               end loop;
            end if;

            if dbg then
               declare
                  ss         : Ustr := Null_Ustr;
                  p1, p2, p3 : Line_Number;
               begin
                  p1 := Line_Number'Max (1, (Can_Print_UpTo - 15));
                  p2 := Line_Number'Min (Num_Lines, (Can_Print_UpTo + 15));
                  for K in p1 .. p2 loop
                     if Lines (K).Contains_Result then ss := ss & "1";
                     else ss := ss & "0"; end if;
                  end loop;
                  PutL2 (LImg (Line_Num) & ": (" &
                           LImg (cp) & "->" & LImg (Can_Print_UpTo) & ") " &
                           LImg (p1) & " [" & fs (ss) & "]" & LImg (p2) );
               end;
            end if;

         exception
            when X : others =>
               Raise_Msg ("Put_Data ended: Task # =" & TImg (Task_Num), X);
         end Put_Data;


                     --  9.5.3(29) mentions forcing update of entry barriers exprs

         entry Get_Slice_LB (Can_Print_UpTo_Copy_2 : out Line_Number)
                                                    --  the entry barrier expression
            when (Printed_UpTo_Copy < Can_Print_UpTo) or else
                        (Can_Print_UpTo >= Num_Lines)
         is
         begin
            Can_Print_UpTo_Copy_2  := Can_Print_UpTo;
         end Get_Slice_LB;


         entry Block_Printing_and_Force_ReCheck_of_Get_Next_Barrier (Printed_UpTo_Copy_val : Line_Number)

            when (Put_Data'Count = 0) and then (Get_Next'Count = 0 or else Must_Slow_Down)

                  --  In GNAT, this routine might not actually update the Get_Data entry
                  --   barrier, but it does block the printer task fairly well.
                  --  (So, either Get_Data runs or the program stalls, and it doesn't
                  --   seem to stall.)
                  --  (Ordering entries or using priority pragmas, both allowed the
                  --   printer task to seize CPU for a long time (at least in GNAT
                  --   3.11p in Windows 95).)
         is
         begin
            Printed_UpTo_Copy := Printed_UpTo_Copy_val;
         end Block_Printing_and_Force_ReCheck_of_Get_Next_Barrier;


      end Lines_Server;


            --  (Max of 59 tasks in Windows 95 (NT) (programs stops if more used). Sad,
            --   and fixable somehow. How?.)

      task body Printer_Task is

         Buffer_Flush_Size_Bytes : constant Positive := 512;
         FB     : Long_Integer := 0;

         procedure Print_Slice (Can_Print_UpTo_Copy_2 : Line_Number) is
         begin
            PutL ("Print_Slice Start: " & LImg (Can_Print_UpTo_Copy_2 - Printed_UpTo) &
                  "  (" & SF.Trim (LImg (Printed_UpTo), Left) & " .. " &
                  LImg (Can_Print_UpTo_Copy_2) & ") ...");

            for K in Printed_UpTo + 1 .. Can_Print_UpTo_Copy_2 loop
               --  if dbg then for J in Lines (K).Results.all'Range loop
               --         PutL ("Print_Slice   ---: " & LImg (K) &
               --               Integer'Image (J) & "'" & Lines (K).Results.all (J) & "'"); end loop;  end if;
               for J in Lines (K).Results.all'Range loop
                  declare
                     S : Ustr renames Lines (K).Results.all (J);
                  begin
                     if Null_Ustr /= S then   --  "Inline bug 1"
                        Put_Line (S);
                        S := Null_Ustr;
                        FB := FB + Long_Integer (SU.Length (S));
                     end if;
                  end;
                  if FB >= 1024 then
                     FB := 0;
                     tio.Flush;
                  end if;
               end loop;
               Was_Printed (K) := True;
               Printed_UpTo := K;
               Lines_Server.Block_Printing_and_Force_ReCheck_of_Get_Next_Barrier (
                        Printed_UpTo_Copy_val => Printed_Upto);
            end loop;
            --  PutL ("Print_Slice: slice printed");

         end Print_Slice;

         Can_Print_UpTo_Copy_2 : Line_Number;

      begin
         Printed_UpTo := 0;
         PutL ("Lines_Server.Block_Printing_and_Force_ReCheck_of_Get_Next_Barrier (Printed_UpTo_Copy_val => 0) ...");
         Lines_Server.Block_Printing_and_Force_ReCheck_of_Get_Next_Barrier (
                  Printed_UpTo_Copy_val => 0);

         accept Start;
         tio.Flush;
         loop
            PutL (" Printer: Waiting for lines ...");

            Lines_Server.Get_Slice_LB (Can_Print_UpTo_Copy_2);

            PutL (" Printer: Slice obtained; Can_Print_UpTo_Copy_2 =" & LImg (Can_Print_UpTo_Copy_2) &
                     ".  Slice is ...");

            Print_Slice (Can_Print_UpTo_Copy_2);

            exit when Printed_UpTo >= Num_Lines;
         end loop;
         tio.Flush;
         for K in Was_Printed'Range loop
            if not Was_Printed (K) then
               Put_Line ("! ERROR: output for some lines was not printed");  tio.Flush;
            end if;
            for J in Lines (K).Results.all'Range loop
               if Null_Ustr /= Lines (K).Results.all (J) then
                     -- Was being executed due to a GNAT 3.11p pragma Inline bug
                     --  which was adequately reported on 8-Apr-99.
                  Put_Line ("! ERROR: output for result" & Integer'Image (J) &
                        " of line" & LImg (K) &
                        " '" & Lines (K).Text & "' was not printed");
                  tio.Flush;
               end if;
            end loop;
         end loop;

         PutL ("Printer: ending");

      exception
         when X : others =>
            Raise_Msg ("! Task Print_Task stopped", X);
      end Printer_Task;


   end Lines_pkg;



   --  =======================================================================



   procedure Ltrim_Zeroes (Source : in out Ustr) is
   begin
      if Source /= Null_Ustr and then SU.Element (Source, 1) = '0' then
         SU.Delete (Source, 1, 1);
         LTrim_Zeroes (Source);
      end if;
   end Ltrim_Zeroes;



   type Action_etype is (to_Proxy_Hunter_fmt, to_Hostname_fmt, to_IP_fmt,
                    no_Action);
   type Ports_array is array (Positive range <>) of Ustr;
   type Ports_array_access is access all Ports_array;

   type Globals_rec is
      record
         Action          : Action_etype;
         pPorts_List     : Ports_array_access;
         Print_Extra     : Boolean;
         Print_Super     : Boolean;
         Print_Only_One_Hostname : Boolean;
         PH_LHS_Cmnt_Trimmed : Ustr;
      end record;


   generic              --  'generic' to allow a a procedure in generic package
                        --    Lines_pkg to be called
      with procedure Put_Data (
                  Task_Num     : Task_Number;
                  Line_Num     : Line_Number;
                  Results      : Results_array);
   procedure Convert_Print (
               Task_Num   : Task_Number;
               Line_Num   : Line_Number;
               host       : Ustr;
               G          : Globals_rec);


   procedure Convert_Print (
               Task_Num   : Task_Number;
               Line_Num   : Line_Number;
               host       : Ustr;
               G          : Globals_rec)
   is

      Action      : Action_etype renames G.Action;
      Ports_List  : Ports_array  renames G.pPorts_List.all;
      Print_Extra : Boolean      renames G.Print_Extra;
      Print_Super : Boolean      renames G.Print_Super;

      function get_Ports (port : Ustr) return Ports_array is
         subtype P is Ports_array;
         port2   : Ustr := port;
         N       : Natural := 0;
      begin
         port2 := port;
         Ltrim_Zeroes (port2);
         if ("" = port2) then
            if (0 = Ports_List'Length) then
               if (Action = to_Proxy_Hunter_fmt) then return P'(1 => us ("80"));
               else return P'(1 => Null_Ustr); end if;
            else
               return Ports_List;
            end if;
         else
                  --  Return (port2) & (Ports_List with 1st duplicate of port2 removed)
            for J in Ports_List'Range loop
               if (port2 /= Ports_List (J)) or else (N < J - 1) then
                  N := N + 1;
               end if;
            end loop;
            declare
               Res     : Ports_array (1 .. 1 + N);
            begin
               N := 0;
               Res (1) := port2;
               for J in Ports_List'Range loop
                  if (port2 /= Ports_List (J)) or else (N < J - 1) then
                     N := N + 1;
                     Res (1 + N) := Ports_List (J);
                  end if;
               end loop;
               return Res;
            end;
         end if;
      end get_Ports;

      procedure PutL (S : String) is
      begin
         if dbg then
            PlPre.PutL ("--t" & SF.Trim (TImg (Task_Num), Left) & " " & S);
         end if;
      end PutL;

      procedure PutL (S : Ustr) is
      begin
         if dbg then
            Convert_Print.PutL (fs (S));
         end if;
      end PutL;

      pragma Inline (Convert_Print.PutL);

      procedure Abbreviate_F_msg (A1ID : Ustr; F_msg : in out Ustr) is
            --  Require (A1ID /= ""). It ought be right trimmed.
         A      : String := fs (A1ID);
         AI     : constant Natural := SU.Index (F_msg, A);
         ALen   : constant Natural := A'Length;
         MLen   : Natural := SU.Length (F_msg);
      begin
         if AI = MLen - ALen + 1 then
            F_msg := SU.Trim (SU.Delete (F_msg, AI, AI + ALen - 1), Right);
            MLen := SU.Length (F_msg);
            if MLen = SU.Index (F_msg, ":", Backward) then
               F_msg := SU.Trim (SU.Delete (F_msg, MLen, MLen), Right);
            end if;
         end if;
      end Abbreviate_F_msg;


      Cmnt            : Ustr;
      Orig_ID         : Ustr;
      Orig_Line       : Ustr;
      to_PH_IP        : Boolean := (Action = to_Proxy_Hunter_fmt);
      to_IP_IP        : Boolean := (Action = to_IP_fmt);
      Prev_F_msg      : Ustr;



      procedure Preamble1 is
         t0, t1       : Ustr;
         Sq_Brkt_ID   : Ustr;
         K_curly, K_hash, K_space : Natural;
      begin

         Orig_Line := SU.Trim (SU.Translate (Source => host,
                        Mapping => WhiteSpace_mapping), Both);

         for Remove_SpaceColon in 1 .. 2 loop
            K_hash := Index_L (Orig_Line, "#");
            K_space := Index_L (Orig_Line, " ");
            Split (Orig_Line, From => Positive'Min (K_hash, K_space), Gap => 1,
                     R1 => Orig_ID, R2 => Cmnt);    --  Cmnt is LTrimmed

            if Cmnt /= "" and then K_space < K_hash
                     and then ':' = SU.Element (Cmnt, 1)
            then
                     --  E.g. Change: "10.1.1.1:83@HTTP   :5:A  [ x:y # c"
                     --         into: "10.1.1.1:83@HTTP:5:A  [ x:y # c"
               Orig_line := Orig_ID & Cmnt;
            else
               exit;
            end if;
         end loop;

         K_hash := Index_L (Cmnt, "#");

         K_curly := SU.Index (Cmnt, "{");
         if (0 < K_curly and K_curly < K_hash) then
            Prev_F_msg := us (SF.Trim (SU.Slice (Cmnt, K_curly + 1, K_hash - 1), Both));
            SU.Delete (Cmnt, K_curly, K_hash - 1);
         else
            Prev_F_msg := Null_Ustr;
         end if;
         PutL ("Cmnt(1) = '" & Cmnt  & "'");

         if 1 = SU.Index (Cmnt, "[") then
            Split (SU.Delete (Cmnt, 1, 1), "#", Sq_Brkt_ID, Cmnt);
            PutL ("Sq_Brkt_ID = '" & Sq_Brkt_ID  & "'");
         else
            --  Sq_Brkt_ID := Null_Ustr;
            t0 := Cmnt;
            Split (t0, "#", t1, Cmnt);
            PutL ("t1 = '" & t1  & "'");
         end if;

         PutL ("Orig_ID = '" & Orig_ID  & "'");
         PutL ("Cmnt = '" & Cmnt  & "'");
         PutL ("Prev_F_msg = '" & Prev_F_msg  & "'");

      end Preamble1;


      Socks5         : Ustr;
      ID, ID_port, ID_note : Ustr;
      ID_Sq       : Ustr;



      procedure Preamble2 is
         L           : Natural;
         attrs       : Ustr;
         OrID        : Ustr := Orig_ID;
         Kat         : Natural := SU.Index (OrID, "@");     --  OID="a:92@HTTP3:A:x", Kat=5
         notes       : Ustr := Null_Ustr;
      begin

         if Kat > 0 then notes := SU.Delete (OrID, 1, Kat); end if;

         if 0 = Kat then                  --       12345678901234
            Socks5 := Null_Ustr;
         else
            L := SU.Index (                               --  L=6
                     Source => notes,                     --  "HTTP3:A:x"
                     Set    => SMC.Alphanumeric_Set,      --   123456789
                     Test   => Outside);
			   if L = 0 then L := SU.Length (notes) + 1; end if;
            Socks5 := us (SU.Slice (OrID, Kat, Kat + L - 1)); --  ="@HTTP3"
            SU.Delete (OrID, Kat, Kat + L - 1);       -- "a:92:A:x"
            if Socks5 = "@HTTP" then
                           --  Args: "-n -aa 10.1.1.1:83@HTTP:A  [  pq # c"
               Orig_ID := OrID;
            end if;
         end if;

         Split (OrID, ":", ID, attrs);               -- ("a", "92:A:x")
         ID := Clean_IP_Number (ID);                -- "001.2.3.4"->"1.2.3.4"

         Split (attrs, ":", ID_port, ID_note, Keep_Pattern);
         PutL ("attrs = '" & attrs  & "'");
                                                        -- ("92", ":A:x")
         if 0 < SU.Index (ID_port, SMC.Decimal_Digit_Set, Outside) then
            ID_port := Null_Ustr;
            ID_note := ':' & attrs;
         end if;

         if Print_Super then ID_Sq := Orig_ID;
         elsif Print_Extra then ID_Sq := ID;
         else ID_Sq := Null_Ustr; end if;

         PutL ("OrID = '" & OrID  & "'");
         PutL ("ID = '" & ID  & "'");
         PutL ("ID_port = '" & ID_port  & "'");
         PutL ("ID_note = '" & ID_note & "'");
         PutL ("ID_Sq = '" & ID_Sq  & "'");

      end Preamble2;


      Line0           : Ustr;
      pIDs            : NameIPs_AFtype;
      Was_Already_Converted_No_F_msg : Boolean;
      AtHttp          : constant Ustr := us ("@HTTP");
      Put_Data_Called : Boolean := false;

   begin
      PutL ("Convert_and_Print starts...");
      Preamble1;
      if "" = Orig_ID then goto QUIT; end if;
      Preamble2;
      if "" = ID then goto QUIT; end if;

      PutL ("Socks5 = '" & Socks5  & "'");
      if to_PH_IP then
         if Socks5 = "" then Socks5 := AtHttp;
         else Socks5 := toUpperCase (Socks5); end if;
      else
         if Socks5 = "@HTTP" then Socks5 := Null_Ustr; end if;
      end if;
      PutL ("Socks5 = '" & Socks5  & "'");


      if to_PH_IP or to_IP_IP then
         ------------------------------------------------------------

         PlIPs.IP_of_ID (ID, From_Name_to_IP, pIDs, Find_Only_One_Hostname => false,
                  Was_Already_Converted_No_F_msg => Was_Already_Converted_No_F_msg);

         declare
            IDs     : NameIPs_array renames pIDs.Reference.all;
            Ports   : constant Ports_array := get_Ports (ID_port);
            Results : Results_array (1 .. Ports'Last * IDs'Last);
            C       : Ustr := pqIFq (" ", Cmnt);
            E       : Ustr := Socks5 & ifelse (to_PH_IP and then ""/=ID_note,
                                     " ", "") & ID_note;
            C3      : Ustr := (" #" * ("" /= C)) & C;
            R, B, F_msg : Ustr;
            Is_IP   : Boolean;
            H       : Positive;
         begin
            PutL ("1 Action = " & Action_etype'Image (Action));
            for J in 1 .. IDs'Last loop
               if Was_Already_Converted_No_F_msg and Print_Extra then
                  F_msg := Prev_F_msg;
               else
                  F_msg := IDs (J).failed_msg;
               end if;
               Is_IP := PlIPs.Is_IP_Address (IDs (J).id) and then
                        (1 /= SU.Index (IDs (J).id, "255.255.255.255"));
               if not Is_IP or ("" = IDs (J).id) then
                  if F_msg = "" then
                     F_msg := us ("No IP number");
                  elsif "" /= ID then
                     Abbreviate_F_msg (ID, F_msg);
                  end if;
                  Results (J) := PlIPs.PH_LHS_comment & Orig_ID & pqIFq (" { ", F_msg) & C3;
                  for PN in 2 .. Ports'Last loop
                     Results (Ports'Last * (J - 1) + PN) := Null_Ustr;
                  end loop;
               else
                  if "" /= ID_Sq and then ID /= IDs (J).id then
                     R := " [ " & ID_Sq;
                              --  Abbreviate_F_msg (ID, F_msg);
                  else
                     R := Null_Ustr;
                  end if;
                  B := E & R & pqIFq (" { ", F_msg) & C3;
                  for PN in Ports'Range loop
                     H := Ports'Last * (J - 1) + PN;
                     Results (H) := IDs (J).id & pqIFq (":", Ports (PN)) & B;
                  end loop;
               end if;
               PutL ("HN->IP/PH: " & fs (Results (J)) & ", IDs(.)='" & fs (IDs (J).id) & "'");
            end loop;
            PutL ("Line_Num = " & LImg (Line_Num));
            Put_Data (Task_Num, Line_Num, Results);
            Put_Data_Called := true;
         end;

      elsif Action /= to_Hostname_fmt then null;

      elsif (0 < SU.Index (Orig_Line, "{")) and then (
            (G.PH_LHS_Cmnt_Trimmed /= Null_Ustr)
                  and then 1 = SU.Index (Orig_Line, fs (G.PH_LHS_Cmnt_Trimmed)))
      then
         null;

      else                                              --  from IP number to Name

         PlIPs.IP_of_ID (ID, From_IP, pIDs, G.Print_Only_One_Hostname,
               Was_Already_Converted_No_F_msg => Was_Already_Converted_No_F_msg);
                                                   --  ID="11.22.33.44"
         declare
            IDs     : NameIPs_array renames pIDs.Reference.all;
            M       : Natural := ifelse (G.Print_Only_One_Hostname, 1, IDs'Last);
            Ports   : constant Ports_array := get_Ports (ID_port);
            Results : Results_array (1 .. M * Ports'Last);
            R, B, F_msg : Ustr;
            D       : Ustr := pqIFq (" #", pqIFq (" ", Cmnt));
            H       : Positive;
            E       : Ustr := Socks5 & ID_note;

         begin
            for J in 1 .. M loop
               if Was_Already_Converted_No_F_msg and Print_Extra then
                  F_msg := Prev_F_msg;
               else
                  F_msg := IDs (J).failed_msg;
               end if;
               if "" /= ID_Sq and then ID /= IDs (J).id then
                  R := " [ " & ID_Sq;
                  Abbreviate_F_msg (ID, F_msg);
               else
                  R := Null_Ustr;
               end if;
               B := E & R & pqIFq (" { ", F_msg) & D;
               for PN in 1 .. Ports'Last loop
                  H := Ports'Last * (J - 1) + PN;
                  Results (H) := SU.Trim (IDs (J).id & pqIFq(":", Ports (PN)) & B, Left);
                  PutL ("IP/PH->HN: " & fs (Results (H)));
               end loop;
               Put_Data (Task_Num, Line_Num, Results);
               Put_Data_Called := true;
            end loop;
         end;
      end if;  --  end of (Action = to_Hostname_fmt) case

      <<QUIT>>
      if not Put_Data_Called then
            --  Program will hang if Put_Data isn't called for this line
         declare
            Results : Results_array (1 .. 1) := (1 => Line0);
         begin
            Put_Data (Task_Num, Line_Num, Results);
         end;
      end if;
   end Convert_Print;



   procedure Process (
               Num_Threads : Task_Number;
               From_File   : Boolean;
               Input_FName : Ustr;
               host        : Ustr;
               G           : Globals_rec)
   is
         --  Read Lines into a linked list and then instantiates a Generic
         --   package. An array allows the output to be not disordered.
         --  Starts DNS lookup tasks and the task that does background printing.

      type E_rec;
      type E_rec_access is access E_rec;
      type E_rec is record
         Text    : Ustr;
         next    : E_rec_access := null;
      end record;
      procedure Free_E is new Ada.Unchecked_Deallocation (E_rec, E_rec_access);

      First_E    : E_rec_access;


      procedure Commence (Num_Lines : Line_Number)
      is
         package IO is new Lines_pkg (Num_Lines => Num_Lines);

         procedure Put_Data_2 (
                     Task_Num     : Task_Number;
                     Line_Num     : Line_Number;
                     Results      : Results_array) is
         begin
            IO.Lines_Server.Put_Data (Task_Num, Line_Num, Results);
         end Put_Data_2;

         procedure DoAndPrint is new Convert_Print (Put_Data => Put_Data_2);

         task type Worker_task_type is

            entry Start (Num : Task_Number);

         end Worker_task_type;

         task body Worker_task_type is
            Task_Num      : Task_Number;
            Line, Result  : Ustr;
            Line_Num      : Line_Number;
         begin
            select
               accept Start (Num : Task_Number) do
                  Task_Num := Num;
               end Start;
            or
               terminate;
            end select;

            loop
               PutL ("DNS task #" & TImg (Task_Num) &
                      " does lookup. (a)");
               IO.Lines_Server.Get_Next (Line_Num, Line);
               PutL ("  Line" & LImg (Line_Num) &
                      ": '" & Line & "' ...");
               exit when 0 = Line_Num;    --  finished
               DoAndPrint (Task_Num, Line_Num, host => Line, G => G);
            end loop;

         exception
            when X : others =>
               Raise_Msg ("! DNS task #" & TImg (Task_Num) &
                      " stopped.  Line number =" & LImg (Line_Num), X);
         end Worker_task_type;


         procedure Work (
                     Num_Threads : Task_Number)
         is
            Workers : array (1 .. Num_Threads) of Worker_task_type;
                  --   All run until they reach the accept entry
         begin
            PutL (" Prog Work started, num threads = " & TImg (Num_Threads));
            tio.Flush;
            IO.Printer_Task.Start;
            for ID in Workers'Range loop
               Workers (ID).Start (ID);
            end loop;
         end Work;

         ----------------------------------------------------------------

         subtype L is Line_Number;
         Num_Threads_2 : Task_Number;
         LC          : constant Natural := SU.Length (PlIPs.PH_LHS_comment);
         F, H        : E_rec_access;

      begin
         PutL ("Commence: Worker_task_type'Storage_Size = " & Integer'Image (Worker_task_type'Storage_Size));

         PutL ("Commence: Moving data to an array ...");

         F := First_E;
         for K in 1 .. Num_Lines loop
            if 0 < LC and then 1 = SU.Index (F.Text, fs (PlIPs.PH_LHS_comment)) then
               SU.Delete (F.Text, 1, LC);
            end if;
            IO.Lines_Server.Assign (Line_Num => K, Text => F.Text);
            H := F.next;
            Free_E (F);
            F := H;
         end loop;

         if From_File then Put_Line (" ..."); tio.Flush; end if;

         Num_Threads_2 := Task_Number (L'Min (L (Num_Threads), Num_Lines));

                     --  L'Min (Num_Lines,L (R'Rounding (8.0 + 0.7*(R(Num_Lines)-8.0)))))),

         PutL ("Commence: Starting lookup tasks ...");

         Work (Num_Threads_2);

         IO.Finalize;

      end Commence;


      Node       : E_rec_access;
      Num_Lines  : Line_Number := 0;

      procedure Add_To_List (line : Ustr) is
         H          : E_rec_access;
      begin
         H := new E_rec'(Text => line, next => null);

         Num_Lines := Num_Lines + 1;
         if Num_Lines = 1 then
            First_E := H;
         else
            Node.next := H;
         end if;
         Node := H;
      end Add_To_List;

   begin
      PutL ("Reading ...");

      if From_File then
         declare
            package Inp_IO is new PlPre.IO;
            File        : tio.File_Type renames Inp_IO.File;
            line        : Ustr;
         begin
            Inp_IO.Fname := Input_FName;
            loop
               Inp_IO.Read_Next_Line (line);
               exit when line = Null_Ustr;
               Add_To_List (line);
               exit when tio.End_Of_File (File);
            end loop;
            PutL ("File read");

            if G.Action = to_Proxy_Hunter_fmt then
               tio.Put (SF.Trim (fs (PlIPs.PH_LHS_comment), Right));
            else
               tio.Put ("#");
            end if;
            tio.Put (LImg (Num_Lines) &
                        " lines read from " & tio.Name (File));

            Inp_IO.Close_File;
            tio.Flush;
         end;
      else
         if host /= Null_Ustr then
            Add_To_List (host);
         end if;
      end if;

      if Num_Lines >= 1 then
         PutL ("Initializing sockets");
         PlIPs.PHSocket.Start_Sockets;
         PutL ("Initialized sockets");
         Commence (Num_Lines);
         PutL ("Finishing");
         PlIPs.PHSocket.Finishup_Sockets;
      end if;

   end Process;



--  ==================================================================

      --  To get extra threads in Win32, this was done: Include something
      --    like "-largs -v -Xlinker --stack=0x10" to the
      --    Gnatmake option list. Seems to make about no difference to
      --    memory use which of following is used: 0x1, 0x10, 0x100.
      --  Smaller sizes seem to allow more threads.
      --  (Win95 System monitor, Norton System Information)

   --  Max_Threads    : constant Task_Number := 58;   --  if 60 in Win 95(NT), then prog stalls
   Max_Threads    : constant Task_Number := 9999;
   Num_Threads    : Task_Number := 60;

   Default_Ports_List : aliased Ports_array := Ports_array'(1 => us ("80"));


   Num_args       : constant Natural := Ada.Command_Line.Argument_Count;
   LastArg        : Natural := Num_args;
   Args           : array (1 .. Num_args) of Ustr;

   procedure Drop (K : Positive) is begin
      if K <= LastArg then
         LastArg := LastArg - 1;
         for H in K .. LastArg loop Args (H) := Args (H + 1); end loop;
      end if;
   end Drop;

   X              : Ustr;
   --  type Mode_type is (From_Name, From_IP, Prt_Help, None);
   Print_Help     : Boolean := False;
   Bad_Option     : Boolean := False;
   S, R1, R2      : Ustr;
   L              : Integer;
   Input_FName    : Ustr := Null_Ustr;
   host           : Ustr := Null_Ustr;
   From_File      : Boolean;
   G              : Globals_rec := (
      Action => no_Action,
      pPorts_List => null,
      Print_Extra => false,
      Print_Super => false,
      Print_Only_One_Hostname => false,
      PH_LHS_Cmnt_Trimmed => Null_Ustr);

   Action         : Action_etype renames G.Action;

begin

   dbg := False;

   tio.Set_Output (tio.Standard_Output);

   for J in 1 .. Num_args loop
      Args (J) := us (Ada.Command_Line.Argument (J));
   end loop;

   loop
      exit when LastArg <= 0;
      X := us (toLowerCase (fs (Args (1))));
            --  Neither IP numbers nor public hostnames start with "-"

      if ("-ph" = X or "-p" = Args (1)) then
         if (Action /= no_Action) then Bad_Option := true; exit; end if;
         Action := to_Proxy_Hunter_fmt;

      elsif ("-ip" = X or "-i" = X) then
         if (Action /= no_Action) then Bad_Option := true; exit; end if;
         Action := to_IP_fmt;

      elsif ("-n" = X or "-name" = X) or ("-n1" = X) then
         if (Action /= no_Action) then Bad_Option := true; exit; end if;
         Action := to_Hostname_fmt;
         if ("-n1" = X) then   --  Don't print hostname synonyms
            G.Print_Only_One_Hostname := true;
         end if;

      elsif ("-a" = X or "-all" = X) then   --  print original name input
         G.Print_Extra := true;

      elsif ("-aa" = X or "-all" = X) then   --   & print port+@SOCKS5+notes
         G.Print_Extra := true;
         G.Print_Super := true;

      elsif "-f" = X then
         if 2 <= LastArg then Input_FName := Args (2);
         else Bad_Option := true; end if;
         Drop (1);

      elsif "-P" = Args (1) or "-po" = X or "-port" = X or "-ports" = X then
         if LastArg < 2 or else
                  not Is_List_of_Numbers (fs (Args (2)), ",", Long_Integer'Last)
         then
            PutL (" Check -po: " & Boolean'Image (Is_List_of_Numbers (fs (Args (2)), ",", Long_Integer'Last)));
            Bad_Option := true;
         else
            S := Args (2);
            G.pPorts_List := new Ports_array (1 .. 1 + SU.Count (S, ","));
            for K in G.pPorts_List.all'Range loop
               Split (S, ",", R1, R2);
               G.pPorts_List.all (K) := R1;
               S := R2;
            end loop;
         end if;
         Drop (1);

      elsif "-t" = X then
         if LastArg < 2 then
            Bad_Option := true;
         else
            S := Args (2);
            L := SU.Length (S);
            if 0 = SU.Index (S, SMC.Decimal_Digit_Set, Outside) and    L >= 1 then --  all in 0 .. 9
               Num_Threads := Task_Number'Min (Max_Threads,
                        Task_Number'Max (1,                -- Num_Threads <= 9999
                           Task_Number'Value (SU.Slice (S, 1, Integer'Min (4, L)))));
            end if;
         end if;
         Drop (1);

      elsif "-dbg" = X then
         dbg := True;

      elsif ("-h" = X or "--help" = X or "/h" = X or "-help" = X) then
                    --  In Win95 GNAT, "?" & "*" match with filenames in directory
         Print_Help := True;

      elsif 1 = SU.Index (X, "-") then
         Bad_Option := True;

      else        -- doesn't start with "-"
         exit;
      end if;
      Drop (1);
   end loop;

   for J in 1 .. LastArg loop
      host := us (fs (host) & String'(1 .. Natural'Min (1, J - 1) => ' ') & fs (Args (J)));
   end loop;

   From_File := ("" /= Input_FName);

   --  PutL ("Hostname/ipnumber = '" & host & "'");
   --  PutL ("Print_Extra = '" & Boolean'Image (G.Print_Extra) & "'");


   if (Action = no_Action) and not From_File and ("" /= host) then
      Split (host, ":", R1, R2);
      if PlIPs.Is_IP_Address (R1) then
         Action := to_Hostname_fmt;
      else
         Action := to_IP_fmt;
      end if;
   end if;

   if Bad_Option or (0 = Num_args)
            or ((Action = no_Action) = (From_File or "" /= host))
   then
      Put_Line ("PLOOKUP: Invalid option(s).  Try '-h' for more information");

   elsif Print_Help or (Action = no_Action) then
      Put_Line ("PLOOKUP 2.51 (22-Aug-99).  Usage: plookup OPTIONS [HOSTNAME/IPNUM/PH_IP]");
      Put_Line ("Converts between: hostnames, IP numbers, and Proxy Hunter IP numbers.");
      Put_Line ("  -i, -ip         : convert to IP number(s)");
      Put_Line ("  -n,             : convert to hostName(s)");
      Put_Line ("  -n1             : print only the most website-ish un-proxy-ish hostname");
      Put_Line ("  -p, -ph         : convert to Proxy Hunter IP number(s)");
      Put_Line ("  -po LIST, -P    : a comma separated list of Ports to add");
      Put_Line ("  -a              : print all original source name(s) or IP number(s)");
      Put_Line ("  -aa             : like '-a', but print port etc. following hostname or IP#");
      Put_Line ("  -f FILE         : name of File (if any) to read lines from");
      Put_Line ("  -t THREADS      : number of Threads. (Default =" &
               TImg (Num_Threads) & ")");
      Put_Line ("  -h, --help, /h  : display this Help and exit");
      Put_Line ("Report bugs to C. Carey at <research@ijs.co.nz> Auckland");
         -- option -dbg also

   else

      if (G.pPorts_List = null) then
         G.pPorts_List := new Ports_array (1 .. 0);
      else
         for K in G.pPorts_List.all'Range loop
            Ltrim_Zeroes (G.pPorts_List.all (K));
         end loop;
      end if;

      if Action = to_Proxy_Hunter_fmt then PlIPs.PH_LHS_comment := us ("0.0.0.0:80 # ");
      else PlIPs.PH_LHS_comment := Null_Ustr; end if;

      G.PH_LHS_Cmnt_Trimmed := SU.Trim (PlIPs.PH_LHS_comment, Right);

      Process (Num_Threads, From_File, Input_FName, host, G => G);

   end if;

   --  Set exit code.

   exception

      when Halt_Without_Err_Msg =>
         null;

      --  when System.Assertions.Assert_Failure =>
      --   .. . raise;

      when X : others =>
         Raise_Msg ("! ERROR. Execution ends, Ada exception raised. ", X);
         --  tio.Flush;
         --  BUG?: add "break exception" to GDB file ".gdbinit" (if compiled
         --   with debugging ("gnatmake -g").
            --   "Try rerunning using the GNU 'gdb' debugger with a " &
            --   "breakpoint on '__gnat_raise' or equiv. 'break exception'.
            --    Also try gdb commands: " &
            --   "h, info, run, frame n, bt, list, set args; etc.  ", X);
         --  close any open files, etc.
         raise;

end plookup;

--  To use Adasockets of enst.fr (in GNAT but for Unix) in Win95, do initialise Win32 sockets,
--  and change "import" pragmas.

--  Other Ada sockets packages: htp://www.ida.liu.se/~tobri, Botton's.
