-----------------------------------------------------------------------------
--                                                                         --
--                             P L O O K U P                               --
--                                                                         --
--                         P L P R E   B o d y                             --
--                                                                         --
--                        $ReleaseVersion: -----  $                         --
--                                                                         --
--  Copyright (C) 1999   Mr G. A. Craig Carey, Auckland, New Zealand.      --
--                                                                         --
--   Plookup is free software; you can  redistribute  it  and/or  modify   --
--   it  under terms of the GNU  General  Public License as published by   --
--   the Free Software Foundation; either version 0, or (at your option)   --
--   any later version.   Plookup is distributed  in the  hope  that  it   --
--   will be useful, but WITHOUT ANY  WARRANTY; without even the implied   --
--   warranty of MERCHANTABILITY  or FITNESS FOR  A PARTICULAR PURPOSE.    --
--   See the GNU General Public  License  for more details.  You  should   --
--   have received a copy of the  GNU General Public License distributed   --
--   with Plookup; see   file COPYING.  If  not,  write  to  the Free      --
--   Software  Foundation, 59   Temple Place -   Suite  330,  Boston, MA   --
--   02111-1307, USA.                                                      --
--                                                                         --
--   As a special exception, if  other  files instantiate generics  from   --
--   this unit, or  you link this  unit with other  files to produce  an   --
--   executable,  this  unit does  not  by  itself cause  the  resulting   --
--   executable to be  covered by the  GNU General Public License.  This   --
--   exception does  not  however invalidate any  other reasons  why the   --
--   executable file might be covered by the GNU Public License.           --
--                                                                         --
--   The author of the software is Mr Craig Carey. The homepage of the     --
--   author is located at:                                                 --
--       http://home.clear.net.nz/pages/research/                          --
--   Dated 8 March 1999                                                    --
--                                                                         --
-----------------------------------------------------------------------------

   --  Program By Mr Craig Carey, Auckland city, research@clear.net.nz
   --   http://home.clear.net.nz/pages/research/

--  pragma Source_File_Name (UNIT_NAME => PlPre, BODY_FILE_NAME => "plpre.adb");

--  pragma Normalize_Scalars;   --  Set uninitialized scalar objects out of range


package body PlPre is


   package body Sort is
        --  Elements that are equal (Lt = false) are not disordered.
        --  Slow on big arrays.

      procedure Sort_Up (LB, UB : Sort_Natural) is
         pragma Assert (1 <= LB);
      begin
         if LB < UB then
            if LB+1 < UB then  Sort_Up (LB+1, UB);  end if;
            if Lt (LB+1, LB) then
               Move (LB, 0);
               Move (LB+1, LB);
               Move (0, LB+1);
               if LB+1 < UB then  Sort_Up (LB+1, UB);  end if;
            end if;
         end if;
      end Sort_Up;

   end Sort;

   ---------------------------------------------------------------------------

   use type Ustr;     --  Make visible operators "=", "&" (L:String; R:Ustr).

   procedure Put_Line (Item : in Ustr) is
   begin
      tio.Put_Line (Item => fs (Item));
   end Put_Line;


   procedure PutL0 (line : String) is begin
      Put_Line (Pre & line);
      tio.Flush;
   end PutL0;

   procedure PutL0 (line : Ustr) is begin
      Put_Line (Pre & fs (line));
      tio.Flush;
   end PutL0;


   procedure PutL (line : String) is begin
      if dbg then PutL0 (line); end if;
   end PutL;

   procedure PutL (line : Ustr) is begin
      if dbg then PutL0 (line); end if;
   end PutL;


   ----------------------------------------------------------------------------

   function toLowerCase (Source : in String) return String is
   begin
      return SF.Translate (Source, SMC.Lower_Case_Map);
   end toLowerCase;

   function toLowerCase (Source : in Ustr) return Ustr is
   begin
      return SU.Translate (Source, SMC.Lower_Case_Map);
   end toLowerCase;

   function toUpperCase (Source : in Ustr) return Ustr is
   begin
      return SU.Translate (Source, SMC.Upper_Case_Map);
   end toUpperCase;

   function Clean_WhiteSpace (Source : in Ustr) return Ustr is
      --  Trim for the read line procedure
   begin
      return SU.Trim (
            SU.Translate (Source => Source, Mapping => WhiteSpace_mapping),
                  Side => Both);
   end Clean_WhiteSpace;


   ----------------------------------------------------------------------------


   function pqIFq (p : String; q : Ustr) return Ustr is
   begin
      if Null_Ustr = q then return Null_Ustr; else return p & q; end if;
   end pqIFq;


   function pqIFp (p : String; q : Ustr) return Ustr is
   begin
      if Null_Ustr = p then return Null_Ustr; else return p & q; end if;
   end pqIFp;


   function "*" (p : Ustr; b : Boolean) return Ustr is
   begin
      if b then return p; else return Null_Ustr; end if;
   end "*";


   function "*" (p : String; b : Boolean) return String is
   begin
      if b then return p; else return ""; end if;
   end "*";


   function ifelse (b : Boolean; p, q : String) return String is
   begin
      if b then return p; else return q; end if;
   end ifelse;


   function ifelse (b : Boolean; p, q : Ustr) return Ustr is
   begin
      if b then return p; else return q; end if;
   end ifelse;


   function ifelse (b : Boolean; p, q : Integer) return Integer is
   begin
      if b then return p; else return q; end if;
   end ifelse;


   ----------------------------------------------------------------------------


   procedure Raise_Msg (
            whaterr : String;
            X       : Ada.Exceptions.Exception_Occurrence)
   is
      mesg    : String := Ada.Exceptions.Exception_Message (X);
      info    : String := Ada.Exceptions.Exception_Information (X);
      name    : String := Ada.Exceptions.Exception_Name (X);
   begin
      tio.Put_Line (">>>  Exception raised: " & whaterr);
      if mesg /= "" then
         tio.Put (" >>> Exception mesg: " & mesg);
         if 0 = SF.Index (mesg, name) then
            tio.Put (". " & Ada.Exceptions.Exception_Name (X));
         end if;
         tio.New_Line;
      end if;
      if info /= "" then
         tio.Put_Line (" >>> Exception info: " & info);
      end if;
      tio.Flush;
   end Raise_Msg;


   ----------------------------------------------------------------------------


   function Index_L (Source : Ustr; Pat : String) return Positive is
      K    : Natural := SU.Index (Source, Pat);
   begin
      if K = 0 then return 1 + SU.Length (Source);
      else return K;
      end if;
   end Index_L;


   procedure Split (Source   : Ustr;
                    Pat      : String;
                    R1, R2   : out Ustr;
                    Mode     : Split_Mode := Remove_Pattern)
   is
      Divide   : Natural := SU.Index (Source, Pat);
      From     : Natural;
   begin
      if 0 = Divide then
         R1 := Source;  R2 := Null_Ustr;
      else
         R1 := us (SF.Trim (SU.Slice (Source, 1, Divide-1), Side => Both));
         if Mode = Remove_Pattern then From := Divide + Pat'Length;
         else From := Divide; end if;
         R2 := us (SF.Trim (SU.Slice (Source, From, SU.Length (Source)),
                     Side => Both));
      end if;
   end Split;


   procedure Split (Source : Ustr; From : Positive; Gap : Natural; R1, R2 : out Ustr) is
      Len : Natural := SU.Length (Source);
   begin
      R1 := us (SF.Trim (SU.Slice (Source, 1, From - 1), Side => Both));
      R2 := us (SF.Trim (SU.Slice (Source, Natural'Min (From + Gap, Len + 1), Len),
                  Side => Both));
   end Split;


   ----------------------------------------------------------------------------


   --   generic

   package body IO is
   ---------------

      Io_Fname    : Ustr := Null_Ustr;

      Input_Line_Buffer_Length : constant := 1024;  --  size of each Get_Line step

      procedure Get_Line (File : in tio.File_Type; Item : out Ustr) is

         function More_Input return Ustr is
            Input : String (1 .. Input_Line_Buffer_Length);
            Last  : Natural;
         begin
            tio.Get_Line (File, Input, Last);
            if Last < Input'Last then return us (Input (1 .. Last));
            else return us (Input (1 .. Last)) & More_Input;
            end if;
         end More_Input;

      begin
         Item := More_Input;
      end Get_Line;


      procedure Open_File
      -------------------
      is
         fn        : string := SF.Trim (fs (Fname), Both);
      begin
         --  Put ("> Opening (read-only) file '" & fn & "' .. . ");
         begin
            tio.Open (File => File, Mode => tio.In_File, Name => fn);
         exception
            when X : tio.Name_Error =>
               tio.New_Line;
               Put_Line (">>> Error opening file: File not found: '" &
                        fs (Fname) & "'");
               raise Halt_Without_Err_Msg;
            when X : others =>
               tio.New_Line;
               Raise_Msg ("Error opening file: '" & fs (Fname) & "'", X);
               raise;
         end;

         IO.Fname := us (tio.Name (File));
         File_Opened := True;

      end Open_File;


      procedure Read_Next_Line (Line : out Ustr) is
      begin
         loop
            Line := Null_Ustr;
            if not File_Opened then Open_File; end if;
            exit when tio.End_of_File (File);
            Linenum := Linenum + 1;
            begin
               Get_Line (File => File, Item => Line);
            exception
               when X : others =>
                  Raise_Msg ("Error reading line" & Linenum'Img &
                                 " of file " & fs (Fname), X);
                  Put_Line ("> Line='" & Line & "'");
                  raise;
            end;
            Line := SU.Trim (Clean_WhiteSpace (Line), Both);
            exit when Line /= Null_Ustr;
            --  exit when Line /= Null_Ustr and then 1 /= SU.Index (Line, "#");
         end loop;
      end Read_Next_Line;


      procedure Close_File is
      --------------------
      begin
         if File_Opened then tio.Close (File); end if;
      exception
         when X : others =>
            Raise_Msg ("Error closing file '" & fs (Fname) & "'", X);
      end Close_File;


   end IO;  --  end package









end PlPre;
