-----------------------------------------------------------------------------
--                                                                         --
--                         ADASOCKETS COMPONENTS                           --
--                                                                         --
--                             S O C K E T S                               --
--                                                                         --
--                                S p e c                                  --
--                                                                         --
--                         ReleaseVersion: 0.1.2                           --
--                                                                         --
--  Copyright (C) 1998  Ecole Nationale Superieure des Telecommunications  --
--                                                                         --
--   AdaSockets is free software; you can  redistribute it and/or modify   --
--   it  under terms of the GNU  General  Public License as published by   --
--   the Free Software Foundation; either version 2, or (at your option)   --
--   any later version.   AdaSockets is distributed  in the hope that it   --
--   will be useful, but WITHOUT ANY  WARRANTY; without even the implied   --
--   warranty of MERCHANTABILITY   or FITNESS FOR  A PARTICULAR PURPOSE.   --
--   See the GNU General Public  License  for more details.  You  should   --
--   have received a copy of the  GNU General Public License distributed   --
--   with AdaSockets; see   file COPYING.  If  not,  write  to  the Free   --
--   Software  Foundation, 59   Temple Place -   Suite  330,  Boston, MA   --
--   02111-1307, USA.                                                      --
--                                                                         --
--   As a special exception, if  other  files instantiate generics  from   --
--   this unit, or  you link this  unit with other  files to produce  an   --
--   executable,  this  unit does  not  by  itself cause  the  resulting   --
--   executable to be  covered by the  GNU General Public License.  This   --
--   exception does  not  however invalidate any  other reasons  why the   --
--   executable file might be covered by the GNU Public License.           --
--                                                                         --
--   The main repository for this software is located at:                  --
--       http://www-inf.enst.fr/ANC/                                       --
--                                                                         --
-----------------------------------------------------------------------------
--                                                                         --
--   Adapted to the Win32 and WinSock 1.1 environment                      --
--                                                                         --
--   Rev:  0.2                                                             --
--   Date: 05-may-1998                                                     --
--                                                                         --
--   Jerry van Dijk                                                        --
--   Billie Holidaystraat 28                                               --
--   2324 LK  LEIDEN                                                       --
--   THE NETHERLANDS                                                       --
--   tel int +31 (0)71 531 43 65                                           --
--   email jdijk@acm.org                                                   --
--                                                                         --
--   The main repository for this software is located at:                  --
--       http://stad.dsl.nl/~jvandyk                                       --
--                                                                         --
-----------------------------------------------------------------------------

with Ada.Streams;
with Interfaces.C;

package Sockets is

   type Socket_FD is private;
   --  A socket

   type Socket_Domain is (AF_INET);
   --  AF_INET: Internet sockets

   type Socket_Type is (SOCK_STREAM);
   --  SOCK_STREAM: Stream mode

   function Socket
     (Domain : Socket_Domain := AF_INET;
      Typ    : Socket_Type   := SOCK_STREAM)
     return Socket_FD;
   --  Create a socket of the given mode

   Connection_Refused : exception;
   WSAVERNOTSUPPORTED : exception;

   procedure Connect
     (Socket : in Socket_FD;
      Host   : in String;
      Port   : in Positive);
   --  Connect a socket on a given host/port. Raise Connection_Refused if
   --  the connection has not been accepted by the other end.

   procedure Bind
     (Socket : in Socket_FD;
      Port   : in Positive);
   --  Bind a socket on a given port

   procedure Listen
     (Socket     : in Socket_FD;
      Queue_Size : in Positive := 5);
   --  Create a socket's listen queue

   type Socket_Level is (SOL_SOCKET);

   type Socket_Option is (SO_REUSEADDR);

   procedure Setsockopt
     (Socket  : in Socket_FD;
      Level   : in Socket_Level := SOL_SOCKET;
      Optname : in Socket_Option;
      Optval  : in Integer);
   --  Set a socket option

   function Accept_Socket (Socket : Socket_FD) return Socket_FD;
   --  Accept a connection on a socket

   Connection_Closed : exception;

   procedure Send (Socket : in Socket_FD;
                   Data   : in Ada.Streams.Stream_Element_Array);
   --  Send data on a socket. Raise Connection_Closed if the socket
   --  has been closed.

   function Receive (Socket : Socket_FD)
     return Ada.Streams.Stream_Element_Array;
   --  Receive data from a socket. May raise Connection_Closed

   procedure Receive (Socket : in Socket_FD;
                      Data   : out Ada.Streams.Stream_Element_Array);
   --  Fill data from a socket. Raise Connection_Closed if the socket has
   --  been closed before the end of the array.

   type Shutdown_Type is (Receive, Send, Both);

   procedure Shutdown (Socket : in Socket_FD;
                       How    : in Shutdown_Type := Both);
   --  Close a previously opened socket

   ---------------------------------
   -- String-oriented subprograms --
   ---------------------------------

   procedure Put (Socket : in Socket_FD;
                  Str    : in String);
   --  Send a string on the socket

   procedure New_Line (Socket : in Socket_FD;
                       Count  : in Natural := 1);
   --  Send CR/LF sequences on the socket

   procedure Put_Line (Socket : in Socket_FD;
                       Str    : in String);
   --  Send a string + CR/LF on the socket

   function Get (Socket : Socket_FD) return String;
   --  Get a string from the socket

   function Get_Line (Socket : Socket_FD) return String;
   --  Get a full line from the socket. CR is ignored and LF is considered
   --  as an end-of-line marker.

private

   type Socket_FD is new Interfaces.C.int;

end Sockets;
