/****************************************************************************
*																			*
*					Certificate Attribute Checking Routines					*
*						Copyright Peter Gutmann 1996-1999					*
*																			*
****************************************************************************/

#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#if defined( INC_ALL ) ||  defined( INC_CHILD )
  #include "asn1.h"
  #include "asn1oid.h"
  #include "cert.h"
  #include "certattr.h"
#else
  #include "keymgmt/asn1.h"
  #include "keymgmt/asn1oid.h"
  #include "keymgmt/cert.h"
  #include "keymgmt/certattr.h"
#endif /* Compiler-specific includes */

/****************************************************************************
*																			*
*							Attribute Field Type Checking					*
*																			*
****************************************************************************/

/* Validate and preprocess a set of attributes in preparation for writing
   them to a certificate or CRL and set up links to the information in the
   attribute information table prior to encoding the attributes.  This is a
   rather complex process which relies on stepping through the list of
   attribute fields and the attribute information table in sync and making
   sure the list of fields is consistent with the attribute information
   table.  In addition we set up sync points between the list and table which
   are used during the encoding process.  For example assume we have the
   following attribute:

	attribute ::= SEQUENCE {
		foo		BOOLEAN DEFAULT TRUE,
		bar		SEQUENCE OF OBJECT IDENTIFIER
		}

   The attribute information table would encode this attribute as:

	t1:	OID	SEQUENCE	MORE
	t2:		BOOLEAN		MORE	OPTIONAL
	t3:		SEQUENCE	MORE
	t4:		OID

   The first table entry t1 contains the OID, the SEQUENCE wrapper, and a
   continuation flag.  For the purposes of comparison with the list, this is
   a nop and can be skipped since it's only used for encoding purposes.  The
   next table entry t2 contains the first attribute field, an optional
   boolean and a continuation flag.  The next table entry t3 contains another
   SEQUENCE wrapper which again is only used for encoding and can be skipped
   for comparing with the list, and a continuation flag.  Finally, the last
   table entry t4 contains the second attribute field, an OID.

   Assuming the attribute list contains the following:

	BOOLEAN	FALSE	-> t1
	OID		xxx		-> t3

   The attribute validation process sets the sync point for the first
   attribute list entry to point to t1, and the second one to point to t3.
   When we encode the attribute, we encode t1 (the OID, critical flag, and
   SEQUENCE wrapper), since the field ID's won't match we step to t2 and use
   that to encode the boolean.  We then do the same for t3 with the SEQUENCE
   and OID.

   If the attribute list instead contained only:

	OID		xxx		-> t1

   then this time the attribute validation process sets the sync point to t1.
   When encoding we encode t1 as before, step to t2, the field ID's won't
   match but t2 is optional so we skip it, then encode t3 as for t1 and
   finally encode the OID using t4.

   At this point we also evaluate the encoded size of each attribute.  For
   invidual fields, we just store their encoded size.  For constructed
   objects, we stack the attribute list entry where the constructed object
   starts and, until we reach the end of the constructed object, accumulate
   the total size of the fields which make up the object.  When we reach the
   end of the object, we unstack the pointer to the attribute list and store
   the total size in it.

   To handle nested constructed objects, we only update the size of the
   topmost item on the stack.  When this is unstacked, we add the size of
   that entry, plus the size of its tag and length information, to the next
   entry on the stack.

   In addition to updating the size, we also record the sequence of table
   entries which are required to encode the constructed item.  A worst-case
   sequence of entries would be:

	SEQUENCE {
		SEQUENCE OPTIONAL { ... }		| Not encoded
		SEQUENCE {
			SEQUENCE OPTIONAL { ... }	| Not encoded
			SEQUENCE {
				value
				}
			}
		}

   which contains an alternating sequence of encoded and non-encoded fields.
   Because of this, the validation check performs the complex task of
   recording which table entries are used for the encoding by stacking and
   unstacking them and discarding the ones which evaluate to a zero size
   during the unstacking process.

   Each entry in the stack contains the list item it applies to, the table
   entry which is used to encode the stacked item, and the size of the item */

#define ATTRIBUTE_STACKSIZE		10

typedef struct {
	ATTRIBUTE_LIST *attributeListPtr;	/* List entry that this applies to */
	ATTRIBUTE_INFO *attributeInfoPtr;	/* Encoding point for sequence */
	int size;							/* Size of sequence */
	} ATTRIBUTE_STACK;

#include <assert.h>					/*!!!!!!!!*/

/* Once we reach the end of the constructed item, we need to unwind the stack
   and update everything we've gone past.  If it's an optional item (so that
   nothing gets encoded), we don't do anything.  The count argument specifies
   the level of unwinding to perform, this can be relative (in which case we
   undo 'count' levels of nesting, which may be more than count stack
   positions of non-nested data was stacked) or absolute (in which case we
   undo 'count' stack positions */

static void updateStackedInfo( ATTRIBUTE_STACK *stack, int *stackPosPtr,
							   int count, const BOOLEAN isRelative )
	{
	int stackPos = *stackPosPtr;

assert( count <= stackPos );		/*!!!!!!!!*/

	while( count-- )
		{
		ATTRIBUTE_LIST *attributeFifoPtr = stack[ --stackPos ].attributeListPtr;
		const ATTRIBUTE_INFO *attributeInfoPtr = stack[ stackPos ].attributeInfoPtr;
		const int size = stack[ stackPos ].size;

assert( attributeInfoPtr != NULL );	/*!!!!!!!!*/
assert( size >= 0 );				/*!!!!!!!!*/
assert( count >= 0 );				/*!!!!!!!!*/
assert( stackPos >= 0 );			/*!!!!!!!!*/

		/* If there's nothing to encode, continue.  There are a few special
		   cases here where even if the sequence is of zero length, we may
		   have to encode something.  Firstly, if there's a member with a
		   default value present (resulting in nothing being encoded) we still
		   have to encode a zero-length sequence.  In addition if all the
		   members have non-encoding values (eg OID's and fixed attributes,
		   none of which are specified by the user) we have to encode these
		   even though there's no actual value associated with them since
		   their mere presence conveys the necessary information.

		   In addition sometimes we can reach the end of the attribute list
		   but there are further actions defined in the encoding table (for
		   example cleanup actions in nested sequences).  In this case the
		   stacked attributeFifoPtr is NULL and the size is zero, so we
		   perform an additional check to make sure the pointer is non-null */
		if( !( size || ( attributeFifoPtr != NULL && \
						 ( attributeFifoPtr->isDefaultValue || \
						   attributeInfoPtr->flags & FL_NONENCODING ) ) ) )
			continue;

assert( attributeFifoPtr != NULL );	/*!!!!!!!!*/

		/* Remember the size and table entry used to encode this stack entry */
		attributeFifoPtr->sizeFifo[ attributeFifoPtr->fifoEnd ] = size;
		attributeFifoPtr->encodingFifo[ attributeFifoPtr->fifoEnd++ ] = \
										stack[ stackPos ].attributeInfoPtr;

		/* If there are no further items on the stack, continue */
		if( !stackPos )
			continue;

		/* If it's a non-constructed field, add the length of the existing and
		   new fields */
		if( attributeInfoPtr->fieldType != BER_SEQUENCE && \
			attributeInfoPtr->fieldType != BER_SET )
			{
			const int newLength = \
					( attributeInfoPtr->fieldType == FIELDTYPE_IDENTIFIER ) ? \
					sizeofOID( attributeInfoPtr->oid ) : \
					( int ) attributeInfoPtr->defaultValue;

			/* Add the new length to the existing data size.  Since this is a
			   non-constructed field, it doesn't count as a reduction in the
			   nesting level, so if we're unnesting by a relative amount we
			   adjust the nesting count to give a net change of zero for this
			   item */
			stack[ stackPos - 1 ].size += size + newLength;
			if( isRelative )
				count++;
			}
		else
			/* It's a constructed field, percolate the encapsulated content
			   size up the stack */
			stack[ stackPos - 1 ].size += ( int ) sizeofObject( size );
		}

	*stackPosPtr = stackPos;
	}

/* Some attributes contain a sequence of items of the attributeTypeAndValue
   form (ie OID, ANY DEFINED BY OID).  To process these, a check is made to
   determine whether the named value component in the attribute list is
   present in the current attributeTypeAndValue definition.  If it isn't, the
   item is given a zero length which means it's never encoded since the field
   is marked as optional.  The following function checks whether a named
   value component is present in the item */

static BOOLEAN checkComponentPresent( const CRYPT_CERTINFO_TYPE fieldID,
									  ATTRIBUTE_INFO **attributeInfoPtrPtr )
	{
	ATTRIBUTE_INFO *attributeInfoPtr = *attributeInfoPtrPtr;
	int nestLevel = 0;

	/* Check each field we find until we find the end of the
	   attributeTypeAndValue */
	while( TRUE )
		{
		/* Adjust the nesting level depending on whether we're entering or
		   leaving a sequence */
		if( attributeInfoPtr->fieldType == BER_SEQUENCE )
			nestLevel++;
		nestLevel -= decodeNestingLevel( attributeInfoPtr->flags );

		/* If the field is present in this attributeTypeAndValue, return */
		if( attributeInfoPtr->fieldID == fieldID )
			return( TRUE );

		/* If we're at the end of the attribute or the attributeTypeAndValue,
		   exit the loop before adjusting the attributeInfoPtr so that we're
		   still pointing at the end-of-attribute field */
		if( nestLevel <= 0 || !( attributeInfoPtr->flags & FL_MORE ) )
			break;

		attributeInfoPtr++;
		}

	/* The field isn't present, update the pointer to the next
	   attributeTypeAndValue or the end of the attribute */
	*attributeInfoPtrPtr = attributeInfoPtr;
	return( FALSE );
	}

/* State machine for checking a CHOICE.  When we get to the start of a
   CHOICE, we move from CHOICE_NONE to CHOICE_START.  Once we've checked one
   of the CHOICE options, we move to CHOICE_DONE.  If a further option is
   found in the CHOICE_DONE state, we record an error.  This is a somewhat
   crude mechanism which works because the only CHOICE fields which can't be
   handled by rewriting them as alternative representations are complete
   attributes so that the CHOICE applies over the entire attribute.  If a
   CHOICE is ever present as an attribute subfield, the checking would be
   handled by recursively checking it as a subtyped field */

typedef enum { CHOICE_NONE, CHOICE_START, CHOICE_DONE } CHOICE_STATE;

/* Check an entry in the attribute table.  While we're performing the check
   we need to pass a lot of state information around, this is contained in
   the following structure */

typedef struct {
	/* State information.  When we're encoding a subtyped field (using an
	   alternative encoding table), we need to remember the field ID of the
	   parent to both tell the encoding routines that we're using an
	   alternative encoding table, and to remember the overall field ID so we
	   don't treat two adjacent field subfields as though they were part of
	   the same parent field.  If we're not currently encoding a subtyped
	   field, this field is set to CRYPT_CERTINFO_NONE */
	ATTRIBUTE_LIST *attributeListPtr;	/* Position in attribute list */
	ATTRIBUTE_INFO *attributeInfoPtr;	/* Position in attribute table */
	CRYPT_CERTINFO_TYPE subtypeParent;	/* Parent of subtype being processed */
	CHOICE_STATE choiceState;			/* State of CHOICE processing */

	/* Encoding stack.  When we're encoding subfields, the stack contains
	   items from both the subfield and the encapsulating field, so we also
	   record the current stack top to make sure we don't go past this level
	   when popping items after we've finished encoding a subfield */
	ATTRIBUTE_STACK *stack;			/* Encoding stack */
	int stackPos;					/* Encoding stack position */
	int stackTop;

	/* Error information */
	CRYPT_CERTINFO_TYPE errorLocus;	/* Error locus */
	CRYPT_CERTERROR_TYPE errorType;	/* Error type */
	} ATTRIBUTE_CHECK_INFO;

static int checkAttribute( ATTRIBUTE_CHECK_INFO *attributeCheckInfo );

static int checkAttributeEntry( ATTRIBUTE_CHECK_INFO *attributeCheckInfo )
	{
	ATTRIBUTE_LIST *attributeListPtr = attributeCheckInfo->attributeListPtr;
	ATTRIBUTE_INFO *attributeInfoPtr = attributeCheckInfo->attributeInfoPtr;
	ATTRIBUTE_STACK *stack = attributeCheckInfo->stack;
	CRYPT_CERTINFO_TYPE fieldID;

	/* Determine the fieldID for the current attribute field */
	if( attributeListPtr == NULL )
		/* If we've reached the end of the list of attributes, use a non-ID
		   which doesn't match any table entry */
		fieldID = CRYPT_UNUSED;
	else
		/* If we're encoding a subtyped field, the fieldID is the field ID
		   within the parent field, or the subFieldID */
		if( attributeCheckInfo->subtypeParent == attributeListPtr->fieldID )
			fieldID = attributeListPtr->subFieldID;
		else
			/* It's a standard attribute field */
			fieldID = attributeListPtr->fieldID;

	/* If the field in the attribute list matches the one in the table,
	   process it and move on to the next one */
	if( attributeListPtr != NULL && attributeInfoPtr->fieldID == fieldID )
		{
		/* If it's a subtyped or CHOICE field, check the components using
		   their own encoding table */
		if( attributeInfoPtr->fieldType == FIELDTYPE_SUBTYPED || \
			attributeInfoPtr->fieldType == FIELDTYPE_CHOICE )
			{
			int status;

			/* Switch to the new encoding table and record the fact that
			   we've done this, and set the new stack top to the level at
			   which we start encoding the subtype */
			if( attributeInfoPtr->fieldType == FIELDTYPE_CHOICE )
				{
				/* Stack the value start position in the attribute list and
				   record the fact that we're processing a CHOICE */
				stack[ attributeCheckInfo->stackPos ].size = 0;
				stack[ attributeCheckInfo->stackPos ].attributeListPtr = attributeListPtr;
				stack[ attributeCheckInfo->stackPos++ ].attributeInfoPtr = attributeInfoPtr;
				attributeCheckInfo->choiceState = CHOICE_START;
				}
			attributeCheckInfo->attributeInfoPtr = \
							( ATTRIBUTE_INFO * ) attributeInfoPtr->extraData;
			attributeCheckInfo->subtypeParent = attributeListPtr->fieldID;
			attributeCheckInfo->stackTop = attributeCheckInfo->stackPos;
			status = checkAttribute( attributeCheckInfo );
			attributeCheckInfo->attributeInfoPtr = attributeInfoPtr;
			attributeCheckInfo->subtypeParent = CRYPT_CERTINFO_NONE;
			attributeCheckInfo->stackTop = 0;
			return( status );
			}

		/* If there's an extended validation function attached to this field,
		   call it */
		if( attributeInfoPtr->extraData != NULL )
			{
			VALIDATION_FUNCTION validationFunction = \
						( VALIDATION_FUNCTION ) attributeInfoPtr->extraData;

			attributeCheckInfo->errorType = validationFunction( attributeListPtr );
			if( attributeCheckInfo->errorType != CRYPT_CERTERROR_NONE )
				return( CRYPT_INVALID );
			}

		/* If this is an optional field and the value is the same as the
		   default value, remember that it doesn't get encoded */
		if( ( attributeInfoPtr->flags & FL_DEFAULT ) && \
			( attributeInfoPtr->defaultValue == attributeListPtr->value ) )
			{
			attributeListPtr->isDefaultValue = TRUE;
			attributeCheckInfo->attributeListPtr = attributeListPtr->next;
			return( CRYPT_OK );
			}

		/* Remember the encoded size of this field */
		attributeListPtr->attributeInfoPtr = attributeInfoPtr;
		attributeListPtr->encodedSize = writeAttributeField( NULL, attributeListPtr );
		if( attributeCheckInfo->stackPos )
			stack[ attributeCheckInfo->stackPos - 1 ].size += attributeListPtr->encodedSize;

		/* If this is a CHOICE field, update the choice state */
		if( attributeCheckInfo->choiceState != CHOICE_NONE )
			{
			if( attributeCheckInfo->choiceState == CHOICE_DONE )
				{
				/* If we've already processed one of the CHOICE options, there
				   can't be another one present */
				attributeCheckInfo->errorType = CRYPT_CERTERROR_PRESENT;
				return( CRYPT_INVALID );
				}
			if( attributeCheckInfo->choiceState == CHOICE_START )
				/* Remember that we've seen a CHOICE option */
				attributeCheckInfo->choiceState = CHOICE_DONE;
			}

		/* Move on to the next attribute field */
		attributeCheckInfo->attributeListPtr = attributeListPtr->next;
		return( CRYPT_OK );
		}

	/* If it's an attributeTypeAndValue sequence, check whether it contains
	   the field we want */
	if( attributeInfoPtr->flags & FL_IDENTIFIER )
		{
		BOOLEAN endOfAttributeField = FALSE;

		if( !checkComponentPresent( fieldID, &attributeInfoPtr ) )
			{
			/* Since we've jumped over several items we may be pointing at an
			   end-of-sequence flag for which no sequence start was stacked,
			   so we skip the stack update step */
			attributeCheckInfo->attributeInfoPtr = attributeInfoPtr;
			return( OK_SPECIAL );
			}

		/* Stack the position of the sequence start and the following OID */
		stack[ attributeCheckInfo->stackPos ].size = 0;
		stack[ attributeCheckInfo->stackPos ].attributeListPtr = attributeListPtr;
		stack[ attributeCheckInfo->stackPos++ ].attributeInfoPtr = attributeInfoPtr++;
		stack[ attributeCheckInfo->stackPos ].size = 0;
		stack[ attributeCheckInfo->stackPos ].attributeListPtr = attributeListPtr;
		stack[ attributeCheckInfo->stackPos++ ].attributeInfoPtr = attributeInfoPtr;

		/* If the OID entry is marked as the end-of-sequence, there are no
		   parameters attached so we move on to the next entry */
		if( attributeInfoPtr->flags & FL_SEQEND )
			endOfAttributeField = TRUE;

		/* Sometimes the OID is followed by a fixed-value blob field which
		   constitutes parameters for the OID, if this is present we stack it
		   as well */
		if( attributeInfoPtr[ 1 ].flags & FL_NONENCODING )
			{
			attributeInfoPtr++;
			stack[ attributeCheckInfo->stackPos ].size = 0;
			stack[ attributeCheckInfo->stackPos ].attributeListPtr = attributeListPtr;
			stack[ attributeCheckInfo->stackPos++ ].attributeInfoPtr = attributeInfoPtr;

			/* If the fields are fixed-value, we always move on to the next
			   entry since there are no user-supplied parameters present */
			endOfAttributeField = TRUE;
			}

		attributeCheckInfo->attributeInfoPtr = attributeInfoPtr;
		if( endOfAttributeField )
			/* If this is all that needs to be encoded, move on to the next
			   attribute field */
			attributeCheckInfo->attributeListPtr = attributeListPtr->next;
		return( CRYPT_OK );
		}

	/* If it's a sequence/set or a non-encoding value then it's a nop entry
	   used only for encoding purposes and can be skipped, however we need to
	   remember it for later encoding */
	if( attributeInfoPtr->fieldType == BER_SEQUENCE || \
		attributeInfoPtr->fieldType == BER_SET || \
		attributeInfoPtr->flags & FL_NONENCODING )
		{
		/* Stack the sequence or value start position in the attribute list */
		stack[ attributeCheckInfo->stackPos ].size = 0;
		stack[ attributeCheckInfo->stackPos ].attributeListPtr = attributeListPtr;
		stack[ attributeCheckInfo->stackPos++ ].attributeInfoPtr = attributeInfoPtr;
		return( CRYPT_OK );
		}

	/* If it's a non-optional field and the attribute field doesn't match,
	   it's an error - attribute attributeID is missing field
	   attributeInfoPtr->fieldID (optional subfield
	   attributeInfoPtr->subFieldID) */
	if( !( attributeInfoPtr->flags & FL_OPTIONAL ) )
		{
		attributeCheckInfo->errorType = CRYPT_CERTERROR_ABSENT;
		return( CRYPT_NOTINITED );
		}

	return( CRYPT_OK );
	}

/* Check an individual attribute */

static int checkAttribute( ATTRIBUTE_CHECK_INFO *attributeCheckInfo )
	{
	BOOLEAN attributeContinues;

	/* Step through the attribute comparing the fields which are present in
	   the attribute list with the fields which should be present according
	   to the table, and set encoding sync points as required */
	do
		{
		int status;

		/* Check the current encoding table entry */
		status = checkAttributeEntry( attributeCheckInfo );
		if( cryptStatusError( status ) )
			{
			attributeCheckInfo->errorLocus = attributeCheckInfo->attributeInfoPtr->fieldID;
			return( status );
			}

		/* If this is the end of a constructed item, unstack it and update
		   the attribute list entry with the length information.  If it's a
		   sequence with all fields optional (so that nothing gets encoded),
		   we don't do anything */
		if( status != OK_SPECIAL )
			updateStackedInfo( attributeCheckInfo->stack,
				&attributeCheckInfo->stackPos,
				decodeNestingLevel( attributeCheckInfo->attributeInfoPtr->flags ),
				TRUE );

		/* Move on to the next table entry.  We have to check the
		   continuation flag before we move to the next table entry in order
		   to include processing of the last field in an attribute */
		attributeContinues = ( attributeCheckInfo->attributeInfoPtr->flags & FL_MORE ) ? TRUE : FALSE;
		attributeCheckInfo->attributeInfoPtr++;
		}
	while( attributeContinues );
	attributeCheckInfo->choiceState = CHOICE_NONE;

	/* We've reached the end of the attribute, if there are still constructed
	   objects stacked, unstack them and update their length information.  If
	   it's a sequence with all fields optional (so that nothing gets
	   encoded), we don't do anything */
	updateStackedInfo( attributeCheckInfo->stack, &attributeCheckInfo->stackPos,
					   attributeCheckInfo->stackPos - attributeCheckInfo->stackTop,
					   FALSE );

	return( CRYPT_OK );
	}

/* Check the entire list of attributes */

int checkAttributes( const ATTRIBUTE_TYPE attributeType,
					 const ATTRIBUTE_LIST *listHeadPtr,
					 CRYPT_CERTINFO_TYPE *errorLocus,
					 CRYPT_CERTERROR_TYPE *errorType )
	{
	ATTRIBUTE_CHECK_INFO attributeCheckInfo;
	const ATTRIBUTE_INFO *attributeInfoStartPtr = \
		( attributeType == ATTRIBUTE_CMS ) ? cmsAttributeInfo : extensionInfo;
	ATTRIBUTE_LIST *attributeListPtr;
	ATTRIBUTE_STACK stack[ ATTRIBUTE_STACKSIZE ];

	/* If we've already done a validation pass, some of the fields will
	   contain values which were previously set, so before we begin we walk
	   down the last resetting the fields which are updated by this
	   function */
	for( attributeListPtr = ( ATTRIBUTE_LIST * ) listHeadPtr;
		 attributeListPtr != NULL && attributeListPtr->fieldID != CRYPT_CERTINFO_NONE;
		 attributeListPtr = attributeListPtr->next )
		{
		attributeListPtr->attributeInfoPtr = NULL;
		attributeListPtr->encodedSize = attributeListPtr->fifoPos = \
			attributeListPtr->fifoEnd = 0;
		attributeListPtr->isDefaultValue = FALSE;
		}

	/* Set up the attribute-checking state information */
	memset( &attributeCheckInfo, 0, sizeof( ATTRIBUTE_CHECK_INFO ) );
	attributeCheckInfo.attributeListPtr = ( ATTRIBUTE_LIST * ) listHeadPtr;
	attributeCheckInfo.attributeInfoPtr = ( ATTRIBUTE_INFO * ) attributeInfoStartPtr;
	attributeCheckInfo.stack = stack;

	/* Walk down the list of known attributes checking each one for
	   consistency */
	while( attributeCheckInfo.attributeListPtr != NULL && \
		   attributeCheckInfo.attributeListPtr->fieldID != CRYPT_CERTINFO_NONE )
		{
		int status;

		/* Find the start of this attribute in the attribute info table and
		   remember it as an encoding sync point.  Comparing the field ID
		   with the attribute ID is usually valid because the attribute info
		   table always begins the series of entries for an attribute with
		   the attribute ID.  The one exception is where the attribute ID is
		   the same as the field ID but they're separate entries in the
		   table, in which case the first entries will contain a
		   FIELDID_FOLLOWS code to indicate that a following field contains
		   the attribute/fieldID */
		while( attributeCheckInfo.attributeInfoPtr->fieldID != \
			   attributeCheckInfo.attributeListPtr->attributeID )
			attributeCheckInfo.attributeInfoPtr++;
		while( attributeCheckInfo.attributeInfoPtr != attributeInfoStartPtr && \
			   attributeCheckInfo.attributeInfoPtr[ -1 ].fieldID == FIELDID_FOLLOWS )
			attributeCheckInfo.attributeInfoPtr--;

		/* Check this attribute */
		status = checkAttribute( &attributeCheckInfo );
		if( cryptStatusError( status ) )
			{
			*errorLocus = attributeCheckInfo.errorLocus;
			*errorType = attributeCheckInfo.errorType;
			return( status );
			}
		}

	return( CRYPT_OK );
	}
