/****************************************************************************
*																			*
*					cryptlib Diffie-Hellman Key Exchange Routines			*
*						Copyright Peter Gutmann 1995-1998					*
*																			*
****************************************************************************/

#include <stdlib.h>
#include <string.h>
#include "crypt.h"
#include "cryptctx.h"

/* Prototypes for functions in lib_kg.c */

int getDLExpSize( const int primeBits );
int generateDLvalues( BIGNUM *p, const int pBits, BIGNUM *q, int qBits,
					  BIGNUM *g, CRYPT_INFO *cryptInfo );

/* The DH key exchange process is somewhat complex since we can either use
   pre-agreed parameters on both sides (which is somewhat inflexible) or have
   one side read the exported parameters from the other side and work with
   those.  In this case the exchange is as follows:

	A.load:		set p, g from fixed or external values
				x(A) = rand

	A.export	y(A) = g^x(A) mod p		error if y != 0
				output = y(A)

				read p, g
	B.load		set p, g from external values
				x(B) = rand

	B.import	y(A) = input
				z = y(A)^x(B) mod p

	B.export	y(B) = g^x(B) mod p		error if y != 0
				output = y(B)

	A.import	y(B) = input
				z = y(B)^x(A) mod p

   Note that we have to set x when we load p and g because otherwise we'd
   have to set x(A) on export and x(B) on import, which is tricky since the
   DH code doesn't know whether it's working with A or B */

/****************************************************************************
*																			*
*						Predefined DH p and g Parameters					*
*																			*
****************************************************************************/

/* Diffie-Hellman public values, generated by Colin Plumb for SKIP (these are
   also use in PGPPhone and PGP 5).  These values were chosen as
   representative well-known public values to allay fears of possible
   trapdoors in self-generated values.  The generation method and actual
   values can be checked against the SKIP standards document or (more
   recently) the PGP 5 source code.  The time figures, except for the
   4096-bit prime, are for a 50 MHz SPARC processor.

   These values can also be used as shared parameters for other discrete-log
   based systems such as ElGamal */

/* 512-bit time = 22s */

const BYTE FAR_BSS prime512[] = {
	0xF5, 0x2A, 0xFF, 0x3C, 0xE1, 0xB1, 0x29, 0x40,
	0x18, 0x11, 0x8D, 0x7C, 0x84, 0xA7, 0x0A, 0x72,
	0xD6, 0x86, 0xC4, 0x03, 0x19, 0xC8, 0x07, 0x29,
	0x7A, 0xCA, 0x95, 0x0C, 0xD9, 0x96, 0x9F, 0xAB,
	0xD0, 0x0A, 0x50, 0x9B, 0x02, 0x46, 0xD3, 0x08,
	0x3D, 0x66, 0xA4, 0x5D, 0x41, 0x9F, 0x9C, 0x7C,
	0xBD, 0x89, 0x4B, 0x22, 0x19, 0x26, 0xBA, 0xAB,
	0xA2, 0x5E, 0xC3, 0x55, 0xE9, 0x2A, 0x05, 0x5F
	};

/* 768-bit time = 105s (1m 45s) */

const BYTE FAR_BSS prime768[] = {
	0xF3, 0xC9, 0x23, 0xC0, 0x43, 0xF0, 0xA5, 0x5B,
	0x18, 0x8D, 0x8E, 0xBB, 0x55, 0x8C, 0xB8, 0x5D,
	0x38, 0xD3, 0x34, 0xFD, 0x7C, 0x17, 0x57, 0x43,
	0xA3, 0x1D, 0x18, 0x6C, 0xDE, 0x33, 0x21, 0x2C,
	0xB5, 0x2A, 0xFF, 0x3C, 0xE1, 0xB1, 0x29, 0x40,
	0x18, 0x11, 0x8D, 0x7C, 0x84, 0xA7, 0x0A, 0x72,
	0xD6, 0x86, 0xC4, 0x03, 0x19, 0xC8, 0x07, 0x29,
	0x7A, 0xCA, 0x95, 0x0C, 0xD9, 0x96, 0x9F, 0xAB,
	0xD0, 0x0A, 0x50, 0x9B, 0x02, 0x46, 0xD3, 0x08,
	0x3D, 0x66, 0xA4, 0x5D, 0x41, 0x9F, 0x9C, 0x7C,
	0xBD, 0x89, 0x4B, 0x22, 0x19, 0x26, 0xBA, 0xAB,
	0xA2, 0x5E, 0xC3, 0x55, 0xE9, 0x2B, 0x1A, 0xB3
	};

/* 1024-bit time = 499s (8m 19s) */

const BYTE FAR_BSS prime1024[] = {
	0xF4, 0x88, 0xFD, 0x58, 0x4E, 0x49, 0xDB, 0xCD,
	0x20, 0xB4, 0x9D, 0xE4, 0x91, 0x07, 0x36, 0x6B,
	0x33, 0x6C, 0x38, 0x0D, 0x45, 0x1D, 0x0F, 0x7C,
	0x88, 0xB3, 0x1C, 0x7C, 0x5B, 0x2D, 0x8E, 0xF6,
	0xF3, 0xC9, 0x23, 0xC0, 0x43, 0xF0, 0xA5, 0x5B,
	0x18, 0x8D, 0x8E, 0xBB, 0x55, 0x8C, 0xB8, 0x5D,
	0x38, 0xD3, 0x34, 0xFD, 0x7C, 0x17, 0x57, 0x43,
	0xA3, 0x1D, 0x18, 0x6C, 0xDE, 0x33, 0x21, 0x2C,
	0xB5, 0x2A, 0xFF, 0x3C, 0xE1, 0xB1, 0x29, 0x40,
	0x18, 0x11, 0x8D, 0x7C, 0x84, 0xA7, 0x0A, 0x72,
	0xD6, 0x86, 0xC4, 0x03, 0x19, 0xC8, 0x07, 0x29,
	0x7A, 0xCA, 0x95, 0x0C, 0xD9, 0x96, 0x9F, 0xAB,
	0xD0, 0x0A, 0x50, 0x9B, 0x02, 0x46, 0xD3, 0x08,
	0x3D, 0x66, 0xA4, 0x5D, 0x41, 0x9F, 0x9C, 0x7C,
	0xBD, 0x89, 0x4B, 0x22, 0x19, 0x26, 0xBA, 0xAB,
	0xA2, 0x5E, 0xC3, 0x55, 0xE9, 0x2F, 0x78, 0xC7
	};

/* 1280-bit time = 632s (10m 32s) */

const BYTE FAR_BSS prime1280[] = {
	0xE1, 0xCE, 0x7C, 0xC9, 0x66, 0xFF, 0x15, 0xF9,
	0xBB, 0xFD, 0x91, 0x5E, 0xC7, 0x01, 0xAA, 0xD3,
	0x5B, 0x9E, 0x8D, 0xA0, 0xA5, 0x72, 0x3A, 0xD4,
	0x1A, 0xF0, 0xBF, 0x46, 0x00, 0x58, 0x2B, 0xE5,
	0xF4, 0x88, 0xFD, 0x58, 0x4E, 0x49, 0xDB, 0xCD,
	0x20, 0xB4, 0x9D, 0xE4, 0x91, 0x07, 0x36, 0x6B,
	0x33, 0x6C, 0x38, 0x0D, 0x45, 0x1D, 0x0F, 0x7C,
	0x88, 0xB3, 0x1C, 0x7C, 0x5B, 0x2D, 0x8E, 0xF6,
	0xF3, 0xC9, 0x23, 0xC0, 0x43, 0xF0, 0xA5, 0x5B,
	0x18, 0x8D, 0x8E, 0xBB, 0x55, 0x8C, 0xB8, 0x5D,
	0x38, 0xD3, 0x34, 0xFD, 0x7C, 0x17, 0x57, 0x43,
	0xA3, 0x1D, 0x18, 0x6C, 0xDE, 0x33, 0x21, 0x2C,
	0xB5, 0x2A, 0xFF, 0x3C, 0xE1, 0xB1, 0x29, 0x40,
	0x18, 0x11, 0x8D, 0x7C, 0x84, 0xA7, 0x0A, 0x72,
	0xD6, 0x86, 0xC4, 0x03, 0x19, 0xC8, 0x07, 0x29,
	0x7A, 0xCA, 0x95, 0x0C, 0xD9, 0x96, 0x9F, 0xAB,
	0xD0, 0x0A, 0x50, 0x9B, 0x02, 0x46, 0xD3, 0x08,
	0x3D, 0x66, 0xA4, 0x5D, 0x41, 0x9F, 0x9C, 0x7C,
	0xBD, 0x89, 0x4B, 0x22, 0x19, 0x26, 0xBA, 0xAB,
	0xA2, 0x5E, 0xC3, 0x55, 0xE9, 0x2C, 0xA6, 0x6F
	};

/* 1536-bit time = 10787s (2h 59m 47s) */

const BYTE FAR_BSS prime1536[] = {
	0xF8, 0xF5, 0x59, 0x5F, 0xDC, 0x76, 0x52, 0x4F,
	0x3D, 0x3D, 0x8D, 0xDB, 0xCE, 0x99, 0xE1, 0x57,
	0x92, 0x59, 0xCD, 0xFD, 0xB8, 0xAE, 0x74, 0x4F,
	0xC5, 0xFC, 0x76, 0xBC, 0x83, 0xC5, 0x47, 0x30,
	0x61, 0xCE, 0x7C, 0xC9, 0x66, 0xFF, 0x15, 0xF9,
	0xBB, 0xFD, 0x91, 0x5E, 0xC7, 0x01, 0xAA, 0xD3,
	0x5B, 0x9E, 0x8D, 0xA0, 0xA5, 0x72, 0x3A, 0xD4,
	0x1A, 0xF0, 0xBF, 0x46, 0x00, 0x58, 0x2B, 0xE5,
	0xF4, 0x88, 0xFD, 0x58, 0x4E, 0x49, 0xDB, 0xCD,
	0x20, 0xB4, 0x9D, 0xE4, 0x91, 0x07, 0x36, 0x6B,
	0x33, 0x6C, 0x38, 0x0D, 0x45, 0x1D, 0x0F, 0x7C,
	0x88, 0xB3, 0x1C, 0x7C, 0x5B, 0x2D, 0x8E, 0xF6,
	0xF3, 0xC9, 0x23, 0xC0, 0x43, 0xF0, 0xA5, 0x5B,
	0x18, 0x8D, 0x8E, 0xBB, 0x55, 0x8C, 0xB8, 0x5D,
	0x38, 0xD3, 0x34, 0xFD, 0x7C, 0x17, 0x57, 0x43,
	0xA3, 0x1D, 0x18, 0x6C, 0xDE, 0x33, 0x21, 0x2C,
	0xB5, 0x2A, 0xFF, 0x3C, 0xE1, 0xB1, 0x29, 0x40,
	0x18, 0x11, 0x8D, 0x7C, 0x84, 0xA7, 0x0A, 0x72,
	0xD6, 0x86, 0xC4, 0x03, 0x19, 0xC8, 0x07, 0x29,
	0x7A, 0xCA, 0x95, 0x0C, 0xD9, 0x96, 0x9F, 0xAB,
	0xD0, 0x0A, 0x50, 0x9B, 0x02, 0x46, 0xD3, 0x08,
	0x3D, 0x66, 0xA4, 0x5D, 0x41, 0x9F, 0x9C, 0x7C,
	0xBD, 0x89, 0x4B, 0x22, 0x19, 0x26, 0xBA, 0xAB,
	0xA2, 0x5E, 0xC3, 0x55, 0xE9, 0x58, 0x7C, 0x47
	};

/* 2048-bit time = 5010s (1h 23m 30s) */

const BYTE FAR_BSS prime2048[] = {
	0xF6, 0x42, 0x57, 0xB7, 0x08, 0x7F, 0x08, 0x17,
	0x72, 0xA2, 0xBA, 0xD6, 0xA9, 0x42, 0xF3, 0x05,
	0xE8, 0xF9, 0x53, 0x11, 0x39, 0x4F, 0xB6, 0xF1,
	0x6E, 0xB9, 0x4B, 0x38, 0x20, 0xDA, 0x01, 0xA7,
	0x56, 0xA3, 0x14, 0xE9, 0x8F, 0x40, 0x55, 0xF3,
	0xD0, 0x07, 0xC6, 0xCB, 0x43, 0xA9, 0x94, 0xAD,
	0xF7, 0x4C, 0x64, 0x86, 0x49, 0xF8, 0x0C, 0x83,
	0xBD, 0x65, 0xE9, 0x17, 0xD4, 0xA1, 0xD3, 0x50,
	0xF8, 0xF5, 0x59, 0x5F, 0xDC, 0x76, 0x52, 0x4F,
	0x3D, 0x3D, 0x8D, 0xDB, 0xCE, 0x99, 0xE1, 0x57,
	0x92, 0x59, 0xCD, 0xFD, 0xB8, 0xAE, 0x74, 0x4F,
	0xC5, 0xFC, 0x76, 0xBC, 0x83, 0xC5, 0x47, 0x30,
	0x61, 0xCE, 0x7C, 0xC9, 0x66, 0xFF, 0x15, 0xF9,
	0xBB, 0xFD, 0x91, 0x5E, 0xC7, 0x01, 0xAA, 0xD3,
	0x5B, 0x9E, 0x8D, 0xA0, 0xA5, 0x72, 0x3A, 0xD4,
	0x1A, 0xF0, 0xBF, 0x46, 0x00, 0x58, 0x2B, 0xE5,
	0xF4, 0x88, 0xFD, 0x58, 0x4E, 0x49, 0xDB, 0xCD,
	0x20, 0xB4, 0x9D, 0xE4, 0x91, 0x07, 0x36, 0x6B,
	0x33, 0x6C, 0x38, 0x0D, 0x45, 0x1D, 0x0F, 0x7C,
	0x88, 0xB3, 0x1C, 0x7C, 0x5B, 0x2D, 0x8E, 0xF6,
	0xF3, 0xC9, 0x23, 0xC0, 0x43, 0xF0, 0xA5, 0x5B,
	0x18, 0x8D, 0x8E, 0xBB, 0x55, 0x8C, 0xB8, 0x5D,
	0x38, 0xD3, 0x34, 0xFD, 0x7C, 0x17, 0x57, 0x43,
	0xA3, 0x1D, 0x18, 0x6C, 0xDE, 0x33, 0x21, 0x2C,
	0xB5, 0x2A, 0xFF, 0x3C, 0xE1, 0xB1, 0x29, 0x40,
	0x18, 0x11, 0x8D, 0x7C, 0x84, 0xA7, 0x0A, 0x72,
	0xD6, 0x86, 0xC4, 0x03, 0x19, 0xC8, 0x07, 0x29,
	0x7A, 0xCA, 0x95, 0x0C, 0xD9, 0x96, 0x9F, 0xAB,
	0xD0, 0x0A, 0x50, 0x9B, 0x02, 0x46, 0xD3, 0x08,
	0x3D, 0x66, 0xA4, 0x5D, 0x41, 0x9F, 0x9C, 0x7C,
	0xBD, 0x89, 0x4B, 0x22, 0x19, 0x26, 0xBA, 0xAB,
	0xA2, 0x5E, 0xC3, 0x55, 0xE9, 0x32, 0x0B, 0x3B
	};

/* 3072-bit time = 58391s (16h 13m 11s) */

const BYTE FAR_BSS prime3072[] = {
	0xCC, 0x1D, 0x77, 0x57, 0x24, 0x38, 0x4A, 0xE2,
	0xC4, 0xF0, 0xE8, 0x8E, 0x13, 0x67, 0x97, 0x4E,
	0x92, 0x13, 0x61, 0xF6, 0xDB, 0xEB, 0x25, 0x0E,
	0x17, 0xFD, 0xF6, 0x98, 0xF7, 0xC8, 0x7C, 0x79,
	0xB0, 0x72, 0x1D, 0x38, 0x75, 0xFB, 0xF6, 0xC1,
	0x73, 0xC4, 0x83, 0x11, 0x26, 0x2B, 0x43, 0x60,
	0xC3, 0xE3, 0xE8, 0xD6, 0x0A, 0xFD, 0xA1, 0x28,
	0x26, 0x0B, 0xAE, 0xA9, 0xAE, 0xB3, 0x65, 0x0F,
	0xA2, 0x00, 0x53, 0x01, 0xA0, 0x7C, 0xD6, 0xAB,
	0xA3, 0x12, 0x1E, 0xFA, 0x0F, 0x2A, 0xCE, 0x1F,
	0x74, 0x84, 0x4F, 0xCA, 0xF3, 0x17, 0xF3, 0xA4,
	0x40, 0xE9, 0xD7, 0xD2, 0x77, 0xB6, 0x42, 0x2D,
	0x02, 0x36, 0xC1, 0x26, 0xCB, 0x68, 0x5E, 0x9D,
	0x7C, 0x98, 0x09, 0x0A, 0x8D, 0x7E, 0x2D, 0xED,
	0xE4, 0x5D, 0x79, 0xF5, 0xD4, 0x92, 0x4F, 0x9B,
	0x18, 0x8E, 0xFC, 0x2A, 0xA7, 0x4B, 0x7C, 0x32,
	0xF6, 0x42, 0x57, 0xB7, 0x08, 0x7F, 0x08, 0x17,
	0x72, 0xA2, 0xBA, 0xD6, 0xA9, 0x42, 0xF3, 0x05,
	0xE8, 0xF9, 0x53, 0x11, 0x39, 0x4F, 0xB6, 0xF1,
	0x6E, 0xB9, 0x4B, 0x38, 0x20, 0xDA, 0x01, 0xA7,
	0x56, 0xA3, 0x14, 0xE9, 0x8F, 0x40, 0x55, 0xF3,
	0xD0, 0x07, 0xC6, 0xCB, 0x43, 0xA9, 0x94, 0xAD,
	0xF7, 0x4C, 0x64, 0x86, 0x49, 0xF8, 0x0C, 0x83,
	0xBD, 0x65, 0xE9, 0x17, 0xD4, 0xA1, 0xD3, 0x50,
	0xF8, 0xF5, 0x59, 0x5F, 0xDC, 0x76, 0x52, 0x4F,
	0x3D, 0x3D, 0x8D, 0xDB, 0xCE, 0x99, 0xE1, 0x57,
	0x92, 0x59, 0xCD, 0xFD, 0xB8, 0xAE, 0x74, 0x4F,
	0xC5, 0xFC, 0x76, 0xBC, 0x83, 0xC5, 0x47, 0x30,
	0x61, 0xCE, 0x7C, 0xC9, 0x66, 0xFF, 0x15, 0xF9,
	0xBB, 0xFD, 0x91, 0x5E, 0xC7, 0x01, 0xAA, 0xD3,
	0x5B, 0x9E, 0x8D, 0xA0, 0xA5, 0x72, 0x3A, 0xD4,
	0x1A, 0xF0, 0xBF, 0x46, 0x00, 0x58, 0x2B, 0xE5,
	0xF4, 0x88, 0xFD, 0x58, 0x4E, 0x49, 0xDB, 0xCD,
	0x20, 0xB4, 0x9D, 0xE4, 0x91, 0x07, 0x36, 0x6B,
	0x33, 0x6C, 0x38, 0x0D, 0x45, 0x1D, 0x0F, 0x7C,
	0x88, 0xB3, 0x1C, 0x7C, 0x5B, 0x2D, 0x8E, 0xF6,
	0xF3, 0xC9, 0x23, 0xC0, 0x43, 0xF0, 0xA5, 0x5B,
	0x18, 0x8D, 0x8E, 0xBB, 0x55, 0x8C, 0xB8, 0x5D,
	0x38, 0xD3, 0x34, 0xFD, 0x7C, 0x17, 0x57, 0x43,
	0xA3, 0x1D, 0x18, 0x6C, 0xDE, 0x33, 0x21, 0x2C,
	0xB5, 0x2A, 0xFF, 0x3C, 0xE1, 0xB1, 0x29, 0x40,
	0x18, 0x11, 0x8D, 0x7C, 0x84, 0xA7, 0x0A, 0x72,
	0xD6, 0x86, 0xC4, 0x03, 0x19, 0xC8, 0x07, 0x29,
	0x7A, 0xCA, 0x95, 0x0C, 0xD9, 0x96, 0x9F, 0xAB,
	0xD0, 0x0A, 0x50, 0x9B, 0x02, 0x46, 0xD3, 0x08,
	0x3D, 0x66, 0xA4, 0x5D, 0x41, 0x9F, 0x9C, 0x7C,
	0xBD, 0x89, 0x4B, 0x22, 0x19, 0x26, 0xBA, 0xAB,
	0xA2, 0x5E, 0xC3, 0x55, 0xE9, 0x4C, 0x32, 0x6F
	};

/* This prime was generated on a 275 MHz DEC Alpha - and it took 76 hours
   (more than 3 days) on *that*.  It performed 59592 modular exponentiations
   (dots printed) before it found the prime.  You *can* verify it yourself,
   and I hope that someone does, but be warned!  It will take a *long* time.
   (You can time the dots and multiply by 60,000 to estimate how long).
   Another run, started several days earlier on a 50 MHz SPARC, has used 12
   days and 18 hours *of CPU time* and is not finished yet */

/* 4096-bit time = 274903.168s (76h 21m 43s) */

const BYTE FAR_BSS prime4096[] = {
	0xF9, 0x18, 0xA0, 0x7E, 0x5A, 0x06, 0x61, 0x7A,
	0x43, 0x90, 0x95, 0xDC, 0x05, 0x6C, 0x87, 0x86,
	0xEC, 0x61, 0xEC, 0xCD, 0x45, 0x1F, 0x0E, 0xD8,
	0xE0, 0xA3, 0x79, 0xC6, 0xC9, 0xDC, 0x7A, 0x0B,
	0xAC, 0xE4, 0x3F, 0xE3, 0x46, 0x94, 0xB6, 0x30,
	0x4A, 0x53, 0xD7, 0x7C, 0x02, 0x16, 0x48, 0x80,
	0xB5, 0x15, 0xE5, 0x29, 0x99, 0xA9, 0x9F, 0x07,
	0x74, 0xD3, 0xFF, 0xE3, 0xA1, 0xC5, 0x96, 0x20,
	0x4E, 0x98, 0x65, 0xB8, 0xD8, 0x0D, 0xEE, 0x10,
	0x5D, 0xAB, 0xB6, 0x17, 0x1C, 0x51, 0xD8, 0x50,
	0xCA, 0x22, 0x57, 0x43, 0x29, 0xBE, 0x95, 0xE8,
	0x56, 0x2B, 0x38, 0x78, 0x5C, 0x0B, 0xDB, 0xF8,
	0x4C, 0x4D, 0xD5, 0xE3, 0xAA, 0x46, 0xCC, 0xFB,
	0xCE, 0x17, 0xE8, 0x2A, 0x9D, 0x14, 0x61, 0xE3,
	0x84, 0xA9, 0x4F, 0xD1, 0x83, 0x84, 0xA8, 0x79,
	0xB6, 0xEF, 0x8F, 0xA7, 0x43, 0x46, 0x08, 0xC6,
	0xCC, 0x1D, 0x77, 0x57, 0x24, 0x38, 0x4A, 0xE2,
	0xC4, 0xF0, 0xE8, 0x8E, 0x13, 0x67, 0x97, 0x4E,
	0x92, 0x13, 0x61, 0xF6, 0xDB, 0xEB, 0x25, 0x0E,
	0x17, 0xFD, 0xF6, 0x98, 0xF7, 0xC8, 0x7C, 0x79,
	0xB0, 0x72, 0x1D, 0x38, 0x75, 0xFB, 0xF6, 0xC1,
	0x73, 0xC4, 0x83, 0x11, 0x26, 0x2B, 0x43, 0x60,
	0xC3, 0xE3, 0xE8, 0xD6, 0x0A, 0xFD, 0xA1, 0x28,
	0x26, 0x0B, 0xAE, 0xA9, 0xAE, 0xB3, 0x65, 0x0F,
	0xA2, 0x00, 0x53, 0x01, 0xA0, 0x7C, 0xD6, 0xAB,
	0xA3, 0x12, 0x1E, 0xFA, 0x0F, 0x2A, 0xCE, 0x1F,
	0x74, 0x84, 0x4F, 0xCA, 0xF3, 0x17, 0xF3, 0xA4,
	0x40, 0xE9, 0xD7, 0xD2, 0x77, 0xB6, 0x42, 0x2D,
	0x02, 0x36, 0xC1, 0x26, 0xCB, 0x68, 0x5E, 0x9D,
	0x7C, 0x98, 0x09, 0x0A, 0x8D, 0x7E, 0x2D, 0xED,
	0xE4, 0x5D, 0x79, 0xF5, 0xD4, 0x92, 0x4F, 0x9B,
	0x18, 0x8E, 0xFC, 0x2A, 0xA7, 0x4B, 0x7C, 0x32,
	0xF6, 0x42, 0x57, 0xB7, 0x08, 0x7F, 0x08, 0x17,
	0x72, 0xA2, 0xBA, 0xD6, 0xA9, 0x42, 0xF3, 0x05,
	0xE8, 0xF9, 0x53, 0x11, 0x39, 0x4F, 0xB6, 0xF1,
	0x6E, 0xB9, 0x4B, 0x38, 0x20, 0xDA, 0x01, 0xA7,
	0x56, 0xA3, 0x14, 0xE9, 0x8F, 0x40, 0x55, 0xF3,
	0xD0, 0x07, 0xC6, 0xCB, 0x43, 0xA9, 0x94, 0xAD,
	0xF7, 0x4C, 0x64, 0x86, 0x49, 0xF8, 0x0C, 0x83,
	0xBD, 0x65, 0xE9, 0x17, 0xD4, 0xA1, 0xD3, 0x50,
	0xF8, 0xF5, 0x59, 0x5F, 0xDC, 0x76, 0x52, 0x4F,
	0x3D, 0x3D, 0x8D, 0xDB, 0xCE, 0x99, 0xE1, 0x57,
	0x92, 0x59, 0xCD, 0xFD, 0xB8, 0xAE, 0x74, 0x4F,
	0xC5, 0xFC, 0x76, 0xBC, 0x83, 0xC5, 0x47, 0x30,
	0x61, 0xCE, 0x7C, 0xC9, 0x66, 0xFF, 0x15, 0xF9,
	0xBB, 0xFD, 0x91, 0x5E, 0xC7, 0x01, 0xAA, 0xD3,
	0x5B, 0x9E, 0x8D, 0xA0, 0xA5, 0x72, 0x3A, 0xD4,
	0x1A, 0xF0, 0xBF, 0x46, 0x00, 0x58, 0x2B, 0xE5,
	0xF4, 0x88, 0xFD, 0x58, 0x4E, 0x49, 0xDB, 0xCD,
	0x20, 0xB4, 0x9D, 0xE4, 0x91, 0x07, 0x36, 0x6B,
	0x33, 0x6C, 0x38, 0x0D, 0x45, 0x1D, 0x0F, 0x7C,
	0x88, 0xB3, 0x1C, 0x7C, 0x5B, 0x2D, 0x8E, 0xF6,
	0xF3, 0xC9, 0x23, 0xC0, 0x43, 0xF0, 0xA5, 0x5B,
	0x18, 0x8D, 0x8E, 0xBB, 0x55, 0x8C, 0xB8, 0x5D,
	0x38, 0xD3, 0x34, 0xFD, 0x7C, 0x17, 0x57, 0x43,
	0xA3, 0x1D, 0x18, 0x6C, 0xDE, 0x33, 0x21, 0x2C,
	0xB5, 0x2A, 0xFF, 0x3C, 0xE1, 0xB1, 0x29, 0x40,
	0x18, 0x11, 0x8D, 0x7C, 0x84, 0xA7, 0x0A, 0x72,
	0xD6, 0x86, 0xC4, 0x03, 0x19, 0xC8, 0x07, 0x29,
	0x7A, 0xCA, 0x95, 0x0C, 0xD9, 0x96, 0x9F, 0xAB,
	0xD0, 0x0A, 0x50, 0x9B, 0x02, 0x46, 0xD3, 0x08,
	0x3D, 0x66, 0xA4, 0x5D, 0x41, 0x9F, 0x9C, 0x7C,
	0xBD, 0x89, 0x4B, 0x22, 0x19, 0x26, 0xBA, 0xAB,
	0xA2, 0x5E, 0xC3, 0x55, 0xEB, 0x3D, 0xD6, 0x17
	};

/* The structure for storing the DH public values */

typedef struct {
	const int baseLen; const BYTE base[ 1 ];
	const int primeLen; const BYTE *prime;
	} DH_PUBLIC_VALUES;

static const DH_PUBLIC_VALUES dhPublicValues[] = {
	{ 2, { 0x02 }, 512, prime512 },
	{ 2, { 0x02 }, 768, prime768 },
	{ 2, { 0x02 }, 1024, prime1024 },
	{ 2, { 0x02 }, 1280, prime1280 },
	{ 2, { 0x02 }, 1536, prime1536 },
	{ 2, { 0x02 }, 2048, prime2048 },
	{ 2, { 0x02 }, 3072, prime3072 },
	{ 2, { 0x02 }, 4096, prime4096 },
	{ 0, { 0 }, 0, NULL }
	};

/****************************************************************************
*																			*
*									Utility Routines						*
*																			*
****************************************************************************/

/* Generate the DH private value x */

static int generatePrivateValue( CRYPT_INFO *cryptInfo )
	{
	int status;

	/* Generate the DH private value x s.t. 0 < x < p-1.  Because bnMod() is
	   expensive and would normally never be called (the exponent is much
	   smaller than the prime), we do a quick check to make sure it's really
	   necessary before calling it */
	status = generateBignum( cryptInfo->ctxPKC.dhParam_x,
					getDLExpSize( cryptInfo->ctxPKC.keySizeBits ), 0xC0, 0 );
	if( cryptStatusError( status ) )
		return( status );
	if( BN_num_bits( cryptInfo->ctxPKC.dhParam_x ) > \
										cryptInfo->ctxPKC.keySizeBits - 1 )
		{
		BN_CTX *bnCTX;

		if( ( bnCTX = BN_CTX_new() ) == NULL )
			return( CRYPT_NOMEM );

		BN_mod( cryptInfo->ctxPKC.dhParam_x, cryptInfo->ctxPKC.dhParam_x,
				cryptInfo->ctxPKC.dhParam_p, bnCTX );
		BN_sub_word( cryptInfo->ctxPKC.dhParam_x, 1 );

		BN_CTX_free( bnCTX );
		}

	return( CRYPT_OK );
	}

/****************************************************************************
*																			*
*						Diffie-Hellman Self-test Routines					*
*																			*
****************************************************************************/

/* Test the Diffie-Hellman implementation using a sample key exchange.
   Because a lot of the high-level encryption routines don't exist yet, we
   cheat a bit and set up a dummy encryption context with just enough
   information for the following code to work */

int dhInitKey( CRYPT_INFO *cryptInfo, const void *key, const int keyLength );
int dhEncrypt( CRYPT_INFO *cryptInfo, BYTE *buffer, int noBytes );
int dhDecrypt( CRYPT_INFO *cryptInfo, BYTE *buffer, int noBytes );

int dhSelfTest( void )
	{
	CRYPT_INFO cryptInfo1, cryptInfo2;
	CRYPT_PKCINFO_DH *dhKey;
	const CAPABILITY_INFO capabilityInfo = { CRYPT_ALGO_DH, CRYPT_MODE_PKC,
								0, NULL, NULL, 64, 128, 512, 0, 0, 0,
								NULL, NULL, NULL, NULL, NULL, NULL, NULL,
								NULL, NULL, NULL, NULL, CRYPT_ERROR };
	BYTE buffer1[ 64 ], buffer2[ 64 ];
	int status = CRYPT_OK;

	/* Allocate room for the public-key components */
	if( ( dhKey = ( CRYPT_PKCINFO_DH * ) malloc( sizeof( CRYPT_PKCINFO_DH ) ) ) == NULL )
		return( CRYPT_NOMEM );

	/* Initialise the BigNum information and components */
	cryptInitComponents( dhKey, 512, CRYPT_UNUSED );
	memset( &cryptInfo1, 0, sizeof( CRYPT_INFO ) );
	cryptInfo1.ctxPKC.param1 = BN_new();
	cryptInfo1.ctxPKC.param2 = BN_new();
	cryptInfo1.ctxPKC.param3 = BN_new();
	cryptInfo1.ctxPKC.param4 = BN_new();
	cryptInfo1.ctxPKC.param5 = BN_new();
	cryptInfo1.capabilityInfo = &capabilityInfo;
	memset( &cryptInfo2, 0, sizeof( CRYPT_INFO ) );
	cryptInfo2.ctxPKC.param1 = BN_new();
	cryptInfo2.ctxPKC.param2 = BN_new();
	cryptInfo2.ctxPKC.param3 = BN_new();
	cryptInfo2.ctxPKC.param4 = BN_new();
	cryptInfo2.ctxPKC.param5 = BN_new();
	cryptInfo2.capabilityInfo = ( CAPABILITY_INFO * ) &capabilityInfo;

	/* Perform the test key exchange on a block of data */
	memset( buffer1, 0, 64 );
	memcpy( buffer1, "abcde", 5 );
	memset( buffer2, 0, 64 );
	memcpy( buffer2, "efghi", 5 );
	if( dhInitKey( &cryptInfo1, dhKey, 42 ) != CRYPT_OK ||
		dhInitKey( &cryptInfo2, dhKey, 43 ) != CRYPT_OK ||
		cryptStatusError( dhEncrypt( &cryptInfo1, buffer1, 20 ) ) ||
		cryptStatusError( dhEncrypt( &cryptInfo2, buffer2, 20 ) ) ||
		cryptStatusError( dhDecrypt( &cryptInfo1, buffer2, CRYPT_USE_DEFAULT ) ) ||
		cryptStatusError( dhDecrypt( &cryptInfo2, buffer1, CRYPT_USE_DEFAULT ) ) ||
		memcmp( buffer1, buffer2, 64 ) )
		status = CRYPT_SELFTEST;

	/* Clean up */
	cryptDestroyComponents( dhKey );
	BN_clear_free( cryptInfo1.ctxPKC.param1 );
	BN_clear_free( cryptInfo1.ctxPKC.param2 );
	BN_clear_free( cryptInfo1.ctxPKC.param3 );
	BN_clear_free( cryptInfo1.ctxPKC.param4 );
	BN_clear_free( cryptInfo1.ctxPKC.param5 );
	zeroise( &cryptInfo1, sizeof( CRYPT_INFO ) );
	BN_clear_free( cryptInfo2.ctxPKC.param1 );
	BN_clear_free( cryptInfo2.ctxPKC.param2 );
	BN_clear_free( cryptInfo2.ctxPKC.param3 );
	BN_clear_free( cryptInfo2.ctxPKC.param4 );
	BN_clear_free( cryptInfo2.ctxPKC.param5 );
	zeroise( &cryptInfo2, sizeof( CRYPT_INFO ) );
	free( dhKey );

	return( status );
	}

/****************************************************************************
*																			*
*							Init/Shutdown Routines							*
*																			*
****************************************************************************/

/* Not needed for the DH routines */

/****************************************************************************
*																			*
*						Diffie-Hellman Key Exchange Routines				*
*																			*
****************************************************************************/

/* Perform phase 1 of Diffie-Hellman ("export") */

int dhEncrypt( CRYPT_INFO *cryptInfo, BYTE *buffer, int noBytes )
	{
	BN_CTX *bnCTX;
	int length, status = 0;

	UNUSED( noBytes );

	/* If the caller has already generated and exported the public value y
	   they can't do it a second time so we check for this and return an
	   initialised error */
	if( !BN_is_zero( cryptInfo->ctxPKC.dhParam_y ) )
		return( CRYPT_INITED );

	if( ( bnCTX = BN_CTX_new() ) == NULL )
		return( CRYPT_NOMEM );

	/* Export y = g^x mod p.  There is no input data since x was set when the
	   DH values were loaded */
	BN_mod_exp( cryptInfo->ctxPKC.dhParam_y, cryptInfo->ctxPKC.dhParam_g,
				cryptInfo->ctxPKC.dhParam_x, cryptInfo->ctxPKC.dhParam_p,
				bnCTX );
	length = BN_bn2bin( cryptInfo->ctxPKC.dhParam_y, buffer );

	BN_CTX_free( bnCTX );

	if( noBytes == CRYPT_USE_DEFAULT )
		length = CRYPT_OK;	/* External calls don't return a length */
	return( ( status == -1 ) ? CRYPT_PKCCRYPT : length );
	}

/* Perform phase 2 of Diffie-Hellman ("import") */

int dhDecrypt( CRYPT_INFO *cryptInfo, BYTE *buffer, int noBytes )
	{
	BN_CTX *bnCTX;
	BIGNUM *z;
	int length = ( noBytes == CRYPT_USE_DEFAULT ) ? \
				 bitsToBytes( cryptInfo->ctxPKC.keySizeBits ) : noBytes;
	int status = CRYPT_OK;

	if( ( bnCTX = BN_CTX_new() ) == NULL )
		return( CRYPT_NOMEM );

	/* If we're being called from user code the other parties y value will be
	   in the buffer rather than having been read in when we read the DH
	   public key */
	if( noBytes != CRYPT_UNUSED )
		BN_bin2bn( buffer, length, cryptInfo->ctxPKC.dhParam_yPrime );

	/* Export z = y^x mod p.  We need to use separate y and z values because
	   the bignum code can't handle modexp with the first two parameters the
	   same */
	z = BN_new();
	BN_mod_exp( z, cryptInfo->ctxPKC.dhParam_yPrime,
				cryptInfo->ctxPKC.dhParam_x, cryptInfo->ctxPKC.dhParam_p,
				bnCTX );
	length = BN_bn2bin( z, buffer );
	BN_clear_free( z );

#if 0
	y = BN_new();
	BN_bin2bn( buffer, length, y );
	zeroise( buffer, length );	/* Clear buffer while data is in bignum */
	BN_mod_exp( y, y, cryptInfo->ctxPKC.dhParam_x,
				cryptInfo->ctxPKC.dhParam_p, bnCTX );
	length = BN_bn2bin( y, buffer );
	BN_clear_free( y );
#endif

	BN_CTX_free( bnCTX );

	if( noBytes == CRYPT_UNUSED )
		length = CRYPT_OK;	/* External calls don't return a length */
	return( ( status == -1 ) ? CRYPT_PKCCRYPT : length );
	}

/****************************************************************************
*																			*
*						Diffie-Hellman Key Management Routines				*
*																			*
****************************************************************************/

/* Load DH public key components into an encryption context */

int dhInitKey( CRYPT_INFO *cryptInfo, const void *key, const int loadType )
	{
	CRYPT_PKCINFO_DH *dhKey = ( CRYPT_PKCINFO_DH * ) key;
	int status = CRYPT_OK;

	/* Load the key component from the external representation into the
	   internal BigNums unless we're doing an internal load */
	if( loadType != LOAD_INTERNAL_PUBLIC && loadType != LOAD_INTERNAL_PRIVATE )
		{
		/* Make sure the last parameter is correct.  Sinc the DH parameters
		   are somewhat different from those used for normal PKC's, this is a
		   useful test to make sure the caller hasn't confused things */
		if( dhKey->isPublicKey != CRYPT_UNUSED )
			return( CRYPT_BADPARM3 );

		/* If a key size is given, use the default public values */
		if( dhKey->endianness > CRYPT_COMPONENTS_LITTLENDIAN )
			{
			int index, size = dhKey->endianness;

			/* Determine which parameters to use */
			for( index = 0; dhPublicValues[ index ].primeLen && \
				 dhPublicValues[ index ].primeLen != size; index++ );
			if( !dhPublicValues[ index ].primeLen )
				return( CRYPT_BADPARM2 );

			/* Load them into the bignums */
			BN_bin2bn( ( BYTE * ) dhPublicValues[ index ].prime,
					   bitsToBytes( size ), cryptInfo->ctxPKC.dhParam_p );
			BN_bin2bn( ( BYTE * ) dhPublicValues[ index ].base, 1,
					   cryptInfo->ctxPKC.dhParam_g );
			}
		else
			{
			/* Load the key components into the bignums */
			BN_bin2bn( dhKey->p, bitsToBytes( dhKey->pLen ),
					   cryptInfo->ctxPKC.dhParam_p );
			BN_bin2bn( dhKey->g, bitsToBytes( dhKey->gLen ),
					   cryptInfo->ctxPKC.dhParam_g );
			}
		}
	cryptInfo->ctxPKC.lastPublicComponent = DH_LAST_PUBLIC;
	cryptInfo->ctxPKC.lastPublicMontCTX = 0;

	/* Make sure the necessary key parameters have been initialised */
	if( BN_is_zero( cryptInfo->ctxPKC.dhParam_p ) || \
		BN_is_zero( cryptInfo->ctxPKC.dhParam_g ) )
		return( CRYPT_BADPARM2 );

	/* Set the keysize and generate a key ID for this key */
	cryptInfo->ctxPKC.keySizeBits = BN_num_bits( cryptInfo->ctxPKC.dhParam_p );
	status = calculateKeyID( CRYPT_ALGO_DH, &cryptInfo->ctxPKC,
							 cryptInfo->ctxPKC.keyID );
	if( cryptStatusError( status ) )
		return( status );

	/* Generate the DH private value x.  During the initial self-test
	   the random data pool may not exist yet, and may in fact never exist in
	   a satisfactory condition if there isn't enough randomness present in
	   the system to generate cryptographically strong random numbers.  To
	   bypass this problem, if the caller passes in a loadType value which
	   can't be passed in via a call to cryptLoadKey() we know it's an 
	   internal self-test call and use a fixed bit pattern for k which avoids
	   having to call generateBignum().  This is a somewhat ugly use of 'magic
	   numbers', but it's safe because cryptLoadKey() won't allow any value
	   for loadType than CRYPT_USE_DEFAULT so there's no way an external
	   caller can pass in a value like this */
	if( loadType == 42 || loadType == 43 )
		{
		static const char dh1[] = "abcdefgh";
		static const char dh2[] = "ijklmnop";

		BN_bin2bn( ( loadType == 42 ) ? ( BYTE * ) dh1 : ( BYTE * ) dh2, 8,
				   cryptInfo->ctxPKC.dhParam_x );
		return( CRYPT_OK );
		}

	return( generatePrivateValue( cryptInfo ) );
	}

/****************************************************************************
*																			*
*						Diffie-Hellman Key Generation Routines				*
*																			*
****************************************************************************/

/* Generate a Diffie-Hellman key into an encryption context */

int dhGenerateKey( CRYPT_INFO *cryptInfo, const int keySizeBits )
	{
	int status;

	/* Determine how many bits to give to p */
	cryptInfo->ctxPKC.keySizeBits = keySizeBits;

	/* Generate large prime p.  We don't care about q, and g is fixed at 2 */
	status = generateDLvalues( cryptInfo->ctxPKC.dhParam_p, keySizeBits, NULL, 0,
							   NULL, cryptInfo );
	if( cryptStatusOK( status ) )
		/* Generate the DH private value x */
		status = generatePrivateValue( cryptInfo );
	if( cryptStatusError( status ) )
		return( status );
	BN_set_word( cryptInfo->ctxPKC.dhParam_g, 2 );
	cryptInfo->ctxPKC.lastPublicComponent = DH_LAST_PUBLIC;
	cryptInfo->ctxPKC.lastPublicMontCTX = 0;
	status = calculateKeyID( CRYPT_ALGO_DH, &cryptInfo->ctxPKC,
							 cryptInfo->ctxPKC.keyID );

	return( status );
	}
