/* miniproc.c

David Mathog, Biology Division, Caltech

Copyright 1997,1998 by David Mathog and California Instititute of Technology.

This software may be used freely, but may not be redistributed.  You may
modify this sofware for your own use, but you may not incorporate any part
of the original code into any other piece of software which will then be
distributed (whether free or commercial) unless prior written consent is
obtained.  For more information, or to report bugs, contact:

   mathog@seqaxp.bio.caltech.edu

For more information, see miniproc.doc

Revision history

1.07 02-FEB-1997.
     Added getenv function.  Some operating systems may not support it, in
       which case it returns a FALSE status.
     Added option to allow MAXINLINE to be set at compilation, and
       changed default to 32768.
     Added ALTPREFIX mechanism/symbol, so that scripts can run with
       a prefix different from "#__".  Note that #__ is ALWAYS active,
       but a second prefix can now be enabled as well.
     Modified f$out syntax, so that either new,append,"new","append", or
       a symbol holding new or append could be used.
     Fixed bug in endif implementation - valid statements following and
       endif were not executed if within another logical block.
     Fixed EXIT_FAILURE versus FAILURE, now consistently uses EXIT_FAILURE.
     Fixed bug.  continuation lines read from macros were not working.
     Fixed bug.  if/elseif/else/endif labels were bleeding down into other
       modules so that labels could not be reused.
     Fixed bug.  Zero length passthrough strings ( #__"") were not working.
       These now correctly output blank lines.
1.06 22-DEC-1997.
     Fixed bug.  Line continuation - buffer was not being reset so that
       if the first line of a continuation was shorter than the preceding
       input, the tail of the preceding would be inserted.
     Fixed bug.  *var was not working - at all.
     Fixed bug.  Assignment of local variables when passed to a macro
       was happening AFTER the macro name changed, so that the name
       lookup would fail.
1.05 17-DEC-1997.
     Fixed bug.  Setting an existing string variable to a null string
       left it as a NULL, rather than a pointer to the character '\0'.
1.04 09-DEC-1997.
     Added f$break.  This is an f$exit that can be used within an
       if/elseif/else/endif structure, but ONLY inside an input file.  That
       is, you cannot exit a macro through one, only an f$in file.
     Fixed bug in error reporting for end of file encountered (wrong file 
       name was returned.)
1.03 03-DEC-1997.
     Modified if logic so that if a if f$(function) works for all functions.
       (If f$evaluate returns with status 0 though, it is a fatal error.)
     Modified f$<- to return value in RESULT instead of P9.
     Modified local variables to work within macros (that is, local to
       macro, previously it was local to input file.)
     Fixed bug, f$macro_return always returned 1
1.02 01-DEC-1997.  Fixed bug in finr[] access, array was 9 elements, but
        accessed by 11-19 index.  Now it works on Solaris. (Thanks Pete.Lee@decus.ch)
     Added :name variables, that is, "local" variables and macros.
     Added #__"whatever" and #__&whatever processing. 
     Added f$<- operator.
     Added buffering in f$evaluate, so that variables may be both
       result and operand.
     Added ifnot, elseifnot
1.01 28-NOV-1997.  Fixed bug in f$macro_repeat - incorrect handling of
       less than 3 repeat count values. 
     Fixed bug in f$evaluate for lexhigh,lexlow - incorrect clean up
       code for extending result string.
     Added f$evaluate resize.
1.0  28-OCT-1997, first release version

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <ctype.h>                    /* for toupper, tolower */
#include <sys/stat.h>                 /* no sys needed for openvms, but others are picky */

/* make the variables and tests global */

#ifdef Boolean
#undef Boolean
#endif
#define Boolean unsigned int

/* #define FAILURE 0 */

#define HASHSPACE 4096                /* hash size for variable access */
#define MAXSTACK 1024                 /* greatest number of tokens on a function line, 
                                         anybody who hits this limit should be using a 
                                         different language or method!!!*/
#ifndef MAXINLINE                     /* can set from the compiler */
#define MAXINLINE   32768             /* biggest fully substituted line, also sum of all 
                                         strings held in the action stack */
#endif
#define MAXALTPREFIX   16             /* largest alternate prefix which can be STORED in the ifstack*/
#define MAXVARLEN     256             /* biggest variable name */
#define MAXLABELLEN    64             /* biggest if label name */
#define MAXIFDEPTH   1024             /* if stack depth */
#define DEFMACRO     1024             /* default allocation for a macro, to start with */
#define MAXFNAME      256             /* biggest file name */
#define MAXMACRODEPTH 100             /* maximum depth for macros calling macros */

int ddosubnum=1;                      /* dummy SUBNUM variable */
int *dosubnum=&ddosubnum;             /* substitute to one level */
int dmacrosubnum=0;                   /* dummy MACROSUBNUM variable */
int *domacrosubnum=&dmacrosubnum;     /* substitute to zero levels */

int dtrace=0;                         /* dummy TRACE variable, needed just until user 
                                         visible variables are initialized */
int *trace=&dtrace;                   /* trace disabled */
#define TRACE_COMMAND    1
#define TRACE_INPUT      2
#define TRACE_ADDVAR     4
#define TRACE_SETVAR     8
#define TRACE_MACRO     16
#define TRACE_FUNC      32
#define TRACE_OUTPUT    64
#define TRACE_SUBS     128
#define TRACE_FULLNAME 256

int *instatus=&dtrace;                /* internal status variable, maps to STATUS */

enum isspecial {IS_SPECIAL,IS_ORDINARY};
enum istype {STRINGVAR,INTVAR,MACRO};
enum macrostate {EMPTY,RECORDING,DONE,PLAYING};
typedef struct isavar VARIABLE;
struct isavar {                       /* a VARIABLE node */
   enum istype type;                  /* STRINGVAR,INTVAR,MACRO */
   char * name;                       /* pointer to the name of the variable */
   VARIABLE *next;                    /* next VARIABLE in chain */
   int    ival;                       /* integer, if integer */
   char * string;                     /* string pointed to, if string or macro*/
   char * macro;                      /* pointer used when playing back macros.  It
                                         Starts at the beginning of string and 
                                         increments out through the storage area.
                                         Also, macros are not recursive, so if macro
                                         is not NULL an attempt to play that macro is
                                         an error. */
   char * altprefix;                  /* altprefix within this macro */
   int ssize;                         /* size of the string/macro storage area, when it
                                         was created with malloc or realloc, in bytes */
   int msize;                         /* of all of the strings in a macro, in bytes 
                                         up to and including the final \0 */
   enum macrostate mstate;            /* State of the macro.  Macros most be DONE
                                         to play, and once DONE, they may not be 
                                         extended.*/
   enum isspecial special;            /* ordinary or special, ordinary has type locked */
   int  mc[3];                        /* macro loop indices, used when looping */
   int  mcmax[3];                     /* macro loop index limits, used when looping */
   };

char *actionstack[MAXSTACK];          /* maximum number of tokens in the stack */
int stackptr=-1;                      /* index, begins as invalid */

enum iftype  {IFLABEL,IFMACRO,IFIN};
enum ifstate {IF,ELSEIF,ELSE,ENDIF};  /* includes the NOT forms */
enum ifdoneit  {YES,NO};
typedef struct isanif IFNODE;
struct isanif {                       /* an if stack node */
   enum iftype type;                  /* IFLABEL,IFMACRO,IFIN  */
   enum ifstate state;                /* IF,IFNOT,ELSEIF,ELSEIFNOT,ELSE,ENDIF */
   enum ifdoneit doneit;              /* YES, NO , passed a test in this particular logic block */
   enum ifdoneit invert;              /* YES, NO, whether or not to invert the logic */
   char string[MAXLABELLEN];
   char altprefix[MAXALTPREFIX];      /* altprefix which was in effect on f$in or macro call */
};
IFNODE ifstack[MAXIFDEPTH];           /* if stack with nodes of this type */
int ifptr=-1;                         /* pointer into stack, initially invalid */
enum ifdoneit ifscan=NO;              /* start with if scanning off */

int dsafety=0;                        /* dummy safety variable, needed just until user 
                                         visible variables are initialized */
int *safety=&dsafety;                 /* safety variables, set on command line ONLY to 
                                         restrict actions taken by possibly hostile input files */
#define SAFE_PATH       1             /* if set, use only string to the right of /\]>: 
                                         in file names, disabling paths, including on
                                         the file passed from the command line */ 
#define SAFE_IN         2             /* if set, disables f$in */
#define SAFE_OUT        4             /* if set, disables f$out (all output to stdout) */
#define SAFE_FILE       8             /* if set, disables f$file_info */


VARIABLE *head[HASHSPACE];            /* pointers to first in the chains */
VARIABLE *tail[HASHSPACE];            /* pointers to last in the chains */
unsigned int hashed;                  /* hash value from last hash operation*/
VARIABLE *result=NULL;                /* pointer to "result" */
VARIABLE *recmacro=NULL;              /* pointer to a macro being recorded */

VARIABLE *altprefix;                  /* altprefix variable */

/* the MC* and MC*MAX variables are referenced so often that it is easiest
to just keep pointers for them around, rather than having to findvar()
them over and over */

VARIABLE *mcshort[3];
VARIABLE *mcmaxshort[3];


enum fromtype {INFROMMACRO,INFROMFILE,INFROMCOMMAND};
enum fromtype fromsource =INFROMCOMMAND; /* where input comes from */
enum fromtype nextfromsource = INFROMCOMMAND; /* where input comes from */

#define HANDLE_NORMAL 0
#define HANDLE_RECORD 1
int howtohandle = HANDLE_NORMAL;

VARIABLE *activemacro[MAXMACRODEPTH]; /* macro calling stack */
int isactivemacro=-1;                 /* index into macro calling stack, here, none active */

struct isanop {                       /* an operator node for do_evaluate */
   char *name;                        /* operator name*/
   int opnum;                         /* used in a switch statement */
   int minoperands;                   /* smallest number of operands */
   int maxoperands;                   /* largest number of operands */
   enum istype restype;               /* type of result */
   enum istype op1type;               /* type of operand1*/
   enum istype op2type;   
   enum istype op3type;   
   enum istype opntype;               /* type of operands 4 to N */
} all_optypes[]={
   {"power"     ,0  ,2 ,2         ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR},
   {"modulo"    ,1  ,2 ,2         ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR},
   {"add"       ,2  ,2 ,MAXSTACK  ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR},
   {"subtract"  ,3  ,2 ,MAXSTACK  ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR},
   {"multiply"  ,4  ,2 ,MAXSTACK  ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR},
   {"divide"    ,5  ,2 ,MAXSTACK  ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR},
   {"eq"        ,6  ,2 ,MAXSTACK  ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR},
   {"ne"        ,7  ,2 ,MAXSTACK  ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR},
   {"ge"        ,8  ,2 ,MAXSTACK  ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR},
   {"gt"        ,9  ,2 ,MAXSTACK  ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR},
   {"lt"        ,10 ,2 ,MAXSTACK  ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR},
   {"le"        ,11 ,2 ,MAXSTACK  ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR},
   {"xor"       ,12 ,1 ,MAXSTACK  ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR},
   {"not"       ,13 ,1 ,1         ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR},
   {"and"       ,14 ,2 ,MAXSTACK  ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR},
   {"or"        ,15 ,2 ,MAXSTACK  ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR},
   {"nand"      ,16 ,2 ,MAXSTACK  ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR},
   {"nor"       ,17 ,2 ,MAXSTACK  ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR    ,INTVAR},
   {"head"      ,18 ,2 ,MAXSTACK  ,STRINGVAR ,INTVAR    ,STRINGVAR ,STRINGVAR ,STRINGVAR},
   {"tail"      ,19 ,2 ,MAXSTACK  ,STRINGVAR ,INTVAR    ,STRINGVAR ,STRINGVAR ,STRINGVAR},
   {"segment"   ,20 ,3 ,MAXSTACK  ,STRINGVAR ,INTVAR    ,INTVAR    ,STRINGVAR ,STRINGVAR},
   {"tostring"  ,21 ,2 ,2         ,STRINGVAR ,INTVAR    ,STRINGVAR ,INTVAR    ,INTVAR},
   {"tointeger" ,22 ,1 ,1         ,INTVAR    ,STRINGVAR ,INTVAR    ,INTVAR    ,INTVAR},
   {"tohex"     ,23 ,1 ,1         ,INTVAR    ,STRINGVAR ,INTVAR    ,INTVAR    ,INTVAR},
   {"tooctal"   ,24 ,1 ,1         ,INTVAR    ,STRINGVAR ,INTVAR    ,INTVAR    ,INTVAR},
   {"append"    ,25 ,2 ,MAXSTACK  ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR},
   {"uppercase" ,26 ,1 ,MAXSTACK  ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR},
   {"lowercase" ,27 ,1 ,MAXSTACK  ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR},
   {"shortest"  ,28 ,1 ,MAXSTACK  ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR},
   {"longest"   ,29 ,1 ,MAXSTACK  ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR},
   {"eliminate" ,30 ,2 ,2         ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR},
   {"retain"    ,31 ,2 ,2         ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR},
   {"element"   ,32 ,2 ,MAXSTACK  ,STRINGVAR ,INTVAR    ,STRINGVAR ,STRINGVAR ,STRINGVAR},
   {"locate"    ,33 ,2 ,2         ,INTVAR    ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR},
   {"compare"   ,34 ,2 ,MAXSTACK  ,INTVAR    ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR},
   {"ccompare"  ,35 ,2 ,MAXSTACK  ,INTVAR    ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR},
   {"length"    ,36 ,1 ,MAXSTACK  ,INTVAR    ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR},
   {"lexhigh"   ,37 ,1 ,MAXSTACK  ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR},
   {"lexlow"    ,38 ,1 ,MAXSTACK  ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR},
   {"stringdel" ,39 ,2 ,MAXSTACK  ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR},
   {"resize"    ,40 ,1 ,1         ,STRINGVAR ,INTVAR    ,STRINGVAR ,STRINGVAR ,STRINGVAR},
   {"getenv"    ,41 ,1 ,1         ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR},
   {""          ,42 ,2 ,MAXSTACK  ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR ,STRINGVAR},
   };


FILE *fin[10];                        /* space for 10 input stream files (all at filenumber 10) */
FILE *finr[9];                        /* space for 9 regular input files (filenumber 11-19, all offset by -11) */
char finname[10][MAXFNAME];           /* names of currently open input files */
int finc=-1;                          /* array pointer for fin, initially invalid */
FILE *fout[]= {NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL};
                                      /* output files filenumber 0-9*/

char cdeck[MAXVARLEN];                /* holds deck characters */
char cfullname[MAXVARLEN+MAXVARLEN];  /* for expanded local names */
char clastfullname[MAXVARLEN+MAXVARLEN]="\0"; 
                                      /* for expanded local names */
char *fullname=&cfullname[0];
char *lastfullname=&clastfullname[0];
char *deck=&cdeck[0];

int wasfevaluate=0;
char fevaluateconstant[]="f$evaluate";
char resultconstant[]="RESULT";       /* not to be confused with internal "result" in f$evaluate!!!*/

/* ------------------------------
  makefullname().  Expands local variable or macronames and also calculates
  the hash value.  Local variables (form=":var") expand based on fromsource.
  For instance, if :var is in myfile.mpc, the variable is stored as
  "^myfile.mpc^var", and if it is in a local macro ":amacro" running from that file 
  it becomes "^^myfile.mpc^amacro^var".  Kind of ugly, but the point is to 
  keep the local variables out of the global namespace, and this pretty 
  much does it unless you go to great lengths to force an overlap.
*/
void makefullname(char *name){
unsigned int i,last;
char *from;
VARIABLE *play;

  if(*name==':'){ /* local variables must have their names expanded */
    (void)strcpy(fullname,"^");
    switch (fromsource){
      case  INFROMMACRO:
        play=activemacro[isactivemacro];
        if(play==NULL){
          (void) fprintf(stderr,"miniproc, fatal programming error, infrommacro in makefullname\n");
          exit(EXIT_FAILURE);
        }
        (void)strcat(fullname,play->name);
        break;
      case  INFROMFILE:
        (void)strcat(fullname,&finname[finc][0]); /* names of currently open input file */
        break;
      case  INFROMCOMMAND:
        (void) fprintf(stderr,"miniproc, fatal error, local variable [%s]used on command line\n",name);
        exit(EXIT_FAILURE);
    }
    (void)strcat(fullname,"^");
    from=name;
    from++;
    (void)strcat(fullname,from);
    (void)strcat(fullname,"\0");
  }
  else {
    (void) strcpy(fullname,name);
  }

  /* now hash it if it is not exactly the same name as the preceding one, 
  that is, if it will have a different hash value */

  if(strcmp(lastfullname,fullname)!=0){
    (void)strcpy(lastfullname,fullname); /* for the next time */
    for(i=1, last=1, hashed=0, from=fullname ; *from!='\0' ; from++,i++){
      hashed=(i * last) * (unsigned int)(*from) + hashed;
      last=*from;
    }
    last = hashed/HASHSPACE;
    hashed = hashed - last*HASHSPACE;
  }
  if((*trace & TRACE_FULLNAME)==TRACE_FULLNAME){
    (void) fprintf(stderr,"  name [%s] -> hash [%d], fullname [%s]\n",name,hashed,fullname);
  }
}

/*---------------------------------------------------------
   findvar(), return the pointer to a variable based on its name, NULL
   if there is no match 
*/

VARIABLE *findvar(char *name){
VARIABLE *slide, *current;

  makefullname(name);

  current = slide = head[hashed];
  while (slide != NULL){
    current = slide;
    slide = (VARIABLE *)(current->next);
    if(strcmp(current->name,fullname)==0)return current;
  }
  return NULL;
}


/*---------------------------------------------------------
   addvar(), creates another variable based on the name supplied,
   Issues a warning and exits on duplicate names or lack of space.
*/

void addvar(char *name, char *sval, int ival, enum istype type,
  enum isspecial special){

  int slen;
  VARIABLE *newvar=NULL;

  if(sval!=NULL)slen=1 + strlen(sval);
  if((*trace & TRACE_ADDVAR)==TRACE_ADDVAR){
    switch (type) {
    case INTVAR:
      (void) fprintf(stderr,"  add integer variable >%s<\n",name);
      (void) fprintf(stderr,"  new value            [%d]\n",ival);
      break;
    case STRINGVAR:
      (void) fprintf(stderr,"  add string variable  >%s<\n",name);
      (void) fprintf(stderr,"  new value            [%s]\n",sval);
      break;
    case MACRO:
      (void) fprintf(stderr,"  adding macro         >%s<\n",name);
      break;
    }
  }

/* make sure the name is new */

  if(findvar(name) != NULL){
    (void) fprintf(stderr,"miniproc, fatal error, duplicate name %s\n",name);
    exit(EXIT_FAILURE);
  }

/* create the variable */

  newvar = malloc(sizeof(VARIABLE));
  if(newvar == NULL){
    (void) fprintf(stderr,"miniproc, fatal error, could not allocate space for %s\n",name);
    exit(EXIT_FAILURE);
  }

/* put the name on */

  newvar->name = malloc((1 + strlen(fullname))*sizeof(char)); /* leave space for the final \0 !! */
  if(newvar->name == NULL){
    (void) fprintf(stderr,"miniproc, fatal error, could not allocate space for %s\n",name);
    exit(EXIT_FAILURE);
  }
  (void) strcpy(newvar->name,fullname);

/* set all fields, then overwrite those that are particular to certain 
   types*/

  newvar->string=NULL;
  newvar->macro=NULL;
  newvar->altprefix=NULL;
  newvar->ssize=0;
  newvar->msize=0;
  newvar->ival=0;
  newvar->mc[0]=1;    /* active loop counters, default to 1 pass*/
  newvar->mc[1]=0;
  newvar->mc[2]=0;
  newvar->mcmax[0]=1; /* max loop size counters, default is 1 pass */
  newvar->mcmax[1]=0;
  newvar->mcmax[2]=0;
  switch (type) {
    case STRINGVAR:
      newvar->type=STRINGVAR;
      if(sval !=NULL){
        newvar->string=malloc(slen*sizeof(char)); /* leave space for the final \0 !!*/
        if(newvar->string== NULL){
           (void) fprintf(stderr,"miniproc, fatal error, could not allocate space for %s\n",name);
           exit(EXIT_FAILURE);
           }
        (void) strcpy(newvar->string,sval);
        newvar->ssize=slen;
      }
      newvar->special=special;
      break;
    case MACRO:
      newvar->type=MACRO;
      newvar->string=malloc(DEFMACRO*sizeof(char)); /* initial allocation for a macro */
      if(newvar->string== NULL){
         (void) fprintf(stderr,"miniproc, fatal error, could not allocate space for %s\n",name);
         exit(EXIT_FAILURE);
     }
      newvar->mstate=EMPTY;
      newvar->macro=newvar->string;
      newvar->ssize=DEFMACRO;
      newvar->msize=0;
      newvar->special=IS_ORDINARY;  /* macros cannot be reassigned, ever */
      newvar->altprefix=malloc((1+strlen(altprefix->string))*sizeof(char)); /* space to hold altprefix */
      if(newvar->altprefix== NULL){
        (void) fprintf(stderr,"miniproc, fatal error, could not allocate space for %s\n",name);
        exit(EXIT_FAILURE);
      }
      if(strlen(altprefix->string) !=0){     /* altprefix at time of recording is permanent */
        (void) strcpy(newvar->altprefix,altprefix->string); /* a string of some sort */
      }
      else {
        *newvar->altprefix='\0';  /* no altprefix in effect */
     }
      recmacro=newvar;             /*the newest macro is always the one recording */
      break;
    case INTVAR:
      newvar->type=INTVAR;
      newvar->special=special;
      newvar->ival=ival;
      break;
    default:
      (void) fprintf(stderr,"miniproc, fatal programming error, call to addvar() with illegal type\n");
      exit(EXIT_FAILURE);
  }

/* put it into head/tail strings. */

  if(head[hashed]==NULL){
    head[hashed] = newvar;       /* this is the first variable, so assign head/tail */
    tail[hashed] = newvar;
  }
  else {                         /* already some, so fix tail */
    tail[hashed]->next=newvar;   /* stick in the link from the old tail */
    tail[hashed] = newvar;       /* update tail */
  }
  newvar->next=NULL;           /* no link yet forward from the current one */

  return;
}

/*---------------------------------------------------------
   setvar(), modifies the value of an existing variable
   Issues a warning and exits on lack of space or an attempt
   to set a nonexistant variable.
   Use macro_record to modify (set) macros. */

void setvar(char *name, char *sval, int ival, enum istype type,
  enum isspecial special){
  VARIABLE *old;

  if((*trace & TRACE_SETVAR)==TRACE_SETVAR){
    (void) fprintf(stderr,"setting variable: %s\n",name);
    switch (type) {
    case INTVAR:
      (void) fprintf(stderr,"  new value =[%d]\n",ival);
      break;
    case STRINGVAR:
      (void) fprintf(stderr,"  new value =[%s]\n",sval);
      break;
    }
  }

  /* special case.  safety may only be set from the command line */

  if(  (strcmp("safety",name)==0)  && fromsource != INFROMCOMMAND){
    (void) fprintf(stderr,"miniproc, fatal safety error, attempt to set safety variable within code\n");
    exit(EXIT_FAILURE);
  }

  /* make sure the variable exists */

  old=findvar(name);
  if(old == NULL){
    (void) fprintf(stderr,"miniproc, fatal error, attempt to set nonexistant variable, %s\n",name);
    exit(EXIT_FAILURE);
  }

  /* make sure the types match OR this is a special variable */

  if (type != old->type && IS_ORDINARY == old->special){
    (void) fprintf(stderr,"miniproc, fatal error, type mismatch when setting variable %s\n",name);
    exit(EXIT_FAILURE);
  }

  /* make sure this is not a Macro, those can be recorded, but not set */

  if (old->type == MACRO){
    (void) fprintf(stderr,"miniproc, fatal error, attempt to set macro= %s\n",name);
    exit(EXIT_FAILURE);
  }

  /* everything is fine, so SET it.  Only set those bits that change - don't 
   waste time freeing strings for a special variable that goes stringvar->
   intvar since that string will be freed later if need be, and once type goes to
   intvar nothing will be able to get to the old string.   */

  switch (type) {  /* will only be stringvar or intvar */
    case STRINGVAR:
      if(old->string != NULL)
        free(old->string);     /* loose old data */
      if(sval !=NULL){
        old->string=malloc((1 + strlen(sval))*sizeof(char)); /* leave space for the final \0 !!*/
        if(old->string== NULL){
           (void) fprintf(stderr,"miniproc, fatal error, could not allocate space for %s\n",name);
           exit(EXIT_FAILURE);
        }
        (void) strcpy(old->string,sval);
        old->ssize=strlen(sval);
      }
      else {
        old->string=NULL;
        old->ssize=0;
      }
      break;
    case INTVAR:
      old->ival=ival;
      break;

  } /* end of switch on type */

  old->type=type;
  old->special=special;


} /* end of setvar() */

/* setaltprefix, this is done often enough that it is worth a separate 
   routine to skip all of the hash lookups */

void setaltprefix(char *newaltprefix){
  free(altprefix->string); /* safe - it cannot be NULL*/
  altprefix->string = malloc((strlen(newaltprefix) + 1 )*sizeof(char));
  if(altprefix->string == NULL){
    (void) fprintf(stderr,"miniproc, fatal error, out of memory\n");
    exit(EXIT_FAILURE);
  }
  if(strlen(newaltprefix) != 0){
    (void) strcpy(altprefix->string,newaltprefix);
  }
  else {
    *altprefix->string='\0';
  }
}
/* -----------------------------------
   stuffaltprefix() sets the altprefix field in an ifstack entry.
   This is called by f$in and do_macro
*/
void stuffaltprefix(void){
  if(strlen(altprefix->string) > MAXALTPREFIX-1){
    (void) fprintf(stderr,"miniproc, fatal error, altprefix [%s] is too long\n",altprefix->string);
    exit(EXIT_FAILURE);
  }
  *(ifstack[ifptr].altprefix)='\0';
  (void) strncat(ifstack[ifptr].altprefix,altprefix->string,MAXALTPREFIX-1);
}
/* -----------------------------------
   yankaltprefix() sets the altprefix field from the value in an ifstack entry.
   This is called by f$exit, f$break, f$macro_break, f$macro_return.
*/

void yankaltprefix(void){
  setaltprefix(ifstack[ifptr].altprefix);
}

/* -----------------------------------
   pathsafe().  Modify a file descriptor string to
   return only the string to the right of any /\]>:
   that might be present, and so restrict access to
   the current directory (hopefully on any OS that this might run on!)
*/
void pathsafe(char *string){
char *from;
char *to;
  for(to=string,from=string; *from!='\0'; from++){
    switch (*from){
      case '/':
      case '\\':  /* escaped \ character */
      case ']':
      case '>':
      case ':':
        to=string;
        break;
      default:
        *to=*from;
        to++;
        break;
    }
  }
  *to='\0'; /* terminate string */
} /* end of pathsafe */

void setcounters(VARIABLE *play,int setmc, int setvismc){
int i;

  if(setmc !=0 ){
    for (i=0 ; i<=2 ; i++){
      if (play->mcmax[i] == 0)
        play->mc[i] = 0;
      else
        play->mc[i] = 1;
    }
  }

  if(setvismc !=0 ){  /* copy mc, mcmax into the visible variables */
    for (i=0 ; i<=2 ; i++){
      mcshort[i]->ival = play->mc[i];
      mcmaxshort[i]->ival = play->mcmax[i];
    }
  }

} /*end of setcounters */


/*---------------------------------------------------------
   resolveright(), interprets the right hand side of an =
   statement.  Handles, *,&," operators.  Determines the type
   of the variable.  This is how it maps out:
   name        contents of variable "name"
   &name       immediate string value
   "name sdf sdf jlsdf "  immediate string value
   [-,0-9]     immediate integer value
   ***name     indirect value from variable

   mode 0 rejects resolving a "plain" macro name
   mode 1 resolves a "plain" macro name to itself (so no &name
       or "name" is required.)  The resolved name comes back in sval
       and the type is set to STRINGVAR
   mode 2 resolves a macro name, will return a MACRO type.  It will also
          return a string that is not stored as an immediate string.  Used
          to pass f$evaluate back to do_ifelse
*/

void resolveright(char *string, char **sval, int *ival, enum istype *type, int mode){
  char first;
  int  redirect;
  VARIABLE *ourvar;

  first=*string;
  if(first == '&'){ /* immediate string value */
    *type=STRINGVAR;
    *sval=string;
    (*sval)++;
    *ival=0;
    return;
  }
  if(first == '"'){ /* immediate string value */
    *type=STRINGVAR;
    *sval=strrchr(string,'"');
    if(*sval == string){
      (void) fprintf(stderr,"miniproc, fatal syntax error, unmatched double quotes\n");
      exit(EXIT_FAILURE);
    }
    **sval='\0';
    *sval=string;
    (*sval)++;
    ival=0;
    return;
  }
  if(first == '-' || (first >= '0' && first <= '9')){ /* immediate integer value */
    *type=INTVAR;
    if(sscanf(string,"%d",ival)==EOF){
      (void) fprintf(stderr,"miniproc, fatal syntax error, invalid integer %d \n",string);
      exit(EXIT_FAILURE);
    }
    *sval=string;  /* have to set it to something... */
    return;
  }
/* find out how many levels of variable redirection have been
   requested, ie *, **, ***, etc. It could be NONE */

   for (*sval = string, redirect=0;
        redirect < strlen(string);
        redirect++,(*sval)++){
      if(**sval!='*')break;
   }
   if(redirect==strlen(string)){
      (void) fprintf(stderr,"miniproc, fatal syntax error, rvalue may not be %s \n",string);
      exit(EXIT_FAILURE);
   }

/* now extract the value from the variable, or die trying.  In mode 2, if
   the value does NOT map to a preexisting variable, then that string will be
   returned verbatim.  */

   for (ourvar=findvar(*sval);redirect>0;redirect--){
     if(ourvar!=NULL){
       if(ourvar->string !=NULL) {ourvar=findvar(ourvar->string);}
       else {ourvar=NULL;}
     }
     else {
       break;
     }
   }
   if(ourvar==NULL){
     if(mode ==2){
       *type=STRINGVAR;
       *sval=string;
       *ival=0;
       return;
     }
     else {
       (void) fprintf(stderr,"miniproc, fatal error, could not resolve value of %s \n",string);
       exit(EXIT_FAILURE);
     }
   }

/* if we have not bailed out yet, then ourvar hold the REAL value that we 
   want */

   switch (ourvar->type){
      case STRINGVAR:
      case INTVAR:
         *sval=ourvar->string;
         *ival=ourvar->ival;
         *type=ourvar->type;
         break;
      default:
        switch (mode){
          case 0:
            (void) fprintf(stderr,"miniproc, fatal error, right value references macro, not variable %s \n",string);
            exit(EXIT_FAILURE);
          case 1: /* used to coerce the use of macro names as plain strings*/
            *sval=ourvar->name;
            *ival=ourvar->ival; /* just be sure something is in there */
            *type=STRINGVAR;
             break;
          case 2: /* used to coerce a macro type out, with name in sval*/
            *sval=ourvar->name;
            *ival=ourvar->ival; /* just be sure something is in there */
            *type=MACRO;
             break;
        }
   }
   return;
}

void trim_command_line(char *line){
char *p;
char *from;
char *to;
int i,j;

  if(strlen(line)<1){
    (void) fprintf(stderr,"miniproc, fatal error, command line has zero length\n");
    exit(EXIT_FAILURE);
  }

  /* remove any preceding tabs or spaces, if any */

  for (from=line,i=1,j=1; i<=strlen(line); i++,from++){
    switch (*from){
       case ' ':
       case '\t':
         break;
       default:
         j=i;
         from--; /* because for will increment it again */
         i=strlen(line)+10; /* force exit on for loop */
         break;
    }
  }
  if( j != 1 ){  /*drag it all left*/
    for (i=j, to=line; i<=strlen(line); *to=*from,i++,from++,to++){}
    *to='\0';  /* terminate the string at the new end */
  }

  p = strrchr(line,' '); /* find final space */
  if (p == NULL) return; /* no spaces */
  p++;                   /* see which character is next */
  if( *p != '\0')return; /* last space is not a trailing space */
   for (p--;p!=line;p--){ /* starting with the space, trim backwards */
    if(*p!=' ')
      return;
    else
      *p='\0';
  }
/* only get here if the line has (maybe) one character, then all spaces
   which isn't a legal line*/

  (void) fprintf(stderr,"miniproc, fatal error, command line has nothing on \n");
  exit(EXIT_FAILURE);
}

/* do_subs makes the determined number of passes through the line
   replacing <<variable>> constructs with the variable's contents. Have
   to be careful, as something like <<fo<foobar>> is legal text, it
   just should not be substituted as fo<foobar is an illegal variable name.
   However, issue warnings for things like this!  The maximum length of
   the expanded string is MAXINLINE -1 (for the terminator character)
   didsubs keeps track of the number of changes on a line, if dosubnum
   is large it stops doing substitutions after the first pass that did
   none.  */

void do_subs(char *line){
int count,maybefront,maybeback,fragfront;
char cinline[MAXINLINE];
char coutline[MAXINLINE];
char cmaybevar[MAXVARLEN];
char csintvar[32]; /* will be used for expanding integer variables */
char *vinline=&cinline[0];
char *outline=&coutline[0];
char *maybevar=&cmaybevar[0];
char *sintvar=&csintvar[0];
char *runner;
char *mrunner;
char *rl1;
int ilength;
int didsubs;
int subsruns=0;
char *lastfrag;
VARIABLE *ourvar;

  if(strlen(line) > MAXINLINE){
    (void) fprintf(stderr,"miniproc, fatal error, line is longer than the maximum allowed [%d]\n",MAXINLINE);
    exit(EXIT_FAILURE);
  }
  (void) strcpy(vinline,line);
  switch (howtohandle){
    case HANDLE_NORMAL:
      subsruns=*dosubnum;
      break;
    case HANDLE_RECORD:
      subsruns=*domacrosubnum;
      break;
  }
  
  for (didsubs=1  ; (didsubs>0 && subsruns>0) ; subsruns--){
    for (vinline=&cinline[0],
         runner=vinline,
         count=1,
         didsubs=0,
         maybefront=strlen(vinline)+1,
         maybeback=strlen(vinline)+1,
         lastfrag=vinline,
         outline=&coutline[0],
         *outline='\0',
         fragfront=0,rl1=NULL
         ;
         count<=strlen(&cinline[0])
         ;
         rl1=runner,
         runner++,
         count++){
      if(count >= 2){
        switch (*runner){
          case '>':
            if(*rl1=='>')
              maybeback=count-2;
            else
              maybeback=0;
            break;
          case '<':
            if(*rl1=='<'){
              maybefront=count+1;
	      mrunner=runner;
              mrunner++;
              }
            else
              maybefront=0;
            break;
          default:
          break;
        }

/* test if maybefront and maybeback are defined */
        if(maybefront!=0 && maybeback!=0 && 
          maybefront<=maybeback && maybeback <= strlen(vinline)){
  
          if( maybeback-maybefront+1 > 0){
            *maybevar='\0';
            (void) strncat(maybevar,mrunner,maybeback-maybefront+1);
          }
          ourvar = findvar(maybevar);
  
/* the variable is indeed real, so time to do the substitution */
  
          if(ourvar != NULL){
            didsubs++;
            (void) strncat(outline,lastfrag,maybefront-fragfront-3);
            fragfront=maybeback+2;
            switch (ourvar->type){
              case STRINGVAR:
                if(strlen(ourvar->string)+strlen(outline) > MAXINLINE){
                  (void) fprintf(stderr,"miniproc, fatal error, substituted line is longer than the maximum allowed [%d]\n",MAXINLINE);
                  exit(EXIT_FAILURE);
                }
                (void) strcat(outline,ourvar->string);
                break;
              case INTVAR:
                (void) sprintf(sintvar,"%d",ourvar->ival);
                (void) strcat(outline,sintvar);
                break;
              case MACRO:
              default:
                (void) fprintf(stderr,"miniproc, warning, <<%s>> is invalid - macros may not be inserted \n",&maybevar[0]);
                break;
            }
            lastfrag = runner;
            lastfrag++; /* next piece to be copied over, maybe */
            maybefront=0;
            maybeback=0;
          }
        }
      } /* end of test on count */
    } /* end of inner for loop */

/* there may be some remaining, if so, pass it through */
   if(count - fragfront >= 2){
     if( ( strlen(outline) + count - fragfront) > MAXINLINE){
       (void) fprintf(stderr,"miniproc, fatal error, substituted line is longer than the maximum allowed [%d]\n",MAXINLINE);
       exit(EXIT_FAILURE);
     }
     (void) strncat(outline,lastfrag,count - fragfront);
   }
   (void) strcpy(vinline,outline); /* copy expanded string back to vinline */
   if( (didsubs !=0) && ((*trace & TRACE_SUBS)==TRACE_SUBS))(void) fprintf(stderr,"  substitute >%s\n",vinline);
  } /* end of outer for loop */
  (void) strcpy(line,vinline); /* copy final string back to input area */
}

/*---------------------------------------------------------
   macro_record().  If the macro is in an appropriate state, it
   appends lines to the macro buffer.  These are stored as 
   string\0string\0string\0 and the total length of this string
   is kept in msize (including the final null).  The total number
   of available bytes for storage is kept in ssize.
*/

void macro_record(char *name, char *sval){
  int i;

  if(sval == NULL){ /* creating a new macro */
    if(recmacro!=NULL){
      (void) fprintf(stderr,"miniproc, fatal error, trying to simultaneously record a second macro %s\n",name);
      exit(EXIT_FAILURE);
    }
    addvar(name, NULL, 1, MACRO, IS_ORDINARY); /* create the macro */
    howtohandle = HANDLE_RECORD;
    return;
  }

  if (recmacro == NULL){
    (void) fprintf(stderr,"miniproc, fatal programming error, attempt to record nonexistant macro %s\n",name);
    exit(EXIT_FAILURE);
  }

  /* check the state of the macro, it must not be DONE */

  switch (recmacro->mstate){
     case EMPTY:
       recmacro->mstate=RECORDING;
     case RECORDING:
       break;
     case DONE:
     case PLAYING:
       (void) fprintf(stderr,"miniproc, fatal error, attempt to rerecord macro %s\n",name);
       exit(EXIT_FAILURE);
       break;
  }

  /* find out if there is room for the next string, if not realloc */

  if(strlen(sval) + recmacro->msize + 1 > recmacro->ssize){
    recmacro->string=realloc(recmacro->string,(recmacro->ssize+DEFMACRO)*sizeof(char));
    if(recmacro->string == NULL){
      (void) fprintf(stderr,"miniproc, fatal error, ran out of space recording macro %s\n",name);
      exit(EXIT_FAILURE);
    }
    recmacro->ssize=recmacro->ssize+DEFMACRO;

   /* fix the recacro->macro pointer, it may point to freed memory, note
      that the pointer must be positioned immediately after the last
      defined character (which is \0) */

    for (recmacro->macro=recmacro->string , i=0   ;
        i < recmacro->msize                       ;
        recmacro->macro++,i++){}

  }

  /* append the string to the macro */

  (void) strcpy(recmacro->macro,sval); /*add the string witha \0 on the end */
  for(i=1;i<=1+strlen(sval);recmacro->macro++,i++,recmacro->msize++){}
}


/*---------------------------------------------------------
   macro_repeat().  Sets the 3 repeat counter buffers in the 
   named macro.  Repeat counts must all be >=0, and the first
   must be like 1,1,0, not 1,0,1 or 0,1,1  (that is, loops
   from left, and cannot have a null loop "inside").
*/

void macro_repeat(char *name){
char *sval;
char **tsval=&sval;
int ival;
enum istype type;
VARIABLE *play;
int i;
int last;

  play=findvar(name);
  if(play == NULL){
    (void) fprintf(stderr,"miniproc, fatal error, f$macro_repeat acting on nonexistant macro %s\n",name);
    exit(EXIT_FAILURE);
  }

  if(play->mstate != DONE){
    (void) fprintf(stderr,"miniproc, fatal error, f$macro_repeat setting counters on active macro \n");
    exit(EXIT_FAILURE);
  }

  /* the next part of the loop will only set those values that were 
  supplied.  So here set first to 1 and all others to zero, which is
  the default*/

  play->mcmax[0]=1;
  play->mcmax[1]=0;
  play->mcmax[2]=0;

  for ( last=1 , i=0 ; (i <= stackptr-2 && i<=2); i++){
    resolveright(actionstack[i+2],tsval,&ival,&type,0);
    if(type != INTVAR){
      (void) fprintf(stderr,"miniproc, fatal error, f$macro_repeat had noninteger parameter\n");
      exit(EXIT_FAILURE);
    }

    if(ival<0){
      (void) fprintf(stderr,"miniproc, fatal error, f$macro_repeat parameters must be >0 \n");
      exit(EXIT_FAILURE);
    }    

    if(last==0 && ival > 0 ){
      (void) fprintf(stderr,"miniproc, fatal error, f$macro_repeat must be like 1,1,0, NOT 1,0,1 or 0,0,1\n");
      exit(EXIT_FAILURE);
    }

    last = ival;
    play->mcmax[i]=ival;
  }

} /* end of macro_repeat */

int macro_gets(char *vinline){
  int i;
  VARIABLE *play;

  if(isactivemacro < 0 || isactivemacro > MAXMACRODEPTH-1){
    (void) fprintf(stderr,"miniproc, fatal programming error, attempting to run invalid macro");
    exit(EXIT_FAILURE);
  }

  play = activemacro[isactivemacro];

  if(play==NULL){
    (void) fprintf(stderr,"miniproc, fatal programming error, attempting to run invalid macro");
    exit(EXIT_FAILURE);
  }

  if (play->mstate != PLAYING){
       (void) fprintf(stderr,"miniproc, fatal programming error, cannot read from macro which is not PLAYING\n");
       exit(EXIT_FAILURE);
  }

  if(play->ival >= play->msize ){ /* hit the end of the macro, there should
                                     have been a f$macro_return first, so 
                                     this is a syntax error */
    (void) fprintf(stderr,"miniproc, fatal error, macro %s does not end with f$macro_return \n",play->name);
    exit(EXIT_FAILURE);
  }

  /* copy the current string to the output */

  (void) strcpy(vinline,play->macro);

  /* modify the pointers*/

  for(i=1;i<=1+strlen(vinline);play->macro++,i++,play->ival++){}

  return 0;
}

/*------------------------------------------
  macro_return().  Indicates the end of a macro.  The
  loop counters are examined, and if appropriate incremented
  appropriately and the macro restarted.  Otherwise, the
  macro call stack is rolled up one.  If it rolls up off
  the top, redirect back to INFROMFILE.

  if mode is 0 it does the full f$macro_return clean up and checking.
  if mode is 1 it forces an immediate exit from the macro with minimal
  clean up and no syntax checking.
*/

void macro_return(int mode){
char *sval;
char **tsval=&sval;
int ival;
enum istype type;
VARIABLE *play;
int i,carry;

  switch (stackptr){
    case 0:
      ival=1; /*default status*/
      break;
    case 1:
      resolveright(actionstack[1],tsval,&ival,&type,0);
      break;
    default: 
      (void) fprintf(stderr,"miniproc, fatal error, f$macro_return, too many parameters\n");
      exit(EXIT_FAILURE);
      break;
  }
  *instatus=ival;

  play = activemacro[isactivemacro];


  /* examine the counters, increment if necessary 
     if we are in here not all max counters are zero */

  if(mode==0){
    for (carry = 1, i=0 ; i<=2 ; i++){
      if ( play->mcmax[i] == 0){
          break;
      }
      play->mc[i] =  play->mc[i] + carry;
      carry = 0;
      if ( play->mc[i] > play->mcmax[i]){
         carry = 1;
         play->mc[i]  = 1;
      }
      else
       break;
    }
  }
  else {  /* this forces a macro in any state to immediately exit */
    carry = 1;
  }

  /* reset internal bits so that it can be played again, either now or later */

  play->macro = play->string;  /* pointer back to the beginning */
  play->ival=0;                /* count of positions back to the beginning */

  if(carry == 1){              /* completely done with this macro */
    play->mstate=DONE;
    setcounters(play,1,0); /* reset the current macros counters */
    isactivemacro--;
    if(isactivemacro < 0)
       nextfromsource = INFROMFILE;
    else {
       play = activemacro[isactivemacro];
       setcounters(play,0,1); /* restore the visible counters for
                                 the current macro */
    }

    /* for break mode just roll up to the next ifmacro label. */

    if(mode==1){ /* roll up the ifstack until the ifmacro is found */
      for( ;( (ifstack[ifptr].type != IFMACRO) && ifptr>=0);ifptr--){}
      if(ifptr < 0){
        (void) fprintf(stderr,"miniproc, fatal programming error, f$macro_break destroyed ifstack\n");
        exit(EXIT_FAILURE);
      }
    }

    /* clean up the ifstack before returning control.  The last tag in ifstack 
       better be MACRO. If so roll it up. 
       ifptr better be valid or there is a bug elsewhere in the program!*/

    if(ifstack[ifptr].type != IFMACRO){
      (void) fprintf(stderr,"miniproc, fatal error, incomplete if/then/else in macro %s\n",play->name);
      exit(EXIT_FAILURE);
    }
    else{

       yankaltprefix(); /* restore the altprefix variable to what it was 
                           when the macro was called */

       /* Macro may have executed in some sort of if/else/elseif context. The 
       first case is for the macro internal to a logic block.  In that
       case it must NOT have been scanning, and it should not begin
       scanning until it hits another part of the if/elseif/else/endif
       structure.  The second case is when the macro is the TEST for a logic
       block.  Here if the test fails it should begin scanning, and if the
       test passes, it should not begin scanning, but rather executing. */ 
  
      ifptr--;
      ifscan = NO;
      if(ifptr >= 0 ){
        if( ifstack[ifptr].type == IFLABEL){
          if(ifstack[ifptr].doneit == YES) 
            ifscan=NO;
          else {
            if(  (ifstack[ifptr].invert==NO  && *instatus == 0) ||
                 (ifstack[ifptr].invert==YES && *instatus != 0) ){
              ifscan = YES;
            }
            else {
              ifscan = NO;
              ifstack[ifptr].doneit = YES;
            }
          }
        }
      }
    }
  }
  else { /* going to play it again */
    setcounters(play,0,1); /* update the visible counters to match internal ones*/

    /* roll back the ifstack - there may be a bunch of if structures built 
       within this macro and they must be rebuilt at each iteration */

    for (;ifptr>=0;ifptr--){
      if( ifstack[ifptr].type == IFMACRO)
         break;
    }

    /* set altprefix to the recorded type.  Otherwise if the macro
       changed altprefix it will blow up on the next loop */

    setaltprefix(play->altprefix);
  }

  return;
} /* end of macro_return*/

void enlarge_string(char **string,int *ssize, int newsize,char *oops){
  if(newsize < *ssize)return;
  /*make it 1 bigger for the terminator.  Note that it is possible that
  for special variables the string may not yet exist, so test that.*/
  if(*string==NULL){
    *string=malloc((newsize+1)*sizeof(char)); 
  }
  else {
    *string=realloc(*string,(newsize+1)*sizeof(char)); 
  }
  if(*string==NULL){
    (void) fprintf(stderr,"miniproc, fatal error, %s\n",oops);
    exit(EXIT_FAILURE);
  }
  *ssize=newsize+1;
}

/* --------------------------------------
   fixorder() is used by f$evaluate tail and segment.
   It takes a string which is rotated right by r characters
   and puts it back into the proper position.
*/

void fixorder(char *string,int rot, int ssize){
char *from;
char *to;
char *temp;
int i,j;

   if(rot==1)return;
   temp=malloc(ssize*sizeof(char));
   if(temp==NULL){
     (void) fprintf(stderr,"miniproc, fatal error, f$evaluate out of memory\n");
     exit(EXIT_FAILURE);
   }
   (void) strncpy(temp,string,ssize);

   /* position the from pointer */

   for(j=1,from=temp;j<rot;j++,from++){}

   /* shuffle everything back into place */

   for(i=1,to=string;  i<=ssize ; i++,to++){
      *to=*from;
      if(j==ssize){
         from=temp;
         j=1;
      }
      else {
         from++;
         j++;
      }
   }
   *to='\0';
   free(temp);
   temp=NULL;
}


/* ----------------------------------------------------------------
  do_evaluate().  Perform verious operations on operands and return the
  result.
*/

void do_evaluate(void){
int i,j,k;
struct isanop *opptr;
char *sval;
char **tsval=&sval;
int ival;
enum istype type;
VARIABLE *result;
int ifirst;
int fsize;
char *holdsval;
char *from;
char *to;
char *temp;
int dofixorder;
int sure;
int rival;   /* buffers to hold result */
char *rsval=NULL;
int rssize;

  /* which operator is it? */
  for(opptr=all_optypes ; strlen(opptr->name)!=0 ; opptr++){
    if(strcmp(opptr->name,actionstack[2])==0)break;
  }

  if (strlen(opptr->name) == 0){
    (void) fprintf(stderr,"miniproc, fatal error, f$evaluate unknown operator %s\n",actionstack[2]);
    exit(EXIT_FAILURE);
  }

  /* validate the number of arguments */

  if (stackptr < opptr->minoperands + 2){ /* 2=f$evaluate + result + operand  - 1*/
    (void) fprintf(stderr,"miniproc, fatal error, f$evaluate too few operands for %s\n",opptr->name);
    exit(EXIT_FAILURE);
  }
  if (stackptr > opptr->maxoperands + 2){ 
    (void) fprintf(stderr,"miniproc, fatal error, f$evaluate too many operands for %s\n",opptr->name);
    exit(EXIT_FAILURE);
  }

  /* validate the result type */

  result=findvar(actionstack[1]);
  if(result == NULL){
    (void) fprintf(stderr,"miniproc, fatal error, f$evaluate, result variable %s does not exist\n",actionstack[1]);
    exit(EXIT_FAILURE);
  }
  if(result->type != opptr->restype){ /*if result is special, set it to match*/
    if(result->special == IS_SPECIAL){
       result->type = opptr->restype;
    }
    else { /* otherwise, this is an error */
      (void) fprintf(stderr,"miniproc, fatal error, f$evaluate result is incorrect type for %s\n",opptr->name);
      exit(EXIT_FAILURE);
    }
  }
  if(result->type == INTVAR){
    rival=0;
  }
  else {
    rsval=malloc(2048*sizeof(char));
    if(rsval == NULL){
      (void) fprintf(stderr,"miniproc, fatal error, f$evaluate could not obtain memory\n");
      exit(EXIT_FAILURE);
    }
    rssize=2048;
  }     

  /* do the operation.  The key is an ENORMOUS switch statement... */

  *instatus = 1;  /* assume that everything goes ok */
  for (i=3; i<=stackptr; i++){

    /* validate the operand type */

    resolveright(actionstack[i],tsval,&ival,&type,0);

    switch (i-2){
      case 1:
        if(type != opptr->op1type){
          (void) fprintf(stderr,"miniproc, fatal error, f$evaluate operand 1 is incorrect type for %s\n",opptr->name);
          exit(EXIT_FAILURE);
        }
        break;
      case 2:
        if(type != opptr->op2type){
          (void) fprintf(stderr,"miniproc, fatal error, f$evaluate operand 2 is incorrect type for %s\n",opptr->name);
          exit(EXIT_FAILURE);
        }
        break;
      case 3:
        if(type != opptr->op3type){
          (void) fprintf(stderr,"miniproc, fatal error, f$evaluate operand 3 is incorrect type for %s\n",opptr->name);
          exit(EXIT_FAILURE);
        }
        break;
      default:
        if(type != opptr->opntype){
          (void) fprintf(stderr,"miniproc, fatal error, f$evaluate operand N is incorrect type for %s\n",opptr->name);
          exit(EXIT_FAILURE);
        }
        break;
    }

    switch (opptr->opnum) {

       case  0:  /* power */
         if(i==3)
           ifirst=ival;
         else {
           if(ival<0){
             (void) fprintf(stderr,"miniproc, fatal error, f$evaluate power exponent may not be negative\n");
             exit(EXIT_FAILURE);
           }
           for(j=1; ival!=0; ival--){
              j=ifirst*j;
           }
           rival = j;
         }
         break;

       case  1:  /* modulo */
         if(ival<0){
           (void) fprintf(stderr,"miniproc, fatal error, f$evaluate modulo operands may not be negative\n");
           exit(EXIT_FAILURE);
         }
         if(i==3)
           ifirst=ival;
         else {
           if(ival==0){
             (void) fprintf(stderr,"miniproc, fatal error, f$evaluate modulo modulus may not be zero\n");
             exit(EXIT_FAILURE);
           }
           i=ifirst/ival;
           rival= ifirst-(i*ival);
        }
         break;
       case  2:  /* add */
         if(i==3)
           rival=ival;
         else
           rival=rival + ival;
         break;
       case  3:  /* subtract */
         if(i==3)
           rival=ival;
         else
           rival=rival - ival;
         break;
       case  4:  /* multiply */
         if(i==3)
           rival=ival;
         else
           rival=rival * ival;
         break;
       case  5:  /* divide */
         if(i==3)
           rival=ival;
         else
           if(ival == 0){
             (void) fprintf(stderr,"miniproc, fatal error, f$evaluate attermpt to divide by zero\n");
             exit(EXIT_FAILURE);
           }
           rival=rival / ival;
         break;
       case  6:  /* eq */
         if(i==3){
           ifirst = ival;
           rival=1;  /* assume true */
         }
         else {
           if(ival != ifirst){
             result->ival=0;
             return;
           }
         }
         break;
       case  7:  /* neq */
         if(i==3){
           ifirst = ival;
           rival=1;  /* assume true */
         }
         else {
           if(ival == ifirst){
             result->ival=0;
             return;
           }
         }
         break;
       case  8:  /* ge */
         if(i==3){
           ifirst = ival;
           rival=1; /* assume true */
         }
         else {
           if(ifirst >= ival)
             break;
           else {
             result->ival=0;
             return;
           }
         }
         break;
       case  9:  /* gt */
         if(i==3){
           ifirst = ival;
           rival=1; /* assume true */
         }
         else {
           if(ifirst > ival)
             break;
           else {
             result->ival=0;
             return;
           }
         }
         break;
       case 10:  /* lt */
         if(i==3){
           ifirst = ival;
           rival=1; /* assume true */
         }
         else {
           if(ifirst < ival)
             break;
           else {
             result->ival=0;
             return;
           }
         }
         break;
       case 11:  /* le */
         if(i==3){
           ifirst = ival;
           rival=1; /* assume true */
         }
         else {
           if(ifirst <= ival)
             break;
           else {
             result->ival=0;
             return;
           }
         }
         break;
       case 12:  /* xor */
         if(i==3){
           ifirst = ival;
           rival=1; /* default = true */
         }
         else {
           if(ifirst != 0){
             if(ival != 0){
               result->ival=0;
               return;
             }
           }
           else{
             if(ival == 0){
               result->ival=0;
               return;
             }
           }
         }
         break;
       case 13:  /* not */ 
         if(ival == 0)
           result->ival=1;
         else
           result->ival=0;
         return;
       case 14:  /* and */ 
         if(i==3){
           ifirst = ival;
           rival=1; /* assume true */
         }
         else {
           if(ifirst !=0 && ival !=0)
             break;
           else {
             result->ival=0;
             return;
           }
         }
         break;
       case 15:  /* or */  
         if(i==3) {
           rival=0; /* assume false */
         }
         if(ival !=0){
           result->ival=1;
           return;
         }
         break;
       case 16:  /* nand */
         if(i==3) {
           ifirst = ival;
           rival=0; /* assume false */
         }
         else {
           if(ifirst !=0 && ival !=0)
             break;
           else {
             result->ival=1;
             return;
           }
         }
         break;
       case 17:  /* nor */ 
         if(i==3) {
           rival=1; /* assume true */
         }
         if(ival != 0){
           result->ival=0;
           return;
         }
         break;
       case 18:  /* head */
         if(i==3){
           if(ival < 0){
             (void) fprintf(stderr,"miniproc, fatal error, f$evaluate head for < zero characters\n"); 
             exit(EXIT_FAILURE);
           }
           fsize = ival;
           if(rssize < fsize + 1)
             enlarge_string(&rsval,&rssize,fsize,"f$evaluate head failed while enlarging result string");
           to=rsval;
           j=1;
         }
         else {
           from=sval;
           for(; (j<=fsize && *from!='\0'); j++,from++,to++){
             *to=*from;
           }
           if(j==fsize || i==stackptr)*to='\0';  /*one way or the other, terminate string */
         }
         break;
       case 19:  /* tail */
         /* this one is a bit odd.  Since the arguments pretty much have to 
            come in in the wrong order, just write the results modulo style
            (wrapping) into the output space.  Then at the end slide 
            everything around into the right position */
         if(i==3){
           if(ival < 0){
             (void) fprintf(stderr,"miniproc, fatal error, f$evaluate tail for < zero characters\n");
             exit(EXIT_FAILURE);
           }
           fsize = ival;
           if(rssize < fsize + 1)
             enlarge_string(&rsval,&rssize,fsize,"f$evaluate tail failed while enlarging result string");
           to=rsval;
           j=1;
           dofixorder=0;
         }
         else {
           from=sval;
           for(; *from!='\0'; from++){
             *to=*from;
             if(j==fsize){
               j=1;
               to=rsval;
               dofixorder=1;
             }
             else {
               j++;
               to++;
             }
           }
         }
         if(i==stackptr){
           if(fsize == 0){
             *rsval='\0';
           }
           else {	  
             if(dofixorder){
               for (k=j;k<=fsize;k++,to++){}
               *to='\0';  /*terminate string */
               fixorder(rsval,j,fsize);
             }
             else {
               to='\0';
             }
           }
         }
         break;
       case 20:  /* segment */
         /* this one is also a bit odd.  Since the arguments pretty much have to 
            come in in the wrong order, just write the results modulo style
            (wrapping) into the output space. After first skipping
            the required number of characters.   Then at the end slide 
            everything around into the right position */

         if(i==3){
           if(ival < 1){
             (void) fprintf(stderr,"miniproc, fatal error, f$evaluate segment first character <1 \n");
             exit(EXIT_FAILURE);
           }
           ifirst=ival;
         }

         if(i==4){
           if(ival < 1){
             (void) fprintf(stderr,"miniproc, fatal error, f$evaluate segment size <1 character \n"); 
             exit(EXIT_FAILURE);
           }
           if(rssize < ival + 1)
             enlarge_string(&rsval,&rssize,ival+1,"f$evaluate segment failed while enlarging result string");
           fsize=ifirst + ival -1;
           to=rsval;
           j=1;
         }

         if(i>=5){
           from=sval;
           for(; (j<=fsize && *from!='\0'); j++,from++){
             if(j>=ifirst){
                *to=*from;
                to++;
             }
           }
           if(j==fsize || i==stackptr)*to='\0';  /*one way or the other, terminate string */
         }
         break;
       case 21:  /* tostring */
         if(i==3){
           ifirst=ival;
         }
         else {
           if(rssize < 32)
             enlarge_string(&rsval,&rssize,32,"f$evaluate segment failed while enlarging result string");
           if(sprintf(rsval,sval,ifirst)<=0){
             (void) fprintf(stderr,"miniproc, fatal error, f$evaluate tostring\n");
             exit(EXIT_FAILURE);
           }
         }
         break;
       case 22:  /* tointeger */
         if(sscanf(sval,"%d",&(rival))==EOF){
           (void) fprintf(stderr,"miniproc, fatal error, f$evaluate tointeger \n");
           exit(EXIT_FAILURE);
         }
         break;
       case 23:  /* tohex */
         if(sscanf(sval,"%x",&(rival))==EOF){
           (void) fprintf(stderr,"miniproc, fatal error, f$evaluate tohex \n");
           exit(EXIT_FAILURE);
         }
         break;
       case 24:  /* tooctal */
         if(sscanf(sval,"%o",&(rival))==EOF){
           (void) fprintf(stderr,"miniproc, fatal error, f$evaluate tooctal \n");
           exit(EXIT_FAILURE);
         }
         break;
       case 25:  /* append */ 
         if (i==3){  /* have to make result big enough to hold what comes out*/
           holdsval=sval;
           for(fsize=strlen(sval)+strlen(rsval)+1,j=3; j<=stackptr; j++){
             resolveright(actionstack[j],tsval,&ival,&type,0);
             if(type != STRINGVAR){
               (void) fprintf(stderr,"miniproc, fatal error, f$evaluate operand %d is incorrect type for append\n");
               exit(EXIT_FAILURE);
             }
             fsize=fsize + strlen(sval);
           }
           *rsval='\0';
           if(fsize > rssize)
             enlarge_string(&rsval,&rssize,fsize,"f$evaluate append failed while enlarging result string");
           sval=holdsval;
         }
         (void) strcat(rsval,sval);
         break;
       case 26:  /* uppercase */
         if (i==3){  /* have to make result big enough to hold what comes out*/
           holdsval=sval;
           for(fsize=strlen(sval)+strlen(rsval)+1,j=3; j<=stackptr; j++){
             resolveright(actionstack[j],tsval,&ival,&type,0);
             if(type != STRINGVAR){
               (void) fprintf(stderr,"miniproc, fatal error, f$evaluate operand %d is incorrect type for append\n");
               exit(EXIT_FAILURE);
             }
             fsize=fsize + strlen(sval);
           }
           *rsval='\0';
           if(fsize > rssize)
             enlarge_string(&rsval,&rssize,fsize,"f$evaluate append failed while enlarging result string");
           sval=holdsval;
	   to=rsval;
         }
         for (from=sval; *from!='\0'; from++,to++){
           *to=toupper(*from);
         }
         *to='\0';
         break;
       case 27:  /* lowercase */
         if (i==3){  /* have to make result big enough to hold what comes out*/
           holdsval=sval;
           for(fsize=strlen(sval)+strlen(rsval)+1,j=3; j<=stackptr; j++){
             resolveright(actionstack[j],tsval,&ival,&type,0);
             if(type != STRINGVAR){
               (void) fprintf(stderr,"miniproc, fatal error, f$evaluate operand %d is incorrect type for append\n");
               exit(EXIT_FAILURE);
             }
             fsize=fsize + strlen(sval);
           }
           *rsval='\0';
           if(fsize > rssize)
             enlarge_string(&rsval,&rssize,fsize,"f$evaluate append failed while enlarging result string");
           sval=holdsval;
	   to=rsval;
         }
         for (from=sval; *from!='\0'; from++,to++){
           *to=tolower(*from);
         }
         *to='\0';
         break;
       case 28:  /* shortest, final operation at end of loop */
         if(i==3){
           fsize=strlen(sval);
           j = i;
         }
         else {
           if(strlen(sval) < fsize){
             fsize=strlen(sval);
             j = i;
           }
         }
         break;
       case 29:  /* longest, final operation at end of loop */
         if(i==3) {
           fsize=strlen(sval);
           j = i;
         }
         else {
           if(strlen(sval) > fsize){
             fsize=strlen(sval);
             j = i;
           }
         }
         break;
       case 30:  /* eliminate */
         if(i==3){
           holdsval=sval;
           fsize=strlen(sval);
           if(fsize > rssize)
             enlarge_string(&rsval,&rssize,fsize,"f$evaluate eliminate failed while enlarging result string");
         }
         else {
           for(from=holdsval,to=rsval; *from!='\0'; from++){
             if(strchr(sval,*from)==NULL){
               *to=*from;
                to++;
             }
           }
           *to='\0';
         }
         break;
       case 31:  /* retain */ 
         if(i==3){
           holdsval=sval;
           fsize=strlen(sval);
           if(fsize > rssize)
             enlarge_string(&rsval,&rssize,fsize,"f$evaluate eliminate failed while enlarging result string");
         }
         else {
           for(from=holdsval,to=rsval; *from!='\0'; from++){
             if(strchr(sval,*from)!=NULL){
               *to=*from;
                to++;
             }
           }
           *to='\0';
         }
         break;
       case 32:  /* element */  
         if(i==3){
           if(ival<1){
             (void) fprintf(stderr,"miniproc, fatal error, f$evaluate element, negative index value\n");
             exit(EXIT_FAILURE);
           }
           ifirst=ival;
         }
         if(i==4){
           if(strlen(sval)<1){
             (void) fprintf(stderr,"miniproc, fatal error, f$evaluate element, empty delimiter string\n");
             exit(EXIT_FAILURE);
           }
           holdsval=sval;
         }
         if(i==5){
           from=malloc((1+strlen(sval))*sizeof(char));
           if(from==NULL){
             (void) fprintf(stderr,"miniproc, fatal error, f$evaluate could not obtain memory\n");
             exit(EXIT_FAILURE);
           }
           (void) strcpy(from,sval); /* because strtok is destructive */
           to=strtok(from,holdsval);
           for(j=1; (to!=NULL && j<ifirst) ; j++){
             to=strtok(NULL,holdsval);
           }
           if(to==NULL){  /* asked for too many */
             *instatus = 0;
             *rsval='\0';
           }
           else {  /* got one */
             fsize=strlen(to) + 1;
             if(fsize > rssize)
               enlarge_string(&rsval,&rssize,fsize,"f$evaluate element failed while enlarging result string");
             (void) strcpy(rsval,to);                          
           }
           free(from);
           from=NULL;
         }
         break;
       case 33:  /* locate */ 
         if(i==3)
           holdsval=sval;
         else {
           temp=strstr(sval,holdsval);
           if(temp==NULL)
             rival=0;
           else{
             for(j=1,from=sval;from!=temp;from++,j++){}
             rival=j;
           }
         }
         break;
       case 34:  /* compare */
         if(i==3){
           holdsval=sval;
           rival=1;
         }
         else {
           for (from=sval; *from!='\0';from++,holdsval++){
             if(*from!=*holdsval){
               result->ival=0;
               return; 
             }
           }
         }
         break;
       case 35:  /* ccompare - some systems don't have strccmp*/
         if(i==3){
           holdsval=sval;
           rival=1;
         }
         else {
           for (from=sval; *from!='\0';from++,holdsval++){
             if(toupper(*from)!=toupper(*holdsval)){
               result->ival=0;
               return; 
             }
           }
         }
         break;
       case 36:  /* length */
         if(i==3)
           rival=strlen(sval);
         else
           rival=rival+strlen(sval);
         break;
       case 37: /* lexhigh, final operation at end of loop */
         if(i==3){
           holdsval=sval;
           j = i;
         }
         else {
           for (sure=0,to=holdsval,from=sval; sure==0 ;from++,to++){
             if(*from > *to){ /* new is bigger */
               j=i;
               holdsval=sval;
               sure=1;
             }
             if(*from < *to) /* new one is smaller */
               sure=1;
             if(*from=='\0') /*from=to=0, so strings are identical*/
               sure=1;
           }
         }
         break;
       case 38: /* lexlow, final operation at end of loop */
         if(i==3){
           holdsval=sval;
           j = i;
         }
         else {
           for (sure=0,to=holdsval,from=sval; sure==0 ;from++,to++){
             if(*from < *to){ /* new is smaller */
               j=i;
               holdsval=sval;
               sure=1;
             }
             if(*from > *to) /* new one definitely is not smaller */
               sure=1;
             if(*from=='\0') /*from=to=0, so strings are identical*/
               sure=1;
           }
         }
         break;
       case 39: /* string del */
         if(i==3){
           temp=malloc((1+strlen(sval))*sizeof(char));
           if(temp==NULL){
             (void) fprintf(stderr,"miniproc, fatal error, f$evaluate stringdel, failed memory allocation\n");
             exit(EXIT_FAILURE);
           }
           (void) strcpy(temp,sval);
         }
         else {
           if(strlen(sval)==0){
             (void) fprintf(stderr,"miniproc, fatal error, f$evaluate stringdel, patterns has zero length\n");
             exit(EXIT_FAILURE);
           }
           to=strstr(temp,sval);
           if(to!=NULL){  /* remove the match */
             for(k=1,from=to;*from!='\0';k++,from++){           
                if(k>strlen(sval)){
                   *to=*from;
                   to++;
                }
             }
             *to='\0';
           }
           if(i==stackptr){
             if(rssize < 1 + strlen(temp))
               enlarge_string(&rsval,&rssize,strlen(temp),"f$evaluate stringdel failed while enlarging result string");
             (void) strcpy(rsval,temp);
             free(temp);             
             temp=NULL;
           }
         }
         break;
       case 40: /* resize (string) */
         if(ival <=0){
           (void) fprintf(stderr,"miniproc, fatal error, f$evaluate resize, size is invalid\n");
           exit(EXIT_FAILURE);
         }
         ifirst=strlen(result->string);
         result->string=realloc(result->string,ival*sizeof(char));
         if(result->string==NULL){
           (void) fprintf(stderr,"miniproc, fatal error, f$evaluate resize could not obtain or release memory\n");
           exit(EXIT_FAILURE);
         }
         rssize=ival;
         if(ifirst>=ival){ /*it truncated */
           *instatus=0;
           for (to=result->string, j=1; j<ival; j++,to++){}
           *to='\0';
         }
         else { /*no truncation*/
           *instatus=1;
         }
         free(rsval);
         rsval=NULL;
         return;
       case 41: /* Result is getenv (op1) */
         holdsval = getenv(sval);
         if(holdsval == NULL){
           *instatus=2;  /* status is true, but = 2 to show it wasn't there */  
           rssize=1;
           *rsval='\0';
         }
         else{
           *instatus=1;  /* status is true, and 1, to show that it is there */
           if(strlen(holdsval) > rssize){
             (void) fprintf(stderr,"miniproc, fatal error, getenv string is too long\n");
             exit(EXIT_FAILURE);
           }
           (void) strcpy(rsval,holdsval);
           free(holdsval);
           holdsval=NULL;
         }
         break;
       default:
         (void) fprintf(stderr,"miniproc, fatal programming error, f$evaluate unknown operator\n");
         exit(EXIT_FAILURE);
    } /* end of switch */
  } /* end of for loop */

  /* some operations have postop cleanup */

  switch (opptr->opnum) {
     case 28:  /* shortest, final operation at end of loop */
     case 29:  /* longest, final operation at end of loop */
       resolveright(actionstack[j],tsval,&ival,&type,0);
       if(fsize + 1 > result->ssize)
         enlarge_string(&result->string,&result->ssize,fsize,
           "f$evaluate shortest or longest failed while enlarging result string");
       (void) strcpy(result->string,sval);
       free(rsval);
       rsval=NULL;
       return;
     case 37:  /* lexhigh, final operation at end of loop */
     case 38:  /* lexlow, final operation at end of loop */
       fsize=strlen(holdsval);
       if(fsize +1 > result->ssize)
         enlarge_string(&result->string,&result->ssize,fsize,
           "f$evaluate lexhigh or lexlow failed while enlarging result string");
       (void) strcpy(result->string,holdsval);
       free(rsval);
       rsval=NULL;
       return;
  }

  /* for most operations, copy rsval or rival to result */
  if(result->type == INTVAR){
    result->ival=rival;
  }
  else {  /* must be STRINGVAR */
    fsize=strlen(rsval)+1;
    if(fsize + 1 > result->ssize) /* won't fit */
         enlarge_string(&result->string,&result->ssize,fsize,
           "f$evaluate failed while enlarging result string");
       (void) strcpy(result->string,rsval);
     free(rsval);
     rsval=NULL;
  }

}


/*------------------------------------
shift the actionstack left "left" positions.
Used to lop off tokens like "if label" or
"elseif label".  It can also shift right, but that
leaves the now emptied positions in actionstack at NULL.
*/

void shiftactive(int left){
int i;
  if(left > 0){
    for(i=0 ; i <= stackptr-left ;  actionstack[i]=actionstack[i+left], i++){}
    stackptr = stackptr - left;
    return;
  }
  if(left < 0){
    if(stackptr - left > MAXSTACK -1){
      (void) fprintf(stderr,"miniproc, fatal error, command has too many operands\n");
      exit(EXIT_FAILURE);
    }
    stackptr = stackptr - left;
    for(i=stackptr; i >= -left ;  actionstack[i]=actionstack[i+left], i--){}
    return;
  }
} /* end of shiftactive */



/* --------------------------------------
   do_function - routine where all of the f$whatever
   functions live (or at least, a stub that calls out
   to them is located.)
*/


void do_function(void){
char ctemp[MAXVARLEN];
char *temp=&ctemp[0];
char *sval;
char **tsval=&sval;
int ival;
enum istype type;
char disposition[3];
static struct stat *statbuffer=NULL;
VARIABLE *tempvar;
char *newline;
int filenum;
FILE *ffrom;
int isabreak;

  if((*trace & TRACE_FUNC)==TRACE_FUNC)(void) fprintf(stderr," function    >%s< \n",actionstack[0]);

  wasfevaluate=0;

/* f$break implementation, this intentionally falls through into f$exit! */

  if(strcmp(actionstack[0],"f$break")==0){
    if(ifstack[ifptr].type == IFIN){
      (void) fprintf(stderr,"miniproc, fatal error, f$break executed outside of if/then/else structure\n");
      exit(EXIT_FAILURE);
    }
    (void) strcpy(actionstack[0],"f$exit");
    isabreak=1;  /* only used in f$exit, which follows immediately */
  }
  else {
    isabreak=0;
  }

/* f$exit implementation, do not separate from f$break! */

  if(strcmp(actionstack[0],"f$exit")==0){
    ival=EXIT_SUCCESS;
    switch (stackptr){
      case 0:
        break;
      case 1:
      case 2: /* doesn't matter what 2nd param is, just so long as it exists */
        resolveright(actionstack[1],tsval,&ival,&type,0);
        if(type != INTVAR){
          (void) fprintf(stderr,"miniproc, fatal error, f$exit had noninteger parameter\n");
          exit(EXIT_FAILURE);
        }
        break;
      default:
        (void) fprintf(stderr,"miniproc, fatal error, f$exit, too many parameters\n");
        exit(EXIT_FAILURE);
    }

    if(stackptr==2){ /* Program triggered an immediate exit, pass ival to OS */
        exit(ival);
    }

    if(fromsource!=INFROMFILE){
      (void) fprintf(stderr,"miniproc, fatal error, f$exit or f$break used to exit a macro\n");
      exit(EXIT_FAILURE);
    }

    if (finc > 0){   /* close existing file and keep going */
      if(fclose(fin[finc]) != 0){
        (void) fprintf(stderr,"miniproc, fatal error, f$exit, could not close current input file \n");
        exit(EXIT_FAILURE);
      }
      finc--;

      /* if this is an f$exit verify that the last tag in the ifstack is an
         IN, then roll it up. If it is an f$break, we don't do this test.
         ifptr better be valid or there is a bug elsewhere in the program!*/

      if(ifstack[ifptr].type != IFIN){
        if(isabreak == 0){
          (void) fprintf(stderr,"miniproc, fatal error, at f$exit incomplete if/then/else structure detected\n");
          exit(EXIT_FAILURE);
        }
        else { /* roll up the if stack to the next file mark */
          for( ; (ifstack[ifptr].type!=IFIN && ifptr>=0) ;ifptr-- ){}
        }
      }

      yankaltprefix(); /* restore altprefix which was in force when the 
                          f$in was invoked */

      ifptr--;
      ifscan = NO;
/* on return from an f$exit we should NEVER be scanning, since we are 
   absolutely in a block of executing code - subsequent elseif or else
   may trigger a scan though.

      if(ifptr >= 0 ){
        if( (ifstack[ifptr].type == IFLABEL) && (ifstack[ifptr].doneit == YES))
        ifscan=YES;
      }
*/
      *instatus=1;
      return;
    }
    else {           /* all files closed, exit program */
      if(ifptr >= 0){
        (void) fprintf(stderr,"miniproc, fatal error, at program exit incomplete if/then/else structure detected\n");
        exit(EXIT_FAILURE);
      }
      exit(ival);
    }
    *instatus=1;
    return;

  } /* end of f$exit */

/* f$in implementation */

  if(strcmp(actionstack[0],"f$in")==0){

    if( (*safety & SAFE_IN) == SAFE_IN){
      (void) fprintf(stderr,"miniproc, fatal safety error, f$in attempted\n");
      exit(EXIT_FAILURE);
    }
    switch (stackptr) {
       case 0:
         (void) fprintf(stderr,"miniproc, fatal error, f$in, not enough parameters\n");
         exit(EXIT_FAILURE);
       case 2:
         resolveright(actionstack[2],tsval,&ival,&type,0);
         if(type != INTVAR){
           (void) fprintf(stderr,"miniproc, fatal error, f$in parameter 2 must be a filenumber\n");
           exit(EXIT_FAILURE);
         }
         if(ival <10 || ival > 19){
           (void) fprintf(stderr,"miniproc, fatal error, f$in filenumber is illegal\n");
           exit(EXIT_FAILURE);
         }
         filenum=ival;
         break;
       case 1:
         filenum=10;
         break;
       default:
         (void) fprintf(stderr,"miniproc, fatal error, f$in, too many parameters\n");
         exit(EXIT_FAILURE);
    }

    resolveright(actionstack[1],tsval,&ival,&type,0);
    if(type != STRINGVAR){
      (void) fprintf(stderr,"miniproc, fatal error, f$in parameter must be a filename\n");
      exit(EXIT_FAILURE);
    }

    switch (filenum){
      case 10: /* the command stream = special case */
        if (finc < 9){   /* space to open a new file */
          finc++;
          if( (*safety & SAFE_PATH) == SAFE_PATH)pathsafe(sval);
          if(strlen(sval) > MAXFNAME - 1 ){
            (void) fprintf(stderr,"miniproc, fatal error, f$in, input filename too long\n");
            exit(EXIT_FAILURE);
          }
          fin[finc]=fopen(sval,"r");
          if(fin[finc]==NULL){
            (void) fprintf(stderr,"miniproc, couldn't open %s \n",sval);
            exit(EXIT_FAILURE);
          }
          (void) strcpy(&finname[finc][0],sval);

  
          /* drop a tag into the ifstack so that label searches don't go up
             past this point */
  
          ifptr++;
          if(ifptr < MAXIFDEPTH){
            ifstack[ifptr].type = IFIN;
            stuffaltprefix();
          }
          else{
            (void) fprintf(stderr,"miniproc, fatal error, if stack overflow when macro %s was invoked \n",actionstack[0]);
            exit(EXIT_FAILURE);
          }
        }
        else {           /* maximum number of files already open */
          (void) fprintf(stderr,"miniproc, fatal error, f$in %s, max. # of files already open \n",sval);
          exit(EXIT_FAILURE);
        }
        break;
      default: /*files 11->19 */
        if(finr[filenum-11]!=NULL){
          if(fclose(finr[filenum-11])!=0){
            (void) fprintf(stderr,"miniproc, fatal error, f$out, could not close current output file\n");
            exit(EXIT_FAILURE);
          }
          finr[filenum-11]=NULL;
        }

        *instatus=1;
        if(strlen(sval)==0)return;  /* empty filename closes the file */

        if( (*safety & SAFE_PATH) == SAFE_PATH)pathsafe(sval);
        finr[filenum-11]=fopen(sval,"r");
        if(finr[filenum-11] == NULL){           /* oops, bad input file */
          (void) fprintf(stderr,"miniproc, fatal error, f$in, could not open %s \n",sval);
          exit(EXIT_FAILURE);
        }
    }

    *instatus=1;
    return;

  } /* end of f$in */

/* f$out implementation */

  if(strcmp(actionstack[0],"f$out")==0){

    if( (*safety & SAFE_OUT) == SAFE_OUT){
      (void) fprintf(stderr,"miniproc, fatal safety error, f$out attempted\n");
      exit(EXIT_FAILURE);
    }
    switch (stackptr){
      case 0:
        (void) fprintf(stderr,"miniproc, fatal error, f$out, no filename supplied\n");
        exit(EXIT_FAILURE);
      case 1:
        (void) strcpy(disposition,"w");
        filenum=0; /*default to first filenum*/
        break;
      case 3: /* case 3 intentionally falls through into case 2 !!!
                 Use resolveright so that "append" or append can be
                 the third argument.  Former form is preferred. */
        resolveright(actionstack[3],tsval,&ival,&type,2);
        (void) strcpy(disposition,"X");
        if(strcmp(sval,"append")==0)
          (void) strcpy(disposition,"a");
        if(strcmp(sval,"new")==0)
          (void) strcpy(disposition,"w");
        if(strcmp(disposition,"X")==0){
          (void) fprintf(stderr,"miniproc, fatal error, f$out disposition must be append or new\n");
          exit(EXIT_FAILURE);
        }
      case 2:
        resolveright(actionstack[2],tsval,&ival,&type,0);
        if(type != INTVAR){
          (void) fprintf(stderr,"miniproc, fatal error, f$out filenumber is invalid\n");
          exit(EXIT_FAILURE);
        }
        else {
          if(ival <0 || ival >9){
            (void) fprintf(stderr,"miniproc, fatal error, f$out filenumber is invalid\n");
            exit(EXIT_FAILURE);
          }
          else {
            filenum=ival;
          }
        }
        break;
      default:
        (void) fprintf(stderr,"miniproc, fatal error, f$out, too many parameters\n");
        exit(EXIT_FAILURE);
    }

    resolveright(actionstack[1],tsval,&ival,&type,0);
    if(type != STRINGVAR){
      (void) fprintf(stderr,"miniproc, fatal error, f$out parameter must be a filename\n");
      exit(EXIT_FAILURE);
    }

    if(fout[filenum]!=NULL){
       if(fclose(fout[filenum])!=0){
         (void) fprintf(stderr,"miniproc, fatal error, f$out, could not close current output file\n");
         exit(EXIT_FAILURE);
       }
       fout[filenum]=NULL;
    }

    *instatus=1;
    if(strlen(sval)==0)return;  /* empty filename closes the file */

    if( (*safety & SAFE_PATH) == SAFE_PATH)pathsafe(sval);
    fout[filenum]=fopen(sval,&disposition[0]);
    if(fout[filenum] == NULL){           /* oops, bad output file */
      (void) fprintf(stderr,"miniproc, fatal error, f$out, could not open %s \n",sval);
      exit(EXIT_FAILURE);
    }
    *instatus=1;
    return;

  } /* end of f$out */

/* f$read implementation */

  if(strcmp(actionstack[0],"f$read")==0){

    switch (stackptr){
      case 2:
        break;
      default:
        (void) fprintf(stderr,"miniproc, fatal error, f$read, wrong number of parameters\n");
        exit(EXIT_FAILURE);
    }

    tempvar=findvar(actionstack[1]);
    if(tempvar==NULL){
      (void) fprintf(stderr,"miniproc, fatal error, f$read, string variable does not exist\n");
      exit(EXIT_FAILURE);
    }
    if(tempvar->type != STRINGVAR){
      (void) fprintf(stderr,"miniproc, fatal error, f$read, variable is not a string\n");
      exit(EXIT_FAILURE);
    }

    resolveright(actionstack[2],tsval,&ival,&type,0);
    if(type != INTVAR){
      (void) fprintf(stderr,"miniproc, fatal error, f$read 2nd parameter must be a filenumber\n");
      exit(EXIT_FAILURE);
    }

    switch (ival) {
      case 10:
        ffrom=fin[finc];
        break;
      case 11:
      case 12:
      case 13:
      case 14:
      case 15:
      case 16:
      case 17:
      case 18:
      case 19:
        ffrom=finr[ival-11];
        break;
      default:
        (void) fprintf(stderr,"miniproc, fatal error, f$read filenumber is invalid\n");
        exit(EXIT_FAILURE);
    }

    if(fgets(tempvar->string,tempvar->ssize,ffrom) == NULL){
      *instatus=0;
      return;
    }

    /* if the whole line fit, there will be a \n on the end, if not, string 
       was too small to hold it */

    newline=strstr(tempvar->string,"\n");
    if(newline == NULL){ /* bad news, string truncated */
      (void) fprintf(stderr,"miniproc, fatal error, f$read, %s too small to hold string\n",actionstack[1]);
      exit(EXIT_FAILURE);
    }
    *newline='\0';
    *instatus=1;
    return;

  } /* end of f$read */

/* f$write implementation */

  if(strcmp(actionstack[0],"f$write")==0){

    switch (stackptr){
      case 2:
        break;
      default:
        (void) fprintf(stderr,"miniproc, fatal error, f$write, wrong number of parameters\n");
        exit(EXIT_FAILURE);
    }


    resolveright(actionstack[2],tsval,&ival,&type,0);
    if(type != INTVAR){
      (void) fprintf(stderr,"miniproc, fatal error, f$read 2nd parameter must be a filenumber\n");
      exit(EXIT_FAILURE);
    }
    if(ival >= 0 && ival <=9){
      filenum=ival;
    }
    else {
      (void) fprintf(stderr,"miniproc, fatal error, f$write filenumber is invalid\n");
      exit(EXIT_FAILURE);
    }

    resolveright(actionstack[1],tsval,&ival,&type,0);
    if(type != STRINGVAR){
      (void) fprintf(stderr,"miniproc, fatal error, f$write, variable is not a string\n");
      exit(EXIT_FAILURE);
    }

    if(fprintf(fout[filenum],"%s\n",sval) <= 0){
      *instatus=0;
    }
    else {
      *instatus=1;
    }
    return;

  } /* end of f$write */

/* f$<- implementation. Very similar to f$evaluate*/

  if(strcmp(actionstack[0],"f$<-")==0){
    shiftactive(-1); /* make one more slot */
    actionstack[0]=&fevaluateconstant[0]; /* now it's an f$evaluate */
    actionstack[1]=&resultconstant[0];        /* and result is RESULT */
    do_evaluate();
    wasfevaluate=1;
    return;
  }

/* f$evaluate implementation */

  if(strcmp(actionstack[0],"f$evaluate")==0){
    do_evaluate();
    wasfevaluate=1;
    return;
  }

/* f$type implementation */
  if(strcmp(actionstack[0],"f$type")==0){
    switch (stackptr){
      case 0:
        (void) fprintf(stderr,"miniproc, fatal error, f$type, no variable name supplied\n");
        exit(EXIT_FAILURE);
      case 1:
        break;
      default:
        (void) fprintf(stderr,"miniproc, fatal error, f$type, too many parameters\n");
        exit(EXIT_FAILURE);
    }

    tempvar=findvar(actionstack[1]);
    if(tempvar==NULL){
      *instatus=0;
      return;
    }
    else {
      switch (tempvar->type){
         case INTVAR:
           *instatus=1;
           return;
         case STRINGVAR:
           *instatus=2;
           return;
         case MACRO:
           *instatus=3;
           return;
      }
    }
  } /* end of f$type */

/* f$file_info implementation */

  if(strcmp(actionstack[0],"f$file_info")==0){
    if( (*safety & SAFE_FILE) == SAFE_FILE){
      (void) fprintf(stderr,"miniproc, fatal safety error, f$file_info attempted\n");
      exit(EXIT_FAILURE);
    }
    switch (stackptr){
      case 0:
        (void) fprintf(stderr,"miniproc, fatal error, f$file_info, no filename supplied\n");
        exit(EXIT_FAILURE);
      case 1:
        break;
      default:
        (void) fprintf(stderr,"miniproc, fatal error, f$file_info, too many parameters\n");
        exit(EXIT_FAILURE);
    }

    if( (*safety & SAFE_PATH) == SAFE_PATH)pathsafe(actionstack[1]);
    if(statbuffer==NULL){
      statbuffer=malloc(sizeof(struct stat));
      if(statbuffer==NULL){
        (void) fprintf(stderr,"miniproc, fatal error, f$file_info, memory allocation error\n");
        exit(EXIT_FAILURE);
      }
      addvar("file_exists", NULL, 0, INTVAR, IS_ORDINARY);
      addvar("file_size", NULL, 0, INTVAR, IS_ORDINARY);
      addvar("file_modified", NULL, 0, INTVAR, IS_ORDINARY);
    }

    if(stat(actionstack[1],statbuffer) == 0)
      *instatus=1;
    else
      *instatus=0;

    setvar("file_exists", NULL, *instatus, INTVAR, IS_ORDINARY);
    setvar("file_size", NULL, statbuffer->st_size, INTVAR, IS_ORDINARY);
    setvar("file_modified", NULL, statbuffer->st_mtime, INTVAR, IS_ORDINARY);

    *instatus=1;
    return;

  } /* end of f$file_info */

/* f$macro_record implementation */

  if(strcmp(actionstack[0],"f$macro_record")==0){
    switch (stackptr){
      case 0:
        (void) fprintf(stderr,"miniproc, fatal error, f$macro_record needs a macro name\n");
        exit(EXIT_FAILURE);
      case 1:
        (void) strcpy(deck,"f$macro_end");
        break;
      case 2:
        (void) strcpy(deck,actionstack[2]);
        break;
      default:
        (void) fprintf(stderr,"miniproc, fatal error, f$macro_record, too many parameters\n");
        exit(EXIT_FAILURE);
    }

    macro_record(actionstack[1], NULL); /* create a new macro and set it to record */
    *instatus=1;
    return;

  } /* end of f$macro_record */

/* f$macro_repeat implementation */

  if(strcmp(actionstack[0],"f$macro_repeat")==0){
    switch (stackptr){
      case 1:
      case 2:
      case 3:
      case 4:
        resolveright(actionstack[1],tsval,&ival,&type,1); /* note mode = 1 !*/
        if(type != STRINGVAR){
          (void) fprintf(stderr,"miniproc, fatal error, f$macro_repeat invalid macro name\n");
          exit(EXIT_FAILURE);
        }
        break;
      default:
         (void) fprintf(stderr,"miniproc, fatal error, f$macro_repeat illegal number of parameters\n");
         exit(EXIT_FAILURE);
    }

    macro_repeat(sval); /* set the repeat counts on this one macro */
    *instatus=1;
    return;

  } /* end of f$macro_repeat */

/* deck/f$macro_end implementation */

  if(strcmp(actionstack[0],"f$macro_end")==0){
 
    if(stackptr >= 2){
      (void) fprintf(stderr,"miniproc, fatal error, deck/f$macro_end, too many parameters\n");
      exit(EXIT_FAILURE);
    }

    if (recmacro == NULL){
      (void) fprintf(stderr,"miniproc, fatal error, cannot end recording a macro before starting\n");
      exit(EXIT_FAILURE);
    }

    /* check the state of the macro, it must not be RECORDING */

    if(recmacro->mstate != RECORDING){
      (void) fprintf(stderr,"miniproc, fatal error, nothing recorded in macro %s\n",recmacro->name);
      exit(EXIT_FAILURE);
    }

    /* reset internal bits so that it can be played */

    recmacro->mstate=DONE;
    recmacro->macro = recmacro->string;  /* pointer back to the beginning */
    recmacro->ival=0;               /* count of positions back to the beginning */
    recmacro = NULL; /* dereference the recording macro */

    /* reset normal line processing */

    howtohandle = HANDLE_NORMAL;
    *instatus=1;
    return;

  } /* end of f$macro_end */

/* f$macro_return implementation */

  if(strcmp(actionstack[0],"f$macro_return")==0){
    macro_return(0);    
    return;
  } /* end of f$macro_return */

/* f$macro_break implementation */

  if(strcmp(actionstack[0],"f$macro_break")==0){
    macro_return(1);    
    *instatus=1;
    return;
  } /* end of f$macro_return */

  (void) fprintf(stderr,"miniproc, fatal error, no such function as %s\n",actionstack[0]);
  exit(EXIT_FAILURE);

} /* end of do_function */

/* ---------------------------------------------------------
   do_macro(), start playing the defined macro.
*/

void do_macro(void){
  int i;
  char cvarname[MAXVARLEN];
  char *varname=&cvarname[0];
  char *sval;
  char **tsval=&sval;
  int ival;
  enum istype type;
  VARIABLE *play;
  int doit;

  if(stackptr > 10){
    (void) fprintf(stderr,"miniproc, fatal error, more than 9 parameters passed to macro %s \n",actionstack[0]);
    exit(EXIT_FAILURE);
  }

  if((*trace & TRACE_MACRO)==TRACE_MACRO)(void) fprintf(stderr," macro       >%s<\n",actionstack[0]);
  isactivemacro++;
  if(isactivemacro > MAXMACRODEPTH - 1){
    (void) fprintf(stderr,"miniproc, fatal error, call stack exhausted at macro %s \n",actionstack[0]);
    exit(EXIT_FAILURE);
  }

  activemacro[isactivemacro]=findvar(actionstack[0]);
  play=activemacro[isactivemacro];
  if(play == NULL){
    (void) fprintf(stderr,"miniproc, fatal error, attempt to execute nonexistant macro %s \n",actionstack[0]);
    exit(EXIT_FAILURE);
  }

  if(play->mstate != DONE){
    (void) fprintf(stderr,"miniproc, fatal error, attempt to recursively execute macro %s \n",actionstack[0]);
    exit(EXIT_FAILURE);
  }

  /* make sure that all max counters aren't zero, if they are, this macro 
     has been set to loop zero times, so roll up and return*/

  for (doit=0, i=0 ; i<=2 ; doit=doit+play->mcmax[i],i++){}
  if(doit==0){   /* all indices are zero, so skip the macro */
    play=NULL;
    isactivemacro--;
    *instatus=0;  /*return a status of 1 for this case*/
    return;
  }

  /* Looks like the macro is ok to execute. 
     Set up the internal loop counters and P and MC parameters */

  play->mstate=PLAYING;
  setcounters(play,1,1);  /* init counters and visible counters */

  /* local variables must be for CALLING macro, but at this point
     we have already set isactivemacro to indicated CALLED macro.
     This will cause local variables to resolve to the CALLED
     macro, rather than to the CALLING macro.  Kick the isactivemacro
     back to the CALLING macro temporarily, then set it back
     again to the called. */

  isactivemacro--;
  for (i=1;i<=9, i<=stackptr ;i++){
    (void) sprintf(varname,"P%d",i);
    resolveright(actionstack[i],tsval,&ival,&type,0);
    setvar(varname, sval , ival, type, IS_SPECIAL);
  }
  isactivemacro++;

  nextfromsource=INFROMMACRO;

  /* drop a tag into the ifstack so that label searches don't go up
     past this point */

  ifptr++;
  if(ifptr < MAXIFDEPTH){
    ifstack[ifptr].type=IFMACRO;
    stuffaltprefix();
  }
  else{
    (void) fprintf(stderr,"miniproc, fatal error, if stack overflow when macro %s was invoked \n",actionstack[0]);
    exit(EXIT_FAILURE);
  }

  /* reset user definable variable altprefix to match what was in force
     when the macro was recorded */

  setaltprefix(play->altprefix);

} /*end of do_macro */

/* --------------------------------------------------
   test().  If the test *fails* then start scanning.
   If it passes, then don't start scanning, but set the doneit
   flag.  If invert is YES, then invert the logic of the test.
*/

void test(char *sval, int ival, enum istype type, enum ifdoneit invert){
  switch (type) {
    case INTVAR:
      if(  (invert==NO  && ival == 0) || 
           (invert==YES && ival != 0) ){
        ifscan = YES;
      }
      else {
        ifscan = NO;
        ifstack[ifptr].doneit = YES;
      }
      break;
    case STRINGVAR:
      if(  (invert==NO  && (sval == NULL || strlen(sval) == 0))  ||
           (invert==YES && (sval != NULL && strlen(sval) != 0))  ){
        ifscan = YES;
      }
      else {
        ifscan = NO;
        ifstack[ifptr].doneit = YES;
      }
      break;
    case MACRO:
      (void) fprintf(stderr,"miniproc, fatal error, TEST resolves to macro name, not integer or string variable \n");
      exit(EXIT_FAILURE);
  }
}

/* ----------------------------------------
  do_ifelse() has most of the logic for if/else/elseif/endif,
  the rest being in macro_return.  If invert is YES it means
  ifnot or elseifnot. This is stored in the ifstack and 
  used to invert the logic when the final test completes */

void do_ifelse(enum ifstate instate, enum ifdoneit invert){
char slabel[MAXLABELLEN];
char *sval;
char **tsval=&sval;
int ival;
enum istype type;
VARIABLE *play;
int i;

  switch (instate){
     case IF:
     case ELSEIF:
       if(stackptr == 1){
         (void) fprintf(stderr,"miniproc, fatal error, if or elseif requires missing TEST\n");
         exit(EXIT_FAILURE);
       }
       if(stackptr == 0){
         (void) fprintf(stderr,"miniproc, fatal error, if or elseif missing mandatory LABEL\n");
         exit(EXIT_FAILURE);
       }
       break;
     case ELSE:
     case ENDIF:
     default:
       if(stackptr > 1){
         (void) fprintf(stderr,"miniproc, fatal error, else or endif with extra parameters\n");
         exit(EXIT_FAILURE);
       }
       if(stackptr == 0){
         (void) fprintf(stderr,"miniproc, fatal error, else or endif missing mandatory LABEL\n");
         exit(EXIT_FAILURE);
       }
       break;    
  }

  /* number of parameters are ok , The second one is ALWAYS an immediate label,
     move it into sval */

  if(strlen(actionstack[1]) >= MAXLABELLEN){
    (void) fprintf(stderr,"miniproc, fatal error, LABEL is too long: %s\n",sval);
    exit(EXIT_FAILURE);
  }

  sval=&slabel[0];
  (void) strcpy(sval,actionstack[1]);

  /* very abbreviated checking when scanning.  Looking for else/elseif/
     endif lines with matching labels, if they don't match, just return.
     Structure within scanned regions may be incorrect, since logic
     checking only occurs when we pass through a region in normal mode */

  if(ifscan == YES){
    if(ifptr < 0 || ifptr > MAXIFDEPTH -1){
      (void) fprintf(stderr,"miniproc, fatal error, invalid if/elseif/else/endif structure, label= %s\n",sval);
      exit(EXIT_FAILURE);
    }
    if(ifstack[ifptr].state != IFLABEL ){
      (void) fprintf(stderr,"miniproc, fatal error, invalid if/elseif/else/endif structure, label= %s\n",sval);
      exit(EXIT_FAILURE);
    }
    if( strcmp(ifstack[ifptr].string,sval) !=0 ) /* label doesn't match, so do nothing */
      return;
  }


  /* verify that we can use this label in this position in this manner.  
     If this is not IF, it must be the last thing on the stack,
     if it is IF, it must not be elsewhere on the stack up to the last 
     MACRO or IN tag. */

  if(instate == IF){
    for(i=ifptr ; i>=0 ; i--){
      switch (ifstack[i].type){
        case IFIN:        /* don't care what labels are in other modules */
        case IFMACRO:
          i=-1; /* force exit from for loop */
          break;
        case IFLABEL:  /* is the label already in the stack for this module? */
          if(strcmp(ifstack[i].string,sval)==0){  /* it is */
              (void) fprintf(stderr,"miniproc, fatal error, invalid if/elseif/else/endif structure, label= %s\n",sval);
              exit(EXIT_FAILURE);
          }
          break;
      }
    }
  }
  else {
    if ( strcmp( ifstack[ifptr].string,sval) != 0 ){
      (void) fprintf(stderr,"miniproc, fatal error, invalid if/elseif/else/endif structure, label= %s\n",sval);
      exit(EXIT_FAILURE);
    }
  }

  /* further verify the if/elseif/else/endif structure, but only if the 
     label matches! */

  if(ifptr >= 0){
    if(ifstack[ifptr].type==IFLABEL){  /* don't care about IFMACRO or IFIN */
      if(strcmp(sval,ifstack[ifptr].string)==0){
        if( ((ifstack[ifptr].state == IF) && (instate == IF)) ||
            ((ifstack[ifptr].state == ELSEIF) && (instate == IF)) ||
            ((ifstack[ifptr].state == ELSE) && (instate != ENDIF)) ){
          (void) fprintf(stderr,"miniproc, fatal error, invalid if/elseif/else/endif structure, label= %s\n",sval);
          exit(EXIT_FAILURE);
        }
    }
    }
  }

  switch (instate){
    case IF:  /* start a new if/elseif/else/endif structure, always test */
      ifptr++;
      if(ifptr >= MAXIFDEPTH){
        (void) fprintf(stderr,"miniproc, fatal error, if structures nested too deeply\n");
        exit(EXIT_FAILURE);
      }
      (void) strcpy(ifstack[ifptr].string,sval);
      ifstack[ifptr].type   = IFLABEL;
      ifstack[ifptr].doneit = NO;
      ifstack[ifptr].invert = invert;
      ifscan=NO;
      break;
    case ELSEIF: /* only test if a block has not already been done */
      if( ifstack[ifptr].doneit == YES){
        ifscan=YES;
        return;
      }
      else {
        ifscan=NO;
        ifstack[ifptr].invert = invert;
      }
      break;
    case ELSE:  /*do this block if no other blocks have marked doneit,
                  otherwise start scanning.  Never test*/
      if( ifstack[ifptr].doneit == NO){
        ifscan=NO;
        ifstack[ifptr].doneit=YES;
      }
      else 
        ifscan=YES;
      return;
    case ENDIF:  /* end this if/elseif/else/endif if block, reset ifscan to 
                    match last block.*/
      ifptr--;
      ifscan = NO;
/* next encountered if,else,elseif will set ifscan as required, but there 
   may be more valid statements in this block following an endif, example:

  if   blah
  else blah
    if    blah2
    endif blah2
    statements that need to be done following the endif, so ifscan must be NO
  endif blah
*/
      return;
  }

  /* wasn't that fun?  Now the ifstack is set correctly, and we can move on
     to the heart of the matter, which is what to do about the actual tests
     for IF and ELSEIF.  The simplest case is that it is just a variable
     so try that first.  For macros and functions rotate the actionstack
     and call do_macro or do_function.  If the test FAILS, then start 
     scanning.  Resolveright is used in a mode which will return
     a macro name in sval, and a type of MACRO, if it sees one.

     If a macro is used as the test parameter, control passes into
     that macro.  Cannot know until that macro completes what the status
     will be, and that will happen only on the final macro_return.
     Look in macro_return for the actions.  */

  resolveright(actionstack[2],tsval,&ival,&type,2); /**/
  switch (type){
    case MACRO:   
      shiftactive(2);
      do_macro();
      break;
    case INTVAR:  /* this one is easy */
      test(sval,ival,type,invert);
      break;
    case STRINGVAR:  /* may be a function, or a real stringvar */
      if(strncmp(sval,"f$",2)==0){  /* it is a function */
        shiftactive(2);
        do_function();
        if(wasfevaluate){
          if(*instatus == 0){
            (void) fprintf(stderr,"miniproc, fatal error, f$evaluate error in if/elseif/else/endif structure \n");
            exit(EXIT_FAILURE);
          }
          resolveright(actionstack[1],tsval,&ival,&type,0);
          test(sval,ival,type,invert);
        }
        else {
          test(NULL,*instatus,INTVAR,invert);
        }
      }
      else{ /* it is a really a string */
        test(sval,ival,type,invert);
      }
      break;
  }

}

void print_out(char *line){
  if(fout[0]==NULL){
    (void) fprintf(stderr,"miniproc, fatal error, write to output with no output file defined \n");
    (void) fprintf(stderr,"miniproc, fatal error, at: %s \n",line);
    exit(EXIT_FAILURE);
  }
  if((*trace & TRACE_OUTPUT)==TRACE_OUTPUT)(void) fprintf(stderr,"   output    >%s<\n",line);
  (void) fprintf(fout[0],"%s\n",line);
}


void parse_and_run(char *line){
int equalpos;
int spacepos;
char cvarleft[MAXVARLEN];
char *varleft=&cvarleft[0];
char *varright;
char *temp;
VARIABLE *ourvar;
int ival;
char *sval;
char **tsval=&sval;
enum istype type;
int i,bail;
char *from;
char *to;
int equalcount;

  if((*trace & TRACE_COMMAND)==TRACE_COMMAND)(void) fprintf(stderr,"command line >%s\n",line);

  /* First parsing phase.  See if it fits pattern:
     #__"whatever"  or #__&whatever.  If so, output "whatever"
     and return.  This is shorthand for either of these other
     operations:
         #__a="#__some long string"
         <<a>>
             or
         #__write a 
  */
  
  switch (*line){
    case '&':
      if (ifscan == YES)return; /* none of these operations when scanning */
      from=line;
      from++;
      print_out(from);
      return;
    case '"':
      if (ifscan == YES)return; /* none of these operations when scanning */
      from=line;
      to=strrchr(from,'"');
      if(to == from){
        (void) fprintf(stderr,"miniproc, fatal syntax error, unmatched double quotes\n");
        exit(EXIT_FAILURE);
      }
      *to='\0';
      from++;
      print_out(from);
      return;
    default:  /* do nothing */
      break;
  }

  /* Second parsing phase.  See if it fits pattern:
     name = whatever
     The key point is that there must be an "=" between the first two 
     tokens, maybe with extra space.  If so, it is a simple variable 
     addition or reassignment */

  for( from=line , i=1  ;
       ((*from != '=') && (*from != ' ') && (*from != '\t') && i<=strlen(line))  ;
       i++,from++){}
  if(i>MAXVARLEN-1){
    (void) fprintf(stderr,"miniproc, fatal error, variable name is too long %s\n",line);
    exit(EXIT_FAILURE);
  }
  temp=from;
  for(bail=0, equalcount=0; i<=strlen(line)  ; i++,from++){
    switch (*from){
        case '=':
          equalcount++;
        case '\t':
        case ' ':
          break;
        default:
          bail=1;
          break;
    }
    if(bail==1)break;
  }

  switch (equalcount) {
     case 0:  /* function, macro, if, or something else*/
       break;
     case 1:  /* looks good, exactly one = in this region */
       if (ifscan == YES)return; /* no variable operations when scanning */
       *temp='\0'; /* terminate first token */
       (void) strcpy(varleft,line);
       ourvar= findvar(varleft);
       resolveright(from,tsval,&ival,&type,0);
       if(ourvar==NULL)   /* make a new variable with this name */
         addvar(varleft, sval, ival, type, IS_ORDINARY);
       else               /* reset an existing variable with this name */
         setvar(varleft, sval, ival, type, IS_ORDINARY);
       return;
     default: /*2 or more =, not valid syntax */
       (void) fprintf(stderr,"miniproc, fatal error, too many = in command line: %s\n",line);
       exit(EXIT_FAILURE);
  }

  /* Third parsing phase - the command is either a function, a macro,
     or an if,elseif,else,endif structure.
     Break it up into tokens and stuff them into the action stack. In
     this mode " " has no special meaning - they are just regular characters!!!
     (Translate the tokens only when the stack executes) */

    stackptr=-1;  /* next token goes into the first position in the stack */
    sval = strtok(line," ");
    for (;sval!=NULL;){
         stackptr++;
         if(stackptr > MAXSTACK){
           (void) fprintf(stderr,"miniproc, fatal error, more than %d tokens in a function or macro line \n",MAXSTACK);
           exit(EXIT_FAILURE);
         }
         actionstack[stackptr]=sval;
         sval=strtok(NULL," ");
    }

  /* Fourth parsing phase - decide if it is a macro or a function, and act 
     appropriately */

    if(strcmp(line,"if")==0){      /* it is an if */
      do_ifelse(IF,NO);
      return;
    }
    if(strcmp(line,"ifnot")==0){      /* it is an ifnot */
      do_ifelse(IF,YES);
      return;
    }
    if(strcmp(line,"elseif")==0){      /* it is an elseif */
      do_ifelse(ELSEIF,NO);
      return;
    }
    if(strcmp(line,"elseifnot")==0){      /* it is an elseifnot */
      do_ifelse(ELSEIF,YES);
      return;
    }
    if(strcmp(line,"else")==0){      /* it is an else */
      do_ifelse(ELSE,NO);
      return;
    }
    if(strcmp(line,"endif")==0){      /* it is an endif */
      do_ifelse(ENDIF,NO);
      return;
    }

    if( ifscan == YES)return;          /* scanning for part of if/elseif/else/endif */

    if(strncmp(line,"f$",2)==0){      /* it is a built in function */
      do_function();
      return;
    }
    /* all that is left is that it is a macro */

    if(fromsource == INFROMCOMMAND){
      (void) fprintf(stderr,"miniproc, fatal error, macros may not run from the command line \n");
      exit(EXIT_FAILURE);
    }
    do_macro();

}


/* set the day/date/time variables */
void do_date_time(int first){
#define LOCALMAXBUF 16
  struct  tm  *time_structure;
  time_t time_val;
  char cbuffer[LOCALMAXBUF];
  char *string=&cbuffer[0];

  (void) time(&time_val);
  time_structure = localtime(&time_val);


  if(first){
      addvar("unixtime", NULL, (int) time_val, INTVAR, IS_ORDINARY);
      (void) strftime(string, LOCALMAXBUF, "%a", time_structure); /* Sun - Sat */
      addvar("day", string, 0, STRINGVAR, IS_ORDINARY);
      (void) strftime(string, LOCALMAXBUF, "%b", time_structure); /* Jan - Dec */
      addvar("month", string, 0, STRINGVAR, IS_ORDINARY);
      addvar("wday", NULL, 1+time_structure->tm_wday, INTVAR, IS_ORDINARY); /* weekday 1-7 */
      addvar("yday", NULL, 1+time_structure->tm_yday, INTVAR, IS_ORDINARY); /* yearday 1-365 */
      addvar("dd", NULL, time_structure->tm_mday, INTVAR, IS_ORDINARY); /* date 1-31 */
      addvar("mm", NULL, 1+time_structure->tm_mon, INTVAR, IS_ORDINARY); /* month 1-12*/
      addvar("yyyy", NULL, 1900+time_structure->tm_year, INTVAR, IS_ORDINARY); /* year */
      addvar("hour", NULL, time_structure->tm_hour, INTVAR, IS_ORDINARY); /* hour 0-23 */
      addvar("minute", NULL, time_structure->tm_min, INTVAR, IS_ORDINARY); /* minute 0-59 */
      addvar("second", NULL, time_structure->tm_sec, INTVAR, IS_ORDINARY); /* minute 0-59 */
  }
  else {
      setvar("unixtime", NULL, (int) time_val, INTVAR, IS_ORDINARY);
      (void) strftime(string, LOCALMAXBUF, "%a", time_structure); /* Sun - Sat */
      setvar("day", string, 0, STRINGVAR, IS_ORDINARY);
      (void) strftime(string, LOCALMAXBUF, "%b", time_structure); /* Jan - Dec */
      setvar("month", string, 0, STRINGVAR, IS_ORDINARY);
      setvar("wday", NULL, 1+time_structure->tm_wday, INTVAR, IS_ORDINARY); /* weekday 1-7 */
      setvar("yday", NULL, 1+time_structure->tm_yday, INTVAR, IS_ORDINARY); /* yearday 1-365 */
      setvar("dd", NULL, time_structure->tm_mday, INTVAR, IS_ORDINARY); /* date 1-31 */
      setvar("mm", NULL, 1+time_structure->tm_mon, INTVAR, IS_ORDINARY); /* month 1-12*/
      setvar("yyyy", NULL, 1900+time_structure->tm_year, INTVAR, IS_ORDINARY); /* year */
      setvar("hour", NULL, time_structure->tm_hour, INTVAR, IS_ORDINARY); /* hour 0-23 */
      setvar("minute", NULL, time_structure->tm_min, INTVAR, IS_ORDINARY); /* minute 0-59 */
      setvar("second", NULL, time_structure->tm_sec, INTVAR, IS_ORDINARY); /* minute 0-59 */
  }
#undef LOCALMAXBUF
}

/* create the special variables P1-P9, then all of the standard variables,
     such as, MC*,MX*MAX, dates, and so forth  */

void make_standard_variables(void){
  char cvarname[MAXVARLEN];
  char *varname=&cvarname[0];
  int i;
  VARIABLE *ourvar;

  /* initialize the head and tail variables */

  for(i=0; i < HASHSPACE; i++){
    head[i]=NULL;
    tail[i]=NULL;
  }

  /* initialize finname to be all null strings */

  for(i=0; i < 10; i++){
    finname[i][0]='\0';
  }
  /* put something in deck */

  (void) strcpy(deck,"f$macro_end");

 /* any internal variables linked to user visible variables must be done 
    first, otherwise the pointers are undefined on first use! */

  addvar("trace", NULL, 0, INTVAR, IS_ORDINARY);
  ourvar= findvar("trace");
  trace=&(ourvar->ival);       /* internal trace variable points to
                                variable's integer value */
  addvar("subs", NULL, 1, INTVAR, IS_ORDINARY);
  ourvar= findvar("subs");
  dosubnum=&(ourvar->ival);       /* internal subs variable points to
                                variable's integer value */
  addvar("macrosubs", NULL, 0, INTVAR, IS_ORDINARY);
  ourvar= findvar("macrosubs");
  domacrosubnum=&(ourvar->ival);       /* internal subs variable points to
                                variable's integer value */
  addvar("safety", NULL, 0, INTVAR, IS_ORDINARY);
  ourvar= findvar("safety");
  safety=&(ourvar->ival);       /* internal subs variable points to
                                variable's integer value */
  addvar("altprefix", "", 0, STRINGVAR, IS_ORDINARY);
  ourvar= findvar("altprefix");
  altprefix=ourvar;    /* internal altprefix variable points to
                          variable with same name, NOT string, which may
                          change */

  addvar("STATUS", NULL, 1, INTVAR, IS_ORDINARY);
  ourvar= findvar("STATUS");
  instatus=&(ourvar->ival);       /* internal status variable points to
                                variable's integer value */

  addvar("RESULT", NULL, 1, STRINGVAR, IS_SPECIAL); /* used by f$<- */


  for (i=1;i<=9;i++){
    (void) sprintf(varname,"P%d",i);
    addvar(varname, NULL , 0, STRINGVAR, IS_SPECIAL);
  }

  addvar("MC1", NULL, 1, INTVAR, IS_ORDINARY);
  addvar("MC2", NULL, 0, INTVAR, IS_ORDINARY);
  addvar("MC3", NULL, 0, INTVAR, IS_ORDINARY);
  addvar("MC1MAX", NULL, 1, INTVAR, IS_ORDINARY);
  addvar("MC2MAX", NULL, 0, INTVAR, IS_ORDINARY);
  addvar("MC3MAX", NULL, 0, INTVAR, IS_ORDINARY);
  mcshort[0]=findvar("MC1");  /* shortcuts to these visible variables */
  mcshort[1]=findvar("MC2");
  mcshort[2]=findvar("MC3");
  mcmaxshort[0]=findvar("MC1MAX");
  mcmaxshort[1]=findvar("MC2MAX");
  mcmaxshort[2]=findvar("MC3MAX");

  do_date_time(1); /* 1 creates the variables*/

}

int main(int argc, char *argv[]){
char cfname[MAXFNAME];
char cmacroname[MAXVARLEN];
char cinline[MAXINLINE];
char csumline[MAXINLINE];
char *fname=&cfname[0];
char *macroname=&cmacroname[0];
char *vinline=&cinline[0];
char *sumline=&csumline[0];
char *noleader=&cinline[3];
char *temp;
int prefix;
int instat;
int endstring;
int i;
int ok=1;
int argptr;
int commandmatch;
enum continuetypes {IN_MORE,IN_NEW} linestatus=IN_NEW;

  make_standard_variables();
  
  switch (argc){
    case 0:  /* maybe for some machine with no command line??? */
    case 1:
      (void) fprintf(stderr,"\nEnter the name of the file to process: ");
      if(scanf("%s",fname) != 1){
        (void) fprintf(stderr,"miniproc, aborted processing, no input file\n");
        exit(EXIT_FAILURE);
      }
      nextfromsource=INFROMFILE;
      break;

    case 2:
      if(strlen(argv[1]) > MAXFNAME - 1 ){
        (void) fprintf(stderr,"miniproc, aborted processing, input filename too long\n");
        exit(EXIT_FAILURE);
      }
      (void) strcpy(fname,argv[1]);  /* set the input file name */
      nextfromsource=INFROMFILE;
      break;

    default:
      (void) strcpy(fname,argv[1]);  /* set the input file name */
      nextfromsource=INFROMCOMMAND;
      argptr=2;
      break;
  }

  /* open the input file that is in fname */
  
 if(  (strcmp(fname,"HELP") == 0) ||
      (strcmp(fname,"help") == 0) ||
      (strcmp(fname,"-help") == 0) ||
      (strcmp(fname,"-HELP") == 0) ||
      (strcmp(fname,"-h") == 0) ||
      (strcmp(fname,"-H") == 0) ||
      (strcmp(fname,"?") == 0)){
    (void) fprintf(stderr,"see the file miniproc.doc for miniproc information\n");
    exit(EXIT_FAILURE);
  }

  finc=0;
  fin[finc]=fopen(fname,"r");
  if(fin[finc]==NULL){
    (void) fprintf(stderr,"miniproc, couldn't open %s \n",fname);
    exit(EXIT_FAILURE);
  }
  (void) strcpy(&finname[finc][0],fname);


  /* enter the main loop, which just keeps reading, processing, reading,
     processing.  However, it may either be reading from a file or out of
     a macro */
  
  csumline[0]='\0'; /* clear the line accumulator */
  while (ok) {

    fromsource = nextfromsource;  /* redirects take effect here */ 
 
    switch (fromsource){
      case INFROMFILE:
 
        if(fgets(vinline,MAXINLINE,fin[finc]) == NULL){
          (void) fprintf(stderr,"miniproc, fatal error, encountered end of input file %s \n"
             ,&finname[finc][0]);
          exit(EXIT_FAILURE);
        }

        /* put a series of \0 on top of the new line character, if any */

        prefix=strcspn(vinline,"\n\r");
        if(prefix < strlen(vinline)){
           cinline[prefix]='\0';
           endstring=prefix-1;
        }
        else
          endstring=strlen(vinline)-1; /* endstring is an array index, 
                                          arrays start at 0 */

        break;

      case INFROMMACRO:
        if(macro_gets(vinline) != 0){
          (void) fprintf(stderr,"miniproc, could not read macro %s \n",macroname);
          exit(EXIT_FAILURE);
        }
        endstring=strlen(vinline)-1; /* crud on line ends was removed at first reading*/
        break;

      case INFROMCOMMAND:
        (void) strcpy(vinline,"#__");  /* make it look like a command from a file or macro*/
        (void) strcpy(&cinline[3],argv[argptr]);
        argptr++;
        if(argptr>=argc)nextfromsource=INFROMFILE;
        endstring=strlen(vinline)-1;
        break;  

    } /* end of switch of fromsource */
  
    /* test to see if this line has a command line prefix */

    commandmatch=strlen(altprefix->string);
    if(commandmatch != 0){ /* test for NOT alternate command line prefix */
      if( strncmp(vinline,altprefix->string,commandmatch) != 0) commandmatch=0;
    }
    if(commandmatch == 0){
      if(strncmp(vinline,"#__",3)==0)commandmatch=3;  /* test for command with a standard prefix */
    }

    /* either recording a macro, in which case the lines are stored verbatim, or 
       we doing normal processing */

    switch (howtohandle){ 
      case HANDLE_RECORD: /* scanning for deck/f$macro_end command line */
        if((*trace & TRACE_INPUT)==TRACE_INPUT)(void) fprintf(stderr,"input line   >%s\n",vinline);

        /* check for deck/f$macro_end, which would end macro recording */

        if(commandmatch){  /* is a command */
          temp=strchr(&cinline[commandmatch],cdeck[0]);     /* first place first character appears */
          if(temp!=NULL){                /* if it appears */ 
            if(strcmp(temp,deck)==0){ /* is it deck/f$macro_end? */
              (void) strcpy(sumline,&cinline[commandmatch]);
              trim_command_line(sumline);
              /* whatever the deck string was, always pass in f$macro_end */
              if(strcmp(sumline,deck)==0)(void) strcpy(sumline,"f$macro_end"); 
              parse_and_run(sumline);
              break;
            }
          }
        }

        /* must not have been a deck/f$macro_end, so record it */

        do_subs(vinline);  /* controlled by domacrosubnum */
        macro_record(NULL, vinline); /* add this line to the currently 
                                   recording macro */
        break;
      case HANDLE_NORMAL:

        /* see if it is a command line, if not, process it and output it */

        if( commandmatch ==0){  /* not a command line */
          if(ifscan==NO){
            if(linestatus == IN_MORE){
              (void) fprintf(stderr,"miniproc, fatal syntax error, unterminated line continuation\n");
              exit(EXIT_FAILURE);
            }
            if((*trace & TRACE_INPUT)==TRACE_INPUT)(void) fprintf(stderr,"input line   >%s\n",vinline);
            do_subs(vinline);
            print_out(vinline);
          }
        }

        /* ok, it is a command line */

        else {

        /* take care of continuation.  command lines ending with "-" are continued

         onto subsequent command lines */
          if(cinline[endstring]=='-'){
            if(linestatus==IN_NEW){  /* first line of continuation */
              (void) strncpy(sumline,&cinline[commandmatch],endstring-commandmatch);
              csumline[endstring-commandmatch]='\0';  /* need a new 0 on end!!!! */              
              linestatus=IN_MORE;
            }
            else {/*already have some lines put together here */
              if(endstring+strlen(sumline) >= MAXINLINE){
                (void) fprintf(stderr,"miniproc, fatal error, continued line of exceeds %d characters\n",MAXINLINE);
                exit(EXIT_FAILURE);
              }
              (void) strncat(sumline,&cinline[commandmatch],endstring-commandmatch); /* minus #__ and - */
            }  
          }
          else { 
            if(linestatus==IN_NEW) /*this isn't part of a continued line*/
              (void) strcpy(sumline,&cinline[commandmatch]);
            else { /*this part ends a continued line*/
              if(endstring+strlen(sumline) >= MAXINLINE){
                (void) fprintf(stderr,"miniproc, fatal error, continued line of exceeds %d characters\n",MAXINLINE);
                exit(EXIT_FAILURE);
              }
              (void) strcat(sumline,&cinline[commandmatch]);
              linestatus=IN_NEW;
            }
          }
      
          /* If the whole command has been read in, and it it isn't a comment
          we need to do final preprocessing on it, then parse and
          run it */
      
          if(linestatus == IN_NEW && csumline[0]!='!'){
            trim_command_line(sumline);
            do_subs(sumline);  /* do the <<>> replacements*/
            parse_and_run(sumline);
            csumline[0]='\0'; /* clear the line accumulator before next loop */
         }  
  
         /* that's it for the main loop, go back for more input.  Note that
            as a result of changes made by the preceding function the next
            line may be read out of a different file, or from a macro */
      
        } /* processing for #__ lines*/
        break;

    } /* switch on howtohandle */

  } /* outer read loop */

} /* end of main() */
