/*
** COPYRIGHT (c) 1995 BY
** DIGITAL EQUIPMENT CORPORATION, MAYNARD, MASSACHUSETTS.
** ALL RIGHTS RESERVED.
**
** THIS SOFTWARE IS FURNISHED UNDER A LICENSE AND MAY BE USED AND COPIED
** ONLY  IN  ACCORDANCE  OF  THE  TERMS  OF  SUCH  LICENSE  AND WITH THE
** INCLUSION OF THE ABOVE COPYRIGHT NOTICE. THIS SOFTWARE OR  ANY  OTHER
** COPIES THEREOF MAY NOT BE PROVIDED OR OTHERWISE MADE AVAILABLE TO ANY
** OTHER PERSON.  NO TITLE TO AND  OWNERSHIP OF THE  SOFTWARE IS  HEREBY
** TRANSFERRED.
**
** THE INFORMATION IN THIS SOFTWARE IS  SUBJECT TO CHANGE WITHOUT NOTICE
** AND  SHOULD  NOT  BE  CONSTRUED  AS A COMMITMENT BY DIGITAL EQUIPMENT
** CORPORATION.
**
** DIGITAL ASSUMES NO RESPONSIBILITY FOR THE USE  OR  RELIABILITY OF ITS
** SOFTWARE ON EQUIPMENT WHICH IS NOT SUPPLIED BY DIGITAL.
**
** COMPONENT:
** 
**   This module is part of the OpenVMS port of the Apache web server.
** 
** MODULE DESCRIPTION:
** 
**   Apache_privileged.c is a user-written system service that provides
**   two functions to the web server that normally require privileges:
**
**        - Bind a TCP/IP socket to a privileged port.
**	  - Lookup UIC and login directory information for a supplied
**          username.
**
**   By putting these operations in a protected shareable image, the
**   main image itself does not need to be installed with privileges.
**   The main image therefore can be linked with traceback and 
**   dynamically loaded modules do not have to be made known images either.
**   
**   When first loaded, the module attempts to translate the exec mode
**   logical name APACHE_PLV_ENABLE_xxxxxxxxxxxx, where 'xxxxxxxxxxxx'
**   is the current process's username.  If defined, this logical must
**   translate to a search list containing 3 equivalence names consisting
**   of decimal numeric charaters.  These equivalence names are interpreted
**   as integers with the following meanings:
**   
**        flags		Bit mask indicating the functions enabled:
**			    <0>   vms_bind_privileged().
**			    <1>   vms_get_login_info().
**
**	  min_port	Minimum port number user can bind to.
**	  max_port	Maximum port number user can bind to.
**
**    The intent of the APACHE_PLV_ENABLE logical is to limit the capabilities
**    that this function grants the caller.
**   
**
**   NOTE: because the change mode is now handled by the image
**   activator, the system service itself must do the argument
**   verification to prevent ACCVIOs and other such problems.
**
**   The associated link procedure, UWSSLNK.COM, will link the object
**   and create the executable. The final step is to copy the
**   executable to SYS$LIBRARY and install it using INSTALL with the
**   /HEADER_RES, /OPEN, /SHARED, and /PROTECTED qualifiers.
**
** AUTHORS:
** 
**   Digital Equipment Corporation (original)
**   
** 
** CREATION DATE: 21-Aug-1999
** 
**  DESIGN ISSUES:
**
**	To be appropriately upwardly-compatible, this module should
**	use the plvdef structure definition from the plvdef.h header
**	file.
**
**	Ideally, all access to processor status (PS) register
**	information should be made using some standard structure
**	definition from the appropriate header file, such as found in
**	pr$r_psdef_bits.  For the sake of convenience, the definition
**	of pr$r_psdef_bits resides in prdef.h, since prdef.h already
**	contains all the bitfield definitions for the "architected"
**	internal processor registers (which, unlike the PS, are
**	accessed by MTPR/MFPR PAL calls).  As part of starlet, prdef
**	now realizes all its definitions using the variant_struct VAX
**	C extension (to declare various members of the union with tag
**	prdef), rendering code using pr$r_psdef_bits as so defined
**	technically non-ANSI-standard.  Furthermore, since the
**	__PAL_READ_PS built-in has type uint64 (not directly
**	compatible with the bit-field definition) it is difficult to
**	avoid causing a store if one decides to use the bitfield
**	designations, which is contrary to one of the goals in this
**	routine (see below).  The routines use defined mask bits to
**	get at the previous mode, but this only works because they're
**	the low-order two bits.
**
**	The OpenVMS VAX MACRO-32 example and the OpenVMS Alpha V1.0
**	BLISS example used "DEFINE_SERVICE" macros in a manner which
**	allowed declaration of kernel- and executive-mode routines in
**	separate statements, mixing of kernel- and executive-mode
**	routines in successive statements and automatic counting of
**	the numbers of both kinds of routines.  The macro capabilities
**	of C are limited, by comparison, so only the counts are
**	calculated (as opposed to hard-coded) here.  Specifically,
**	this example relies on sizeof() constructs to avoid having to
**	hard-code the numbers of routines, but still requires that all
**	executive-mode routines be declared in a single declaration,
**	and that all kernel-mode routines be declared in a single
**	declaration.
**
**	Since this is an example "template" for services which may be
**	executed many times, and since PALcode calls can be somewhat
**	expensive, the services in this routine work to minimize
**	the number of PAL calls made.  Thus, the RD_PS call, needed
**	both (1) to get the previous processor mode as part of
**	checking write access to arguments, and (2) to get the first
**	value to return to the caller, stores its result in the
**	variable ps_temp.  ps_temp should not, and likely will not, be
**	materialized other than in a register (since we don't know
**	where we can safely put it) but there is currently no
**	enforceable compiler guarantee of this, or of similar behavior
**	for a "full-sized" user-written system service.  The link of
**	the image should fail if allocation of storage results in a C
**	RTL call.
**
**
**  COMPILATION, LINKING AND INSTALLATION PROCEDURES:
**   
**  	To compile this routine, use
**
**	$ CC APACHE_PRIVILEGED + SYS$LIBRARY:SYS$LIB_C /LIBRARY
**	
**	To link this routine, use the following link procedure (also
**	present with more comments as UWSS_LNK.COM):
**	
**      $ LINK  /SHARE=UWSS  /PROTECT  /MAP=UWSS -
**	/SYSEXE  /FULL  /CROSS  /NOTRACE -
**	UWSS,  SYS$INPUT:/OPTIONS
**	GSMATCH=LEQUAL,1,1
**	SYMBOL_VECTOR = ( -
**	                     vms_bind_privileged   = PROCEDURE, -
**			     vms_get_login_inf  = PROCEDURE )
**
**	PSECT=USER_SERVICES,VEC,NOWRT,NOEXE
**	COLLECT=PLV,USER_SERVICES
**
**	To use this routine, copy the executable to SYS$LIBRARY, and
**	install it as follows:
**
**	$ INSTALL REPLACE SYS$LIBRARY:APACHE_PRIVILEGED /HEADER_RES /OPEN /SHARED /PROTECTED
**
**	 
**  PORTABILITY ISSUES:
**   
**	This program is only intended to run as an example collection
**	of user-written system services, in a privileged shareable
**	image, on OpenVMS Alpha.  The program assumes that OpenVMS Alpha
**	will continue to support ONLY the hierarchical OpenVMS VAX
**	page protection model.  That model is more restrictive than
**	the architected Alpha Alpha PTE model, which, for example, may
**	allow user write access but no kernel write access to a given
**	page.  If the restriction is ever lifted, a probew PAL call
**	for current mode as well as one for previous mode will be
**	required for safe operation.
**
**   
** MODIFICATION HISTORY:
**
** X-1	DBM001		               		21-Aug-1992
**	Module created.
** 
** X-2  DCP002		                        11-Jan-1993
**	Use extern_model pragma for PLV, make cosmetic and substantial
**	textual changes, do probew from current mode to avoid both
**	accvios and poss sys space corruption, calc # rtns
**	automatically.
**
** X-3  DCP003		                        17-Mar-1993
**	Correct file names in usage instructions.
**
** X-4  DCP004		                        29-Dec-1993
**	Use the system-supplied PLV definition, remove local copy
**	and adjust comments accordingly.  Also, re-type common
**	run-down stub as returning int to match PLV definition.
**
** X-5	NYK500					17-Oct-1995
**	Modify two of the existing services to be 64-bit capable.
**
*/

/* And now, on with the program... */

#include <builtins.h>
#include <chfdef.h>
#include <far_pointers.h>
#include <ints.h>
#include <lib$routines.h>
#include <plvdef.h>
#include <prdef.h>
#include <prvdef.h>
#include <ssdef.h>
#include <ssdescrdef.h>
#include <iodef.h>
#include <ucx$inetdef.h>
#ifndef SOCKADDRIN
#define SOCKADDRIN _SOCKADDRIN
#endif
#include <uaidef.h>
#include <jpidef.h>
#include <lnmdef.h>
#include <descrip.h>
#include <pwd.h>
#ifdef TESTIT
#include <string.h>
#include <stdio.h>
#endif
int SYS$QIOW(), SYS$GETJPIW(), SYS$TRNLNM(), SYS$SETPRV();

/*
** To make things easy, the number and name of the user-written system
** services must be known at compile time.  So first define the number
** of services (kernel and exec) and allocate the routine lists, which
** in C-language terms are arrays of pointers to functions returning
** ints.
*/


/* "Forward routine" declarations */
int     vms_bind_privileged();
int	vms_create_privileged_socket();
int	vms_get_login_info();
int     rundown_handler();

/* Kernel and exec routine lists: */

int (*(exec_table[]))() = {
       vms_create_privileged_socket,
       vms_bind_privileged,
       vms_get_login_info };

/*
** Kernel and exec flags.  Don't allow functions to take 64 bit arguments.
*/
int
    exec_flags [] = {
	0, 0 };
/*
** The next two defines allow the kernel and executive routine counts
** to be filled in automatically after lists have been declared for
** kernel and exec mode.  They must be placed before the PLV
** declaration and initialization, and for this module will be
** functionally equivalent to:
**
** #define KERNEL_ROUTINE_COUNT 0
** #define EXEC_ROUTINE_COUNT 1
** 
*/

#define EXEC_ROUTINE_COUNT sizeof(exec_table)/sizeof(int *)
#define KERNEL_ROUTINE_COUNT 0

/*
** Now build and initialize the PLV structure.  Since the PLV must have
** the VEC psect attribute, and must be the first thing in that psect,
** we use the strict external ref-def model which allows us to put the
** PLV structure in its own psect.  This is like the globaldef
** extension in VAX C, where you can specify in what psect a global
** symbol may be found; unlike globaldef, it allows the declaration
** itself to be ANSI-compliant.  Note that the initialization here
** relies on the change-mode-specific portion (plv$r_cmod_data) of the
** PLV being declared before the portions of the PLV which are specific
** to message vector PLVs (plv$r_msg_data) and system service intercept
** PLVs (plv$r_ssi_data).
**
*/

#ifdef __ALPHA
#pragma extern_model save
#pragma extern_model strict_refdef "USER_SERVICES"
#endif
extern const PLV user_services = {
        PLV$C_TYP_CMOD,         /* type */
        0,                      /* version */
        {
        {KERNEL_ROUTINE_COUNT,	/* # of kernel routines */
        EXEC_ROUTINE_COUNT,     /* # of exec routines */
        0,	                /* kernel routine list */
        exec_table,             /* exec routine list */
        0,		        /* kernel rundown handler */
        rundown_handler,        /* exec rundown handler */
        0,                      /* no RMS dispatcher */
        0,			/* kernel routine flags */
        exec_flags}		/* exec routine flags */
        }
        };
#ifdef __ALPHA
#pragma extern_model restore
#endif
/**************************************************************************/
/* Define global data structures.  The page protection will only allow
 * modifying from exec or kernel mode.
 */
static long uaf_context = 0;		/* context for SYS$GETUAI calls */
static int control_vec_initialized = 0;	/* status. */
static int control_vec[3];		/* 0-flags, 1-min port, 2-max port */

/**************************************************************************/
/* Support routines.
 * safe_copy appends count bytes plua NULL null-terminated string to the 
 * current buffer starting at offset *pos and then updates *pos by count.
 * The return value is the address of the first byte written.
 */
static char * safe_copy ( const char *src, int count, char *buffer, 
	int *pos, int bufsize )
{
    int i, j;
    i = *pos;
    if ( count+i < bufsize ) {
        for ( j = 0; j < count; j++ ) buffer[i+j] = src[j];
        buffer[i+count] = '\0';		/* terminate string */
        *pos = (i+count);
    }
    return &buffer[i];
}
/*
 * Enable sysprv, returning previous privilege mask.
 */
static long required_privs[2] = { 
	PRV$M_SYSPRV|PRV$M_OPER|PRV$M_SYSNAM|PRV$M_NETMBX,
	0 };
static int disable_privileges ( long *previous )
{
    /*
     * Only disable privileges that user didn't have before.
     */
    long extra_privs[2];
    extra_privs[0] = (required_privs[0] & (~previous[0]));
    extra_privs[1] = (required_privs[1] & (~previous[1]));
    /*
     * Skip $setprv call if nothing to disable
     */
    if ( (extra_privs[0] == 0) && (extra_privs[1] == 0) ) return 1;

    return SYS$SETPRV ( 0, extra_privs, 0, 0 );
}

static int enable_privileges ( long *previous )
{
    int status;
    status = SYS$SETPRV ( 1, required_privs, 0, previous );
    if ( status == SS$_NOTALLPRIV ) {
        disable_privileges ( previous );
	status = SS$_NOSETPRV;
    }
    return status;
}

/*
 * Translate the logical APACHE_PLV_ENABLE_xxxxxxxxxxxx and initialize
 * the control vector.
 */
static int verify_caller()
{
    int status, length, i, j, attr;
    struct { short length, code; char *buffer; int *ret_len; } item[8];
    long iosb[2];
    char username[12], *src;
    char equiv[768];		/* buffer for result */
    int equiv_len[3], name_index[3];
    static char logical_name[32];
    static $DESCRIPTOR(logical_name_dx,logical_name);
    static $DESCRIPTOR(name_table, "LNM$FILE_DEV" );
    /*
     * Default to basic access.
     */
    control_vec_initialized = 1;
    control_vec[0] = 0;		/* no access */
    control_vec[1] = 1024;		
    control_vec[2] = 65535;	/* default to non-priv range */
    /*
     * Get current processes username.  Note that we can't call LIB$ routines
     * from inside the protected image.
     */
    item[0].length = sizeof(username);
    item[0].code = JPI$_USERNAME;
    item[0].buffer = username;
    item[0].ret_len = &length;
    item[1].length = item[1].code = 0;		/* terminate list */
    length = 0;

    status = SYS$GETJPIW ( 0, 0, 0, item, iosb, 0, 0 );
    if ( (status&1) == 0 ) {
	return status;
    }
    /*
     * Incorporate the username into the logical name to test.
     */
    src = "APACHE_PLV_ENABLE_";
    for ( i = 0; src[i]; i++ ) logical_name[i] = src[i];
    if ( i + length < sizeof(logical_name) ) {
	for ( j = 0; j < length; j++ ) {
	    if ( username[j] == ' ' ) break;
	    logical_name[i++] = username[j];
	}
    }
    logical_name_dx.dsc$w_length = i;
    logical_name_dx.dsc$a_pointer = logical_name;
    /*
     * Translate the logical, we expect 3 equivalence names.
     */
    for ( i = 0; i < 3; i++ ) {
	item[i*2].length = 255;
	item[i*2].code = LNM$_STRING;
	item[i*2].buffer = &equiv[i*256];
	item[i*2].ret_len = &equiv_len[i];
        equiv_len[i] = 0;
	name_index[i] = i;
	if ( i < 2 ) {
	    item[(i*2)+1].length = sizeof(int);
	    item[(i*2)+1].code = LNM$_INDEX;
	    item[(i*2)+1].buffer = (char *) &name_index[i+1];
	    item[(i*2)+1].ret_len = (int *) 0;
	}
    }
    item[5].code = item[5].length = 0;		/* terminate list */
    attr = LNM$M_CASE_BLIND;
    status = SYS$TRNLNM ( &attr, &name_table, &logical_name_dx, 0, item );
    /*
     * Convert the decimal strings into integers.
     */
    if ( (status&1) == 1 ) {
	for ( i = 0; i < 3; i++ ) {
	    int value;
	    char *s;
	    /* Scan string and decode, we can't use RTL function */
	    s = &equiv[i*256];
	    value = 0;
	    for ( j = 0; j < equiv_len[i]; j++ ) {
		if ( (s[j] >= '0') && (s[j] <= '9') ) {
		    value = (value*10) + (s[j] - '0');
		} else break;
		value = value&0x0ffff;
	    }
	    control_vec[i] = value;
	}
    }
    return status;
}
/**************************************************************************
** And now, the service routines.  The routines in this example
** execute no instructions which would require privilege to complete,
** but they give different results when run privileged than they would
** if run from a non-privileged shareable image.  Each routine will
** expects to be passed a pointer to an array of 2 longwords.  The
** first longword will be set by the service routine to contain the
** low 32-bits of the processor status register, from which a caller
** can verify that the service routine is executing in the proper mode
** (kernel or exec).  The second longword is then set to a unique
** integer (1 for first_service, and so on up to 4) to verify that the
** correct service routine has been called.
**
** Each routine will verify that it can write to the longword array in
** the previous mode, since we're running in one of the inner modes.
** If a bogus address were passed (either an inaccessible address,
** causing an access violation, or an address accessible in an inner
** mode that shouldn't be accessible in user mode), then without this
** check, either the process would get blown away (exec mode) or the
** system would crash (in kernel mode, right away if an accvio or
** later if system space were corrupted).  Clearly, neither failure
** mode is desirable.  Note that the use of the single PROBEW PAL call
** for previous mode is enough because only a single contiguous pair
** of longword data (equivalent to a single quadword datum) will be
** stored.  For storage of larger amounts of data which may cross more
** than one page boundary, more stringent checking is required.
**
** One final note on the service routines.  As a security precaution,
** protected shareable images aren't allowed to call other shareable
** images, unless they too are installed protected.  Watch your calls
** (including implicit ones) to external routines!  They must either
** be linked (from object files or libraries) into your image, or be
** in other protected images.  If you fail to heed this warning, the
** image activator will 'kindly' remind you with the error:
**
** SYSTEM-F-NOSHRIMG, privileged shareable image cannot have outbound calls
*/
/*
 * This service creates a privileged TCP socket.
 */
int vms_create_privileged_socket (int chan )
{
    register unsigned int ps_temp;  /* Assumed to take no real storage*/
    int status, i, port_num;
    char *src, *dst;
    struct { unsigned short status, count; long spare; } iosb;
    long prev_privs[2];
    struct socknet { short protocol; char ptype, domain; } sockchar;
    /*
     * Validate the caller and the caller's arguments.
     */
    if ( !control_vec_initialized ) verify_caller();
    if ( (control_vec[0]&1) == 0 ) return SS$_NOSYSPRV;

    /*
     * Issue UCX call.
     */
    sockchar.protocol = 6;	/* TCP */
    sockchar.ptype = 1;		/* STREAM */
    sockchar.domain = 2;	/* AF_INET */

    status = enable_privileges ( prev_privs );
    if ( (status&1) == 1 ) {
        status = SYS$QIOW ( 0, chan, IO$_SETMODE, &iosb, 0, 0,
	    &sockchar, 0, 0, 0, 0, 0 );
        disable_privileges ( prev_privs );
	if ( (status&1) && (iosb.status == SS$_NOPRIV) ) status = SS$_NOSYSNAM;
    }

    if ( (status&1) == 1 ) status = iosb.status;
    return status;
}
/*
** This service takes a sockaddr_in structure and issues a bind request from
** executive mode.
*/
int vms_bind_privileged (int chan, struct SOCKADDRIN *address, int backlog )
{
    register unsigned int ps_temp;  /* Assumed to take no real storage*/
    int status, i, port_num;
    char *src, *dst;
    struct { unsigned short status, count; long spare; } iosb;
    long prev_privs[2];

    struct sockname { long length; int *va; int *retadr; long list_end;
	struct SOCKADDRIN a; } local;
    /*
     * Validate the caller and the caller's arguments.
     */
    if ( !control_vec_initialized ) verify_caller();
    if ( (control_vec[0]&1) == 0 ) return SS$_NOSYSPRV;

    ps_temp = __PAL_RD_PS();	/* Get the value of the proc status reg*/
    if (		/* Verify write access to the array from  */
	__PAL_PROBER(address, sizeof(struct SOCKADDRIN), 
		ps_temp & PR$M_PS_PRVMOD) == 0 /* previous mode */
       )
	return SS$_ACCVIO;		/* nope, sorry */
    src = (char *) address;
    dst = (char *) &local.a;
    for ( i = 0; i < sizeof(struct SOCKADDRIN); i++ ) dst[i] = src[i];

    port_num = address->SIN$W_PORT;
    port_num = ((port_num>>8)&255) | ((port_num&255)<<8);
    if ( port_num < 1024 ) {
        if ( port_num < control_vec[1] ) return SS$_SUBRNG;
        if ( port_num > control_vec[2] ) return SS$_SUBRNG;
    }
    /*
     * Issue UCX call.
     */
    local.length = sizeof(struct SOCKADDRIN);
    local.va = (int *) &local.a;
    local.retadr = 0;
    local.list_end = 0;

    status = enable_privileges ( prev_privs );
    if ( (status&1) == 1 ) {
        status = SYS$QIOW ( 0, chan, IO$_SETMODE, &iosb, 0, 0,
	    0, 0, &local, backlog, 0, 0 );
        disable_privileges ( prev_privs );
	if ( (status&1) && (iosb.status == SS$_NOPRIV) ) status = SS$_NOSYSNAM;
    }

    if ( (status&1) == 1 ) status = iosb.status;
    return status;
}
/*
** The get_login_info function issues a GETUAI request to retrieve
** the login information about a user.
**
*/

int vms_get_login_info ( const char *username, int ulen, struct passwd *pw, 
	int (*sys_getuai)(), char *buffer, int bufsize )
{
    register unsigned int ps_temp;  /* Assumed to take no real storage*/
    struct dsc$descriptor_s username_dx;
    struct { short length, code; char *buffer; int *ret_len; } item[5];
    int status, flags, devlen, dirlen, uic;
    long prev_privs[2];
    char device[40], defdir[64];
    /*
     * Validate the callers aguments.
     */
    if ( !control_vec_initialized ) verify_caller();
    if ( (control_vec[0]&2) == 0 ) return SS$_NOPRIV;

    ps_temp = __PAL_RD_PS();	/* Get the value of the proc status reg*/
    if ( __PAL_PROBER(username, ulen, 
		ps_temp & PR$M_PS_PRVMOD) == 0 /* previous mode */
       ) return SS$_ACCVIO;		/* nope, sorry */
    if ( __PAL_PROBEW(pw, sizeof(struct passwd), 
		ps_temp & PR$M_PS_PRVMOD) == 0 /* previous mode */
       ) return SS$_ACCVIO;		/* nope, sorry */
    if ( __PAL_PROBEW(buffer, bufsize, 
		ps_temp & PR$M_PS_PRVMOD) == 0 /* previous mode */
       ) return SS$_ACCVIO;		/* nope, sorry */

    /*
     * Build arguments for GETUAI call, results will first go to local
     * storage (exec-mode access only).
     */
    username_dx.dsc$b_dtype = DSC$K_DTYPE_T;
    username_dx.dsc$b_class = DSC$K_CLASS_S;
    username_dx.dsc$w_length = ulen;
    username_dx.dsc$a_pointer = (char *) username;

    item[0].code = UAI$_FLAGS; item[0].length = sizeof(flags);
    item[0].buffer = (char *) &flags; item[0].ret_len = (int *) 0;

    item[1].code = UAI$_DEFDEV; item[1].length = sizeof(device);
    item[1].buffer = device; item[1].ret_len = &devlen; 
    devlen = 0; device[0] = '\0';

    item[2].code = UAI$_DEFDIR; item[2].length = sizeof(defdir);
    item[2].buffer = defdir; item[2].ret_len = &dirlen; 
    dirlen = 0; defdir[0] = '\0';

    item[3].code = UAI$_UIC; item[3].length = sizeof(int);
    item[3].buffer = (char *) &uic; item[3].ret_len = (int *) 0;

    item[4].code = 0; item[4].length = 0;       /* terminate list */
    /*
     * Retreive the data.
     */
    buffer[0] = '\0';
    status = enable_privileges ( prev_privs );
    if ( (status&1) == 1 ) {
        status = sys_getuai ( 0, &uaf_context, &username_dx, item, 0, 0, 0 );
        disable_privileges ( prev_privs );
    }
    if ( (status&1) == 1 ) {
	/*
	 * Concatenate the returned data into strings inside the caller's
	 * buffer and set the pw structure elements to point to this data.
	 */
	int pos; char *dummy;
	pos = 0;
	pw->pw_name = safe_copy ( username, ulen, buffer, &pos, bufsize );
	safe_copy ( "", 1, buffer, &pos, bufsize );
	pw->pw_uid = uic&0x07fffffff;
	pw->pw_gid = (uic>>16)&0x0ffff;
	pw->pw_dir = safe_copy (&device[1], device[0], buffer, &pos, bufsize);
	dummy = safe_copy (&defdir[1], defdir[0], buffer, &pos, bufsize);
	safe_copy ( "", 1, buffer, &pos, bufsize );
	pw->pw_shell = safe_copy ( "DCL", 3, buffer, &pos, bufsize );

    } else {
	/*
	 * Fill in pw struct with null data.
	 */
	pw->pw_name = buffer;
	pw->pw_uid = pw->pw_gid = -1;
	pw->pw_dir = buffer;
	pw->pw_shell = buffer;
    }
    return status;
}
/*
** Our kernel and exec rundown handlers are one and the same.
** The rundown handler is invoked before any system rundown
** is performed.
**
** This routine is provided simply as a placeholder for a
** real rundown handler.  A user-written rundown handler
** should not invoke any RMS services or RTL routines, and
** must not signal any exceptions.  User-written rundown
** handlers can invoke most system services except those
** that use RMS (i.e.  $PUTMSG).
*/
int rundown_handler()
{
    return SS$_NORMAL;			/* Indicate success */
}

#ifdef TESTIT
#include <stdio.h>
int main ( int argc, char **argv )
{
    int status;
    status = verify_caller ();
    printf ( "verify_caller result: %d ctlvec: %d %d %d\n", status,
	control_vec[0], control_vec[1], control_vec[2] );
    return status;
}
#endif
