/*
 * This module manages groups of related mailboxes via a data structure
 * known as mailbox set.  The set contains up to 3 mailboxes that are
 * specialized in function:
 * 	input	Read data.
 *	output	Write data.
 *	error   Read asynch.
 *
 * Author: David Jones
 * Date:	7-AUG-1999
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stat.h>
#include <fcntl.h>
#include <unixio.h>		/* for isapipe() function */
#include <errno.h>

#include <ssdef.h>		/* VMS system service status return codes */
#include <iodef.h>		/* QIO function codes */
#include <dvidef.h>		/* $GETDVI item codes */
#include <cmbdef.h>		/* Extended mailbox functions */
#include <descrip.h>		/* string descriptors */
#include <msgdef.h>		/* mailbox message codes */
#include <jpidef.h>		/* VMS job/process information codes */
#include <lckdef.h>		/* VMS distributed lock manager codes */

int SYS$CREMBX(), SYS$DASSGN(), SYS$QIOW(), SYS$QIO(), LIB$GETDVI();
int LIB$SPAWN(), LIB$GET_EF(), SYS$DCLAST(), SYS$CANCEL(), SYS$SETAST();
int SYS$HIBER(), SYS$WAKE(), SYS$ENQW(), SYS$DEQ(), LIB$GETJPI();
int SYS$SYNCH();

#include "mailbox_set.h"	/* verify prototypes */
/*
 * Structure definitions.
 */
struct mbx_iosb {
    unsigned short status, count;
    long pid;
}; 
struct process_info {
    struct process_info *next;
    long pid;				/* process id */
    long lockid;
    int state;				/* 0-new, 1-ready, 2-busy, 3-done */
    int waitable;			/* can return a wait state */
    int last_status;			/* completion status */
};
struct mbx_message {
    unsigned short type;		/* message type */
    unsigned short unit;		/* not used */
    int finalsts;			/* exit status */
    char data[248];
};
/*
 * Create a temporary mailbox with flags and return the channel  unit number.
 * Protection is: SY:RWED, OW:RWED, GR: WO:
 */
int vms_create_mailbox ( int max_msg, int quota, int flags,
	vms_mailbox *mbx )
{
    int status, code, fd_flags;
    char mbxname[64];
    /*
     * Default structure to NULL setting and test arguments.
     */
    mbx->chan = 0;
    mbx->maxmsg = max_msg;
    mbx->fd = -1;
    mbx->check = 0;			/* don't check for partner */
    mbx->fname = "";
    mbx->bufpos = 0;
    mbx->buffer = (char *) 0;
    if ( max_msg <= 0 ) return 1;	/* OK */
    if ( max_msg > quota ) return SS$_BADPARAM;
    /*
     * Create the mailbox.
     */
    status = SYS$CREMBX (0, &mbx->chan, max_msg, quota, 0x0ff00, 0, 0, flags);
    if ( (status&1) == 0 ) return status;
    /*
     * Lookup unit number.
     */
    code = DVI$_UNIT;
    mbx->unit = 0;
    status = LIB$GETDVI ( &code, &mbx->chan, 0, &mbx->unit, 0, 0 );
    if ( (status&1) == 0 ) {
	/* Cleanup for error */
	mbx->maxmsg = 0;
	SYS$DASSGN ( mbx->chan );
	return status;
    }
    return status;
}
/***************************************************************************/
/* AST to handle asynchronous reads of the error mailbox.
 */
static void async_err_ast ( vms_mailbox_set *set )
{
    vms_async_buf *tail;
    int status;
    /*
     * Set the length for the current tail of the chain.
     */
    set->last->length = set->err_iosb.status;
    if ( (set->err_iosb.status&1) == 1 ) {
	set->last->length = set->err_iosb.count;
    } else {
	if ( set->err_iosb.status == SS$_NOWRITER ) set->async_err_active = 0;
	set->last->length = -1;
    }
    if ( set->async_err_active == 0 ) return;	/* AST thread cancelled */
    /*
     * Append a new buffer to the chain and issue another read.
     */
    tail = (vms_async_buf *) malloc ( sizeof(vms_async_buf) +
	set->err.maxmsg );
    if ( !tail ) return;
    tail->next = (vms_async_buf *) 0;
    tail->status = SS$_ABORT;
    tail->length = -1;
    set->last->next = tail;
    set->last = tail;

    status = SYS$QIO ( 0, set->err.chan, IO$_READVBLK|IO$M_WRITERCHECK, 
	&set->err_iosb, async_err_ast, set,
	&set->last->data[0], set->err.maxmsg, 0, 0, 0, 0 );

    if ( (status&1) == 0 ) {
	tail->status = status;
	set->err_iosb.status = status;
    }
}
static vms_async_buf *dequeue_async_buf ( vms_mailbox_set *set, int keep_active )
{
    vms_async_buf *head;
    SYS$SETAST(0);
    head = set->first;
    if ( head && set->async_err_active ) {
	if ( !keep_active ) {
	    /*
	     * Cancel any pending read and wait for cancel to complete.
	     */
	    SYS$CANCEL ( set->err.chan );
	    set->async_err_active = 0;
	    SYS$SETAST(1);
	    SYS$SYNCH ( 0, &set->err_iosb );
	    SYS$SETAST(0);
	} else {
	    /*
	     * Don't return the last buffer, instead wait for completion
	     * of next buffer that lets AST routine add to chain.
	     */
	    if ( head == set->last && !set->err.check ) {
		int i;
		for ( i = 0; (head == set->last) && i < 500; i++ ) {
	    	    SYS$SETAST(1);
		    SYS$SYNCH ( 0, &set->err_iosb );
		    SYS$SETAST(0);
		}
		if ( head == set->last ) head = (vms_async_buf *) 0;
	    }
	}
    }
    if ( head ) set->first = head->next;
    SYS$SETAST(1);
    return head;
}
/***************************************************************************/
/* Create 3 directional mailboxes and use to initialize a mailbox set
 * structure.
 */
int vms_create_mailbox_set ( int in_maxmsg, int in_quota,
	int out_maxmsg, int out_quota, int err_maxmsg, vms_mailbox_set *set )
{
    int status;

    set->task = 0;
    set->in.maxmsg = 0;
    set->out.maxmsg = 0;
    set->out.maxmsg = 0;
    set->async_err_active = 0;
    set->first = set->last = (vms_async_buf *) 0;
    /*
     * Create the 3 mailboxes.
     */
    status = vms_create_mailbox(in_maxmsg, in_quota, CMB$M_READONLY, &set->in);
    if ( (status&1) == 1 ) {
	status = vms_create_mailbox ( out_maxmsg, out_quota, CMB$M_WRITEONLY,
		&set->out );
	if ( (status&1) == 1 ) {
	    status = vms_create_mailbox ( err_maxmsg, err_maxmsg, CMB$M_READONLY,
		&set->err );
	    if ( ((status&1) == 1) && (err_maxmsg > 0) ) {
		/*
		 * Initiate asyncrhonous reads on error mailbox.
		 */
		set->async_err_active = 1;
		set->first = (vms_async_buf *) malloc (
			sizeof(vms_async_buf) + err_maxmsg );
		set->last = set->first;
		set->first->next = (vms_async_buf *) 0;
		set->first->length = -1;

		status = SYS$QIO( 0, set->err.chan, IO$_READVBLK,
			&set->err_iosb, async_err_ast, set,
			&set->first->data[0], err_maxmsg, 0, 0, 0, 0 );
	    }
	}
    }
    if ( (status&1) == 0 ) {
	/*
	 * Error, rollback any mailboxes that were created.
	 */
	vms_destroy_mailbox_set ( set );
    }
#ifdef DEBUG
else printf("process %x created mbxset %d %d %d\n", getpid(),
set->in.unit, set->out.unit, set->err.unit );
#endif
    return status;
}
/***********************************************************************/
/*
 * Return file descriptor.
 */
int vms_open_mailbox ( vms_mailbox *mbx, int flags )
{
    int status;
    char *fname, mbxname[32];

    if ( mbx->maxmsg > 0 ) {
	fname = mbxname;
	sprintf ( mbxname, "_MBA%d:", mbx->unit );
    } else fname = mbx->fname;

    mbx->fd = open ( fname, flags, 0700 );
    return (mbx->fd < 0) ? 0 : 1;
}
/**************************************************************************/
/* Emulate the apache ap_bgets function.
 *
 * Reads from the stream into the array pointed to by buff, until
 * a (CR)LF sequence is read, or end-of-file condition is encountered
 * or until n-1 bytes have been stored in buff. If a CRLF sequence is
 * read, it is replaced by a newline character.  The string is then
 * terminated with a null character.
 *
 * Returns the number of bytes stored in buff (not including the terminating
 * NULL), or zero on end of transmission, or -1 on an error.
 *
 * Notes:
 *  If null characters are expected in the data stream, then
 * buff should not be treated as a null terminated C string; instead
 * the returned count should be used to determine the length of the
 * string.
 *  CR characters in the byte stream not immediately followed by a LF
 * will be preserved.
 */
int vms_mailbox_gets ( char *buf, int bufsize, vms_mailbox *in )
{
    int status, count, i;
    struct { unsigned short status, count; long pid; } iosb;
    char c;

    if ( in->maxmsg <= 0 ) return -1;	/* not a valid mailbox */
    if ( in->buffer == (char *) 0 ) {
	/*
	 * allocate a buffer, include room for LF+NULL to be tacked to end.
	 */
	in->buffer = malloc ( in->maxmsg+2 );
	if ( !in->buffer ) return -1;
	in->bufpos = 0;		/* next element to read */
	in->buflen = 0;
    }

    if ( bufsize > in->maxmsg ) bufsize = in->maxmsg+1;
    for ( count = 0; ; ) {
	if ( in->bufpos >= in->buflen ) {
	    /*
	     * read more data.  Check for no writer only after first read.
	     */
	    status = SYS$QIOW ( 0, in->chan, 
	        IO$_READVBLK|(in->check?IO$M_WRITERCHECK:0),
	        &iosb, 0, 0,
	        in->buffer, in->maxmsg, 0, 0, 0, 0 );

	    in->check = 1;
	    if ( (status&1) == 1 ) status = iosb.status;
	    if ( (status&1) == 1 ) {
	        in->buflen = iosb.count;
		in->buffer[in->buflen++] = '\n';
		in->buffer[in->buflen] = '\0';
		in->bufpos = 0;
	    } else if ( status == SS$_NOWRITER ) {
		free ( in->buffer );
		in->buffer = (char *) 0;
	        break;
	    } else if ( status == SS$_ENDOFFILE ) {
	        continue;
	    } else {
	        count = -1;
	        errno = EVMSERR;
	        vaxc$errno = status;
	        return count;
	    }
	}
	/*
	 * Move characters into buffer.
	 */
	c = '\0';
	for ( i = in->bufpos; (c!='\n') && (i < in->buflen); i++ ) {

	    if ( count >= (bufsize-1) ) break;
	    c = in->buffer[i];
	    if ( c == '\r' ) {
		/* since LF was appended we know buffer[i+1] exists. */
		if ( in->buffer[i+1] == '\n' ) buf[count++] = c;
	    } else {
		buf[count++] = c;
	    }
	}
	in->bufpos = i;
	if ( c == '\n' ) break;
    }
    /*
     * Always terminate the string (differs from fgets() function).
     */
    buf[count] = '\0';
    return count;
}
/*
 * Duplicate gets functionality only with err mailbox.
 */
int vms_err_mailbox_gets ( char *buf, int bufsize, vms_mailbox_set *set )
{
    int status, count, i;
    vms_mailbox *in;
    struct { unsigned short status, count; long pid; } iosb;
    char c;

    in = &set->err;
    if ( set->err.maxmsg <= 0 ) return -1;	/* not a valid mailbox */
    if ( in->buffer == (char *) 0 ) {
	/*
	 * allocate a buffer, include room for LF+NULL to be tacked to end.
	 */
	in->buffer = malloc ( in->maxmsg+2 );
	if ( !in->buffer ) return -1;
	in->bufpos = 0;		/* next element to read */
	in->buflen = 0;
    }

    if ( bufsize > in->maxmsg ) bufsize = in->maxmsg+1;
    for ( count = 0; ; ) {
	if ( in->bufpos >= in->buflen ) {
	    /*
	     * read more data, pull from head of async queue.
	     */
	    vms_async_buf *cur; int length;

	    cur = dequeue_async_buf ( set, 1 );
	    if ( cur ) { 
		/*
		 * Copy contents of buffer and free async buffer.
		 */
		status = cur->status; 
		length = cur->length; 
		memcpy ( in->buffer, cur->data, length );
		free ( cur );
	    }
	    else { status = SS$_ABORT; length = 0; }
	    if ( (status&1) == 1 ) {
	        in->buflen = length;
		in->buffer[in->buflen++] = '\n';
		in->buffer[in->buflen] = '\0';
		in->bufpos = 0;
	    } else if ( status == SS$_NOWRITER ) {
		free ( in->buffer );
		in->buffer = (char *) 0;
	        break;
	    } else if ( status == SS$_ENDOFFILE ) {
	        continue;
	    } else {
	        count = -1;
	        errno = EVMSERR;
	        vaxc$errno = status;
	        return count;
	    }
	}
	/*
	 * Move characters into buffer.
	 */
	c = '\0';
	for ( i = in->bufpos; (c!='\n') && (i < in->buflen); i++ ) {

	    if ( count >= (bufsize-1) ) break;
	    c = in->buffer[i];
	    if ( c == '\r' ) {
		/* since LF was appended we know buffer[i+1] exists. */
		if ( in->buffer[i+1] == '\n' ) buf[count++] = c;
	    } else {
		buf[count++] = c;
	    }
	}
	in->bufpos = i;
	if ( c == '\n' ) break;
    }
    /*
     * Always terminate the string (differs from fgets() function).
     */
    buf[count] = '\0';
    return count;
}
/*
 * Return pointer to remaining buffer.
 */
int vms_mailbox_buffer ( vms_mailbox *in, char **buffer, int *buflen )
{
    if ( in->maxmsg <= 0 ) return SS$_BADPARAM;
    if ( !in->buffer ) return SS$_ENDOFFILE;
    *buflen = in->buflen - in->bufpos;
    *buffer = &in->buffer[in->bufpos];
    in->bufpos = in->buflen;		/* only return data once */
    return 1;
}
/*************************************************************************/
/*  Deassign mailbox channels in mailbox set.
 */
int vms_destroy_mailbox_set ( vms_mailbox_set *set )
{
#ifdef DEBUG
    printf("process %x destroying mbxset %d %d %d\n", getpid(),
	set->in.unit, set->out.unit, set->err.unit );
#endif
    if ( set->in.maxmsg > 0 ) {
	SYS$DASSGN ( set->in.chan );
        if ( set->in.buffer ) free ( set->in.buffer );
    }
    if ( set->out.maxmsg > 0 ) SYS$DASSGN ( set->out.chan );
    if ( set->err.maxmsg > 0 ) {
	/*
	 * Drain any accumated async message buffers.
	 */
	vms_async_buf *cur;

	while ( cur = dequeue_async_buf ( set, 0 ) ) free ( cur );
	SYS$DASSGN ( set->err.chan );
	if ( set->err.buffer ) free ( set->err.buffer );
    }

    set->in.maxmsg = set->out.maxmsg = set->err.maxmsg = 0;
    return 1;
}
