/*
 * This file will include OS specific functions which are not inlineable.
 * Any inlineable functions should be defined in os-inline.c instead.
 */

#include <wait.h>
#include <pwd.h>
#include <ssdef.h>
#include "ap_config.h"
#include "os.h"
#ifndef HARD_SERVER_LIMIT
#define HARD_SERVER_LIMIT 256
#endif

#if defined __DECC
# if __CRTL_VER < 70100000 
int fcntl (int fd, int cmd, ...);

int fcntl (int fd, int cmd, ...)
{
return 0;

}	/* fcntl() */

# endif
#endif

/* some linkers complain unless there's at least one function in each
 * .o file... and extra prototype is for gcc -Wmissing-prototypes
 */
extern void ap_is_not_here(void);
void ap_is_not_here(void) {}

/*
 * Insert the DSO emulation code for AIX
 */
#ifdef AIX
#include "os-aix-dso.c"
#endif

/*
 *  Abstraction layer for loading
 *  Apache modules under run-time via 
 *  dynamic shared object (DSO) mechanism
 */

#ifdef RHAPSODY
#include <mach-o/dyld.h>
#include "httpd.h"
#include "http_log.h"

ap_private_extern
void undefined_symbol_handler(const char *symbolName)
{
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_EMERG, NULL,
                 "dyld found undefined symbol: %s\n"
                 "Aborting.\n",
                 symbolName);
    abort();
}

ap_private_extern
NSModule multiple_symbol_handler (NSSymbol s, NSModule old, NSModule new)
{
    /*
     * Since we can't unload symbols, we're going to run into this
     * every time we reload a module. Workaround here is to just
     * rebind to the new symbol, and forget about the old one.
     * This is crummy, because it's basically a memory leak.
     * (See Radar 2262020 against dyld).
     */

#ifdef DEBUG
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, NULL,
                 "dyld found a multiply defined symbol %s in modules:\n"
                 "%s\n%s\n",
                 NSNameOfSymbol(s),
                 NSNameOfModule(old), NSNameOfModule(new));
#endif

    return(new);
}

ap_private_extern
void linkEdit_symbol_handler (NSLinkEditErrors c, int errorNumber,
                              const char *fileName, const char *errorString)
{
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_EMERG, NULL,
                 "dyld errors during link edit for file %s\n%s\n",
                 fileName, errorString);
    abort();
}

#endif /*RHAPSODY*/

#ifndef NO_DLOPEN

void ap_os_dso_init(void)
{
#if defined(RHAPSODY)
    NSLinkEditErrorHandlers handlers;

    handlers.undefined = undefined_symbol_handler;
    handlers.multiple  = multiple_symbol_handler;
    handlers.linkEdit  = linkEdit_symbol_handler;

    NSInstallLinkEditErrorHandlers(&handlers);
#endif
}

void *ap_os_dso_load(const char *path)
{
#if defined(HPUX) || defined(HPUX10)
    shl_t handle;
    handle = shl_load(path, BIND_IMMEDIATE|BIND_VERBOSE|BIND_NOSTART, 0L);
    return (void *)handle;

#elif defined(RHAPSODY)
    NSObjectFileImage image;
    if (NSCreateObjectFileImageFromFile(path, &image) !=
        NSObjectFileImageSuccess)
        return NULL;
    return NSLinkModule(image, path, TRUE);

#elif defined(OSF1) ||\
    (defined(__FreeBSD_version) && (__FreeBSD_version >= 220000))
    return dlopen((char *)path, RTLD_NOW | RTLD_GLOBAL);

#else
    return dlopen(path, RTLD_NOW | RTLD_GLOBAL);
#endif
}

void ap_os_dso_unload(void *handle)
{
#if defined(HPUX) || defined(HPUX10)
    shl_unload((shl_t)handle);

#elif defined(RHAPSODY)
    NSUnLinkModule(handle,FALSE);

#else
    dlclose(handle);
#endif

    return;
}

void *ap_os_dso_sym(void *handle, const char *symname)
{
#if defined(HPUX) || defined(HPUX10)
    void *symaddr = NULL;
    int status;

    errno = 0;
    status = shl_findsym((shl_t *)&handle, symname, TYPE_PROCEDURE, &symaddr);
    if (status == -1 && errno == 0) /* try TYPE_DATA instead */
        status = shl_findsym((shl_t *)&handle, symname, TYPE_DATA, &symaddr);
    return (status == -1 ? NULL : symaddr);

#elif defined(RHAPSODY)
    NSSymbol symbol;
    char *symname2 = (char*)malloc(sizeof(char)*(strlen(symname)+2));
    sprintf(symname2, "_%s", symname);
    symbol = NSLookupAndBindSymbol(symname2);
    free(symname2);
    return NSAddressOfSymbol(symbol);

#elif defined(DLSYM_NEEDS_UNDERSCORE)
    char *symbol = (char*)malloc(sizeof(char)*(strlen(symname)+2));
    void *retval;
    sprintf(symbol, "_%s", symname);
    retval = dlsym(handle, symbol);
    free(symbol);
    return retval;

#else
    return dlsym(handle, symname);
#endif
}

const char *ap_os_dso_error(void)
{
#if defined(HPUX) || defined(HPUX10)
    return strerror(errno);
#elif defined(RHAPSODY)
    return NULL;
#else
    return dlerror();
#endif
}

#endif	/* NO_DLOPEN */
/****************************************************************************/
/* Extensions for OpenVMS added by David Jones:
 *
 *    Routines to support sharing listen sockets between processes:
 *	int vms_create_shared_socket ( int af, int type, int protocol );
 *	int vms_export_shared_sockets ( char *logical_name );
 *	(struct vms_shared_socket { struct vms_share_socket *next,
 *		char tag[xx]; int fd, namelen; void *name };)
 *	int vms_import_shared_sockets ( char *logical_name, 
 *		struct vms_shared_socket **list );
 *	int vms_assign_shared_socket ( char *tag );
 *
 *    Routines to share the scoreboard structure between processes:
 *	void *vms_create_scoreboard ( size_t size, int is_master_process );
 *
 *    Hack to force socket close from inside signal handler:
 *	int vms_close_busy_socket ( int sd );
 *
 *    Child process creation routines/wait:
 *	int vms_create_child(char *command, long *pid);
 *	int vms_wait_for_child_exit(int pid, int *exit_status, int flags);
 */
/*#include <ucx$inetdef.h>*/		/* TCP device driver definitions */
#define UCX$C_SHARE 4105
#define UCX$C_SOCKOPT 1

#include <iodef.h>			/* QIO service definitions */
#include <descrip.h>			/* VMS string descriptors */
#include <dvidef.h>			/* GETDVI service defnitions */
#include <lnmdef.h>			/* logical name services */
#include <jpidef.h>			/* process information */
#include <secdef.h>			/* global sections */

int SYS$ASSIGN(), SYS$DASSGN(), SYS$QIOW(), LIB$GETDVI(), SYS$CRELNM();
int SYS$TRNLNM(), SYS$CRMPSC(), SYS$MGBLSC(), SYS$SETAST(), SYS$HIBER();
int SYS$WAKE(), SYS$CANCEL(), SYS$GETUAI();
int LIB$FIND_IMAGE_SYMBOL();
/*
 * Define static descriptors
 */
static $DESCRIPTOR(ucx_device, "UCX$DEVICE");
static $DESCRIPTOR(name_table_dx,"LNM$PROCESS");
/*
 * Use statically declared pointer to keep track of all shared sockets
 * created since last export() done.
 */
static struct vms_shared_socket *shared_list = (struct vms_shared_socket *) 0;

/****************************************************************************/
static int (*vms_bind_privileged)
	(int chan, const void *address, int backlog );
static int (*vms_get_login_info) ( const char *username, int ulen, 
	struct passwd *pw, int (*sys_getuai)(), char *buffer, int bufsize );
static int (*vms_create_privileged_socket) ( int channel );
static int plv_status = 0;
static load_privileged_library ( )
{
    int status;
    static $DESCRIPTOR(apache_plv_image,"APACHE_PRIVILEGED");
    static $DESCRIPTOR(apache_plv_dir,"APACHE_ROOT:[000000]");
    static $DESCRIPTOR(info_rtn,"VMS_GET_LOGIN_INFO" );
    static $DESCRIPTOR(create_rtn,"VMS_CREATE_PRIVILEGED_SOCKET" );

    status = LIB$FIND_IMAGE_SYMBOL ( &apache_plv_image,
	    &info_rtn, &vms_get_login_info, &apache_plv_dir );

    if ( (status&1) == 1 ) {
	status = LIB$FIND_IMAGE_SYMBOL ( &apache_plv_image,
	    &create_rtn, &vms_create_privileged_socket, &apache_plv_dir );
    }
    plv_status = status;
}
/***************************************************************************/
/* The vms_create_shared_socket function returns a socket descriptor similar
 * to the RTL socket() function except this socket is given the special UCX
 * SHARE attribute that allows the device to be accessed by multiple processes.
 * Sharing a single socket between processes is better than using multiple
 * socket with REUSADDR attribute.
 *
 * The VMS device name of the created socket device is added to a saved list
 * to be exported by vms_export_shared_sockets().
 */
int vms_create_shared_socket ( int af, int type, int protocol, int port )
{
    int status, flag, sd, code;
    int channel, length;
    struct vms_shared_socket *ss;
    char tag[SHARED_SOCKET_MAX_TAG];
    static $DESCRIPTOR(dev_name_dx, "");
    struct sockchar { short protocol; char ptype, domain; } sockchar;
    struct { unsigned short status, count; long devdepend; } iosb;
    struct  itlst { short length, code;
	union { struct itlst *lst; int *val; } addr; } olst[4];
    /*
     * Create the network pseudo-device that will become our socket.
     */
    channel = 0;
    status = SYS$ASSIGN ( &ucx_device, &channel, 0, 0, 0 );
    if ( (status&1) == 0 ) {
	return -1;
    }
    /*
     * Initialize on first call.
     */
    if ( (port < 1024) && (plv_status&1) == 0 ) {
	if ( plv_status == 0 ) load_privileged_library();
    }
    /*
     * Initialize the device as a socket using parameters suppiled by caller
     * Plus the additional option to set the device SHARED which allows other
     * processes to access it.
     */
    sockchar.domain = af;
    sockchar.ptype = type;
    sockchar.protocol = protocol;

    flag = 1;			/* set true */
    olst[0].length = sizeof(struct itlst);		/* primary descriptor */
    olst[0].code = UCX$C_SOCKOPT;
    olst[0].addr.lst = &olst[1];		/* points to array */
    olst[1].length = sizeof(flag);
    olst[1].code = UCX$C_SHARE;
    olst[1].addr.val = &flag;

    if ( (port<1024) && ((plv_status&1) == 1) ) {
	status = vms_create_privileged_socket ( channel );
    } else {
        status = SYS$QIOW ( 0, channel, IO$_SETMODE, &iosb, 0, 0,
	    &sockchar, 0, 0, 0, olst, 0 );

	if ( (status&1) == 1 ) status = iosb.status;
    }
    if ( (status&1) == 0 ) {
	SYS$DASSGN ( channel );
	return -1;
    }
    /*
     * Create tag string, for now just use device name.
     */
    dev_name_dx.dsc$a_pointer = tag;
    dev_name_dx.dsc$w_length = SHARED_SOCKET_MAX_TAG - 1;
    length = 0;
    code = DVI$_DEVNAM;
    status = LIB$GETDVI ( &code, &channel, 0, 0, &dev_name_dx, &length );
    if ( (status&1) == 0 ) {
	SYS$DASSGN ( channel );
	return -1;
    }
    if ( length < 0 ) length = 0;
    if ( length >= SHARED_SOCKET_MAX_TAG ) length =  SHARED_SOCKET_MAX_TAG-1;
    tag[length] = '\0';	/* terminate string */
    /*
     * Convert channel number to socket file descriptor that can be used
     * by the C RTL socket routines.
     */
    sd = socket_fd ( channel );
    if ( sd < 0 ) {
	SYS$DASSGN ( channel );
	return sd;
    }
    /*
     * Allocate descriptor and place on shared list, all we really need
     * are the device names.
     */
    ss = (struct vms_shared_socket *) malloc(sizeof(struct vms_shared_socket));
    if ( !ss ) {
	return sd;
    }
    strcpy ( ss->tag, tag );
    ss->next = shared_list;
    shared_list = ss;

    return sd;
}

/***************************************************************************/
/* The vms_export_shared_sockets function creates a logical name that
 * holds the list of shared socket devices created by previous calls to
 * vms_create_shared_socket().  After creating the logical, the internal list 
 * is reset to empty.
 */
int vms_export_shared_sockets ( char *log_name )
{
    static $DESCRIPTOR(log_name_dx,"");
    int status, i, LIB$SET_LOGICAL();
    struct vms_shared_socket *ss;
    struct itmlst { unsigned short length, code; char *bufptr; int *retlen; }
	item[128];
    /*
     * Convert list of names into equivalence names for multi-valued
     * logical name.
     */
    ss = shared_list;
    for ( i = 0; ss && (i < 127); i++ ) {
	item[i].length = strlen ( ss->tag );
	item[i].code = LNM$_STRING;
	item[i].bufptr = ss->tag;
	item[i].retlen = (int *) 0;

	ss = ss->next;
    }
    item[i].length = item[i].code = 0;		/* terminate list */
    if ( i <= 0 ) return 2160;
    /*
     * Create the logical name.
     */
    log_name_dx.dsc$a_pointer = log_name;
    log_name_dx.dsc$w_length = strlen(log_name_dx.dsc$a_pointer);
    status = LIB$SET_LOGICAL ( &log_name_dx, 0, &name_table_dx, 0,
	item );
    return status;
}
/*************************************************************************/
static int vms_assign_shared_socket ( char *tag )
{
    static $DESCRIPTOR(tag_dx, "");
    int status, channel, sd, flag;
    struct { unsigned short status, count; long devdepend; } iosb;
    struct  itlst { short length, code;
	union { struct itlst *lst; int *val; } addr; } olst[4];
    /*
     * Assign another channel to the the existing UCX device.
     */
    tag_dx.dsc$a_pointer = (char *) tag;
    tag_dx.dsc$w_length = strlen ( tag );

    channel = 0;
    status = SYS$ASSIGN ( &tag_dx, &channel, 0, 0, 0 );
    if ( (status&1) == 0 ) {
	fprintf(stderr,
		"Error assigning channel to shared socket device '%s': %d\n",
		tag,status );
	return -1;
    }
    /*
     * Do sanity checks on assigned device.
     */
    /*
     * Release ownership of device, set sockopt SHARE.
     */
    flag = 1;
    olst[0].length = sizeof(struct itlst);		/* primary descriptor */
    olst[0].code = UCX$C_SOCKOPT;
    olst[0].addr.lst = &olst[1];		/* points to array */
    olst[1].length = sizeof(flag);
    olst[1].code = UCX$C_SHARE;
    olst[1].addr.val = &flag;

    status = SYS$QIOW ( 0, channel, IO$_SETMODE, &iosb, 0, 0,
	0, 0, 0, 0, olst, 0 );

    if ( (status&1) == 1 ) status = iosb.status;
    if ( (status&1) == 0 ) {
	fprintf(stderr,"Error sharing socket device, status: %d\n", status );
    }
    /*
     * Allocate an RTL socket descriptor and initialize with our channel.
     */
    sd = socket_fd ( channel );
    if ( sd < 0 ) {
	/* Failed to get sd, cleanup */
	SYS$DASSGN ( channel );
	return sd;
    }
    return sd;
}
/**************************************************************************/
/* The vms_import_shared_socket function translates the multi-valued logical
 * name specified and for each equivalence name initializes and appends a 
 * vms_shared_socket structure to a list to be returned to the caller.  
 */
int vms_import_shared_socket ( char *log_name, 
	struct vms_shared_socket **list )
{
    static $DESCRIPTOR(log_name_dx,"");
    struct itmlst { unsigned short length, code; char *bufptr; short *retlen; }
	item[260];
    struct vms_shared_socket *ss;
    int j, i, status, max_index, zero;
    int index[128];
    short length[128];
    char tag[SHARED_SOCKET_MAX_TAG];
    /*
     * Make initial call to get number of names and first equivalence string.
     */
    log_name_dx.dsc$a_pointer = log_name;
    log_name_dx.dsc$w_length = strlen(log_name_dx.dsc$a_pointer);
    item[0].length = sizeof(max_index);
    item[0].code = LNM$_MAX_INDEX;
    item[0].bufptr = (char *) &max_index;
    item[0].retlen = (short *) 0;
    max_index = 0;
    item[1].length = sizeof(zero);
    item[1].code = LNM$_INDEX;
    item[1].bufptr = (char *) &zero; zero = 0;
    item[1].retlen = (short *) 0;
    item[2].length = sizeof(tag)-1;
    item[2].code = LNM$_STRING;
    item[2].bufptr = tag;
    item[2].retlen = &length[0];
    length[0] = 0;			/* clear high word */
    item[3].length = item[3].code = 0;

    status = SYS$TRNLNM ( 0, &name_table_dx, &log_name_dx, 0, item );
    if ( (status&1)==0 ) fprintf(stderr,
	"Import shared translation (%s) status: %d, max: %d\n", 
	log_name, status, max_index );
    if ( (status&1) == 0 ) return status;
    /*
     * Allocate list.
     */
    if ( max_index == 0 ) {
	/* common case, 1 name only. */
	ss = (struct vms_shared_socket *) 
		malloc ( sizeof(struct vms_shared_socket) );
	if ( !ss ) return 44;
	ss->next = (struct vms_shared_socket *) 0;
	strncpy( ss->tag, tag, length[0] );
	ss->tag[length[0]] = '\0';
	ss->fd = -1;
	ss->namelen = 0;
	ss->name = (void *) 0;
    } else if ( max_index > 0 ) {
	ss = (struct vms_shared_socket *) malloc ( (max_index+1) *
		sizeof(struct vms_shared_socket) );
	for ( i = 0; i <= max_index; i++ ) {
	    ss[i].next = &ss[i+1];
	    ss[i].fd = -1;
	    ss[i].namelen = 0;
	    ss[i].name = (void *) 0;

	    j = (i*2);
	    index[i] = i;
	    length[i] = 0;
	    item[j].length = sizeof(index[i]);
	    item[j].code = LNM$_INDEX;
	    item[j].bufptr = (char *) &index[i];
	    item[j].retlen = (short *) 0;
	    item[j+1].length = sizeof(ss[i].tag)-1;
	    item[j+1].code = LNM$_STRING;
	    item[j+1].bufptr = ss[i].tag;
	    item[j+1].retlen = &length[i];
	}
	ss[max_index].next = (struct vms_shared_socket *) 0;
	j = (max_index+1) * 2;
	item[j].length = item[j].code = 0;	/* terminate list; */

	status = SYS$TRNLNM ( 0, &name_table_dx, &log_name_dx, 0, item );
	if ( status&1 ) {
	    for ( i = 0; i <= max_index; i++ ) {
		ss[i].tag[length[i]] = '\0';
	    }
	} else fprintf(stderr,
		"Import muliplt shared translation (%s) status: %d, max: %d\n", 
		log_name, status, max_index );
    } else {
	*list = (struct vms_shared_socket *) 0;
	return 20;
    }
    *list = ss;
    /*
     * Make sockets.
     */
    for ( ; ss; ss = ss->next ) {
	ss->fd = vms_assign_shared_socket ( ss->tag );
    }
    return status;
}
/*************************************************************************/
/*
 * Force rundown of socket after failed normal close.
 */
int vms_close_busy_socket ( int sd )
{
	int status, vms_chan;
	unsigned short iosb[4];
	/*
	 * Get the VMS channel and kill any pending read or write.
	 */
    vms_chan = vaxc$get_sdc ( sd );
    status = SYS$CANCEL ( vms_chan );
    if ( (status&1) == 1 ) {
	    /*
	     * Force connection close, though this won't 'unbusy' the file
	     * descriptor.
	     */
	status = SYS$QIOW ( 0, vms_chan, IO$_DEACCESS, iosb, 0, 0,
		0, 0, 0, 0, 0, 0 );
	if ( status&1 ) status = iosb[0];
	if ( status&1 ) status = SYS$DASSGN ( vms_chan );
    }
    return status;
}
/*************************************************************************/
/* Create global section.  Section name is APACHE_SCOREBOARD_xxxxxxxx,
 * where xxxxxxxx is PID of the 'master' process.  If is_master flag is
 * true, this process is the master process so the current PID is used.
 * When is_master is false, we assume we are a sub-process that has the
 * master process as its owner.
 */
void *vms_create_scoreboard ( size_t size, int is_master )
{
    static char section_name[32];
    $DESCRIPTOR(section_name_dx, section_name);
    int code, status, LIB$GETJPI();
    long pid;
    void *in_addr[2], *out_addr[2];
    /*
     * construct section name.
     */
    code = (is_master) ? JPI$_PID : JPI$_OWNER;
    status = LIB$GETJPI ( &code, 0, 0, &pid, 0, 0 );
    if ( (status&1) == 0 ) return (void *) -1;
    sprintf ( section_name, "APACHE_SCOREBOARD_%08x", pid );
    section_name_dx.dsc$w_length = strlen ( section_name );
    /*
     * Allocate pagefile section at next free address.  Pagecount is
     * in number of pagelets (512 byte units).
     */
    in_addr[0] = section_name;		/* just selects P0 region */
    in_addr[1] = &section_name[32];
    status = SYS$CRMPSC ( in_addr, out_addr, 0, 
	SEC$M_WRT | SEC$M_GBL | SEC$M_PAGFIL | SEC$M_EXPREG,
	&section_name_dx, 0, 0, 0, (size+511)/512, 0, 0x0FF00, 0 );
	if(!(status&1)) fprintf(stderr,"error creating section: %d %x:%x\n", 
		status, out_addr[0], out_addr[1] );
    if ( (status&1) == 0 ) return (void *) -1;
    return out_addr[0];
}
/*************************************************************************/
/*
 * Send signal to process.
 */
int ap_os_kill ( int pid, int sig )
{
    return kill ( pid, sig );
}
/*************************************************************************/
/*
 * Process management routines. build list of allocated PIDs.  These
 * variables are updated by ASTs.
 */
static int active_list_init = 0, waiting_for_completion;
static struct server_ctl {
   struct server_ctl *next;
   long pid;			/* process ID */
   int completed;
   int completion_status;
} active_list[HARD_SERVER_LIMIT], *active_ctl, *free_ctl;

static void exit_notify_ast ( struct server_ctl *ctl )
{
    /*
     * Mark completed for retrieval.
     */
    ctl->completed = 1;
    if ( waiting_for_completion ) {
	waiting_for_completion = 0;
	SYS$WAKE(0,0);
    }
}

int vms_create_child ( char *command, long *pid )
{
    static $DESCRIPTOR(command_dx,"");
    int i, status, flags, LIB$SPAWN(), *completion_status;
    struct server_ctl *ctl;

    if ( active_list_init == 0 ) {	/* initialize */
	active_list_init = 1;
	waiting_for_completion = 0;
	for ( i = 0; i < HARD_SERVER_LIMIT-1; i++ ) {
	    active_list[i].next = &active_list[i+1];
	}
	active_list[HARD_SERVER_LIMIT-1].next = (struct server_ctl *) 0;
	free_ctl = active_list;
	active_ctl = (struct server_ctl *) 0;
    }
    /*
     * Alocate a control block.
     */
    SYS$SETAST(0);
    ctl = free_ctl;
    if ( ctl ) {
	free_ctl = ctl->next;
	ctl->next = active_ctl;
	active_ctl = ctl;
	ctl->completed = 0;
	ctl->pid = 0;
    }
    SYS$SETAST(1);
    if ( !ctl ) return 44;	/* no control blocks */
    /*
     * create the process.
     */
    command_dx.dsc$a_pointer = command;
    command_dx.dsc$w_length = strlen ( command_dx.dsc$a_pointer );

    flags = 1;		/* nowait */
    status = LIB$SPAWN ( &command_dx, 0, 0, &flags, 0, &ctl->pid,
	&ctl->completion_status, 0, exit_notify_ast, ctl, 0, 0, 0 );
    *pid = ctl->pid;
    if ( !(status&1) ) {
	/*
	 * Call the completion routine.
	 */
	ctl->completion_status = status;
	ctl->completed = 1;
	fprintf(stderr,"Spawn '%s' status: %d, pid: %x\n", command, status, *pid );
    }
    return status;
}
/*
 * Returns PID of exited child or 0 if WNOHANG flag present or -1 if error.
 */
int vms_wait_for_child_exit ( int pid, int *estatus, int flags )
{
    struct server_ctl *ctl, *prev, copy;
    /*
     * Revert to real waitpid if vms_create_child never called.
     */
#ifdef waitpid
#undef waitpid
    if ( !active_list_init ) {
	return waitpid ( pid, estatus, flags );
    }
#endif
    /*
     * Disable ASTs while scanning structures they update.
     */
    SYS$SETAST(0);
    if ( !active_ctl ) {
	SYS$SETAST(1);
	return -1;		/* no active clients */
    }
    /*
     * Scan active list for completed.
     */
    for ( ; ; ) {
        prev = (struct server_ctl *) 0;
        for ( ctl = active_ctl; ctl; ctl = ctl->next ) {
	  if ( ctl->completed ) {
	   /*
	    * See if completed server is one we aer looking for.
	    */
           if ( (pid == -1) || (pid == ctl->pid) ) {
	       /*
	        * Move control block from active to free list.
	        */
	        if ( prev ) {
		    prev->next = ctl->next;
	        } else {
		    active_ctl = ctl->next;
	        }
		ctl->next = free_ctl;
		free_ctl = ctl;
		break;
	    }
	  }
          prev = ctl;
       }
	if ( !ctl && (flags&WNOHANG) == 0 ) {		/* no match, stall */
	    waiting_for_completion = 1;
	    SYS$SETAST(1);
	    SYS$HIBER();
	    SYS$SETAST(0);
	} else if ( !ctl ) {
	    /* Nohang flag set, return 0 */
	    SYS$SETAST(1);
	    return 0;
	} else {
	    copy = *ctl;
	    break;
	}
    }
    SYS$SETAST(1);
    /*
     * return data to caller.
     */
    *estatus = copy.completion_status;
    return copy.pid;
}
/****************************************************************************/
/*
 * Substitute for C RTL's getpwnam call.  Convert the string returned in
 * pw_dir to unix syntax.
 */
void *vms_getpwnam ( const char *username )
{
    static int pw_valid = 0;
    static struct passwd pw;		/* static data area. */
    static char buffer[1024];
    int status, i, state;
    char c, *colon, *p, *filename;
    /*
     * Return previous if it matches.
     */
    if ( pw_valid ) {
	if ( strcmp ( username, pw.pw_name ) == 0 ) return &pw;
    }
    /*
     * Initialize on first call.
     */
    if ( (plv_status&1) == 0 ) {
	/*
	 * Don't bother loading if username is ours.  Subsequent
	 * calls will match the pw_valid.
	 */
	if ( strcasecmp ( getlogin(), username ) != 0 ) {
	    load_privileged_library();
	}
    }
    /*
     * Retrieve the data.
     */
    if ( (plv_status&1) == 1 ) {
	/*
	 * Use user-written system service.
	 */
        status = vms_get_login_info ( username, strlen(username),
	    &pw, SYS$GETUAI, buffer, sizeof(buffer) );
#ifdef DEBUG
	printf("Status of PLV login info: %d\n", status );
#endif

        if ( (status&1) == 0 ) return (void *) 0;
    } else {
	/*
	 * Fallback to RTL version.
	 */
	struct passwd *alt;

	alt = getpwnam ( username );
	if ( alt ) {
	    int nlen;

	    nlen = strlen(alt->pw_name);
	    pw = *alt;
	    pw_valid = 0;
	    if ( (1+ strlen ( pw.pw_dir ) + nlen) < sizeof(buffer) ) {
		    strcpy ( buffer, pw.pw_name );
		    pw.pw_name = buffer;
		    pw.pw_dir = &buffer[nlen];
		    strcpy ( &buffer[nlen], alt->pw_dir );
	    } else return (void *) 0;	/* can't fixup */
	} else return (void *) 0;
    }
    /*
     * Convert pw_dir member to unix syntax, no trailing slash.
     */
    pw_valid = 0;		/* don't make valid again until done */
    filename = pw.pw_dir;
    if ( !filename[0] ) return (void *) &pw;
    if ( filename[0] == '/' ) return (void *) &pw;
    colon = strchr ( filename, ':' );
    if ( !colon ) return (void *) &pw;		/* unexpected */
    /*
     * replace "disk:" portion with "/disk" (shift everything to right).
     */
    for ( p = colon; p > filename; --p ) *p = p[-1];
    filename[0] = '/';
    /*
     * Replace the directory portion following the disk ([dir1.dir2]) with
     * a unix-style path (/dir1/dir2).
     */
    p = colon+1;
    for ( i = 1; (i<512) && colon[i]; i++ ) {
	if ( colon[i] == '.' ) {
	    /* Directory separator. Skip 'root' dir '.][' */
	    *p++ = '/';
	    if ( (colon[i+1] == ']') || (colon[i+1] == '>') ) {
		if ( colon[i+2] ) i+=2; else i++;
	    }
	} else if ( (colon[i] == '[') || (colon[i] == '<') ) {
	    /* Change open-bracket to '/' */
	    *p++ = '/';
	} else if ( (colon[i] == ']') || (colon[i] == '>') ) {
	    /* Ignore the closing brackets (shortens string) */

	} else {
	    /* Keep everything else as it */
	    *p++ = colon[i];
	}
    }
    /*
     * Terminate string and return it to caller.
     */
    *p = '\0';
    pw_valid = 1;
#ifdef DEBUG
    printf("getpw path out: '%s', uid = %x\n", filename, pw.pw_uid );
#endif
    return (void *) &pw;
}
/* End of file */
