/*
 * Locks
 *    APACHE_PROCESS		top-level lock all others are sublocks ofr
 *				this lock.
 * 
 *    MANAGER_xxxxxxxx		xxxxxxxx is PID of owner process of
 *				manager process.  Value block contains:
 *				    PID		Manager process.
 *				    req-unit	Mailbox unit number.
 *				    rpt-unit	Report mailbox.
 *				    cre-uni	Mailbox unit number.
 *
 *    EXIT_yyyyyyyy		yyyyyyyy is PID of worker process.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stat.h>
#include <fcntl.h>
#include <errno.h>

#include <ssdef.h>		/* VMS system service status return codes */
#include <iodef.h>		/* QIO function codes */
#include <dvidef.h>		/* $GETDVI item codes */
#include <cmbdef.h>		/* Extended mailbox functions */
#include <descrip.h>		/* string descriptors */
#include <msgdef.h>		/* mailbox message codes */
#include <jpidef.h>		/* VMS job/process information codes */
#include <lckdef.h>		/* VMS distributed lock manager codes */

int SYS$CREMBX(), SYS$DASSGN(), SYS$QIOW(), SYS$QIO(), LIB$GETDVI();
int LIB$SPAWN(), LIB$GET_EF(), SYS$CANCEL(), SYS$SETAST(), SYS$SETIMR();
int SYS$ENQW(), SYS$DEQ(), LIB$GETJPI(), LIB$EMUL(), SYS$CANTIM();
int SYS$SYNCH(), SYS$ASSIGN();

#include "mailbox_set.h"	/* verify prototypes */
/*
 * Structure definitions.
 */
struct mbx_iosb {
    unsigned short status, count;
    long pid;
}; 
struct process_info {
    struct process_info *next;
    long pid;				/* process id */
    long lockid;
    int state;				/* 0-new, 1-ready, 2-busy, 3-done */
    int waitable;			/* can return a wait state */
    int last_status;			/* completion status */
};
struct mbx_message {
    unsigned short type;		/* message type */
    unsigned short unit;		/* not used */
    int finalsts;			/* exit status */
    char data[120];
};
/*
 * Global data.  Note that the lists are accessed at AST level.
 */
static char process_startup_cmd[256];
static char manager_lock_name[32];
static $DESCRIPTOR(root_lock_name_dx, "APACHE_PROCESS");
static $DESCRIPTOR(manager_lock_name_dx, manager_lock_name );
static $DESCRIPTOR(process_startup_dx,process_startup_cmd);
static $DESCRIPTOR(manager_startup, "@PROCESS_MANAGER" );
static long root_lock_id, our_pid;
static struct {
    long status;
    long lock_id;
    long pid;				/* PID of manager process */
    long request_unit;			/* mailbox unit numbers */
    long report_unit;
    long spare;
} manager_lksb;
static vms_mailbox request, report;

static vms_mailbox_set ctl;
static int report_ef;
static int waiting_for_completion;
static int initial_used = 0;
static struct process_info *free_pcb;	/* lookaside list for procinfo structs*/
static struct process_info *active_pcb;
#define STATIC_PCB_ALLOCATION 128
static struct process_info initial_pcb[STATIC_PCB_ALLOCATION];

static int assign_mailbox ( int unit_num, int flags, vms_mailbox *mbx )
{
    int status, code, fd_flags;
    static char mbxname[64];
    static $DESCRIPTOR(mbxname_dx,mbxname);
    /*
     * Default structure to NULL setting and test arguments.
     */
    mbx->chan = 0;
    mbx->maxmsg = 512;
    mbx->fd = -1;
    mbx->check = 0;			/* don't check for partner */
    mbx->fname = "";
    mbx->bufpos = 0;
    mbx->buffer = (char *) 0;
    /*
     * Assign channel.
     */
    sprintf ( mbxname, "_MBA%d:", unit_num );
    mbxname_dx.dsc$w_length = strlen ( mbxname );

    status = SYS$ASSIGN ( &mbxname_dx, &mbx->chan, 0, 0, flags );

    return status;
}

static int qio_and_wait ( int chan, int func, struct mbx_iosb *iosb,
	void *buffer, size_t bufsize )
{
    int status;

    status = SYS$QIOW ( 0, chan, func, iosb, 0, 0,
	buffer, bufsize, 0, 0, 0, 0 );
    if ( (status&1) == 1 ) status = iosb->status;
    return status;
}
/***************************************************************************/
/* Lock control routines.
 */
static void destroy_exit_lock ( struct process_info *pcb )
{
    SYS$DEQ ( pcb->lockid, 0, 0, 0 );
    pcb->lockid = 0;
}

static int create_exit_lock ( struct process_info *pcb, void *blk_ast )
{
    char lock_name[32];
    long lksb[2];
    int status;
    static $DESCRIPTOR(child_lock_dx,"");

    sprintf ( lock_name, "EXIT_%x", pcb->pid );
    child_lock_dx.dsc$a_pointer = lock_name;
    child_lock_dx.dsc$w_length = strlen ( child_lock_dx.dsc$a_pointer );

    status = SYS$ENQW ( 0, LCK$K_PWMODE, lksb, 0, &child_lock_dx,
	root_lock_id, 0, pcb, blk_ast, 0, 0 );
    if ( (status&1) == 1 ) status = lksb[0];
    if ( (status&1) == 1 ) pcb->lockid = lksb[1];
    else {
	fprintf(stderr,"Error creating lock %s: %d\n", lock_name, status );
	pcb->lockid = 0;
    }
    return status;
}

static void convert_exit_lock ( struct process_info *pcb, int new_mode )
{
    int status;
    long lksb[2];

    lksb[1] = pcb->lockid;
    status = SYS$ENQW ( 0, new_mode, lksb, LCK$M_CONVERT, 0, root_lock_id,
	0, 0, 0, 0, 0 );
    if ( status&1 ) status = lksb[0];
    if ( (status&1) == 0 ) {
	fprintf(stderr, "Error converting lock EXIT_%x (%x): %d\n", 
		pcb->pid, pcb->lockid, status );
    }
}
/******************************************************************************/
/*
 * Allocation routine for pcb.
 */
static struct process_info *new_pcb()
{
    struct process_info *pcb;
    /*
     * Allocate block.
     */
    pcb = free_pcb;
    if ( pcb ) {
	/*
	 * Block available on lookaside list.
	 */
	free_pcb = pcb->next;
    } else if ( initial_used < STATIC_PCB_ALLOCATION ) {
	/*
	 * Block available from static allocation area.
	 */
	pcb = &initial_pcb[initial_used];
	initial_used++;
    } else {
	/*
	 * Take block from heap.
	 */
	pcb = (struct process_info *) malloc ( sizeof(struct process_info) );
    }
    /*
     * Initialize block.
     */
    if ( pcb ) {
	pcb->pid = 0;
	pcb->next = (struct process_info *) 0;
	pcb->state = 0;
	pcb->waitable = 1;
	pcb->last_status = 1;
    }
    return pcb;
}

void child_exit_ast ( struct process_info *pcb )
{
    struct process_info *blk, *prev;
    /*
     * ensure the pcb in on the active list.
     */
    for ( blk = active_pcb; blk; blk = blk->next ) if ( blk == pcb ) break;
    if ( !blk ) {
	fprintf(stderr,"Bugcheck, exit detected on inactive pcb\n" );
	return;
    }
    /*
     * Release the lock so worker can continue.
     */
    destroy_exit_lock ( pcb );
    /*
     * cancel any pending I/O on the lock.
     */
}
void child_death_ast ( struct process_info *pcb )
{
    struct process_info *blk, *prev;
    int status;
    /*
     * spawned child died, find the pcb on the active list and remove.
     */
    if ( !active_pcb ) {
	fprintf(stderr,"Bugcheck, dead child not on pcb list\n" );
	return;
    } else if ( active_pcb == pcb ) { 		/* easy case, pcb first on list */
	active_pcb = pcb->next;
    } else {
	prev = active_pcb;
	for ( blk = prev->next; blk; blk = blk->next ) if ( blk == pcb ) {
	    prev->next = blk->next;
	    break;
	}
    }
    /*
     * Create an exit lock on behalf of this worker to help ensure
     * any clients get notified of the exit.
     */
    status = create_exit_lock ( pcb, 0 );
    if ( (status&1) == 1 ) destroy_exit_lock ( pcb );

    pcb->next = free_pcb;
    free_pcb = pcb;
}

void manager_death_ast ( int dummy )
{
    /*
     * Manager process we spawned died. recover best we can.
     */
    printf("\nScript manager process exitted!\n\n");
}
/******************************************************************************/
/* Initialize the apache_process root lock and  manager_xxxx locks.
 * Process types:
 *     -1	We are job master process.
 *	0	We are manager process. (job master is parent).
 *	1	We are client process. (job master is parent)
 *	2	We are worker process (job master is parent's parent).
 */
int vms_initialize_process_manager ( int proc_type, char *worker_shell,
	char *manager_shell ) 
{
    int status, code, mode, flags;
    long lksb[2], parent_pid, grandparent_pid, master_pid;
    static int initialized = 0;
    /*
     * Only initialize once.
     */
    if ( strlen(worker_shell) >= sizeof(process_startup_cmd) ) return 20;
    strcpy ( process_startup_cmd, worker_shell );
    process_startup_dx.dsc$w_length = strlen ( process_startup_cmd );

    if ( initialized ) return 3;
    free_pcb = (struct process_info *) 0;
    active_pcb = (struct process_info *) 0;
    initialized = 1;
    /*
     * Get the PID to use in the manager_xxxxxx lock and generate name.
     */
    code = JPI$_OWNER;
    our_pid = 0;
    status = LIB$GETJPI ( &code, &our_pid, 0, &parent_pid, 0, 0 );
    if ( (status&1) == 0 ) return status;
    if ( proc_type < 0 ) {
	master_pid = our_pid;
    } else if ( proc_type < 2 ) {
	master_pid = parent_pid;
    } else {
	status = LIB$GETJPI ( &code, &parent_pid, 0, &grandparent_pid, 0, 0 );
	if ( (status&1) == 0 ) return status;
	master_pid = grandparent_pid;
    }
    sprintf ( manager_lock_name, "MANAGER_%x", master_pid );
    /*
     * Create the root lock that will be parent for all other locks.
     */
    status = SYS$ENQW ( 0, LCK$K_CRMODE, lksb, 
	LCK$M_NODLCKBLK, &root_lock_name_dx,
	0, 0, 0, 0, 0, 0 );
    if ( (status&1) == 1 ) status = lksb[0];
    if ( (status&1) == 0 ) return status;
    root_lock_id = lksb[1];
    /*
     * create the manager lock, mode depends upon whether we are manager
     * process or not.  The manager gets a high level lock to allow it
     * to fill in the lock value block and ensure only 1 process acts
     * as manager.
     */
    mode = (proc_type) ? LCK$K_CRMODE : LCK$K_PWMODE;
    flags = LCK$M_VALBLK;
    if ( proc_type < 0 ) {
	mode = LCK$K_PWMODE;
	flags = LCK$M_NOQUEUE;
    }
    manager_lock_name_dx.dsc$w_length = strlen ( manager_lock_name );
    status = SYS$ENQW ( 0, mode, &manager_lksb, flags, &manager_lock_name_dx,
	root_lock_id, 0, 0, 0, 0, 0, 0 );

    if ( status&1 ) status = manager_lksb.status;
    if ( status == SS$_VALNOTVALID ) {
	printf("Manager has not initialized the value block yet.\n");
        status = 1;
    } else if ( status == SS$_NOTQUEUED ) {
	fprintf ( stderr, "Existing manager already running\n" );
	mode = LCK$K_CRMODE;		/* pretend to be client */
	proc_type = 1;
	status = SYS$ENQW ( 0, mode, &manager_lksb, LCK$M_VALBLK,
		&manager_lock_name_dx,
		root_lock_id, 0, 0, 0, 0, 0, 0 );
	if ( status&1 ) status = manager_lksb.status;
	if ( (status&1) == 0 ) return status;

    } else if ( (status&1) == 0 ) return status;
    /*
     * Rest of initialization depends upon proc_type.
     */
    if ( proc_type < 0 ) {
	/*
	 * We are master process, create mailboxes.
	 */
	status = vms_create_mailbox ( 512, 512, CMB$M_WRITEONLY, &request );
	if ( (status&1) == 0 ) return status;
	status = vms_create_mailbox ( 128, 128, CMB$M_WRITEONLY, &report );
	if ( (status&1) == 0 ) return status;
	/*
	 * Create a new manager process and load PID into LKSB.
	 */
	flags = 1;		/* no wait */
	if ( manager_shell ) {
	    manager_startup.dsc$w_length = strlen ( manager_shell );
	    manager_startup.dsc$a_pointer = manager_shell;
	};
	status = LIB$SPAWN ( &manager_startup, 0, 0, &flags, 0, 
		&manager_lksb.pid, 0, 0, manager_death_ast, 0, 0, 0, 0 );
	if ( (status&1) == 0 ) return status;
        /*
	 * Lower the lock to let the manager proceed.
	 */
	manager_lksb.request_unit = request.unit;
	manager_lksb.report_unit = report.unit;
	mode = LCK$K_CRMODE;
	status = SYS$ENQW ( 0, mode, &manager_lksb, 
		LCK$M_VALBLK|LCK$M_CONVERT, 0,
		root_lock_id, 0, 0, 0, 0, 0, 0 );
    } else {
	/*
	 * Assign channels to mailboxes:
	 *   request	Read by workers, written by others.
	 *   report	read by manager, written by others.
	 */
	if ( manager_lksb.status == SS$_VALNOTVALID ) return SS$_VALNOTVALID;

	status = assign_mailbox ( manager_lksb.request_unit,
	    (proc_type > 1) ? CMB$M_READONLY : CMB$M_WRITEONLY, &request );
	if ( (status&1) == 0 ) return status;

	status = assign_mailbox ( manager_lksb.report_unit,
	    (proc_type == 0) ? CMB$M_READONLY : CMB$M_WRITEONLY, &report );
	if ( (status&1) == 0 ) return status;
    }
    return status;
}
/******************************************************************************/
/* Main routine for the process manager.
 */
int vms_process_manager_main ( char *worker_shell )
{
    int status;
    struct mbx_iosb iosb;
    struct mbx_message msg;
    struct process_info *pcb;
    /*
     * Initialize.
     */
    status = vms_initialize_process_manager ( 0, worker_shell,  (char *) 0 );
    if ( (status &1) == 0 ) return status;
    /*
     * Main loop.
     */
    for ( ; ; ) {
	int flags;
        /*
         * Read report mailbox.
         */
	status = qio_and_wait ( report.chan, IO$_READVBLK, &iosb,
		&msg, sizeof(msg) );
#ifdef DEBUG
	printf("Mailbox read completed: %d, sender: %x, list: %x\n", 
		status, iosb.pid, active_pcb );
#endif
	if ( status == SS$_ENDOFFILE ) continue;	/* ignore EOF */
	if ( (status&1) == 0 ) break;	/* ignore errors (EOF) */
	if ( msg.unit != report.unit ) {
	    printf("invalid message sent to report mailbox, ignored\n");
	    continue;
	}
	/*
	 * See if this request came from a client we are creating.
	 */
	SYS$SETAST(0);
	if ( msg.type == SS$_LINKDISCON ) {
	    for ( pcb = active_pcb; pcb; pcb = pcb->next ) {
	        if ( (pcb->pid == iosb.pid) ) {
		    pcb->state = 1;
		    break;
		}
	    }
	    SYS$SETAST(1);
	    continue;
	}
	/*
	 * Create new worker.
	 */
	pcb = new_pcb();
	if ( !pcb ) {
	    fprintf(stderr, "failed to allocate a pcb\n" );
	    SYS$SETAST(1);
	    return SS$_INSFMEM;
	}
	pcb->next = active_pcb;
	active_pcb = pcb;
	SYS$SETAST(1);
	flags = 1;			/* nowait */
	status = LIB$SPAWN ( &process_startup_dx, 0, 0, &flags, 0,
		&pcb->pid, &pcb->last_status, &report_ef, child_death_ast,
		pcb, 0, 0, 0 );
	if ( (status&1) == 1 ) {
	    /*
	     * Spawn successfull, wait on it to initialize.
	     */
	} else {
	    /*
	     * Cleanup.
	     */
	}
    }
    return status;
}

/******************************************************************************/
/* Main function for use by clients.  Build special request line to send
 * to request mailbox that is read by worker process.  If no readers,
 * send message to manager and retry.
 */
int vms_create_task ( vms_mailbox_set *set, char *command, char **env,
	long *pid )
{
    int i, status, req_len, flags;
    char req_line[512];
    char *input, *output, *error, *envmbx;
    char input_fname[32], output_fname[32], error_fname[32];
    struct mbx_iosb iosb, req_iosb, rpt_iosb;
    struct process_info *pcb;
    struct mbx_message message;
    /*
     * Build request line, delimit with NULs (allows spaces in filenames).
     */
    if ( set->in.maxmsg > 0 ) {
	input = input_fname;
	sprintf ( input_fname, "_MBA%d:", set->in.unit );
    } else input = set->in.fname;

    if ( set->out.maxmsg > 0 ) {
	output = output_fname;
	sprintf ( output_fname, "_MBA%d:", set->out.unit );
    } else {
	env = (char **) 0;	/* Can't transfer env if not a mailbox */
	output = set->out.fname;
    }

    if ( set->err.maxmsg > 0 ) {
	error = error_fname;
	sprintf ( error_fname, "_MBA%d:", set->err.unit );
    } else error = set->err.fname;

    envmbx = "";
    if ( env ) envmbx = output;
    req_len = strlen(input)+strlen(output)+strlen(error)+strlen(command)
	+ strlen(envmbx) + 5;
    if ( req_len > sizeof(req_line) ) return SS$_DATAOVERUN;
    sprintf ( req_line, "%s%c%s%c%s%c%s%c%s%c", input, 0, output, 0,
	error, 0, command, 0, envmbx, 0 );
    /*
     * Attempt to send to an already-spawned process.
     */
    status = qio_and_wait( request.chan, IO$_WRITEVBLK|IO$M_READERCHECK,
	&iosb, req_line, req_len );
    if ( (status == SS$_NOREADER) ) {
	/*
	 * No process available, queue a blocking read to the request
	 * mailbox with a timeout. and kick the manager to make it
	 * create a new one.
	 */
	status = SYS$QIO ( 18, request.chan, IO$_WRITEVBLK, &iosb,
		0, 0,
		req_line, req_len, 0, 0, 0, 0 );
	if ( (status&1) == 0 ) {
	    return status;
	}
	message.type = SS$_NORMAL;
	message.unit = report.unit;
	message.finalsts = 0;
	status = qio_and_wait ( report.chan, IO$_WRITEVBLK|IO$M_READERCHECK, 
		&rpt_iosb,  &message, 8 );
	if ( (status&1) == 0 ) {
	    /* The manager went away unexpectedly */
	    SYS$CANCEL ( request.chan );
	}
	status = SYS$SYNCH ( 18, &iosb );
	if ( (status&1) == 1 ) status = iosb.status;
    }
    if ( (status&1) == 1 ) {
	/*
	 * A process was waiting or created, get lock with blocking AST
	 * to detect if worker starts reading from request MBX again
	 * (or goes away).
	 */
	pcb = new_pcb();
	pcb->pid = iosb.pid;
	status = create_exit_lock ( pcb, (void *) child_exit_ast );
	/*
	 * Send environment array if present.
	 */
	if ( env ) {
	    for ( i = 0; (status&1) && env[i]; i++ ) {
	        status = qio_and_wait ( set->out.chan, 
			IO$_WRITEVBLK|(set->out.check?IO$M_READERCHECK:0),
			&iosb, env[i], strlen(env[i]) );
		set->out.check = 1;
	    }
	    if ( (status&1) ) {
	        status = qio_and_wait ( set->out.chan, 
		    IO$_WRITEOF|(set->out.check?IO$M_READERCHECK:0),
			&iosb, (void *) 0, 0 );
	    }
	}
	destroy_exit_lock ( pcb );
    } else {
	/*
	 * Unexpected error.
	 */
    }
    return status;
}
/*
 * Timeout AST simply issues cancel on channel.  If mailbox read completes
 * before AST cancels the operation, ignore the timeout.
 */
void timeout_ast ( vms_mailbox *mbx )
{
    SYS$CANCEL ( mbx->chan );
}
/******************************************************************************/
/* Main function for use by workers to get next program request.
 * Request read from mailbox is parsed into 5 strings.  If 5th
 * parameter specified a mailbox, it is read and used to populate
 * the env array;
 */
int vms_get_task_request ( int is_first, int timeout,
	char *req_parm[5], char ***env, long *pid )
{
    int status, req_len, i, j, parm_start;
    long delta[2], factor, offset;
    struct mbx_iosb iosb;
    struct process_info pcb;
    char req_line[512];
    /*
     * If this is very first time process getting a request, notify
     * the manager process by writing a message to the report mailbox.
     * It is OK to treat every time as the first time, but adds needless
     * overhead.
     */
    if ( is_first ) {
	struct mbx_message message;
	message.type = SS$_LINKDISCON;
	message.unit = report.unit;
	message.finalsts = 1;
	status = qio_and_wait ( report.chan, IO$_WRITEVBLK|IO$M_READERCHECK,
		&iosb, &message, 8 );
	if ( (status&1) == 0 ) return status;
    }
    /*
     * Get exit lock to indicate we are ready to read the next request.
     * Any clients still thinking we are talking to them will get
     * a blocking AST delivered to them.  On the initial call, the
     * manager blocks waiting on us to request the lock.
     */
    pcb.pid = our_pid;
    status = create_exit_lock ( &pcb, 0 );
    if ( (status&1) == 0 ) return status;
    /*
     * Read request from mailbox.
     */
    if ( timeout > 0 ) {
	/*
	 * Wait no longer than timelimit seconds for a message.
	 */
	factor = -10000000;		/* convert seconds to 100 nsec ticks */
	offset = 0;
	LIB$EMUL ( &timeout, &factor, &offset, delta );
	status = SYS$SETIMR ( 0, delta, timeout_ast, &request, 0 );
	status = SYS$QIO ( 7, request.chan, IO$_READVBLK, &iosb, 0, 0,
		req_line, sizeof(req_line), 0, 0, 0, 0 );
	if ( status & 1) {
	    status = SYS$SYNCH  ( 7, &iosb );
	    SYS$CANTIM ( &request, 0 );
	    status = iosb.status;
	}

    } else {
	/*
	 * no time limit.
	 */
        status = qio_and_wait ( request.chan, IO$_READVBLK, &iosb,
	    req_line, sizeof(req_line) );
    }
    if ( (status&1) == 0 ) return status;
    req_len = iosb.count;
    /*
     * parse the request on the nulls embeded in the request line.
     */
    parm_start = 0;
    for ( j = i = 0; (j < 5); i++ ) if ( (i>=req_len) || !req_line[i] ) {
	req_parm[j] = malloc ( i-parm_start+1 );
	if ( !req_parm[j] ) return SS$_INSFMEM;
	strncpy ( req_parm[j], &req_line[parm_start], i-parm_start );
	req_parm[j][i-parm_start] = '\0';
	parm_start = i + 1;
	j++;
    }
    if ( j != 5 ) {
	/*
	 * Wrong number of parameters on line.
	 */
	return SS$_ABORT;
    }
    /*
     * Release lock, client must get this lock before it sends the env
     * array.
     */
    destroy_exit_lock ( &pcb );
    /*
     * Read the environment array if present.  Client terminates with EOF
     * one the mailbox.
     */
    if ( req_parm[4][0] ) {
	static $DESCRIPTOR(mbxname_dx,"");
	int channel;
	char **new_env;

	new_env = (char **) malloc ( sizeof(char *) *128 );
	*env = new_env;
	new_env[0] = (char *) 0;
	channel = 0;
	mbxname_dx.dsc$w_length = strlen ( req_parm[4] );
	mbxname_dx.dsc$a_pointer = req_parm[4];
	status = SYS$ASSIGN ( &mbxname_dx, &channel, 0, 0, CMB$M_READONLY );
	i = 0;
	while ( (status&1) ) {
	    char buffer[512];
	    status = qio_and_wait ( channel, IO$_READVBLK|IO$M_WRITERCHECK,
		&iosb, buffer, sizeof(buffer) );
	    if ( (status&1) == 0 ) break;
	    if ( i > 127 ) continue;

	    new_env[i] = malloc ( iosb.count + 1 );
	    if ( new_env[i] ) {
		memcpy ( new_env[i], buffer, iosb.count );
		new_env[i][iosb.count] = '\0';
		i++;
		new_env[i] = (char *) 0;
	    }
	}
	if ( status == SS$_ENDOFFILE ) status = 1;
    }
   return status;
}

#ifdef TESTIT
int main ( int argc, char **argv )
{
    int status, mode; char line[800];
	long pid;

    mode = (argc > 1) ? atoi(argv[1]) : -1;
    status = vms_initialize_process_manager ( mode, 
		"mcr sys$disk:[]process_manager 2",	/* start worker */
		"mcr sys$disk:[]process_manager 0" );	/* start manager */
    printf("status %d, mode: %d\n", status, mode );

    if ( mode < 0 ) {			/* master */
	system ( "mcr sys$disk:[]process_manager 1" );
	printf("test client completed, hit return to finish\n");
	gets ( line );
    } else if ( mode == 0 ) {		/* manager */
	status = vms_process_manager_main ( "mcr sys$disk:[]process_manager 2" );
    } else if ( mode == 1 ) {		/* client */
	vms_mailbox_set mbx_set;
	char *env[] = { "a=1", "b=3", (char *) 0 };

	status = vms_create_mailbox_set ( 4096, 5000, 512, 512, 512, &mbx_set);
	printf("client created mailbox set, status: %d\n", status );
	if ( (status&1) == 0 ) return status;

	status = vms_create_task ( &mbx_set, "sho time", env, &pid );
	printf("create_task status: %d, pid: %x\n", status, pid );
    } else if ( mode == 2 ) {		/* worker */
	char *req_parm[5], **env;
	int i;

	status = vms_get_task_request ( 1, 60, req_parm, &env, &pid );
	printf ( "get_task status: %d, client pid: %x\n", status, &pid );
	if ( (status&1) == 0 ) return status;

	for ( i = 0; i < 5; i++ ) printf ( "req_parm[%d] = '%s'\n", i,
		req_parm[i] ) ;
	for ( i = 0; env[i]; i++ ) printf ( "env[%d] = '%s'\n", i, env[i] );
    } else {
	printf("Bad mode: %d\n", mode );
    }
    return status;
}
#endif
