/******************************************************************************
 *
 * Copyright (c) 1998,99 by Mindbright Technology AB, Stockholm, Sweden.
 *                 www.mindbright.se, info@mindbright.se
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 *****************************************************************************
 * $Author: mats $
 * $Date: 2000/01/30 16:56:27 $
 * $Name: rel1-2pre3 $
 *****************************************************************************/
package mindbright.net;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.IOException;
import java.net.Socket;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.net.SocketException;

public class SocksProxySocket extends Socket {

    private String proxyHost;
    private int    proxyPort;
    private String targetHost;
    private int    targetPort;

    String     serverDesc;

    public String getServerDesc() {
	return serverDesc;
    }

    private SocksProxySocket(String host, int port, String proxyHost, int proxyPort)
	throws IOException, UnknownHostException
    {
	super(proxyHost, proxyPort);

	this.proxyHost  = proxyHost;
	this.proxyPort  = proxyPort;
	this.targetHost = targetHost;
	this.targetPort = targetPort;
    }

    public static SocksProxySocket getSocks4Proxy(String host, int port, String proxyHost,
						  int proxyPort, String userId)
	throws IOException, UnknownHostException
    {
	SocksProxySocket proxySocket = new SocksProxySocket(host, port, proxyHost, proxyPort);

	try {
	    InputStream  proxyIn  = proxySocket.getInputStream();
	    OutputStream proxyOut = proxySocket.getOutputStream();
	    InetAddress  hostAddr = InetAddress.getByName(host);

	    proxyOut.write(0x04); // V4
	    proxyOut.write(0x01); // CONNECT
	    proxyOut.write((port >>> 8) & 0xff);
	    proxyOut.write(port & 0xff);
	    proxyOut.write(hostAddr.getAddress());
	    proxyOut.write(userId.getBytes());
	    proxyOut.write(0x00); // NUL terminate userid string
	    proxyOut.flush();

	    if(proxyIn.read() != 0x00)
		throw new IOException("Invalid response from SOCKS4 server");

	    int code = proxyIn.read();
	    if(code != 90)
		throw new IOException("SOCKS4 server unable to connect: " + code);

	    byte[] data = new byte[6];

	    if(proxyIn.read(data, 0, 6) != 6)
		throw new IOException("SOCKS4 error reading destination address/port");

	    proxySocket.serverDesc = data[2]  + "." + data[3] +  "." + data[4] +
		"." + data[5] + ":" + ((data[0] << 8) | data[1]);

	} catch (SocketException e) {
	    throw new SocketException("Error communicating with SOCKS4 server " +
				      proxyHost + ":" + proxyPort);
	}

	return proxySocket;
    }

    public static SocksProxySocket getSocks5Proxy(String host, int port, String proxyHost,
						  int proxyPort, ProxyAuthenticator authenticator)
	throws IOException, UnknownHostException
    {
	SocksProxySocket proxySocket = new SocksProxySocket(host, port, proxyHost, proxyPort);

	try {
	    InputStream  proxyIn  = proxySocket.getInputStream();
	    OutputStream proxyOut = proxySocket.getOutputStream();

	    // Simplest form, only no-auth and cleartext username/password
	    //
	    byte[] request = { (byte) 0x05, (byte) 0x02, (byte) 0x00, (byte) 0x02 };
	    byte[] reply  = new byte[2];

	    proxyOut.write(request);
	    proxyOut.flush();

	    if(proxyIn.read() != 0x05)
		throw new IOException("Invalid response from SOCKS5 server");
	    
	    int method = proxyIn.read();
	    switch(method) {
	    case 0x00:
		break;
	    case 0x02:
		doAuthentication(proxyIn, proxyOut, authenticator);
		break;
	    default:
		throw new IOException("SOCKS5 server does not support our authentication methods");
	    }

	    // Request connect to proxyHost (as 'domain-name') : proxyPort
	    //
	    request = new byte[] { (byte) 0x05, (byte) 0x01, (byte) 0x00, (byte) 0x03 };
	    proxyOut.write(request);
	    proxyOut.write(host.length());
	    proxyOut.write(host.getBytes());
	    proxyOut.write((port >>> 8) & 0xff);
	    proxyOut.write(port & 0xff);
	    proxyOut.flush();

	    if(proxyIn.read() != 0x05)
		throw new IOException("Invalid response from SOCKS5 server");

	    int status = proxyIn.read();
	    if(status != 0x00)
		throw new IOException("SOCKS5 server unable to connect: " + status);

	    proxyIn.read(); // 0x00 RSV

	    int aType = proxyIn.read();
	    byte[] data = new byte[255];
	    switch(aType) {
	    case 0x01:
		if(proxyIn.read(data, 0, 4) != 4)
		    throw new IOException("SOCKS5 error reading address");
		proxySocket.serverDesc = data[0]  + "." + data[1] +  "." + data[2] +
		    "." + data[3];
		break;
	    case 0x03:
		int n = proxyIn.read();
		if(proxyIn.read(data, 0, n) != n)
		    throw new IOException("SOCKS5 error reading address");
		proxySocket.serverDesc = new String(data);
		break;
	    default:
		throw new IOException("SOCKS5 gave unsupported address type: " + aType);
	    }

	    if(proxyIn.read(data, 0, 2) != 2)
		throw new IOException("SOCKS5 error reading port");
	    proxySocket.serverDesc += ":" + ((data[0] << 8) | data[1]);

	} catch (SocketException e) {
	    throw new SocketException("Error communicating with SOCKS5 server " +
				      proxyHost + ":" + proxyPort);
	}

	return proxySocket;
    }

    static void doAuthentication(InputStream proxyIn, OutputStream proxyOut,
				 ProxyAuthenticator authenticator) throws IOException {
	String username = authenticator.getProxyUsername("SOCKS5", null);
	String password = authenticator.getProxyPassword("SOCKS5", null);

	proxyOut.write(0x01);
	proxyOut.write(username.length());
	proxyOut.write(username.getBytes());
	proxyOut.write(password.length());
	proxyOut.write(password.getBytes());

	if(proxyIn.read() != 0x05)
	    throw new IOException("Invalid response from SOCKS5 server");

	if(proxyIn.read() != 0x00)
	    throw new IOException("Invalid username/password for SOCKS5 server");
    }

    public String toString() {
	return "SocksProxySocket[addr=" + getInetAddress() +
	    ",port=" + getPort() +
	    ",localport=" + getLocalPort() + "]";
    }

}
