// **********************************************************************
//
// Copyright (c) 1999
// Object Oriented Concepts, Inc.
// Billerica, MA, USA
//
// All Rights Reserved
//
// **********************************************************************

//
// This implementation supports persistence as well as the
// extended naming service with timestamps and dynamic updates.
//

package com.ooc.CosNaming;

import org.omg.CORBA.*;
import org.omg.CosNaming.*;
import org.omg.CosNaming.NamingContextPackage.*;
import com.ooc.CosNaming.OBNamingContextPackage.*;

//
// Not final because LocalNamingContext (ORBacus Names) inherits from it
//
public class NamingContext extends _OBNamingContextImplBase
    implements Runnable
{
    static private int count_; // Running static count for keys.

    private BOA boa_;

    private ORB orb_;
    private String key_;

    private java.util.Hashtable bindings_; // Set of bindings in this context.
    private java.util.Hashtable ncs_; // Global set of naming contexts.
    private NamingDatabase store_; // The database.
    private boolean noUpdates_; // No automatic updates?
    boolean destroyed_; // Has the context been destroyed?

    //
    // Clients to which this service reports events.
    //
    private java.util.Vector listeners_ = new java.util.Vector();

    //
    // NcOrObjBinding
    //
    private static class NcOrObjBinding
    {
	private NcOrObj ncOrObj_;
	private NameComponent comp_;
	private long timestamp_;

        NcOrObjBinding(NameComponent nc,
                       NcOrObj ncOrObj,
                       long timestamp)
        {
            ncOrObj_ = ncOrObj;
            comp_ = nc;
            timestamp_ = timestamp;
        }
                       
        synchronized Binding
        toBinding()
        {
            Binding b = new Binding();
            b.binding_name = new NameComponent[1];
            b.binding_name[0] = new NameComponent(comp_.id,
                                                  comp_.kind);
            b.binding_type = ncOrObj_.discriminator();

            return b;
        }

        synchronized ExtendedBinding
        toExtBinding()
        {
            ExtendedBinding b = new ExtendedBinding();
            b.binding_name = new NameComponent[1];
            b.binding_name[0] =
                new NameComponent(comp_.id, comp_.kind);
            b.ncOrObj = ncOrObj_;
            b.timestamp = (int)timestamp_;

            return b;
        }

        synchronized void
        rename(NameComponent nc)
        {
            comp_.id = nc.id;
            comp_.kind = nc.kind;
        }

        synchronized NcOrObj
        ncOrObj()
        {
            return ncOrObj_;
        }

        synchronized void
        rebind(NameComponent[] n, NcOrObj ncOrObj, long timestamp)
            throws NotFound
        {
            //
            // Contexts must be rebound to contexts, objects
            // to objects.
            //
            if(ncOrObj.discriminator() != ncOrObj_.discriminator())
            {
                //
                // INS (98-10-11) 4-12:
                //
                // If rebind or rebind_context raise a
                // NotFound exception because an already
                // existing binding is of the wrong type, the
                // rest_of_name member of the exception has a
                // sequence length of 1.
                //
                if(ncOrObj_.discriminator() == BindingType.ncontext)
                    throw new NotFound(NotFoundReason.not_object, n);
                else
                    throw new NotFound(NotFoundReason.not_context, n);
            }
            
            //
            // Update the binding information.
            //
            comp_ = n[0];
            ncOrObj_ = ncOrObj;
            timestamp_ = timestamp;
        }

        synchronized boolean
        isContext()
        {
            return ncOrObj_.discriminator() == BindingType.ncontext;
        }

        synchronized org.omg.CORBA.Object
        object()
        {
            if(ncOrObj_.discriminator() == BindingType.ncontext)
                return ncOrObj_.nc();
            return ncOrObj_.obj();
        }

        synchronized org.omg.CosNaming.NamingContext
        context(NameComponent[] n)
            throws NotFound
        {
	    //
	    // Binding must belong to a context.
	    //
	    if(ncOrObj_.discriminator() != BindingType.ncontext)
		throw new NotFound(NotFoundReason.not_context, n);

            return ncOrObj_.nc();
        }
    }

    //
    // This is used to create a unique name for a NameComponent for
    // use in the hashtable. We use the same algorithm as the
    // to_string method except we are only interested in the first
    // component.
    //
    private String
    bindingHashString(NameComponent[] name)
    {
	String result = new String();
	result += addEscapes(name[0].id);
	if(name[0].kind.length() > 0)
	{
	    result += '.';
	    result += addEscapes(name[0].kind);
	}

	return result;
    }

    private void
    cleanup()
    {
	//
	// Remove ourself from the NamingContextSet. This will stop the
	// naming database from dumping this context.
	//
	if(ncs_ != null)
	    ncs_.remove(key_);

	if(store_ != null)
	    store_.destroy(this);
	
	//
	// Disconnect the servant from the ORB.
	//
	orb_.disconnect(this);
    }
    
    private void
    doBind(NameComponent[] n, NcOrObj ncOrObj, boolean rebind, long timestamp)
	throws NotFound, AlreadyBound, CannotProceed, InvalidName
    {
	//
	// Check if name is valid.
	//
	if(n.length == 0)
	    throw new InvalidName();
	if(n[0].id.length() == 0 && n[0].kind.length() == 0)
	    throw new InvalidName();

        if(n.length == 1)
        {
            synchronized(bindings_)
            {
                NcOrObjBinding binding =
                    (NcOrObjBinding)bindings_.get(bindingHashString(n));
        
                if(binding != null)
                {
                    //
                    // Name is already bound.
                    //
                    if(!rebind)
                        throw new AlreadyBound();
                    
                    //
                    // Update the binding information.
                    //
                    
                    binding.rebind(n, ncOrObj, timestamp);
                }
                else
                {
                    //
                    // Bind the name and the object.
                    //
                    bindings_.put(bindingHashString(n), 
                                  new NcOrObjBinding(n[0], ncOrObj,
                                                     timestamp));
                }
            }
            
            if(store_ != null)
                store_.bind(this, n, ncOrObj, timestamp, rebind);
            return;
        }

        NcOrObjBinding binding =
            (NcOrObjBinding)bindings_.get(bindingHashString(n));
        
        if(binding == null)
            throw new NotFound(NotFoundReason.missing_node, n);

        //
        // The context in which to bind.
        //
        org.omg.CosNaming.NamingContext context = binding.context(n);
    
        //
        // Create the rest of the name.
        //
        NameComponent[] rest = new NameComponent[n.length - 1];
        System.arraycopy(n, 1, rest, 0, rest.length);
        
        //
        // Let the next context handle the binding.
        //
        if(ncOrObj.discriminator() == BindingType.ncontext)
        {
            if(rebind)
                context.rebind_context(rest, ncOrObj.nc());
            else
                context.bind_context(rest, ncOrObj.nc());
        }
        else
        {
            if(rebind)
                context.rebind(rest, ncOrObj.obj());
            else
                context.bind(rest, ncOrObj.obj());
	}
    }

    //
    // Notify all listeners that a binding has been added.
    //
    private void
    bindingAdded(NcOrObj ncOrObj, NameComponent[] n)
    {
	//
	// Check whether change occured in this context.
	//
	if(n.length != 1)
	    return;

	if(!noUpdates_)	synchronized(listeners_)
	{
	    boolean isContext =
		(ncOrObj.discriminator() == BindingType.ncontext);
		
	    com.ooc.CORBA.MessageViewer.instance().trace(4, "Binding added");
		
	    java.util.Enumeration e = listeners_.elements();
	    while(e.hasMoreElements())
	    {
		com.ooc.CosNaming.BindingListener l =
		    (com.ooc.CosNaming.BindingListener)e.nextElement();
		try
		{
		    if(isContext)
			l.context_added(this, ncOrObj.nc(), n[0]);
		    else
			l.object_added(this, ncOrObj.obj(), n[0]);
		}
		catch(SystemException ex)
		{
		    com.ooc.CORBA.MessageViewer.instance().trace(3,
		         "SystemException: Removing listener");
		    listeners_.removeElement(l);
		}
	    }
	}
    }

    //
    // Notify all listeners that a binding has been removed.
    //
    private void
    bindingRemoved(NameComponent[] n, boolean isContext)
    {
	//
	// Check whether change occured in this context
	//
	if(n.length != 1)
	    return;

	if(!noUpdates_) synchronized(listeners_)
	{
	    com.ooc.CORBA.MessageViewer.instance().trace(4, "Binding removed");
	    
	    java.util.Enumeration e = listeners_.elements();
	    while(e.hasMoreElements())
	    {
		com.ooc.CosNaming.BindingListener l =
		    (com.ooc.CosNaming.BindingListener)e.nextElement();
		
		try
		{
		    if(isContext)
			l.context_removed(this, n[0]);
		    else
			l.object_removed(this, n[0]);
		}
		catch(SystemException ex)
		{
		    com.ooc.CORBA.MessageViewer.instance().trace(3,
		         "SystemException: Removing listener");
		    listeners_.removeElement(l);
		}
	    }
	}
    }

    //
    // Notify all listeners that a binding has been changed.
    //
    private void
    bindingChanged(NcOrObj ncOrObj, NameComponent[] oldName,
		   NameComponent[] newName)
    {
	//
	// Check whether change occured in this context
	//
	if(oldName.length != 1)
	    return;

	if(!noUpdates_) synchronized(listeners_)
	{
	    boolean isContext =
		(ncOrObj.discriminator() == BindingType.ncontext);
	    com.ooc.CORBA.MessageViewer.instance().trace(4, "Binding changed");
	    
	    java.util.Enumeration e = listeners_.elements();
	    while(e.hasMoreElements())
	    {
		com.ooc.CosNaming.BindingListener l =
		    (com.ooc.CosNaming.BindingListener)e.nextElement();
		
		try
		{
		    if(isContext)
			l.context_changed(this, ncOrObj.nc(), oldName[0],
					  newName[0]);
		    else
			l.object_changed(this, ncOrObj.obj(), oldName[0],
					 newName[0]);
		}
		catch(SystemException ex)
		{
		    com.ooc.CORBA.MessageViewer.instance().trace(3,
		         "SystemException: Removing listener");
		    listeners_.removeElement(l);
		}
	    }
	}
    }

    //
    // Add stringified-name escapes. The characters '.', '/' and '\\'
    // must be escaped with a '\\'.
    //
    private String
    addEscapes(String str)
    {
	String result = new String();

	int len = str.length();
	for(int i = 0 ; i < len ; i++)
	{
	    char c = str.charAt(i);
	    switch(c)
	    {
	    case '.':
		result += "\\.";
		break;

	    case '/':
		result += "\\/";
		break;

	    case '\\':
		result += "\\\\";
		break;

	    default:
		result += c;
		break;
	    }
	}

	return result;
    }

    //
    // Escape characters according to URL encoding scheme
    // except for the characters listed in 4.6.3. We cannot use the
    // java.net.URLEncoder class to do this since it appears to
    // escape the '/' character.
    //
    static private String
    addURLEscapes(String s)
    {
	String result = new String();
	for(int i = 0 ; i < s.length() ; i++)
	{
	    char ch = s.charAt(i);
	    //
	    // As per RFC 2396.
	    //
	    if(Character.isLetter(ch) || Character.isDigit(ch))
		result += ch;
	    else
	    {
		switch(ch)
		{
		case ';': case '/': case '?': case ':':
		case '@': case '&': case '=': case  '+':
		case '$': case ',': case '-': case '_':
		case '.': case '!': case '~': case '*':
		case '\'':case '(': case ')':
		    result += ch;
		    break;

		default:
		    result += '%';
		    result += Character.toUpperCase(
			Character.forDigit((ch >> 4) & 0xF, 16));
		    result += Character.toUpperCase(
			Character.forDigit(ch & 0xF, 16));
		}
	    }
	}

	return result;
    }

    NamingContext
    createNewContext()
    {
	//
	// Create new unique key. The key is composed of the current
	// time and a running count.
	//
	String name = new java.util.Date() + "-" + count_++;

	return new NamingContext(orb_, store_, ncs_, name, noUpdates_);
    }

    //
    // Constructor and destructor
    //
    NamingContext(ORB orb, NamingDatabase store, java.util.Hashtable ncs,
		  String key, boolean noUpdates)
    {
	orb_ = orb;
	key_ = key;
	bindings_ = new java.util.Hashtable(1023);
	ncs_ = ncs;
	store_ = store;
	noUpdates_ = noUpdates;
	destroyed_ = false;

	boolean isRoot = key_.equals("NameService");

	((com.ooc.CORBA.ORB)orb_).connect(this, key_);

	if(ncs_ != null)
	    ncs_.put(key_, this);

	if(store_ != null)
	{
	    if(isRoot)
		store_.info();
	    store_.create(this, (int)(System.currentTimeMillis() / 1000));
	}

        if(isRoot)
        {
            String prop = com.ooc.CORBA.Properties.instance().
                getProperty("ooc.naming.display_root_ior");
            if(prop != null && prop.equals("true"))
                System.out.println(orb_.object_to_string(this));
        }
    }

    //
    // For ORBacus Names built-in naming service
    //
    public NamingContext(ORB orb, BOA boa)
    {
	this(orb, null, null, "NameService", false);

	boa_ = boa;
    }

    protected void
    finalize()
        throws Throwable
    {
	if(ncs_ != null)
	    ncs_.remove(this);
        super.finalize();
    }

    //
    // From java.lang.Thread, for ORBacus Names
    //
    public void
    run()
    {
	boa_.impl_is_ready(null);
    }

    // ------------------------------------------------------------------
    // Standard IDL to Java Mapping
    // ------------------------------------------------------------------

    //
    // Implementation of CosNaming::NamingContext.
    //

    public void
    bind(NameComponent[] n, org.omg.CORBA.Object obj)
        throws NotFound, CannotProceed, InvalidName, AlreadyBound
    {
	if(destroyed_)
	    throw new OBJECT_NOT_EXIST();

	NcOrObj ncOrObj = new NcOrObj();
	ncOrObj.obj(obj);

	doBind(n, ncOrObj, false, System.currentTimeMillis() / 1000);

	bindingAdded(ncOrObj, n);
    }

    public void
    rebind(NameComponent[] n, org.omg.CORBA.Object obj)
        throws NotFound, CannotProceed, InvalidName
    {
	if(destroyed_)
	    throw new OBJECT_NOT_EXIST();

	NcOrObj ncOrObj = new NcOrObj();
	ncOrObj.obj(obj);

	try
	{

	    doBind(n, ncOrObj, true, System.currentTimeMillis() / 1000);
	}
	catch(AlreadyBound ex)
	{
	    //
	    // AlreadyBound must not be thrown here, so convert it.  This
	    // shouldn't happen -- therefore an application error.
	    //
	    throw new InternalError();
	}

	bindingChanged(ncOrObj, n, n);
    }

    public void
    bind_context(NameComponent[] n, org.omg.CosNaming.NamingContext nc)
        throws NotFound, CannotProceed, InvalidName, AlreadyBound
    {
	if(destroyed_)
	    throw new OBJECT_NOT_EXIST();

	if(nc == null)
	    throw new BAD_PARAM();

	NcOrObj ncOrObj = new NcOrObj();
	ncOrObj.nc(nc);

	doBind(n, ncOrObj, false, System.currentTimeMillis() / 1000);
 
	bindingAdded(ncOrObj, n);
    }

    public void
    rebind_context(NameComponent[] n, org.omg.CosNaming.NamingContext nc)
        throws NotFound, CannotProceed, InvalidName
    {
	if(destroyed_)
	    throw new OBJECT_NOT_EXIST();

	if(nc == null)
	    throw new BAD_PARAM();

	NcOrObj ncOrObj = new NcOrObj();
	ncOrObj.nc(nc);

	try
	{
	    doBind(n, ncOrObj, true, System.currentTimeMillis() / 1000);
	}
	catch(AlreadyBound ex)
	{
	    //
	    // AlreadyBound must not be thrown here, so convert it.
	    // This shouldn't happen -- therefore an application
	    // error.
	    //
	    throw new InternalError();
	}

	bindingChanged(ncOrObj, n, n);
    }

    public org.omg.CORBA.Object
    resolve(NameComponent[] n)
        throws NotFound, CannotProceed, InvalidName
    {
	if(destroyed_)
	    throw new OBJECT_NOT_EXIST();

	//
	// Check if name is valid.
	//
	if(n.length == 0)
	    throw new InvalidName();
	if(n[0].id.length() == 0 && n[0].kind.length() == 0)
	    throw new InvalidName();

        NcOrObjBinding binding =
            (NcOrObjBinding)bindings_.get(bindingHashString(n));
        
        //
        // Missing node if binding was not found.
        //
        if(binding == null)
            throw new NotFound(NotFoundReason.missing_node, n);

	if(n.length == 1)
            return binding.object();

        //
        // Context in which to resolve.
        //
        org.omg.CosNaming.NamingContext context = binding.context(n);

	//
	// Create the rest of the name.
	//
	NameComponent[] rest = new NameComponent[n.length - 1];
	System.arraycopy(n, 1, rest, 0, rest.length);

	//
	// Have the context handle the resolving of the name.
	//
	return context.resolve(rest);
    }

    public void
    unbind(NameComponent[] n)
        throws NotFound, CannotProceed, InvalidName
    {
	if(destroyed_)
	    throw new OBJECT_NOT_EXIST();

	//
	// Check if name is valid.
	//
	if(n.length == 0)
	    throw new InvalidName();
	if(n[0].id.length() == 0 && n[0].kind.length() == 0)
	    throw new InvalidName();

        NcOrObjBinding binding;

        if(n.length == 1)
        {
            synchronized(bindings_)
            {
                binding = (NcOrObjBinding)bindings_.get(bindingHashString(n));
                
                //
                // Missing node if binding was not found.
                //
                if(binding == null)
                    throw new NotFound(NotFoundReason.missing_node, n);
                
                //
                // Do unbind.
                //
                bindings_.remove(bindingHashString(n));
            }
            
            if(store_ != null)
                store_.unbind(this, n);
            
            bindingRemoved(n, binding.isContext());
            return;
        }

        binding = (NcOrObjBinding)bindings_.get(bindingHashString(n));
        
        //
        // Missing node if binding was not found.
        //
        if(binding == null)
            throw new NotFound(NotFoundReason.missing_node, n);
        
        //
        // The context in which to unbind.
        //
        org.omg.CosNaming.NamingContext context = binding.context(n);
        
        //
        // Create the rest of the name.
        //
        NameComponent[] rest = new NameComponent[n.length - 1];
        System.arraycopy(n, 1, rest, 0, rest.length);
        
        //
        // Let the next context handle the resolving of the name.
        //
        context.unbind(rest);
    }

    public org.omg.CosNaming.NamingContext
    new_context()
    {
	if(destroyed_)
	    throw new OBJECT_NOT_EXIST();

	return createNewContext();
    }

    public org.omg.CosNaming.NamingContext
    bind_new_context(NameComponent[] n)
        throws NotFound, AlreadyBound, CannotProceed, InvalidName
    {
	if(destroyed_)
	    throw new OBJECT_NOT_EXIST();

	NamingContext newContext = createNewContext();
	org.omg.CosNaming.NamingContext nc = newContext;

	try
	{
	    bind_context(n, nc);
	}
	catch(SystemException ex)
	{
	    if(ex.completed == org.omg.CORBA.CompletionStatus.COMPLETED_NO)
		newContext.cleanup();

	    throw ex;
	}
	catch(NotFound ex)
	{
	    newContext.cleanup();
	    throw ex;
	}
	catch(CannotProceed ex)
	{
	    newContext.cleanup();
	    throw ex;
	}
	catch(InvalidName ex)
	{
	    newContext.cleanup();
	    throw ex;
	}
	catch(AlreadyBound ex)
	{
	    newContext.cleanup();
	    throw ex;
	}

	return nc;
    }

    public void
    destroy()
        throws NotEmpty
    {
	if(destroyed_)
	    throw new OBJECT_NOT_EXIST();

	if(!bindings_.isEmpty())
	    throw new NotEmpty();

	destroyed_ = true;

	cleanup();
    }

    public void
    list(int how_many, BindingListHolder bl, BindingIteratorHolder bi)
    {
	if(destroyed_)
	    throw new OBJECT_NOT_EXIST();

	//
	// Note that we have to be careful in this method. Since we don't
	// synchronize the set of bindings_ it's possible that the set of
	// bindings is altered during processing.
	//
	java.util.Enumeration keys = bindings_.elements();

	java.util.Vector blvec = new java.util.Vector();

	//
	// Copy the first how_many bindings into bl, taking into
	// account that some bindings may not be present. We may not
	// use the calculated `num' above since some bindings may no
	// longer be present.
	//
	while(keys.hasMoreElements() && blvec.size() < how_many)
	{
	    NcOrObjBinding binding = (NcOrObjBinding)keys.nextElement();
	    if(binding != null)
                blvec.addElement(binding.toBinding());
	}
	bl.value = new Binding[blvec.size()];
        blvec.copyInto(bl.value);
	
	//
	// If we have more bindings then create a BindingIterator with the
	// remainder of the existing bindings, otherwise we return a nil
	// BindingIterator.
	//
	blvec.removeAllElements();
	while(keys.hasMoreElements())
	{
	    NcOrObjBinding binding = (NcOrObjBinding)keys.nextElement();
	    if(binding != null)
                blvec.addElement(binding.toBinding());
	}
	
	if(blvec.size() > 0)
	{
	    Binding[] rest = new Binding[blvec.size()];
            blvec.copyInto(rest);
	    
	    bi.value = new BindingIterator(orb_, rest);
	}
	else
	{
	    bi.value = null;
	}
    }

    //
    // Implementation of CosNaming::NamingContextExt
    //
    public String
    to_string(NameComponent[] name)
    {
	String result = new String();

	for(int i = 0 ; i < name.length ; i++)
	{
	    if(i != 0)
		result += '/';

	    result += addEscapes(name[i].id);
	    if(name[i].kind.length() > 0)
	    {
		result += '.';
		result += addEscapes(name[i].kind);
	    }
	}

	return result;
    }

    public NameComponent[]
    to_name(String s)
        throws InvalidName
    {
	if(s.length() == 0)
	    throw new InvalidName();

	com.ooc.CORBA.IIOPNameParser parser =
	    new com.ooc.CORBA.IIOPNameParser(s);
	if(!parser.isValid())
	    throw new InvalidName();

	String[] contents = parser.getContents();

	NameComponent[] name = new NameComponent[contents.length / 2];
	int c = 0;
	for(int i = 0; i < contents.length; i += 2, c++)
	    name[c] = new NameComponent(contents[i], contents[i + 1]);

	return name;
    }

    public String
    to_url(String addr, String n)
    {
	String result = "iiopname://";

	result += addr;
	result += '/';

	result += addURLEscapes(n);

	return result;
    }

    public org.omg.CORBA.Object
    resolve_str(String n)
        throws NotFound, CannotProceed, InvalidName, AlreadyBound
    {
	return resolve(to_name(n));
    }

    //
    // Implementation of CosNaming::OBNamingContext
    //
    public ExtendedBinding[]
    list_extended()
    {
	if(destroyed_)
	    throw new OBJECT_NOT_EXIST();

	java.util.Enumeration keys = bindings_.elements();

	java.util.Vector ebvec = new java.util.Vector();
	while(keys.hasMoreElements())
	{
	    NcOrObjBinding binding = (NcOrObjBinding)keys.nextElement();
	    if(binding != null)
                ebvec.addElement(binding.toExtBinding());
	}
	
	ExtendedBinding[] eb = new ExtendedBinding[ebvec.size()];
        ebvec.copyInto(eb);

	return eb;
    }

    public void
    rename(NameComponent[] oldName, NameComponent[] newName)
	throws NotFound, AlreadyBound, InvalidName
    {
	if(destroyed_)
	    throw new OBJECT_NOT_EXIST();

	if(oldName.length != 1 || newName.length != 1)
	    throw new InvalidName();

	if(newName[0].id.length() == 0 && newName[0].kind.length() == 0)
	    throw new InvalidName();

	//
	// If the names are the same then we have nothing to do.
	//
	if(oldName[0].id.equals(newName[0].id) &&
	   oldName[0].kind.equals(newName[0].kind))
	    return;

        NcOrObj ncOrObj;

	//
	// We have to synchronize on the bindings and the store_ here
	// since we don't want the set of bindings changed in the middle
	// of the operation. Note that we have to acquire the lock on the
	// store_ before locking the binding set to avoid a deadlock.
	//

	//
	// This deals with the fact that store_ may be null.
	//
	java.lang.Object syncStore;
	if(store_ != null)
	    syncStore = store_;
	else
	    syncStore = bindings_;

	synchronized(syncStore)
	{
	synchronized(bindings_)
	{
            //
            // Locking the binding itself is not necessary below since the
            // bindings_ set is locked for the entire time that we have
            // the binding, and it's not possible to acquire a reference
            // to a binding without locking the bindings.
            //
            // Get the current binding.
            //
            NcOrObjBinding binding = (NcOrObjBinding)bindings_.get(
		bindingHashString(oldName));

	    if(binding == null)
		throw new NotFound(NotFoundReason.missing_node, oldName);

	    //
	    // Check if newName already exists
	    //
	    if(bindings_.containsKey(bindingHashString(newName)))
		throw new AlreadyBound();

	    //
	    // Remove the current, add the new.
	    //
	    bindings_.remove(bindingHashString(oldName));

            binding.rename(newName[0]);

	    bindings_.put(bindingHashString(newName), binding);

            ncOrObj = binding.ncOrObj();

	    if(store_ != null)
	    {
		store_.unbind(this, oldName);
		store_.bind(this, newName, ncOrObj,
			    System.currentTimeMillis() / 1000,
			    false);
	    }
	}
	}

	bindingChanged(ncOrObj, oldName, newName);
    }

    public boolean
    add_listener(com.ooc.CosNaming.BindingListener listener)
    {
	if(destroyed_)
	    throw new OBJECT_NOT_EXIST();

	if(!noUpdates_) synchronized(listeners_)
	{
	    //
	    // Check if listener is already registered.
	    //
	    java.util.Enumeration e = listeners_.elements();
	    while(e.hasMoreElements())
	    {
		com.ooc.CosNaming.BindingListener l =
		    (com.ooc.CosNaming.BindingListener)e.nextElement();
		if(listener._is_equivalent(l))
		    return true;
	    }

	    com.ooc.CORBA.MessageViewer.instance().trace(3, "Adding listener");
	    listeners_.addElement(listener);
	    return true;
	}
	return false;
    }

    public boolean
    remove_listener(com.ooc.CosNaming.BindingListener listener)
    {
	if(destroyed_)
	    throw new OBJECT_NOT_EXIST();

	if(!noUpdates_) synchronized(listeners_)
	{
	    java.util.Enumeration e = listeners_.elements();
	    while(e.hasMoreElements())
	    {
		com.ooc.CosNaming.BindingListener l =
		    (com.ooc.CosNaming.BindingListener)e.nextElement();
		if(listener._is_equivalent(l))
		{
		    com.ooc.CORBA.MessageViewer.instance().trace(
			3, "Removing listener");
		    listeners_.removeElement(l);
		    return true;
		}
	    }
	}

	return false;
    }

    public boolean
    callbacks_active()
    {
	return !noUpdates_;
    }

    //
    // Internal methods.
    //

    String
    _OB_getKey()
    {
	return key_;
    }

    void
    _OB_bindWithTimestamp(NameComponent[] n, NcOrObj ncOrObj, long timestamp)
	throws NotFound, AlreadyBound, CannotProceed, InvalidName
    {
	doBind(n, ncOrObj, false, timestamp);
    }
}
