// **********************************************************************
//
// Copyright (c) 1999
// Object Oriented Concepts, Inc.
// Billerica, MA, USA
//
// All Rights Reserved
//
// **********************************************************************

package com.ooc.CosNaming;

import java.io.*;
import org.omg.CORBA.*;
import org.omg.CosNaming.*;
import org.omg.CosNaming.NamingContextPackage.*;
import com.ooc.CosNaming.OBNamingContextPackage.*;
import com.ooc.CosNaming.Database.*;

//
// This class reads or writes the contents of a NamingContext to or from
// a persistent store.
//
final class NamingDatabase
{
    private ORB orb_; // The ORB.

    private String host_; // The host.
    private short port_; // The port.

    String logFile_; // The name of the logfile.
    String logFileNew_; // The name of the new logfile.
    String logFileBack_; // The name of the backup logfile.

    java.util.Hashtable ncs_; // The current set of naming contexts.
    FileOutputStream os_; // The current output stream.

    boolean noUpdates_; // No automatic updates.
    boolean initializing_; // Are we restoring the naming contexts?
    boolean update_; // Do we need a journal update?
    boolean noflush_; // Do we want to flush on each update?

    // ----------------------------------------------------------------------
    // NamingDatabase private member implementation
    // ----------------------------------------------------------------------
 
    //
    // Dump the contents of a naming context. This is the set of bindings
    // in the context.
    //
    private void
    dumpContext(org.omg.CosNaming.NamingContext nc)
	throws UserException
    {
	orb_.connect(nc);
	org.omg.CORBA.DynamicImplementation impl =
	    (org.omg.CORBA.DynamicImplementation)nc;
	if(!((com.ooc.CORBA.Delegate)impl._get_delegate()).is_local())
	    throw new RuntimeException();

	//
	// Retrieve all bindings.
	//
	ExtendedBinding[] b =
	    ((com.ooc.CosNaming.OBNamingContext)nc).list_extended();

	for(int i = 0 ; i < b.length ; i++)
	    bind(nc, b[i].binding_name, b[i].ncOrObj, b[i].timestamp, false);
    }

    private void
    writeAny(Any any)
    {
	com.ooc.CORBA.OutputStream out = new com.ooc.CORBA.OutputStream();

	//
	// We marshal our endianess
	//
	out._OB_writeEndian();

	//
	// Now marshal the actual data.
	//
	out.write_any(any);

	try
	{
	    os_.write(Converter.octetsToAscii(out._OB_buffer(),
					      out._OB_count()).getBytes());

	    os_.write('\n');
	    if(!noflush_)
		os_.flush();
	}
	catch(IOException ex)
	{
            com.ooc.CORBA.MessageViewer.instance().error("Write failed!");
            throw new PERSIST_STORE();
	}
    }

    //
    // Rename the file identified by `from' to the new name identified by
    // `to'.
    //
    private boolean
    doRename(String from, String to)
    {
	boolean result = (new File(from)).renameTo(new File(to));
	if(!result)
	{
	    com.ooc.CORBA.MessageViewer.instance().
		error("Rename " + from + " to " + to + " failed.");

	    return false;
	}

	return true;
    }

    //
    // Check to see if the file identified by `file' exists.
    //
    private boolean
    exists(String file)
    {
	return (new File(file).exists());
    }

    //
    // Open the logfile.
    //
    private void
    openLog()
    {
	//
	// Open the logFile_.
	//
	try
	{
	    os_ = new FileOutputStream(logFile_, true);
	}
	catch(IOException ex)
	{
	    com.ooc.CORBA.MessageViewer.instance().
		error("Cannot open `" + logFile_ + "': " + ex.toString());
	    throw new PERSIST_STORE();
	}
    }

    //
    // Constructor
    //
    NamingDatabase(BOA boa, ORB orb, String log, java.util.Hashtable ncs,
 		   boolean noUpdates, boolean start)
    {
	orb_ = orb;
	ncs_ = ncs;
	noUpdates_ = noUpdates;
	initializing_ = false;
	update_ = false;
	noflush_ = false;

	//
	// Determine host and port.
	//
	com.ooc.OCI.AccRegistry registry =
	    ((com.ooc.CORBA.BOA)boa).get_acc_registry();
	com.ooc.OCI.Acceptor[] acceptors = registry.get_acceptors();

	for(int i = 0 ; i < acceptors.length ; i++)
	{
	    com.ooc.OCI.AcceptorInfo accInfo = acceptors[i].get_info();
	    com.ooc.OCI.IIOP.AcceptorInfo iiopInfo =
		com.ooc.OCI.IIOP.AcceptorInfoHelper.narrow(accInfo);
	    if(iiopInfo != null)
	    {
		host_ = iiopInfo.host();
		port_ = iiopInfo.port();

		break;
	    }
	}

	//
	// Setup the filenames.
	//
	logFile_ = log;
	logFileNew_ = log + ".new";
	logFileBack_ = log + ".bak";

	com.ooc.CORBA.MessageViewer viewer =
	    com.ooc.CORBA.MessageViewer.instance();

	//
	// If we're running for the first time, then we need to verify
	// that the logfile is not already present.
	//
	if(start)
	{
	    if(exists(logFile_))
	    {
		viewer.error("Cannot specify --start option. " +
			     "There is an existing long file: `" +
			     logFile_ + "'.");
		throw new PERSIST_STORE();
	    }

	    openLog();
	    return;
	}

	//
	// If the logfile doesn't exist, and the backup file does then
	// this is probably indicative of a crash.
	//
	if(!exists(logFile_) && exists(logFileBack_))
	{
	    viewer.warning("NamingDatabase file is not present: " +
			   logFile_ + ". Using backup file: " + logFileBack_);
	    
	    if(!doRename(logFileBack_, logFile_))
	    {
		viewer.warning("Renamed of backup file failed.");
		throw new PERSIST_STORE();
	    }
	}
  
	//
	// Does the logfile exist?
	//
	if(!exists(logFile_))
	{
	    viewer.error("NamingDatabase file is not present: `" +
			 logFile_ + "'. Run with --start option if " +
			 "running for the first time.");
	    throw new PERSIST_STORE();
	}

	//
	// Ok, let's get down to business. Read the contents of the
	// logfile.  This is a set of strings, each of which is an octet
	// encoding of a marshaled any.
	//
	FileInputStream is = null;
	BufferedReader reader = null;
	try
	{
	    is = new FileInputStream(logFile_);
	    reader = new BufferedReader(new InputStreamReader(is));
	}
	catch(IOException ex)
	{
	    viewer.error("Cannot open store file: " + logFile_);
	    throw new PERSIST_STORE();
	}

	initializing_ = true;
	int rec = 0;
	try
	{
	    try
	    {
		try
		{
		    while(true)
		    {
			String cmd = reader.readLine();
			if(cmd == null)
			    break;

			//
			// Convert the octet character string to an
			// octet sequence and create input stream.
			//
			com.ooc.CORBA.InputStream in =
			    new com.ooc.CORBA.InputStream(
				Converter.asciiToOctets(cmd));

			//
			// Unmarshal the endianness, and the any.
			//
			in._OB_readEndian();
			Any any = in.read_any();

			//
			// Next extract the instruction.
			//
			Version version;
			Info inf;
			Create create;
			Bind bind;
			Destroy destroy;
			Unbind unbind;

			TypeCode tc = any.type();
			if(tc.equal(VersionHelper.type()))
			{
			    // No action has to be taken here
			}
			else if(tc.equal(CreateHelper.type()))
			{
			    create = CreateHelper.extract(any);
			    
			    //
			    // Create a new naming context with the
			    // provided key.
			    //
			    new com.ooc.CosNaming.NamingContext(orb_, this,
								ncs_,
								create.key,
								noUpdates_);
			}
			else if(tc.equal(BindHelper.type()))
			{
			    bind = BindHelper.extract(any);

			    com.ooc.CosNaming.NamingContext nc =
				(com.ooc.CosNaming.NamingContext)(bind.nc);

			    //
			    // Create a new binding.
			    //
			    if(bind.type.discriminator() ==
                               BindingType.ncontext)
			    {
				if(bind.rebind)
				    nc.rebind_context(bind.name,
						      bind.type.nc());
				else
				{
				    NcOrObj ncOrObj = new NcOrObj();
				    ncOrObj.nc(bind.type.nc());
				    nc._OB_bindWithTimestamp(bind.name,
							     ncOrObj,
							     bind.timestamp);
				}
			    }
			    else if(bind.rebind)
			    {
				nc.rebind(bind.name, bind.type.obj());
			    }
			    else
			    {
				NcOrObj ncOrObj = new NcOrObj();
				ncOrObj.obj(bind.type.obj());
				nc._OB_bindWithTimestamp(bind.name,
							 ncOrObj,
							 bind.timestamp);
			    }
			}
			else if(tc.equal(DestroyHelper.type()))
			{
			    destroy = DestroyHelper.extract(any);

			    //
			    // Destroy a context.
			    //
			    destroy.nc.destroy();
			}
			else if(tc.equal(UnbindHelper.type()))
			{
			    unbind = UnbindHelper.extract(any);

			    //
			    // Do an unbind.
			    //
			    unbind.nc.unbind(unbind.name);
			}
			else if(tc.equal(InfoHelper.type()))
			{
			    inf = InfoHelper.extract(any);
			    if(!host_.equals(inf.host) || port_ != inf.port)
			    {
				viewer.error("Wrong host or port: " + host_ +
					     ":" + port_ +
					     " Expected host and port" +
					     inf.host + ":" + inf.port);
				throw new PERSIST_STORE();
			    }
			}
			else
			{
			    viewer.error("Unknown any in database.");
			    throw new PERSIST_STORE();
			}
			rec++;
		    }
		}
		catch(NotFound ex)
		{
		    String msg = new String();
		    msg += "Restore failed: NotFound: ";
		    msg += "Reason: ";
		    if(ex.why == NotFoundReason.missing_node)
			msg += "missing_node";
		    else if(ex.why == NotFoundReason.not_context)
			msg += "not_context";
		    else
			msg += "not_object";
		    msg += "Rest of name: ";
		    for(int i = 0 ; i < ex.rest_of_name.length ; i++)
		    {
			msg += "(" + ex.rest_of_name[i].id + "," +
			    ex.rest_of_name[i].kind + ") ";
		    }
		    viewer.error(msg);
		    throw new PERSIST_STORE();
		}
		catch(CannotProceed ex)
		{
		    String msg = new String();
		    msg += "Restore failed: CannotProceed";
		    msg += " Rest of name: ";
		    for(int i = 0 ; i < ex.rest_of_name.length ; i++)
			msg += "(" + ex.rest_of_name[i].id + "," +
					 ex.rest_of_name[i].kind + ") ";
		    viewer.error(msg);
		    throw new PERSIST_STORE();
		}
		catch(InvalidName ex)
		{
		    viewer.error("Restore failed: InvalidName");
		    throw new PERSIST_STORE();
		}
		catch(AlreadyBound ex)
		{
		    viewer.error("Restore failed: AlreadyBound");
		    throw new PERSIST_STORE();
		}
		catch(NotEmpty ex)
		{
		    viewer.error("Restore failed: NotEmpty");
		    throw new PERSIST_STORE();
		}
	    }
	    catch(PERSIST_STORE ex)
	    {
		viewer.error("Error processing record: " + rec);
		is.close();
		throw ex;
	    }
	}
	catch(IOException ex)
	{
	}

	try
	{
	    is.close();
	}
	catch(IOException ex)
	{
	}

	initializing_ = false;

	//
	// Re-open the logfile for new journal entries.
	//
	openLog();
	update_ = true;
    }

    protected void
    finalize()
        throws Throwable
    {
	try
	{
	    os_.close();
	}
	catch(IOException ex)
	{
	}
        super.finalize();
    }

    synchronized public void
    updateJournal()
    {
	if(!update_)
	    return;

	com.ooc.CORBA.MessageViewer viewer =
	    com.ooc.CORBA.MessageViewer.instance();;
	try
	{
	    //
	    // Close old journal.
	    //
	    os_.close();

	    os_ = new FileOutputStream(logFileNew_);
	}
	catch(IOException ex)
	{
	    viewer.error("Cannot open `" + logFileNew_ + "': " +
			 ex.toString());
	    throw new PERSIST_STORE();
	}

	//
	// Write
	//
	noflush_ = true;

	//
	// Write version, host and port
	//
	info();

	//
	// We write the entire contents of our local naming tree to the
	// journal.  We do this in two steps. First we write the create
	// instructions.  Next we write the bindings. It's important that
	// this order is preserved else we can have a binding to a local
	// naming context *before* that context exists.
	//
	try
	{
	    try
	    {
		synchronized(ncs_)
		{
		    java.util.Enumeration keys = ncs_.elements();
		    while(keys.hasMoreElements())
		    {
			org.omg.CosNaming.NamingContext nc =
			    (org.omg.CosNaming.NamingContext)
			    keys.nextElement();
			create(nc, 0);
		    }
		    keys = ncs_.elements();
		    while(keys.hasMoreElements())
		    {
			org.omg.CosNaming.NamingContext nc =
			    (org.omg.CosNaming.NamingContext)
			    keys.nextElement();
			try
			{
			    dumpContext(nc);
			}
			catch(OBJECT_NOT_EXIST ex)
			{
			    //
			    // Ignore this. If this occurs then the
			    // context is in the middle of being
			    // destroyed and the destroy record will
			    // be added to the store once we've
			    // finished dumping the journal.
			    //
			}
		    }
		}
	    }
	    catch(CannotProceed ex)
	    {
		String msg = new String();
		msg += "Compact failed: CannotProceed";
		msg += " Rest of name: ";
		for(int i = 0 ; i < ex.rest_of_name.length ; i++)
		    msg += "(" + ex.rest_of_name[i].id + "," +
			ex.rest_of_name[i].kind + ") ";
		viewer.error(msg);
		throw new PERSIST_STORE();
	    }
	    catch(InvalidName ex)
	    {
		viewer.error("Compact failed: InvalidName");
		throw new PERSIST_STORE();
	    }
	    catch(AlreadyBound ex)
	    {
		viewer.error("Compact failed: AlreadyBound");
		throw new PERSIST_STORE();
	    }
	    catch(UserException ex)
	    {
		viewer.error("Compact failed: UserException");
		throw new PERSIST_STORE();
	    }
	    catch(SystemException ex)
	    {
		viewer.error("Compact failed: SystemException");
		throw new PERSIST_STORE();
	    }
	}
	catch(PERSIST_STORE ex)
	{
	    noflush_ = false;
	    try
	    {
		os_.close();
	    }
	    catch(IOException e)
	    {
	    }

	    //
	    // Re-open old log file.
	    //
	    openLog();

	    throw ex;
	}

	try
	{
	    os_.close();
	}
	catch(IOException e)
	{
	}

	noflush_ = false;

	//
	// Move the current to backup. If this fails, it's not a big deal.
	//
	doRename(logFile_, logFileBack_);
 
	//
	// Move the new logfile to current. This failing IS a big deal!
	//
	if(!doRename(logFileNew_, logFile_))
	    throw new PERSIST_STORE();
 
	(new File(logFileBack_)).delete();

	//
	// Open new.
	//
	openLog();
	update_ = false;
    }

    //
    // Add a create instruction to the journal.
    //
    synchronized public void
    create(org.omg.CosNaming.NamingContext nc, long timestamp)
    {
	if(initializing_)
	    return;
 
	com.ooc.CosNaming.NamingContext impl =
	    (com.ooc.CosNaming.NamingContext)nc;
	Create create = new Create(impl._OB_getKey(), (int)timestamp);

	Any any = orb_.create_any();
	CreateHelper.insert(any, create);

	writeAny(any);

	update_ = true;
    }

    //
    // Add a destroy instruction to the journal.
    //
    synchronized public void
    destroy(org.omg.CosNaming.NamingContext nc)
    {
	if(initializing_)
	    return;
 
	Destroy destroy = new Destroy(nc);
	Any any = orb_.create_any();
	DestroyHelper.insert(any, destroy);
	writeAny(any);
 
	update_ = true;
    }

    //
    // Add a bind instruction to the journal.
    //
    synchronized public void
    bind(org.omg.CosNaming.NamingContext nc, NameComponent[] name,
	 NcOrObj ncOrObj, long timestamp, boolean rebind)
    {
	if(initializing_)
	    return;

	Bind bind = new Bind(rebind, nc, name, ncOrObj, (int)timestamp);
	Any any = orb_.create_any();
	BindHelper.insert(any, bind);
	writeAny(any);
 
	update_ = true;
    }

    //
    // Add an unbind instruction to the journal.
    //
    synchronized public void
    unbind(org.omg.CosNaming.NamingContext nc, NameComponent[] name)
    {
	if(initializing_)
	    return;
 
	Unbind unbind = new Unbind(nc, name);
	Any any = orb_.create_any();
	UnbindHelper.insert(any, unbind);
	writeAny(any);
 
	update_ = true;
    }

    //
    // Add host and port to the journal.
    //
    synchronized public void
    info()
    {
	if(initializing_)
	    return;
 
	Any any = orb_.create_any();

	//
	// Write version.
	//
	Version version = new Version(com.ooc.CORBA.Version.version);
	VersionHelper.insert(any, version);
	writeAny(any);

	//
	// Write host and port.
	//
	Info inf = new Info(host_, port_);
	InfoHelper.insert(any, inf);
	writeAny(any);
 
	update_ = true;
    }
}
