// **********************************************************************
//
// Copyright (c) 1999
// Object Oriented Concepts, Inc.
// Billerica, MA, USA
//
// All Rights Reserved
//
// **********************************************************************

package chat;

import org.omg.CORBA.*;
import chat.BroadcasterPackage.*;

final public class Broadcaster_impl extends _BroadcasterImplBase
{
    private ReceiverDescription[] receiverDescSeq_ =
        new ReceiverDescription[0];

    //
    // Add element to ReceiverDescription array
    //
    private int
    inc()
    {
	int len = receiverDescSeq_.length + 1;
	ReceiverDescription[] r = new ReceiverDescription[len];
	System.arraycopy(receiverDescSeq_, 0, r, 0, len - 1);
	receiverDescSeq_ = r;
	receiverDescSeq_[len - 1] = new ReceiverDescription();

	return len;
    }

    //
    // Remove last element from ReceiverDescription array
    //
    private void
    dec()
    {
	int len = receiverDescSeq_.length - 1;
	ReceiverDescription[] r = new ReceiverDescription[len];
	System.arraycopy(receiverDescSeq_, 0, r, 0, len);
	receiverDescSeq_ = r;
    }

    //
    // Add receiver
    //
    public void
    add(Receiver receiver, String id, String host, String nick)
	throws AlreadyRegistered, NickExists
    {
	//
	// Check if already registered
	//
	for(int i = 0 ; i < receiverDescSeq_.length ; i++)
	    if(receiverDescSeq_[i].receiver._is_equivalent(receiver))
		throw new AlreadyRegistered();
 
	//
	// Check if nick already exists
	//
	for(int i = 0 ; i < receiverDescSeq_.length ; i++)
	    if(receiverDescSeq_[i].nick.equals(nick))
		throw new NickExists();

	//
	// Add receiver description
	//
	int len = inc();
	receiverDescSeq_[len - 1].receiver = receiver;
	receiverDescSeq_[len - 1].id = id;
	receiverDescSeq_[len - 1].host = host;
	receiverDescSeq_[len - 1].nick = nick;

    }

    //
    // Remove receiver
    //
    void
    remove(Receiver receiver)
	throws UnknownReceiver
    {
	//
	// Search for receiver
	//
	for(int i = 0 ; i < receiverDescSeq_.length ; i++)
	{
	    if(receiverDescSeq_[i].receiver._is_equivalent(receiver))
	    {
		//
		// Receiver found, remove it
		//
		while(i < receiverDescSeq_.length - 1)
		{
		    receiverDescSeq_[i] = receiverDescSeq_[i + 1];
		    i++;
		}
		dec();

		return;
	    }
	}

	//
	// Receiver was not found
	//
	throw new UnknownReceiver();
    }

    //	
    // Register receiver
    //
    public void
    register(Receiver receiver, String id, String host, String nick)
	throws AlreadyRegistered, NickExists
    {
	//
	// Do nothing if this is a nil object reference
	//
	if(receiver == null)
	    return;

	//
	// Add receiver
	//
	add(receiver, id, host, nick);
    
	//
	// Send `join' message to others
	//
	String message = "*** " + nick + " (" + id + "@" + host +
	    ") has joined the chat";

	say(message);
    }

    //	
    // Register receiver with check
    //
    public void
    registerWithCheck(Receiver receiver, String id, String host, String nick)
	throws ReceiverUnreachable, AlreadyRegistered, NickExists
    {
	//
	// Do nothing if this is a nil object reference
	//
	if(receiver == null)
	    return;

	//
	// Try to ping the receiver
	//
	boolean nonExist = true;
	try
	{
	    nonExist = receiver._non_existent();
	}
	catch(COMM_FAILURE ex)
	{
	}
	
	if(nonExist)
	    throw new ReceiverUnreachable();

	//
	// Register receiver
	//
	register(receiver, id, host, nick);
    }

    //	
    // Unregister receiver
    //
    public void
    unregister(Receiver receiver)
	throws UnknownReceiver
    {
	//
	// Do nothing if this is a nil object reference
	//
	if(receiver == null)
	    return;
 
	//
	// Remember nick name
	//
	String nick = null;
	for(int i = 0 ; i < receiverDescSeq_.length ; i++)
        if(receiverDescSeq_[i].receiver._is_equivalent(receiver))
            nick = receiverDescSeq_[i].nick;

	//
	// Remove receiver
	//
	remove(receiver);
 
	//
	// Send `leave' message to other
	//
	if(nick != null)
	{
	    String message = "*** " + nick + " has left the chat";

	    say(message);
	}	
    }

    //	
    // Set the nick name
    //
    public void
    setNickName(Receiver receiver, String nick)
	throws NickExists, UnknownReceiver
    {
	//
	// Do nothing if this is a nil object reference
	//
	if(receiver == null)
	    return;
 
	//
	// Check if nick already exists
	//
	for(int i = 0 ; i < receiverDescSeq_.length ; i++)
	{
	    if(receiverDescSeq_[i].nick.equals(nick))
		throw new NickExists();
	}
	//
	// Search for receiver
	//
	for(int i = 0 ; i < receiverDescSeq_.length ; i++)
	{
	    if(receiverDescSeq_[i].receiver._is_equivalent(receiver))
	    {
		//
		// Receiver found, set nick name
		//
		String oldNick = receiverDescSeq_[i].nick;
		receiverDescSeq_[i].nick = nick;
 
		//
		// Send `is known as' message to all
		//
		String message = "*** " + oldNick +
		    " is now known as " + nick;

		say(message);

		return;
	    }
	}

	//
	// Receiver was not found
	//
	throw new UnknownReceiver();
    }

    //	
    // Get a receiver by its nick name
    //
    public void
    getReceiverByNick(String nick, ReceiverDescriptionHolder desc)
	throws UnknownNick
    {
	//
	// Search for nick name
	//
	for(int i = 0 ; i < receiverDescSeq_.length ; i++)
	    if(receiverDescSeq_[i].nick.equals(nick))
        {
            desc.value = receiverDescSeq_[i];
            return;
        }
 
	//
	// Nick name was not found
	//
	throw new UnknownNick();
    }

    //	
    // Get all receiver names
    //
    public String[]
    getReceiverNames()
    {
	String[] names = new String[receiverDescSeq_.length];

	for(int i = 0 ; i < receiverDescSeq_.length ; i++)
	    names[i] = receiverDescSeq_[i].nick + " (" +
		receiverDescSeq_[i].id + "@" +
		receiverDescSeq_[i].host + ")";

	return names;
    }

    //
    // Say something
    //
    public void
    say(String text)
    {
	//
	// Remember `bad' object references in a `bad sequence'
	//
        java.util.Vector badSeq = new java.util.Vector();

	for(int i = 0 ; i < receiverDescSeq_.length ; i++)
	{
	    try
	    {
		receiverDescSeq_[i].receiver.print(text);
	    }
	    catch(COMM_FAILURE ex)
	    {
		badSeq.addElement(receiverDescSeq_[i]);
	    }
	}

	//
	// Unregister `bad' object references
	//
	try
	{
	    for(int i = 0 ; i < badSeq.size() ; i++)
		remove(((ReceiverDescription)(badSeq.elementAt(i))).receiver);
	}
	catch(UnknownReceiver ex)
	{
	}

	//
	// Send `vanish' messages
	//
	for(int i = 0 ; i < badSeq.size() ; i++)
	{
	    String message = "*** " +
		((ReceiverDescription)(badSeq.elementAt(i))).nick +
		" vanished";

	    say(message);
	}

	//
	// Print text as log message, too
	//
	System.out.println(text);
    }
}
