// **********************************************************************
//
// Copyright (c) 1999
// Object Oriented Concepts, Inc.
// Billerica, MA, USA
//
// All Rights Reserved
//
// **********************************************************************

package chat;

import chat.BroadcasterPackage.*;
import org.omg.CORBA.*;
import java.util.*;
import java.net.*;

final class Receiver_impl extends _ReceiverImplBase
{
    //
    // The client window
    //
    private ClientWindow window_;

    //
    // The server object reference
    //
    private Broadcaster server_;

    //
    // The BOA
    //
    private BOA boa_;

    //
    // My ID, host and nickname
    //
    private String id_;
    private String host_;
    private String nick_;

    //
    // Constructor
    //
    Receiver_impl(ClientWindow window, Broadcaster server, BOA boa)
    {
	window_ = window;
	server_ = server;
	boa_ = boa;

	//
	// Get ID and nick
	//
	try
	{
	    Properties props = System.getProperties();

	    id_ = props.getProperty("user.name");
	    nick_ = props.getProperty("CHATNICK");
	}
	catch(SecurityException ex)
	{
	}

	if(id_ == null)
	    id_ = "<unknown>";

	if(nick_ == null)
	    nick_ = id_;

	//
	// Try to get canonical hostname
	//
	try
	{
	    InetAddress host = InetAddress.getLocalHost();
	    host_ = host.getHostName();
	}
	catch(UnknownHostException ex)
	{
	    host_ = "<unknown>";
	}

	//
	// Register with the server
	//
	register();
    }

    //
    // Parse input
    //
    void
    parse(String input)
    {
	if(input.charAt(0) != '/')
	{
	    //
	    // No command, "say" it
	    //
	    String message = "<" + nick_ + "> " + input;

	    while(true)
	    {
		try
		{
		    server_.say(message);
		    return;
		}
		catch(COMM_FAILURE ex)
		{
		    registerAgain(ex);
		}
	    }
	}
	else
	{
	    //
	    // Command, execute it
	    //
	    execute(input.substring(1));
	}
    }

    //
    // Execute commands
    //
    void
    execute(String cmds)
    {
        StringTokenizer tokenizer = new StringTokenizer(cmds, " \n");

	while(tokenizer.hasMoreTokens())
	{
	    String token = tokenizer.nextToken();

	    if(token.equals("nick"))
	    {
		if(tokenizer.hasMoreTokens())
		    cmdNick(tokenizer.nextToken());
		else
		    print("*** Nick name is missing");
	    }
	    else if(token.equals("who"))
	    {
		cmdWho();
	    }
	    else if(token.equals("whois"))
	    {
		if(tokenizer.hasMoreTokens())
		    cmdWhois(tokenizer.nextToken());
		else
		    print("*** Nick name is missing");
	    }
	    else if(token.equals("msg"))
	    {
		if(tokenizer.hasMoreTokens())
		{
		    String nick = tokenizer.nextToken();

		    if(tokenizer.hasMoreTokens())
			cmdMsg(nick, tokenizer.nextToken());
		    else
			print("*** Message is missing");
		}
		else
		    print("*** Nick name is missing");
	    }
	    else if(token.equals("about"))
		cmdAbout();
	    else if(token.equals("help"))
		cmdHelp();
	    else if(token.equals("quit"))
		cmdQuit();
	    else
		print("*** Unknown command `" + token + "'");
	}
    }

    //
    // `quit' command
    //
    void
    cmdQuit()
    {
	quit();
    }

    //
    // `about' command
    //
    void
    cmdAbout()
    {
	about();
    }

    //
    // `help' command
    //
    void
    cmdHelp()
    {
	help();
    }

    //
    // `msg' command
    //
    void
    cmdMsg(String nick, String msg)
    {
	while(true)
	{
	    try
	    {
		ReceiverDescriptionHolder desc =
		    new ReceiverDescriptionHolder();
		server_.getReceiverByNick(nick, desc);

		print("-> *" + nick + "* " + msg);

		try
		{
		    desc.value.receiver.print("*" + nick_ + "* " + msg);
		}
		catch(COMM_FAILURE ex)
		{
		    print("*** Communication failure with other client");
		    print("*** Can't send message");
		}

		return;
	    }
	    catch(COMM_FAILURE ex)
	    {
		registerAgain(ex);
	    }
	    catch(UnknownNick ex)
	    {
		unknownNick(nick);
		return;
	    }
	}
    }

    //
    // `nick' command
    //
    void
    cmdNick(String nick)
    {
	try
	{
	    server_.setNickName(this, nick);
	    nick_ = nick;
	    return;
	}
	catch(COMM_FAILURE ex)
	{
	    registerAgain(ex);
	}
	catch(NickExists ex)
	{
	    nickExists(nick);
	    return;
	}
	catch(UnknownReceiver ex)
	{
	    print("*** Warning: client is not known to server");
            print("*** Registering client now");
            register();
	}
    }

    //
    // `who' command
    //
    void
    cmdWho()
    {
	while(true)
	{
	    try
	    {
		String[] names = server_.getReceiverNames();

		for(int i = 0 ; i < names.length ; i++)
		    print(names[i]);

		return;
	    }
	    catch(COMM_FAILURE ex)
	    {
		registerAgain(ex);
	    }
	}
    }

    //
    // `whois' command
    //
    void
    cmdWhois(String nick)
    {
	try
	{
	    ReceiverDescriptionHolder desc = new ReceiverDescriptionHolder();
	    server_.getReceiverByNick(nick, desc);

	    print("*** " + desc.value.nick + " is " + desc.value.id +
		  "@" + desc.value.host);

	    return;
	}
	catch( COMM_FAILURE ex)
	{
	    registerAgain(ex);
	}
	catch(UnknownNick ex)
	{
	    unknownNick(nick);
	    return;
	}
    }

    //
    // Print `about' text
    //
    void
    about()
    {
	print("****************************************************");
	print("");
	print("ORBacus Chat Demo Application");
	print("");
	print("Copyright (c) 1999");
	print("Object Oriented Concepts, Inc.");
	print("Billerica, MA, USA");
	print("");
	print("All Rights Reserved");
	print("");
	print("****************************************************");
    }

    //
    // Print `help' text
    //
    void
    help()
    {
	print("****************************************************");
	print("");
	print("Available commands:");
	print("");
	print("/about             Display `about' info");
	print("/help              Show this help text");
	print("/msg <NAME>        Send personal message to <NAME>");
	print("/nick <NAME>       Change nick name to <NAME>");
	print("/quit              Leave the chat");
	print("/who               List chat participants");
	print("/whois <NAME>      Identify <NAME>");
	print("");
	print("****************************************************");
    }

    //
    // Quit program
    //
    void
    quit()
    {
	try
	{
	    server_.unregister(this);
	}
	catch(UnknownReceiver ex)
	{
	    print("*** Warning: client was already unregistered");
	}
	catch(SystemException ex)
	{
	}

	boa_.deactivate_impl(null);
    }

    //
    // Register with the server
    //
    void
    register()
    {
	while(true)
	{
	    try
	    {
		server_.registerWithCheck(this, id_, host_, nick_);
		window_.setTitle(nick_ + " (" + id_ + "@" + host_ + ")");
		return;
	    }
	    catch(COMM_FAILURE ex)
	    {
		print("*** Can't register with the server");
		print("*** Retry in 1 second...");

		try
		{
		    Thread.currentThread().sleep(1000);
		}
		catch(InterruptedException e)
		{
		}
	    }
	    catch(ReceiverUnreachable ex)
	    {
		print("*** Chat server can't call back to chat client");
		print("*** (Usually caused by a firewall or " +
		      "name server problem)");
		print("*** Retry in 1 second...");

		try
		{
		    Thread.currentThread().sleep(1000);
		}
		catch(InterruptedException e)
		{
		}
	    }
	    catch(NickExists ex)
	    {
		nickExists(nick_);

		nick_ += "_";

		print("*** Trying to use `" + nick_ + "' instead");
	    }
	    catch(AlreadyRegistered ex)
	    {
		print("*** Warning: client was already registered");
		print("*** Warning: unregistering and registering client now");

		try
		{
		    server_.unregister(this);
		}
		catch(UnknownReceiver e)
		{
		}
	    }
	}
    }

    //
    // Print message
    //
    public void
    print(String text)
    {
	window_.print(text);
    }

    //
    // Handle common errors
    //

    void
    registerAgain(COMM_FAILURE ex)
    {
	print("*** Communication failure with the server");
	print("*** Trying to register with the server again in 1 second...");

	try
	{
	    Thread.currentThread().sleep(1000);
	}
	catch(InterruptedException e)
	{
	}

	register();
    }

    void
    unknownNick(String nick)
    {
	print("*** Unknown nick name `" + nick + "'");
    }

    void
    nickExists(String nick)
    {
	print("*** Nick name `" + nick + "' already exists");
    }
}
