// **********************************************************************
//
// Copyright (c) 1999
// Object Oriented Concepts, Inc.
// Billerica, MA, USA
//
// All Rights Reserved
//
// **********************************************************************

package com.ooc.CORBA;

final class GIOPClient extends Client
{
    //
    // The connector
    //
    private com.ooc.OCI.Connector connector_;

    //
    // The worker
    //
    private GIOPClientWorker worker_;
    private Object workerMutex_ = new Object();

    //
    // Add message header and request header
    //
    private int
    addHeaders(Delegate p, String op, int reqId, boolean ow, OutputStream out)
    {
	org.omg.GIOP.MessageHeader hdr = new org.omg.GIOP.MessageHeader();
	hdr.magic = new char[4];
	hdr.magic[0] = 'G';
	hdr.magic[1] = 'I';
	hdr.magic[2] = 'O';
	hdr.magic[3] = 'P';
	hdr.version = new org.omg.GIOP.Version((byte)1, (byte)0);
	hdr.flag = false ? (byte)1 : (byte)0; // false is big endian
	hdr.message_type = (byte)org.omg.GIOP.MsgType._Request;
	hdr.message_size = out.count_ - 12;

        // If key is not cached, extract from IOR
	if(p._OB_key() == null || p._OB_key().length == 0)
	{
	    byte[] k = connector_.is_usable(p._OB_ior());
	    p._OB_key(k);
	    if(p._OB_key().length == 0)
		throw new InternalError();
	}

	org.omg.GIOP.RequestHeader reqH = new org.omg.GIOP.RequestHeader();
	reqH.service_context = new org.omg.IOP.ServiceContext[0];
	reqH.request_id = reqId;
	reqH.response_expected = !ow;
	reqH.reserved = new byte[3];
	reqH.object_key = p._OB_key();
	reqH.operation = op;
	reqH.requesting_principal = new Principal();

	int old = out.count_;
	out.count_ = 0;
	org.omg.GIOP.MessageHeaderHelper.write(out, hdr);
	org.omg.GIOP.RequestHeaderHelper.write(out, reqH);
	out.count_ = old;

	return reqH.request_id;
    }

    //
    // Get the worker
    //
    private GIOPClientWorker
    getWorker(int t)
    {
	synchronized(workerMutex_)
	{
	    if(worker_ == null)
	    {
		//
		// Create new transport, using the connector
		//
		com.ooc.OCI.Transport transport;
		if(t >= 0)
		{
		    transport = connector_.connect_timeout(t);
		    
		    //
		    // Was there a timeout?
		    //
		    if(transport == null)
			throw new org.omg.CORBA.NO_RESPONSE();
		}
		else
		{
		    transport = connector_.connect();
		}
		
		//
		// Create new worker
		//
		switch(ORB.conc_model().value())
		{
		case ORB.ConcModel._ConcModelBlocking:
		    worker_ = new GIOPClientWorkerBlocking(this, transport);
		    break;
		    
		case ORB.ConcModel._ConcModelThreaded:
		    worker_ = new GIOPClientWorkerThreaded(this, transport);
		    break;
		    
		default:
		    throw new InternalError();
		}
	    }
	
	    return worker_;
	}
    }

    //
    // Constructor
    //
    GIOPClient(ORB orb, com.ooc.OCI.Connector connector)
    {
	super(orb);
	//System.out.println("GIOPClient");

	connector_ = connector;
    }

    //
    // Destructor
    //
    protected void
    finalize()
	throws Throwable
    {
	//System.out.println("~GIOPClient");

	//
	// If worker_ is not nil, destroy() was not called
	//
	if(worker_ != null)
	    throw new InternalError();

	super.finalize();
    }

    //
    // Destroy the server
    //
    void
    _destroy()
    {
	//
	// Destroy the worker. Make sure that any further destructions
	// do not block.
	//
	GIOPClientWorker w;
	
	synchronized(workerMutex_)
	{
	    w = worker_;
	    worker_ = null;
	}
	
	if(w != null)
	    w._destroy();
    }

    //
    // Destroy the worker
    //
    void
    destroyWorker(GIOPClientWorker worker)
    {
	if(worker == null)
	    throw new InternalError();

	//
	// Destroy the worker. Make sure that any further destructions
	// do not block.
	//
	GIOPClientWorker w = null;
	
	synchronized(workerMutex_)
	{
	    if(worker_ == worker)
	    {
		w = worker_;
		worker_ = null;
	    }
	}
	
	if(w != null)
	    w._destroy();
    }

    //
    // Get OCI connector
    //
    com.ooc.OCI.Connector
    connector()
    {
	return connector_;
    }
    
    //
    // Get OCI transport
    //
    com.ooc.OCI.Transport
    transport()
    {
	GIOPClientWorker worker = getWorker(-1);
        return worker.transport();
    }

    //
    // Calculate offset
    //
    int
    offset(Delegate p, String op)
    {
        // If key is not cached, extract from IOR
	if(p._OB_key() == null || p._OB_key().length == 0)
	{
	    byte[] k = connector_.is_usable(p._OB_ior());
	    p._OB_key(k);
	    if(p._OB_key().length == 0)
		throw new InternalError();
	}

	OutputStream out = new OutputStream(1024);
	out.count_ = 12;
	org.omg.GIOP.RequestHeader reqH = new org.omg.GIOP.RequestHeader();
	reqH.service_context = new org.omg.IOP.ServiceContext[0];
	reqH.request_id = 0;
	reqH.response_expected = true;
	reqH.reserved = new byte[3];
	reqH.object_key = p._OB_key();
	reqH.operation = op;
	reqH.requesting_principal = new Principal();
	org.omg.GIOP.RequestHeaderHelper.write(out, reqH);

	return out.count_;
    }

    //
    // Send request and receive response
    // With explicit timeout (throws exception on timeout)
    // With optional retry on communication failure
    //
    InputStream
    request(Delegate p, String op, OutputStream out,
	    org.omg.CORBA.BooleanHolder exc,
	    org.omg.CORBA.BooleanHolder fwd,
	    int t, boolean retry)
    {
	org.omg.GIOP.ReplyStatusTypeHolder status =
	    new org.omg.GIOP.ReplyStatusTypeHolder();
	InputStream in;

	while(true)
	{
	    //
	    // Get the worker
	    //
	    GIOPClientWorker worker = getWorker(t);
	    
	    //
	    // Add headers
	    //
	    int reqId = worker.getRequestId();
	    addHeaders(p, op, reqId, false, out);
	    
	    try
	    {
		//
		// Send request and receive reply, with timeout
		//
		in = worker.sendReceiveTimeout(reqId, out, status, t);
	    }
	    catch(org.omg.CORBA.COMM_FAILURE ex)
	    {
		//
		// Retry once on communication failure if the retry
		// parameter is set to true
		//
		if(retry && out.count_ > 0)
		{
		    retry = false;
		    continue;
		}
		
		throw ex;
	    }
	    catch(org.omg.CORBA.TRANSIENT ex)
	    {
		//
		// Retry on transient failure
		//
		if(out.count_ > 0)
		    continue;

		throw ex;
	    }
	    
	    //
	    // Was there a timeout?
	    //
	    if(in == null)
	    {
		if(t < 0)
		    throw new InternalError();
		
		//
		// Mark the request ID as discarded and throw a timeout
		// exception
		//
		worker.discardRequest(reqId);
		throw new org.omg.CORBA.NO_RESPONSE();
	    }

	    break;
	}
	
	//
	// Evaluate reply
	//
	switch(status.value.value())
	{
	case org.omg.GIOP.ReplyStatusType._NO_EXCEPTION:
	    exc.value = false;
	    fwd.value = false;
	    break;
	    
	case org.omg.GIOP.ReplyStatusType._USER_EXCEPTION:
	    exc.value = true;
	    fwd.value = false;
	    break;
	    
	case org.omg.GIOP.ReplyStatusType._SYSTEM_EXCEPTION:
	    Util.unmarshalSystemExceptionThrow(in);
	    break;
	    
	case org.omg.GIOP.ReplyStatusType._LOCATION_FORWARD:
	    exc.value = false;
	    fwd.value = true;
	    break;
	    
	default:
	    throw new InternalError();
	}

	return in;
    }

    //
    // Send request oneway
    // With implicit zero timeout (does not throw exception on timeout)
    // With optional retry on communication failure
    //
    void
    oneway(Delegate p, String op, OutputStream out, boolean retry)
    {
	while(true)
	{
	    //
	    // Get the worker
	    //
	    // TODO: The -1 timeout argument is wrong! What really needs
	    // to be done is to use a 0 timeout here. But then the
	    // connect() in getWorker() will fail in (nearly) all cases,
	    // meaning that all oneways to servers to which the client has
	    // not yet connected will be discarded! The right solution is
	    // to decouple the buffers from the worker, so that the oneway
	    // can be put into the send buffers, even if no worker exits.
	    //
	    // The decoupling of worker and buffers has to be done anyway,
	    // in order to more easily support bi-directional GIOP. That
	    // is, it should be possible that both, a GIOPClientWorker and
	    // a GIOPServerWorker have access to the same buffers.
	    //
	    GIOPClientWorker worker = getWorker(-1);
	    
	    //
	    // Add headers
	    //
	    int reqId = worker.getRequestId();
	    addHeaders(p, op, reqId, true, out);
	    
	    try
	    {
		//
		// Send request non-blocking
		//
		worker.sendNonBlocking(out);
	    }
	    catch(org.omg.CORBA.COMM_FAILURE ex)
	    {
		//
		// Retry once on communication failure if the retry
		// parameter is set to true
		//
		if(retry && out.count_ > 0)
		{
		    retry = false;
		    continue;
		}
		
		throw ex;
	    }
	    catch(org.omg.CORBA.TRANSIENT ex)
	    {
		//
		// Retry on transient failure
		//
		if(out.count_ > 0)
		    continue;

		throw ex;
	    }

	    break;
	}
    }

    //
    // Send request deferred
    // With implicit zero timeout (does not throw exception on timeout)
    // With optional retry on communication failure
    //
    void
    deferred(Delegate p, String op, OutputStream out,
	     org.omg.CORBA.IntHolder reqId, boolean retry)
    {
	while(true)
	{
	    //
	    // Get the worker
	    //
	    // TODO: The -1 argument is wrong here. See the comments in
	    // "oneway" above.
	    //
	    GIOPClientWorker worker = getWorker(-1);
	    
	    //
	    // Add headers
	    //
	    reqId.value = worker.getRequestId();
	    addHeaders(p, op, reqId.value, false, out);

	    try
	    {
		//
		// Send request non-blocking
		//
		// This should really be sendTimeout, but I don't have such an
		// operation right now.
		//
		worker.sendNonBlocking(out);
	    }
	    catch(org.omg.CORBA.COMM_FAILURE ex)
	    {
		//
		// Retry once on communication failure if the retry
		// parameter is set to true
		//
		if(retry && out.count_ > 0)
		{
		    retry = false;
		    continue;
		}
		
		throw ex;
	    }
	    catch(org.omg.CORBA.TRANSIENT ex)
	    {
		//
		// Retry on transient failure
		//
		if(out.count_ > 0)
		    continue;

		throw ex;
	    }

	    break;
	}
    }

    //
    // Get response
    // With explicit timeout (throws exception on timeout)
    //
    InputStream
    response(int reqId,
	     org.omg.CORBA.BooleanHolder exc, org.omg.CORBA.BooleanHolder fwd,
	     int t)
    {
	org.omg.GIOP.ReplyStatusTypeHolder status =
	    new org.omg.GIOP.ReplyStatusTypeHolder();
	InputStream in;

	while(true)
	{
	    //
	    // Get the worker
	    //
	    GIOPClientWorker worker = getWorker(t);
	    
	    try
	    {
		//
		// Receive reply, with timout
		//
		in = worker.receiveTimeout(reqId, status, t);
	    }
	    catch(org.omg.CORBA.TRANSIENT ex)
	    {
		//
		// Retry on transient failure
		//
		continue;
	    }
	    
	    //
	    // Was there a timeout?
	    //
	    if(in == null)
	    {
		if(t < 0)
		    throw new InternalError();
		
		//
		// Mark the request ID as discarded and throw a timeout
		// exception
		//
		worker.discardRequest(reqId);
		throw new org.omg.CORBA.NO_RESPONSE();
	    }

	    break;
	}
	
	//
	// Evaluate reply
	//
	switch(status.value.value())
	{
	case org.omg.GIOP.ReplyStatusType._NO_EXCEPTION:
	    exc.value = false;
	    fwd.value = false;
	    break;
	    
	case org.omg.GIOP.ReplyStatusType._USER_EXCEPTION:
	    exc.value = true;
	    fwd.value = false;
	    break;
	    
	case org.omg.GIOP.ReplyStatusType._SYSTEM_EXCEPTION:
	    Util.unmarshalSystemExceptionThrow(in);
	    break;
	    
	case org.omg.GIOP.ReplyStatusType._LOCATION_FORWARD:
	    exc.value = false;
	    fwd.value = true;
	    break;
	    
	default:
	    throw new InternalError();
	}

	return in;
    }

    //
    // Poll response
    // With implicit zero timeout (returns zero on timeout)
    //
    InputStream
    poll(int reqId,
	 org.omg.CORBA.BooleanHolder exc, org.omg.CORBA.BooleanHolder fwd)
    {
	org.omg.GIOP.ReplyStatusTypeHolder status =
	    new org.omg.GIOP.ReplyStatusTypeHolder();
	InputStream in;

	while(true)
	{
	    //
	    // Get the worker
	    //
	    // TODO: The -1 argument is wrong here. See the comments in
	    // "oneway" above.
	    //
	    GIOPClientWorker worker = getWorker(-1);

	    try
	    {
		//
		// Receive reply, with timeout
		//
		in = worker.receiveTimeout(reqId, status, 0);
	    }
	    catch(org.omg.CORBA.TRANSIENT ex)
	    {
		//
		// Retry on transient failure
		//
		continue;
	    }
	    
	    //
	    // Was there a timeout?
	    //
	    if(in == null)
		return null;

	    break;
	}
	    
	//
	// Evaluate reply
	//
	switch(status.value.value())
	{
	case org.omg.GIOP.ReplyStatusType._NO_EXCEPTION:
	    exc.value = false;
	    fwd.value = false;
	    break;
	    
	case org.omg.GIOP.ReplyStatusType._USER_EXCEPTION:
	    exc.value = true;
	    fwd.value = false;
	    break;
	    
	case org.omg.GIOP.ReplyStatusType._SYSTEM_EXCEPTION:
	    Util.unmarshalSystemExceptionThrow(in);
	    break;
	    
	case org.omg.GIOP.ReplyStatusType._LOCATION_FORWARD:
	    exc.value = false;
	    fwd.value = true;
	    break;
	    
	default:
	    throw new InternalError();
	}

	return in;
    }
}
