// **********************************************************************
//
// Copyright (c) 1999
// Object Oriented Concepts, Inc.
// Billerica, MA, USA
//
// All Rights Reserved
//
// **********************************************************************

package com.ooc.CORBA;

final class GIOPClientWorkerThreaded extends GIOPClientWorker
{
    private boolean destroy_; // True if destroy() was called
    private org.omg.CORBA.SystemException exception_; // The buffered exception

    //
    // The sender thread
    //
    private final class SenderThread extends Thread
    {
        SenderThread()
        {
            super("ORBACUS:Client:SenderThread");
        }
        
	public void run()
	{
	    senderRun();

	    synchronized(GIOPClientWorkerThreaded.this)
	    {
		senderThread_ = null;
		GIOPClientWorkerThreaded.this.notifyAll();
	    }
	}
    };
    Thread senderThread_;

    //
    // The receiver thread
    //
    private final class ReceiverThread extends Thread
    {
        ReceiverThread()
        {
            super("ORBACUS:Client:ReceiverThread");
        }
        
	public void run()
	{
	    receiverRun();

	    synchronized(GIOPClientWorkerThreaded.this)
	    {
		receiverThread_ = null;
		GIOPClientWorkerThreaded.this.notifyAll();
	    }
	}
    };
    Thread receiverThread_;

    //
    // Add outgoing message
    //
    protected synchronized void
    add(com.ooc.OCI.Buffer buf)
    {
	if(!destroy_) // Don't add if already destroyed
	{
	    super.add(buf);

	    //
	    // Notify the sender thread. This is only necessary if
	    // this is the first outgoing message, since if there is
	    // more then one message in the buffers, threads are
	    // already notified.
	    //
	    if(outVec_.size() == 1)
		notifyAll();
	}
    }

    //
    // Remove outgoing message
    //
    protected synchronized com.ooc.OCI.Buffer
    remove()
    {
	//
	// Wait for a message
	//
	while(!destroy_ && outVec_.size() == 0)
	{
	    try
	    {
		wait();
	    }
	    catch(InterruptedException ex)
	    {
	    }
	}
	
	//
	// Return nil buffer if destroy_ was set
	//
	if(destroy_)
	    return null;

	//
	// Remove and return first message from message buffers
	//
	if(outVec_.size() == 0)
	    throw new InternalError();

	com.ooc.OCI.Buffer buf =
	    (com.ooc.OCI.Buffer)outVec_.firstElement();

	outVec_.removeElementAt(0);

	return buf;
    }

    //
    // Constructor
    //
    GIOPClientWorkerThreaded(GIOPClient client,
			     com.ooc.OCI.Transport transport)
    {
	super(client, transport);

	//
	// Start sender and receiver thread
	//
	senderThread_ = new SenderThread();
	senderThread_.start();
	receiverThread_ = new ReceiverThread();
	receiverThread_.start();
    }

    //
    // Destructor
    //
    protected void
    finalize()
	throws Throwable
    {
	if(!destroy_)
	    throw new InternalError();

	super.finalize();
    }

    //
    // Destroy the worker
    //
    synchronized void
    _destroy()
    {
	//
	// Don't destroy twice
	//
	if(destroy_)
	    return;
	
	//
	// Set the destroy flag
	//
	destroy_ = true;
	
	//
	// Notify the sender and the receiver thread
	//
	notifyAll();
	
	//
	// Wait for the sender thread to terminate
	//
	if(senderThread_ != null &&
	   !senderThread_.equals(Thread.currentThread()))
	{
	    while(senderThread_ != null)
	    {
		try
		{
		    wait();
		}
		catch(InterruptedException ex)
		{
		}
	    }
	}
	
	//
	// Unblock threads blocking in the send() and recv()
	// operations
	//
	transport_.unblock_threads(true);
	
	//
	// Wait for the receiver thread to terminate
	//
	if(receiverThread_ != null &&
	   !receiverThread_.equals(Thread.currentThread()))
	{
	    while(receiverThread_ != null)
	    {
		try
		{
		    wait();
		}
		catch(InterruptedException ex)
		{
		}
	    }
	}
	
	//
	// Close the transport
	//
	closeTransport();
    }

    //
    // Send a request and receive a reply, with timeout
    //
    synchronized InputStream
    sendReceiveTimeout(int reqId, OutputStream out,
		       org.omg.GIOP.ReplyStatusTypeHolder status, int t)
    {
	//
	// Check whether there is an exception
	//
	if(exception_ != null)
	{
	    //
	    // Destroy the worker and throw
	    //
	    client_.destroyWorker(this);
	    throw exception_;
	}

	//
	// Convert output stream to OCI buffer
	// Add the OCI buffer
	//
	com.ooc.OCI.Buffer b = new com.ooc.OCI.Buffer();
	b.data(out.buf_, out.count_);
	add(b);

	//
	// From this point on, the call may not be retried, since this
	// could violate "at most once" semantics. Therefore the
	// output stream is reset.
	//
	out.count_ = 0;

	//
	// Return the reply
	//
	return receiveTimeout(reqId, status, t);
    }

    //
    // Receive the reply
    //
    InputStream
    receiveTimeout(int reqId, org.omg.GIOP.ReplyStatusTypeHolder status, int t)
    {
	//
	// Yield if zero timeout
	//
	if(t == 0)
	    Thread.yield();

	synchronized(this)
	{
	    InputStream in = null;

	    if(t < 0)
	    {
		//
		// Wait for reply or exception, without timeout
		//
		while(exception_ == null &&
		      (in = findReply(reqId, status)) == null)
		{
		    try
		    {
			wait();
		    }
		    catch(InterruptedException ex)
		    {
		    }
		}
	    }
	    else if(t == 0)
	    {
		//
		// Zero timeout, don't wait for reply
		//
		in = findReply(reqId, status);

		if(in == null)
		    return null; // Timeout
	    }
	    else if(t > 0)
	    {
		//
		// Get current time
		//
		long t1 = System.currentTimeMillis();
		int millis = 0;

		//
		// Wait for reply or exception, with timeout
		//
		while(exception_ == null &&
		      (in = findReply(reqId, status)) == null)
		{
		    try
		    {
			wait(t);
		    }
		    catch(InterruptedException ex)
		    {
		    }

		    //
		    // Calculate time needed
		    //
		    millis = (int)(System.currentTimeMillis() - t1);

		    //
		    // Check if time limit has been exceeded
		    //
		    if(millis >= t)
			return null; // Timeout
		}
	    }

	    //
	    // Check whether there is an exception
	    //
	    if(exception_ != null)
	    {
		//
		// Destroy the worker and throw
		//
		client_.destroyWorker(this);
		throw exception_;
	    }

	    if(in == null)
		throw new InternalError();

	    return in;
	}
    }

    //
    // Send a request, non-blocking
    //
    synchronized void
    sendNonBlocking(OutputStream out)
    {
	//
	// Check whether there is an exception
	//
	if(exception_ != null)
	{
	    //
	    // Destroy the worker and throw
	    //
	    client_.destroyWorker(this);
	    throw exception_;
	}

	//
	// Convert output stream to OCI buffer
	// Add the OCI buffer
	//
	com.ooc.OCI.Buffer b = new com.ooc.OCI.Buffer();
	b.data(out.buf_, out.count_);
	add(b);

	//
	// Nothing else to do here
	//
    }

    //
    // Run method for sender thread
    //
    public void
    senderRun()
    {
	while(!destroy_)
	{
	    try
	    {
		//
		// Remove outgoing message from the outgoing message buffers
		//
		com.ooc.OCI.Buffer buf = remove();
		
		//
		// Check if thread should be stopped and destroyed
		//
		if(buf == null)
		    break;
		
		//
		// Send buffer, blocking
		//
		transport_.send(buf, true);
		if(!buf.is_full())
		    throw new InternalError();
	    }
	    catch(org.omg.CORBA.SystemException ex)
	    {
		synchronized(this)
		{
		    //
		    // It is possible that the sender thread has
		    // already set an exception
		    //
		    if(exception_ == null)
			exception_ = ex;
		    
		    notifyAll();
		    
		    //
		    // The worker may *not* be removed from the client,
		    // otherwise the client will not detect the
		    // communication failure.
		    //
		    //client_ -> destroyWorker(this);
		    _destroy();
		}

		return;
	    }
	}

	//
	// Send all remaining messages, ignoring any errors
	//
	sendRemainingMessages();
    }

    //
    // Run method for receiver thread
    //
    public void
    receiverRun()
    {
	while(!destroy_)
	{
	    try
	    {
		//
		// Setup the incoming message buffer
		//
		if(buf_ != null)
		    throw new InternalError();
		setupBuffer();
		
		//
		// Receive header, blocking
		//
		transport_.receive(buf_, true);
		if(!buf_.is_full())
		    throw new InternalError();
		
		//
		// Check if thread should be stopped and destroyed
		//
		if(destroy_)
		    break;

		//
		// Header is complete
		//
		extractHeader();
		
		if(!buf_.is_full())
		{
		    //
		    // Receive body, blocking
		    //
		    transport_.receive(buf_, true);
		    if(!buf_.is_full())
			throw new InternalError();
		}
		
		//
		// Execute the message
		//
		synchronized(this)
		{
		    if(!destroy_)
		    {
			execute();
			notifyAll();
		    }
		}
	    }
	    catch(org.omg.CORBA.SystemException ex)
	    {
		synchronized(this)
		{
		    //
		    // It is possible that the sender thread has
		    // already set an exception
		    //
		    if(exception_ == null)
			exception_ = ex;
		    
		    notifyAll();

		    //
		    // The worker may *not* be removed from the client,
		    // otherwise the client will not detect the
		    // communication failure.
		    //
		    //client_ -> destroyWorker(this);
		    _destroy();
		}

		return;
	    }
	}
    }
}
