// **********************************************************************
//
// Copyright (c) 1999
// Object Oriented Concepts, Inc.
// Billerica, MA, USA
//
// All Rights Reserved
//
// **********************************************************************

package com.ooc.CORBA;

//
// This class encapsulates the some of the various INS
// object-reference encodings.
//
class INS
{
    //
    // Exception thrown in the case of an INSURL error.
    //
    static private class MalformedINSURLException extends Exception
    {
	int minor_;

	MalformedINSURLException(int minor, String msg)
	{
	    super(msg);
	    minor_ = minor;
	}

	int getMinor()
	{
	    return minor_;
	}
    }

    //
    // This class represents a INS address.
    //
    static private class INSAddress
    {
	int major;
	int minor;
	String host;
	int port;
    };

    //
    // This class represents an INS URL. This is a URL of the form
    // "protocol://addresses/stringified_name".
    //
    static private class INSURL
    {
	private String protocol_; // The protocol name.

	private INSAddress[] addresses_; // The parsed INS addresses.

	private byte[] file_; // The parsed file portion of the INS URL.

	//
	// Take a string of the form:
	//
	// <addr_list> = [<address> ,]* <address>
	// <address> = [<version> <host> [: <port>]]
	// <host> = DNS Style Host Name | ip_address
	// <version> = <major> . <minor> @ | empty_string
	// <port> = number
	// <major> = number
	// <minor> = number
	// <string_name>= stringified Name | empty_string
	//
	// and produce an array of INSAddress classes.
	//
	private void
	parseAddresses(String address)
	    throws MalformedINSURLException
	{
	    int numBodies = 0;

	    //
	    // Count the number of addresses.
	    //
	    int comma = 0;
	    do
	    {
		comma = address.indexOf(',', comma);
		//
		// If we're not at the end, skip the ',' character.
		//
		if(comma != -1)
		    ++comma;

		numBodies++;
	    }
	    while(comma != -1);
	
	    //
	    // Allocate space for the bodies.
	    //
	    addresses_ = new INSAddress[numBodies];

	    //
	    // Reset numBodies and comma.
	    //
	    numBodies = 0;
	    comma = 0;
	    do
	    {
		int start = comma;

		//
		// Find the next comma.
		//
		comma = address.indexOf(',', start);

		//
		// This is the end of the address portion. Its either the
		// end of the string, or the character before the ','.
		//
		int end = comma;
		if(end == -1)
		    end = address.length();

		//
		// If we're not at the end, skip the ',' character.
		//
		if(comma != -1)
		    ++comma;

		//
		// Initialize the defaults for the address.
		//
		String host = "localhost";
		int port = 9999;
		int major = 1;
		int minor = 0;

		int addrstart = start;

		//
		// Do we have an '@' portion?
		//
		int at = address.indexOf('@', start);
		if(at != -1 && at <= end)
		{
		    int dot = address.indexOf('.', start);
		    if(dot == -1 || dot > end)
			throw new MalformedINSURLException(
			    MinorBadParam._BadAddress,
			    "Bad version number");
		    try
		    {
			String m = address.substring(start, dot);
			major = Integer.parseInt(m);
			m = address.substring(dot + 1, at);
			minor = Integer.parseInt(m);
		    }
		    catch(NumberFormatException e)
		    {
			throw new MalformedINSURLException(
			    MinorBadParam._BadAddress,
			    "Bad version number");
		    }

                    //
                    // ORBacus 3.x can only create IIOP 1.0 IORs
                    //
                    if(major != 1 || minor != 0)
			throw new MalformedINSURLException(
			    MinorBadParam._BadAddress,
			    "Bad version number");

		    //
		    // The end of the address portion is now the character
		    // after the '@'.
		    //
		    addrstart = at + 1;
		}

		//
		// Find ':' separator.
		//
		int colon = address.indexOf(':', addrstart);

		//
		// No colon? It's only the host. Otherwise both host and
		// port are specified.
		//
		if(colon == -1 || colon > end)
		{
		    if(addrstart != end)
			host = address.substring(addrstart, end);
		}
		else
		{
		    if(colon != addrstart)
			host = address.substring(addrstart, colon);
		    String portString = address.substring(colon + 1, end);
		    try
		    {
			port = Integer.parseInt(portString);
		    }
		    catch(NumberFormatException e)
		    {
			throw new MalformedINSURLException(
			    MinorBadParam._BadAddress,
			    "Bad port: " + portString);
		    }
		}

		//
		// Copy in the address information.
		//
		addresses_[numBodies] = new INSAddress();
		addresses_[numBodies].major = major;
		addresses_[numBodies].minor = minor;
		addresses_[numBodies].host = host;
		addresses_[numBodies].port = port;
		++numBodies;
	    }
	    while(comma != -1);
	}

	//
	// Parse the character escapes. A '%' character is used as an
	// escape.  Two hexadceimal characters follow a '%' -- the first
	// is the high order nibble and the second is the low-order
	// nibble. If a '%' is not followed by two hex digits the name is
	// synactically invalid.
	//
	private void
	parseFile(String file)
	    throws MalformedINSURLException
	{
	    try
	    {
		String out = new String();
		int len = file.length();
		for(int curr = 0; curr < len; curr++)
		{
		    char ch = file.charAt(curr);
		    if(ch == '%')
		    {
			char c1 = file.charAt(++curr);
			char c2 = file.charAt(++curr);
			ch = (char)((Character.digit(c1, 16) << 4) |
				    Character.digit(c2, 16));
		    }
		    out += ch;
		}

		//
		// Don't use getBytes because this can cause unwanted
		// conversions.
		//
		file_ = new byte[out.length()];
		for(int i = 0; i < file_.length; i++)
		{
		    if(out.charAt(i) > 255)
			throw new InternalError();
		    file_[i] = (byte)out.charAt(i);
		}
	    }
	    catch(StringIndexOutOfBoundsException e)
	    {
		throw new MalformedINSURLException(
			    MinorBadParam._BadSchemeSpecificPart,
			    "Bad octet duple");
	    }
	}

	//
	// Construct a new parsed INS URL.
	//
	INSURL(String s)
	    throws MalformedINSURLException
	{
	    int start = s.indexOf("://");
	    if(start == -1)
		throw new MalformedINSURLException(MinorBadParam._Other,
						   "No protocol");
	    protocol_ = s.substring(0, start).toLowerCase();

	    //
	    // Skip the  "://"
	    //
	    start += 3;
	    int end = s.indexOf('/', start);
	    if(end == -1)
		throw new MalformedINSURLException(MinorBadParam._Other,
						   "No file");

	    //
	    // Parse the addresses and the file portion of the INS
	    // URL.
	    //
	    parseAddresses(s.substring(start, end));
	    parseFile(s.substring(end+1));
	}

	INSURL(String protocol, INSAddress[] addrs, byte[] file)
	{
	    protocol_ = protocol;
	    addresses_ = addrs;
	    file_ = file;
	}

	//
	// Get the protocol.
	//
	String getProtocol()
	{
	    return protocol_;
	}

	//
	// Get the parsed addresses.
	//
	INSAddress[] getAddresses()
	{
	    return addresses_;
	}

	//
	// Get the file portion.
	//
	byte[] getFile()
	{
	    return file_;
	}
    }

    //
    // Do an IIOPLOC resolve.
    //
    static private org.omg.CORBA.Object
    IIOPLOCToObject(ORB orb, INSURL url)
    {
	INSAddress[] addrs = url.getAddresses();
	byte[] key = url.getFile();

        //
        // Although no object-key is allowed by the INS specification
        // we're going to make this an invalid IOR.
        //
        if(key.length == 0)
            throw new org.omg.CORBA.BAD_PARAM(
                "", MinorBadParam._Other,
                org.omg.CORBA.CompletionStatus.COMPLETED_NO);

	//
	// We only use the first address in the list; all others are ignored
	//
        org.omg.IOP.IOR ior = com.ooc.OCI.IIOP.impl.Util.createIOR(
            addrs[0].host, addrs[0].port, key, "");
	return orb._OB_createObject(ior);
    }

    //
    // Do an IIOPNAME resolve.
    //
    static private org.omg.CORBA.Object
    IIOPNAMEToObject(ORB orb, INSURL url)
    {
	//
	// Object-key of the root naming context. See 98-10-11 3-5 for
	// details. Unfortunately there is a ambiguity in the
	// specification.  On pg. 4-20 the key is identified as
	// "NamingService". Since "NameService" is used first, and
	// more than once in the specification we use this.
	//
	final String nameKey = "NameService";

	//
	// Create the object-key from the string. We don't use the
	// getBytes() method since this may cause unwanted
	// conversions.
	//
	// TODO: Is this correct on non US ASCII systems -- should we
	// code the key directly?
	//
	byte key[] = new byte[nameKey.length()];
	for(int i = 0; i < nameKey.length(); i++)
	{
	    if(nameKey.charAt(i) > 255)
		throw new InternalError();
	    key[i] = (byte)nameKey.charAt(i);
	}

	//
	// Do an IIOPLOC resolve of this key.
	//
	INSURL nUrl = new INSURL("iioploc", url.getAddresses(), key);
	
	org.omg.CORBA.Object service = IIOPLOCToObject(orb, nUrl);

	byte[] file = url.getFile();

	//
	// If we don't have a specification then we return the naming
	// context as specified on 4-18:
	//
	// iiopname:///
	//
	// This URL repesents the naming context returned by the agent
	// running on the local host at the default port. It is
	// equivalent to iioploc:///NameService.
	//
	if(file.length == 0)
	    return service;

	try
	{
	    //
	    // Create typecodes for Name and NameComponent.
	    //
	    org.omg.CORBA.StructMember[] contents =
		new org.omg.CORBA.StructMember[2];

	    contents[0] = new org.omg.CORBA.StructMember();
	    contents[0].name = "id";
	    contents[0].type = orb.create_string_tc(0);

	    contents[1] = new org.omg.CORBA.StructMember();
	    contents[1].name = "kind";
	    contents[1].type = orb.create_string_tc(0);

	    org.omg.CORBA.TypeCode tcNameComponent =
	        orb.create_struct_tc("IDL:omg.org/CosNaming/NameComponent:1.0",
				     "NameComponent", contents);

	    org.omg.CORBA.TypeCode tcName = 
		orb.create_sequence_tc(0, tcNameComponent);

	    //
	    // Parse path, create NameComponent sequence.
	    //
	    IIOPNameParser parser = new IIOPNameParser(file);
	    if(!parser.isValid())
		throw new org.omg.CORBA.BAD_PARAM(
		    "", MinorBadParam._BadSchemeSpecificPart,
		    org.omg.CORBA.CompletionStatus.COMPLETED_NO);

	    String[] content = parser.getContents();

	    org.omg.CORBA.Any[] as =
		new org.omg.CORBA.Any[content.length / 2];
	    for(int i = 0; i < content.length; i += 2)
	    {
		//
		// Create the DynStruct containing the id and kind
		// fields.
		//
		org.omg.CORBA.DynStruct name =
		    orb.create_dyn_struct(tcNameComponent);
		name.insert_string(content[i]);
		name.insert_string(content[i + 1]);
		as[i / 2] = name.to_any();
	    }

	    //
	    // Create the DynSequence representing the name.
	    //
	    org.omg.CORBA.DynSequence seq = orb.create_dyn_sequence(tcName);
	    seq.length(as.length);
	    seq.set_elements(as);

	    //
	    // Create the DII Request to the naming service to resolve
	    // the name.
	    //
	    org.omg.CORBA.Request request = service._request("resolve");

	    //
	    // Copy in the arguments.
	    //
	    org.omg.CORBA.Any any = request.add_in_arg();
	    org.omg.CORBA.Any seqany = seq.to_any();
	    any.read_value(seqany.create_input_stream(), seqany.type());
	    seq.destroy();

	    request.set_return_type(orb.get_primitive_tc(
		org.omg.CORBA.TCKind.tk_objref));
	
            try
            {
                //
                // Invoke the request.
                //
                request.invoke();

                //
                // Return the result if there was no exception
                //
                if(request.env().exception() == null)
                    return request.return_value().extract_Object();
            }
            catch(org.omg.CORBA.SystemException ex)
            {
                // Fall through
            }

            throw new org.omg.CORBA.BAD_PARAM(
                "", MinorBadParam._Other,
                org.omg.CORBA.CompletionStatus.COMPLETED_NO);
	}
	//
	// All of the following exceptions denote some sort of
	// internal error.
	//
	catch(org.omg.CORBA.ORBPackage.InconsistentTypeCode e)
	{
	    throw new InternalError();
	}
	catch(org.omg.CORBA.DynAnyPackage.InvalidValue e)
	{
	    throw new InternalError();
	}
	catch(org.omg.CORBA.DynAnyPackage.Invalid e)
	{
	    throw new InternalError();
	}
	catch(org.omg.CORBA.DynAnyPackage.InvalidSeq e)
	{
	    throw new InternalError();
	}
	catch(org.omg.CORBA.BAD_OPERATION e)
	{
	    throw new InternalError();
	}
    }

    //
    // Convert an IOR: IOP object-reference to an Object.
    //
    static private org.omg.CORBA.Object
    IORStringToObject(ORB orb, String s)
    {
	byte[] buf = Util.asciiToOctets(s);
	if(buf == null)
	    throw new org.omg.CORBA.BAD_PARAM(
		"", MinorBadParam._BadSchemeSpecificPart,
		org.omg.CORBA.CompletionStatus.COMPLETED_NO);
	try
	{
	    InputStream in = new InputStream(buf);
	    in._OB_readEndian();
	    return orb._OB_createObject(org.omg.IOP.IORHelper.read(in));
	}
	catch(RuntimeException ex)
	{
	    throw new org.omg.CORBA.BAD_PARAM(
		"", MinorBadParam._BadSchemeSpecificPart,
		org.omg.CORBA.CompletionStatus.COMPLETED_NO);
	}
    }

    //
    // Convert an iiop:// object-URL to an Object.
    //
    static private org.omg.CORBA.Object
    IIOPStringToObject(ORB orb, String s)
    {
	//
	// This avoids the need for a new protocol handler.
	//
	String u = "http:" + s;
	org.omg.IOP.IOR ior;
	try
	{
	    java.net.URL url = new java.net.URL(u);
	    
	    String host = url.getHost();
	    
	    int port = url.getPort();
	    if(port == -1)
		throw new org.omg.CORBA.BAD_PARAM(
		    "", MinorBadParam._BadAddress,
		    org.omg.CORBA.CompletionStatus.COMPLETED_NO);

	    String name = url.getFile().substring(1);

	    return getInetObject(orb, host, port, name, "");
	}
	catch(java.net.MalformedURLException ex)
	{
	    throw new org.omg.CORBA.BAD_PARAM(
		"", MinorBadParam._Other,
		org.omg.CORBA.CompletionStatus.COMPLETED_NO);

	}
    }

    static protected org.omg.CORBA.Object
    stringToObject(ORB orb, String s)
    {
	//
	// Various IOR prefixes.
	//
	final String iorPrefix = "ior:";
	final String iiopPrefix = "iiop:";
	final String iiopnamePrefix = "iiopname";
	final String iioplocPrefix = "iioploc";

	String lowerS = s.toLowerCase();

	if(lowerS.startsWith(iorPrefix))
	    return IORStringToObject(orb, s.substring(iorPrefix.length()));

	if(lowerS.startsWith(iiopPrefix))
	    return IIOPStringToObject(orb, s.substring(iiopPrefix.length()));

	try
	{
	    INSURL url = new INSURL(s);

	    if(url.getProtocol().equals(iiopnamePrefix))
		return IIOPNAMEToObject(orb, url);

	    if(url.getProtocol().equals(iioplocPrefix))
		return IIOPLOCToObject(orb, url);
	}
	catch(MalformedINSURLException e)
	{
	    throw new org.omg.CORBA.BAD_PARAM("", e.getMinor(),
	        org.omg.CORBA.CompletionStatus.COMPLETED_NO);
	}

        throw new org.omg.CORBA.BAD_PARAM("", MinorBadParam._BadSchemeName,
	    org.omg.CORBA.CompletionStatus.COMPLETED_NO);
    }

    static protected org.omg.CORBA.Object
    getInetObject(ORB orb, String host, int port, String name, String id)
    {
	//
	// Makeup object key.
	//
	// I don't use string marshal functions here because I
	// don't want to have a string length encoded. That would
	// cause byte order dependencies.
	//
	byte key[] = new byte[name.length() + 1];
	for(int i = 0 ; i < name.length() ; i++)
	{
	    if(name.charAt(i) > 255)
		throw new org.omg.CORBA.BAD_PARAM(
		    "",  MinorBadParam._BadSchemeSpecificPart,
		    org.omg.CORBA.CompletionStatus.COMPLETED_NO);
	    key[i] = (byte)name.charAt(i);
	}
	key[name.length()] = (byte)0;

	return orb._OB_createObject(com.ooc.OCI.IIOP.impl.Util.
				    createIOR(host, port, key, id));
    }
}
