// $Id: $
//
// $Log: $
// Revision 1.0  1998/03/30  raif
// + original version based on cryptix.tools.InstallKit
//
// $Endlog$
/*
 * Copyright (c) 1997, 1998 Systemics Ltd on behalf of
 * the Cryptix Development Team. All rights reserved.
 */
package NIST;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
//import java.util.Properties;

/**
 * Install the NIST-Kit template sources in a user-specified directory.<p>
 *
 * <b>Copyright</b> &copy; 1998
 * <a href="http://www.systemics.com/">Systemics Ltd</a> on behalf of the
 * <a href="http://www.systemics.com/docs/cryptix/">Cryptix Development Team</a>.
 * <br>All rights reserved.<p>
 *
 * <b>$Revision: $</b>
 * @author  Raif S. Naffah
 */
public final class InstallKit
{
// Constants and variables
//...........................................................................

    private static final String NAME = "InstallKit";

    private static final String NL = System.getProperty("line.separator");
    private static final String VERSION = "$Revision: 1.0$";
    private static final String COPYRIGHT =
        "Copyright (c) 1998 Systemics Ltd. on behalf of" + NL +
        "the Cryptix Development Team.  All rights reserved.";
    private static final String ID =
        "Cryptix NIST-Kit template source code installer" + NL +
        VERSION + NL +
        COPYRIGHT;

    private int exitValue = 0;

    // current values of switches as set from the command line arguments
    private boolean mute = true;       // -v if true become verbose
    private boolean overWrite = false; // -f if true overwrite existing files
    private String srcDirName;         // -s  source directory if != user.dir
    private String destDirName;        // -d  destination directory if != user.dir
    private String algorithm = null;   // candidate algorithm name == provider

    private File srcDir;               // source directory File object
    private File destDir;              // destination directory File object


// main
//...........................................................................

    public static final void main (String[] args) {
        System.out.println(ID + NL + NL);
        InstallKit cmd = new InstallKit();
        cmd.run(args);
    }


// methods
//...........................................................................

    /** main action. */
    private boolean run (String[] args) {
        long time = System.currentTimeMillis();
        if (!processOptions(args))
            return false;
        notify("AES Candidate Algorithm: \""+algorithm+"\"");
        notify("     Template directory: \""+srcDir+"\"");
        notify(" Installation directory: \""+destDir+"\"");
        try {
            destDir = new File(destDir, algorithm);

            // create "classes" and "classes/Foo" directories in destination
            File dir = new File(destDir, "classes");
            dir.mkdirs();
            dir = new File(dir, algorithm);
            dir.mkdirs();

            // process the .properties file
            File props = new File(srcDir, "XXX.properties");
            if (! props.exists()) {
                error("\"XXX.properties\" file not found. Check the -s argument.");
                return false;
            }
            if (!copyReplace(props, dir))
                return false;

            // process the src/*.java files
            srcDir =  new File(srcDir,  "src");
            destDir = new File(destDir, "src");
            String[] files = srcDir.list();
            for (int i = 0; i < files.length; i++)
                if (files[i].startsWith("XXX"))
                    if (!copyReplace(new File(srcDir, files[i]), destDir))
                        return false;
        } catch (Exception x) {
            x.printStackTrace();
            error("Exception encountered: " + x.getMessage());
            return false;
        }
        notify("Java interpreter used: Version "+System.getProperty("java.version"));
        notify("Java Just-In-Time (JIT) compiler: "+System.getProperty("java.compiler"));
        notify("Total execution time (ms): "+(System.currentTimeMillis() - time));
        return true;
    }
    
    /** Process command line arguments and initialise instance fields. */
    private boolean processOptions (String[] args) {
        int argc = args.length;
        if (argc == 0) {
            printHelp();
            return false;
        }
        System.out.println(
            "(type \"java NIST."+NAME+"\" with no arguments for help)"+NL+NL);
        int i = -1;
        String cmd = "";
        boolean next = true;
        while (true) {
            if (next) {
                i++;
                if (i >= argc)
                    break;
                else
                    cmd = args[i];
            } else
                cmd = "-" + cmd.substring(2);

            if (cmd.startsWith("-v")) {
                mute = false;
                next = (cmd.length() == 2);
            } else if (cmd.startsWith("-f")) {
                overWrite = true;
                next = (cmd.length() == 2);
            } else if (cmd.startsWith("-s")) { // source directory
                srcDirName = args[i + 1];
                i++;
                next = true;
            } else if (cmd.startsWith("-d")) { // destination directory
                destDirName = args[i + 1];
                i++;
                next = true;
            } else // it's the algorithm
                algorithm = cmd;
        }
        // fill the blanks
        if (algorithm == null) {
            error("Missing candidate Algorithm name");
            return false;
        }
        if (srcDirName == null)
            srcDirName =  System.getProperty("user.dir");
        if (destDirName == null)
            destDirName = System.getProperty("user.dir");
        srcDir = new File(new File(srcDirName), "XXX");
        if (! srcDir.isDirectory()) {
            error("Source \""+srcDir.getAbsolutePath()+"\" is not a directory");
            return false;
        }
        destDir = new File(destDirName);
        if (! destDir.isDirectory()) {
            error("Destination \""+destDir.getAbsolutePath()+"\" is not a directory");
            return false;
        }
        return true;
    }

    /**
     * Copy a specified file, whose name starts with "XXX" to a designated
     * directory replacing each occurence of the string "XXX", within the
     * contents of the source file with the name of the candidate algorithm.<p>
     *
     * Destination file name is also renamed "algorithm*" instead of "XXX*".
     */
    private boolean copyReplace (File srcFile, File destDir)
    throws FileNotFoundException, IOException {
        String destFileName = algorithm + srcFile.getName().substring(3);
        File destFile = new File(destDir, destFileName);
        if (destFile.exists() && !overWrite) {
            error("File \""+destFile.getAbsolutePath()+"\" already exists");
            return false;
        }
        // make sure everything in the destination path exists
        File parent = new File(destFile.getParent());
        if (parent != null)
            parent.mkdirs();

        // we'll do the copy/replacement to a temp file. later we'll rename it
        // to the designated final destination file
        File tmp = createTempFile("aes", destDir);

        // now read from source into temp replacing XXX's with real name
        BufferedReader in = null;
        try {
            in = new BufferedReader(new FileReader(srcFile));
        } catch (FileNotFoundException x) {
            // very unlikely
        }
        PrintWriter out = new PrintWriter(
                            new BufferedWriter(
                                new FileWriter(tmp)));
        String line;
        while ((line = in.readLine()) != null)
            out.println(replace(line, "XXX", algorithm));
        if (out.checkError())
            notify("Error encountered while processing \"" +
                srcFile.getAbsolutePath() + "\"");
        out.close();
        in.close();

        tmp.renameTo(destFile);

        notify("File \""+srcFile.getAbsolutePath()+"\" processed");
        return true;
    }
    
    /**
     * Method to return a randomly built string, with a designated prefix to
     * be used as the name of a temporary file in a designated directory
     */
    private static File createTempFile (String prefix, File destDir) {
        java.util.Random random = new java.util.Random();
        int x;
        File it;
        do {
            x = (int) (Math.abs(random.nextDouble()) * 1000000);
            it = new File(destDir, prefix + String.valueOf(x));
        } while (it.exists());
        return it;
    }

    /**
     * Return a new String where all occurences of oldToken are replaced
     * with newToken.
     */
    private static String
    replace (String line, String oldToken, String newToken) {
        int limit = line.length();
        int delta = oldToken.length();
        StringBuffer result = new StringBuffer(2 * limit);
        int end;
        int start = 0;
        while (start < limit && (end = line.indexOf(oldToken, start)) != -1) {
            result.append(line.substring(start, end)).append(newToken);
            start = end + delta;
        }
        return start < limit ?
            result.append(line.substring(start)).toString() :
            result.toString();
    }

    /** Print an error message to System.err. */
    private void error (String s) {
        System.err.println(NL + "*** "+s+"...");
        exitValue = -1;
    }

    /** @return The exeucution result code: 0 means OK. */
    public int getExitValue() { return exitValue; }

    /** Write a notification message to System.out. */
    private void notify (String s) {
        if (!mute)
            System.out.println(NAME+": "+s+"...");
    }
    
    /** write help to System.out. */
    void printHelp () {
        System.out.println(
        "NAME" + NL +
        "  "+NAME+": Installs the NIST-Kit template sources for a specified" + NL +
        "  candidate algorithm name." + NL + NL +
        "SYNTAX" + NL +
        "  java NIST."+NAME + NL +
        "    [ -v ]" + NL +
        "    [ -f ]" + NL +
        "    [ -s <source-directory>]" + NL +
        "    [ -d <destination-directory>]" + NL +
        "    <algorithm>" + NL + NL +
        "DESCRIPTION" + NL +
        "  Creates a new directory named after the candidate algorithm and" + NL +
        "  copies the template source files there, renaming them by replacing" + NL +
        "  the 'XXX' prefix with the algorithm's name." + NL +
        "  The contents of these files are also scaned and every occurence of" + NL +
        "  the token 'XXX' is replaced with <algorithm>."+NL+NL+
        "OPTIONS" + NL +
        "  -v   Verbose." + NL + NL +
        "  -f   Overwrite existing files and directoriues if already there." + NL + NL +
        "  -s <source-directory>" + NL +
        "       Pathname where the 'src' directory and the template source" + NL +
        "       files are located (<dir-1>/ in the documentation). If left" + NL +
        "       unspecified, the current user directory is considered." + NL + NL +
        "  -d <destination-directory>" + NL +
        "       Pathname where the new 'src' directory and the properly named" + NL +
        "       blank source files will be installed (<dir-2>/ in the" + NL +
        "       documentation). If left unspecified, the current user directory" + NL +
        "       is considered." + NL + NL +
        "  <algorithm>" + NL +
        "       Candidate algorithm name (<Foo> in the documentation)." + NL + NL +
        COPYRIGHT);
    }
}
