// $Id: $
//
// $Log: $
// Revision 1.0.1  1998/04/16  raif
// + fixed the IV bug.
//
// Revision 1.0  1998/04/08  raif
// + original version.
//
// $Endlog$
/*
 * Copyright (c) 1997, 1998 Systemics Ltd on behalf of
 * the Cryptix Development Team. All rights reserved.
 */
package NIST;

import java.io.PrintWriter;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.security.InvalidKeyException;
import java.security.InvalidParameterException;

//...........................................................................
/**
 * A class to operate any AES candidate algorithm implementation, complying
 * with level-0 API, in 1-bit CFB mode.<p>
 *
 * <b>References:</b><ol>
 *   <li> <a name="HAC">[HAC]</a>
 *      A. J. Menezes, P. C. van Oorschot, S. A. Vanstone,
 *      <cite>Handbook of Applied Cryptography</cite>,
 *      CRC Press 1997, p. 231.<p>
 *   <li> <a name="AC2">[AC2]</a>
 *      <a href="mailto:schneier@counterpane.com">Bruce Schneier</a>,
 *      <cite>Applied Cryptography, 2nd edition</cite>,
 *      John Wiley &amp; Sons, 1996. pp. 200-202.<p>
 *   <li>NIST,
 *      <cite>FIPS PUB 81</cite>, "DES Modes of Operation,"
 *      December 2, 1980.</ol><p>
 *
 * <b>Copyright</b> &copy; 1997, 1998
 * <a href="http://www.systemics.com/">Systemics Ltd</a> on behalf of the
 * <a href="http://www.systemics.com/docs/cryptix/">Cryptix Development Team</a>.
 * <br>All rights reserved.<p>
 *
 * <b>$Revision: $</b>
 * @author  Raif S. Naffah
 */
public final class NIST_CFB
implements NIST_CipherSpi
{
// Debugging methods and variables
//...........................................................................

    static final String NAME = "NIST_CFB";
    static final boolean IN = true, OUT = false;

    static final boolean DEBUG = NIST_Properties.GLOBAL_DEBUG;
    static final int debuglevel = DEBUG ? NIST_Properties.getLevel(NAME) : 0;
    static final PrintWriter err = DEBUG ? NIST_Properties.getOutput() : null;

    static final boolean TRACE = NIST_Properties.isTraceable(NAME);

    static void debug (String s) { err.println(">>> "+NAME+": "+s); }
    static void trace (boolean in, String s) {
        if (TRACE) err.println((in?"==> ":"<== ")+NAME+"."+s);
    }
    static void trace (String s) { if (TRACE) err.println("<=> "+NAME+"."+s); }


// Constants and variables
//...........................................................................

    // transient is used to prevent serialization of sensitive data

    private static final int BLOCK_SIZE = 16; // bytes in an AES block

    private static final char[] HEX_DIGITS = {
        '0','1','2','3','4','5','6','7','8','9','A','B','C','D','E','F'
    };

    private transient Object sessionKey; // current session key

    private transient byte[] iv; // CFB shift register
    protected transient byte[] userIV; // initial user-supplied IV value

    private int state;

    // variables for the Reflection API
    private Method makeKey = null; // the makeKey([B) method
    private Method encrypt = null; // the blockEncrypt([B, int, Object) method
    private Method decrypt = null; // the blockDecrypt([B, int, Object) method
    private Object[] args; // will contain actual arguments of invoked methods

    private String aes; // full name of the underlying _Algorithm class


// Constructor
//...........................................................................

    /**
     * @param algorithm  The distinguished name of the AES candidate
     *      algorithm implementation, assumed to comply with level-0
     *      API within a class called aes_Algorithm in package aes.
     * @exception ClassNotFoundException  If unable to locate the
     *      aes.aes_Algorithm class supposed to implement the level-0 API
     * @exception NoSuchMethodException  If at least one of the three
     *      required methods of the level-0 API with the correct number
     *      of parameters was not found.
     */
    protected NIST_CFB (String aes)
    throws ClassNotFoundException, NoSuchMethodException {
        try { // get references to underlying *_Algorithm methods
            aes = aes + "." + aes + "_Algorithm";
            Class algorithm = Class.forName(aes);
            // inspect the *_Algorithm class
            Method[] methods = algorithm.getDeclaredMethods();
            for (int i = 0, n = methods.length; i < n; i++) {
                String name = methods[i].getName();
                int params = methods[i].getParameterTypes().length;
                if (name.equals("makeKey") && (params == 1))
                    makeKey = methods[i];
                else if (name.equals("blockEncrypt") && (params == 3))
                    encrypt = methods[i];
                else if (name.equals("blockDecrypt") && (params == 3))
                    decrypt = methods[i];
            }
            if (makeKey == null)
                throw new NoSuchMethodException("makeKey()");
            if (encrypt == null)
                throw new NoSuchMethodException("blockEncrypt()");
            if (decrypt == null)
                throw new NoSuchMethodException("blockDecrypt()");
        } catch (ClassNotFoundException x1) {
            debug("Class " + aes + " not found");
            throw (ClassNotFoundException) x1.fillInStackTrace();
        } catch (NoSuchMethodException x2) {
            debug("Method " + aes + "." + x2.getMessage() + " not found");
            throw (NoSuchMethodException) x2.fillInStackTrace();
        }
        this.aes = aes;
        userIV = null;
        iv = new byte[BLOCK_SIZE];
        sessionKey = null;
    }

    /**
     * Throws a CloneNotSupportedException. Cloning of ciphers is not
     * allowed for security reasons.
     *
     * @exception CloneNotSupportedException Not allowed for security reasons.
     */
    public final Object clone() throws CloneNotSupportedException {
        throw new CloneNotSupportedException();
    }


// Implementation of Level-1 API methods
//...........................................................................

    public void init (int state, byte[] key) throws InvalidKeyException {
if (DEBUG) trace(IN, "init()");
        generateKey(key);
        engineInit();
        this.state = state;
if (DEBUG) trace(OUT, "init()");
    }

    public void setIV (byte[] iv) throws InvalidParameterException {
if (DEBUG) trace(IN, "setIV("+(iv[0]&0x01)+")");
        if ((state == NIST_Cipher.ENCRYPT_STATE ||
                state == NIST_Cipher.DECRYPT_STATE) && userIV != null)
            throw new IllegalStateException();

        if (iv.length != BLOCK_SIZE)
            throw new InvalidParameterException(""+iv.length);

        userIV = (byte[]) (iv.clone());
        this.iv = (byte[]) (userIV.clone());
if (DEBUG) trace(OUT, "setIV()");
    }

    public byte[] update (byte[] in, int inOff, int inLen) {
if (DEBUG) trace(IN, "update("+in+", "+inOff+", "+inLen+")");
        if (iv == null)
            throw new NullPointerException("IV");

        byte[] out = new byte[inLen];
        int outOff = 0;

        switch (state) {
        case NIST_Cipher.ENCRYPT_STATE:
            for (int i = 0; i < inLen; i++)
                out[outOff++] = byteEncrypt(in[inOff++] & 0xFF);
            break;
        case NIST_Cipher.DECRYPT_STATE:
            for (int i = 0; i < inLen; i++)
                out[outOff++] = byteDecrypt(in[inOff++] & 0xFF);
            break;
        default:
            throw new IllegalStateException();
        }
if (DEBUG) trace(OUT, "update()");
        return out;
    }

    public byte[] doFinal (byte[] in, int inOff, int inLen) {
if (DEBUG) trace(IN, "doFinal("+in+", "+inOff+", "+inLen+")");
        byte[] out = this.update(in, inOff, inLen);
        state = 0; // cipher is now un-initialized
if (DEBUG) trace(OUT, "doFinal()");
        return out;
    }

    public boolean self_test() {
if (DEBUG) trace(IN, "self_test()");
        boolean ok = false;
        try {
            byte[] key = new byte[] {
                0, 1, 2, 3, 4, 5, 6, 7, 0, 1, 2, 3, 4, 5, 6, 7 };
            byte[] iv = new byte[] {
                0, 1, 2, 3, 4, 5, 6, 7, 0, 1, 2, 3, 4, 5, 6, 7 };
            byte[] pt = new byte[] { // need not be n*BLOCK_SIZE
                0, 1, 2, 3, 4, 5, 6, 7, 0, 1, 2, 3, 4, 5, 6, 7,
                0, 1, 2, 3, 4 };

            NIST_CFB cipher = new NIST_CFB("NIST");
            cipher.setIV(iv);
            cipher.init(NIST_Cipher.ENCRYPT_STATE, key);
            byte[] ct = cipher.doFinal(pt, 0, pt.length);
if (DEBUG) debug(" plain text: " + toString(pt));
            cipher.init(NIST_Cipher.DECRYPT_STATE, key);
            byte[] cpt = cipher.doFinal(ct, 0, ct.length);
if (DEBUG) debug("plain2 text: " + toString(cpt));
            ok = areEqual(pt, cpt);
            if (!ok) throw new
                RuntimeException("Level-1 API CFB failed");
        } catch (Exception x) {
if (DEBUG) debug("Exception encountered during self-test: " + x.getMessage());
            x.printStackTrace();
            return false;
        }
        debug("Self-test OK? " + ok);
if (DEBUG) trace(OUT, "self_test()");
        return true;
    }


// own methods
//...........................................................................

    private void generateKey (byte[] key) throws InvalidKeyException {
if (DEBUG) trace(IN, "generateKey("+key+")");
        if (key == null)
            throw new InvalidKeyException("Null key");

        int length = key.length;
        if (!(length == 16 || length == 24 || length == 32))
             throw new InvalidKeyException("Incorrect length: "+length);

        args = new Object[] { key };
        try {
            sessionKey = makeKey.invoke(null, args);
        } catch (IllegalAccessException x1) {
if (DEBUG) debug("Illegal access to method " + aes + ".makeKey()");
            throw new InvalidKeyException(x1.getMessage());
        } catch (InvocationTargetException x2) {
if (DEBUG) debug("Exception occured in method " + aes + ".makeKey()");
            throw new InvalidKeyException(x2.getMessage());
        }
if (DEBUG) trace(OUT, "generateKey()");
    }

    private void engineInit() {
        if (userIV != null)
            iv = (byte[]) (userIV.clone());
    }

    /**
     * Encrypt an 8-bit entity right justified in a Java int, with the
     * entity's bit number 1 being the most significant bit of a Java byte.
     *
     * @return An 8-bit ciphertext entity.
     */
    private byte byteEncrypt (int pt) {
        int Ki, Pi, bit, Ci = 0;
        for (int j = 0; j < 8; j++) {
            // in CFB always encrypt xorBlock
            args = new Object[] { iv, new Integer(0), sessionKey };
            try {
                byte[] ciphertext = (byte[]) encrypt.invoke(null, args);
                Ki = (ciphertext[0] >>> 7) & 0x01; // bit 1 is leftmost
            } catch (Exception x) {
if (DEBUG) debug("Exception occured during encryption");
                x.printStackTrace();
                throw new RuntimeException(x.getMessage());
            }
            Pi = (pt >>> (7-j)) & 0x01;
            bit = (Ki ^ Pi) & 0x01;
            // shift iv 1 position to the left
            for (int i = 0; i < BLOCK_SIZE; i++) {
                iv[i] = (byte)(
                    (iv[i] & 0xFF)  << 1 |
                    ((i != BLOCK_SIZE-1) ?
                        (iv[i+1] & 0x80) >>> 7 :
                        bit
                    )
                );
            }
            Ci |= bit << (7-j);
        }
        return (byte) Ci;
    }

    /**
     * Decrypt an 8-bit entity right justified in a Java int, with the
     * entity's bit number 1 being the most significant bit of a Java byte.
     *
     * @return An 8-bit plaintext entity.
     */
    private byte byteDecrypt (int ct) {
        int Ki, Pi = 0, bit, Ci;
        for (int j = 0; j < 8; j++) {
            // in CFB always encrypt xorBlock
            args = new Object[] { iv, new Integer(0), sessionKey };
            try {
                byte[] plaintext = (byte[]) encrypt.invoke(null, args);
                Ki = (plaintext[0] >>> 7) & 0x01; // bit 1 is leftmost
            } catch (Exception x) {
if (DEBUG) debug("Exception occured during decryption");
                x.printStackTrace();
                throw new RuntimeException(x.getMessage());
            }
            Ci = (ct >>> (7-j)) & 0x01;
            bit = (Ki ^ Ci) & 0x01;
            // shift iv 1 position to the left
            for (int i = 0; i < BLOCK_SIZE; i++) {
                iv[i] = (byte)(
                    (iv[i] & 0xFF)  << 1 |
                    ((i != BLOCK_SIZE-1) ?
                        (iv[i+1] & 0x80) >>> 7 :
                        Ci
                    )
                );
            }
            Pi |= bit << (7-j);
        }
        return (byte) Pi;
    }


// utility static methods (from cryptix.util.core.ArrayUtil and Hex classes)
//...........................................................................
    
    /** @return True iff the arrays have identical contents. */
    private static boolean areEqual (byte[] a, byte[] b) {
        int aLength = a.length;
        if (aLength != b.length)
            return false;
        for (int i = 0; i < aLength; i++)
            if (a[i] != b[i])
                return false;
        return true;
    }

    /**
     * Returns a string of hexadecimal digits from a byte array. Each
     * byte is converted to 2 hex symbols.
     */
    private static String toString (byte[] ba) {
        int length = ba.length;
        char[] buf = new char[length * 2];
        for (int i = 0, j = 0, k; i < length; ) {
            k = ba[i++];
            buf[j++] = HEX_DIGITS[(k >>> 4) & 0x0F];
            buf[j++] = HEX_DIGITS[ k        & 0x0F];
        }
        return new String(buf);
    }


// main(): Basic self-test
//...........................................................................

    public static void main (String[] args) {
if (DEBUG) trace(IN, "main()");
        try {
if (DEBUG) debug("Self-test OK? " + new NIST_CFB("NIST").self_test());
        } catch (Exception x) {
            System.err.println("Exception occured: "+x.getMessage());
            x.printStackTrace();
        }
if (DEBUG) trace(OUT, "main()");
    }
}
