// $Id: $
//
// $Log: $
// Revision 1.0.1  1998/04/12  raif
// + activated CFB_1 mode (was left deactivated).
// + added test case for null cipher in CFB-1 mode.
// + identified cryptix.util.core as the source for utility methods.
//
// Revision 1.0  1998/04/04  raif
// + original version.
//
// $Endlog$
/*
 * Copyright (c) 1997, 1998 Systemics Ltd on behalf of
 * the Cryptix Development Team. All rights reserved.
 */
package NIST;

import java.io.PrintWriter;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.security.InvalidKeyException;
import java.security.InvalidParameterException;

//...........................................................................
/**
 * A class to operate any AES candidate algorithm implementation, complying
 * with either level-0 or level-1 APIs, in any NIST required feedback mode.<p>
 *
 * <b>Copyright</b> &copy; 1997, 1998
 * <a href="http://www.systemics.com/">Systemics Ltd</a> on behalf of the
 * <a href="http://www.systemics.com/docs/cryptix/">Cryptix Development Team</a>.
 * <br>All rights reserved.<p>
 *
 * <b>$Revision: $</b>
 * @author  Raif S. Naffah
 */
public final class NIST_Cipher
{
// Constants and variables
//...........................................................................

    public static final int ENCRYPT_STATE = 1;
    public static final int DECRYPT_STATE = 2;

    public static final int ECB_MODE =      1;
    public static final int CBC_MODE =      2;
    public static final int CFB_1BIT_MODE = 3;

    NIST_CipherSpi cipher;


// Constructor
//...........................................................................

    /**
     * Instantiate an object that implements an AES candidate algorithm
     * operating in one of the NIST required modes.<p>
     *
     * First attempt to instantiate a class that implements NIST_CipherSpi
     * Extended API and is supposed to be called <code>aes_suffix</code>,
     * located in package <code>aes</code>. <code>suffix</code> Will be ECB,
     * CBC or CFB depending on the value of <code>mode</code> according to
     * the following table:
     * <pre>
     *    Mode              Suffix     Full class name
     *    =============     =======    ===============
     *    ECB_MODE          ECB        aes.aes_ECB
     *    CBC_MODE          CBC        aes.aes_CBC
     *    CFB_1BIT_MODE     CFB        aes.aes_CFB
     * </pre><p>
     *
     * If the above fails, this class then tries to use NIST own
     * implementations of the feedback modes which assume a class named
     * <code>aes_Algorithm</code> in package <code>aes</code> implementing
     * the Basic API.<p>
     *
     * If that also fails, this class then throws the appropriate Exception.
     *
     * @param algorithm  The distinguished name of the AES candidate
     *      algorithm implementation.
     * @param mode   The identifier of the feedback mode in which
     *      the instantiated cipher will operate. Valid values are
     *      ECB_MODE, CBC_MODE, and CFB_1BIT_MODE.
     * @exception ClassNotFoundException  If unable to locate the
     *      aes.aes_Algorithm class supposed to implement the level-0 API
     * @exception NoSuchMethodException  If at least one of the three
     *      required methods of the level-0 API with the correct number
     *      of parameters was not found.
     * @see NIST.NIST_Cipher#ECB_MODE
     * @see NIST.NIST_Cipher#CBC_MODE
     * @see NIST.NIST_Cipher#CFB_1BIT_MODE
     */
    public NIST_Cipher (String aes, int mode)
    throws ClassNotFoundException, NoSuchMethodException {
        // first attempt as if it has its own mode implementations
        String suffix;
        switch (mode) {
        case ECB_MODE:      suffix = "ECB"; break;
        case CBC_MODE:      suffix = "CBC"; break;
        case CFB_1BIT_MODE: suffix = "CFB"; break;
        default:
            throw new IllegalArgumentException(
                "Invalid feedback mode: "+mode);
        }
        boolean ok = false;
        String className = "";
        try {
            System.err.println("Attempting to use Extended API...");
            className = aes + "." + aes + "_" + suffix;
            Class clazz = Class.forName(className);
            cipher = (NIST_CipherSpi) clazz.newInstance();
            ok = true;
        } catch (ClassNotFoundException x1) {
            System.err.println("Class not found: "+className);
        } catch (InstantiationException x2) {
            System.err.println("Exception while instantiating: "+className);
        } catch (IllegalAccessException x3) {
            System.err.println("Illegal access to: "+className);
        } catch (NoSuchMethodError x4) {
            System.err.println("Does not exist or does not have a no-argument constructor: "+className);
        }
        if (! ok) { // try ours
            System.err.println("Attempting to use Basic API...");
            switch (mode) {
            case ECB_MODE:      cipher = new NIST_ECB(aes); break;
            case CBC_MODE:      cipher = new NIST_CBC(aes); break;
            case CFB_1BIT_MODE: cipher = new NIST_CFB(aes); break;
            }
        }
    }

    /**
     * Throws a CloneNotSupportedException. Cloning of ciphers is not
     * allowed for security reasons.
     *
     * @exception CloneNotSupportedException Not allowed for security reasons.
     */
    public final Object clone() throws CloneNotSupportedException {
        throw new CloneNotSupportedException();
    }


// definition of Level-1 API methods
//...........................................................................

    /**
     * Initialize this cipher for encryption/decryption, using designated
     * key material.
     *
     * @param state  Operation state of this instance. Valid values are
     *          ENCRYPT_STATE or DECRYPT_STATE.
     * @param key   User key material to use for the desired operation.
     * @exception  InvalidKeyException  If the key is invalid.
     * @see NIST.NIST_Cipher#ENCRYPT_STATE
     * @see NIST.NIST_Cipher#DECRYPT_STATE
     */
    public void init (int state, byte[] key) throws InvalidKeyException {
        cipher.init(state, key);
    }

    /**
     * Set the starting value of the IV.<p>
     *
     * @param  iv   The user-supplied IV value.
     * @exception IllegalStateException If the cipher is already initialized
     *      in an encryption/decryption state and the IV has already been set.
     * @exception InvalidParameterException  If the supplied IV value
     *      is of the wrong length.
     */
    public void setIV (byte[] iv) throws InvalidParameterException {
        cipher.setIV(iv);
    }

    /**
     * Continue a multi-part encryption/decryption operation.<p>
     *
     * @param in  Input buffer containing the plaintext/ciphertext (depending
     *      on the current cipher's operational state).
     * @param inOff  Offset into <code>in</code> specifying where data starts.
     *      Caller must ensure it's a positive value that will not cause an
     *      ArrayIndexOutOfBoundsException when accessing <code>in</code>.
     * @param inLen  Length of the subarray. Caller must ensure inLen is a
     *      non-zero positive value.
     * @return The encrypted/decrypted input text.
     * @exception  NullPointerException  If the IV has not been set.
     * @exception  IllegalStateException  If the cipher's operational state is
     *      not properly set.
     */
    public byte[] update (byte[] in, int inOff, int inLen) {
        return cipher.update(in, inOff, inLen);
    }

    /**
     * Encrypt/decrypt data in a single-part operation or finish a
     * multi-part operation.<p>
     *
     * @param in  Input buffer containing the plaintext/ciphertext (depending
     *      on the current cipher's operational state).
     * @param inOff  Offset into <code>in</code> specifying where data starts.
     *      Caller must ensure it's a positive value that will not cause an
     *      ArrayIndexOutOfBoundsException when accessing <code>in</code>.
     * @param inLen  Length of the subarray. Caller must ensure inLen is
     *      zero or a positive value.
     * @return The encrypted/decrypted input text including any other data
     *      that may have been buffered in one or more earlier invocations
     *      to <code>update</code>.
     * @exception  NullPointerException  If the IV has not been set.
     * @exception  IllegalStateException  If the cipher is not properly
     *      initialized.
     * @exception  InvalidParameterException  If the operation when carried
     *      out, will cause an illegal block size exception; eg. the total
     *      length of data to encrypt is not a multiple of the cipher's block
     *      size.
     */
    public byte[] doFinal (byte[] in, int inOff, int inLen) {
        return cipher.doFinal(in, inOff, inLen);
    }


// own private method
//...........................................................................

    private void test() {
        System.err.println("Underlying cipher self test(s) OK? " +
            cipher.self_test());
    }


// utility static method (from cryptix.util.core.ArrayUtil class)
//...........................................................................
    
    /** @return True iff the arrays have identical contents. */
    private static boolean areEqual (byte[] a, byte[] b) {
        int aLength = a.length;
        if (aLength != b.length)
            return false;
        for (int i = 0; i < aLength; i++)
            if (a[i] != b[i])
                return false;
        return true;
    }


// main(): Basic self-test
//...........................................................................

    public static void main (String[] args) {
        boolean ok = false;
        try {
            NIST_Cipher cipher = new NIST_Cipher("NIST", ECB_MODE);
            cipher.test();
            cipher = new NIST_Cipher("NIST", CBC_MODE);
            cipher.test();
            cipher = new NIST_Cipher("NIST", CFB_1BIT_MODE);
            cipher.test();

            byte[] key = new byte[] {
                0, 1, 2, 3, 4, 5, 6, 7, 0, 1, 2, 3, 4, 5, 6, 7 };
            byte[] iv = new byte[] {
                0, 1, 2, 3, 4, 5, 6, 7, 0, 1, 2, 3, 4, 5, 6, 7 };
            byte[] input = new byte[] {
                0, 1, 2, 3, 4, 5, 6, 7, 0, 1, 2, 3, 4, 5, 6, 7,
                0, 1, 2, 3, 4, 5, 6, 7, 0, 1, 2, 3, 4, 5, 6, 7,
                0, 1, 2, 3, 4, 5, 6, 7, 0, 1, 2, 3, 4, 5, 6, 7 };

            cipher.setIV(iv);
            cipher.init(ENCRYPT_STATE, key);
            byte[] ct1 = cipher.doFinal(input, 0, input.length);
            int i = (input.length / 2) - 3;

            cipher.init(DECRYPT_STATE, key);
            byte[] ct2 = new byte[ct1.length];
            byte[] t1 = cipher.update(ct1, 0, i);
            byte[] t2 = cipher.doFinal(ct1, i, input.length - i);
            System.arraycopy(t1, 0, ct2, 0, t1.length);
            System.arraycopy(t2, 0, ct2, t1.length, t2.length);
            ok = areEqual(input, ct2);
            if (!ok) throw new
                RuntimeException("Level-1 API failed");
        } catch (Exception x) {
            x.printStackTrace();
        }
    }
}
