// $Id: $
//
// $Log: $
// Revision 1.0  1998/04/04  raif
// + original version.
//
// $Endlog$
/*
 * Copyright (c) 1997, 1998 Systemics Ltd on behalf of
 * the Cryptix Development Team. All rights reserved.
 */
package NIST;

import java.io.PrintWriter;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.security.InvalidKeyException;
import java.security.InvalidParameterException;

//...........................................................................
/**
 * A class to operate any AES candidate algorithm implementation, complying
 * with level-0 API, in ECB mode.<p>
 *
 * This object handles buffering but does not handle padding.<p>
 *
 * <b>Copyright</b> &copy; 1997, 1998
 * <a href="http://www.systemics.com/">Systemics Ltd</a> on behalf of the
 * <a href="http://www.systemics.com/docs/cryptix/">Cryptix Development Team</a>.
 * <br>All rights reserved.<p>
 *
 * <b>$Revision: $</b>
 * @author  Raif S. Naffah
 */
public final class NIST_ECB
implements NIST_CipherSpi
{
// Debugging methods and variables
//...........................................................................

    static final String NAME = "NIST_ECB";
    static final boolean IN = true, OUT = false;

    static final boolean DEBUG = NIST_Properties.GLOBAL_DEBUG;
    static final int debuglevel = DEBUG ? NIST_Properties.getLevel(NAME) : 0;
    static final PrintWriter err = DEBUG ? NIST_Properties.getOutput() : null;

    static final boolean TRACE = NIST_Properties.isTraceable(NAME);

    static void debug (String s) { err.println(">>> "+NAME+": "+s); }
    static void trace (boolean in, String s) {
        if (TRACE) err.println((in?"==> ":"<== ")+NAME+"."+s);
    }
    static void trace (String s) { if (TRACE) err.println("<=> "+NAME+"."+s); }


// Constants and variables
//...........................................................................

    // transient is used to prevent serialization of sensitive data

    private static final int BLOCK_SIZE = 16; // bytes in an AES block
    private transient Object sessionKey; // current session key

    private transient byte[] buffer;
    private transient int buffered;

    private int state;

    // variables for the Reflection API
    private Method makeKey = null; // the makeKey([B) method
    private Method encrypt = null; // the blockEncrypt([B, int, Object) method
    private Method decrypt = null; // the blockDecrypt([B, int, Object) method
    private Object[] args; // will contain actual arguments of invoked methods

    private String aes; // full name of the underlying _Algorithm class


// Constructor
//...........................................................................

    /**
     * @param algorithm  The distinguished name of the AES candidate
     *      algorithm implementation, assumed to comply with level-0
     *      API within a class called aes_Algorithm in package aes.
     * @exception ClassNotFoundException  If unable to locate the
     *      aes.aes_Algorithm class supposed to implement the level-0 API
     * @exception NoSuchMethodException  If at least one of the three
     *      required methods of the level-0 API with the correct number
     *      of parameters was not found.
     */
    protected NIST_ECB (String aes)
    throws ClassNotFoundException, NoSuchMethodException {
        try { // get references to underlying *_Algorithm methods
            aes = aes + "." + aes + "_Algorithm";
            Class algorithm = Class.forName(aes);
            // inspect the *_Algorithm class
            Method[] methods = algorithm.getDeclaredMethods();
            for (int i = 0, n = methods.length; i < n; i++) {
                String name = methods[i].getName();
                int params = methods[i].getParameterTypes().length;
                if (name.equals("makeKey") && (params == 1))
                    makeKey = methods[i];
                else if (name.equals("blockEncrypt") && (params == 3))
                    encrypt = methods[i];
                else if (name.equals("blockDecrypt") && (params == 3))
                    decrypt = methods[i];
            }
            if (makeKey == null)
                throw new NoSuchMethodException("makeKey()");
            if (encrypt == null)
                throw new NoSuchMethodException("blockEncrypt()");
            if (decrypt == null)
                throw new NoSuchMethodException("blockDecrypt()");
        } catch (ClassNotFoundException x1) {
if (DEBUG) debug("Class " + aes + " not found");
            throw (ClassNotFoundException) x1.fillInStackTrace();
        } catch (NoSuchMethodException x2) {
if (DEBUG) debug("Method " + aes + "." + x2.getMessage() + " not found");
            throw (NoSuchMethodException) x2.fillInStackTrace();
        }
        this.aes = aes;

        buffer = new byte[BLOCK_SIZE];
        buffered = 0;
        sessionKey = null;
    }

    /**
     * Throws a CloneNotSupportedException. Cloning of ciphers is not
     * allowed for security reasons.
     *
     * @exception CloneNotSupportedException Not allowed for security reasons.
     */
    public final Object clone() throws CloneNotSupportedException {
        throw new CloneNotSupportedException();
    }


// Implementation of Level-1 API methods
//...........................................................................

    public void init (int state, byte[] key) throws InvalidKeyException {
if (DEBUG) trace(IN, "init()");
        generateKey(key);
        engineInit();
        this.state = state;
if (DEBUG) trace(OUT, "init()");
    }

    /** Meaningless in an ECB mode. Always throws an Error. */
    public void setIV (byte[] iv) throws InvalidParameterException {
        throw new Error("Not allowed in ECB mode");
    }

    public byte[] update (byte[] in, int inOff, int inLen) {
if (DEBUG) trace(IN, "update("+in+", "+inOff+", "+inLen+")");
        byte[] temp;
        byte[] out = new byte[BLOCK_SIZE * ((inLen + buffered) / BLOCK_SIZE)];
        int outOff = 0;
        for (int i = 0; i < inLen; i++) {
            buffer[buffered++] = in[inOff++];
            if (buffered >= BLOCK_SIZE) {
                temp = processBuffer();
                System.arraycopy(temp, 0, out, outOff, temp.length);
                outOff += temp.length;
            }
        }
if (DEBUG) trace(OUT, "update()");
        return out;
    }

    public byte[] doFinal (byte[] in, int inOff, int inLen) {
if (DEBUG) trace(IN, "doFinal("+in+", "+inOff+", "+inLen+")");
        if ((inLen + buffered) % BLOCK_SIZE != 0)
            throw new InvalidParameterException();

        byte[] out = this.update(in, inOff, inLen);

        state = 0; // cipher is now un-initialized
if (DEBUG) trace(OUT, "doFinal()");
        return out;
    }

    public boolean self_test() {
if (DEBUG) trace(IN, "self_test()");
        boolean ok = false;
        try {
            byte[] key = new byte[] {
                0, 1, 2, 3, 4, 5, 6, 7, 0, 1, 2, 3, 4, 5, 6, 7 };
            byte[] input = new byte[] {
                0, 1, 2, 3, 4, 5, 6, 7, 0, 1, 2, 3, 4, 5, 6, 7,
                0, 1, 2, 3, 4, 5, 6, 7, 0, 1, 2, 3, 4, 5, 6, 7,
                0, 1, 2, 3, 4, 5, 6, 7, 0, 1, 2, 3, 4, 5, 6, 7 };

            NIST_ECB cipher = new NIST_ECB("NIST"); // the null cipher
            cipher.init(NIST_Cipher.ENCRYPT_STATE, key);
            byte[] ct1 = cipher.doFinal(input, 0, input.length);
            int i = (input.length / 2) - 3;

            cipher.init(NIST_Cipher.ENCRYPT_STATE, key);
            byte[] ct2 = new byte[ct1.length];
            byte[] t1 = cipher.update(input, 0, i);
            byte[] t2 = cipher.doFinal(input, i, input.length - i);
            System.arraycopy(t1, 0, ct2, 0, t1.length);
            System.arraycopy(t2, 0, ct2, t1.length, t2.length);
            ok = areEqual(ct1, ct2);
            if (!ok) throw new
                RuntimeException("Level-1 API ECB encryption failed");
        } catch (Exception x) {
if (DEBUG) debug("Exception encountered during self-test: " + x.getMessage());
            x.printStackTrace();
            return false;
        }
if (DEBUG) debug("Self-test OK? " + ok);
if (DEBUG) trace(OUT, "self_test()");
        return true;
    }


// Own private methods
//...........................................................................

    private void generateKey (byte[] key) throws InvalidKeyException {
if (DEBUG) trace(IN, "generateKey("+key+")");
        if (key == null)
            throw new InvalidKeyException("Null key");

        int length = key.length;
        if (!(length == 16 || length == 24 || length == 32))
             throw new InvalidKeyException("Incorrect length: "+length);

        args = new Object[] { key };
        try {
            sessionKey = makeKey.invoke(null, args);
        } catch (IllegalAccessException x1) {
if (DEBUG) debug("Illegal access to method " + aes + ".makeKey()");
            throw new InvalidKeyException(x1.getMessage());
        } catch (InvocationTargetException x2) {
if (DEBUG) debug("Exception occured in method " + aes + ".makeKey()");
            throw new InvalidKeyException(x2.getMessage());
        }
if (DEBUG) trace(OUT, "generateKey()");
    }

    private void engineInit() {
        for (int i = 0; i < BLOCK_SIZE; )
            buffer[i++] = 0;
        buffered = 0;
    }

    private byte[] processBuffer() {
        byte[] result;
        switch (state) {
        case NIST_Cipher.ENCRYPT_STATE:
            args = new Object[] { buffer, new Integer(0), sessionKey };
            try {
                result = (byte[]) encrypt.invoke(null, args);
            } catch (Exception x) {
if (DEBUG) debug("Exception occured during encryption");
                x.printStackTrace();
                throw new RuntimeException(x.getMessage());
            }
            break;
        case NIST_Cipher.DECRYPT_STATE:
            args = new Object[] { buffer, new Integer(0), sessionKey };
            try {
                result = (byte[]) decrypt.invoke(null, args);
            } catch (Exception x) {
if (DEBUG) debug("Exception occured during decryption");
                x.printStackTrace();
                throw new RuntimeException(x.getMessage());
            }
            break;
        default:
            throw new IllegalStateException();
        }
        engineInit();
        return result;
    }


// utility static method (from cryptix.util.core.ArrayUtil class)
//...........................................................................
    
    /** @return True iff the arrays have identical contents. */
    private static boolean areEqual (byte[] a, byte[] b) {
        int aLength = a.length;
        if (aLength != b.length)
            return false;
        for (int i = 0; i < aLength; i++)
            if (a[i] != b[i])
                return false;
        return true;
    }


// main(): Basic self-test
//...........................................................................

    public static void main (String[] args) {
if (DEBUG) trace(IN, "main()");
        try {
if (DEBUG) debug("Self-test OK? " + new NIST_ECB("NIST").self_test());
        } catch (Exception x) {
            System.err.println("Exception occured: "+x.getMessage());
            x.printStackTrace();
        }
if (DEBUG) trace(OUT, "main()");
    }
}
