/* This module implements a 'completion_port' object managing asynchronous
 * operations to multiple streams and values.
 *
 * Author: David Jones
 * Revised: 8-AUG-1998		release lock while cancelling pending I/O
 *				due to timeout.
 * 10-MAR-1999			Do small optimization for wrapper completion.
 */
#include <stdio.h>
#include <stdlib.h>
#include <descrip.h>			/* VMS string descriptors */
#include <iodef.h>			/* VMS QIO function codes */
#include <ssdef.h>			/* VMS sys. service error codes */
int SYS$ASSIGN(), SYS$DASSGN(), SYS$CANCEL(), SYS$QIO();

#include "completion_port.h"		/* prototypes for exported functons */

#include "pthread_1c_np.h"
#include "tutil.h"
#include "tmemory.h"
/*
 * Define internal structure pointed to by cport_port handle.
 */
struct cport_port_instance {
    struct cport_port_instance *next;
    /*
     * Pthreads synchronization objects to serialize access to port
     * structure.
     */
    pthread_mutex_t lock;
    pthread_cond_t io_completion;
    tm_destructor_handle mem_lock;
    /*
     * if timed_busy > 0, soonest_timeout is pointer to isb with earliest
     * timeout time.
     */
    int ref_count;
    int timed_busy;		/* count of busy isb with timeouts */
    cport_isb soonest_timeout;
    /*
     * Use three blank ISBs as the queue heads.  An isb is always on
     * one of three queues circular queues:
     *    busy		stream driver/qio operation in progess (AST)
     *    completed	awating delivery to user by cport_next_completion
     *    processed	Ready to become busy again.
     */
   struct cport_io_sync_block busy, completed, processed;
   /*
    * Last_assign_status and last_assign_errmsg store condition code
    * giving reason last assign to device or stream failed.  Failure to
    * assign means no isb is created so there is no place in it to store
    * this information.
    */
   int last_assign_status;
   char last_assign_errmsg[256];
};
typedef struct cport_port_instance *cport_instance;
/*
 * Define global lookaside lists for port instance and isb allocation.
 */
static pthread_once_t module_setup = PTHREAD_ONCE_INIT;
static pthread_key_t open_port;
static pthread_mutex_t pool;
static cport_instance free_port_instance;
static cport_isb free_isb;
#define PORT_ALLOC_GROUP 4
#define ISB_ALLOC_GROUP 24
static cport_stream_handler no_driver;
/*
 * forward references for internal functions;
 */
static void port_rundown ( cport_instance );
/*
 * Macros for queue manipuation:
 *   ENQUEUE - add isb to tail of queue.
 *   DEQUEUE - remove isb from arbitrary spot in queue.
 *   REQUEUE - move isb from current queue and move to tail (isb must be at head).
 *
 * You must hold port->lock mutex while manipulating the queues.
 */
#define ENQUEUE(b,d) b->q.flink = &d; b->q.blink = d.q.blink; \
	d.q.blink->q.flink = b; d.q.blink = b;

#define DEQUEUE(b) b->q.flink->q.blink = b->q.blink; \
	b->q.blink->q.flink = b->q.flink;

#define REQUEUE(q1,b,q2) q1.q.flink=b->q.flink; b->q.flink->q.blink = &q1; \
	b->q.flink = &q2; b->q.blink = q2.q.blink; \
	q2.q.blink->q.flink=b; q2.q.blink = b;

static void dump_q ( cport_isb isb, char *text ) {
    cport_instance port;
    printf ( "%s\n", text );
    printf("   isb %x: flink=%x, blink=%x, port=%x\n", isb,
	isb->q.flink, isb->q.blink, isb->port );
    port = (cport_instance) isb->port;
    printf("   busy   %x: flink=%x blink=%x   timed=%d\n", 
	&port->busy, port->busy.q.flink, port->busy.q.blink, port->timed_busy );
    printf("   cmplt  %x: flink=%x blink=%x\n", 
	&port->completed, port->completed.q.flink, port->completed.q.blink );
    printf("   procd  %x: flink=%x blink=%x\n", 
	&port->processed, port->processed.q.flink, port->processed.q.blink );
}
/*************************************************************************/
/* One-time setup routine, initialize pthread objects and lists.
 */
void module_init ( )
{
    INITIALIZE_MUTEX ( &pool );
    tm_initialize();
    CREATE_KEY ( &open_port, (pthread_destructor_t) port_rundown );
    free_isb = (cport_isb) 0;
    free_port_instance = (cport_instance) 0;
}
/*************************************************************************/
/* Define pool allocation/deallocation functions.  All these functions
 * acquire pool mutex for serializing access to the global structures.
 *
 * Port instances are allocated from system memory in blocks of 4 and
 * isb in blocks of 24.
 */
static cport_instance alloc_instance()
{
    cport_instance new;
    struct iblock { 
	struct cport_port_instance instance[PORT_ALLOC_GROUP]; } *block;
    int i;

    pthread_mutex_lock ( &pool );
    if ( !free_port_instance ) {
	/*
	 * allocate group and place on free list.  Pre-initalize the
	 * pthreads objects on each block.
	 */
	block = (struct iblock *) malloc ( sizeof(struct iblock ) );
	if ( block ) {
	    for ( i = 0; i < PORT_ALLOC_GROUP; i++ ) {
		block->instance[i].next = &block->instance[i+1];
		INITIALIZE_MUTEX ( &block->instance[i].lock );
		INITIALIZE_CONDITION ( &block->instance[i].io_completion );
	    }
	    block->instance[PORT_ALLOC_GROUP-1].next = (cport_instance) 0;
	    free_port_instance = &block->instance[0];
	} else {
	    fprintf(stderr, "malloc failure in completion_port module\n" );
	}
    }

    new = free_port_instance;
    if ( new ) {
	/*
	 * Remove block from free list and initlaize queues.  Queues are
	 * double-linked ciruclar lists.
	 */
	free_port_instance = new->next;
	new->next = (cport_instance) 0;	
	new->ref_count = 1;
	new->timed_busy = 0;
	new->busy.q.flink = new->busy.q.blink = &new->busy;
	new->completed.q.flink = new->completed.q.blink = &new->completed;
	new->processed.q.flink = new->processed.q.blink = &new->processed;
    }
    pthread_mutex_unlock ( &pool );
    return (cport_instance) new;
}
static void dealloc_instance(cport_instance port)
{
    pthread_mutex_lock ( &pool );
    port->next = free_port_instance;
    port->ref_count = 0;
    free_port_instance = port;
    pthread_mutex_unlock ( &pool );
}
static cport_isb alloc_isb ( cport_instance port )
{
    int i;
    struct iblock { 
	struct cport_io_sync_block instance[ISB_ALLOC_GROUP];
	struct timespec timeout[ISB_ALLOC_GROUP];
    } *block;
    cport_isb new;
    /*
     * Expand pool if needed.
     */
    pthread_mutex_lock ( &pool );
    if ( !free_isb ) {
	/*
	 * allocate group and place on free list.  Pre-initalize the
	 * pthreads objects on each block.
	 */
	block = (struct iblock *) malloc ( sizeof(struct iblock ) );
	if ( block ) {
	    for ( i = 0; i < ISB_ALLOC_GROUP; i++ ) {
	        block->instance[i].q.flink = &block->instance[i+1];
		block->instance[i].timeout =  (void *) &block->timeout[i];
	    }
	    block->instance[ISB_ALLOC_GROUP-1].q.flink = (cport_isb) 0;
	    free_isb = &block->instance[0];
	} else {
	    fprintf(stderr, "malloc failure in completion_port module\n" );
	}
    }
    /*
     * Allocate and initialize block.
     */
    new = free_isb;
    if ( new ) {
	/*
	 * Remove from free list and place on port's processed queue.
	 */
	free_isb = new->q.flink;
	/*
	 * Initialize the isb for use.
	 */
	new->user_ptr = (void *) 0;
	new->user_val = 0;
	new->iosb = (void *) &new->default_iosb;
	new->port = (cport_port) port;		/* cast to opaque data type */
	new->driver = no_driver;
	new->completion_callback = 0;
	new->wrapper = (cport_isb) 0;
	new->timer = 0;				/* not timed */
	new->channel = new->flags = 0;
	new->default_iosb[0] = 0;
	new->drv = (void *) 0;			/* no context */
    }
    pthread_mutex_unlock ( &pool );
    return new;
}
static void dealloc_isb ( cport_isb isb )
{
    pthread_mutex_lock ( &pool );
    isb->port = (cport_port) 0;
    isb->q.flink = free_isb;
    free_isb = isb;
    pthread_mutex_unlock ( &pool );
}
/*
 * Rundown streams, deassign all streams assigned to a port instance.
 */
static void rundown_streams ( cport_instance port )
{
    cport_isb isb;
    cport_isb qhead;
    int i;

    pthread_mutex_lock ( &port->lock );
    qhead = &port->busy;
    for ( i = 0; i < 3; i++ ) {
	while ( qhead[i].q.flink != &qhead[i] ) {
	    isb = qhead[i].q.flink;
	    while ( isb->wrapper ) {
		if ( isb->wrapper->port != isb->port ) {
		    fprintf(stderr,"Dead isb found: %x ->%x\n", isb,
			isb->wrapper );
		    isb->wrapper = (cport_isb) 0;
		} else
		isb = isb->wrapper;	/* find root stream */
	    }
	    if ( !isb->port || !isb->driver.ftable ) {
		fprintf(stderr,"BUGCHECK, attempted deassign on detached isb\n");
		DEQUEUE(isb);
		pthread_mutex_lock ( &pool );
		isb->driver.ftable = (cport_start_function *) 0;
		isb->port = (cport_port) 0;
		isb->q.flink = free_isb;
		free_isb = isb;
		pthread_mutex_unlock ( &pool );
		continue;
	    }
	    pthread_mutex_unlock ( &port->lock );
	    cport_deassign ( isb );
	    pthread_mutex_lock ( &port->lock );
	}
    }
    pthread_mutex_unlock ( &port->lock );
}
/****************************************************************************/
/* Destructor routine for open_port context key, which is list of
 * ports created by a thread.  Destructor ensures the ports are rundown
 * and freed. This routine should never execute because threads should
 * explicitly destroy their ports.
 */
static void port_rundown ( cport_instance port )
{
    cport_instance cur, last;
    /*
     * Vist each port on list and rundown streams.  Skipping those
     * with wrappers.
     */
    for ( cur = last = port; cur; cur = cur->next ) {
	/*
	 * Save pointer to end of chain for adding to free list below.
	 */
	last = cur;
	/*
	 * scan busy queue and cancel I/O streams.
	 */
	rundown_streams ( cur );
    }
    /*
     * release per-thread heaps that want to run down.
     */
    tm_unlock_destructor ( port, port->mem_lock );
    /*
     * Add whole group to free port list.
     */
    if ( last ) {
	pthread_mutex_lock ( &pool );
        last->next = free_port_instance;
	free_port_instance = last;
        pthread_mutex_unlock ( &pool );
    }
    return;
}
/****************************************************************************/
/* Create and destroy port instances.  Flag on create are not currently used
 * and must be zero.  On error (malloc fail, pthread init failure) a null
 * pointer is returned.
 */
cport_port cport_create_port ( int flags )
{
    cport_instance new, cur;
    /*
     * Do one-time setup and allocate new instance.
     */
    pthread_once ( &module_setup, module_init );
    new = alloc_instance();
    if ( !new ) return (cport_port) new;
    /*
     * Add to thread's list of open ports (push to front of list
     */
    GET_SPECIFIC ( open_port, new->next );
    pthread_setspecific ( open_port, new );
    if ( !new->next ) new->mem_lock = tm_lock_destructor ( new );
    
    return (cport_port) new;		/* cast to the opaque type */
}

int cport_destroy_port ( cport_port port )
{
    cport_instance p, cur, prev;
    /*
     * Ensure all streams are closed.
     */
#ifdef DEBUG
    printf("Destroying port %x\n", port );
#endif
    p = (cport_instance) port;
    rundown_streams ( p );
    /*
     * Remove from thread's open port list and place on free queue.
     */
    GET_SPECIFIC ( open_port, cur );
    if ( cur == p ) {
	/*
	 * Our block first in list (usualy true because only 1 in list).
	 * If last port, unlock tmemory destructor routine.
	 */
	pthread_setspecific ( open_port, p->next );
	if ( !p->next ) tm_unlock_destructor ( p, p->mem_lock );
    } else if ( cur ) {
	/*
	 * Not first in list and more than 1 port.
	 */
	prev = cur;
        for ( cur = cur->next; cur; cur = cur->next ) {
	    if ( cur == p ) {
		prev->next = p->next;
		break;
	    }
	}
    }
    dealloc_instance ( p );

    if ( !cur ) {
	/*
	 * Error, port not found on list.
	 */
	fprintf(stderr, "BUGCHECK: %s\n",
	    "thread destroyed completion port it did not own" );
	return 0;
    }
    return 1;
}
/***************************************************************************/
/* Define private handler block for QIO-based operation.
 */
static int qio_start_io ( cport_isb isb, int func, void *buffer, int length )
{
    if ( func < 0 || func > 1 ) return 20;
    return cport_qio2 ( isb, (func == 0) ? IO$_WRITEVBLK : IO$_READVBLK,
	buffer, length );
}
static cport_start_function qio_ftable[1] = { qio_start_io };

static int qio_create (cport_isb isb, void *device, int flags, 
	char errmsg[256])
{
    int status;
    struct { int length; void *data; } device_dx;

    device_dx.length = tu_strlen ( (char *) device );
    device_dx.data = device;
    status = SYS$ASSIGN ( &device_dx, &isb->channel, 0, 0, flags );
    errmsg[0] = '\0';
    return status;
}

static int qio_destroy ( cport_isb isb )
{
    return SYS$DASSGN ( isb->channel );
}

static int qio_cancel ( cport_isb isb )
{
    return SYS$CANCEL ( isb->channel );
}

static cport_stream_handler qio_driver = {
    0,				/* all start_io handle by 1 rout. */
    qio_ftable,			/* I/O function */
    qio_create,			/* assign channel */
    qio_destroy,
    qio_cancel
};
/****************************************************************************/
/* Allocate synchronization block and setup for use by client.
 */
cport_isb cport_assign_channel ( cport_port cport, 
	void *device, 			/* device for SYS$ASSIGN */
	int flags )			/* flags arg for SYS$ASSIGN */
{
     /*
      * Assign channel just uses the qio driver table.
      */
     return cport_assign_stream ( cport, &qio_driver, device, flags );
}

cport_isb cport_assign_stream ( 
	cport_port port_handle, 
	cport_stream_handler *handler,  
	void *arg1, int arg2 )
{
    cport_isb isb;
    cport_instance port;
    int status;
    char errmsg[256];
    /*
     * Allocate a provisional ISB
     */
    port = (cport_instance) port_handle;
    isb = alloc_isb ( port );
    if ( !isb ) {
	pthread_mutex_lock ( &port->lock );
	port->last_assign_status = SS$_INSFMEM;
	tu_strcpy ( port->last_assign_errmsg,
		"Error allocating memory for I/O synchronization block" );
	pthread_mutex_unlock ( &port->lock );
	return isb;
    }
    /*
     * Load driver functions into dispatch table and call the create
     * function to to driver-specific initialization.  (the function table
     * is shared, but other entry points get per-instance copies).
     */
    isb->driver = *handler;
    status = isb->driver.attach ( isb, arg1, arg2, port->last_assign_errmsg );
    pthread_mutex_lock ( &port->lock );
    port->last_assign_status = status;
    if ( (status&1) == 1 ) { 
	/*
	 * Move isb to port's processed queue to ensure cleanup on thread exit.
	 */
	ENQUEUE(isb,port->processed);
    }
    pthread_mutex_unlock ( &port->lock );
    if ( (status&1) == 0 ) {
	/*
	 * Create failed, cleanup.
	 */
	dealloc_isb ( isb );
	isb = (cport_isb) 0;
    }
    return isb;
}
int cport_last_assign_status ( cport_port port_handle, char errmsg[256] )
{
    cport_instance port;
    port = (cport_instance) port_handle;
    tu_strnzcpy ( errmsg, port->last_assign_errmsg, 255 );
    return  port->last_assign_status;
}
/***************************************************************************/
/* Disconnect stream and free associated i/o sync. block.
 */
int cport_deassign ( cport_isb isb )
{
    int status;
    cport_instance port;
    /*
     * Call driver to rundown channel.
     */
    status = isb->driver.detach ( isb );
    /*
     * remove isb from port's data base and move to global lookaside list.
     */
    port = (cport_instance) isb->port;
    pthread_mutex_lock ( &port->lock );
    DEQUEUE(isb);
    pthread_mutex_unlock ( &port->lock );

    pthread_mutex_lock ( &pool );
    isb->driver.ftable = (cport_start_function *) 0;	/* mark free */
    isb->port = (cport_port) 0;			/* mark free */
    isb->q.flink = free_isb;
    free_isb = isb;
    pthread_mutex_unlock ( &pool );
    
    return 0;
}


int cport_cancel ( cport_isb isb )
{
    /*
     * call through dispatch table, driver responsible for moving isb to
     * completed queue.
     */
    int status;
    status = isb->driver.cancel ( isb );
    return status;
}

/***************************************************************************/
/* Setup a timeout for an isb.
 */
int cport_set_timeout ( cport_isb isb, int seconds, int nanoseconds )
{
    if ( seconds == 0 && nanoseconds == 0 ) {
	isb->timer = 0;			/* cancel timeout */
    } else {
	struct timespec delta, *timeout;
	timeout = (struct timespec *) isb->timeout;
	delta.tv_sec = seconds;
	delta.tv_nsec = nanoseconds;
	if ( 0 == pthread_get_expiration_np ( &delta, isb->timeout ) ) {
	    isb->timer = 1;
	} else {
	    return 0;			/* error */
	}
    }
    return 1;
}

static void add_timed_busy ( cport_isb isb, cport_instance port )
{
    if ( port->timed_busy <= 0 ) {
	port->soonest_timeout = isb;
	port->timed_busy = 1;
    } else {
	/*
	 * see if earlier than current soonest.  Assumed caller holds
	 * lock mutex for port.
	 */
	struct timespec *soonest, *cand;
	soonest = (struct timespec *) port->soonest_timeout->timeout;
	cand = (struct timespec *) isb->timeout;

	if ( soonest->tv_sec < cand->tv_sec ) {
	} else if ( soonest->tv_sec > cand->tv_sec ) {
	    port->soonest_timeout = isb;
	} else if ( soonest->tv_sec == cand->tv_sec &&
		soonest->tv_nsec > cand->tv_nsec  ) {
	    port->soonest_timeout = isb;
	}
        port->timed_busy++;
    }
}
static void subtract_timed_busy ( cport_isb isb, cport_instance port )
{
    if ( port->timed_busy <= 0 ) return;
    if ( port->timed_busy > 1 && (port->soonest_timeout == isb) ) {
	/*
	 * current timeout was the earliest.  Scan all isb and
	 * find new earliest time.
	 */
	cport_isb cur, earliest;
	earliest = (cport_isb) 0;

	for ( cur=port->busy.q.flink; cur!=&port->busy; cur=cur->q.flink ) {
	    struct timespec *e, *c;

	    if ( cur->timer == 0 || cur == isb ) continue;
	    c = (struct timespec *) cur->timeout;
	    e = (struct timespec *) earliest->timeout;
	    if ( !earliest ) {
		earliest = cur;
	    } else if ( e->tv_sec > c->tv_sec ) {
		earliest = cur;
	    } else if ( e->tv_sec == c->tv_sec && e->tv_nsec > c->tv_nsec ) {
		earliest = cur;
	    }
	}
	/*
	 * Update port struct to port to new earliest time.
	 */
	if ( !earliest ) {
	    fprintf(stderr, "BUGCHECK: timeout busy counting error\n" );
	    port->timed_busy = 0;
	} else {
	    port->soonest_timeout = earliest;
	}
    }
    --port->timed_busy;
}
/***************************************************************************/
/* Start_io functions, initialize and synchronous I/O operation.
 *
 * The QIO functions take VMS IO$_* function codes for the func argument.
 * cport_qio2 lets you call with only p1 and p2, p3..p6 are set to zero.
 * Note that cport_qio does not save P1/P2 in isb->buffer/isb->length,
 * while cport_qio2 does.
 *
 * Caller MUST set isb->iosb to point to the IOSB to receive the result
 * of the query.
 */
int cport_qio ( cport_isb isb, int func, void *p1, void *p2, void *p3,
	void *p4, void *p5, void *p6 )
{
    int status;
    cport_instance port;

    port = (cport_instance) isb->port;
    status = SYS$QIO ( 4, isb->channel, func, isb->iosb,
	pthread_cond_signal_int_np, &port->io_completion, 
	p1, p2, p3, p4, p5, p6 );
    if ( (status&1) == 1 ) {
	/*
	 * I/O started, move isb from wherever it is to tail of busy queue.
	 */
	pthread_mutex_lock ( &port->lock );
	DEQUEUE(isb);
	ENQUEUE(isb,port->busy );
	if ( isb->timer == 1 ) add_timed_busy ( isb, port );
	pthread_mutex_unlock ( &port->lock );
    }
    return status;
}

int cport_qio2 ( cport_isb isb, int func, void *p1, int p2 )
{
    int status;
    cport_instance port;

    port = (cport_instance) isb->port;
    isb->length = p2;
    isb->buffer = p1;
    status = SYS$QIO ( 4, isb->channel, func, isb->iosb,
	pthread_cond_signal_int_np, &port->io_completion, 
	p1, p2, 0, 0, 0, 0 );
    if ( (status&1) == 1 ) {
	/*
	 * I/O started, move isb from wherever it is to tail of busy queue.
	 */
	pthread_mutex_lock ( &port->lock );
	DEQUEUE(isb);
	ENQUEUE(isb,port->busy );
	if ( isb->timer == 1 ) add_timed_busy ( isb, port );
	pthread_mutex_unlock ( &port->lock );
    }
    return status;
}

int cport_start_io ( cport_isb isb, int func, void *buffer, int length )
{
    int status, fcode;
    /*
     * mask out requested function.
     */
    fcode = ( func & isb->driver.fmask );
    isb->length = length;
    isb->buffer = buffer;
#ifdef DEBUG
    printf("Start_io on isb %x, func=%d, buf=%x, len=%d\n", isb, func, 
		buffer,length);
#endif
    status = (isb->driver.ftable[fcode]) ( isb, func, buffer, length );
#ifdef DEBUG
    printf("ftable[%d](%x,%d,%x,%d)  returned: %d\n", fcode, isb,
	func, buffer, length, status );
#endif
    return status;
}
/****************************************************************************/
/*  The following routines are used by drivers to manipulate the state
 *  of I/Os initiated via the driver's start_io function.  A driver
 *  that starts an asynchronous operation other than through 
 *  cport_qio/cport_qio2, must call cport_mark_busy to place the isb on
 *  the busy (I/O pending) queue.  When the I/O completes the driver
 *  calls cport_completion_ast (if at AST level) or cport_mark_completed
 *  with is_busy=1 (non-AST level).
 *
 *  If the driver completes a requested start I/O immediately without
 *  marking the isb busy, it must call cport_mark_completed with is_busy=0.
 */
int cport_mark_busy ( cport_isb isb )
{
    cport_instance port;
    port = (cport_instance) isb->port;		/* x-ray through opaque ptr */
    pthread_mutex_lock ( &port->lock );
    DEQUEUE(isb);
    ENQUEUE(isb,port->busy);
    if ( isb->timer ) add_timed_busy ( isb, port );
    pthread_mutex_unlock ( &port->lock );
    return 1;
}
void cport_completion_ast ( cport_isb isb ) 
{
    /*
     * Lookup port
     */
    unsigned short int *iosb; int status;
    cport_instance port;
    port = (cport_instance) isb->port;		/* x-ray through opaque ptr */
    status = pthread_cond_signal_int_np(&port->io_completion);
}
int cport_mark_completed ( cport_isb isb, int is_busy )
{
    cport_instance port;
    port = (cport_instance) isb->port;		/* x-ray through opaque ptr */
    pthread_mutex_lock ( &port->lock );
    DEQUEUE(isb);
    ENQUEUE(isb,port->completed);
    if ( is_busy ) if ( isb->timer ) subtract_timed_busy ( isb, port );
    pthread_mutex_unlock ( &port->lock );
    return 1;
}
static void complete_wrapper ( cport_instance port, cport_isb wrapper )
{
    /*
     * Internal call for use by completing wrapper isb's (port mutex locked).
     */
    if ( port != (cport_instance) wrapper->port ) { 
	fprintf(stderr, "BUGCHECK - complete_wrapper passed invalid port\n" );
	port = (cport_instance) wrapper->port;
    }
    DEQUEUE(wrapper)
    ENQUEUE(wrapper,port->completed);
    if ( wrapper->timer ) subtract_timed_busy ( wrapper, port );
}

int cport_lock_port ( cport_isb isb )
{
    cport_instance port;
    port = (cport_instance) isb->port;		/* x-ray through opaque ptr */
    pthread_mutex_lock ( &port->lock );
    return 1;
}
int cport_unlock_port ( cport_isb isb, int signal_completion )
{
    cport_instance port;
    port = (cport_instance) isb->port;		/* x-ray through opaque ptr */
    if ( signal_completion ) pthread_cond_signal ( &port->io_completion );
    pthread_mutex_unlock ( &port->lock );
    return 1;
}
int cport_copy_timeout ( cport_isb src, cport_isb dst )
{
    /* Copy timeout info from one isb (src) to another (dst), stream handlers 
     * use this function to propagate timeout settings from internal streams.
     */
    dst->timer = src->timer;
    dst->timeout = src->timeout;
    return 1;
}
/*****************************************************************************/
/* Main synchronization routine for module.  Return i/o sync. block (isb)
 * of next completed I/O.
 */
cport_isb cport_next_completion ( cport_port port_handle )
{
    cport_instance port;
    cport_isb isb;

    port = (cport_instance) port_handle;

    pthread_mutex_lock ( &port->lock );
    /*
     * Loop until completed queue not empty, on exit isb is completed i/o.
     */
    for ( ; ; ) {
	/*
	 * Scan busy list and move completed I/Os to completed list.
	 */
	for (isb=port->busy.q.flink; isb != &port->busy; isb=isb->q.flink) {
	    long *iosb;
	    cport_isb prev;
	    iosb = (long *) isb->iosb;
	    if ( *iosb ) {
		prev = isb->q.blink;
		DEQUEUE ( isb );
		if ( isb->timer ) subtract_timed_busy ( isb, port );
		ENQUEUE ( isb, port->completed );
		isb = prev;
	    }
	}
	/*
	 * Examine head of completed queue to see if queue non-empty.
	 */
	if ( port->completed.q.flink != &port->completed ) {
	    /*
	     * Move isb at head of completed queue to tail of processed queue.
	     */
	    isb = port->completed.q.flink;
	    REQUEUE ( port->completed, isb, port->processed );
	    /*
	     * Check for completion routine and do callout instead of exit.
	     */
#ifdef DEBUG
	    printf("I/O completion on %x, callback=%x, wrapper=%x\n", isb,
		isb->completion_callback, isb->wrapper );
#endif
	    if ( isb->completion_callback ) {
		int disposition;
		pthread_mutex_unlock ( &port->lock );
		disposition = isb->completion_callback ( isb );
		pthread_mutex_lock ( &port->lock );
#ifdef DEBUG
		printf("completion callback for %x returned %d\n", isb, 
			disposition );
#endif
		if ( disposition == 0 ) {
		    DEQUEUE(isb);
		    ENQUEUE(isb,port->busy);
		    if ( isb->timer ) add_timed_busy ( isb, port );
		} else if ( disposition == 1 ) {
		    DEQUEUE(isb);
		    ENQUEUE(isb,port->completed);
		} else if ( disposition != 2 ) {
		    fprintf(stderr, "BUGCHECK: %s (%d)\n",
			"unexpected return value from completion callback",
			disposition );
		}
		if ( disposition != 2 ) continue;
	    }
	    if ( isb->wrapper ) {
		/*
		 * This isb is internal isb for another driver, complete
		 * that I/O instead.
		 */
#ifdef NOWRAPPEROPT
		pthread_mutex_unlock ( &port->lock );
		cport_mark_completed ( isb->wrapper, 1 );
		pthread_mutex_lock ( &port->lock );
#else
		complete_wrapper ( port, isb->wrapper );
#endif
		continue;
	    }
	    break;
	}
	/*
	 * Completed queue was empty, wait for signal.
	 */
	if ( port->timed_busy > 0 ) {
	    int status;
	    status = pthread_cond_timedwait ( &port->io_completion,
		&port->lock, port->soonest_timeout->timeout );
	    if ( status != 0 ) {
		/*
		 * assume timeout occured.
		 */
		cport_isb expired;
		expired = port->soonest_timeout;
		expired->timer = 2;		/* flag as expired */
		pthread_mutex_unlock ( &port->lock );
		expired->driver.cancel ( expired );
		pthread_mutex_lock ( &port->lock );
		
	    }
	} else {
	    pthread_cond_wait ( &port->io_completion, &port->lock );
	}
    }
    pthread_mutex_unlock ( &port->lock );
    return isb;
}
/*
 * Conveinence routine to simplify synchronized I/O operations.
 */
int cport_do_io ( cport_isb isb, int func, void *buffer, int length, int
	*transferred )
{
    int status;
    cport_isb cur;
    cport_instance port;
    unsigned short int *iosb;
    /*
     * Start I/O normally.
     */
    status = cport_start_io ( isb, func, buffer, length );
    if ( (status&1) == 0 ) return status;
    /*
     * wait till our isb shows up on the completed queue.
     */
    for ( port = (cport_instance) isb->port; ; ) {
	cur = cport_next_completion ( (cport_port) port );
	if ( cur == isb ) break;
	if ( !cur ) {
	    /* serious error */
	    return 0;
	}
    }
    /*
     * Extract the data from the iosb.
     */
    iosb = (unsigned short int *) isb->iosb;
    status = iosb[0];
    *transferred = iosb[1];
    return status;
}
