/*
 * This module handles execution of command mode connection where
 * the username has been qualified by /administrate.  Rather than creating
 * a client process, the command is processed internally.
 *
 * The user the system connects as must be present in the administrators
 * parameter list.
 *
 * Author:	David Jones
 * Date:	6-SEP-1998
 * Revised:	17-SEP-1998	Expand matched tokens to full string
 *				when relaying to monitor pipe.
 * Revised:	28-OCT-1998	Cleanup stray system service calls calls.
 */
#include <stdio.h>
#include <stdlib.h>
#include <descrip.h>			/* VMS string descriptors */
#include <string.h>

#include "pthread_1c_np.h"
#include "tutil.h"
#include "tmemory.h"
#define CPORT_DRIVER_SPECIFIC struct private_st
#include "completion_port.h"
#include "parameters.h"
#include "user_info.h"
#include "helper.h"
#include "initiator_client.h"
#include "cport_sshsess.h"
#include "cport_admin.h"		/* verify prototypes */
#include "cport_pipe.h"
#include "monitor.h"

struct mbx_iosb {
    unsigned short status, count;
    int pid;
};
#define MAX_INPUT_LINE 512
/*
 * A private_st structure is allocated for each open asynch_stream and is
 * pointed to by the asynch_stream structure's .private member.
 */
struct out_line {
    struct out_line *next;
    int length;			/* negative length marks EOF */
    int pos;			/* Current position reading from. */
    char text[1];		/* variable length */
};
struct private_st {
    int ref_count;
    int exitstatus;		/* Command exit status. */
    int input_pos;		/* Current buffer position */
    int pend_in_size;		/* Bytes of pending input remaining */
    int pend_out_size;
    char *pending_in;
    char *pending_out;
    struct out_line *output_queue, *output_tail;
    sshsess_session session;
    cport_port port;		/* completion port for PIPE access */
    cport_isb monitor;		/* channel to monitor pipe */
    cport_isb blocked_isb;
    char exit_message[100];
    char input_buffer[MAX_INPUT_LINE];
};
static $DESCRIPTOR(template_dx,"");

/*
 * One-time initialization for module: Initialize per-thread heap and
 * create pipe used to send control message to monitor thread.
 */
static void *monitor_pipe;
static pthread_mutex_t admin_lock;
static pthread_cond_t admin_done;
static int admin_pending;
static struct private_st *admin_owner;

cport_isb execadmin_init ( cport_port monitor_port )
{
    int status;
    cport_isb pipe;
    struct parameter_list_elem *adm;

    tm_initialize();
    status = cport_pipe_open ( monitor_port, &monitor_pipe );
    if ( (status&1) == 1 ) {
	pipe = cport_assign_stream ( monitor_port,
		&cportpipe_driver, monitor_pipe, 1 );
    } else {
	pipe = (cport_isb) 0;
    }
    /*
     * Access to the administrator functions is serialized by a mutex.
     */
    INITIALIZE_MUTEX ( &admin_lock );
    INITIALIZE_CONDITION ( &admin_done );
    admin_owner = (struct private_st *) 0;
    admin_pending = 0;
    /*
     * Upcase the administrators list for caseless compares.
     */
    for ( adm = param.administrator; adm; adm = adm->next ) {
	tu_strupcase ( adm->value, adm->value );
    }
    return pipe;
}
/******************************************************************************/
/* The execadmin_done routine is a callback done by the monitor to signal
 * it has completed processing the command sent to it on the pipe.
 */
void execadmin_done ( int dstatus, char *message )
{
    pthread_mutex_lock ( &admin_lock );
    admin_pending = 0;
    if ( admin_owner ) {
	admin_owner->exitstatus = dstatus;
	tu_strncpy ( admin_owner->exit_message, message, 
		sizeof(admin_owner->exit_message)-1 );
    }
    pthread_cond_broadcast ( &admin_done );
    pthread_mutex_unlock ( &admin_lock );
}
/******************************************************************************/
static int read_from_out_buf ( struct private_st *ctx, 
	char *buffer, int bufsize )
{
    /*
     * Copy contents of ctx->pending_out to specified buffer and update
     * queue element position.
     */
    int segsize, transferred;
    struct out_line *elem;
    elem = ctx->output_queue;
    segsize = elem->length - elem->pos;
    if ( segsize > bufsize ) segsize = bufsize;

    memcpy ( buffer, &elem->text[elem->pos], segsize );
    elem->pos += segsize;
    transferred = segsize;
    if ( (segsize < bufsize) && (elem->pos == elem->length) ) {
	/*
	 * actualize the virtual newline and dequeue the element.
	 */
	buffer[transferred++] = '\n';
	ctx->output_queue = elem->next;
	tm_free ( elem );
	return segsize+1;
    }
    return segsize;
}

static void output_line ( struct private_st *ctx, char *text, int eof )
{
    int length;
    /*
     * Allocate structure to hold line to output.
     */
    struct out_line *elem;
    length = tu_strlen ( text );
    elem = (struct out_line *) tm_malloc ( sizeof(struct out_line) + length );
    if ( !elem ) return;
    elem->next = (struct out_line *) 0;
    elem->length = length;
    elem->pos = 0;
    tu_strcpy ( elem->text, text );
    if ( eof ) elem->length = -1;
    /*
     * Link into queue.
     */
    if ( !ctx->output_queue ) {		/* first in queue */
	ctx->output_queue = elem;
    } else {
	ctx->output_tail->next = elem;
    }
    ctx->output_tail = elem;
    /*
     * Complete pending read if one is waiting.
     */
    if ( ctx->blocked_isb ) {
	cport_isb isb;
	isb = ctx->blocked_isb;
	ctx->blocked_isb = (cport_isb) 0;

	isb->default_iosb[0] = 1;
	if ( ctx->output_queue->length >= 0 ) {
	    isb->default_iosb[1] = read_from_out_buf ( 
		ctx, isb->buffer, isb->length );
	} else {
	    /* End of file. */
	    isb->default_iosb[1] = 0;
	}
	cport_mark_completed ( isb, 1 );
    }
}
/******************************************************************************/
/* Main routine for creating a stream.  Verfify the username is in the
 * allowed list.
 * under the given username and password and the stream context is
 * initialized for asynchrounous communication with this process.
 */
int execadmin_open ( void **ctx, 		/* return context pointer */
	sshsess_session session,
	struct user_account_info *uai,		/* Username/password */
	char errmsg[256] )
{
    int status, ncblen, pwd_len, obj_len;
    struct parameter_list_elem *administrator;
    struct { int length; char *data; } ncb_desc;
    struct mbx_iosb iosb;
    struct private_st *pctx;
    char username[40];
    /*
     * Scan administrators list.  Administrator variable will end up
     * null if user is not an administator.
     */
    tu_strnzcpy ( username, uai->username, sizeof(username)-1 );
    tu_strupcase ( username, username );
    for ( administrator = param.administrator; administrator;
		administrator = administrator->next ) {
	if ( tu_strncmp(username, administrator->value, 100) == 0 ) break;
    }
    if ( !administrator ) {
	tu_strcpy ( errmsg, 
	   "Authenticated user is not designated an administrator" );
	return 20;
    }
    /*
     * Allocate private structure from thread-private heap (automatically
     * freed on thread exit.
     */
    pctx = (struct private_st *) tm_malloc ( sizeof(struct private_st) );
    if ( !pctx ) {
	tu_strcpy ( errmsg, "memory allocation failure" );
	return 0;
    }
    pctx->session = session;
    pctx->ref_count = 0;
    pctx->input_pos = 0;
    pctx->blocked_isb = (cport_isb) 0;
    pctx->output_queue = pctx->output_tail = (struct out_line *) 0;
    pctx->port = cport_create_port ( 0 );
    if ( pctx->port ) pctx->monitor = cport_assign_stream ( pctx->port,
	&cportpipe_driver, monitor_pipe, 0 );
    else pctx->monitor = (cport_isb) 0;

    output_line ( pctx, "Logged into /ADMINISTRATE interface", 0 );
    *ctx = (void *) pctx;
    return 1;
}
/******************************************************************************/
int display_session ( int index, sshsess_session session, void *ctx )
{
    char line[512], number[20];
    struct private_st *pctx;
    int length, i, indent;
    unsigned char *p;
    /*
     * Make prefix
     */
    tu_strint ( index, number );
    length = tu_strlen ( number );
    tu_strcpy ( line, "      " );
    tu_strcpy ( &line[(length < 7) ? 7-length : 1], number );
    length = tu_strlen(line);
    line[length++] = ':'; line[length++] = ' ';
    indent = length;
    /*
     * Add remote host and port.
     */
    p = session->remote_host;
    tu_strint ( *p, &line[length] ); 
    p++; length += tu_strlen(&line[length]); line[length++] = '.';
    tu_strint ( *p, &line[length] ); 
    p++; length += tu_strlen(&line[length]); line[length++] = '.';
    tu_strint ( *p, &line[length] ); 
    p++; length += tu_strlen(&line[length]); line[length++] = '.';
    tu_strint ( *p, &line[length] ); 
    length += tu_strlen(&line[length]); line[length++] = ':';
    tu_strint ( session->remote_port, &line[length] );
    length += tu_strlen(&line[length]); line[length++] = ' ';
    /*
     * Add username if present.
     */
    if ( session->ext_username ) {
	line[length++] = '"';
	tu_strnzcpy ( &line[length], session->ext_username, 64 );
	length += tu_strlen ( &line[length] );
	line[length++] = '"';
    }
    /*
     * Write line.
     */
    line[length] = '\0';
    pctx = (struct private_st *) ctx;
    output_line ( pctx, line, 0 );
    return 1;
}
/**************************************************************************/
/* Parse and execute administration command stored in ctx->input_buffer
 * (length given by ctx->input_pos).  Results are displayed by calling
 * output_line with text lines to be output.
 *
 * Input_buffer is destroyed by this procedure.
 */
void execute_command ( struct private_st *ctx )
{
    int i, j, state, length;
    char *cmd, *monitor_cmd;
    char *token[10];
    /*
     * Parse command string into tokens and upcase first keyword.
     */
    cmd = ctx->input_buffer;
    for ( i = j = state = 0; cmd[i]; i++ ) switch ( state ) {
	case 0:
	    if ( cmd[i] != ' ' && cmd[i] != '\t' ) {
		token[j++] = &cmd[i];
		state = (j < 10) ? 1 : 2;
	    }
	    break;
	case 1:	 
	    if ( cmd[i] == ' ' || cmd[i] == '\t' ) {
		cmd[i] = '\0';		/* terminate string */
		state = 0;
	    }
	    break;
	case 2:		/* too many tokens */
	    break;
    }
    if ( j == 0 ) { j = 1; token[0] = "HELP"; }
    else tu_strupcase ( token[0], token[0] );
    /*
     * dispatch on keyword.
     */
    length = tu_strlen ( token[0] );
    monitor_cmd = (char *) 0;
    if ( tu_strncmp ( token[0], "HELP", length ) == 0 ) {
	output_line ( ctx, "Commands:", 0 );
	output_line ( ctx, "    HELP      Show available commands", 0 );
	output_line ( ctx, "    CLOSE     Stop accepting new connections", 0 );
	output_line ( ctx, "    SHOW xxx  Show:", 0 );
	output_line ( ctx, "                 SESSIONS", 0 );
	output_line ( ctx, "    SHUTDOWN  Stop the server.", 0 );
    } else if ( tu_strncmp ( token[0], "SHELL", 6 ) == 0 ) {
	output_line ( ctx, 
	"\rNo shell available, client must supply command on command line\r",0);
    } else if ( tu_strncmp ( token[0], "SHOW", length ) == 0 ) {
	/*
	 * Show command has options specified in token[1].
	 */
	if ( j < 2 ) token[1] = "";
	else tu_strupcase ( token[1], token[1] );
	length = tu_strlen ( token[1] );
	if ( length == 0 ) length = 1;
	if ( tu_strncmp ( token[1], "SESSIONS", length ) == 0 ) {
	    output_line ( ctx, "Active sessions:", 0 );
	    ssh_scan_sessions ( display_session, ctx );
	} else {
	    output_line ( ctx,
		"Missing or unknown SHOW option", 0 );
	}
    } else if ( tu_strncmp ( token[0], "SHUTDOWN", length ) == 0 ) {
	monitor_cmd = "SHUTDOWN";	/* notify monitor */
	if ( j > 1 ) {	/* test for restart option */
	    length = tu_strlen ( token[1] );
	    tu_strupcase ( token[1], token[1] );
	    if ( length == 0 ) length = 1;
	    if (tu_strncmp(token[1], "RESTART", length) == 0)  
		monitor_cmd = "SHUTDOWN_RESTART";
	}
    } else if ( tu_strncmp ( token[0], "CLOSE", length ) == 0 ) {
	monitor_cmd = "CLOSE";
	if ( j > 1 ) {
	    length = tu_strlen ( token[1] );
	    tu_strupcase ( token[1], token[1] );
	    if ( length == 0 ) length = 1;
	    if (tu_strncmp(token[1], "RESTART", length) == 0)  
		monitor_cmd = "CLOSE_RESTART";
	}
    } else {
	output_line ( ctx, 
		"Unknown command, use HELP to get command summary", 0 );
    }
    /*
     * Commands that affect the monitor thread (shutdown, close) do so
     * by writing a message to pipe stream that the thread is reading from.
     * The monitor thread processes the command and signals completion by
     * calling execadmin_done
     */
    if ( ctx->monitor && monitor_cmd ) {
	int xfer, status;
        /*
         * Allow only one /admin connection at a time to send messages
         * to the monitor thread, admin_owner is address of owning
	 * connection's private context.
         */
	pthread_mutex_lock ( &admin_lock );
	while ( admin_owner ) pthread_cond_wait ( &admin_done, &admin_lock );
	admin_owner = ctx;
        /*
         * Send message and wait for execadmin_done to clear flag.
         * Use broadcast since more than one predicate is using the same
         * condition variable.
         */
	admin_pending = 1;
        status = cport_do_io ( ctx->monitor, CPORT_WRITE, monitor_cmd, 
		tu_strlen(monitor_cmd),	&xfer );
	while ( admin_pending ) pthread_cond_wait (&admin_done, &admin_lock);
	admin_owner = (struct private_st *) 0;
	pthread_cond_broadcast ( &admin_done );
	pthread_mutex_unlock ( &admin_lock );
    }
    /*
     * Mark end of output, connection will close after sending EOF mark.
     */
    output_line ( ctx, "", 1 );	/* EOF */
}
/**************************************************************************/
/* The write opration buffers written characters until a newline seen,
 * at which point it processes the command.
 */
/*
 * Copy data from buffer to ctx->input_buffer until a newline
 * character encountered or pending_in is full, then update pending_in
 * and pend_in_size to point to remainder of buffer.  Return value is 0 
 * if newline not seen.
 */
static int parse_input_data ( 
	struct private_st *ctx, char *buffer, int bufsize )
{
    int i, pos;
    char c;

    pos = ctx->input_pos;
    for ( i = 0; bufsize; i++ ) {
	c = buffer[i];
	if ( (c == '\n') || (pos >= MAX_INPUT_LINE) ) {
	    /*
	     * Checkpoint position and flag that next output needed.
	     */
	    ctx->input_pos = pos;
	    if ( c == '\n' ) i++;	/* skip the newline */
	    ctx->pend_in_size = (ctx->pend_in_size-i);
	    ctx->pending_in = &buffer[i];
	    return 1;		/* Complete command seen */
	}
	else ctx->input_buffer[pos++] = c;
    }
    /*
     * Buffer all loaded and no newlines, leave in in_buffer.
     */
    ctx->input_pos = pos;
    ctx->pend_in_size = 0;
    ctx->pending_in = (char *) 0;

    return 0;
}

static int start_write ( cport_isb isb, int func, void *buffer, int bufsize )
{
    struct private_st *ctx;
    int status;

    ctx = isb->drv;
    /*
     * Get data.
     */
    if ( bufsize == 0 ) {
	/*
	 * Special case, write EOF.
	 */
	isb->default_iosb[0] = 1;
	isb->default_iosb[1] = 0;
	status = cport_mark_completed ( isb, 0 );
	return status;
    }
    ctx->pending_in = (char *) buffer;
    ctx->pend_in_size = bufsize;
    while ( parse_input_data (ctx, ctx->pending_in, ctx->pend_in_size ) ) {
	/*
	 * Input buffer now contains a complete command, process it.
	 */
	ctx->input_buffer[ctx->input_pos] = '\0';
        execute_command ( ctx );
	ctx->input_pos = 0;	/* reset buffer */
    }
    /*
     * Fill in IOSB with success status, all data written and complete I/O.
     */
    isb->iosb = (void *) isb->default_iosb;
    isb->default_iosb[0] = 1;
    isb->default_iosb[1] = bufsize;
    status = cport_mark_completed ( isb, 0 );
    return status;
}
/**************************************************************************/
/* The read operation returns the (text) data generated as a result of
 * processing the command.  If the queue it reads from is empty, block
 * completion of the I/O so it can be resumed by the next write operation.
 */

static int start_read ( cport_isb isb, int func, void *buffer, int bufsize )
{
    int status, segsize, transferred;
    struct private_st *ctx;

    ctx = isb->drv;
    /*
     * Return characters from remaining input or start new read from
     * network.
     */
    if ( ctx->output_queue ) {
	/*
	 * Return queued data to client.
	 */
	isb->iosb = (void *) isb->default_iosb;
	isb->default_iosb[0] = 1;
	if ( ctx->output_queue->length >= 0 ) {
	    isb->default_iosb[1] = read_from_out_buf(ctx, 
			(char *)buffer, bufsize);
	} else {
	    /* End of file. */
	    isb->default_iosb[1] = 0;
	}
	status = cport_mark_completed ( isb, 0 );
    } else {
	/*
	 * Client hasn't sent command yet, wait for it.
	 */
	ctx->blocked_isb = isb;
	isb->iosb = (void *) isb->default_iosb;
	isb->default_iosb[0] = isb->default_iosb[1] = 0;
	return cport_mark_busy ( isb );
    }

    return status;
}
/*****************************************************************************/
/* The change window function has no meaning, but include for compatibility
 * with the cmd and pty modes.
 */
static int change_window ( cport_isb isb, int func, void *buffer, int length )
{
    struct private_st *pctx;
    /*
     * Since we aren't really a terminal, do nothing about geometry changes.
     */
    isb->default_iosb[0] = 1;
    return cport_mark_completed ( isb, 0 );
}

static int query_exit_status ( cport_isb isb, int func, 
	void *buffer, int length )
{
    *((int *) buffer) = isb->drv->exitstatus;
    isb->default_iosb[0] = 1;
    isb->default_iosb[1] = 4;
    return 1;			/* synchronous completion */
}
/****************************************************************************/
/* Declare the functions used in in the stream handler table.
 */
static cport_start_function ftable[4] = {
    start_write,
    start_read,
    change_window,		/* Change terminal geometry */
    query_exit_status		/* return exit status */
};

static int new_stream ( cport_isb isb, void *context, int dir,
	char errmsg[256] )
{
    struct private_st *ctx;

    isb->drv = (struct private_st *) context;
    ctx = isb->drv;

    ctx->ref_count++;
    /* isb->channel = (dir=0)? ctx->mbx : ctx->net; */
    return 1;
}

static int destroy_stream ( cport_isb isb )
{
    int status;
    struct private_st *ctx;

    ctx = isb->drv;
    isb->drv->ref_count--;
    if ( isb->drv->ref_count <= 0 ) {
        struct mbx_iosb iosb;
	/*
	 * Rundown the pipe used to talk to the monitor thread.
         */
	if ( ctx->monitor ) cport_deassign ( ctx->monitor );
	ctx->monitor = (cport_isb) 0;
	if ( ctx->port ) cport_destroy_port ( ctx->port );
	ctx->port = (cport_port) 0;
        /*
         * Rundown pthreads objects and free allocated memory.
         */
        tm_free ( ctx );
        isb->drv = (struct private_st *) 0;
        return 1;
    }
    else status = 1;
    isb->channel = 0;
    return status;
}

static int cancel_io ( cport_isb isb )
{
    struct private_st *ctx;
    int status;
    /*
     * Cancel the active QIO's
     */
    ctx = isb->drv;
    if ( isb->flags == 1 ) {
	isb->flags = 0;
    }
    return 1;
}
/*
 * The cportucx_driver table will be specified in cport_assign_stream
 * calls as the handler argument.
 */
cport_stream_handler cportadmin_driver = {
	3,					/* mask 3 => 4 functions */
	ftable,
	new_stream,
	destroy_stream,
	cancel_io
};
