/*
 * This module provides support for executing SSH_CMSG_EXEC_CMD-style
 * commands for an arbitrary user.  The command is actually executed
 * by a network process created with the username and password supplied
 * by the caller.  The command process reads its input from a mailbox
 * and directs output to the network connection (SYS$NET).
 *
 * Input and output streams for the command process are accessed via the
 * 'async_terminal' asynchronous interface.  The open function initializes
 * a structure with the following function pointers:
 *     start_io ( execcmd, int direction, buffer, size );
 *
 * direction flag:
 *    1		channel to process's sys$Input (write to this stream)
 *    0		channel to process's sys$Output (read from this stream).
 *
 * the caller must also initialize the async_terminal structure with
 * a callback function to signal when a poll operation would be worthwhile.
 *
 * Execution of user commands by SSH daemon:
 *
 *    1. Open connection (N) to network task 0"user password"::"0=sshexec".
 *    2. create directional mailbox (M), write only. Add acl to permit
 *       read access from target user's UIC.
 *    3. Write mailbox name to N as message.
 *    4. read boundary tag from N.
 *    5. Write command to execute to M.
 *        +-- Write stdin data to M, finish with EOF.
 *    6. -+                                                  > (in parallel)
 *        +-- Read stdout data from N until bound. tag read.
 *    7. Delete mailbox.
 *    8. Close network connection.
 *
 * SSHEXEC.COM network task:
 *    1. Open sys$net /read/write.
 *    2. Read mailbox name from network link.
 *    3. open mailbox /read
 *    4. write boundary tag string to network link.
 *    5. read command to be executed from mailbox.
 *    6. assign the mailbox and network link to sys$input and sys$output.
 *    7.  Execute command.
 *    8 write boundary tag to network link.
 *    9 close mailbox and network link.
 *
 * Author:	David Jones
 * Date:	12-MAY-1998
 * Revised:	23-SEP-1998		! bug in new_stream() dir interpretation.
 * Revsed:	25-SEP-1998		! force detached process deletes.
 * Revised:	22-OCT-1998		Supply pty_map data.
 */
#include <stdio.h>
#include <stdlib.h>
#include <iodef.h>			/* QIO symbolic definitions */
#include <cmbdef.h>			/* mailbox function codes */
#include <dvidef.h>
#include <descrip.h>			/* VMS string descriptors */
#include <acedef.h>			/* VMS access control lists */
#include <ossdef.h>			/* VMS object security services */
#include <string.h>

#include "pthread_1c_np.h"
#include "tutil.h"
#include "tmemory.h"
#define CPORT_DRIVER_SPECIFIC struct private_st
#include "completion_port.h"
#include "parameters.h"
#include "user_info.h"
#include "pty_map.h"
#include "initiator_client.h"

int SYS$CREMBX(), SYS$QIO(), SYS$ASSIGN(), SYS$GETDVI(), SYS$CANCEL();
int LIB$GET_EF(), SYS$DASSGN(), SYS$SET_SECURITY(), SYS$DELPRC();

static pthread_once_t execcmd_once = PTHREAD_ONCE_INIT;

struct mbx_iosb {
    unsigned short status, count;
    int pid;
};
#define MAX_INPUT_LINE 512
#define MAX_OUTPUT_LINE 4096
/*
 * A private_st structure is allocated for each open asynch_stream and is
 * pointed to by the asynch_stream structure's .private member.
 */
struct private_st {
    int net;			/* VMS I/O channel to DECnet link */
    int mbx;			/* channel to mailbox */
    int ref_count;
    int bndtag_len;		/* length of boundary tag */
    int exitstatus;		/* Command exit status. */
    int input_pos;		/* Current buffer position */
    int output_pos;		/* */
    int pend_in_size;		/* Bytes of pending input remaining */
    int pend_out_size;
    struct mbx_iosb in_iosb, out_iosb;
    int pid;
    char *pending_in;
    char *pending_out;
    char *boundary_tag;		/* Boundary tag to mark output */
    pthread_mutex_t lock;
    pthread_cond_t io_done;
    char input_buffer[MAX_INPUT_LINE];
    char output_buffer[MAX_OUTPUT_LINE];
};
static $DESCRIPTOR(template_dx,"");
static $DESCRIPTOR(decnet_device0,"_NET:");
static $DESCRIPTOR(security_class,"DEVICE");

/*
 * One-time initialization for module: Initialize per-thread heap and
 * create AST-to-Thread queue for running AST in a normal thread context.
 */
static void execcmd_init()
{
    tm_initialize();
}
/******************************************************************************/
/*
 * Fill in error message and rundown private context.
 */
static int abort_open ( struct private_st *pctx, 
	char *msg, int status, char *errmsg )
{
    tu_strcpy ( errmsg, msg );
    if ( status != 0 ) {
	tu_strint ( status, &errmsg[tu_strlen(errmsg)] );
    }
    pthread_mutex_destroy ( &pctx->lock );
    pthread_cond_destroy ( &pctx->io_done );
    if ( pctx->net != -1 ) SYS$DASSGN ( pctx->net );
    if ( pctx->mbx != -1 ) SYS$DASSGN ( pctx->mbx );
    if ( pctx->boundary_tag ) tm_free ( pctx->boundary_tag );
    tm_free ( pctx );
    return status;
}
/*
 * Synchronization routine for QIO calls.  Specify io_synch as the astadr
 * and pthread_cond_wait_int_np.
 */
int io_synch ( int status, struct private_st *pctx, struct mbx_iosb *iosb )
{
    if ( (status&1) == 1 ) {
        pthread_mutex_lock ( &pctx->lock );
	while ( iosb->status == 0 ) {
	    pthread_cond_wait ( &pctx->io_done, &pctx->lock );
	}
	status = iosb->status;
        pthread_mutex_unlock ( &pctx->lock );
    }
    return status;
}

static int protected_mailbox ( struct private_st *pctx,
	int maxmsg, long uic, int flags,
	int *chan, char mbx_name[64] )
{
    int status, mbxname_len;
    struct mbx_iosb iosb;
    struct { short length, code; void *buffer; int *retlen;} item[2];
    /*
     * Create mailbox to be used for processes input and get it's name.  
     * Protect from world access.
     */
    *chan = 0;
    status = SYS$CREMBX ( 0, chan, maxmsg, maxmsg, 0xff00, 0, 0, flags );
    if ( (status&1) == 1 ) {
	item[0].length = 63;
	item[0].code = DVI$_DEVNAM;
	item[0].buffer = mbx_name; 
	item[0].retlen = &mbxname_len;
	item[1].length = item[1].code = 0;

	mbxname_len = 0;
	status = SYS$GETDVI ( 0, *chan, 0, &item, &iosb, 0, 0, 0 );
	status = io_synch ( status, pctx, &iosb );
	if ( (status&1) == 0 ) SYS$DASSGN ( *chan );
	mbx_name[mbxname_len] = 0;
    }
    /*
     * Add ACL to mailbox to permit the target user access.
     */
    if ( (status&1) == 1 && (uic != 0) ) {
	struct {
	    unsigned char length, type;
	    unsigned short flags;
	    unsigned long access;
	    long identifier;
	} ace;
	ace.type = ACE$C_KEYID;
	ace.length = sizeof(ace);
	ace.flags = 0;
	ace.access = 15;		/* full access */
	ace.identifier = uic;

	item[0].length = sizeof(ace);
	item[0].code = OSS$_ACL_ADD_ENTRY;
	item[0].buffer = (void *) &ace;
	item[0].retlen= (int *) 0;
	item[1].length = item[1].code = 0;

	status= SYS$SET_SECURITY ( 
		&security_class, 0, chan, 0, &item, 0, 0 );
	if ( (status&1) == 0 ) SYS$DASSGN ( *chan );
    }
    return status;
}
/*************************************************************************/
/* Main routine for creating a stream.  A network process is created
 * under the given username and password and the stream context is
 * initialized for asynchrounous communication with this process.
 */
int execcmd_open ( void **ctx, 		/* return context pointer */
	char *object,				/* e.g. "0=SSHEXEC" */
	struct user_account_info *uai,		/* Username/password */
	char *x11_info,				/* port forwarding info */
	char errmsg[256] )
{
    int status, ncblen, pwd_len, obj_len;
    struct { int length; char *data; } ncb_desc;
    struct mbx_iosb iosb;
    struct private_st *pctx;
    struct { short length, code; void *buffer; int *retlen;} item[2];
    char mbx_name[64], ncb[128];
    char boundary_tag[2048];
    int bndtag_len;
    /*
     * Make sure module initialized.
     */
    pthread_once ( &execcmd_once, execcmd_init );
    /*
     * Allocate private structure from thread-private heap (automatically
     * freed on thread exit.
     */
    pctx = (struct private_st *) tm_malloc ( sizeof(struct private_st) );
    if ( !pctx ) {
	tu_strcpy ( errmsg, "memory allocation failure" );
	return 0;
    }
    pctx->net = -1;
    pctx->mbx = -1;
    pctx->exitstatus = 0;
    pctx->boundary_tag = (char *) 0;
    pctx->input_pos = 0;
    pctx->output_pos = 0;
    pctx->pid = 0;
    pctx->out_iosb.count = 0;
    pctx->out_iosb.status = 0;
    pctx->in_iosb.count = 0;
    pctx->in_iosb.status = 0;
    INITIALIZE_MUTEX ( &pctx->lock );
    INITIALIZE_CONDITION ( &pctx->io_done );
    /*
     * Build network connect block used to connect to task.
     *   0"~ ~"::"0=~/xxc1234567890123456"
     *   12 3 456789 012345678901234567890 = 30 + 39 + 32
     */
    ncb[0] = '0';
    ncb[1] = '"';
    tu_strnzcpy ( &ncb[2], uai->username, sizeof(ncb) );
    ncblen = tu_strlen ( ncb );
    pwd_len = tu_strlen(uai->password);
    obj_len = tu_strlen(object);
    if ( (obj_len + pwd_len + ncblen + 32) > sizeof(ncb) ) {
	return abort_open ( pctx, "invalid arguments", 0, errmsg );
    } else if ( pwd_len > 0 ) {
        ncb[ncblen++] = ' ';
	tu_strcpy ( &ncb[ncblen], uai->password ); ncblen += pwd_len;
    }
    tu_strcpy ( &ncb[ncblen], "\"::\"" );
    ncblen += 4;
    tu_strcpy ( &ncb[ncblen], object ); ncblen += obj_len;
    ncb[ncblen++] = '/';
    ncb[ncblen++] = 0;  ncb[ncblen++] = 0; /* zero word, (new connection) */
    ncb[ncblen++] = 15;			/* count of optional data */
    tu_strcpy ( &ncb[ncblen], "SSH.1.2.OSU-1.0" );   /* 15 chars + 0 */
    ncblen += 16;
    ncb[ncblen++] = '"';		/* closing quote */
    /*
     * Assign channel to network device and connect to task.
     */
    errmsg[0] = '\0';
    if ( param.decnet_cmd ) {
        status = SYS$ASSIGN ( &decnet_device0, &pctx->net, 0, 0, 0 );
	if ( (status&1) == 1 ) {
	    ncb_desc.length = ncblen;
	    ncb_desc.data = ncb;
	    status = SYS$QIO ( 0, pctx->net, IO$_ACCESS, &iosb, 
		pthread_cond_signal_int_np, &pctx->io_done,
		0, &ncb_desc, 0, 0, 0,0 );
	    status = io_synch ( status, pctx, &iosb );
            if ( (status&1) == 1 ) status = iosb.status;
	 }
    } else {
	/*
	 * Create mailbox and detached process.
	 */
	status = protected_mailbox ( pctx, MAX_OUTPUT_LINE, uai->uic, 0,
		&pctx->net, mbx_name );
        if ( (status&1) == 0 ) return abort_open ( pctx, 
	    "Error creating output mailbox, code: ", status, errmsg );
        /*
	 * Create detached process.
	 */
	status = initiate_user_process ( uai, 0, mbx_name, "nl:",
		"", x11_info, errmsg );
	if ( (status&1) == 0 ) return abort_open ( pctx,
		"Error creating user process, code: ", status, errmsg );
	/*
	 * Set initial command.
	 */
	status = SYS$QIO ( 0, pctx->net, IO$_WRITEVBLK, &iosb, 
		pthread_cond_signal_int_np, &pctx->io_done,
		param.cmd_task, tu_strlen(param.cmd_task), 0, 0, 0, 0 );
	status = io_synch ( status, pctx, &iosb );
	if ( (status&1) == 0 ) return abort_open ( pctx,
		"network write error, code: ", status, errmsg );
    }
    if ( (status&1) == 0 ) return abort_open ( pctx, 
	"Error openning network connection, code: ", status, errmsg );

    status = protected_mailbox ( pctx, MAX_INPUT_LINE, uai->uic, CMB$M_WRITEONLY,
		&pctx->mbx, mbx_name );
    if ( (status&1) == 0 ) return abort_open ( pctx, 
	"Error creating input mailbox, code: ", status, errmsg );

    /*
     * Write mailbox name to network link
     */
    status = SYS$QIO ( 0, pctx->net, IO$_WRITEVBLK, &iosb, 
	pthread_cond_signal_int_np, &pctx->io_done,
	mbx_name, tu_strlen(mbx_name), 0, 0, 0, 0 );
    status = io_synch ( status, pctx, &iosb );
    if ( (status&1) == 0 ) return abort_open ( pctx,
	"network write error, code: ", status, errmsg );
    /*
     * read boundary tag from network, save PID.
     */
    status = SYS$QIO ( 0, pctx->net, IO$_READVBLK, &iosb, 
	pthread_cond_signal_int_np, &pctx->io_done,
	pctx->output_buffer, sizeof(pctx->output_buffer), 0, 0, 0, 0 );
    status = io_synch ( status, pctx, &iosb );
    if ( (status&1) == 0 ) return abort_open ( pctx,
	"error reading boundary tag, code: ", status, errmsg );
    else if ( !param.decnet_cmd ) pctx->pid = iosb.pid;

    pctx->bndtag_len = iosb.count+8;	/* include exit status */
    pctx->boundary_tag = (char *) tm_malloc ( iosb.count+1 );
    memcpy ( pctx->boundary_tag, pctx->output_buffer, iosb.count );
    pctx->boundary_tag[iosb.count] = '\0';

    *ctx = (void *) pctx;
    return status;
}
/**************************************************************************/
/* The write opration buffers written characters until a newline seen,
 * at which point it writes the line to the input mailbox.
 */
/*
 * Copy data from buffer to ctx->input_buffer until a newline
 * character encountered or pending_in is full, then update pending_in
 * and pend_in_size to point to remainder of buffer.  Return value is 0 
 * if newline not seen.
 */
static int parse_input_data ( 
	struct private_st *ctx, char *buffer, int bufsize )
{
    int i, pos;
    char c;

    pos = ctx->input_pos;
    for ( i = 0; bufsize; i++ ) {
	c = buffer[i];
	if ( (c == '\n') || (pos >= MAX_INPUT_LINE) ) {
	    /*
	     * Checkpoint position and flag that next output needed.
	     */
	    ctx->input_pos = pos;
	    if ( c == '\n' ) i++;	/* skip the newline */
	    ctx->pend_in_size = (ctx->pend_in_size-i);
	    ctx->pending_in = &buffer[i];
	    return 1;
	}
	else ctx->input_buffer[pos++] = c;
    }
    /*
     * Buffer all loaded and no newlines, leave in in_buffer.
     */
    ctx->input_pos = pos;
    ctx->pend_in_size = 0;
    ctx->pending_in = (char *) 0;

    return 0;
}
static int finish_write ( cport_isb isb )
{
    struct private_st *ctx;
    int status;

    isb->flags = 0;
    ctx = isb->drv;
    ctx->input_pos = 0;		/* reset buffer position */
    if ( (ctx->in_iosb.status&1) == 1 && (ctx->pend_in_size > 0) ) {
	/*
	 * Parse next line out of pending input.
	 */
	if ( parse_input_data (ctx, ctx->pending_in, ctx->pend_in_size ) ) {
	    isb->flags = 1;		/* mark QIO in progress for cancels */
	    status = SYS$QIO ( 0, ctx->mbx, IO$_WRITEVBLK, &ctx->in_iosb,
		ctx->input_buffer, ctx->input_pos, 0, 0, 0, 0 );
	    if ( (status&1) == 1 ) return CPORT_COMPLETE_BUSY;
	    /*
	     * Save error condition.
	     */
	    isb->flags = 0;
	    ctx->in_iosb.status = status;
	}
    }
    /*
     * All complete records transferred or write failed, fill in iosb and 
     * return control to caller.
     */
    isb->iosb = (void *) isb->default_iosb;
    isb->default_iosb[0] = ctx->in_iosb.status;
    isb->default_iosb[1] = isb->length;

    return CPORT_COMPLETE_PROCESSED;
}
static int start_write ( cport_isb isb, int func, void *buffer, int bufsize )
{
    struct private_st *ctx;
    int status;

    ctx = isb->drv;
    /*
     * Get data.
     */
    if ( bufsize == 0 ) {
	/*
	 * Special case, write EOF.
	 */
	isb->completion_callback = finish_write;
	isb->iosb = (void *) &ctx->in_iosb;
	isb->flags = 1;		/* mark QIO in progress for cancels */
	status = SYS$QIO ( 0, ctx->mbx, IO$_WRITEOF|IO$M_READERCHECK, 
		&ctx->in_iosb, cport_completion_ast, isb, 0, 0, 0, 0, 0, 0 );
	if ( (status&1) == 0 ) isb->flags = 0;
	else status = cport_mark_busy ( isb );
	return status;
    }
    if ( parse_input_data (ctx, (char *) buffer, bufsize ) ) {
	isb->completion_callback = finish_write;
	isb->iosb = (void *) &ctx->in_iosb;
	isb->flags = 1;		/* mark QIO in progress for cancels */
	status = SYS$QIO ( 0, ctx->mbx, IO$_WRITEVBLK|IO$M_READERCHECK, 
		&ctx->in_iosb, cport_completion_ast, isb,
		ctx->input_buffer, ctx->input_pos, 0, 0, 0, 0 );
	if ( (status&1) == 0 ) isb->flags = 0;
	else status = cport_mark_busy ( isb );
	return status;
    }
    /*
     * Data written did not end in newline and is all in input_buffer awaiting
     * more to be written.
     */
    isb->default_iosb[0] = 1;
    isb->default_iosb[1] = bufsize;
    status = cport_mark_completed ( isb, 0 );
    return status;
}
/**************************************************************************/
/* The read operation is buffers data records read from the network connection
 * and returns data to caller as a byte stream.  A newline is appended
 * to the end of each newtwork buffer read.
 */
static int read_from_out_buf ( struct private_st *ctx, 
	char *buffer, int bufsize )
{
    /*
     * Copy contents of ctx->pending_out to specified buffer and update
     * ctx->output_pos.
     */
    int segsize, transferred;
    segsize = ctx->out_iosb.count - ctx->output_pos - 1;
    if ( segsize > bufsize ) segsize = bufsize;

    memcpy ( buffer, &ctx->output_buffer[ctx->output_pos], segsize );
    ctx->output_pos += segsize;
    transferred = segsize;
    if ( (segsize < bufsize) && 
	((ctx->output_pos+1) == ctx->out_iosb.count) ) {
	/*
	 * actualize the virtual newline.
	 */
	buffer[transferred++] = '\n';
	ctx->output_pos++;
	return segsize+1;
    }
    return segsize;
}
/*
 * Driver completion routine.
 */
static int finish_read ( cport_isb isb )
{
    struct private_st *ctx;
    int segsize, i, value;
    /*
     * Check QIO completion status in IOSB.
     */
    ctx = isb->drv;
    isb->flags = 0;
    isb->iosb = (void *) isb->default_iosb;
    if ( (ctx->out_iosb.status&1) == 0 ) {
	/* Error during read. */
	isb->default_iosb[0] = ctx->out_iosb.status;
	return CPORT_COMPLETE_PROCESSED;
    }
    /*
     * Check for End-of-file boundary marker.
     */
    if ( ctx->out_iosb.count == ctx->bndtag_len ) {
	/*
	 * last 8 chars of boundary tag are completion code, dont' check.
	 */
	if ( tu_strncmp ( ctx->boundary_tag, ctx->output_buffer,
		ctx->bndtag_len-8 ) == 0 ) {
	    /*
	     * decode the exit status (hexadecimal digits).
	     */
	    char c;
	    value = 0;
	    for ( i = ctx->bndtag_len-8; i < ctx->bndtag_len; i++ ) {
		c = ctx->output_buffer[i];
		if ( c >= '0' && c <= '9' ) value = (value * 16) + (c-'0');
		else if (c >= 'A' && c <= 'F') value = (value*16)+(c-'A')+10;
		else break;
	    }
	    ctx->exitstatus = value;
	    /*
	     * Mark eof.
	     */
	    isb->default_iosb[0] = 1;
	    isb->default_iosb[1] = 0;
    	    return CPORT_COMPLETE_PROCESSED;		/* complete I/O */
	}
    }
    /*
     * Reset buffer position and append virtual newline.
     */
    ctx->output_pos = 0;
    ctx->out_iosb.count++;
    /*
     * transfer data.
     */
    isb->default_iosb[0] = 1;
    isb->default_iosb[1] = read_from_out_buf ( 
		ctx, ctx->pending_out, ctx->pend_out_size );
    if ( isb->default_iosb[1] == 0 ) fprintf(stderr,"False EOF on stream\n");
    return CPORT_COMPLETE_PROCESSED;		/* complete I/O */
}


static int start_read ( cport_isb isb, int func, void *buffer, int bufsize )
{
    int status, segsize, transferred;
    struct private_st *ctx;

    ctx = isb->drv;
    /*
     * Return characters from remaining input or start new read from
     * network.
     */
    if ( ctx->output_pos < ctx->out_iosb.count ) {
	isb->iosb = (void *) isb->default_iosb;
	isb->default_iosb[0] = 1;
	isb->default_iosb[1] = read_from_out_buf(ctx, (char *)buffer, bufsize);
	status = cport_mark_completed ( isb, 0 );
    } else {
	isb->flags = 1;
	isb->completion_callback = finish_read;
	ctx->pending_out = buffer;
	ctx->pend_out_size = bufsize;
	isb->iosb = (void *) &ctx->out_iosb;
	status = SYS$QIO ( 0, ctx->net, IO$_READVBLK, &ctx->out_iosb,
		cport_completion_ast, isb,
		ctx->output_buffer, sizeof(ctx->output_buffer), 0, 0, 0, 0 );
	if ( (status&1) == 0 ) {
	    isb->flags = 0;
	    return status;
	}
	status = cport_mark_busy ( isb );
    }

    return status;
}
static int change_window ( cport_isb isb, int func, void *buffer, int length )
{
    struct private_st *pctx;
    /*
     * Since we aren't really a terminal, do nothing about geometry changes.
     */
    isb->default_iosb[0] = 1;
    return cport_mark_completed ( isb, 0 );
}

static int query_exit_status ( cport_isb isb, int func, 
	void *buffer, int length )
{
    *((int *) buffer) = isb->drv->exitstatus;
    isb->default_iosb[0] = 1;
    isb->default_iosb[1] = 4;
    return 1;			/* synchronous completion */
}
/****************************************************************************/
/* Declare the functions used in in the stream handler table.
 */
static cport_start_function ftable[4] = {
    start_write,
    start_read,
    change_window,		/* Change terminal geometry */
    query_exit_status		/* return exit status */
};

static int new_stream ( cport_isb isb, void *context, int dir,
	char errmsg[256] )
{
    struct private_st *ctx;

    isb->drv = (struct private_st *) context;
    ctx = isb->drv;

    ctx->ref_count++;
    isb->channel = (dir==1)? ctx->mbx : ctx->net;
    return 1;
}

static int destroy_stream ( cport_isb isb )
{
    int status;
    struct private_st *ctx;

    ctx = isb->drv;
#ifdef DEBUG
printf("Destroy cmd stream called on %x (drv=%x), ref count: %d\n", isb,
isb->drv, isb->drv->ref_count );
#endif
    isb->drv->ref_count--;
    if ( isb->drv->ref_count <= 0 ) {
        struct mbx_iosb iosb;
	/*
	 * No more streams attached.  Send EOF to mailbox to help ensure 
	 * network process goes away, then deasign the I/O channels.
         */
        status = SYS$QIO ( 0, ctx->mbx, IO$_WRITEOF|IO$M_READERCHECK|IO$M_NOW, 
	&iosb, pthread_cond_signal_int_np, &ctx->io_done,
	0, 0, 0, 0, 0, 0 );
        status = io_synch ( status, ctx, &iosb );
        SYS$DASSGN ( ctx->mbx );
        status = SYS$DASSGN ( ctx->net );
	/*
	 * If a detached process, force kill.
	 */
	if ( ctx->pid != 0 ) {
	    status = SYS$DELPRC ( &ctx->pid, 0 );
	}
        /*
         * Rundown pthreads objects and free allocated memory.
         */
        pthread_mutex_destroy ( &ctx->lock );
        pthread_cond_destroy ( &ctx->io_done );
        if ( ctx->boundary_tag ) tm_free ( ctx->boundary_tag );
        tm_free ( ctx );
        isb->drv = (struct private_st *) 0;
        return 1;
    }
    else status = 1;
    isb->channel = 0;
    return status;
}

static int cancel_io ( cport_isb isb )
{
    struct private_st *ctx;
    int status;
    /*
     * Cancel the active QIO's
     */
    ctx = isb->drv;
    if ( isb->flags == 1 ) {
	isb->flags = 0;
	SYS$CANCEL ( isb->channel );
    }
    return 1;
}
/*
 * The cportucx_driver table will be specified in cport_assign_stream
 * calls as the handler argument.
 */
cport_stream_handler cportcmd_driver = {
	3,					/* mask 3 => 4 functions */
	ftable,
	new_stream,
	destroy_stream,
	cancel_io
};
