/*
 * Example driver for completion_port.
 *
 * Author: David Jones
 * Date:   2-JUN-1998
 * Revised: 25-NOV-1998		Make internal routines static.
 */
#include <descrip.h>
#include <iodef.h>
#include <ssdef.h>

#include <stdlib.h>
#include <stdio.h>
#include "pthread_1c_np.h"
/*
 * The cport_isb structure reserves the field '->drv' for use  by drivers,
 * which completion_port.h defines this field as type void * by default.
 * We override the void * type to be a pointer to our private data
 * structure by defining the symbol CPORT_DRIVER_SPECIFIC prior to
 * including completion_port.h.  The driver module can now refer to
 * field x in the private structure as isb->drv->x without cumbersome
 * casts.  Field drv is always a pointer.
 */
#define CPORT_DRIVER_SPECIFIC struct private_st
#include "completion_port.h"
/*
 * Define structure to hold the driver-specific context information.
 * This structure is completely private to the driver and must
 * be allocated/deallocated by the driver.
 */
struct private_st {
    struct private_st *next;
    int ref_count;		/* # of streams assigned to this context */
    int channel;		/* Channel assigned to null device. */
    int wildcard_syi;		/* flag if wildcard scan in progress. */
    unsigned long csid;		/* Current Cluster System ID for getsyi */
};
/*
 * Define internal routines for maintaining lookaside list of previously
 * allocated (and released) private structs.  Also make once routine.
 */
static pthread_once_t module_setup = PTHREAD_ONCE_INIT;
static pthread_mutex_t module_data;
static struct private_st *free_ctx;
static $DESCRIPTOR(null_device,"_NL:");
int SYS$ASSIGN(), SYS$DASSGN(), SYS$GETTIM(), SYS$GETSYI();

static void module_init ( )
{
    INITIALIZE_MUTEX ( &module_data );
    free_ctx = (struct private_st *) 0;
}

#define GROUP 4
static struct private_st *allocate_context()
{
    struct private_st *new;
    int i;
    struct iblock { struct private_st instance[GROUP]; } *block;
    /*
     * Allocate new block.
     */
    pthread_mutex_lock ( &module_data );
    if ( !free_ctx ) {
	/*
	 * Grow list allocate multiple.
	 */
	block = (struct iblock *) malloc ( sizeof(struct iblock) );
	if ( block ) {
	    /*
	     * Do one-time initialization for block members (mutexes, etc)
	     * and link into free list.
	     */
	    for ( i = 0; i < GROUP; i++ ) {
		block->instance[i].next = &block->instance[i+1];
	    }
	    block->instance[GROUP-i].next = (struct private_st *) 0;
	    free_ctx = &block->instance[0];

	} else {
	    /*
	     * Serious error, allocation failures.
	     */
	    pthread_mutex_unlock ( &module_data );
	    return free_ctx;
	}
    }
    new = free_ctx;
    free_ctx = new->next;
    pthread_mutex_unlock ( &module_data );
    /*
     * Set allocated context to initial state.
     */
    new->channel = 0;
    new->ref_count = 0;
    new->wildcard_syi = 0;
    new->csid = 0;

    return new;
}

static void deallocate_context ( struct private_st *ctx )
{
    /*
     * Simply push context onto lookaside list.
     */
    pthread_mutex_lock ( &module_data );
    ctx->next = free_ctx;
    free_ctx = ctx;
    pthread_mutex_unlock ( &module_data );
}
/******************************************************************************/
/* The open function creates a private I/O context.  The arguments are
 * completely driver-specific and include anything required for the
 * initialization of the private portion of the I/O context.  The open
 * function must return in one form or another a (void *) that points
 * to a driver context, to be used by the caller as the context argument
 * for subsequent calls to cport_assign_stream.
 */
int cport_example_open ( void **context )
{
    struct private_st *ctx;
    int status;
    /*
     * Initialialize module.
     */
    pthread_once ( &module_setup, module_init );
    /*
     * Allocate context.
     */
    ctx = (void *) allocate_context();
    if ( !ctx ) return SS$_INSFMEM;
    /*
     * Allocate a channel to the null device.
     */
    status = SYS$ASSIGN ( &null_device, &ctx->channel, 0, 0, 0 );
    if ( (status&1) == 0 ) {
	deallocate_context ( ctx );
	ctx = (struct private_st *) 0;
    }
    *context = (void *) ctx;
    return status;
}
/******************************************************************************/
/* The new_stream function is called by cport_assign_stream (via the .attach)
 * function pointer) to bind a newly created isb to the i/o context pointer 
 * returned by the driver's open function.
 *
 * Arguments:
 *    isb	New synchronization block to be assigned to context.
 *    context	I/O context returned by cport_example_open();
 *    opt	Assignment flags/options, unused.
 *    errmsg	Receives error text if assign fails.
 *
 * Return value:
 *    1		Normal, successful completion.
 *   even	Error.
 */
static int new_stream (cport_isb isb, void *context, int opt, char errmsg[256])
{
    struct private_st *ctx;
    /*
     * Initialize the driver-specific field in the isb.  This usually points
     * to the I/O context but is does not have to.  The opt argument
     * has driver-specific interpretation.
     */
    isb->drv = (struct private_st *) context;
    ctx = isb->drv;
    isb->channel = ctx->channel;	/* for cport_qio calls */

    ctx->ref_count++;
    return 1;
}
/******************************************************************************/
/* destroy_stream is called to by cport_deassign (via the .detach function
 * pointer). to unbind an isb from its i/o context.  When a program calls
 * cport_destroy, completion_port.c will deassign any streams still associated.
 * with it.
 */
static int destroy_stream ( cport_isb isb )
{
    struct private_st *ctx;

    ctx = isb->drv;
    ctx->ref_count--;
    if ( ctx->ref_count <= 0 ) {
	/*
	 * Last stream using context deassigned, rundown.
	 */
	SYS$DASSGN ( ctx->channel );
	deallocate_context ( ctx );
    }
    isb->drv = (struct private_st *) 0;
    return 1;
}
/******************************************************************************/
/* cancel_io is called by cport_cancel It should take whatever steps are
 * necessary to cancel any pending asynchronous operations (SYS$CANCEL).
 */
static int cancel_io ( cport_isb isb )
{
    struct private_st *ctx;

    ctx = isb->drv;
    return 1;
}
/******************************************************************************/
/* Define completion callback routines referenced by start_io functions
 * below.  A completion callback is a function called by cport_next_completion
 * after it has moved the isb from the port's completed queue to the
 * processed queue but before it has returned to the caller.
 *
 * The return value of the completion_callback determines the action
 * cport_next_completion will take with the isb:
 *    CPORT_COMPLETE_BUSY	Move isb back to busy queue (new I/O started).
 *
 *    CPORT_COMPLETE_COMPLETED	Move isb back to completed queue (little use).
 *
 *    CPORT_COMPLETE_PROCESSED	Leave isb on processed queue and return it
 *				to caller.
 */
static int finish_getsyi ( cport_isb isb )
{
    struct private_st *ctx;

    ctx = isb->drv;
    isb->completion_callback = 0;
    /*
     * Reset wildcard flag if error.
     */
    if ( isb->default_iosb[0]&1 == 0 ) ctx->wildcard_syi = 0;

    return CPORT_COMPLETE_PROCESSED;
}

/******************************************************************************/
/* Define the start_io functions that cport_start_io calls via the ftable
 * array in the stream_handler.
 *
 * A user call to cport_start_io ( isb, func, arg1, arg2 ) results in
 * a call to (*isb->driver.ftable[func&FMASK])( isb, func, arg1, arg2 ).
 *
 * All start_io functions use the following prototype:
 *
 *    int start ( cport_isb isb, int func, void *arg1, void *arg2 );
 *
 *    arguments:
 *        isb		I/O synchronization block.  cport_next_completion
 *			will return this address when I/O completes.
 *
 *	  func		Function value describing operation to perform,
 *			in most cases the function is implied by which
 *			start_io function got called.  Bits outside the
 *			range of FMASK may specify qualifiers.
 *
 *        arg1		Driver and function specific argument.  Being
 *			a pointer, this usually points to the user buffer.
 *
 *        arg2		Driver and function specific argument.  Being an
 *			int type, this argument usually specifies the
 *			buffer length or function qualifier.
 *
 *    return value:
 *        odd		Success.
 *	  even.		Failure.
 *
 * 
 */
static int start_write ( cport_isb isb, int func, void *arg1, int arg2 )
{
    /*
     * For simple requests, use cport_qio2 function, which takes care of
     * issuing qio and syncrhonizing completion.  Arg1 and arg2 become
     * P1/P2 of QIO call.
     */
    return cport_qio2 ( isb, IO$_WRITEVBLK, arg1, arg2 );
}
static int start_read ( cport_isb isb, int func, void *arg1, int arg2 )
{
    return cport_qio2 ( isb, IO$_READVBLK, arg1, arg2 );
}

static int start_gettim ( cport_isb isb, int func, void *arg1, int arg2 )
{
    /*
     * This function demonstrates a start_io that completes immediately.
     */
    return SYS$GETTIM ( arg1 );
}

static int start_getsyi ( cport_isb isb, int func, void *arg1, int arg2 )
{
    int status;
    struct private_st *ctx;
    /*
     * Peform a sys$getsyi call.  Arg1 is pointer to item_list and
     * arg2 is system ID (0-current, -1 wildcard).
     */
    ctx = isb->drv;
    if ( ctx->wildcard_syi == 0 ) {
	if ( arg2 == -1 ) {
	    ctx->csid = (unsigned long) -1;
	    ctx->wildcard_syi = 1;
	} else {
	    ctx->csid = arg2;
	}
    }
    /*
     * Set completion callback pointer in ISB to let our driver see
     * the result prior to returning to caller.  AST and parameter
     * are cport_completion_ast and ISB.  Move isb to busy queue prior
     * to invoking service (ensures cport_completion_ast will see isb on
     * busy queue).
     */
    isb->completion_callback = finish_getsyi;
    cport_mark_busy ( isb );
    status = SYS$GETSYI ( 
	0, &ctx->csid, 0, arg1, isb->iosb, cport_completion_ast, isb );
    return status;
}
/******************************************************************************/
/* Define global stream handler structure to be passed to cport_assign_stream()
 * by clients wishing to use this driver.  Note that the isb contains a
 * copy of the handler struct, not a pointer to a single instance.  Note
 * also that the copies have a common pointer to the start_io ftable.
 *
 * The convention for naming the stream handler is cportXXXXX_driver, where
 * XXXXX is a unqiue driver name (e.g. cportsshmsg_driver is driver for ssh
 * message layer).
 * 
 */
#define FMASK 3			/* Must be 2^n -1 */
static cport_start_function ftable[FMASK+1] = {
    start_write,		/* CPORT_WRITE function */
    start_read,			/* CPORT_READ function */
    start_getsyi,		/* CPORT_FCODES+0, get SYI call */
    start_gettim		/* CPORT_FCODES+1, sync. function */
};

cport_stream_handler cportexample_driver = {
    FMASK,			/* Size of ftable - 1 */
    ftable,			/* start_io function table */
    new_stream,			/* Assign stream to I/O context */
    destroy_stream,		/* Deassign stream */
    cancel_io			/* Stop I/O in progress. */
};
