/*
 * This module handles execution of pty and command mode connections where
 * the username has been qualified by /noprocess.  Rather than creating
 * a client process, the command is processed internally, reading and
 * discarding input until a read error occurs.
 *
 * The user the system connects as must be present in the administrators
 * parameter list.
 *
 * Author:      David Jones
 * Date:        28-OCT-1998
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "tutil.h"
#define CPORT_DRIVER_SPECIFIC struct private_st
#include "completion_port.h"
#include "parameters.h"
#include "user_info.h"
#include "cport_sshsess.h"
#include "cport_noprocess.h"            /* verify prototypes */
/*
 * A private_st structure is allocated for each open asynch_stream and is
 * pointed to by the asynch_stream structure's .private member.
 */
struct private_st {
    int ref_count;
    int exitstatus;             /* Command exit status. */
};

/******************************************************************************/
/* Main routine for creating a stream.  Verfify the username is in the
 * allowed list.
 */
int execnoproc_open ( void **ctx,               /* return context pointer */
        sshsess_session session,
        struct user_account_info *uai,          /* Username/password */
        char errmsg[256] )
{
    int status, ncblen, pwd_len, obj_len;
    struct parameter_list_elem *administrator;
    struct { int length; char *data; } ncb_desc;
    struct private_st *pctx;
    char username[40];
    *ctx = (void *) 0;
    return 1;
}
/**************************************************************************/
/* The write opration is effectively a null operation.
 */
static int start_write ( cport_isb isb, int func, void *buffer, int bufsize )
{
    struct private_st *ctx;
    int status, i;
    char *cp;
    /*
     * Scan for control-Z.
     */
    isb->default_iosb[0] = 1;
    for ( cp = (char *) buffer, i=0; i < bufsize; i++ ) if ( cp[i] == 26 ) {
	/*
	 * force rundown of connection.
	 */
        isb->default_iosb[0] = 44;
    }
    /*
     * Fill in IOSB with success status, all data written and complete I/O.
     */
    isb->iosb = (void *) isb->default_iosb;
    isb->default_iosb[1] = bufsize;
    status = cport_mark_completed ( isb, 0 );
    return status;
}
/**************************************************************************/
/* The read operation just blocks forever.
 */

static int start_read ( cport_isb isb, int func, void *buffer, int bufsize )
{
    isb->iosb = (void *) isb->default_iosb;
    if ( isb->flags == 0 && (bufsize>1) ) {
	isb->flags = 1;			/* mark that status line sent */
	tu_strnzcpy ( (char *) buffer, 
	  "Non-process (/noprocess) login, input ignored (except ctrl-Z)\r\n",
	   bufsize-1 );
	isb->default_iosb[0] = 1;
	isb->default_iosb[1] = tu_strlen ( buffer );
	return cport_mark_completed ( isb, 0 );
    }
    isb->default_iosb[0] = isb->default_iosb[1] = 0;
    return cport_mark_busy ( isb );
}
/*****************************************************************************/
/* The change window function has no meaning, but included for compatibility
 * with the cmd and pty modes.
 */
static int change_window ( cport_isb isb, int func, void *buffer, int length )
{
    struct private_st *pctx;
    /*
     * Since we aren't really a terminal, do nothing about geometry changes.
     */
    isb->default_iosb[0] = 1;
    return cport_mark_completed ( isb, 0 );
}

static int query_exit_status ( cport_isb isb, int func, 
        void *buffer, int length )
{
    *((int *) buffer) = isb->drv->exitstatus;
    isb->default_iosb[0] = 1;
    isb->default_iosb[1] = 4;
    return 1;                   /* synchronous completion */
}
/****************************************************************************/
/* Declare the functions used in in the stream handler table.
 */
static cport_start_function ftable[4] = {
    start_write,
    start_read,
    change_window,              /* Change terminal geometry */
    query_exit_status           /* return exit status */
};

static int new_stream ( cport_isb isb, void *context, int dir,
        char errmsg[256] )
{
    isb->flags = 0;
    return 1;
}

static int destroy_stream ( cport_isb isb )
{
    int status;
    struct private_st *ctx;

    isb->channel = 0;
    return 1;
}

static int cancel_io ( cport_isb isb )
{
    struct private_st *ctx;
    int status;
    /*
     * Cancel pending I/Os (none).
     */
    return 1;
}
/*
 * The cportucx_driver table will be specified in cport_assign_stream
 * calls as the handler argument.
 */
cport_stream_handler cportnoproc_driver = {
        3,                                      /* mask 3 => 4 functions */
        ftable,
        new_stream,
        destroy_stream,
        cancel_io
};
