/*
 * 'Pipe' driver for passing messages between threads with independant
 * completion ports.  Note that for proper synchronization, master must assign
 * stream to pipe prior to partner thread performing I/O.
 *
 * The thread creating the pipe context is known as the master thread, sematics
 * of the pipe read operation varying depending upon whether the caller is
 * the master or partner.  If the master thread reads from the pipe and there
 * is no partner thread, the read blocks.  If a partner thread reads from the
 * pipe and the master has deassigned, the read fails with SS$_LINKDISCON.
 *
 * Author:	David Jones
 * Date:	18-JUN-1998
 * Revised:	14-SEP-1998	Fix bug in deassign handling.
 * Revised:	15-SEP-1998	complete support for set_send_type.
 * Revised:	21-SEP-199	Fix rundown problem with attempts to signal
 *				destroyed streams.
 * Revsied:	25-NOV-1998	make [de]allocate_context routine static.
 */
#include "pthread_1c_np.h"
#include <stdlib.h>
#include <ssdef.h>			/* VMS sys. service status codes */
#include <stdio.h>
#include <string.h>
/*
 * Override definition of drv field in isb to be pointer to our private
 * structure.
 */
#define CPORT_DRIVER_SPECIFIC struct private_st
#include "completion_port.h"
/*
 * Define structure to hold the driver-specific context information.
 * This structure is completely private to the driver and must
 * be allocated/deallocated by the driver.
 */
struct private_st {
    struct private_st *next;
    int ref_count;		/* # of streams assigned to this context */
    int master_ref_count;	/* # of streams assigned by master port */
    /*
     * Use a mutex to synchronize the state changes.
     */
    pthread_mutex_t lock;
    /*
     * A pipe involves the interaction of 2 pairs of streams.
     * Writes to m_0 are read by c_1 and writes to c_0 are read by m_1.
     */
    cport_port master;		/* creator thread's completion port */
    cport_port partner;		/* partner thread's completion port */
    cport_isb waiter_0_0;		/* coordinate master writes. (m_0) */
    cport_isb waiter_0_1;		/* master is reading */
    cport_isb waiter_1_0;		/* partner is writing */
    cport_isb waiter_1_1;		/* partner is reading */
    int send_type_0;		/* moved into IOSB[3] on read */
    int send_type_1;
};
/*
 * Define internal routines for maintaining lookaside list of previously
 * allocated (and released) private structs.  Also make once routine.
 */
static pthread_once_t module_setup = PTHREAD_ONCE_INIT;
static pthread_mutex_t module_data;
static struct private_st *free_ctx;

static void module_init ( )
{
    INITIALIZE_MUTEX ( &module_data );
    free_ctx = (struct private_st *) 0;
}

#define GROUP 4
static struct private_st *allocate_context()
{
    struct private_st *new;
    int i;
    struct iblock { struct private_st instance[GROUP]; } *block;
    /*
     * Allocate new block.
     */
    pthread_mutex_lock ( &module_data );
    if ( !free_ctx ) {
	/*
	 * Grow list allocate multiple.
	 */
	block = (struct iblock *) malloc ( sizeof(struct iblock) );
	if ( block ) {
	    /*
	     * Do one-time initialization for block members (mutexes, etc)
	     * and link into free list.
	     */
	    for ( i = 0; i < GROUP; i++ ) {
		block->instance[i].next = &block->instance[i+1];
		INITIALIZE_MUTEX(&block->instance[i].lock);
	    }
	    block->instance[GROUP-i].next = (struct private_st *) 0;
	    free_ctx = &block->instance[0];

	} else {
	    /*
	     * Serious error, allocation failures.
	     */
	    pthread_mutex_unlock ( &module_data );
	    return free_ctx;
	}
    }
    new = free_ctx;
    free_ctx = new->next;
    pthread_mutex_unlock ( &module_data );
    /*
     * Set allocated context to initial state.
     */
    new->ref_count = 0;
    new->master_ref_count = 0;
    new->master = (cport_port) 0;
    new->partner = (cport_port) 0;
    new->waiter_0_0 = (cport_isb) 0;
    new->waiter_0_1 = (cport_isb) 0;
    new->waiter_1_0 = (cport_isb) 0;
    new->waiter_1_1 = (cport_isb) 0;

    return new;
}

static void deallocate_context ( struct private_st *ctx )
{
    /*
     * Simply push context onto lookaside list.
     */
    pthread_mutex_lock ( &module_data );
    ctx->next = free_ctx;
    free_ctx = ctx;
    pthread_mutex_unlock ( &module_data );
}
/******************************************************************************/
/* The open function creates a private I/O context.
 */
int cport_pipe_open ( cport_port master, void **context )
{
   struct private_st *new;

    pthread_once ( &module_setup, module_init );

    new = allocate_context();
    if ( !new ) return SS$_INSFMEM;

    new->master = master;
    *context = (void *) new;
    return 1;
}
/******************************************************************************/
int cport_pipe_set_send_type ( cport_isb isb, int send_type )
{
    /*
     * determine if master or not.
     */
    pthread_mutex_lock ( &isb->drv->lock );
    if ( isb->port == isb->drv->master ) {
	isb->drv->send_type_0 = send_type;
    } else {
	isb->drv->send_type_1 = send_type;
    }
    pthread_mutex_unlock ( &isb->drv->lock );
    return 1;
}
/******************************************************************************/
/* cancel_io is called by cport_cancel.
 */
static int cancel_io ( cport_isb isb )
{
    struct private_st *ctx;
    /*
     * see if any I/O is in progress and abort it.
     */
    ctx = isb->drv;
    pthread_mutex_lock ( &ctx->lock );
    if ( ctx->waiter_0_0 == isb ) {
	ctx->waiter_0_0 = (cport_isb) 0;
    } else if ( ctx->waiter_0_1 == isb ) {
	ctx->waiter_0_1 = (cport_isb) 0;
    } else if ( ctx->waiter_1_0 == isb ) {
	ctx->waiter_1_0 = (cport_isb) 0;
    } else if ( ctx->waiter_1_1 == isb ) {
	ctx->waiter_1_1 = (cport_isb) 0;
    } else {
	isb = (cport_isb) 0;
    }
    if ( isb ) {
	cport_lock_port ( isb );
	isb->default_iosb[0] = SS$_CANCEL;
	isb->default_iosb[1] = 0;
	isb->default_iosb[2] = 0;
	isb->default_iosb[3] = 0;
	cport_unlock_port ( isb,  1 );
    }
    pthread_mutex_unlock ( &ctx->lock );
    return 0;
}
/******************************************************************************/
/* Define the start_io functions that cport_start_io calls via the ftable
 * array in the stream_handler.
 *
 * A user call to cport_start_io ( isb, func, arg1, arg2 ) results in
 * a call to (*isb->driver.ftable[func&FMASK])( isb, func, arg1, arg2 ).
 *
 * All start_io functions use the following prototype:
 *
 *    int start ( cport_isb isb, int func, void *arg1, void *arg2 );
 *
 *    arguments:
 *        isb		I/O synchronization block.
 *	  func		Function value describing operation to perform,
 *			in most cases the function is implied by which
 *			start_io function got called.  Bits outside the
 *			range of FMASK may specify qualifiers.
 *
 *        arg1		Driver and function specific argument.  Being
 *			a pointer, this usually points to the user buffer.
 *
 *        arg2		Driver and function specific argument.  Being an
 *			int type, this argument usually specifies the
 *			buffer length.
 *
 *    return value:
 *        odd		Success.
 *	  even.		Failure.
 *
 * 
 */
static int start_xfer_0_1 ( cport_isb isb, int func, void *arg1, int arg2 )
{
    /*
     * Start read from isb assigned to master port.  Since writer is other
     * port, use waiter/send_type _1.
     */
    struct private_st *ctx;
    int status, count;

    ctx = isb->drv;
    pthread_mutex_lock ( &ctx->lock );
    if ( ctx->waiter_1_0 ) {
	cport_isb waiter;
	cport_port wport;
	/*
	 * waiter_1_0 points to isb of waiting thread to copy data from
	 * partner's isb and signal completion.
	 */
	waiter = ctx->waiter_1_0;
	ctx->waiter_1_0 = (cport_isb) 0;

	cport_lock_port ( waiter );
	count = waiter->length;
	if ( count <= arg2 ) {
	    waiter->default_iosb[0] = 1;
	    memcpy ( arg1, waiter->buffer, count );
	} else {
	    waiter->default_iosb[0] = SS$_DATAOVERUN;
	}
        waiter->default_iosb[1] = count;
	cport_unlock_port ( waiter, 1 );
        /*
         * Complete the caller's I/O.
	 */
	isb->default_iosb[0] = 1;
	isb->default_iosb[1] = count;
	isb->default_iosb[3] = ctx->send_type_1;
	cport_mark_completed ( isb, 0 );	/* immediate completion */
	status = 1;

#ifdef SYMETRIC_PIPE
    } else if ( (ctx->ref_count == ctx->master_ref_count) && ctx->partner ) {
	/*
	 * partner deassigned all streams.
	 */
	isb->default_iosb[0] = SS$_LINKDISCON;
	isb->default_iosb[1] = 0;
	cport_mark_completed ( isb, 0 );	/* immediate completion */
	status = 1;
#endif
    } else {
	/*
	 * Become the waiter.
	 */
	ctx->waiter_0_1 = isb;
	isb->default_iosb[0] = isb->default_iosb[1] = 0;
	cport_mark_busy ( isb );
	status = 1;
    }
    pthread_mutex_unlock ( &ctx->lock );
    return status;
}

static int start_xfer_0_0 ( cport_isb isb, int func, void *arg1, int arg2 )
{
    /*
     * Start write from isb assigned to master port.  When master port is
     * writer use waiter/send_type _0.
     */
    struct private_st *ctx;
    int status, count;

    ctx = isb->drv;
    pthread_mutex_lock ( &ctx->lock );
    if ( ctx->waiter_1_1 ) {
	cport_isb waiter;
	cport_port wport;
	/*
	 * waiter_1_1 points to isb of waiting thread to copy data to
	 * partner's isb and signal completion.
	 */
	waiter = ctx->waiter_1_1;
	ctx->waiter_1_1 = (cport_isb) 0;

	cport_lock_port ( waiter );
	if ( waiter->length >= arg2 ) {
	    waiter->default_iosb[0] = 1;
	    memcpy ( waiter->buffer, arg1, arg2 );
	} else {
	    waiter->default_iosb[0] = SS$_DATAOVERUN;
	}
        waiter->default_iosb[1] = arg2;
	waiter->default_iosb[3] = ctx->send_type_0;
	cport_unlock_port ( waiter, 1 );

	isb->default_iosb[0] = 1;
	isb->default_iosb[1] = arg2;
	cport_mark_completed ( isb, 0 );	/* immediate completion */
	status = 1;
    } else if ( (ctx->ref_count == ctx->master_ref_count) && ctx->partner ) {
	/*
	 * partner deassigned all streams.
	 */
	isb->default_iosb[0] = SS$_LINKDISCON;
	isb->default_iosb[1] = 0;
	cport_mark_completed ( isb, 0 );	/* immediate completion */
	status = 1;
    } else {
	/*
	 * Become the waiter.
	 */
	ctx->waiter_0_0 = isb;
	isb->default_iosb[0] = isb->default_iosb[1] = 0;
	cport_mark_busy ( isb );
	status = 1;
    }
    pthread_mutex_unlock ( &ctx->lock );
    return status;
}
static int start_xfer_1_1 ( cport_isb isb, int func, void *arg1, int arg2 )
{
    /*
     * Start read from isb assigned to partner port.  Since writer is master
     * port, use waiter/send_type _0.
     */
    struct private_st *ctx;
    int status, count;
	cport_isb waiter;

    ctx = isb->drv;
    waiter = (cport_isb) ctx;
    pthread_mutex_lock ( &ctx->lock );
	waiter = ctx->waiter_0_0;
    if ( ctx->waiter_0_0 ) {
	cport_port wport;
	/*
	 * waiter_0_0 points to isb of waiting thread to copy data from
	 * partner's isb and signal completion.
	 */
	ctx->waiter_0_0 = (cport_isb) 0;

	cport_lock_port ( waiter );
	count = waiter->length;
	if ( count <= arg2 ) {
	    waiter->default_iosb[0] = 1;
	    memcpy ( arg1, waiter->buffer, count );
	} else {
	    waiter->default_iosb[0] = SS$_DATAOVERUN;
	}
        waiter->default_iosb[1] = count;
	waiter->default_iosb[3] = ctx->send_type_1;
	cport_unlock_port ( waiter, 1 );
        /*
         * Complete the caller's I/O.
	 */
	isb->default_iosb[0] = 1;
	isb->default_iosb[1] = count;
	isb->default_iosb[3] = ctx->send_type_0;
	cport_mark_completed ( isb, 0 );	/* immediate completion */
	status = 1;
    } else if ( 0 == ctx->master_ref_count ) {
	/*
	 * partner deassigned all streams.
	 */
	isb->default_iosb[0] = SS$_LINKDISCON;
	isb->default_iosb[1] = 0;
	cport_mark_completed ( isb, 0 );	/* immediate completion */
	status = 1;
    } else {
	/*
	 * Become the waiter.
	 */
	ctx->waiter_1_1 = isb;
	isb->default_iosb[0] = isb->default_iosb[1] = 0;
	cport_mark_busy ( isb );
	status = 1;
    }
    pthread_mutex_unlock ( &ctx->lock );
    return status;
}
static int start_xfer_1_0 ( cport_isb isb, int func, void *arg1, int arg2 )
{
    /*
     * Start write from isb assigned to partner port.  When partner port is
     * writer use waiter/send_type _1.
     */
    struct private_st *ctx;
    int status, count;

    ctx = isb->drv;
    pthread_mutex_lock ( &ctx->lock );
    if ( ctx->waiter_0_1 ) {
	cport_isb waiter;
	cport_port wport;
	/*
	 * waiter_0_1 points to isb of waiting thread to copy data to
	 * partner's isb and signal completion.
	 */
	waiter = ctx->waiter_0_1;
	ctx->waiter_0_1 = (cport_isb) 0;

	cport_lock_port ( waiter );
	if ( waiter->length >= arg2 ) {
	    waiter->default_iosb[0] = 1;
	    memcpy ( waiter->buffer, arg1, arg2 );
	} else {
	    waiter->default_iosb[0] = SS$_DATAOVERUN;
	}
        waiter->default_iosb[1] = arg2;
	waiter->default_iosb[3] = ctx->send_type_1;
	cport_unlock_port ( waiter, 1 );

	isb->default_iosb[0] = 1;
	isb->default_iosb[1] = arg2;
	cport_mark_completed ( isb, 0 );	/* immediate completion */
	status = 1;
    } else if ( 0 == ctx->master_ref_count ) {
	/*
	 * partner deassigned all streams.
	 */
	isb->default_iosb[0] = SS$_LINKDISCON;
	isb->default_iosb[1] = 0;
	cport_mark_completed ( isb, 0 );	/* immediate completion */
	status = 1;
    } else {
	/*
	 * Become the waiter.
	 */
	ctx->waiter_1_0 = isb;
	isb->default_iosb[0] = isb->default_iosb[1] = 0;
	cport_mark_busy ( isb );
	status = 1;
    }
    pthread_mutex_unlock ( &ctx->lock );
    return status;
}
/*****************************************************************************/
/*
 * Define dispatch tables to the start_io functions defined above.
 */
#define FMASK 1			/* Must be 2^n -1 */
static cport_start_function ftable[FMASK+1] = {
    start_xfer_0_0,			/* CPORT_WRITE function */
    start_xfer_0_1			/* CPORT_READ function */
};
static cport_start_function ftable2[FMASK+1] = {
    start_xfer_1_0,		/* CPORT_WRITE function */
    start_xfer_1_1		/* CPORT_READ function */
};

/******************************************************************************/
/* The new_stream function is called by cport_assign_stream (via the .attach)
 * function pointer) to bind a newly created isb to the i/o context pointer 
 * returned by the driver's open function.
 *
 * Arguments:
 *
 * Return value:
 */
static int new_stream (cport_isb isb, void *context, int opt, char errmsg[256])
{
    struct private_st *ctx;

    ctx = (struct private_st *) context;
    isb->drv = context;
    pthread_mutex_lock ( &ctx->lock );
    if ( isb->port == ctx->master ) {
	/*
	 * We are master thread.
	 */
	ctx->master_ref_count++;
    } else {
	/*
	 * We are not master thread.  redirect to alternate function table.
	 */
	isb->driver.ftable = ftable2;
	ctx->partner = isb->port;
    }
    ctx->ref_count++;
    pthread_mutex_unlock ( &ctx->lock );
    return 1;
}
/******************************************************************************/
/* destroy_stream is called to by cport_deassign (via the .detach function
 * pointer). to unbind an isb from its i/o context.  When a program calls
 * cport_destroy, completion_port.c will deassign any streams still associated.
 * with it.
 */
static int destroy_stream ( cport_isb isb )
{
    struct private_st *ctx;
    int ref_count, master_count;
    cport_isb waiter;

    ctx = isb->drv;
    pthread_mutex_lock ( &ctx->lock );
    /*
     * update reference counts.
     */
    ctx->ref_count--;
    if ( isb->port == ctx->master ) {
	/*
	 * We are master thread, see if last stream.
	 */
	ctx->master_ref_count--;
	if ( ctx->master_ref_count == 0 ) {
	    /*
	     * kill any pending partner I/O by partner.
	     */
	    if ( ctx->waiter_1_0 ) {
		waiter = ctx->waiter_1_0;
		ctx->waiter_1_0 = (cport_isb) 0;
		cport_lock_port ( waiter );
		waiter->default_iosb[0] = SS$_LINKDISCON;
		cport_unlock_port ( waiter, 1 );
	    }
	    if ( ctx->waiter_1_1 ) {
		waiter = ctx->waiter_1_1;
		ctx->waiter_1_1 = (cport_isb) 0;
		cport_lock_port ( waiter );
		waiter->default_iosb[0] = SS$_LINKDISCON;
		cport_unlock_port ( waiter, 1 );
	    }
	}
	/*
	 * Clear any pending I/O.
         */
	if ( isb == ctx->waiter_0_0 ) ctx->waiter_0_0 = (cport_isb) 0;
	if ( isb == ctx->waiter_0_1 ) ctx->waiter_0_1 = (cport_isb) 0;
    } else {
	/*
	 * We are not master thread see if last thread.
	 */
	if ( ctx->master_ref_count == ctx->ref_count ) {
	    /*
	     * Kill any pending I/O by master.
	     */
	    if ( ctx->waiter_0_0 ) {
		waiter = ctx->waiter_0_0;
		ctx->waiter_0_0 = (cport_isb) 0;
		cport_lock_port ( waiter );
		waiter->default_iosb[0] = SS$_LINKDISCON;
		cport_unlock_port ( waiter, 1 );
	    }
#ifdef SYMETRIC_PIPE
	    if ( ctx->waiter_0_1 ) {
		waiter = ctx->waiter_0_1;
		ctx->waiter_0_1 = (cport_isb) 0;
		cport_lock_port ( waiter );
		waiter->default_iosb[0] = SS$_LINKDISCON;
		cport_unlock_port ( waiter, 1 );
	    }
#endif
	}
	if ( isb == ctx->waiter_1_0 ) ctx->waiter_1_0 = (cport_isb) 0;
	if ( isb == ctx->waiter_1_1 ) ctx->waiter_1_1 = (cport_isb) 0;
    }
    ref_count = ctx->ref_count;
    master_count = ctx->master_ref_count;
    pthread_mutex_unlock ( &ctx->lock );
    /*
     * Rundown context if no more streams.
     */
    if ( ref_count <= 0 ) {
	deallocate_context ( ctx );
    }
    return 1;
}
/******************************************************************************/
/* Define global stream handler structure to be pass to cport_assign_stream().
 * by clients wishing to the this driver.  Note that the isb contains a
 * copy of the handler struct, not a pointer to a single instance.  Note
 * also that copies have a common pointer to the start_io ftable.
 *
 * The convention for naming the stream handler is cportXXXXX_driver, where
 * XXXXX is a unqiue driver name (e.g. cportsshmsg_driver is driver for
 * 
 */
cport_stream_handler cportpipe_driver = {
    FMASK,			/* Size of ftable - 1 */
    ftable,			/* start_io function table */
    new_stream,			/* Assign stream to I/O context */
    destroy_stream,		/* Deassign stream */
    cancel_io			/* Stop I/O in progress. */
};
