/*
 * This module provides support for executing SSH_CMSG_EXEC_SHELL-style
 * logins for and arbitrary user.  We create a pseudo-terminal (PDT) and
 * login the user using the username and password supplied by the caller.
 * If a command is supplied, it is sent as the first input string.
 *
 * Note that the terminal must maintain /hostsync to function properly.
 *
 *
 * Author:	David Jones
 * Date:	16-MAY-1998
 * Revised:	26-MAY-1998		Update for completion port model
 * Revised:	25-JUN-1998		Set defchar2 field in char. buffer.
 * Revised:     27-JUN-1998		Additional check for idle state.
 * Revsied:	2-JUL-1998		Support param.tt_char.
 * Revised:	15-JUL-1998		Support process create via initiator.
 * Revsed:	25-JUL-1998		Change cport_execpty interface,
 *				 	generate accpornam string.
 * Revised:	18-AUG-1998		Support change window.
 * revised:	26-SEP-1998		Support terminal type string.
 * Revised:	2-OCT-1998		Improve locality for terminal defs.
 * Revised:	21-OCT-1998		Supoprt param.pty_mode and
 * 					param.pty_map_table.
 * Revised:	16-MAR-1999		Convert read errors caused by
 *					idle AST to EOF markers.
 * Revised:	23-MAY-1999		Support pty_accpornam hack.
 */
#include <stdio.h>
#include <stdlib.h>
#include <iodef.h>			/* QIO symbolic definitions */
#include <dcdef.h>			/* VMS device class codes */
#include <dvidef.h>
#include <syidef.h>			/* VMS system information */
#include <ptddef.h>			/* pseudo-terminal defs */
#include <ttdef.h>			/* VMS terminal modes */
#include <tt2def.h>			/* VMS extended. term. modes */
#include <ssdef.h>			/* VMS sys. serv. error codes */
#include <string.h>

#include "pthread_1c_np.h"
#define CPORT_DRIVER_SPECIFIC struct ptd_context
#include "completion_port.h"
#include "parameters.h"
#include "tutil.h"
#include "tmemory.h"
#include "user_info.h"
#include "initiator_client.h"
#include "pty_map.h"

int SYS$SETAST(), SYS$QIO(), SYS$ASSIGN(), SYS$GETDVI(), SYS$CANCEL();
int LIB$GET_EF(), SYS$DASSGN(), SYS$SET_SECURITY(), LIB$GETSYI();

int PTD$CREATE(), PTD$READ(), PTD$WRITE(), PTD$CANCEL(), PTD$DELETE();
int PTD$SET_EVENT_NOTIFICATION();

static pthread_once_t execpty_once = PTHREAD_ONCE_INIT;
static char execpty_io_init_err[256];
static unsigned char ptd_tty_deftype;
static int ptd_page_size;
static int ptd_tty_defchar;		/* default terminal characteristics */
static int ptd_tty_defchar2;

struct terminal_iosb { 			/* IOSB for terminal driver: */
   unsigned short status; 		/*	completion status    */
   unsigned short count; 		/*	offset to terminator */
   unsigned short terminator;		/*	Terminating character*/
   unsigned short term_len;		/*	chars in term. sequence */
};
struct tchar_buf {
    unsigned char class;
    unsigned char type;
    unsigned short screen_width;
    unsigned long tchar_pagelen;		/* page length in high byte */
    unsigned long extended_char;
};
static struct terminal_def {
    struct terminal_def *next;
    struct tchar_buf tchar;		/* terminal characteristics for type */
    char name[1];			/* type name (variable length) */
} *ptd_tty_type;			/* type definition */

static struct tchar_update {
    struct tchar_update *next;
    int namlen;
    short update_pending;
    unsigned short io_chan;
    unsigned short iosb[4];
    char terminal_name[64];
    struct tchar_buf current, pending;
} *update_list, *update_free;

struct ptd_io_buffer {
    unsigned short status;		/* Completion status */
    unsigned short count;		/* bytes transferred */
    char data[512-(2*sizeof(unsigned short))];   /* take full pagelet */
};
struct ptd_context {
    struct ptd_context *next;		/* next in list */
    pthread_mutex_t lock;		/* synchronizes access ot struct */
    pthread_cond_t io_done;
    int ref_count;			/* number of streams assigned */
    int idle;
    int owner;
    unsigned short ctl_chan;		/* control channel */
    unsigned short io_chan;		/* auxillary channel (setmode) */
    struct tchar_buf mode;		/* initial terminal characteristics */
    char *pending_in;
    char *pending_out;
    int pend_in_size;		/* Bytes of pending input remaining */
    int pend_out_size;
    int pending;
    int xoffed;
    cport_isb xon_waiter;
    long xon_iosb[2];
    int line_len;
    int line_done;
    int prompt_action;
    int prompt_start;			/* off of prompt or -1 */
    int prompt_end;
    char prompt[500];			/* holds last 500 char of prompt */
    char devnam[64];			/* device name */
    int buf_count;			/* buffers owned */
    void * in_addr[2];			/* Page range containing I/O buffers */
    struct ptd_io_buffer *buf;		/* array of buffers (buf_per_ctx) */
    char line[1024];
};
#define BUFFERS_PER_CONTEXT 4

static pthread_mutex_t ptd_resource;
static struct ptd_context *ptd_free_context;

#define MAX_INPUT_LINE 512
#define MAX_OUTPUT_LINE 4096
#define MAX_INPUT_SEGMENT 8
static int finish_write ( cport_isb isb );	/* forward referecne */
static void hold_input_ast ( struct ptd_context *);
static void resume_input_ast ( struct ptd_context *);
/*
 * One-time initialization for module: Initialize per-thread heap,
 * create AST-to-Thread queue for running AST in a normal thread context and
 * create page-aligend memory.
 */
static void execpty_init()
{
    int code, status, length;
    struct parameter_mlist_elem *tlist;

    INITIALIZE_MUTEX ( &ptd_resource );
    tm_initialize();
    /*
     * Determine memory system page size.
     */
    code = SYI$_PAGE_SIZE;
    status = LIB$GETSYI ( &code, &ptd_page_size, 0, 0, 0, 0 );
    /*
     * Get default terminal characteristics and fixup for PTY's
     */
    code = SYI$_TTY_DEFCHAR;
    status = LIB$GETSYI ( &code, &ptd_tty_defchar, 0, 0, 0, 0 );
    code = SYI$_TTY_DEFCHAR2;
    status = LIB$GETSYI ( &code, &ptd_tty_defchar2, 0, 0, 0, 0 );
    convert_tt_type_parameters ( "", "VT100", param.tt_char,
		&ptd_tty_deftype, &ptd_tty_defchar, &ptd_tty_defchar2 );
    /*
     * make table of terminal type definitions
     */
    ptd_tty_type = (struct terminal_def *) 0;
    for ( tlist = param.tt_type; tlist; tlist = tlist->next ) {
	/*
	 * Allocate definition block.
	 */
	struct terminal_def *new;
	char *value;

	length = sizeof(struct terminal_def);
	if ( tlist->value ) length += tu_strlen ( tlist->value );
	new = (struct terminal_def *) malloc ( length );
	new->next = ptd_tty_type;
	ptd_tty_type = new;
	tu_strcpy ( new->name, tlist->value ? tlist->value : "" );
	/*
	 * Set characteristics for this type.
	 */
	new->tchar.class = DC$_TERM;
	new->tchar.type = 0;
	new->tchar.tchar_pagelen = ptd_tty_defchar;
	new->tchar.extended_char = ptd_tty_defchar2;
	value = (char *) 0;
	if ( tlist->child ) value = tlist->child->value;
	convert_tt_type_parameters ( new->name, value,
		param.tt_char,
		&new->tchar.type,
		(int *) &new->tchar.tchar_pagelen, 
		(int *) &new->tchar.extended_char );
	if ( new->name[0] == '\0' && value ) {
	    ptd_tty_deftype = new->tchar.type;
	}
    }
    ptd_free_context = (struct ptd_context *) 0;
    update_list = update_free = (struct tchar_update *) 0;
}
/************************************************************************/
/* Allocate and initialize structure for managing a PTD.
 */
static struct ptd_context *allocate_context()
{
    struct ptd_context *new_context;

    pthread_mutex_lock ( &ptd_resource );
    if ( !ptd_free_context ) {
	/*
	 * Expand the free list.
	 */
	int i, status, count, pagelets, LIB$GET_VM_PAGE(), LIB$FREE_VM_PAGE();
	struct ptd_io_buffer *io_buffer;
	count = ptd_page_size / 
		(sizeof(struct ptd_io_buffer)*BUFFERS_PER_CONTEXT);
	if ( count == 0 ) {
	    pagelets = BUFFERS_PER_CONTEXT;
	    count = 1;
	} else {
	    pagelets = ptd_page_size / 512;
	}

	status = LIB$GET_VM_PAGE ( &pagelets, &io_buffer );
	if ( status&1 ) {
	    ptd_free_context = malloc ( sizeof(struct ptd_context) * count );
	    if ( ptd_free_context ) {
		/*
		 * Initialize the pointers.
		 */
		for ( i = 0; i < count; i++ ) {
		    ptd_free_context[i].next = &ptd_free_context[i+1];
		    INITIALIZE_MUTEX(&ptd_free_context[i].lock);
		    INITIALIZE_CONDITION ( &ptd_free_context[i].io_done );
		    ptd_free_context[i].owner = 0;
		    ptd_free_context[i].buf_count = BUFFERS_PER_CONTEXT;
		    ptd_free_context[i].in_addr[0] = io_buffer;
		    ptd_free_context[i].in_addr[1] = 
			&io_buffer[pagelets-1].data[507];
		    ptd_free_context[i].buf = &io_buffer[i*BUFFERS_PER_CONTEXT];
		}
		ptd_free_context[count-1].next = (struct ptd_context *) 0;
	    } else {
		/* Malloc failed, release VM pages */
		LIB$FREE_VM_PAGE ( &pagelets, &io_buffer );
	    }
	}
    }
    new_context = ptd_free_context;
    if ( new_context ) ptd_free_context = new_context->next;
    pthread_mutex_unlock ( &ptd_resource );
    if ( new_context ) new_context->next = (struct ptd_context *) 0;

    return new_context;
}

static void deallocate_context ( struct ptd_context *ctx )
{
    pthread_mutex_lock ( &ptd_resource );
    ctx->next = ptd_free_context;
    ptd_free_context = ctx;
    pthread_mutex_unlock ( &ptd_resource );
    return;
}
/******************************************************************************/
/* Define AST routines used to trap prompt strings for read.
 */
static void ptd_middle_read_ast ( struct ptd_context *ctx )
{
    ctx->prompt_action = 2;
    PTD$SET_EVENT_NOTIFICATION ( ctx->ctl_chan, 0, 0, 0, PTD$C_DISABLE_READ );
}
static void ptd_start_read_ast ( struct ptd_context *ctx )
{
    int status;
    ctx->prompt_action = 1;
}
void ptd_end_read_ast ( struct ptd_context *ctx )
{
}
static int ptd_collect_prompt ( struct ptd_context *ctx )
{
    int status;
    ctx->prompt_start = 0; ctx->prompt_end = -1;

    status = PTD$SET_EVENT_NOTIFICATION ( ctx->ctl_chan, ptd_start_read_ast,
	ctx, 0, PTD$C_START_READ );

    if ( status & 1 ) status = PTD$SET_EVENT_NOTIFICATION ( ctx->ctl_chan, 
		ptd_middle_read_ast, ctx, 0, PTD$C_MIDDLE_READ );

    if ( status & 1 ) status = PTD$SET_EVENT_NOTIFICATION ( ctx->ctl_chan, 
		0, 0, 0, PTD$C_ENABLE_READ );

    if (status&1 == 0) printf("Error setting up wait-for-prompt: %d\n",status);
    return status;
}
static void ptd_get_line_ast ( struct ptd_context *ctx ) 
{
    int status, i, pos, finish;
    char c;
    struct ptd_io_buffer *buf;
    /*
     * Copy chars to line buffer untill full or linefeed.
     */
    buf = &ctx->buf[1];
/* printf("get_ast: status: %d count: %d\n", buf->status, buf->count ); */
    if ( (buf->status&1) == 0 ) {
	ctx->line_done = buf->status;
	pthread_cond_signal_int_np ( &ctx->io_done );
	return;
    }
    pos = ctx->line_len;
    for ( i = finish = 0; i < buf->count; i++ ) {
	c = buf->data[i];
	ctx->line[pos++] = c;
	if ( c == '\r' ) ctx->prompt_start = pos;
	if ( c == '\n' ) {
	    ctx->prompt_start = pos;
	    finish = 1;
	}
    }
    ctx->line_len = pos;
    if ( ctx->prompt_action == 1 ) ctx->prompt_start = pos;
    else if ( ctx->prompt_action == 2 ) ctx->prompt_end = pos;

    if ( finish || pos > 500 || (ctx->prompt_action == 2) ) {
	ctx->line_done = 1;
	pthread_cond_signal_int_np ( &ctx->io_done );
    } else {
	ctx->prompt_action = 0;
        status = PTD$READ ( 17, ctx->ctl_chan, ptd_get_line_ast, ctx,
	    buf, sizeof(buf->data)-7 );
    }
}
static int ptd_get_line ( struct ptd_context *ctx ) {
    int status;
    ctx->line_len = 0;
    ctx->line_done = 0;
    ctx->prompt_action = 0;
    status = PTD$READ ( 17, ctx->ctl_chan, ptd_get_line_ast, ctx,
	&ctx->buf[1], sizeof(ctx->buf[1].data)-7 );
    if ( (status&1) == 0 ) return status;

    pthread_mutex_lock ( &ctx->lock );
    while ( ctx->line_done == 0 && !ctx->idle ) {
	status = pthread_cond_wait ( &ctx->io_done, &ctx->lock );
    }
    pthread_mutex_unlock ( &ctx->lock );
    ctx->line[ctx->line_len] = '\0';
    return 1;
}
void ptd_idle_ast ( struct ptd_context *ctx )
{
    if ( ctx->idle == 0 ) {
	ctx->idle = 1;
	pthread_cond_signal_int_np ( &ctx->io_done );
	PTD$CANCEL ( ctx->ctl_chan );
    }
}
/******************************************************************************/
/*
 * Synchronization routine for QIO calls.  Specify io_synch as the astadr
 * and pthread_cond_wait_int_np.
 */
int ptd_synch ( int status, struct ptd_context *ctx, int buf_num )
{
    struct ptd_io_buffer *buf;

    if ( (status&1) == 1 ) {
	buf = &ctx->buf[buf_num];
        pthread_mutex_lock ( &ctx->lock );
	while ( buf->status == 0 ) {
	    pthread_cond_wait ( &ctx->io_done, &ctx->lock );
	}
	status = buf->status;
        pthread_mutex_unlock ( &ctx->lock );
    }
    return status;
}

/*************************************************************************/
/* Scan POSIX style terminal modes string and reflect requested modes
 * in terminal characteristics.  Ignore errors.
 */
void scan_tty_modes ( char *modes, int modes_len, struct ptd_context *pty )
{
    int i, opcode, arg;
    unsigned char *umodes;

    umodes = (unsigned char *) modes;
    for ( i = 0; i < modes_len && modes[i]; i++ ) {
	/*
	 * Extract next opcode and argument from string.
	 */
	opcode = umodes[i];
	if ( opcode < 128 && (i+1 < modes_len) ) {
	    arg = umodes[i+1];
	    i++;		/* skip argument */
	} else if ( opcode < 160 && (i+4 < modes_len) ) {
	    arg = umodes[i+1];
	    arg = (arg<<24) | (umodes[i+2]<<16) | (umodes[i+3]<<8) |
		umodes[i+4];
	    i += 4;
	} else if ( opcode >= 192 && opcode <= 193 &&
		(i+4 < modes_len) ){
	    arg = umodes[i+1];
	    arg = (arg<<24) | (umodes[i+2]<<16) | (umodes[i+3]<<8) |
		umodes[i+4];
	    i += 4;
	} else {
		break;
	}
	/*
	 * interpret opcode.
	 */
	if ( opcode == 53 ) {		/* echo */
	    if ( arg ) pty->mode.tchar_pagelen &= (~TT$M_NOECHO);
	    else pty->mode.tchar_pagelen |= TT$M_NOECHO;
	}
    }
}
/*************************************************************************/
/* Main routine for creating a context.  A pseudo-terminal is created and
 * logged in under the given username and password and the stream context is
 * initialized for asynchrounous communication with this process.
 */
int execpty_open ( void **ctx,			/* return value */
	char *accpornam,			/* display string, rem. host */
	int geometry[4],			/* Terminal size */
	char *tty_type,				/* terminal type */
	int tty_modes_len,
	char *tty_modes,
	struct user_account_info *uai,		/* Username to login */
	char *x11_info,				/* port forwarding info */
	char errmsg[256] )
{
    int status, pwd_len, obj_len, mbxname_len;
    struct ptd_context *pty;
    struct terminal_def *tdef;
    struct { short length, code; void *buffer; int *retlen;} item[2];
    /*
     * Make sure module initialized.
     */
    pthread_once ( &execpty_once, execpty_init );
    errmsg[0] = '\0';
    /*
     */
    pty = allocate_context();
    if ( !pty ) {
	tu_strcpy ( errmsg, "Error allocating pty context" );
	return 0;
    }
    /*
     * search for matching terminal name in defintions list so we
     * know attributes to use.
     */
    for ( tdef = ptd_tty_type; tdef; tdef = tdef->next ) {
	if ( tu_strncmp ( tty_type, tdef->name, 80 ) == 0 ) {
	    break;
	}
    }
    /*
     * Initialize pty context, set initial terminal characteristics.
     */
    pty->pending = 0;
    pty->xoffed = 0;
    pty->xon_waiter = (cport_isb) 0;
    pty->xon_iosb[0] = 0;
    pty->ref_count = 0;
    pty->idle = 0;
    pty->io_chan = 0;
    pty->mode.class = DC$_TERM;
    if ( tdef ) {
	/* Preserve page length from default char. */

        pty->mode.type = tdef->tchar.type;
        pty->mode.tchar_pagelen = (ptd_tty_defchar&0xff000000) |
		tdef->tchar.tchar_pagelen;
        pty->mode.extended_char = tdef->tchar.extended_char;
    } else {
        pty->mode.type = ptd_tty_deftype;
        pty->mode.tchar_pagelen = ptd_tty_defchar;
        pty->mode.extended_char = ptd_tty_defchar2;
    }
    pty->mode.screen_width = (255&geometry[1]);	/* columns */
    if ( pty->mode.screen_width == 0 ) pty->mode.screen_width = 80;
    if ( geometry[0] > 0 ) pty->mode.tchar_pagelen  =
	(pty->mode.tchar_pagelen&0x0ffffff) | (geometry[0] << 24);

    if ( tty_modes_len > 0 ) scan_tty_modes ( tty_modes, tty_modes_len, pty );
    pty->devnam[0] = '\0';
    /*
     * Create the pseudo_terminal.
     */
    status = PTD$CREATE ( &pty->ctl_chan, 0, &pty->mode, sizeof(pty->mode),
	ptd_idle_ast, pty, 0, pty->in_addr );
    if ( (status&1) == 0 ) {
	tu_strcpy ( errmsg, "Error creating pty, code: " );
	tu_strint ( status, &errmsg[tu_strlen(errmsg)] );
	return status;
    } else {
	/*
	 * Get device name of psuedo terminal for passing to initiator.
	 */
	struct { unsigned short status, length; int fill; } iosb;
	int length;

	item[0].code = DVI$_TT_PHYDEVNAM;
	item[0].length = sizeof(pty->devnam) - 2;
	item[0].buffer = pty->devnam;
	item[0].retlen = &length;
	item[1].code = item[1].length = 0;
	length = 0;

	pthread_mutex_lock ( &pty->lock );
	status = SYS$GETDVI ( 0, pty->ctl_chan, 0, &item,
		&iosb, 	pthread_cond_signal_int_np, &pty->io_done, 0 );
	while ( (status&1) && iosb.status == 0 ) {
	    pthread_cond_wait ( &pty->io_done, &pty->lock );
	}
	pthread_mutex_unlock ( &pty->lock );
	if ( status&1 ) status = iosb.status;
	if ( (status&1) == 1 ) {
	    /*
	     * Ensure terminal name ends in a colon.
	     */
	    pty->devnam[length] = '\0';
	    if ( iosb.length > 0 ) if ( pty->devnam[length-1] != ':' ) {
		pty->devnam[length] = ':';
		pty->devnam[length+1] = '\0';
	    }
	    /*
	     * Create entry in pty_map table and set terminal.
	     */
	    if ( *param.pty_map_table ) {
		pty_map_assign ( pty->devnam, 
			accpornam, uai->username, x11_info );
	    }
	    if ( param.pty_accpornam ) {
		/*
		 * Macro routine to bash UCB is not thread safe.
		 */
		int set_status, nlen;
		nlen = tu_strlen ( accpornam );
		if ( nlen >  63 ) nlen = 63;
		pthread_lock_global_np();
		set_status = pty_set_accpornam ( pty->ctl_chan,
			nlen, accpornam );
		pthread_unlock_global_np();
		
	    }
	}
    }

    if ( *param.initiator ) {
	/*
	 * Create detached interactive process via initiator (skips
	 * prompting for password).
	 */
	if ( (status&1) == 1 ) {
	    /*
	     * Initiator changes device ownership, which causes idle AST
	     * to fire.  Reset idle flag after process create (not a clean
	     * solution).
	     */
	    pty->idle = 1;
	    status = initiate_user_process ( uai, param.pty_mode,
		pty->devnam, pty->devnam, accpornam, x11_info, errmsg );
	    pty->idle = 0;
	}

        if ( status&1 == 0 ) {
	    /*
	     * login failed, delete the pseudo terminal.
	     */
	    PTD$DELETE ( pty->ctl_chan );
	    if ( pty->devnam[0] ) pty_map_deassign ( pty->devnam );
	    deallocate_context ( pty );
	    return status;
	}
		
    } else {
        /*
         * No initiator, set ASTs to capture prompt string and write a 
	 * carriage return to initiate a login sequence.
         */
	status = ptd_collect_prompt ( pty );
        pty->buf[0].data[0] = '\r';
        status = PTD$WRITE ( pty->ctl_chan, pthread_cond_signal_int_np, 
		&pty->io_done, &pty->buf[0], 1, 
		&pty->buf[2], sizeof(pty->buf[2].data) );
	status = ptd_synch ( status, pty, 0 );
        /*
         * Attempt login with username and password fed to terminal input.
         */
        if ( param.pty_mode == 1 ) while ( (status&1) == 1 ) {
	    /*
	     * Capture line from terminal.
	     */
            status = ptd_get_line ( pty );
	    if ( status&1 ) {
		/*
		 * See if we've read a prompt string yet.
		 */
                if ( pty->prompt_end > 0 && pty->prompt_start < pty->prompt_end ) {
		    int k;
		    pty->line[pty->prompt_end] == '\0';
		    pty->prompt_end = 0;
		    /*
		     * Send response appropriate to prompt.
		     */
		    if ( tu_strncmp( &pty->line[pty->prompt_start], 
		      		"Username: ", 16 ) == 0 ) {
		        tu_strnzcpy ( pty->buf[0].data, uai->username,
				sizeof(pty->buf[0].data)-1 );
		        k = tu_strlen ( pty->buf[0].data );
		        pty->buf[0].data[k] = '\r';
		        status = PTD$WRITE ( pty->ctl_chan, 
			    pthread_cond_signal_int_np, &pty->io_done, 
			    &pty->buf[0], k+1,
			    &pty->buf[2], sizeof(pty->buf[2].data) );
		        status = ptd_synch ( status, pty, 0 );
		        ptd_collect_prompt ( pty );
		    } else if ( tu_strncmp ( &pty->line[pty->prompt_start], 
				"Password: ", 16) == 0 ) {
		        tu_strnzcpy ( pty->buf[0].data, uai->password,
			    sizeof(pty->buf[0].data)-1 );
		        k = tu_strlen ( pty->buf[0].data );
		        pty->buf[0].data[k] = '\r';
		        status = PTD$WRITE ( pty->ctl_chan, 
			    pthread_cond_signal_int_np, &pty->io_done, 
			    &pty->buf[0], k+1,
			    &pty->buf[2], sizeof(pty->buf[2].data) );
		        status = ptd_synch ( status, pty, 0 );
		        /*
		         * Once password written, we are done.
		         */
		        break;
		    }
	        } else {
	           /*
		     * Save for output later.
		     */
	            /* printf("%s", pty->line ); */
	        }
	    }
        }
    }
    /*
     * Setup flow control.
     */
    if ( status & 1 ) {
        status = PTD$SET_EVENT_NOTIFICATION ( pty->ctl_chan, 
		hold_input_ast, pty, 0, PTD$C_SEND_XOFF );

	if ( status&1 ) status = PTD$SET_EVENT_NOTIFICATION ( pty->ctl_chan,
		resume_input_ast, pty, 0, PTD$C_SEND_XON );
    }
    if ( (status&1) == 0 ) {
	printf("error code: %d\n", status );
	tu_strcpy ( errmsg, "Error initializing pty, code: " );
	tu_strint ( status, &errmsg[tu_strlen(errmsg)] );
    }
    *ctx = (void *) pty;
    return status;
}
/**************************************************************************/
/* The following routine manage stalling a write completion after
 * an XOFF.  We assume the xoff AST will fire before finish_write executes.
 */
static void hold_input_ast ( struct ptd_context *pty )
{
    pty->xoffed = 1;
}
static void resume_input_ast ( struct ptd_context *pty )
{
    pty->xoffed = 0;
    if ( pty->xon_waiter ) {
	pty->xon_iosb[0] = 1;
	cport_completion_ast ( pty->xon_waiter );
    }
}

static int retry_finish_write ( cport_isb isb )
{
    isb->iosb = (void *) isb->default_iosb;
    isb->drv->xon_waiter = (cport_isb) 0;
    isb->completion_callback = finish_write;
    return finish_write ( isb );
}
/**************************************************************************/
static int finish_write ( cport_isb isb ) 
{
    struct ptd_context *ctx;
    int status, count, segsize;

    ctx = isb->drv;
    if ( (ctx->buf[0].status&1) == 0 ) {
	if ( ctx->buf[0].status == SS$_DATAOVERUN ||
	     ctx->buf[0].status == SS$_DATALOST ) {
	    /*
	     * Keep going.
	     */
	} else {
	    isb->iosb = (void *) isb->default_iosb;
	    isb->default_iosb[0] = ctx->buf[0].status;
	   return 2;	/* error */
	}
    }
    /*
     * Check for xoffed condition.
     */
    SYS$SETAST(0);
    if ( ctx->xoffed ) {
	/*
	 * Repoint isb's iosb pointer to field set by resume_input AST.
	 */
	isb->iosb = (void *) ctx->xon_iosb;
	ctx->xon_iosb[0] = 0;
	ctx->xon_waiter = isb;
	isb->completion_callback = retry_finish_write;
	SYS$SETAST(1);
	return CPORT_COMPLETE_BUSY;
    }
    SYS$SETAST(1);

    count = ctx->buf[0].count;
    isb->default_iosb[1] += count;
    if ( ctx->pend_in_size > 0 ) {
	/*
	 * more characters to write.
	 */
	segsize = ctx->pend_in_size;
	if ( segsize > MAX_INPUT_SEGMENT ) segsize = MAX_INPUT_SEGMENT;
	memcpy ( ctx->buf[0].data, ctx->pending_in, segsize );
	ctx->pend_in_size -= segsize;
	ctx->pending_in += segsize;
	status = PTD$WRITE ( ctx->ctl_chan, cport_completion_ast, isb,
		   &ctx->buf[0], segsize, 0, 0 );
	if ( (status&1) == 0 ) return 2;
	if ( ctx->idle ) PTD$CANCEL ( ctx->ctl_chan );
	return 0;		/* keep isb on busy queue */
    } else {
	/*
	 * All characters written.  Check for xoffed state and inhibit
	 * completion of I/O.
	 */
	isb->iosb = (void *) isb->default_iosb;
	isb->default_iosb[0] = 1;
	return 2;
    }
}
/**************************************************************************/
/* start_write initiates and asynchronous send of message to remote
 * client.  Data buffer is encrypted and send out.
 */
static int start_write ( cport_isb isb, int func, void *data, int length )
{
    int packet_size, i, pad_bytes, status, segsize;
    struct ptd_context *ctx;
    unsigned char type;

    ctx = isb->drv;
    /*
     * Write function.  Convert end-of-file to control-Z.
     * Write at most 8 characters at a time to ensure we don't
     * overflow typeahead buffer before xoff signalled by driver.
     */
    if ( length == 0 ) {
	length = 1;
	data = (void *) "\032";			/* control-Z */
    }
    segsize = length;
    if ( segsize > MAX_INPUT_SEGMENT ) {
	/*
	 * setup pending state for continuation.
	 */
	segsize = MAX_INPUT_SEGMENT;
	ctx->pend_in_size = length - segsize;
	ctx->pending_in = ((char *) data) + segsize;
    } else {
	ctx->pend_in_size = 0;
	ctx->pending_in = (char *) 0;
    }
    memcpy ( ctx->buf[0].data, data, segsize );

    isb->iosb = (void *) &ctx->buf[0];	/* use IOSB in ptd's io buffer */
    isb->default_iosb[1] = 0;		/* track total chars output */
    isb->completion_callback = finish_write;

    status = PTD$WRITE ( ctx->ctl_chan, cport_completion_ast, isb,
		   &ctx->buf[0], segsize, 0, 0 );
    if ( (status&1) ) {
	cport_mark_busy ( isb );
	if ( ctx->idle ) PTD$CANCEL ( ctx->ctl_chan );
    }

    return status;
}
/**************************************************************************/
static int finish_read ( cport_isb isb )
{
    struct ptd_context *ctx;
    int status, count, segsize;

    ctx = isb->drv;
    if ( (ctx->buf[1].status&1) == 0 ) {
	/*
	 * Check for idle cancel condition and convert to EOF signal.
	 */
	SYS$SETAST ( 0 );
	if ( ctx->idle && (ctx->buf[1].status == SS$_CANCEL) ) {
	    ctx->buf[1].status = 1;
	    ctx->buf[1].count = 0;
	}
	SYS$SETAST ( 1 );
	return 2;	/* error */
    }
    /*
     * Copy read data to user's buffer.
     */
    memcpy ( isb->buffer, ctx->buf[1].data, ctx->buf[1].count );
    return 2;
}
/**************************************************************************/
/* Start_receive initiates receiving of next packet.
 */
static int start_read ( cport_isb isb, int func, void *buffer, int bufsize )
{
    int status;
    struct ptd_context *ctx;

    ctx = isb->drv;
    /*
     * Read characters.
     */
    ctx->pending_out = buffer;
    ctx->pend_out_size = bufsize;
    isb->iosb = (void *) &ctx->buf[1];	/* use IOSB in ptd's io buffer */
    isb->completion_callback = finish_read;
    status = PTD$READ ( 0, ctx->ctl_chan, cport_completion_ast, isb,
		&ctx->buf[1], (bufsize < sizeof(ctx->buf[1].data)) ? 
		bufsize : sizeof(ctx->buf[1].data) );
    if ( (status&1) ) {
	cport_mark_busy ( isb );
	if ( ctx->idle ) PTD$CANCEL ( ctx->ctl_chan );
    }

    return status;
}
/****************************************************************************/
/*
 * The follow 2 routines set up an AST thread for managing asynchronous
 * SETMODE operations on the pseudo-terminals.
 */
static void update_term_ast ( struct tchar_update *buf )
{
    if ( buf->iosb[0]&1 == 1 && buf->update_pending ) {
	/*
	 * Issue new QIO to set the pending characteristics but only if
	 * different than what we last set them to.
	 */
	int status;
	buf->update_pending = 0;
	if ( buf->current.screen_width != buf->pending.screen_width ||
	     buf->current.tchar_pagelen != buf->pending.tchar_pagelen ) {

	    buf->current = buf->pending;
	    status = SYS$QIO ( 0, buf->io_chan, IO$_SETMODE, buf->iosb,
	         update_term_ast, buf,
	        &buf->current, sizeof(buf->current), 0, 0, 0, 0 );
	    if ( status&1 == 1 ) return;
	}
    }
    /*
     * free connection.
     */
    SYS$DASSGN ( buf->io_chan );
    if ( buf == update_list ) {	/* first on list */
	update_list = buf->next;
    } else {
	struct tchar_update *prev;
	for ( prev = update_list; prev; prev = prev->next ) {
	    if ( prev->next == buf ) {
		prev->next = buf->next;
		break;
	    }
	}
    }
    buf->next = update_free;
    update_free = buf;
}

static int update_term_char ( char *terminal, int namlen,
	struct tchar_buf *new_char )
{
    int status;
    struct { int l; char *s; } terminal_dx;
    struct tchar_update *buf;

    terminal_dx.l = namlen;
    terminal_dx.s = terminal;
    /*
     * Disable ASTs and scan current update list for matching block (
     * IO completion AST manipulates the  update_list and update_free
     * lists). We want to ensure only 1 outstanding setmode operation at a
     * time is occurring.  If multiple updates requested while a setmode
     * is pending, only apply the last and only if different.
     */
    pthread_mutex_lock ( &ptd_resource );
    SYS$SETAST(0);
    for ( buf = update_list; buf; buf = buf->next ) {
	if ( buf->namlen == namlen ) if ( 0 == tu_strncmp ( 
		terminal, buf->terminal_name, namlen ) ) {
	    break;
	}
    }
    if ( buf ) {
	/*
	 * Matching buffer found, piggyback this update onto the existing
	 * block.
	 */
	buf->update_pending = 1;
	buf->pending = *new_char;
	SYS$SETAST ( 1 );
	pthread_mutex_unlock ( &ptd_resource );
	return 1;

    } else {
	/*
	 * Allocate a new buffer and initialize, placing at head of update list.
	 */
	if ( update_free ) {
	    buf = update_free;
	    update_free = buf->next;
	} else {
	    SYS$SETAST ( 1 );
	    buf = malloc ( sizeof(struct tchar_update) );
	    SYS$SETAST ( 0 );
	}
	if ( buf ) {
	    buf->update_pending = 0;
	    buf->namlen = namlen;
	    tu_strnzcpy ( buf->terminal_name, terminal, namlen );
	    buf->current = *new_char;
	    buf->next = update_list;
	    update_list = buf;
	}
	SYS$SETAST(1);
	/*
	 * Assign channel.
	 */
	if ( buf ) status = SYS$ASSIGN (&terminal_dx, &buf->io_chan, 0, 0, 0);
	else status = 12;
    }
    pthread_mutex_unlock ( &ptd_resource );
    if ( (status&1) == 1 ) {
	/*
	 * Initiate I/O, AST routine will deassign the channel
	 * and return buffer to free list.
	 */
	status = SYS$QIO ( 0, buf->io_chan, IO$_SETMODE, buf->iosb,
	    update_term_ast, buf,
	    &buf->current, sizeof(buf->current), 0, 0, 0, 0 );
    } else if ( buf ) {
	/*
	 * place buffer back on free list.
	 */
	SYS$SETAST(0);
	buf->next = update_free;
	update_free = buf;
	SYS$SETAST(1);
    }
    return status;
}
/****************************************************************************/
/* Change window function is called when client program changes the size
 * of the terminal window.  The buffer for the function is an array
 * of ints specifying the new row, col, x, y values.  Only the rows and
 * columns of the terminal characteristics are updated.
 *
 * Since the SETMODE operation won't complete until any pending reads to
 * the terminal complete, perform the QIO under an AST level context
 * that is independant of the current thread's I/O.  The structures used
 * garantee that at most 1 SETMODE function at a time is queued to the
 * pseudo-terminal.
 */
static int change_window ( cport_isb isb, int func, void *buffer, int length )
{
    char terminal_name[64];
    struct { unsigned short status, length; int fill; } iosb;
    struct { short length, code; void *buffer; int *retlen;} item[8];
    struct tchar_buf cur_char;
    int status, *geometry, new_width, new_page;
    static int devcode[5] = {		/* all return longword */
	DVI$_DEVCLASS, DVI$_DEVTYPE, DVI$_DEVBUFSIZ, DVI$_DEVDEPEND,
	DVI$_DEVDEPEND2 };
    int devinfo[5], i, namlen;
    struct ptd_context *ctx;

    ctx = isb->drv;
    /*
     * Get the terminals name and current characteristics.
     */
    item[0].code = DVI$_TT_PHYDEVNAM;
    item[0].length = sizeof(terminal_name) - 2;
    item[0].buffer = terminal_name;
    item[0].retlen = &namlen;
    for ( i = 0; i < 5; i++ ) { 
	item[i+1].length=sizeof(devinfo[i]); 
	item[i+1].code = devcode[i];
	item[i+1].buffer = &devinfo[i];
	item[i+1].retlen = (int *) 0; 
	devinfo[i] = 0;
    }
    item[6].code = item[6].length = 0;
    namlen = 0;

    pthread_mutex_lock ( &ctx->lock );
    status = SYS$GETDVI ( 0, ctx->ctl_chan, 0, &item,
		&iosb, 	pthread_cond_signal_int_np, &ctx->io_done, 0 );
    while ( (status&1) && iosb.status == 0 ) {
	pthread_cond_wait ( &ctx->io_done, &ctx->lock );
    }
    pthread_mutex_unlock ( &ctx->lock );
    if ( status&1 ) status = iosb.status;
    if ( (status&1) == 0 ) return status;
    cur_char.class = devinfo[0];
    cur_char.type = devinfo[1];
    cur_char.screen_width = devinfo[2];
    cur_char.tchar_pagelen = devinfo[3];
    cur_char.extended_char = devinfo[4];
    /*
     * Update the characteristics buffer with new value specify by
     * caller.  Check for no effective change.
     */
    geometry = (int *) buffer;
    if ( length < (sizeof(int)*2) ) return 20;
    new_width = (255&geometry[1]);
    new_page = (255&geometry[0]);
    if ( new_width == 0 ) new_width = cur_char.screen_width;
    if ( new_page == 0 ) new_page = cur_char.tchar_pagelen >> 24;
    /* if ( new_width == cur_char.screen_width &&
	( (cur_char.tchar_pagelen>>24) & new_page) == new_page ) return 1; */
    cur_char.screen_width = new_width;
    cur_char.tchar_pagelen = (cur_char.tchar_pagelen&0x0ffffff) |
	(new_page<<24);

    status = update_term_char ( terminal_name, namlen, &cur_char );
    return status;
}
/**************************************************************************/
static int query_exit_status ( cport_isb isb, int func, 
	void *buffer, int length )
{
    *((int *) buffer) = 1;
    isb->default_iosb[0] = 1;
    isb->default_iosb[1] = 4;
    return 1;
}
/****************************************************************************/
/* Declare the functions used in in the stream handler table.
 */
static cport_start_function ftable[4] = {
    start_write,
    start_read,
    change_window,
    query_exit_status
};

static int new_stream ( cport_isb isb, void *context, int dir,
	char errmsg[256] )
{
    struct ptd_context *ctx;

    isb->drv = (struct ptd_context *) context;
    ctx = isb->drv;

    ctx->ref_count++;
    isb->channel = 0;
    return 1;
}

static int destroy_stream ( cport_isb isb )
{
    int status;
    isb->drv->ref_count--;
    if ( isb->drv->ref_count <= 0 ) {
	/*
	 * No more streams attached.
	 */
	PTD$CANCEL ( isb->drv->ctl_chan );
	status = PTD$DELETE ( isb->drv->ctl_chan );
	if ( isb->drv->devnam[0] ) pty_map_deassign ( isb->drv->devnam );

	deallocate_context ( isb->drv );
    }
    else status = 1;
    isb->channel = 0;
    return status;
}

static int cancel_io ( cport_isb isb )
{
    struct ptd_context *ctx;
    int status;
    /*
     * Cancel the TCP stuff.
     */
    ctx = isb->drv;
    return PTD$CANCEL ( ctx->ctl_chan );
}
/*
 * The cportucx_driver table will be specified in cport_assign_stream
 * calls as the handler argument.
 */
cport_stream_handler cportpty_driver = {
	3,					/* mask 3 => 4 functions */
	ftable,
	new_stream,
	destroy_stream,
	cancel_io
};

