/*
 * this module manages the server side of an SSH connection from the time
 * the encryption is turned on (SSH_CMSG_SESSION_KEY confirmed) to the
 * time the credentials of the remote end are established (authentication
 * phase).
 *
 * Authentication phase is handled as a big loop, reading authentication
 * messages and responding until a successful login or a timeout has occurred.
 *
 * Configuration:
 *    This module used the login_timeout and multi_user fields in the
 *    global parameters structure.
 *
 *    The multi-user list specifies an ip-addresses (e.g. 101.11.11.1) that
 *    is to be considered a multi-user hosts for purposes of intrusion database
 *    scans.  Normally the source user for the scan is "SSH_CLIENT", but if
 *    the source address is marked as multi-user, the source user will be
 *    the target user.  Ip-address may specify wild cards.
 *					
 *
 * Author:	David Jones
 * Date:	17-MAY-1998
 * Revised:	1-JUL-1998	Check for successful message scan.
 * Revised:	7-JUL-1998	Preserve password case in auth_info.
 * Revised:	17-JUL-1998	Use new user_info.h interface for UAF data.
 * Revised:	 7-AUG-1998	Update logfails field in UAF on failure.
 * Revised:	 5-SEP-1998	Have rsa_authenticate return success on
 *				lookup failure.  Allow ignore portion
 *                              of username after '/'.
 * Revised:	22-OCT-1998	add param.required_id support.
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ssdef.h>		/* VMS condition codes */
#include <ciadef>		/* VMS intrustion database */
#include <jpidef>		/* job/process info */
#include <secsrvdef>		/* Security services. */
#include <descrip.h>		/* VMS string descriptors */
#define PRV$M_SECURITY 64
#ifdef VAXC
globalref int secsrv$_intruder;
#else
extern int secsrv$_intruder;
#endif

#include "parameters.h"
#include "tutil.h"
#include "cport_sshsess.h"
#include "cport_sshlogin.h"			/* prototypes */
int SYS$HASH_PASSWORD(), LIB$GETJPI(), SYS$SCAN_INTRUSION();
int SYS$ASCTOID(), SYS$FIND_HELD(), SYS$FINISH_RDB();

#include "sslinc:rand.h"			/* randome number functions */

struct requirement {		/* defines access requirement */
    long id;			/* UIC or identifier */
    int test;			/* access test: 0-deny if present, 1-allow */
};
static struct requirement *required_id;
static int required_id_count;		/* zero if no requirements */

static long cur_privs[2];
static int auth_type_enable;
#ifdef DEBUG
static void dump_buffer ( char *heading, void *sbuffer, int bufsize )
{
    int i, j;
    unsigned char *buffer;
    buffer = (unsigned char *) sbuffer;
    printf ( "%s (%d bytes):", heading, bufsize );
    j = 15;
    for ( i = 0; i < bufsize; i++ ) {
	j++;
	if ( j > 15 ) {
	    printf("\n   %04x:", i ); j = 0; }
	printf(" %02x", 255&buffer[i] );
    }
    printf("\n");
}
#else
#define dump_buffer(a,b,c) 1
#endif
/***************************************************************************/
/* Scan required_id parameter and build list of requirement structs.
 */
static int init_requirements ()
{
    int count, i, test, status;
    struct parameter_list_elem *id;
    char *value;
    static char identifier[256];
    static $DESCRIPTOR ( ident_dx, identifier );
    /*
     * Count number of IDs.
     */
    for ( count = 0, id = param.required_id; id; id = id->next ) {
	value = id->value;
	test = 1;		/* default to allow if present */
	if ( value[0] == '+' ) value++;
	if ( value[0] == '-' ) { test = 0; value++; }
	if ( id->value[0]  == '\0' ) continue;	/* ignore null strings */
	count++;
    }
    /*
     * allocate array or requirement definitions
     */
    required_id_count = 0;
    if ( count == 0 ) return 1;
    required_id = (struct requirement *) malloc ( count *
		sizeof(struct requirement) );
    if ( !required_id ) {
	fprintf ( stderr, "Faild to allocated required_id array\n" );
	return 0;
    }
    /*
     * Scan list of ID names again and fill in table.
     */
    for ( i = 0, id = param.required_id; id; id = id->next ) {
	long attrib;
	/*
	 * Get identifier name and strip flags.
	 */
	value = id->value;
	test = 1;		/* default to allow if present */
	if ( value[0] == '+' ) value++;
	if ( value[0] == '-' ) { test = 0; value++; }
	if ( id->value[0]  == '\0' ) continue;	/* ignore null strings */
	/*
	 * Look up identifier.
	 */
	tu_strnzcpy ( identifier, value, sizeof(identifier)-1 );
	/* tu_strupcase ( identifier, identifier ); */
	ident_dx.dsc$w_length = tu_strlen ( identifier );

	status = SYS$ASCTOID ( &ident_dx, &required_id[i].id, &attrib );
	if ( (status&1) == 1 ) {
	    /*
	     * Add to array.
	     */
	    required_id[i].test = test;
	    i++;
	} else {
	    /* report error */
	    fprintf ( stderr, 
		"Failed to lookup '%s' in rights database, status: %d\n",
		identifier, status );
	}
    }
    /*
     * Sort list so we can do binary searches.
     */

    required_id_count = i;
    return 1;
}
/***************************************************************************/
/* Check access against requirements, return 1 if allowed, 0 if denied.
 */
static int search_requirements ( long id )
{
    /*
     * return index of matching table entry or required_id_count if no match.
     */
    int i;
    for ( i = 0; (i < required_id_count); i++ ) {
	if ( required_id[i].id == id ) break;
    }
    return i;
}

static int check_requirements ( struct user_account_info *uai ) 
{
    int allow, i, status;
    long holder[2], id, context;

    if ( required_id_count == 0 ) return 1;	/* no checks */
    /*
     * scan list for matching UIC and set base allow/deny if found.
     */
    allow = 0;
    i = search_requirements ( uai->uic );
    if (  i < required_id_count ) {
	if ( required_id[i].test == 0 ) return 0;
	allow = 1;		/* permit */
    }
    /*
     * retrieve identifiers held by user and scan.  If allow is currently 0,
     * Must have matching rightslist ID to access, if allow is 1 must not
     * have any negative required ids.
     */
    holder[0] = uai->uic;
    holder[1] = 0;
    context = 0;
    for ( ; ; ) {
	status = SYS$FIND_HELD ( holder, &id, 0, &context );
	if ( status == SS$_NOSUCHID ) break;	/* all done */
	if ( (status&1) == 1 ) {
	    i = search_requirements ( id );
	    if ( i < required_id_count ) {
		if ( required_id[i].test == 0 ) {	/* explicit deny */
		    allow = 0;
		    SYS$FINISH_RDB ( &context );
		    break;
		} else {
		    allow = 1;
		}
	    }
	} else {
	    /*
	     * error returned from system service, fail safe.
	     */
	    allow = 0;
	    SYS$FINISH_RDB ( &context );
	}
    }
    return allow;			/* final result */
}
/***************************************************************************/
/* Scan intrustion database, return 0 if login should be denied, or original
 * login status (success or failure) if no matching intrusion data.
 */

static int scan_intrusion ( int login_status, int username_valid,
	struct dsc$descriptor_s *username, unsigned char *remote_addr )
{
    int status, flags, len;
    static $DESCRIPTOR(default_user,"SSH_CLIENT");
    $DESCRIPTOR(remote_node_dx, "");
    struct dsc$descriptor_s *source_user;
    char remote_node[20];
    /*
     * Check that process has SECURITY privilege, do null operation if not.
     */
    if ( (PRV$M_SECURITY&cur_privs[1]) == 0 ) return login_status;
    /*
     * Encode remote address as node name
     */
    tu_strint(remote_addr[0], remote_node );
    len = tu_strlen ( remote_node );
    remote_node[len++] = '.';
    tu_strint(remote_addr[1], &remote_node[len]);
    len += tu_strlen ( &remote_node[len] );
    remote_node[len++] = '.';
    tu_strint(remote_addr[2], &remote_node[len]);
    len += tu_strlen ( &remote_node[len] );
    remote_node[len++] = '.';
    tu_strint(remote_addr[3], &remote_node[len]);
    len += tu_strlen ( &remote_node[len] );
    remote_node[len] = '\0';
    /*
     * Determine source user string to use for scan, use "SSH_CLIENT" if
     * username not valid or remote_node not on mhosts list.
     */
    source_user = &default_user;
    if ( username_valid == 1 ) {
	struct parameter_list_elem *host;
	for ( host = param.multi_user; host; host = host->next ) {
	    if ( tu_strmatchwild ( remote_node, host->value ) == 0 ) {
		source_user = username;
		break;
	    }
	}
    }
    /*
     * Build descriptors and call system service to do intrusion DB scan.
     */
    remote_node_dx.dsc$w_length = len;
    remote_node_dx.dsc$a_pointer = remote_node;
    flags = (username_valid) ? CIA$M_REAL_USERNAME : 0;

    status = SYS$SCAN_INTRUSION ( login_status, username, JPI$K_REMOTE,
	0, &remote_node_dx, source_user, 0, 0, 0, 0, flags );
    if (status&1 == 0) printf("scan intrustion: %d\n", status );
    /*
     * Fail request if intruder detected, otherwise return original
     * login status.
     */
    if ( status == (int) &secsrv$_intruder ) return 0;

    return (1&login_status);
}

/***************************************************************************/
/* Initialization routine called by main routine at program start.
 */
int sshlgn_initialize ( int *auth_types, char *errmsg )
{
    int status, code, known_types;
    struct parameter_list_elem *item;
    /*
     * Initialize login module, auth_types argument returns mask of allowed
     * authentication types.  (Allows for initialization to dynamically
     * load new auth_types).  Selectively enable based on paraemters file.
     */
    known_types = 1<<SSH_AUTH_PASSWORD | 1<<SSH_AUTH_RSA;
    *auth_types = 0;
    auth_type_enable = *auth_types;
    for ( item = param.auth_types; item; item = item->next ) {
	char keyword[64];
	tu_strnzcpy ( keyword, item->value, sizeof(keyword)-1 );
	tu_strupcase ( keyword, keyword );
	if ( tu_strncmp ( keyword, "ALL", 4 ) == 0 ) {
	    *auth_types = known_types;
	} else if ( tu_strncmp ( keyword, "PASSWORD", 9 ) == 0 ) {
	    *auth_types |= (1<<SSH_AUTH_PASSWORD);
	} else if ( tu_strncmp ( keyword, "RSA", 4 ) == 0 ) {
	    *auth_types |= (1<<SSH_AUTH_RSA);
	} else {
	    tu_strcpy ( errmsg, 
		"Unknown or unsupported authorization auth_type ignored: " );
	    tu_strnzcpy ( &errmsg[tu_strlen(errmsg)], item->value,
		sizeof(keyword)-1 );
	    return SS$_BADPARAM;
	}
    }
    *auth_types &= known_types;
    auth_type_enable = *auth_types;
    if ( auth_type_enable == 0 ) {
	fprintf(stderr, "Warning, no authtypes allowed\n");
    }
    /*
     * determine if we have privilege to perform scan_intrusion operations.
     */
    cur_privs[0] = cur_privs[1] = 0;
    code = JPI$_CURPRIV;
    status = LIB$GETJPI ( &code, 0, 0, cur_privs, 0, 0 );
    if ( (status&1) ) {
        if ( (cur_privs[1] &PRV$M_SECURITY) == 0 ) {
	    fprintf(stderr, "Warning, %s\n",
	       "process lacks SECURITY privilege, intrusion scans disabled" );
        }
    } else fprintf(stderr,"Error reading process priv info: %d\n", status );
    /*
     * Initialize the rightslists-based access checks.
     */
    init_requirements();
    return status;
}
/***************************************************************************/
/* Handle the RSA authentication, verify public key and issue challenge
 * to client.  Login dir is default login directory for target user account.
 *
 * Note the the login_dir argument may be temporarily modified by this
 * function and the msg argument will be reused for the client dialog.
 *
 * Return value:
 *    1		Success, check auth_type for SSH_AUTH_RSA, if 0 no match
 *		found for key.
 *   even	Fatal error.
 */
static int rsa_authenticate ( 
	sshsess_session ctx, 		/* session context */
	sshmsg_local *msg,		/* Received message, modified */
	struct user_account_info *uai,	/* SYSUAF data */
	 char *errmsg )
{
    int keylen, status, length, i, xcount;
    char *key, *keyfile;
    sshrsa user_rsakey;
    MD5_CTX digest_ctx;
    unsigned char rsa_challenge[32];
    char enc_challenge[512];
    char rsa_expected_response[MD5_DIGEST_LENGTH];
    /*
     * Extract public key modulus from message and lookup full public
     * key.
     */
    if ( 2 == sshmsg_scan_message ( msg, &keylen, &key ) ) {
	int ld_len;
	dump_buffer ( "public key", key, (keylen+7)/8 );
	/*
	 * Generate authorized keys filename from login dir. and 
	 * global parameter.  If param.user_keyfile is relative, make
	 * it relative to user's home directory.
	 */
	keyfile = uai->login_dir;
	ld_len = tu_strlen(uai->login_dir);	/* save original length */
	if ( param.user_keyfile[0] == '/' ) 
	   keyfile = param.user_keyfile;
	else {
	    tu_strnzcpy ( &uai->login_dir[ld_len], param.user_keyfile,
		sizeof(uai->login_dir)-ld_len-1 );
	}
	/*
	 * Get rsa key and match against user file. On success, errmsg 
	 * contains matching key's description string.
	 */
	user_rsakey = sshrsa_create();
	status = sshrsa_lookup_public_key ( keyfile, uai->uic,
		    keylen, key, user_rsakey, errmsg );
	
	if ( (status&1) == 0 ) {
	    /*
	     * lookup failed change return status back to normal completion.
	     */
	    status = 1;
	} else {
	    /*
	     * Lookup found match, generate SSHSMSG_AUTH_RSA_CHALLENGE.
	     */
	    tu_strnzcpy ( uai->password, errmsg, 39 );
	    RAND_bytes ( rsa_challenge, sizeof(rsa_challenge) );
	    dump_buffer ( "Rand challenge", rsa_challenge, sizeof(rsa_challenge) );
	    status = sshrsa_encrypt_number ( sizeof(rsa_challenge)*8,
			(char *) rsa_challenge, user_rsakey, enc_challenge, 
			sizeof(enc_challenge), &length );
	    dump_buffer ( "Encrypted challenge", enc_challenge, length );
	    sshmsg_format_message ( msg, 
			SSH_SMSG_AUTH_RSA_CHALLENGE, length*8, enc_challenge );

	    status = cport_do_io ( ctx->out_msg, CPORT_WRITE,
			msg, 1, &xcount );
	    if ( status&1 == 1 ) {
		/*
		 * Generate expected response.
		 */
		MD5_Init ( &digest_ctx );
		MD5_Update ( &digest_ctx, (unsigned char *) rsa_challenge, 
			sizeof(rsa_challenge) );
		MD5_Update ( &digest_ctx, ctx->session_id,
			sizeof(ctx->session_id) );
		MD5_Final ( (unsigned char *) rsa_expected_response, 
			&digest_ctx );
		dump_buffer ( "Expected response to challenge", rsa_expected_response,
			 sizeof(rsa_expected_response) );
		/*
		 * Read response and compare.
		 */
		status = cport_do_io ( ctx->in_msg, CPORT_READ, msg, 1, 
			&xcount );
		if ( (status&1) == 1 && 
		    msg->length == sizeof(rsa_expected_response) ) {
		    dump_buffer ( "response from client", msg->data, msg->length );
		    for ( i = 0; i < sizeof(rsa_expected_response); i++ ) {
			if ( rsa_expected_response[i] != msg->data[i] ) break;
		    }
		    /*
		     * Getting to end of buffer means match.
		     */
		    if ( i >= sizeof(rsa_expected_response) ) {
			if ( check_requirements ( uai ) ) ctx->auth_type = 
				SSH_AUTH_RSA;
		    }
		}
	    }
	}
	/*
	 * Cleanup, restore login_dir to original value and delete RSA key.
	 */
	if ( param.user_keyfile[0] != '/' ) uai->login_dir[ld_len] = '\0';
	if ( user_rsakey ) sshrsa_destroy ( user_rsakey );
		
    } else {
	tu_strcpy ( errmsg, "scan failure on auth_rsa message" );
	status = SS$_BADPARAM;
    }
    return status;
}
/***************************************************************************/
/* Handle authentication phase of SSH protocol, reading auth messages from
 * client and validating against local user database (SYSUAF).
 *
 * Return values for sshlgn_authenticate:
 *    0		Failure, errmsg has diagnostic.
 *    1		Success, ctx->ext_username points to username and
 *		ctx->auth_type is authorization method.
 */
int sshlgn_authenticate ( sshsess_session ctx, 
	struct user_account_info *uai, void **ext_auth_info, char *errmsg )
{
    int status, code, length, outlen, response, user_status, pwd_status;
    int xcount;
    sshmsg_local msg, success, failure;

    if ( !ctx->server_ctx ) {
	tu_strcpy ( errmsg, "Session is not a server-side context" );
	return 0;
    }
    ctx->username[0] = '\0';
    ctx->ext_username = ctx->username;
    *ext_auth_info = (void *) 0;
    uai->password[0] = '\0';
    /*
     * Initialize standard responses.
     */
    sshmsg_init_locals ( ctx->locus, 3, &msg, &success, &failure );
    sshmsg_format_message ( &success, SSH_SMSG_SUCCESS );
    sshmsg_format_message ( &failure, SSH_SMSG_FAILURE );
    /*
     * Set timeout on connection.
     */
    cport_set_timeout ( ctx->in_msg, param.login_timeout, 0 );
    /*
     * Read auth messages until we get one we can deal with.
     */
    user_status = pwd_status = 0;		/* no referants */

    for ( ctx->auth_type = 0; ctx->auth_type == 0; ) {
	/*
	 * Get packet.
	 */
	status = cport_do_io ( ctx->in_msg, CPORT_READ, &msg, 1, &xcount );
	if ( (status&1) == 0 ) {
	    tu_strcpy ( errmsg, "Error reading setup packet" );
	    break;
        }
	response = SSH_SMSG_FAILURE;

	if ( msg.type == SSH_CMSG_USER ) {
	    /*
	     * Save username in context structure.
	     */
	    int ulen, slash;
	    char *user;
	    if ( 2 != sshmsg_scan_message ( &msg, &ulen, &user) ) {
		tu_strcpy ( errmsg, "Error scanning CMSG_USER block" );
		break;
	    } else if ( ulen >= sizeof(ctx->username) ) {
		ctx->ext_username = malloc ( ulen+1 );
		if ( !ctx->ext_username ) {
		    tu_strcpy ( errmsg, "Error allocating memory for username" );
		    break;
		}
	    }
	    memcpy ( ctx->ext_username, user, ulen );
	    ctx->ext_username[ulen] = '\0';
	    /*
	     * Verify username string.
	     * Look for / embedded in username and only take portion up to
	     * slash.
	     */
	    for ( slash = 0; ctx->ext_username[slash] &&
		(ctx->ext_username[slash] != '/'); slash++ );
	    uai->uic = 0;
	    status = ssh_get_user_information ( ctx->ext_username,
		slash, uai );
	    if ( status&1 ) {
		user_status = 1;	/* user is known */
	    } else {
		user_status = 2;	/* user is not valid */
	    }
	    /*
	     * If we supported a proxy, we could allow success but since
	     * we don't always return failure (client then sends credentials).
	     */
	    status = cport_do_io ( 
		ctx->out_msg, CPORT_WRITE, &failure, 1, &xcount );

 	} else if ( user_status == 0 ) {
	    /*
	     * Protocol violation, CMSG_USER was not the next message.
	     */
	    tu_strcpy ( errmsg, 
		"Unexpect message type received (expecting SSH_CMSG_USER)" );
	    break;

	} else if ( msg.type == SSH_CMSG_AUTH_RSA &&
		(auth_type_enable&(1<<SSH_AUTH_RSA)) ) {
	    /*
	     * Attempt RSA challenge/response dialog with client.
	     */
	    if ( user_status != 1 ) {
		/* User invalid, can't read authorization file. */
		status = SS$_NOSUCHUSER;
	        status = cport_do_io ( 
			ctx->out_msg, CPORT_WRITE, &failure, 1, &xcount );
	    } else {
		/*
		 * User valid, handle with separate function.
		 */
		status = rsa_authenticate ( ctx, &msg, uai, errmsg );
		if ( (status&1) == 0 ) {
		    tu_strcpy ( errmsg, "Error doing RSA authenticate" );
		    break;
		}

	        status = cport_do_io ( ctx->out_msg, CPORT_WRITE, 
		     ctx->auth_type != 0 ? &success : &failure, 1, &xcount );
	    }

	} else if ( msg.type == SSH_CMSG_AUTH_PASSWORD &&
		(auth_type_enable&(1<<SSH_AUTH_PASSWORD)) ) {
	    int plen, slash; char *pwd, password[36];
	    long cand_hash[2];
	    struct dsc$descriptor_s pwd_dx, user_dx;
	    /*
	     * Only compare password if user actually exists.
	     */
	    user_dx.dsc$b_dtype = user_dx.dsc$b_class = 0;
	    for ( slash = 0; ctx->ext_username[slash] &&
		(ctx->ext_username[slash] != '/'); slash++ );
	    user_dx.dsc$w_length = slash;   /* tu_strlen(ctx->ext_username);*/
	    user_dx.dsc$a_pointer = ctx->ext_username;
	    tu_strupcase ( ctx->ext_username, ctx->ext_username );

	    if ( user_status == 1 ) {
	        /*
	         * Hash the password and compare.  Fixup.
	         */

		if ( 2 != sshmsg_scan_message ( &msg, &plen, &pwd ) ) {
		    tu_strcpy ( errmsg, "Error scanning AUTH_PASSWORD block" );
		    break;
		}
		if ( plen > 32 ) plen = 32;		/* truncate? */

		tu_strnzcpy ( password, pwd, plen );
		tu_strupcase ( password, password );
		pwd_dx.dsc$b_dtype = pwd_dx.dsc$b_class = 0;
		pwd_dx.dsc$w_length = plen; pwd_dx.dsc$a_pointer = password;

		status = SYS$HASH_PASSWORD ( &pwd_dx, uai->algorithm, 
			uai->salt, &user_dx, cand_hash );

	    } else {
		status = SS$_NOSUCHUSER;
	    }

	    if ( (status&1) && 
		  (cand_hash[0]==uai->hash[0] && cand_hash[1]==uai->hash[1])
			&& check_requirements ( uai )  ) {
		/*
		 * success, make sure we aren't a intruder.
		 */
		status = scan_intrusion(status, 1, &user_dx, ctx->remote_host);
		
                if ( (status&1) == 1 ) {
		    ctx->auth_type = SSH_AUTH_PASSWORD;
		    tu_strnzcpy ( uai->password, pwd, 39 < plen ? 39 : plen );
	            status = cport_do_io ( 
			ctx->out_msg, CPORT_WRITE, &success, 1, &xcount );
		} else {
	            status = cport_do_io (
			ctx->out_msg, CPORT_WRITE, &failure, 1, &xcount );
		}
	    } else {
		/*
		 * tell security service about the failure.
		 */
		if ( (status&1) == 1 ) status = SS$_INVLOGIN;
		status = scan_intrusion(status, (user_status==1)?1:0, 
			&user_dx, ctx->remote_host);
	        status = cport_do_io ( 
			ctx->out_msg, CPORT_WRITE, &failure, 1, &xcount );
	    }
	    break;
	} else if ( msg.type == SSH_MSG_DISCONNECT ) {
	    /*
	     * abort login.
	     */
	    break;
	} else {
	    /*
	     * Unknown messages just return failure.
	     */
	    status = cport_do_io ( 
			ctx->out_msg, CPORT_WRITE, &failure, 1, &xcount );
	}
    }
    cport_set_timeout ( ctx->in_msg, 0, 0 );	/* clear timeout */
    sshmsg_rundown_locals ( ctx->locus );
    /*
     * increment logfails in UAF if authenticate didn't work and user known.
     */
    if (user_status==1 && ctx->auth_type==0) ssh_update_user_logfails( uai );
    return (ctx->auth_type == 0) ? 0 : 1;
}
