/*
 * Define routines for SSH Packet Assembler/Disassembler (PAD) module.
 * The PAD layer handles converting SSH messages to send their encrypted
 * form and receiving decrypted messages.
 *
 * I/O of the encrypted packets takes place over a TCP/IP connection
 * opened via the cport_ucx comlpetion port interface.
 *
 * Integer return values of functions are odd for success, even for failure.
 *
 * Author:  David Jones
 * Date:    7-MAY-1998
 * Revised: 15-MAY-1998		add cancel function.
 * Revised: 18-MAY-1998		Accept '\r' as protocol string terminator.
 * Revised:  6-JUN-1998		Fix bug in sshpad_destroy, was attempting
 *				to deallocate fixed portion of sshpad_buf
 *				instead of dynamically allocated buffer.
 * Revised:  30-JUN-1998	Add <string.h> to declare memcpy prototype.
 * Revised:  25-JUL-1998	Optimize packet reads will spillover area.
 * Revised:   8-AUG-1998	Propagate timeout settings to internal streams.
 * Revised:   20-OCT-1998	Add sshpad_set_max_packet function.
 * Revised:   31-OCT-1998	Make random fill a little more random.
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <ssdef.h>

#include "pthread_1c_np.h"
#include "tutil.h"
#include "tmemory.h"
#define CPORT_DRIVER_SPECIFIC struct sshpad_ctx
#include "cport_sshpad.h"
#include "cport_ucx.h"

int SYS$GETTIM();		/* random seed initialzation */

#define MAX_IO_SIZE 4096
#define MAX_ENCRYPTED_SIZE 262144
#define MIN_ENCRYPTED_SIZE 5
#define CRC_SIZE 4
#define PAD_SIZE(edata_len) (8-((edata_len)&7))

#define SSHPAD_FIXED_BUFSIZE 512
#define SSHPAD_MAX_SPILLOVER 1024

static pthread_once_t sshpad_once = PTHREAD_ONCE_INIT;

struct sshpad_buf {			/* manages buffer */
    int alloc;				/* size of buffer */
    int size;				/* target size of current I/O */
    int filled;				/* number of chars sent/recv'd */
    int state;				/* additional state. */
    char *buffer;
    int (*encrypt)(void *, char *, int); /* In-place encryption/decryption */
    void *crypt_state;
    char fixed[SSHPAD_FIXED_BUFSIZE];
};


struct sshpad_ctx {
    struct sshpad_ctx *next;
    int is_server;
    int cipher_type;
    int ref_count;
    int max_data_size;			/* limit on writes */
    void *tcp_context;
    int send_type;			/* data type to send */

    int user_size;
    char *user_buf;			/* caller-supplied buffer for read */
    cport_port port;
    cport_isb in_tcp;
    cport_isb out_tcp;

    struct sshpad_buf outbound;
    struct sshpad_buf inbound;
    /*
     * Spillover is where excess bytes are stored if more than 1 packet
     * arrives in a single read.
     */
    int spillover_count;
    char spillover[SSHPAD_MAX_SPILLOVER];
};


static void store_int ( long value, char *buffer )
{
    union { long l; char c[sizeof(long)]; } tmp;
    tmp.l = value;
    buffer[0] = tmp.c[3];
    buffer[1] = tmp.c[2];
    buffer[2] = tmp.c[1];
    buffer[3] = tmp.c[0];
    return;
}

static long load_int ( char *buffer )
{
    union { long l; char c[sizeof(long)]; } tmp;
    tmp.c[0] = buffer[3];
    tmp.c[1] = buffer[2];
    tmp.c[2] = buffer[1];
    tmp.c[3] = buffer[0];
    return tmp.l;
}

static void random_fill ( int bufsize, char *buffer )
{
    int i;
    for ( i = 0; i < bufsize; i++ ) {
	buffer[i] = ((rand()+(int)(&buffer[i])+bufsize) & 255) - 128;
    }
    return;
}
/****************************************************************************/
/* Compute CRC for buffer and storein reversed order.
 * On VAX, LIB$CRC is optimal.  On Alpha LIB$CRC is 2-3 times slower than the 
 * direct-coded alternative from FSF.
 */
static long crc_table[16];
static long crc_coeff;
static long initial_crc;
static void compute_crc ( char *buffer, int bufsize, unsigned char *crc )
{
    long LIB$CRC();
    union { long l; unsigned char c[sizeof(long)]; } tmp;
    struct {int size; char *data; } buffer_dx;

    buffer_dx.size = bufsize;
    buffer_dx.data = buffer;
    tmp.l = LIB$CRC ( crc_table, &initial_crc, &buffer_dx );
    crc[0] = tmp.c[3];
    crc[1] = tmp.c[2];
    crc[2] = tmp.c[1];
    crc[3] = tmp.c[0];
}
/*
 * One-time initialization for module.
 */
static void sshpad_init()
{
    unsigned int  seed[2];
    
    void LIB$CRC_TABLE();
    tm_initialize();			/* per-thread heaps */

    crc_coeff = 0xedb88320;		/* defined by SSH spec. */
    initial_crc = 0;
    LIB$CRC_TABLE ( &crc_coeff, crc_table );   /* for lib$crc */

    SYS$GETTIM ( seed );
    srand ( seed[0] | 6641 );

    return;
}
/****************************************************************************/
/*
 * Initialize new pad structure and do initial handshake (protocol exchange).
 */
sshpad sshpad_create ( 
	cport_port port,
	void *tcp_context,
	int is_server,			/* true if is server */
	char *our_protocol,		/* SSH protocol string */
	char remote_protocol[128],	/* Remote response */
	char errmsg[256] )
{
    struct sshpad_ctx *ctx;
    int status, transferred, limit, i, j;
    cport_isb tcp_in, tcp_out, cur;
    /*
     * Ensure module initialized.
     */
    status = pthread_once ( &sshpad_once, sshpad_init );
    /*
     * Assign input and output streams to port so we can communicate with
     * it.
     */
    tcp_in = cport_assign_stream ( port, &cportucx_driver, tcp_context, 0 );
    tcp_out = cport_assign_stream ( port, &cportucx_driver, tcp_context, 0 );
    if ( !tcp_in || !tcp_out ) {
	tu_strcpy ( errmsg, "Error assigning tcp streams" );
	return (sshpad) 0;
    }
    /*
     * Do handshake prior to allocating anything.
     */
    cport_set_timeout ( tcp_in, 30, 0 );
    cport_set_timeout ( tcp_out, 30, 0 );
    errmsg[0] = '\0';
    if ( is_server ) {		/* we write first */
	limit = tu_strlen ( our_protocol );
	for ( i = 0; i < limit; i += transferred ) {
	    status = cport_do_io ( tcp_out, CPORT_WRITE,
		&our_protocol[i], limit-i, &transferred );
	    if ( status&1 == 0 ) {
		tu_strcpy ( errmsg, "Error writing protocol string" );
		return (sshpad) 0;
	    }
	}
	for ( i = 0; i < 127; i += transferred ) {
	    status = cport_do_io ( tcp_out, CPORT_READ,
		&remote_protocol[i], 127-i, &transferred );
	    if ( (status&1) == 0 )  {
		tu_strcpy ( errmsg, "Error reading protocol string" );
		return (sshpad) 0;
	    }
	    remote_protocol[i+transferred] = '\0';
	    for ( j = 0; j < transferred; j++ ) {
		if ( remote_protocol[i+j] == '\n' ) { i = 128; break; }
		if ( remote_protocol[i+j] == '\r' ) { i = 128; break; }
		if ( remote_protocol[i+j] == '\0' ) { i = 128; break; }
	    }
	}
    } else {			/* we read first */
	for ( i = 0; i < 127; i += transferred ) {
	    status = cport_do_io ( tcp_out, CPORT_READ,
		&remote_protocol[i], 127-i, &transferred );
	    if ( status&1 == 0 ) {
		tu_strcpy ( errmsg, "Error reading protocol string" );
		return (sshpad) 0;
	    }
	    remote_protocol[i+transferred] = '\0';
	    for ( j = 0; j < transferred; j++ ) {
		if ( remote_protocol[i+j] == '\n' ) { i = 128; break; }
		if ( remote_protocol[i+j] == '\r' ) { i = 128; break; }
		if ( remote_protocol[i+j] == '\0' ) { i = 128; break; }
	    }
	}

	limit = tu_strlen ( our_protocol );
	for ( i = 0; i < limit; i += transferred ) {
	    status = cport_do_io ( tcp_out, CPORT_WRITE,
		&our_protocol[i], limit-i, &transferred );
	    if ( status&1 == 0 ) {
		tu_strcpy ( errmsg, "Error writing protocol string" );
		return (sshpad) 0;
	    }
	
	}
    }
    /*
     * Allocate block from lookaside list.
     */
    ctx = (struct sshpad_ctx *) tm_malloc (sizeof(struct sshpad_ctx));
    if ( !ctx ) {
	tu_strcpy ( errmsg, "Error allocating PAD context" );
	return (sshpad) 0;
    }
    /*
     * set to known state.
     */
    ctx->is_server = is_server;
    ctx->tcp_context = tcp_context;
    ctx->ref_count = 0;
    ctx->max_data_size = 0;
    sshpad_set_max_packet ( (sshpad) ctx, MAX_ENCRYPTED_SIZE );
    ctx->user_size = 0;			/* no read active */
    ctx->user_buf = (char *) 0;
    ctx->port = port;
    cport_set_timeout ( tcp_in, 0, 0 );
    cport_set_timeout ( tcp_out, 0, 0 );
    ctx->in_tcp = tcp_in;
    ctx->out_tcp = tcp_out;

    ctx->outbound.alloc = sizeof(ctx->outbound.fixed);
    ctx->outbound.size = ctx->outbound.filled = ctx->outbound.state = 0;
    ctx->outbound.buffer = ctx->outbound.fixed;
    ctx->outbound.encrypt = (void *) 0;
    ctx->outbound.crypt_state = (void *) 0;

    ctx->inbound.alloc = sizeof(ctx->inbound.fixed);
    ctx->inbound.size = ctx->inbound.filled = ctx->inbound.state = 0;
    ctx->inbound.buffer = ctx->inbound.fixed;
    ctx->inbound.encrypt = (void *) 0;
    ctx->inbound.crypt_state = (void *) 0;

    ctx->spillover_count = 0;

    return (sshpad) ctx;
}
/******************************************************************************/
/* Set max data size for writes, return value is previous setting.
 */
int sshpad_set_max_packet ( sshpad pad, int max_packet_size )
{
    /*
     * save callers arguments in context structure, adjusting for CRC.
     */
    int previous;
    struct sshpad_ctx *ctx;
    ctx = (struct sshpad_ctx *) pad;

    previous = ctx->max_data_size + CRC_SIZE;
    ctx->max_data_size = max_packet_size - CRC_SIZE;
    return previous;
}
/******************************************************************************/
/* Set encryption callbacks.
 */
int sshpad_set_encryption ( sshpad pad,
	int (*inbound)(void *, char *, int), void *in_state,
	int (*outbound)(void *, char*, int), void *out_state )
{
    /*
     * save callers arguments in context structure.
     */
    struct sshpad_ctx *ctx;
    ctx = (struct sshpad_ctx *) pad;
    ctx->inbound.encrypt = inbound;
    ctx->inbound.crypt_state = in_state;
    ctx->outbound.encrypt = outbound;
    ctx->outbound.crypt_state = out_state;

    return 1;
}
/*
 * Tear down pad context, note that encrypt states are not destroyed.
 */
int sshpad_destroy ( sshpad pad )
{
    struct sshpad_ctx *ctx;
    ctx = (struct sshpad_ctx *) pad;
    if ( ctx->ref_count > 0 ) return 3;		/* has streams assigned */
    /* teardown PAD structure */
    if ( ctx->outbound.buffer != ctx->outbound.fixed ) 
		tm_free ( ctx->outbound.buffer );
    if ( ctx->inbound.buffer != ctx->inbound.fixed ) 
		tm_free ( ctx->inbound.buffer );
    tm_free ( ctx );
    return 1;
}

static int finish_write ( cport_isb isb )
{
    struct sshpad_ctx *ctx;
    unsigned short int *iosb;
    int status;
    /*
     * write to out_tcp stream completed, see how many more to write.
     */
    ctx = isb->drv;
    iosb = (unsigned short int *) ctx->out_tcp->iosb;
    if ( (iosb[0]&1) == 0 ) {
	/* progate error */
	memcpy ( iosb, ctx->out_tcp->iosb, 8 );
	return 2;
    }

    ctx->outbound.filled += iosb[1];		/* update number written */
    if ( ctx->outbound.filled < ctx->outbound.size ) {
	/*
	 * send next chunk.
	 */
	int seg_size;
	seg_size = ctx->outbound.size - ctx->outbound.filled;
	if ( seg_size > MAX_IO_SIZE ) seg_size = MAX_IO_SIZE;
        status = cport_start_io ( ctx->out_tcp, CPORT_WRITE,
	    &ctx->outbound.buffer[ctx->outbound.filled], seg_size );
	if ( (status&1) == 0 ) {
	    isb->default_iosb[0] = status;
	    return 2;
	}
	return 0;		/* return to busy queue */
    }
    /*
     * fill in mock IOSB and let I/O return to caller.
     */
    iosb = (unsigned short int *) isb->iosb;
    iosb[0] = 1;
    iosb[1] = isb->length;
    return 2;
}
/**************************************************************************/

static int finish_read ( cport_isb isb )
{
    int seg_size, received_bytes, pad_bytes, status;
    struct sshpad_ctx *ctx;
    unsigned short int *iosb;
    /*
     * read from in_tcp stream completed, see how many more to get.
     */
    ctx = isb->drv;
    iosb = (unsigned short int *) ctx->in_tcp->iosb;
#ifdef DEBUG
    printf("pad layer read completed, status: %d %d, buffer: %d/%d\n", 
	iosb[0], iosb[1], ctx->inbound.filled, ctx->inbound.size );
#endif
    if ( (iosb[0]&1) == 0 ) {
	/* progate error */
	memcpy ( iosb, ctx->in_tcp->iosb, 8 );
	return 2;
    }

    ctx->inbound.filled += iosb[1];		/* update number written */
    if ( ctx->inbound.filled < ctx->inbound.size ) {
	/*
	 * whole packet not read yet, continue.
	 */
	seg_size = ctx->inbound.size - ctx->inbound.filled;
	if ( ctx->spillover_count > 0 ) {
	    /*
	     * Retrieve data from spillover buffer.  We assume buffer.alloc
	     * is at least as big and spillover_count (start_read limits
	     * initial read to min. of buffer.alloc and spillover buffer size)
	     */
#ifdef DEBUG
	printf("pad layer using spillover data: seg=%d, count=%d\n",seg_size,
	ctx->spillover_count );
#endif
	    memcpy ( &ctx->inbound.buffer[ctx->inbound.filled], 
		    ctx->spillover, ctx->spillover_count );
	    ctx->inbound.filled += ctx->spillover_count;
	    ctx->spillover_count = 0;
	    seg_size = ctx->inbound.size - ctx->inbound.filled;
	}
	if ( seg_size > 0 ) {
	    /*
	     * get additional data from network connection.
	     */
	    if ( seg_size > MAX_IO_SIZE ) seg_size = MAX_IO_SIZE;
            status = cport_start_io ( ctx->in_tcp, CPORT_READ,
	        &ctx->inbound.buffer[ctx->inbound.filled], seg_size );
	    if ( (status&1) == 0 ) {
	        isb->default_iosb[0] = status;
	        return 2;
	    }
	    return 0;		/* return to busy queue */
	}
    }
    /*
     * Check what we read and update packet size if needed.
     */
    iosb = (unsigned short int *) isb->iosb;
    if ( ctx->inbound.size == 4 ) {
	/*
	 * If read size was 4, we are initiating read of new packet and have 
	 * finished reading the length field.  compute new size and continue 
         * reading.
	 */
	ctx->inbound.state = 1;
	ctx->inbound.size = load_int ( ctx->inbound.buffer );
	if ( ctx->inbound.size < MIN_ENCRYPTED_SIZE || 
		ctx->inbound.size > MAX_ENCRYPTED_SIZE ) {
	    /* received bad packet, abort */
	    iosb[0] = SS$_BADCHKSUM;
	    return 2;
	}
	/*
	 * Compute number of bytes we need read, adjust for padding 
	 */
	pad_bytes = PAD_SIZE(ctx->inbound.size);
	ctx->inbound.size += (pad_bytes + 4);
	if ( ctx->inbound.size-5 > ctx->user_size ) {
	    /* 
	     * User specified too small a buffer, set overflow
	     * indicator and return needed amount in iosb
	     */
	    iosb[0] = SS$_DATAOVERUN;
	    iosb[1] = (ctx->inbound.size-5) & 0x0ffff;
	    iosb[2] = (ctx->inbound.size-5) >> 16;
	    ctx->inbound.state |= SSHPAD_DATA_OVERRUN;
#ifdef DEBUG
	    printf ( "pad layer setting data overrun bit\n");
#endif
	    if ( ctx->inbound.filled > ctx->inbound.size ) {
		/* Save excess bytes read in spillover area. */
		ctx->spillover_count = ctx->inbound.filled - ctx->inbound.size;
		memcpy ( ctx->spillover, 
		    &ctx->inbound.buffer[ctx->inbound.size],
		    ctx->spillover_count );
		    ctx->inbound.filled = ctx->inbound.size;
	    }
	    return 2;
	}
	if ( ctx->inbound.size > ctx->inbound.alloc ) {
	    /* expand buffer, preserving current contents */
	    char *tmp;
	    ctx->inbound.alloc = ctx->inbound.size + 512;
	    tmp = tm_malloc ( ctx->inbound.alloc );
	    if ( tmp ) {
		memcpy ( tmp, ctx->inbound.buffer, ctx->inbound.alloc );
		if ( ctx->inbound.buffer != ctx->inbound.fixed )
			tm_free ( ctx->inbound.buffer );
		ctx->inbound.buffer = tmp;
	    }
	}
	seg_size = ctx->inbound.size - ctx->inbound.filled;
	if ( seg_size > MAX_IO_SIZE ) seg_size = MAX_IO_SIZE;
	else if ( seg_size < 0 ) {
	    /*
	     * Number read included some of next packet, save in spillover
	     * area.
	     */
	    ctx->spillover_count = (-seg_size);
#ifdef DEBUG
	    printf("pad layer saving spillover data, count: %d\n", 
		ctx->spillover_count);
#endif
	    memcpy ( ctx->spillover, &ctx->inbound.buffer[ctx->inbound.size],
			ctx->spillover_count );
	    seg_size = 0;
	}
	status = cport_start_io ( ctx->in_tcp, CPORT_READ,
		&ctx->inbound.buffer[ctx->inbound.filled], seg_size );
	return 0;
    } else {
	/*
	 * Complete encrypted packet read, decrypt it and verify CRC
	 */
	unsigned char crc[4], *cmp_crc;
	if ( ctx->inbound.encrypt ) {
	    ctx->inbound.encrypt ( ctx->inbound.crypt_state,
		&ctx->inbound.buffer[4], ctx->inbound.size - 4 );
	}
	compute_crc ( &ctx->inbound.buffer[4], 
		ctx->inbound.size-8, crc );
	cmp_crc = (unsigned char *) &ctx->inbound.buffer[ctx->inbound.size-4];
	if ( (crc[0] == cmp_crc[0]) && (crc[1] == cmp_crc[1]) &&
	     (crc[2] == cmp_crc[2]) && (crc[3] == cmp_crc[3]) ) {
	    /*
	     * copy data to user buffer.
	     */
	    pad_bytes = PAD_SIZE(load_int(ctx->inbound.buffer));
	    ctx->inbound.state = 0;
	    iosb[0] = SS$_NORMAL;
	    iosb[3] = ctx->inbound.buffer[pad_bytes+4];
	    
	    received_bytes = ctx->inbound.size-pad_bytes-CRC_SIZE-5;
	    iosb[1] = received_bytes & 0x0ffff;
	    iosb[2] = received_bytes >> 16;
	    memcpy ( ctx->user_buf, &ctx->inbound.buffer[pad_bytes+5],
		received_bytes );
	} else {
	    /*
	     * CRC failure.
	     */
	fprintf(stderr, 
	    "CRC check failure, expected %x %x %x %x, saw %x %x %x %x\n",
	    crc[0], crc[1], crc[2], crc[3], 
	    cmp_crc[0], cmp_crc[1], cmp_crc[2], cmp_crc[3] );
	fprintf(stderr,"Bad packet length: %d\n", ctx->inbound.size );
	    iosb[0] = SS$_PARITY;
	}
    }
    return 2;
}
/**************************************************************************/
/* start_send initiates and asynchronous send of message to remote
 * client.  Data buffer is encrypted and send out.
 */
static int start_write ( cport_isb isb, int func,
	void *data, int length )
{
    int packet_size, i, pad_bytes, status;
    struct sshpad_ctx *ctx;
    unsigned char type;
    /*
     * Compute packet size and make sure outbound buffer is big enough,
     * expanding if needed.
     */
    ctx = isb->drv;
    if ( length > ctx->max_data_size ) return 20;
    type = ctx->send_type;

    pad_bytes = PAD_SIZE(length+1+CRC_SIZE);
    packet_size = (length+pad_bytes+9);
    if ( packet_size > ctx->outbound.alloc ) {
	if ( ctx->outbound.buffer != ctx->outbound.fixed ) 
		tm_free ( ctx->outbound.buffer );
	ctx->outbound.alloc = packet_size + 4096;
	ctx->outbound.buffer = tm_malloc ( ctx->outbound.alloc );
	if ( !ctx->outbound.buffer ) {
	    ctx->outbound.alloc = sizeof(ctx->outbound.fixed);
	    ctx->outbound.buffer = ctx->outbound.fixed;
	    return 0;
	}
    }
    /*
     * Contruct packet: length, pad, code, data, CRC
     */
    store_int ( length+1+CRC_SIZE, ctx->outbound.buffer );
    if ( ctx->outbound.encrypt ) {
	random_fill ( pad_bytes, &ctx->outbound.buffer[4] );
    } else for ( i=0; i < pad_bytes; i++ ) ctx->outbound.buffer[i+4] = '\0';
    ctx->outbound.buffer[pad_bytes+4] = (char) type;
    memcpy ( &ctx->outbound.buffer[pad_bytes+5], data, length );
    compute_crc ( &ctx->outbound.buffer[4], packet_size-8,
		(unsigned char *) &ctx->outbound.buffer[packet_size-4] );
    if ( ctx->outbound.encrypt ) {
	/* Encrypt everything but the length field */
	(*ctx->outbound.encrypt) ( ctx->outbound.crypt_state,
		&ctx->outbound.buffer[4], packet_size - 4 );
    }
    /*
     * Begin sending data.
     */
    ctx->outbound.size = packet_size;
    ctx->outbound.filled = 0;
    ctx->outbound.state = 1;			/* write in progress */
    isb->completion_callback = finish_write;
    if ( isb->timer || ctx->out_tcp->timer )
	cport_copy_timeout ( isb, ctx->out_tcp );

    isb->default_iosb[0] = isb->default_iosb[1] = 0;
    status = cport_start_io ( ctx->out_tcp, CPORT_WRITE,
	ctx->outbound.buffer,
	(packet_size > MAX_IO_SIZE) ? MAX_IO_SIZE : packet_size );

    return status;
}
/**************************************************************************/
/* Start_receive initiates receiving of next packet.  Our target length
 * for the inital read is 4, which is the minimum needed to get the
 * packet length.  The actual read requested will be the minimum of
 * the current buffer allocation and SSHPAD_MAX_SPILLOVER.
 */
static int start_read ( cport_isb isb, int func, void *buffer, int bufsize )
{
    int status, read_size;
    struct sshpad_ctx *ctx;

    ctx = isb->drv;
    ctx->inbound.size = 4;		/* read length initially */
    ctx->inbound.filled = 0;
    ctx->user_size = bufsize;
    ctx->user_buf = (char *) buffer;

    isb->default_iosb[0] = isb->default_iosb[1] = 0;
    isb->completion_callback = finish_read;
    if ( isb->timer || ctx->in_tcp->timer )
	cport_copy_timeout ( isb, ctx->in_tcp );

    read_size = ctx->inbound.alloc;
    if ( read_size > SSHPAD_MAX_SPILLOVER ) read_size = SSHPAD_MAX_SPILLOVER;
    if ( ctx->spillover_count > 0 ) read_size = 0;
#ifdef DEBUG
    printf("pad layer starting read l=%d\n", read_size );
#endif
    status = cport_start_io ( ctx->in_tcp, CPORT_READ,
	ctx->inbound.buffer, /* 16  */ read_size );
    return status;
}
static int start_read_retry ( cport_isb isb, int func, void *buffer, int bufsize )
{
    int status, seg_size;
    struct sshpad_ctx *ctx;

    ctx = isb->drv;
    if ( ctx->inbound.size-5 > bufsize ) return 2104;  /* too small */
    /*
     * ensure buffer is big enough.
     */
    if ( ctx->inbound.size > ctx->inbound.alloc ) {
	    /* expand buffer, preserving current contents */
	    char *tmp;
	    ctx->inbound.alloc = ctx->inbound.size + 512;
	    tmp = tm_malloc ( ctx->inbound.alloc );
	    if ( tmp ) {
		memcpy ( tmp, ctx->inbound.buffer, 
			ctx->inbound.alloc );
		if ( ctx->inbound.buffer != ctx->inbound.fixed )
			tm_free ( ctx->inbound.buffer );
		ctx->inbound.buffer = tmp;
	    }
    }
    /*
     * Set to continue reading, except don't initialize size and filled.
     */
    ctx->user_size = bufsize;
    ctx->user_buf = (char *) buffer;

    isb->default_iosb[0] = isb->default_iosb[1] = 0;
    isb->completion_callback = finish_read;
    seg_size = ctx->inbound.size - ctx->inbound.filled;
    if ( seg_size > MAX_IO_SIZE ) seg_size = MAX_IO_SIZE;
#ifdef DEBUG
    printf("pad layer re-starting read l=%d, spillover_count: %d\n", 
	seg_size, ctx->spillover_count );
#endif
    if ( ctx->spillover_count > 0 || seg_size < 0) seg_size = 0;
    status = cport_start_io ( ctx->in_tcp, CPORT_READ,
	&ctx->inbound.buffer[ctx->inbound.filled], seg_size );
    return status;
}
/*
 * Set send type, complete I/O immediately.
 */
static int set_send_type ( cport_isb isb, int func, void *buffer, int bufsize )
{
    unsigned char *tmp;
    tmp = (unsigned char *) buffer;
    isb->drv->send_type = *tmp;
    if ( bufsize ) cport_mark_completed ( isb, 0 );
    return 1;
}
/****************************************************************************/
/* Declare the functions used in in the stream handler table.
 */
static cport_start_function ftable[4] = {
    start_write,
    start_read,
    set_send_type,
    start_read_retry
};

static int new_stream ( cport_isb isb, void *context, int dir,
	char errmsg[256] )
{
    struct sshpad_ctx *ctx;

    isb->drv = (struct sshpad_ctx *) context;
    ctx = isb->drv;
    /* printf("Stream %x assigned to pad driver, tcp stream: %x %x\n",
	isb, ctx->out_tcp, ctx->in_tcp ); */

    if ( dir == 0 ) ctx->out_tcp->wrapper = isb;
    if ( dir == 1 ) ctx->in_tcp->wrapper = isb;
    ctx->ref_count++;
    isb->channel = 0;
    return 1;
}

static int destroy_stream ( cport_isb isb )
{
    int status;
    /*
     * Only deassign the tcp stream attached to this tcp stream.
     */	
    if ( isb->drv->in_tcp ) if ( isb->drv->in_tcp->wrapper == isb )  {
	isb->drv->in_tcp->wrapper = (cport_isb) 0;
	cport_deassign ( isb->drv->in_tcp );
    }
    if ( isb->drv->out_tcp ) if ( isb->drv->out_tcp->wrapper == isb )  {
	isb->drv->out_tcp->wrapper = (cport_isb) 0;
	cport_deassign ( isb->drv->out_tcp );
    }
    isb->drv->ref_count--;
    if ( isb->drv->ref_count <= 0 ) {
	/*
	 * Last stream deassigned, rundown the pad structure.
	 */
	status = sshpad_destroy ( isb->drv );
    }
    else status = 1;
    isb->channel = 0;
    return status;
}

static int cancel_io ( cport_isb isb )
{
    struct sshpad_ctx *ctx;
    int status;
    /*
     * Cancel the TCP stuff.
     */
    status = 1;
    if ( isb->drv->in_tcp ) if ( isb->drv->in_tcp->wrapper == isb )  {
	isb->drv->in_tcp->wrapper = (cport_isb) 0;
	status = cport_cancel ( isb->drv->in_tcp );
    }
    if ( isb->drv->out_tcp ) if ( isb->drv->out_tcp->wrapper == isb )  {
	isb->drv->out_tcp->wrapper = (cport_isb) 0;
	status = cport_cancel ( isb->drv->out_tcp );
    }
    return status;
}
/*
 * The cportucx_driver table will be specified in cport_assign_stream
 * calls as the handler argument.
 */
cport_stream_handler cportsshpad_driver = {
	3,					/* mask 3 => 4 functions */
	ftable,
	new_stream,
	destroy_stream,
	cancel_io
};
