/*
 * Example driver for completion_port.
 *
 * Author: David Jones
 * Date:   2-JUN-1998
 * Revised:25-NOV-1998			Make allocate_context static.
 */
#include <descrip.h>			/* VMS string descriptors */
#include <iodef.h>			/* QIO function codes */
#include <ssdef.h>			/* system service status codes */
#include <dvidef.h>			/* $GETDVI codes */
#include <dcdef.h>
#include <ttdef.h>
#include <tt2def.h>

#include <stdlib.h>
#include <stdio.h>
#include "pthread_1c_np.h"
#include "tutil.h"
struct terminal_iosb { 			/* IOSB for terminal driver: */
   unsigned short status; 		/*	completion status    */
   unsigned short count; 		/*	offset to terminator */
   unsigned short terminator;		/*	Terminating character*/
   unsigned short term_len;		/*	chars in term. sequence */
};
struct tchar_buf {
    unsigned char class;
    unsigned char type;
    unsigned short screen_width;
    unsigned long tchar_pagelen;		/* page length in high byte */
    unsigned long extended_char;
} current;
/*
 * The cport_isb structure reserves the field '->drv' for use  by drivers,
 * which completion_port.h defines this field as type void * by default.
 * We override the void * type to be a pointer to our private data
 * structure by defining the symbol CPORT_DRIVER_SPECIFIC prior to
 * including completion_port.h.  The driver module can now refer to
 * field x in the private structure as isb->drv->x without cumbersome
 * casts.  Field drv is always a pointer.
 */
#define CPORT_DRIVER_SPECIFIC struct private_st
#include "completion_port.h"
#include "cport_terminal.h"
/*
 * Define structure to hold the driver-specific context information.
 * This structure is completely private to the driver and must
 * be allocated/deallocated by the driver.
 */
struct private_st {
    struct private_st *next;
    int ref_count;		/* # of streams assigned to this context */
    int channel;		/* Channel assigned to terminal */
    int is_terminal;
    int echo;			/* Echo chars */
    int cooked;
    int at_eof;			/* if true, ^Z detected */
    struct tchar_buf mode;
    int tty_modes_len;
    char tty_modes[16];
    struct {
	void *fwd_link;
	int (*handler)(int *, struct private_st *);
 	int arg_count;
	int *cond_addr;
	struct private_st *pctx;
	int cond_value;
    } exblk;
};
/*
 * Define internal routines for maintaining lookaside list of previously
 * allocated (and released) private structs.  Also make once routine.
 */
static pthread_once_t module_setup = PTHREAD_ONCE_INIT;
static pthread_mutex_t module_data;
static struct private_st *free_ctx;
static $DESCRIPTOR(null_device,"_NL:");
int SYS$ASSIGN(), SYS$DASSGN(), SYS$CANCEL(), SYS$QIOW(), LIB$GETDVI();
int SYS$CANEXH(), SYS$DCLEXH();

static void module_init ( )
{
    INITIALIZE_MUTEX ( &module_data );
    free_ctx = (struct private_st *) 0;
}

#define GROUP 2
static struct private_st *allocate_context()
{
    struct private_st *new;
    int i;
    struct iblock { struct private_st instance[GROUP]; } *block;
    /*
     * Allocate new block.
     */
    pthread_mutex_lock ( &module_data );
    if ( !free_ctx ) {
	/*
	 * Grow list allocate multiple.
	 */
	block = (struct iblock *) malloc ( sizeof(struct iblock) );
	if ( block ) {
	    /*
	     * Do one-time initialization for block members (mutexes, etc)
	     * and link into free list.
	     */
	    for ( i = 0; i < GROUP; i++ ) {
		block->instance[i].next = &block->instance[i+1];
	    }
	    block->instance[GROUP-i].next = (struct private_st *) 0;
	    free_ctx = &block->instance[0];

	} else {
	    /*
	     * Serious error, allocation failures.
	     */
	    pthread_mutex_unlock ( &module_data );
	    return free_ctx;
	}
    }
    new = free_ctx;
    free_ctx = new->next;
    pthread_mutex_unlock ( &module_data );
    /*
     * Set allocated context to initial state.
     */
    new->channel = 0;
    new->ref_count = 0;
    new->is_terminal = 0;
    new->echo = 1;
    new->cooked = 1;
    new->at_eof = 0;
    new->tty_modes_len = 0;

    return new;
}

static void deallocate_context ( struct private_st *ctx )
{
    /*
     * Simply push context onto lookaside list.
     */
    pthread_mutex_lock ( &module_data );
    ctx->next = free_ctx;
    free_ctx = ctx;
    pthread_mutex_unlock ( &module_data );
}
/******************************************************************************/
/* The open function creates a private I/O context.  The arguments are
 * completely driver-specific and include anything required for the
 * initialization of the private portion of the I/O context.  The open
 * function must return in on form or another a (void *) that points
 * to a driver context, to be used by the caller as the context argument
 * for subsequent calls to cport_assign_stream.
 */
int cport_terminal_open ( void **context, char *device )
{
    struct private_st *ctx;
    int status, code, class;
    struct dsc$descriptor_s device_dx;
    /*
     * Initialialize module.
     */
    pthread_once ( &module_setup, module_init );
    /*
     * Allocate context.
     */
    ctx = (void *) allocate_context();
    if ( !ctx ) return SS$_INSFMEM;
    /*
     * Allocate a channel to the specified device.
     */
    device_dx = null_device;
    device_dx.dsc$w_length = tu_strlen ( device );
    device_dx.dsc$a_pointer = device;
    status = SYS$ASSIGN ( &device_dx, &ctx->channel, 0, 0, 0 );
    if ( (status&1) == 0 ) {
	deallocate_context ( ctx );
	ctx = (struct private_st *) 0;
    }
    /*
     * Get device class and ensure it is a terminal.
     */
    code = DVI$_DEVCLASS;
    class = 0;
    status = LIB$GETDVI ( &code, &ctx->channel, 0, &class, 0, 0 );
    if ( (status&1) == 0 ) {
	deallocate_context ( ctx );
	ctx = (struct private_st *) 0;
    } else if ( class == DC$_TERM ) {
	/*
	 * Save terminal characteristics.
	 */
	struct terminal_iosb iosb;
	ctx->is_terminal = 1;
        status = SYS$QIOW ( 11, ctx->channel, IO$_SENSEMODE, &iosb, 0, 0,
		&ctx->mode, sizeof(ctx->mode), 0, 0, 0, 0 );
    } else {
	deallocate_context ( ctx );
	ctx = (struct private_st *) 0;

	status = SS$_BADPARAM;
    }
    *context = (void *) ctx;
    return status;
}

/*************************************************************************/
static int restore_char ( int *exit_status, struct private_st *pctx )
{
    struct terminal_iosb iosb; int status;

    status = SYS$CANCEL ( pctx->channel );	/* kill read */

    status = SYS$QIOW ( 11, pctx->channel, IO$_SETMODE, &iosb, 0, 0,
		&pctx->mode, sizeof(pctx->mode), 0, 0, 0, 0 );
    return status;
}
int cport_terminal_set_mode ( cport_isb isb, int echo, int cooked )
{
    int status;
    struct private_st *ctx;

    ctx = isb->drv;
    ctx->echo = echo;
    ctx->cooked = cooked;

    if ( !cooked ) {
	/*
	 * Set terminal mode noecho and pasthru.
	 */
	struct tchar_buf current;
	struct terminal_iosb iosb;

	current = ctx->mode;
	current.tchar_pagelen |= (TT$M_NOECHO);
	current.extended_char |= (TT2$M_PASTHRU);
	status = SYS$QIOW ( 11, isb->channel, IO$_SETMODE, &iosb, 0, 0,
	    &current, sizeof(current), 0, 0, 0, 0 );
	/*
	 * setup exit handler to restore original characteristics.
	 */
	ctx->exblk.handler = restore_char;
	ctx->exblk.arg_count = 2;
	ctx->exblk.cond_addr = &ctx->exblk.cond_value;
	ctx->exblk.pctx = ctx;
	status = SYS$DCLEXH ( &ctx->exblk );
	/*
	 * Construct posix-style modes descriptor.
	 */
	ctx->tty_modes_len = 2;
	ctx->tty_modes[0] = 53;		/* echo */
	ctx->tty_modes[1] = 1;		/* on (remote terminal echos */

	return status;
    } else {
	ctx->tty_modes_len = 0;
    }
    return 1;
}
cport_isb cport_terminal_duplex_stream ( cport_isb isb )
{
    return cport_assign_stream(isb->port, &cportterminal_driver, isb->drv, 0);
}
int cport_terminal_get_mode ( cport_isb isb, char **type, int geom[4],
	int *modes_len, char **modes )
{
    struct private_st *ctx;
    /*
     * convert terminal characteristics to caller's arguments.
     */
    ctx = isb->drv;
    if ( ctx->mode.type == TT$_VT100 ) {
        *type = "vt100";
    } else if ( ctx->mode.type == TT$_VT200_SERIES ) {
	*type = "vt200_series";
    }
    geom[0] = ctx->mode.tchar_pagelen >> 24;
    geom[1] = ctx->mode.screen_width;
    geom[2] = 0;
    geom[3] = 0;
    *modes_len = ctx->tty_modes_len;
    *modes = ctx->tty_modes;
    return 1;
}
/******************************************************************************/
/* The new_stream function is called by cport_assign_stream (via the .attach)
 * function pointer) to bind a newly created isb to the i/o context pointer 
 * returned by the driver's open function.
 *
 * Arguments:
 *    isb	New synchronization block to be assigned to context.
 *    context	I/O context returned by cport_example_open();
 *    opt	Assignment flags/options, unused.
 *    errmsg	Receives error text if assign fails.
 *
 * Return value:
 *    1		Normal, successful completion.
 *   even	Error.
 */
static int new_stream (cport_isb isb, void *context, int opt, char errmsg[256])
{
    struct private_st *ctx;
    /*
     * Initialize the driver-specific field in the isb.  This usually points
     * to the I/O context but is does not have to.  The opt argument
     * has driver-specific interpretation.
     */
    isb->drv = (struct private_st *) context;
    ctx = isb->drv;
    isb->channel = ctx->channel;	/* for cport_qio calls */

    ctx->ref_count++;
    return 1;
}
/******************************************************************************/
/* destroy_stream is called to by cport_deassign (via the .detach function
 * pointer). to unbind an isb from its i/o context.  When a program calls
 * cport_destroy, completion_port.c will deassign any streams still associated.
 * with it.
 */
static int destroy_stream ( cport_isb isb )
{
    struct private_st *ctx;

    ctx = isb->drv;
    ctx->ref_count--;
    if ( ctx->ref_count <= 0 ) {
	/*
	 * Last stream using context deassigned, rundown.
	 */
	if ( ctx->is_terminal ) {
	    int status;
	    restore_char ( &status, ctx );	/* Restore charactistics */
	    SYS$CANEXH ( &ctx->exblk );		/* cancel exit handler */
	}
	SYS$DASSGN ( ctx->channel );
	deallocate_context ( ctx );
    }
    isb->drv = (struct private_st *) 0;
    return 1;
}
/******************************************************************************/
/* cancel_io is called by cport_cancel It should take whatever steps are
 * necessary to cancel any pending asynchronous operations (SYS$CANCEL).
 */
static int cancel_io ( cport_isb isb )
{
    int status;
    status = SYS$CANCEL ( isb->drv->channel );
    return status;
}
/******************************************************************************/
/* Define completion callback routines referenced by start_io functions
 * below.  A completion callback is a function called by cport_next_completion
 * after it has moved the isb from the port's completed queue to the
 * processed queue but before it has returned to the caller.
 *
 * The return value of the completion_callback determines the action
 * cport_next_completion will take with the isb:
 *    CPORT_COMPLETE_BUSY	Move isb back to busy queue (new I/O started).
 *
 *    CPORT_COMPLETE_COMPLETED	Move isb back to completed queue (little use).
 *
 *    CPORT_COMPLETE_PROCESSED	Leave isb on processed queue and return it
 *				to caller.
 */
static int finish_write ( cport_isb isb )
{
    int status, count;
    struct terminal_iosb *iosb;

    return CPORT_COMPLETE_PROCESSED;
}

static int finish_read ( cport_isb isb )
{
    int status, count;
    struct terminal_iosb *iosb;
    char *buffer;
    /*
     * Cast iosb to driver-specific struct.
     */
    iosb = (struct terminal_iosb *) isb->iosb;
    if ( (iosb->status&1) == 0 ) return CPORT_COMPLETE_PROCESSED;
    /*
     * Do fixup.
     */
    count = iosb->count + iosb->term_len;
    if ( isb->drv->cooked ) {
	/*
	 * In cooked mode, convert CR to LF and ^Z to eof flag.
	 */
	if ( iosb->term_len == 1 && iosb->terminator == '\r' ) {
	    buffer = (char *) isb->buffer;
	    buffer[iosb->count] = '\n';
        } else if ( iosb->term_len == 1 && iosb->terminator == '\032' ) {
	    isb->drv->at_eof = 1;
	    buffer = (char *) isb->buffer;
	    buffer[iosb->count] = '\n';		/* change ^Z to newline */
	}
    }
    iosb->count = count;

    return CPORT_COMPLETE_PROCESSED;
}
/******************************************************************************/
/* Define the start_io functions that cport_start_io calls via the ftable
 * array in the stream_handler.
 *
 * A user call to cport_start_io ( isb, func, arg1, arg2 ) results in
 * a call to (*isb->driver.ftable[func&FMASK])( isb, func, arg1, arg2 ).
 *
 * All start_io functions use the following prototype:
 *
 *    int start ( cport_isb isb, int func, void *arg1, void *arg2 );
 *
 *    arguments:
 *        isb		I/O synchronization block.  cport_next_completion
 *			will return this address when I/O completes.
 *
 *	  func		Function value describing operation to perform,
 *			in most cases the function is implied by which
 *			start_io function got called.  Bits outside the
 *			range of FMASK may specify qualifiers.
 *
 *        arg1		Driver and function specific argument.  Being
 *			a pointer, this usually points to the user buffer.
 *
 *        arg2		Driver and function specific argument.  Being an
 *			int type, this argument usually specifies the
 *			buffer length or function qualifier.
 *
 *    return value:
 *        odd		Success.
 *	  even.		Failure.
 *
 * 
 */
static int start_write ( cport_isb isb, int func, void *arg1, int arg2 )
{
    int status, carcon;
    struct private_st *ctx;
    char *buffer;

    ctx = isb->drv;
    carcon = 0;
    if ( ctx->cooked && arg2 > 0 ) {
	/*
	 * Adjust for carriage control.
	 */
        buffer = (char *) arg1;
	if ( buffer[arg2-1] == '\n' ) {
	    /*
	     * replace linefeed feed with CR/LF via P4 arg in QIO.
	     */
	    --arg2;
	    carcon = 0x1000000;		/* 1 postfix newline */
	}
    }
    isb->completion_callback = finish_write;

    return cport_qio ( isb, IO$_WRITEVBLK, 
	arg1, (void *) arg2, 0, (void *) carcon, 0, 0);
}

static int start_read ( cport_isb isb, int func, void *arg1, int arg2 )
{
    int status, fcode;
    struct private_st *ctx;
    /*
     */
    ctx = isb->drv;
    if ( ctx->at_eof ) {
	/*
	 * Fail read immediately.
	 */
	return SS$_ENDOFFILE;
    }

    isb->completion_callback = finish_read;

    return cport_qio2 ( isb, 
	(ctx->echo || !ctx->cooked) ? IO$_READVBLK : IO$_READVBLK|IO$M_NOECHO, 
	isb->buffer, ctx->cooked ? isb->length : 1 );
}

/******************************************************************************/
/* Define global stream handler structure to be pass to cport_assign_stream().
 * by clients wishing to the this driver.  Note that the isb contains a
 * copy of the handler struct, not a pointer to a single instance.  Note
 * also that copies have a common pointer to the start_io ftable.
 *
 * The convention for naming the stream handler is cportXXXXX_driver, where
 * XXXXX is a unqiue driver name (e.g. cportsshmsg_driver is driver for
 * 
 */
#define FMASK 1			/* Must be 2^n -1 */
static cport_start_function ftable[FMASK+1] = {
    start_write,		/* CPORT_WRITE function */
    start_read			/* CPORT_READ function */
};

cport_stream_handler cportterminal_driver = {
    FMASK,			/* Size of ftable - 1 */
    ftable,			/* start_io function table */
    new_stream,			/* Assign stream to I/O context */
    destroy_stream,		/* Deassign stream */
    cancel_io			/* Stop I/O in progress. */
};
