/*
 * Completion port driver for TCP/IP access via UCX QIO interface.
 * When assigning the stream, the driver as cport_ucx_driver and
 * arg2 as channel number returned by cportucx_open_tcp_socket(),
 * cportucx_create_tcp_listener(), or cport_start_io ( , CPORT_FCODES+0,
 * on a listener stream.
 *
 * Completion port functions:
 *    create ( )	Bind stream to I/O context in arg 1.
 *    destroy()
 *    cancel()
 *    start_io()	Perform async. operation:
 *			    CPORT_WRITE		write buffer.
 *			    CPORT_READ		read buffer.
 *			    CPORT_FCODES+0	Accept, buffer must be
 *						address of int.
 *
 * Additional setup functions:
 *    cportucx_create_tcp_listener ( int port, int backlog, int attr_count,
 *	char **attributes, int *io_chan, char errmsg[256] );
 *
 *    cportucx_open_tcp_socket ( char *target_host, int target_port,
 *	int attr_count, char **attributes, int *io_chan, char errmsg[256] );
 *
 *  Author:  David Jones
 *  Revised: 1-JUN-1998		bug in finish_read.
 *  Revised: 15-JUN-1998	work around UCX/Multinet emulator
 *				incompatibility (zero length reads).
 *  Revised: 18-JUN-1998	Add UCX$C_KEEPALIVE def for multinet.
 *  Revised: 21-JUN-1998	Fix freelist expansion bug.
 *  Revised: 21-AUG-1998	Deassign accept_chan on stream destroy.
 *  Revised: 26-JAN-1999	Add registration callback to open_tcp_socket.
 *  Revised: 22-FEB-1999	Fix invalid setmode calls in cancel function.
 *  Revised: 25-FEB-1999	Fix calls QIO_AND_WAIT when kernel threads
 *				present.
 * Revised:  16-MAR-1999	More fiddling with attention ASTs.
 */
#include <stdlib.h>
#include <stdio.h>
#include <iodef.h>
#include <ssdef.h>
#include <descrip.h>
#include <string.h>
#define CPORT_DRIVER_SPECIFIC struct ucx_ctx
#include "completion_port.h"

#include "pthread_1c_np.h"
#include "tutil.h"
#include "cport_ucx.h"		/* verify prototypes */

#ifdef MULTINET
#include "MULTINET_ROOT:[MULTINET.INCLUDE.SYS]TYPES.H"
#include "MULTINET_ROOT:[MULTINET.INCLUDE.NETINET]IN.H"
#include "MULTINET_ROOT:[MULTINET.INCLUDE.SYS]SOCKET.H"
#include "MULTINET_ROOT:[MULTINET.INCLUDE.VMS]INETIODEF.H"

#define SOCKADDRIN sockaddr_in
#define SIN$W_FAMILY sin_family
#define SIN$W_PORT sin_port
#define SIN$L_ADDR sin_addr.s_addr
#define UCX$C_TCP IPPROTO_TCP
#define UCX$C_STREAM SOCK_STREAM
#define UCX$C_AF_INET AF_INET
#define UCX$C_SOCKOPT 1
#define UCX$C_LINGER SO_LINGER
#define UCX$C_REUSEADDR SO_REUSEADDR
#define UCX$C_KEEPALIVE SO_KEEPALIVE
#define INETACP$C_TRANS 2
#define INETACP_FUNC$C_GETHOSTBYNAME 1
#else
#include <UCX$INETDEF.H>
#define sockaddr_in SOCKADDRIN
#endif

static $DESCRIPTOR(ucx_device,"UCX$DEVICE");

static pthread_once_t setup = PTHREAD_ONCE_INIT;
static pthread_mutex_t client_io;	/* lock for I/O synchronization */

int os_kernel_threads();
int SYS$DASSGN(), SYS$QIOW(), SYS$QIO(), SYS$ASSIGN(), SYS$CANCEL();

#define HTONS(_x) (((_x)/256)&255) | (((_x)&255)<<8)
struct ucx_ctx {
    struct ucx_ctx *next;		/* for chaining together */
    int channel;			/* I/O channel */
    int ref_count;			/* number of streams attached */
    int type;				/* 0-open, 1-accept */
    int wrt_attn_armed;
    int read_attn_armed;
    int nb_hist;	/* non-blocking write history */
    struct {
	short int status, count; long devdepend;
    } w_iosb, r_iosb;
    /*
     * Used for listen operations.
     */
    int accept_chan;
    int accept_pending;			/* true if accept_chan valid */
    struct sockname { long length; void *va; int *retadr; long list_end;
	struct SOCKADDRIN a; } remote;
    struct { long length; struct SOCKADDRIN *va;
		int *retadr; long list_end; } sockname;

};
static struct ucx_ctx *free_ctx;
static int free_ctx_alloc;
/*************************************************************************/
#ifdef PTHREAD_USE_D4
/*
 * Draft 4 library lacks upcalls so always synchronize via ASTs signalling
 * a condition variable.
 */
#define QIO_AND_WAIT(cond,chan,func,iosb,p1,p2,p3,p4,p5,p6) \
     (pthread_mutex_lock(&client_io),\
      synch_io(SYS$QIO(1,chan,func,iosb,pthread_cond_signal_int_np,cond,\
		p1,p2,p3,p4,p5,p6),cond, (short *) iosb))
#else
/*
 * With standard V7 library, select synchronization method based upon whether
 * kernel thread upcalls enabled.
 */
int os_kernel_threads();
static int kernel_threads = 0;
#ifdef __ALPHA
#define ENF 128
#else
#define ENF 1
#endif
#define QIO_AND_WAIT(cond,chan,func,iosb,p1,p2,p3,p4,p5,p6) \
 (kernel_threads ? ios(SYS$QIOW(10,chan,func,iosb,0,0,p1,p2,p3,p4,p5,p6), \
	(unsigned short *) iosb ) : \
     (pthread_mutex_lock(&client_io),\
     synch_io(SYS$QIO(ENF,chan,func,iosb,pthread_cond_signal_int_np,\
		cond,p1,p2,p3,p4,p5,p6),cond, (short *) iosb)) )
static int ios ( int status, unsigned short *iosb )
{
    /* Return status in IOSB if original status successful */
    if ( (status&1) == 1 ) status = *iosb;
    return status;
}
#endif
static int synch_io ( int status, pthread_cond_t *condition, short *iosb )
{
    /*
     * Make sure operation is pending (success status).
     */
    if ( (status&1) == 1 ) {
	/*
	 * Loop until predicate (iosb.status) is non-zero.
	 */
	do {
	    pthread_cond_wait ( condition, &client_io );
	} while ( iosb[0] == 0 );
	status = (unsigned short) iosb[0];
    }
    /*
     * Condition satisfied, unlock mutex.
     */
    pthread_mutex_unlock ( &client_io );
    return status;
}
/***************************************************************************/
/* Initialization routine, called once.
 */
static void module_init() {
    int status, i;
    /*
     * Initialize pthread structures.
     */
    status = INITIALIZE_MUTEX ( &client_io );
    /*
     * Initialize free list.
     */
    free_ctx_alloc = 4;
    free_ctx = (struct ucx_ctx *) malloc ( sizeof(struct ucx_ctx) * 4 );
    if ( free_ctx ) {
	for ( i = 0; i < 4; i++ ) {
	    free_ctx[i].next = &free_ctx[i+1];
	}
        free_ctx[3].next = (struct ucx_ctx *) 0;
    }
#ifndef PTHREAD_USE_D4
    /*
     * See if kernel threads available so we can use 'W' form of systems
     * services.
     */
    kernel_threads = os_kernel_threads();
#endif
}
struct ucx_ctx *alloc_context()
{
    struct ucx_ctx *new;
    pthread_mutex_lock ( &client_io );
    if ( !free_ctx ) {
	int i;
	free_ctx = (struct ucx_ctx *) malloc ( sizeof(struct ucx_ctx) *
		free_ctx_alloc );
	if ( free_ctx ) {
	    for ( i = 0; i < free_ctx_alloc; i++ ) {
		free_ctx[i].next = &free_ctx[i+1];
	    }
	    free_ctx[free_ctx_alloc-1].next = (struct ucx_ctx *) 0;
	    free_ctx_alloc += (free_ctx_alloc/2);
	}
    }
    new = free_ctx;
    if ( new ) { free_ctx = new->next; new->next = (struct ucx_ctx *) 0; }
    pthread_mutex_unlock ( &client_io );
    return new;
}
/****************************************************************************/
/* Socket creation routines.  Synchronous routines that return VMS I/O
 * channels for binding into cportucx streams.
 */
static int create_socket ( int attr_count, char **attributes, int *port,
	int backlog, pthread_cond_t *cond, int *chan, char *err_msg )
{
    int status, i, j, addr_count;
    int flag;
    struct sockname { long length; void *va; int *retadr; long list_end;
	struct SOCKADDRIN a; };
    struct sockname local;
    struct sockchar { short protocol; char ptype, domain; } sockchar;
    struct { short length, type; void *ptr; short len1, code1;
	void *addr1; int linger_onoff, linger_time; } sockopt;
    struct  itlst { short length, code;
	  union { struct itlst *lst; int *val; } addr; } olst[12];
    struct { unsigned short sts, count, d1, d2; } iosb;

    status = SYS$ASSIGN ( &ucx_device, chan, 0, 0, 0 );
    if ( (status&1) == 0 ) {
	/*
	 * ASSIGN failure, deallocate and abort.
	 */
	tu_strcpy ( err_msg, "Failed to assign to UCX device" );
	return status;
    }
    /*
     * Set up structures needed for socket creation with default values.
     */
    sockchar.protocol = UCX$C_TCP;
    sockchar.ptype =  UCX$C_STREAM;
    sockchar.domain = UCX$C_AF_INET;

    local.length = sizeof(struct SOCKADDRIN);
    local.va = &local.a;
    local.retadr = 0;
    local.list_end = 0;
    local.a.SIN$W_FAMILY = UCX$C_AF_INET;
    local.a.SIN$W_PORT = HTONS(*port);
    local.a.SIN$L_ADDR = 0;

    sockopt.length = 8; sockopt.type = UCX$C_SOCKOPT;
    sockopt.ptr = &sockopt.len1;
    sockopt.len1 = 8;
    sockopt.code1 = UCX$C_LINGER;
    sockopt.addr1 = &sockopt.linger_onoff;
    sockopt.linger_onoff = 1;
    sockopt.linger_time = 40;

    /*
     * Scan attributes and make modifier list for socket create and bind.
     */
    olst[0].code = UCX$C_SOCKOPT; olst[0].length = 0;
    olst[0].addr.lst = &olst[1];
    for ( j = i = 0; i < attr_count; i++ ) {
	if ( tu_strncmp ( attributes[i], "reuseaddr", 10 ) == 0 ) {
	    j++;
	    olst[0].length += sizeof(struct itlst);
	    olst[j].length = 4; olst[j].code = UCX$C_REUSEADDR;
	    olst[j].addr.val = &flag; flag = 1;

	} else if ( tu_strncmp ( attributes[i], "local_host=", 11 ) == 0 ) {
	} else if ( tu_strncmp ( attributes[i], "local_port=", 11 ) == 0 ) {
	    LOCK_C_RTL
	    local.a.SIN$W_PORT = HTONS(atoi(&attributes[i][11]));
	    UNLOCK_C_RTL
	} else if ( tu_strncmp ( attributes[i], "keepalive", 11 ) == 0 ) {
	    j++;
	    olst[0].length += sizeof(struct itlst);
	    olst[j].length = 4; olst[j].code = UCX$C_KEEPALIVE;
	    olst[j].addr.val = &flag; flag = 1;
	}
    }

    status = QIO_AND_WAIT ( cond, *chan, IO$_SETMODE, &iosb,
	&sockchar, 0, 0, 0, 0, 0 );

    if ( (status&1) && j > 0 ) {
	status = QIO_AND_WAIT ( cond, *chan, IO$_SETMODE, &iosb,
		0, 0, 0, 0, olst, 0 );
    }

    if ( (status&1) ) {
	status = QIO_AND_WAIT ( cond, *chan, IO$_SETMODE, &iosb,
		0, 0, &local, backlog, &sockopt, 0 );
    }
    if ( (status&1) && (*port == 0) ) {
	/*
	 * Return port number actually obtained.
	 */
	int nlen;
	nlen = 0;
	local.retadr = &nlen;
	status = QIO_AND_WAIT ( cond, *chan, IO$_SENSEMODE, &iosb,
		0, 0, &local, 0, 0, 0 );
	if ( (status&1) == 1 ) *port = HTONS(local.a.SIN$W_PORT);
    }
    return status;
}
/***************************************************************************/
/* Convert host name to IP addresses.  Ctx argument is assumed to have a
 * valid channel assigned.
 */
static int translate_hostname ( pthread_cond_t *io_done, int chan, char *name, 
	unsigned long *list, int list_size, int *count )
{
    int i, j, length, status;
    struct { unsigned short length, type; void *ptr; } p1, p2, p4;
    unsigned char subfunc[4];
    struct { unsigned short sts, count, d1, d2; } iosb;
    /*
     * Scan name and see if name is 'dot' format numeric.
     */
    for ( i = 0; name[i]; i++ ) if ( name[i] != '.' &&
	(name[i] < '0' || name[i] > '9') ) break;
    if ( !name[i] ) {
	unsigned long addr[4];
	addr[0] = addr[1] = addr[2] = addr[3] = 0;
	for ( i = j = 0; name[i]; i++ ) {
	    if ( name[i] == '.' ) {
		j++;  if ( j >= 4 ) break;
	    } else {
		addr[j] = addr[j]*10 + (name[i]-'0');
	    }
	}
	if ( !name[i] )	{
	    *count = 1;
	    list[0] = addr[0] | (addr[1]<<8) | (addr[2]<<16) | (addr[3]<<24);
	    return 1;
	}
    }
    /*
     * Build arguments for QIO interface.
     */
    subfunc[0] = INETACP_FUNC$C_GETHOSTBYNAME;
    subfunc[1] = INETACP$C_TRANS;
    subfunc[2] = subfunc[3] = 0;
    p1.type = 0; p1.length = 4; p1.ptr = subfunc;
    p2.type = 0; p2.length = tu_strlen(name); p2.ptr = name;
    if ( p2.length <= 0 ) return 20;
    p4.type = 0; p4.length = list_size*4; p4.ptr = list;

    length = 0;
    status = QIO_AND_WAIT ( io_done, chan, IO$_ACPCONTROL, 
	&iosb, &p1, &p2, &length, &p4, 0, 0 );
    if ( status&1 ) *count = length / sizeof(unsigned long);
    else count = 0;
    return status;
}
/***************************************************************************/
int cportucx_open_tcp_socket ( 
    char *target_host, 			/* Remote host to connect to */
    int target_port, 			/* Report port, (local port 'ANY') */
    int attr_count,			/* Optional attributes */
    char **attributes, 			/* Attribute specifcations, see below */
    void **cnx,	 			/* Handle to connection */
    int (*register_cb)(int, int, void *), /* Register allocated port number */
    void *reg_arg,			/* arbitrary argument for callback */
    char err_msg[256] )			/* Description of error code, if any */
{
    int status, i, addr_count, local_port;
    unsigned long addr_list[128];
    int flag;
    struct sockname { long length; void *va; int *retadr; long list_end;
	struct SOCKADDRIN a; };
    struct { unsigned short sts, count, d1, d2; } iosb;
    struct sockname remote;
    struct ucx_ctx *ctx;
    pthread_cond_t io_done;
    int io_chan;
    /*
     * Do one-time and temporary setup.
     */
    pthread_once ( &setup, module_init );
    INITIALIZE_CONDITION(&io_done);
    /*
     * assign channel.
     */
    io_chan = 0;
    local_port = 0;		/* let driver pick port number */
    status = create_socket ( attr_count, attributes, &local_port, 0,
		&io_done, &io_chan, err_msg );
    if ( (status&1) == 0 ) {
	pthread_cond_destroy ( &io_done );
	return status;
    }
    /*
     * Convert target host name to address.
     */
    status = translate_hostname ( &io_done, io_chan, target_host, 
	addr_list, 12, &addr_count);
    if ( (status&1) == 0 ) {
	tu_strcpy ( err_msg, "Host lookup failure" );
	SYS$DASSGN ( io_chan );
	pthread_cond_destroy ( &io_done );
	return status;
    }
    /* printf("/cportucx/ address count: %d, addr[0]: %x, port: %d\n", addr_count, 
	addr_list[0], target_port ); */
    /*
     * Register port number allocated for socket with caller-supplied callback.
     * The intent is the callback can associate the local port number with
     * the remote client being proxied before the connect takes place and
     * thus avoid a race condition.
     */
    if ( register_cb ) {
	/*
	 * First argument to callback is 1 for add, 0 for abort.
	 */
	status = register_cb ( 1, local_port, reg_arg );
	if ( (status&1) == 0 ) {
	    tu_strcpy ( err_msg, "Error registering local port" );
	    SYS$DASSGN ( io_chan );
	    pthread_cond_destroy ( &io_done );
	    return status;
	}
    }
    /*
     * Attempt connection to all addresses in list.
     */
     if ( (status&1) == 1 ) for ( i = 0; i < addr_count; i++ ) {
	/*
	 * Attempt to connect to next address.
	 */
	remote.length = sizeof(struct SOCKADDRIN);
	remote.va = &remote.a;
	remote.retadr = 0;
        remote.list_end = 0;
	remote.a.SIN$W_FAMILY = UCX$C_AF_INET;
	remote.a.SIN$W_PORT = HTONS(target_port);
	remote.a.SIN$L_ADDR = addr_list[i];

	status = QIO_AND_WAIT ( &io_done, io_chan, IO$_ACCESS,
		&iosb, 0, 0, &remote, 0, 0, 0 );
	if ( (status&1) == 1 ) break;	/* Success! quit trying */
	tu_strcpy ( err_msg, "Connect failure" );
     } else {
        tu_strnzcpy ( err_msg, "Unimplemented", 255 );
     }
    /*
     * If connect failed, unwind from per-thread context to keep a brain-dead
     * thread that retries failed connects from consuming resources.
     */
    if ( (status&1) == 0 ) {
	if ( register_cb ) register_cb ( 0, local_port, reg_arg );
	SYS$DASSGN ( io_chan );
	pthread_cond_destroy ( &io_done );
    }
    /*
     * allocate and initialize context.
     */
    ctx = alloc_context();
    ctx->channel = io_chan;
    ctx->ref_count = 0;
    ctx->type = 0;
    ctx->wrt_attn_armed = ctx->read_attn_armed = ctx->nb_hist = 0;
    *cnx = (void *) ctx;
    return status;
}
/****************************************************************************/
int cportucx_create_tcp_listener ( 
	int port, 		/* Port number to listen on */
	int backlog, 		/* Listen queue depth */
	int attr_count,		/* number of optional attributes */
	char **attributes, 	/* Options for listen */
	void **cnx,	 	/* Result */
	char err_msg[256] )	/* Error message if return status != 1 */
{
    int status, i, channel;
    pthread_cond_t io_done;
    struct ucx_ctx *ctx;
    /*
     * Do one-time setup.
     */
    pthread_once ( &setup, module_init );
    INITIALIZE_CONDITION(&io_done);
    /*
     * Create socket.
     */
    status = create_socket ( attr_count, attributes, &port, backlog,
	&io_done, &channel, err_msg );

    pthread_cond_destroy ( &io_done );
    /*
     * allocate and initialize context.
     */
    ctx = alloc_context();
    ctx->channel = channel;
    ctx->ref_count = 0;
    ctx->type = 2;			/* listen socket */
    *cnx = (void *) ctx;

    return status;
}
/****************************************************************************/
/* Asychronous section (completion port driver).  Define structure for
 * driver private section.
 */
/****************************************************************************/
/* The accept operation.  I/O buffer points to a structure with fields to
 * receive the channel, remote port, and remote address.
 */

static int accept_finish ( cport_isb isb )
{
    int status;
    struct ucx_ctx *ctx, *new;
    /*
     * Restore context, check statuc.
     */
    status = isb->default_iosb[0];
    status = (status & 0xffff);
    isb->flags = 0;
    isb->drv->accept_pending = 0;
    if ( status & 1 ) {
	struct cportucx_accept_result *result;
	/*
	 * Allocate context for the new connection.
	 */
	new = alloc_context();
	new->channel = isb->drv->accept_chan;
	new->ref_count = 0;
	new->type = 1;		/* flag as having been an accept */
	/*
	 * Success, save data in caller's buffer.
	 */
	result = (struct cportucx_accept_result *) isb->buffer;
	result->io_ctx = (void *) new;
	result->remote_port = HTONS ( isb->drv->remote.a.SIN$W_PORT );
	memcpy ( result->remote_address, &isb->drv->remote.a.SIN$L_ADDR, 4 );
    } else {
	SYS$DASSGN ( isb->drv->accept_chan );
    }
    return CPORT_COMPLETE_PROCESSED;		/* deliver to caller */
}

static int start_accept ( cport_isb isb, int func,  void *buffer, int length )
{
    int status;
    struct ucx_ctx *ctx;
    /*
     * Assign channel, abort on failure.
     */
    ctx = isb->drv;
    ctx->accept_chan = 0;
    status = SYS$ASSIGN ( &ucx_device, &ctx->accept_chan, 0, 0 );
    if ( (status&1) == 0 ) return status;
    ctx->accept_pending = 1;
    /*
     * Start async. accept operation that will be completed by accept_finish.
     */
    ctx->sockname.length = sizeof(struct SOCKADDRIN);
    ctx->sockname.va = &ctx->remote.a;
    ctx->sockname.retadr = (int *) &ctx->remote.length;
    ctx->sockname.list_end = 0;
    ctx->remote.a.SIN$W_FAMILY = UCX$C_AF_INET;
    ctx->remote.a.SIN$W_PORT = 0;	/* any port */
    ctx->remote.a.SIN$L_ADDR = 0;

    isb->completion_callback = accept_finish;
    isb->length = length;
    isb->buffer = (char *) buffer;		/* save caller's arguments */

    status = cport_qio ( isb, IO$_ACCESS|IO$M_ACCEPT, 0, 0,
	&ctx->sockname, &ctx->accept_chan, 0, 0 );
    return status;
	
}
/****************************************************************************/
/* The write operation is broken up into 4 routines:
 *    start_write	entry in driver function table (function 0)
 *
 *    finish_write	Completion routine, check status of write and
 *			either let start_io complete or queue attention AST.
 *
 *    retry_write_ast   AST called when buffer space available on socket,
 *			simply re-issue write.
 *
 *    setmode_verify_ast AST to check that SETMODE on socket worked, if not
 *			place status in IOSB and complete the I/O.
 */
void retry_write_ast ( cport_isb isb )
{
    isb->drv->wrt_attn_armed = 0;
    isb->flags = 1;
    SYS$QIO ( 4, isb->channel, IO$_WRITEVBLK|IO$M_NOWAIT, isb->iosb,
	cport_completion_ast, isb, 
	isb->buffer, isb->length > 512 ? 512 : isb->length, 0, 0, 0, 0 );
}

void setmode_verify_ast ( cport_isb isb )
{
    struct ucx_ctx *ctx;

    ctx = isb->drv;
    if ( (ctx->w_iosb.status&1) == 0 ) {
	ctx->wrt_attn_armed = 0;
	memcpy ( isb->iosb, &ctx->w_iosb, 8 );
	cport_completion_ast ( isb );
    }
}

static int finish_write ( cport_isb isb )
{
    struct ucx_ctx *ctx;
    int status;
    /*
     * Check status.
     */
    ctx = isb->drv;
    status = isb->default_iosb[0];
    isb->flags = 0;
    status = (status & 0xffff);
    ctx->nb_hist = (ctx->nb_hist*2)&0x0ffe;
    if ( status == SS$_SUSPENDED ) {
	/*
	 * Set write attention IOSB.  Point iosb for asynch routine at
	 * a different block.
	 */
	ctx->nb_hist |= 1;
/* if ( ctx->nb_hist&2 ) printf("ucx write retry, size: %d, %o\n", 
isb->length, ctx->nb_hist ); */
	isb->completion_callback = finish_write;
	isb->default_iosb[0] = isb->default_iosb[1] = 0;
	ctx->wrt_attn_armed = 1;
	status = SYS$QIO ( 4, isb->channel, IO$_SETMODE|IO$M_WRTATTN,
		&ctx->w_iosb, setmode_verify_ast, isb,
		retry_write_ast, isb, 0, 0, 0, 0 );
	return CPORT_COMPLETE_BUSY;	/* continue to keep isb 'busy' */
    } else return CPORT_COMPLETE_PROCESSED;	/* deliver I/O to caller */
}

static int start_write ( cport_isb isb, int func, void *buffer, int length )
{
    int status;
    struct ucx_ctx *ctx;
    /*
     * check for timeout.
     */
    if ( isb->timer == 2 ) return SS$_TIMEOUT;
    /*
     * Setup callback in order to check for EWOULDBLOCK status.
     */
    ctx = isb->drv;
    isb->completion_callback = finish_write;
    /*
     * do non-blocking write.
     */
    isb->flags = 1;
    status = cport_qio2 ( isb, IO$_WRITEVBLK | IO$M_NOWAIT, buffer, length );
    return status;
}
/****************************************************************************/
/* The read operation is broken up into 4 routines:
 *    start_read	entry in driver function table (function 0)
 *
 *    finish_read	Completion routine, check status of read and
 *			either let start_io complete or queue attention AST.
 *
 *    retry_read_ast   AST called when data available on socket,
 *			simply re-issue read.
 *
 *    setmode_verify_ast2 AST to check that SETMODE on socket worked, if not
 *			place status in IOSB and complete the I/O.
 */
static int finish_read ( cport_isb );		/* forward reference */
static void retry_read_ast ( cport_isb isb )
{
    if ( !isb->drv->read_attn_armed || isb->flags ) {
	fprintf(stderr, 
	   "assertion failed in read attn. ast, attn_armed: %d, flags: %d\n",
		isb->drv->read_attn_armed, isb->flags );
	if ( isb->flags ) return;	/* ignore duplicate ASTs */
    }
    isb->drv->read_attn_armed = 0;
    isb->flags = 1;
    SYS$QIO ( 4, isb->channel, IO$_READVBLK|IO$M_NOWAIT, isb->iosb,
	cport_completion_ast, isb, isb->buffer, isb->length, 0, 0, 0, 0 );
}
static void setmode_verify_ast2 ( cport_isb isb )
{
    struct ucx_ctx *ctx;

    ctx = isb->drv;
    if ( (ctx->r_iosb.status&1) == 0 ) {
	/* setmode operation failed, abort the request */
	ctx->read_attn_armed = 0;
	memcpy ( isb->iosb, &ctx->r_iosb, 8 );
	cport_completion_ast ( isb );
    }
}

static int finish_read ( cport_isb isb )
{
    /*
     * Check status.
     */
    struct ucx_ctx *ctx;
    int status;

    ctx = (struct ucx_ctx *) isb->drv;
    isb->flags = 0;
    status = isb->default_iosb[0];
    status = (status & 0xffff);
    /* printf("/ucx/ read QIO completion status(%x): %d %d\n", isb, status,
	isb->default_iosb[1]); */
    if ( status == SS$_SUSPENDED ) {
	/*
	 * Set read attention IOSB.  Point iosb for asynch routine at
	 * a different block.
	 */
	isb->completion_callback = finish_read;
	isb->default_iosb[0] = isb->default_iosb[1] = 0;
	ctx->read_attn_armed = 1;
	status = SYS$QIO ( 4, isb->channel, IO$_SETMODE|IO$M_READATTN,
		&ctx->r_iosb, setmode_verify_ast2, isb,
		retry_read_ast, isb, 0, 0, 0, 0 );
	return CPORT_COMPLETE_BUSY;	/* continue to keep isb 'busy' */
    }
    else return CPORT_COMPLETE_PROCESSED;		/* I/O completed */
}

static int start_read ( cport_isb isb, int func, void *buffer, int length )
{
    int status;
    /*
     * check for timeout.
     */
    if ( isb->timer == 2 ) return SS$_TIMEOUT;
    /*
     * Setup completion callback in order to check for EWOULDBLOCK status.
     */
    isb->completion_callback = finish_read;
    /*
     * do non-blocking read.
     */
    if ( length > 0 ) {
        isb->flags = 1;
        status = cport_qio2 (isb, IO$_READVBLK | IO$M_NOWAIT, buffer, length);
    } else {
	/*
	 * Multinet doesn't like zero-length reads, fake it.
	 */
	isb->default_iosb[0] = 1;
	isb->default_iosb[1] = 0;
	isb->default_iosb[2] = isb->default_iosb[3] = 0;
	status = cport_mark_completed ( isb, 0 );
    }
    return status;
}

/****************************************************************************/
/* Declare the functions used in in the stream handler table.
 */
static cport_start_function ftable[4] = {
    start_write,
    start_read,
    start_accept,
    start_accept
};

static int new_stream ( cport_isb isb, void *attributes, int channel,
	char errmsg[256] )
{
    struct ucx_ctx *ctx;

    isb->drv = (struct ucx_ctx *) attributes;
    ctx = isb->drv;

    ctx->ref_count++;
    ctx->wrt_attn_armed = ctx->read_attn_armed = 0;
    isb->channel = ctx->channel;
    return 1;
}

static int destroy_stream ( cport_isb isb )
{
    int status;
    isb->drv->ref_count--;
    if ( isb->drv->ref_count <= 0 ) {
	if ( isb->drv->accept_pending ) {
	    isb->drv->accept_pending = 0;
	    SYS$DASSGN ( isb->drv->accept_chan );
	}
	status = SYS$DASSGN ( isb->drv->channel );
	pthread_mutex_lock ( &client_io );
	isb->drv->next = free_ctx;
	free_ctx = isb->drv;
	pthread_mutex_unlock ( &client_io );
    }
    else status = 1;
    isb->channel = 0;
    return status;
}

static int cancel_io ( cport_isb isb )
{
    struct ucx_ctx *ctx;
    int status;
    /*
     * We should reset the attention ASTs.
     */
    ctx = isb->drv;
    if ( ctx->read_attn_armed ) {
        status = SYS$QIOW ( 5, isb->channel, IO$_SETMODE|IO$M_READATTN,
		&ctx->r_iosb, 0, 0,
		0, 0, 0, 0, 0, 0 );
	ctx->read_attn_armed = 0;
    }
    if ( ctx->wrt_attn_armed ) {
        status = SYS$QIOW ( 5, isb->channel, IO$_SETMODE|IO$M_WRTATTN,
		&ctx->w_iosb, 0, 0,
		0, 0, 0, 0, 0, 0 );
	ctx->wrt_attn_armed = 0;
    }
    if ( isb->flags ) return SYS$CANCEL ( isb->channel );
    else {
	/*
	 * No actual QIO in progress to cancel, complete the I/O.
	 */
	isb->default_iosb[0] = SS$_CANCEL;
	return cport_mark_completed ( isb, 1 );
    }
}
/*
 * The cportucx_driver table will be specified in cport_assign_stream
 * calls as the handler argument.
 */
cport_stream_handler cportucx_driver = {
	3,					/* mask 3 => 4 functions */
	ftable,
	new_stream,
	destroy_stream,
	cancel_io
};
