/*
 * This driver allows the program using it to act as an X11 server.
 * Process must have SYSNAM privilege to make a listener.  The driver starts
 * a thread that manages the known object database which is a list of
 * channels listening on object X$Xn and listens on the control mailbox.
 * when a connection occurs, it finds a corresponding virtual server and
 * hands off the request to it.
 *
 * Client threads request a listener context and issue accept operation
 * (similar to the CPORT_UCX driver).  The driver allocates a virtual
 * server context and if necessary initiates a known object for a new
 * X11 server.  If all usernames are unique, the same server number
 * (and hence known object) are used.
 *
 * Author: David Jones
 * Date:   19-NOV-1998
 * Revised: 25-NOV-1998		Make [de]allocate_context routines static.
 * Revsied: 3-DEC-1998		Fix bug in deassign_client list scan.
 */
#include <descrip.h>
#include <iodef.h>
#include <cmbdef.h>			/* mailbox function codes */
#include <ssdef.h>
#include <msgdef.h>
#include <dvidef.h>
#include <prvdef.h>
#if defined(__ALPHA) && defined(PTHREAD_USE_D4)
#include <nfbdef.h>
#else
#define NFB$C_DECLNAME 21
#define NFB$C_DECLOBJ 22
#endif
#define LISTEN_DEPTH 8
static $DESCRIPTOR(mailbox_name,"SSH_X11_KNOWN_OBJECTS");

#include <stdlib.h>
#include <stdio.h>
#include "pthread_1c_np.h"
#include "tutil.h"
#include "parameters.h"
/*
 * The cport_isb structure reserves the field '->drv' for use  by drivers,
 * which completion_port.h defines this field as type void * by default.
 * We override the void * type to be a pointer to our private data
 * structure by defining the symbol CPORT_DRIVER_SPECIFIC prior to
 * including completion_port.h.  The driver module can now refer to
 * field x in the private structure as isb->drv->x without cumbersome
 * casts.  Field drv is always a pointer.
 */
#define CPORT_DRIVER_SPECIFIC struct private_st
#include "completion_port.h"
#include "cport_x11.h"
/*
 * Define structure to manage DECnet known object, shared among several
 * virtual servers.
 */
struct known_object {
    struct known_object *next;		/* linked list */
    int server_number;			/* X11 server number */
    unsigned short chan;		/* I/O channel to net control channel */
    short int unit;			/* unit number of net device */
    struct private_st *client;		/* List of client's for this server # */
};
static struct known_object prime_object;
static struct known_object *free_object;

struct mbx_message {
	short int type;			/* MSG$_xxx */
	short int unit;
	unsigned char name_len;
	char info[295];			/* variable portion */
};
/*
 * Define structure to hold the driver-specific context information.
 * This structure is completely private to the driver and must
 * be allocated/deallocated by the driver.
 */
struct private_st {
    struct private_st *next;
    int ref_count;		/* # of streams assigned to this context */
    int type;			/* 1-connection, 2-listener, 0-closed */
    unsigned short mbx, channel;	/* VMS I/O channels. */
    int server_number;
    char username[40];		/* Username for listener */
    pthread_mutex_t lock;
    cport_isb pending_accept;		/* ISB to signal on queue not-empty */
    int connects_pending;		/* queue of received connects */
    unsigned short pending_chan[LISTEN_DEPTH];
    char *pending_host[LISTEN_DEPTH];
};

int os_kernel_threads();
static pthread_once_t module_setup = PTHREAD_ONCE_INIT;
static pthread_mutex_t module_data;
static pthread_mutex_t client_io;
static pthread_mutex_t monitor_lock;
static pthread_cond_t monitor_wake;
static pthread_cond_t monitor_assign;
static struct private_st *free_ctx;
static $DESCRIPTOR(decnet_device,"_NET:");
int SYS$ASSIGN(), SYS$DASSGN(), SYS$GETTIM(), SYS$GETSYI(), SYS$CREMBX();
int SYS$SETPRV(), LIB$GETDVI(), SYS$QIOW(), SYS$QIO(), SYS$CANCEL();

/*************************************************************************/
#ifdef PTHREAD_USE_D4
/*
 * Draft 4 library lacks upcalls so always synchronize via ASTs signalling
 * a condition variable.
 */
#define QIO_AND_WAIT(cond,chan,func,iosb,p1,p2,p3,p4,p5,p6) \
     (pthread_mutex_lock(&client_io),\
      synch_io(SYS$QIO(1,chan,func,iosb,pthread_cond_signal_int_np,cond,\
		p1,p2,p3,p4,p5,p6),cond, (short *) iosb))
#else
/*
 * With standard V7 library, select synchronization method based upon whether
 * kernel thread upcalls enabled.
 */
typedef void * (*startroutine_t)(void *);
int os_kernel_threads();
static int kernel_threads = 0;
#ifdef __ALPHA
#define ENF 128
#else
#define ENF 1
#endif
#define QIO_AND_WAIT(cond,chan,func,iosb,p1,p2,p3,p4,p5,p6) \
 (kernel_threads ? SYS$QIOW(10,chan,func,iosb,0,0,p1,p2,p3,p4,p5,p6) : \
     (pthread_mutex_lock(&client_io),\
     synch_io(SYS$QIO(ENF,chan,func,iosb,pthread_cond_signal_int_np,\
		cond,p1,p2,p3,p4,p5,p6),cond, (short *) iosb)) )
#endif
static int synch_io ( int status, pthread_cond_t *condition, short *iosb )
{
    /*
     * Make sure operation is pending (success status).
     */
    if ( (status&1) == 1 ) {
	/*
	 * Loop until predicaate (iosb.status) is non-zero.
	 */
	do {
	    pthread_cond_wait ( condition, &client_io );
	} while ( iosb[0] == 0 );
	status = (unsigned short) iosb[0];
    }
    /*
     * Condition satisfied, unlock mutex.
     */
    pthread_mutex_unlock ( &client_io );
    return status;
}
/*************************************************************************/
/* Initialize object block and declare X11 listener.  Caller is assumed to
 * hold monitor_lock mutex.
 */
static int init_listener ( struct known_object *object )
{
    int status;
    struct desc { long size; char *va; } dcb_dx, dcb_name_dx;
    struct decl_block {		/* Descriptor block for network QIO */
	unsigned char type;	/* NFB$C_DECLNAME */
	long code;		/* object number (0) */
    } dcb;
    struct { short status, code; long detail; } iosb;
    char objname[32];
    long priv[2], prev_priv[2];
    /*
     * Assign channel with associated mailbox and get unit number.
     */
    status = SYS$ASSIGN ( &decnet_device, &object->chan, 0, &mailbox_name, 0);
    if ( (status&1) == 1 ) {
	int code, unit;
	code = DVI$_UNIT;
	status = LIB$GETDVI ( &code, &object->chan, 0, &unit, 0, 0 );
	object->unit = unit;
	if ( (status&1) == 0 ) SYS$DASSGN ( object->chan );
    }
    /*
     * Declare the network name.
     */
    if ( (status&1) == 1 ) {
        /*
         * Build object name from server number., form is X$Xnnn and initialize
         * descriptors.
         */
	tu_strcpy ( objname, "X$X0" );
	tu_strint ( object->server_number, &objname[3] );
	dcb_dx.size = sizeof(dcb);
	dcb_dx.va = (char *) &dcb;
	dcb_name_dx.size = tu_strlen ( objname );
	dcb_name_dx.va = objname;
	dcb.type = NFB$C_DECLNAME;
	dcb.code = 0;
        /*
	 * Grant ourselves SYSNAM privilege and declare name.
	 * (unsafe, we should be blocking other threads from running)
	 */
	priv[0] = PRV$M_SYSNAM;
	priv[1] = prev_priv[0] = prev_priv[1] = 0;
	SYS$SETPRV ( 1, priv, 0, prev_priv );
        status = QIO_AND_WAIT ( &monitor_assign, object->chan, 
		IO$_ACPCONTROL, &iosb, &dcb_dx, &dcb_name_dx, 0, 0, 0, 0 );
	if ( (prev_priv[0]&PRV$M_SYSNAM) == 0 ) SYS$SETPRV ( 0, 
		prev_priv, 0, 0 );
        /*
         * Cleanup resources on error.
         */
	if ( (status&1) == 0 ) {
	    SYS$DASSGN ( object->chan );
	}
    }
    return status;
}
/*************************************************************************/
/* Scan known object database and insert context block into client list
 * of first object with no matching username, allocating a new object
 * if necessary.  Return value is pointer to the object used.
 */
static struct known_object *assign_client ( struct private_st *ctx )
{
    int status;
    struct known_object *obj;
    struct private_st *cur;

    pthread_mutex_lock ( &monitor_lock );
    for ( obj = &prime_object; obj; obj = obj->next ) {
	for ( cur = obj->client; cur; cur = cur->next ) {
	    if ( tu_strncmp ( cur->username, ctx->username, 40 ) == 0 ) break;
	}
	if ( !cur ) {
	    /* Username not on current object's list, add it.
	     */
	    ctx->next = obj->client;
	    obj->client = ctx;
	    ctx->server_number = obj->server_number;
	    break;
	}
    }
    if ( !obj && free_object ) {
	/*
	 * Allocate a new object and start listening.
	 */
	obj = free_object;
	free_object = obj->next;
	obj->next = prime_object.next;
	prime_object.next = obj;
	status = init_listener ( obj );
	if ( (status&1) ) {
	    ctx->next = (struct private_st *) 0;
	    obj->client = ctx;
	    ctx->server_number = obj->server_number;
	} else {
	    /*
	     * Serious error, failed to declare object.
	     */
	    obj->next = free_object;
	    free_object = obj;
	    obj = (struct known_object *) 0;
	}
    }
    pthread_mutex_unlock ( &monitor_lock );
    return obj;
}
/*
 * Find context in database and remove.
 */
static void deassign_client ( struct private_st *ctx )
{
    int status;
    struct known_object *obj, *tmp;
    struct private_st *cur, *prev;
    /*
     * Scan list of servers to find one matching context's server number.
     */
    pthread_mutex_lock ( &monitor_lock );
    for ( obj = &prime_object; obj; obj = obj->next ) if (
		ctx->server_number == obj->server_number ) {
	/*
	 * Scan this server number's client list for a match, removing if found.
	 */
	if ( ctx == obj->client ) {	/* first in list */
	    obj->client = ctx->next;
	    cur = ctx;
	} else {
	    prev = obj->client;
	    for ( cur = prev->next; cur; cur = cur->next ) {
		if ( ctx == cur ) {
		    prev->next = cur->next;
		    break;
	        }
		prev = cur;
	    }
	}
	if ( cur ) {
	    /*
	     * Username was matched, rundown object if it is secondary
	     * and has no more clients.
	     */
	    if ( !obj->client && obj != &prime_object ) {
		for (tmp=&prime_object; tmp->next != obj; tmp = tmp->next);
		tmp->next = obj->next;
		SYS$DASSGN ( obj->chan );
		obj->next = free_object;
		free_object = obj;
	    }
	} else fprintf(stderr, 
	    "Internal error, X11 deassign_client could not find context\n");
	break;
    }
    pthread_mutex_unlock ( &monitor_lock );
}
/*************************************************************************/
/* Handle processing of MSG$_CONNECT messages read from the mailbox.
 * Calling thread is assumed to hold the monitor_lock mutex.
 */
static int new_connection ( unsigned short length, struct mbx_message *msg )
{
    int status, i, j, host_ndx;
    struct known_object *obj;
    struct private_st *ctx;
    struct { long l; char *s; } ncb_dx;
    struct { unsigned short status, count, dep[2]; } iosb;
    char *ncb, host[256], username[40];
    unsigned short new_chan;
    struct parameter_list_elem *allowed_host;
    /*
     * Construct ncb and descriptor from contents of message 
     */
    i = msg->name_len;	/* skip device name (NETnnn) */
    if ( i > length ) i = 0;
    ncb = &msg->info[i+1];	/* skip count byte */
    ncb_dx.l = msg->info[i];
    ncb_dx.s = ncb;
    if ( ncb_dx.l > (length-i-1) ) ncb_dx.l = length - i - 1;
    /*
     * Scan database for matching client, return pointer to it's context.
     */
    ctx = (struct private_st *) 0;
    username[0] = '\0';
    host_ndx = 0;
    for ( obj = &prime_object; obj; obj = obj->next ) {
	if ( obj->unit == msg->unit ) {
	    /*
	     * Found known object, parse message and extract host and username.
	     */
	    host[0] = '\0';
	    for ( i = 0; i < ncb_dx.l; i++ ) {
		if ( ncb[i] == ':' && (i < sizeof(host)) ) {
		    if ( tu_strncmp (&ncb[i],"::\"0=",5) == 0 ) {
			tu_strnzcpy ( host, ncb, i );
			i += 5;
			for ( j = i; j < ncb_dx.l && ncb[j] != '/'; j++ );
			if (j-i >= sizeof(username)) j = sizeof(username)+i-1;
			tu_strnzcpy ( username, &ncb[i], j-i );
			ncb[j+3] = '\0';	/* no opt data */
			ncb[j+20] = '"';
			ncb_dx.l = j+21;	/* adjust length */
			break;
		    }
		}
	     }
	    /*
	     * Test if remote host is legal.
	     */
	    for ( allowed_host = param.x11_decnet_node; allowed_host;
		allowed_host = allowed_host->next ) {
		if (tu_strncmp (allowed_host->value,  host, 128) == 0) break;
		host_ndx++;
	    }
	    if ( !allowed_host ) {
#ifdef DEBUG
		printf("Remote host '%s' not on allowed list\n", host );
#endif
		break;		/* host not matched */
	    }
	    /*
	     * Scan active clients for matching username.
	     */
	    for ( ctx = obj->client; ctx; ctx = ctx->next ) {
#ifdef DEBUG
printf("Server %d allows username: '%s'\n", obj->server_number, ctx->username);
#endif
	        if ( tu_strncmp ( ctx->username, username, 40 ) == 0 ) break;
	    }
	    break;		/* stop searching objects */
	}
    }
    /*
     * Build NCB and either accept or reject connection based upon whether
     * match found.
     */
    status = SYS$ASSIGN ( &decnet_device, &new_chan, 0, 0, 0 );
    if ( (status&1) == 0 || !obj ) {
    } else if ( ctx ) {
	/*
	 * Accept connection.
	 */
	status = QIO_AND_WAIT ( &monitor_assign, new_chan,
		IO$_ACCESS, &iosb, 0, &ncb_dx, 0, 0, 0, 0 );
	/* 
	 * Add channel to queue and resume stalled client thread if present.
	 */
	pthread_mutex_lock ( &ctx->lock );
	if ( ctx->connects_pending < LISTEN_DEPTH ) {
	    ctx->pending_chan[ctx->connects_pending] = new_chan;
	    ctx->pending_host[ctx->connects_pending] = allowed_host->value;
	    ctx->connects_pending++;
	    if ( ctx->pending_accept ) {
		cport_lock_port ( ctx->pending_accept );
	        ctx->pending_accept->default_iosb[0] = 1;
		cport_unlock_port ( ctx->pending_accept, 1 );
	        ctx->pending_accept = (cport_isb) 0;
	    }
	} else {
	    /* No free slots in queue, abort */
	    SYS$DASSGN ( new_chan );
	}
	pthread_mutex_unlock ( &ctx->lock );
    } else {
	/*
	 * Username not matched, reject connection.
	 */
	status = QIO_AND_WAIT ( &monitor_assign, new_chan,
		IO$_ACCESS|IO$M_ABORT, &iosb, 0, &ncb_dx, 0, 0, 0, 0 );
	SYS$DASSGN ( new_chan );
    }
    return status;
}
/*************************************************************************/
/* Monitor thread listens to mailbox for messages from DECnet driver
 * and locates the corresponding object or client block in the database.
 */
static void *mailbox_monitor ( int *ready )
{
    int status;
    struct mbx_message message;
    struct {
	unsigned short status, count;
	long sender_pid;
    } iosb;
    unsigned short mbx_chan, ef;
    /*
     * Create mailbox.
     */
    ef = 4;
    status = SYS$CREMBX ( 0, &mbx_chan, sizeof(message), 
	sizeof(message)*2, 0x0ff00, 0,
	&mailbox_name, 0, 0 );
    pthread_mutex_lock ( &module_data );
    *ready = 1;
    pthread_cond_signal ( &monitor_assign );
    pthread_mutex_unlock ( &module_data );
    /*
     * Assign channel to prime object.
     */
    pthread_mutex_lock ( &monitor_lock );
    status = init_listener ( &prime_object );
    /*
     * Start initial read.
     */
    status = SYS$QIO ( ef, mbx_chan, IO$_READVBLK, &iosb, 
	pthread_cond_signal, &monitor_wake,
	&message, sizeof(message), 0, 0, 0, 0 );
    while ( (status&1) == 1 ) {
	/*
	 * wait for signal from I/O completion
	 */
	pthread_cond_wait ( &monitor_wake, &monitor_lock );
	if ( iosb.status != 0 ) {
	    /*
	     * Interpret mailbox message.
	     */
	    if ( iosb.status&1 == 0 ) message.type = 0;
	    if ( iosb.count < 4 ) message.type = 0;

	    switch ( message.type ) {
		case MSG$_CONNECT:	/* New connection request */
		    /*
		     * New connection.
		     */
		    new_connection ( iosb.count, &message );
		    break;
		case MSG$_NETSHUT:	/* DECnet shutdown */
		    status = SS$_SHUT;
		    break;
		default:			/* Illegal message */
		    break;
	    }
	    /*
	     * Issue new read.
	     */
	    if ( status&1 ) status = SYS$QIO ( ef, mbx_chan, IO$_READVBLK, 
		&iosb, 	pthread_cond_signal, &monitor_wake,
		&message, sizeof(message), 0, 0, 0, 0 );

	}
    }
    pthread_mutex_unlock ( &monitor_lock );
    return (void *) 0;
}
/*************************************************************************/
/*
 * Define internal routines for maintaining lookaside list of previously
 * allocated (and released) private structs.  Also make once routine.
 */
static void module_init ( )
{
    int status, ready;
    static pthread_attr_t monitor_attr;
    static pthread_t monitor_id;
    struct parameter_list_elem *node;

    INITIALIZE_MUTEX ( &module_data );
    INITIALIZE_MUTEX ( &monitor_lock );
    INITIALIZE_CONDITION ( &monitor_wake );
    INITIALIZE_CONDITION ( &monitor_assign );
    status = INITIALIZE_MUTEX ( &client_io );
    free_ctx = (struct private_st *) 0;
    /*
     * Convert node names to upper case.
     */
    for ( node = param.x11_decnet_node; node; node = node->next ) {
	tu_strupcase ( node->value, node->value );
    }
    /*
     * Initialize known object list.
     */
    if ( param.x11_servers > 0 ) {
	int status, i;
	/*
	 * Initialize prime object.
	 */
	prime_object.next = (struct known_object *) 0;
	prime_object.server_number = param.x11_server_number;
	prime_object.unit = 0;
	prime_object.client = (struct private_st *) 0;
	/*
	 * Pre-initialize free list with server numbers.
	 */
	LOCK_C_RTL
	free_object = (struct known_object *) malloc ( param.x11_servers *
		sizeof(struct known_object) );
	UNLOCK_C_RTL
	if ( !free_object ) {
	    fprintf ( stderr, "Error initializing X11 free object list\n");
	} else {
	    for ( i = 0; i < (param.x11_servers-1); i++ ) {
		free_object[i].next = &free_object[i+1];
		free_object[i].server_number = 1 + param.x11_server_number + i;
	    }
	    free_object[param.x11_servers-1].next = (struct known_object *) 0;
	}
	/*
 	 * Start monitor thread to read mailbox messages.
	 */
	pthread_mutex_lock ( &module_data );
	ready = 0;
	pthread_mutex_unlock ( &module_data );
	INITIALIZE_THREAD_ATTR ( &monitor_attr );
	pthread_attr_setinheritsched ( &monitor_attr, PTHREAD_EXPLICIT_SCHED );
	pthread_attr_setstacksize ( &monitor_attr, 30000 );
    
#ifndef PTHREAD_USE_D4
	/*
	 * See if kernel threads available so we can use 'W' form of system
	 * services.
	 */
	kernel_threads = os_kernel_threads();
	status = pthread_create ( &monitor_id, &monitor_attr,
		(startroutine_t) mailbox_monitor, &ready );
#else
	status = pthread_create ( &monitor_id, monitor_attr,
		(pthread_startroutine_t) mailbox_monitor, &ready );
#endif
	/*
	 * Wait for monitor thread to initialize and signal us.  Since we
	 * are in a once routine, there is no conflict in using monitor_assign.
	 */
	pthread_mutex_lock ( &module_data );
	while (ready == 0) pthread_cond_wait ( &monitor_assign, &module_data );
	pthread_mutex_unlock ( &module_data );
    }
}

#define GROUP 6
static struct private_st *allocate_context()
{
    struct private_st *new;
    int i;
    struct iblock { struct private_st instance[GROUP]; } *block;
    /*
     * Allocate new block.
     */
    pthread_mutex_lock ( &module_data );
    if ( !free_ctx ) {
	/*
	 * Grow list allocate multiple.
	 */
	block = (struct iblock *) malloc ( sizeof(struct iblock) );
	if ( block ) {
	    /*
	     * Do one-time initialization for block members (mutexes, etc)
	     * and link into free list.
	     */
	    for ( i = 0; i < GROUP; i++ ) {
		block->instance[i].next = &block->instance[i+1];
		INITIALIZE_MUTEX ( &block->instance[i].lock );
	    }
	    block->instance[GROUP-i].next = (struct private_st *) 0;
	    free_ctx = &block->instance[0];

	} else {
	    /*
	     * Serious error, allocation failures.
	     */
	    pthread_mutex_unlock ( &module_data );
	    return free_ctx;
	}
    }
    new = free_ctx;
    free_ctx = new->next;
    pthread_mutex_unlock ( &module_data );
    /*
     * Set allocated context to initial state.
     */
    new->channel = 0;
    new->mbx = 0;
    new->ref_count = 0;
    new->connects_pending = 0;
    new->pending_accept = (cport_isb) 0;

    return new;
}

static void deallocate_context ( struct private_st *ctx )
{
    /*
     * Simply push context onto lookaside list.
     */
    pthread_mutex_lock ( &module_data );
    ctx->next = free_ctx;
    free_ctx = ctx;
    pthread_mutex_unlock ( &module_data );
}
/******************************************************************************/
/* The open function creates a private I/O context.  The arguments are
 * completely driver-specific and include anything required for the
 * initialization of the private portion of the I/O context.  The open
 * function must return in one form or another a (void *) that points
 * to a driver context, to be used by the caller as the context argument
 * for subsequent calls to cport_assign_stream.
 */
int cportx11_create_x11_listener (
	char *username,			/* allowed username */
	int *server_number,		/* X11 server number to use */
	void **listen_ctx,		/* Context for assigns */
	char errmsg[256] )		/* Error diagnositc text */
{
    struct private_st *ctx;
    struct known_object *obj;
    int status, maxmsg, bufquo;
    char *slash;
    /*
     * Initialialize module.
     */
    pthread_once ( &module_setup, module_init );
    /*
     * Allocate context.
     */
    *server_number = 0;
    ctx = (void *) allocate_context();
    if ( !ctx ) return SS$_INSFMEM;
    ctx->type = 2;
    /*
     * Upcase and trim username, saving result in context structure.
     */
    tu_strnzcpy ( ctx->username, username, sizeof(ctx->username)-1 );
    tu_strupcase ( ctx->username, ctx->username );
    slash = tu_strstr ( ctx->username, "/" );
    if ( slash ) *slash = '\0';
    /*
     * Assign it to a server not already in use by this username.
     */
    obj = assign_client ( ctx );
    if ( obj ) {
	/* Success */
	*server_number = ctx->server_number;
	status = SS$_NORMAL;

    } else {
	deallocate_context ( ctx );
	ctx = (struct private_st *) 0;
	status = 44;
    }
    *listen_ctx = (void *) ctx;
    return status;
}
/******************************************************************************/
/* The new_stream function is called by cport_assign_stream (via the .attach)
 * function pointer) to bind a newly created isb to the i/o context pointer 
 * returned by the driver's open function.
 *
 * Arguments:
 *    isb	New synchronization block to be assigned to context.
 *    context	I/O context returned by cport_example_open();
 *    opt	Assignment flags/options, unused.
 *    errmsg	Receives error text if assign fails.
 *
 * Return value:
 *    1		Normal, successful completion.
 *   even	Error.
 */
static int new_stream (cport_isb isb, void *context, int opt, char errmsg[256])
{
    struct private_st *ctx;
    /*
     * Initialize the driver-specific field in the isb.  This usually points
     * to the I/O context but is does not have to.  The opt argument
     * has driver-specific interpretation.
     */
    isb->drv = (struct private_st *) context;
    ctx = isb->drv;
    isb->channel = ctx->channel;		/* for cport_qio calls */

    ctx->ref_count++;
    /*
     * Initialzation varies with context type.
     */
    if ( ctx->type == 2 ) {		/* listener, no channel. */
    }
    return 1;
}
/******************************************************************************/
/* destroy_stream is called to by cport_deassign (via the .detach function
 * pointer). to unbind an isb from its i/o context.  When a program calls
 * cport_destroy, completion_port.c will deassign any streams still associated.
 * with it.
 */
static int destroy_stream ( cport_isb isb )
{
    struct private_st *ctx;

    ctx = isb->drv;
    ctx->ref_count--;
    if ( ctx->ref_count <= 0 ) {
	/*
	 * Last stream using context deassigned, rundown.
	 */
	if ( ctx->type == 1 ) {
	    /*
	     * X11 connection closed, deassign channel.
	     */
	    SYS$DASSGN ( ctx->channel );
	} else if ( ctx->type == 2 ) {
	    /*
	     * listener closed, remove from object database.
	     */
	    deassign_client ( ctx );
	}
	deallocate_context ( ctx );
    }
    isb->drv = (struct private_st *) 0;
    return 1;
}
/******************************************************************************/
/* cancel_io is called by cport_cancel It should take whatever steps are
 * necessary to cancel any pending asynchronous operations (SYS$CANCEL).
 */
static int cancel_io ( cport_isb isb )
{
    struct private_st *ctx;

    ctx = isb->drv;
    if ( ctx->type == 1 ) SYS$CANCEL ( ctx->channel );
    else if ( ctx->type == 2 ) {
	pthread_mutex_lock ( &ctx->lock );
	if ( ctx->pending_accept ) {
	    ctx->pending_accept = (cport_isb) 0;
	    isb->default_iosb[1] = SS$_CANCEL;
	}
	pthread_mutex_unlock ( &ctx->lock );
    }
    return 1;
}
/******************************************************************************/
/* Define completion callback routines referenced by start_io functions
 * below.  A completion callback is a function called by cport_next_completion
 * after it has moved the isb from the port's completed queue to the
 * processed queue but before it has returned to the caller.
 *
 * The return value of the completion_callback determines the action
 * cport_next_completion will take with the isb:
 *    CPORT_COMPLETE_BUSY	Move isb back to busy queue (new I/O started).
 *
 *    CPORT_COMPLETE_COMPLETED	Move isb back to completed queue (little use).
 *
 *    CPORT_COMPLETE_PROCESSED	Leave isb on processed queue and return it
 *				to caller.
 */
static int accept_finish ( cport_isb isb )
{
    int status;
    struct private_st *ctx, *new;
    struct cportx11_accept_result *result;

    ctx = isb->drv;
    pthread_mutex_lock ( &ctx->lock );
    if ( isb->length < sizeof(result->io_ctx) ) {
	isb->default_iosb[0] = SS$_BADPARAM;
    } else if ( ctx->connects_pending > 0 ) {

	isb->default_iosb[0] = 1;
	isb->default_iosb[1] = 4;
	new = allocate_context();
	if ( new ) {
	    ctx->connects_pending--;
	    new->server_number = ctx->server_number;
	    new->channel = ctx->pending_chan[ctx->connects_pending];
	    new->ref_count = 0;
	    new->type = 1;
	    /*
	     * Save connection data in callers buffer.
	     */
	    result = (struct cportx11_accept_result *) isb->buffer;
	    result->io_ctx = (void *) new;
	    tu_strnzcpy ( result->remote_address,
		ctx->pending_host[ctx->connects_pending],
		sizeof(result->remote_address)-1 );
	    isb->default_iosb[0] = 1;
	    isb->default_iosb[1] = sizeof(result);
	    isb->default_iosb[3] = ctx->server_number;
	} else {
	    isb->default_iosb[0] = 44;
	}
    } else {	/* spurious call */
	isb->default_iosb[0] = 440;
    }
    pthread_mutex_unlock ( &ctx->lock );
    return CPORT_COMPLETE_PROCESSED;		/* deliver to caller */
}
/******************************************************************************/
/* Define the start_io functions that cport_start_io calls via the ftable
 * array in the stream_handler.
 *
 * A user call to cport_start_io ( isb, func, arg1, arg2 ) results in
 * a call to (*isb->driver.ftable[func&FMASK])( isb, func, arg1, arg2 ).
 *
 * All start_io functions use the following prototype:
 *
 *    int start ( cport_isb isb, int func, void *arg1, void *arg2 );
 *
 *    arguments:
 *        isb		I/O synchronization block.  cport_next_completion
 *			will return this address when I/O completes.
 *
 *	  func		Function value describing operation to perform,
 *			in most cases the function is implied by which
 *			start_io function got called.  Bits outside the
 *			range of FMASK may specify qualifiers.
 *
 *        arg1		Driver and function specific argument.  Being
 *			a pointer, this usually points to the user buffer.
 *
 *        arg2		Driver and function specific argument.  Being an
 *			int type, this argument usually specifies the
 *			buffer length or function qualifier.
 *
 *    return value:
 *        odd		Success.
 *	  even.		Failure.
 *
 * 
 */
static int start_accept ( cport_isb isb, int func, void *arg1, int arg2 )
{
    int status;
    struct private_st *ctx, *new;
    /*
     * Check pending queue.
     */
    ctx = isb->drv;
    isb->length = arg2;
    isb->buffer = (char *) arg1;		/* save caller's arguments */
    isb->completion_callback = accept_finish;
    isb->default_iosb[0] = isb->default_iosb[1] = 0;
    isb->iosb = (void *) isb->default_iosb;
    pthread_mutex_lock ( &ctx->lock );
    if ( ctx->connects_pending > 0 ) {
	isb->default_iosb[0] = 1;
	status = 1;
    } else {
	/*
	 * No pending, prepare block so monitor thread will unblock us.
	 */
	ctx->pending_accept = isb;
	isb->default_iosb[0] = isb->default_iosb[1] = 0;
	status = 1;
    }
    cport_mark_busy ( isb );
    pthread_mutex_unlock ( &ctx->lock );
    return status;
}
static int start_write ( cport_isb isb, int func, void *arg1, int arg2 )
{
    /*
     * For simple requests, use cport_qio2 function, which takes care of
     * issuing qio and syncrhonizing completion.  Arg1 and arg2 become
     * P1/P2 of QIO call.
     */
    return cport_qio2 ( isb, IO$_WRITEVBLK, arg1, arg2 );
}

static int finish_read ( cport_isb isb )
{
    /*
     * Check status.
     */
    struct ucx_ctx *ctx;
    int status;

    ctx = (struct ucx_ctx *) isb->drv;
    status = isb->default_iosb[0];
    status = (status & 0xffff);
    if ( status == SS$_BUFFEROVF ) {
	isb->default_iosb[0] = SS$_NORMAL;
    }
    isb->completion_callback = 0;
    return CPORT_COMPLETE_PROCESSED;		/* I/O completed */
}

static int start_read ( cport_isb isb, int func, void *arg1, int arg2 )
{
    /*
     * Setup completion callback in order to check for SS$_BUFFEROVF status.
     */
    isb->completion_callback = finish_read;

    return cport_qio2 ( isb, IO$_READVBLK|IO$M_MULTIPLE, arg1, arg2 );
}

/******************************************************************************/
/* Define global stream handler structure to be passed to cport_assign_stream()
 * by clients wishing to use this driver.  Note that the isb contains a
 * copy of the handler struct, not a pointer to a single instance.  Note
 * also that the copies have a common pointer to the start_io ftable.
 *
 * The convention for naming the stream handler is cportXXXXX_driver, where
 * XXXXX is a unqiue driver name (e.g. cportsshmsg_driver is driver for ssh
 * message layer).
 * 
 */
#define FMASK 3			/* Must be 2^n -1 */
static cport_start_function ftable[FMASK+1] = {
    start_write,		/* CPORT_WRITE function */
    start_read,			/* CPORT_READ function */
    start_accept,		/* CPORT_FCODES+0, listen for connection */
    start_accept		/* CPORT_FCODES+1, sync. function */
};

cport_stream_handler cportx11_driver = {
    FMASK,			/* Size of ftable - 1 */
    ftable,			/* start_io function table */
    new_stream,			/* Assign stream to I/O context */
    destroy_stream,		/* Deassign stream */
    cancel_io			/* Stop I/O in progress. */
};
