/*
 * This module handles mailbox and sub-process creation for ancillary
 * processes spawned by the SSH server.  The mailbox is created with
 * protection (sy:rwed,ow:rwed,gr,wo).  The spawned process inherits the
 * current process's sys$input and sys$output.
 *
 * Prior to creating the sub-process, a lock is obtained with the name
 * 'mailbox_name'_'parent_pid' that the created process can use to detect
 * image exit.
 *
 * The spawn_helper function is NOT thread safe, it should only be called
 * by the main thread.
 *
 * Author:	David Jones
 * Date:	11-JUL-1998
 *  Revised:	29-OCT-1998		Add helper_restart function.
 * Revised:	4-MAY-1999		Use initial thread PID in kernel threads.
 */
#include <descrip.h>			/* VMS string descriptors */
#include <jpidef.h>			/* VMS $GETJPI codes */
#include <lckdef.h>			/* VMS lock manager symbols */
#include <iodef.h>			/* VMS QIO function codes */
#include <string.h>
#include <stdio.h>

#include "pthread_1c_np.h"

int LIB$GET_SYMBOL(), LIB$GETJPI(), LIB$SPAWN(), SYS$CREMBX(), SYS$DASSGN();
int SYS$ENQW(), SYS$DEQ(), SYS$QIO(), SYS$SETPRV(), LIB$PUT_OUTPUT();

#include "helper.h"			/* verify prototypes */
struct mbx_iosb {
   unsigned short status, length;
   long pid;
};
/***************************************************************************/
int helper_create ( 
	char *mailbox_name,		/* Mailbox name for CREMBX */
	int mailbox_size,		/* max message size/ msg quota */
	char *command,			/* DCL command to spawn */
	helper_ctx ctx )		/* receives result */
{
    int flags, ctl_chan, status, transferred;
    static char lock_name_buf[64];
    static $DESCRIPTOR(lock_name,lock_name_buf);
    static $DESCRIPTOR(mbx_name,"");
    static $DESCRIPTOR(command_dx, "");
    long code, pid;
    /*
     * Create mailbox for communication with created process.
     */
    ctx->channel = 0;
    mbx_name.dsc$w_length = strlen(mailbox_name);
    mbx_name.dsc$a_pointer = mailbox_name;

    status = SYS$CREMBX(0, &ctx->channel, 
	mailbox_size, mailbox_size, 0x0ff00, 0, &mbx_name, 0);
    if ( (status&1) ) {
	/*
	 * Create deadman lock that spawned process can use to detect
	 * this image exiting.
	 */
	code = JPI$_PID;
	status = LIB$GETJPI ( &code, 0,  0, &pid, 0, 0 );
	if ( (status&1) == 0 ) 
		fprintf(stderr,"Error in GETJPI call: %d\n", status);
#ifdef JPI$_INITIAL_THREAD_PID
	else {
	    /*
	     * Replace the pid with the initial thread's pid.
	     */
	    long initial_pid = 0;
	    code = JPI$_INITIAL_THREAD_PID;
	    status = LIB$GETJPI ( &code, &pid, 0, &initial_pid, 0, 0 );
	    if ( (status&1) && initial_pid ) pid = initial_pid;
	}
#endif
	sprintf ( lock_name_buf, "%s_%x", mbx_name.dsc$a_pointer, pid );
	lock_name.dsc$w_length = strlen(lock_name_buf);
	status = SYS$ENQW ( 8, LCK$K_EXMODE, ctx->deadman, 
		LCK$M_NODLCKBLK, &lock_name,
		0, 0, 0, 0, 0, 0, 0 );
	if ( (status&1) == 1 ) status = ctx->deadman[0];
	if ( (status&1) == 0 ) {
	    SYS$DASSGN ( ctx->channel );
	    return status;
	}
	/*
	 * Create sub-process. Flags: nowait, inherit authorized privs only.
	 */
	command_dx.dsc$w_length = strlen(command);
	command_dx.dsc$a_pointer = command;
	flags = 1 | 128;
	status = LIB$SPAWN ( &command_dx, 0, 0, &flags, 0, 
		&ctx->pid, 0, 0, 0, 0, 0, 0, 0 );

	if ( (status&1) == 0 ) {
	    SYS$DASSGN ( ctx->channel );
	    SYS$DEQ ( ctx->deadman[1], 0, 0, 0 );
	    return status;
	}
    }
    if ( (status&1) ) {
	/*
	 * Initialize pthreads syncrhonization objects.
	 */
	INITIALIZE_MUTEX ( &ctx->lock );
	INITIALIZE_MUTEX ( &ctx->io );
	INITIALIZE_CONDITION ( &ctx->io_done );
    }
    return status;
}
/****************************************************************************/
/* Lock and unlock simply invoke pthreads primitives.
 */
int helper_lock ( helper_ctx ctx ) {
    int status;
    status = pthread_mutex_lock ( &ctx->lock );
    if ( status == 0 ) return 1;
    return 44;
}
int helper_unlock ( helper_ctx ctx ) {
    int status;
    status = pthread_mutex_unlock ( &ctx->lock );
    if ( status == 0 ) return 1;
    return 44;
}
/*****************************************************************************/
/* I/O functions use AST with pthread_cond_signal_np to wait for I/O completion
 * using DECthreads scheduler.
 */
int helper_write ( helper_ctx helper, void *buffer, int bufsize )
{
    int status;
    struct mbx_iosb iosb;

    pthread_mutex_lock ( &helper->io );
    status = SYS$QIO ( 0, helper->channel, IO$_WRITEVBLK, &iosb,
	pthread_cond_signal_int_np, &helper->io_done,
	buffer, bufsize, 0, 0, 0, 0 );
    if ( (status&1) == 1 ) {
	while ( iosb.status == 0 ) {
	    pthread_cond_wait ( &helper->io_done, &helper->io );
	}
    }
    pthread_mutex_unlock ( &helper->io );
    status = iosb.status;
    if ( iosb.pid != helper->pid ) {	/* mismatch */
	status = 0;
    }
    return status;
}
int helper_read ( helper_ctx helper, void *buffer, int bufsize, int *recvd )
{
    int status;
    struct mbx_iosb iosb;

    pthread_mutex_lock ( &helper->io );
    status = SYS$QIO ( 0, helper->channel, IO$_READVBLK, &iosb,
	pthread_cond_signal_int_np, &helper->io_done,
	buffer, bufsize, 0, 0, 0, 0 );
    if ( (status&1) == 1 ) {
	while ( iosb.status == 0 ) {
	    pthread_cond_wait ( &helper->io_done, &helper->io );
	}
    }
    pthread_mutex_unlock ( &helper->io );
    status = iosb.status;
    if ( iosb.pid != helper->pid ) {	/* mismatch */
	status = 0;
    }
    *recvd = iosb.length;
    return status;
}
/****************************************************************************/
/* Helper_restart() retrieves the DCL symbols used by ssh-server_startup and
 * spawns a command the re-invokes it (creating a new detached process).
 */
int helper_restart()
{
    int status, length, table, flags, code, p1_len;
    long cur_priv[2], proc_priv[2], delta_priv[2];
    static char command_line[256], startup_proc[200], p1[200];
    static $DESCRIPTOR(cmd_line_dx,command_line);
    static $DESCRIPTOR(startup_proc_dx,startup_proc);
    static $DESCRIPTOR(sym_name,"SSH_STARTUP_PROC");
    static $DESCRIPTOR(p1_name, "P1");
    static $DESCRIPTOR(p1_dx, p1);
    /*
     * Construct command to spawn: @'ssh_startup_proc' 'p1' 'p2' 'p3'
     */
    length = table = 0;
    status = LIB$GET_SYMBOL ( &sym_name, &startup_proc_dx, &length, &table );
    if ( (status&1) == 1 && table == 1 && length <= 200 ) {
	command_line[0] = '@';
	strncpy ( &command_line[1], startup_proc, length );
	p1_len = 0;
	status = LIB$GET_SYMBOL ( &p1_name, &p1_dx, &p1_len );
	if ( (status&1) && p1_len > 0 ) {
	    command_line[1+length] = ' '; length++;
	    strncpy ( &command_line[1+length], p1, p1_len );
	}
	cmd_line_dx.dsc$w_length = length + 1 + p1_len;
    } else {
	    return status;
    }
    /*
     * Set process privs to the process 'permanent' state and spawn command.
     */
    code = JPI$_PROCPRIV;
    status = LIB$GETJPI ( &code, 0, 0, &proc_priv, 0, 0 );
    code = JPI$_CURPRIV;
    if (status&1) status = LIB$GETJPI ( &code, 0, 0, &cur_priv, 0, 0 );
    if ( status&1 ) {
	status = SYS$SETPRV ( 1, proc_priv, 0, cur_priv );
	delta_priv[0] = cur_priv[0] & (~proc_priv[0]);
	delta_priv[1] = cur_priv[1] & (~proc_priv[1]);
	status = SYS$SETPRV ( 0, delta_priv, 0, 0 );
    } else return status;

    flags = 0;
    LIB$PUT_OUTPUT ( &cmd_line_dx );
    status = LIB$SPAWN ( &cmd_line_dx, 0, 0, &flags );
    printf("status of spawn: %d\n", status );
    SYS$SETPRV ( 1, delta_priv, 0, 0 );
    return status;
}
