/*
 * Common functions used by 'helper' programs for the SSH server.  Helper
 * programs communicate with the server via a mailbox and use the lock
 * manager to signal program termination (deadman locks).
 *
 * This module maintains communication state in static variables and is
 * therefore not threadsafe.
 *
 * Author:	David Jones
 * Date:	23-MAY-1999
 * Revised:	26-MAY-1999		missing return statement in setup func.
 */
#include <stdio.h>
#include <stdlib.h>
#include <descrip.h>			/* VMS string descriptors */
#include <prcdef.h>			/* Process create flags */
#include <jpidef.h>
#include <lckdef.h>
#include <impdef.h>			/* VMS persona (impersonate) service */
#include <ossdef.h>			/* Object Set Security service */
#include <iodef.h>
#include <ssdef.h>
#include <string.h>

int SYS$ASSIGN(), SYS$QIOW(), LIB$GETJPI(), SYS$ENQ(), SYS$EXIT(), SYS$SETPRN();
int SYS$CREPRC(), SYS$SET_SECURITY(), SYS$DASSGN(), SYS$SETPRI();
/*
 * Define static information.
 */
static long parent_pid, initial_pid;
static long lksb[5];
static short mbx_chan;
struct mbx_iosb {
   unsigned short status, length;
   long pid;
};
static $DESCRIPTOR(null_desc,"");

#include "helper_com.h"		/* verify prototypes */
/*
 * Set routine initializes the mailbox and deadman locks.
 */
int helper_setup ( char *process_name, char *mbx_name,
	void (*deadman_ast)(), void *ast_arg )
{
    struct dsc$descriptor_s mbx_name_dx, lock_name, process_name_dx;
    int status, code;
    char lock_name_buf[128];
    /*
     * Get PID of owner process, any traffic to mailbox must be to/from this
     * process.  Get our own PID and queue lock  If we ever get lock, then
     * parent image has exitted.
     */
    status = SYS$SETPRI ( 0, 0, 4, 0, 0, 0 );
    code = JPI$_OWNER;
    parent_pid = initial_pid = 0;
    status = LIB$GETJPI ( &code, 0, 0, &parent_pid, 0 );
    if ( parent_pid != 0 ) {
#ifdef JPI$_INITIAL_THREAD_PID
	/*
	 * For kernel threads, replace PID with initial thread PID.
	 */
	code = JPI$_INITIAL_THREAD_PID;
	status = LIB$GETJPI ( &code, &parent_pid, 0, &initial_pid, 0 );
	if ( (status&1) && initial_pid ) parent_pid = initial_pid;
#endif
       /*
	* Change process name.
	*/
	if ( process_name ) {
	    process_name_dx = null_desc;
	    process_name_dx.dsc$a_pointer = process_name;
	    process_name_dx.dsc$w_length = strlen ( process_name );
	    status = SYS$SETPRN ( &process_name );
	}
	/*
	 * Get deadman lock.
	 */
	sprintf ( lock_name_buf, "%s_%x",mbx_name, parent_pid );
	lock_name = null_desc;
	lock_name.dsc$a_pointer = lock_name_buf;
        lock_name.dsc$w_length = strlen ( lock_name.dsc$a_pointer );
	status = SYS$ENQ ( 9, LCK$K_EXMODE, lksb, LCK$M_NODLCKWT, &lock_name,
		0, deadman_ast, ast_arg, 0, 0, 0, 0, 0 );
    }
    /*
     * Assign channel to mailbox.
     */
    mbx_name_dx = null_desc;
    mbx_name_dx.dsc$a_pointer = mbx_name;
    mbx_name_dx.dsc$w_length = strlen ( mbx_name );
    status = SYS$ASSIGN ( &mbx_name_dx, &mbx_chan, 0, 0, 0 );
    if ( (status&1) == 0 ) return status;
    return status;
}
/*****************************************************************************/
/* Send and recv message are low-level QIO's.  The process ID of the sender
 * is verified.
 */
int helper_recv_message ( void *buffer, int bufsize, int *length )
{
    int status;
    struct mbx_iosb iosb;

    status = SYS$QIOW ( 8, mbx_chan, IO$_READVBLK, &iosb, 0, 0,
	buffer, bufsize, 0, 0, 0, 0 );
    if ( (status&1) == 1 ) status = iosb.status;
    if ( ((status&1) == 1) || (status == SS$_ENDOFFILE) ) {
	/*
	 * Message read, verify sender is the process that created us.
	 */
	*length = iosb.length;
	if ( (iosb.pid != parent_pid) && (parent_pid != 0) ) {
	    if ( initial_pid != 0 ) {
		/*
		 * Replace iosb.pid with initial thread pid.
		 */
		int code;
#ifdef JPI$_INITIAL_THREAD_PID
		long sender_initial_pid = 0;
		code = JPI$_INITIAL_THREAD_PID;
		status = LIB$GETJPI ( &code, &iosb.pid, 0, 
			&sender_initial_pid, 0 );
		if ( (status&1) && sender_initial_pid ) 
			iosb.pid = sender_initial_pid;
#endif
	    }
	    if ( iosb.pid != parent_pid ) {
	        fprintf(stderr,"Initiator request written by wrong process: %x\n",iosb.pid);
	        status = SS$_ABORT;
	    }
	}
    } else *length = 0;
    return status;
}

int helper_send_message ( void *buffer, int bufsize )
{
    int status;
    struct mbx_iosb iosb;

    status = SYS$QIOW ( 8, mbx_chan, IO$_WRITEVBLK, &iosb, 0, 0,
	buffer, bufsize, 0, 0, 0, 0 );
    if ( (status&1) == 1 ) status = iosb.status;
    return status;
}
/*****************************************************************************/
/*
 * Cleanup.
 */
int helper_teardown()
{
    return 1;
}
