/*
 * This program is run by a process spawned by the SSH server, it handles
 * creating detached interactive processes by arbitrary user names.
 * Since it bypasses loginout authentication, it must enforce SYSUAF policy
 * checks.
 *
 * The process communicates with the parent via a mailbox with the name
 * SSH_INITIATOR_MAILBOX.  When the server wants to login a user, it
 * sends a mailbox message that includes the username and standard input
 * and output devices.
 *
 * Author:	David Jones
 * Date:	16-JUL-1998
 * Revised:	21-OCT-1998		Inhibit NOPASSWORD flag if is_shell==2.
 * Revised:	Get initial thread.
 */
#include <stdio.h>
#include <stdlib.h>
#include <descrip.h>			/* VMS string descriptors */
#include <prcdef.h>			/* Process create flags */
#include <jpidef.h>
#include <lckdef.h>
#include <impdef.h>			/* VMS persona (impersonate) service */
#include <ossdef.h>			/* Object Set Security service */
#include <iodef.h>
#include <ssdef.h>
#include <string.h>

struct mbx_iosb {
   unsigned short status, length;
   long pid;
};
#include "initiator.h"
#include "tutil.h"

static short mbx_chan;
int SYS$ASSIGN(), SYS$QIOW(), LIB$GETJPI(), SYS$ENQ(), SYS$EXIT(), SYS$SETPRN();
int SYS$CREPRC(), SYS$SET_SECURITY(), SYS$DASSGN(), SYS$SETPRI();

static $DESCRIPTOR(engine_name,"SSH Initiator");
static $DESCRIPTOR(ctlmbx_name, SSH_INITIATOR_MAILBOX_NAME );
static $DESCRIPTOR(classname, "DEVICE");

#ifdef DEBUG
static void dump_buffer ( char *heading, char *buffer, int bufsize )
{
    int i, j;
    printf ( "%s (%d bytes):", heading, bufsize );
    j = 15;
    for ( i = 0; i < bufsize; i++ ) {
	j++;
	if ( j > 15 ) {
	    printf("\n   %04x:", i ); j = 0; }
	printf(" %02x", 255&buffer[i] );
    }
    printf("\n");
}
#else
#define dump_buffer(a,b,c) 1
#endif
/******************************************************************************/
/* Define routines for assuming new persona.
 */
#define PCACHE_SIZE 20
static struct persona_block {
    char username[36];
    long persona;			/* VMS context pointer */
    unsigned int last_access;		/* for LRU replacement */
} pcache[PCACHE_SIZE];
int pcache_waiters;
unsigned int pcache_sequence;
long original_persona, persona_flags;
int SYS$PERSONA_CREATE(), SYS$PERSONA_ASSUME(), SYS$PERSONA_DELETE();

static int become_persona ( char *username, int ulen )
{
    char compare_string[36];
    int oldest_entry, status, i, match;
    /*
     * Do caseless compares.
     */
    if ( ulen >= sizeof(compare_string) ) ulen = sizeof(compare_string)-1;
    tu_strnzcpy ( compare_string, username, ulen );
    tu_strupcase ( compare_string, compare_string );
    /*
     * Search cache for existing match, remember oldest entry.
     */
    for ( match = -1, i = oldest_entry = 0; i < PCACHE_SIZE; i++ ) {
	if ( pcache[i].last_access == 0 ) {
	    oldest_entry = i; break;
	}
	else if ( tu_strncmp (compare_string, pcache[i].username, 39) == 0 ) {
	    match = i; break;
	} else if ( pcache[i].last_access < pcache[oldest_entry].last_access) {
	    oldest_entry = i;
	}
    }
    if ( match == -1 ) {
	/*
	 * No match, select victim.
	 */
	struct dsc$descriptor username_dx;
	match = oldest_entry;
	if ( pcache[match].last_access != 0 ) {
	    /*
	     * delete previous use of entry.
	     */
	    status = SYS$PERSONA_DELETE ( &pcache[i].persona );
	}
	/*
	 * Create new persona.
	 */
	tu_strnzcpy ( pcache[match].username, compare_string, 
		sizeof(pcache[match].username)-1 );
        username_dx.dsc$a_pointer = compare_string;
        username_dx.dsc$w_length = tu_strlen ( pcache[match].username );
	status = SYS$PERSONA_CREATE (&pcache[match].persona, &username_dx, 0);
	if ( (status&1) == 0 ) return status;
    }
    /*
     * Update sequence.
     */
    status = SYS$PERSONA_ASSUME ( &pcache[match].persona, persona_flags );
    pcache_sequence++;
    if ( pcache_sequence == 0 ) pcache_sequence = 1;
    pcache[i].last_access = pcache_sequence;

    return status;
}

static int revert_persona ( )
{
    int status;
    /*
     * Change persona back to original.
     */
    status = SYS$PERSONA_ASSUME ( &original_persona, persona_flags );

    return status;
}
/***********************************************************************/

int main ( int argc, char **argv )
{
    int status, i, flags;
    long initial_pid, pid, code, lksb[2];
    char errmsg[256];
    struct mbx_iosb iosb;
    static char lock_name_buf[64];
    static $DESCRIPTOR(lock_name,lock_name_buf);
    static $DESCRIPTOR(input,"");
    static $DESCRIPTOR(output,"");
    static $DESCRIPTOR(loginout, "sys$system:loginout.exe");
    struct initiator_request request;
    struct initiator_reply reply;
    struct { short length, code; void *buffer; int *retlen;} item[2];
    /*
     * Get PID of owner process, any traffic to mailbox must be to/from this
     * process.  Get our own PID and queue lock  If we ever get lock, then
     * parent image has exitted.
     */
    status = SYS$SETPRI ( 0, 0, 4, 0, 0, 0 );
    code = JPI$_OWNER;
    reply.pid = 0;
    pid = initial_pid = 0;
    status = LIB$GETJPI ( &code, &reply.pid, 0, &pid, 0 );
    if ( pid != 0 ) {
#ifdef JPI$_INITIAL_THREAD_PID
	/*
	 * For kernel threads, replace PID with initial thread PID.
	 */
	code = JPI$_INITIAL_THREAD_PID;
	status = LIB$GETJPI ( &code, &pid, 0, &initial_pid, 0 );
	if ( (status&1) && initial_pid ) pid = initial_pid;
#endif
	/*
	 * Get deadman lock.
	 */
	status = SYS$SETPRN ( &engine_name );
	sprintf ( lock_name_buf, "%s_%x",ctlmbx_name.dsc$a_pointer, pid );
        lock_name.dsc$w_length = strlen ( lock_name.dsc$a_pointer );
	status = SYS$ENQ ( 9, LCK$K_EXMODE, lksb, LCK$M_NODLCKWT, &lock_name,
		0, SYS$EXIT, SS$_ABORT,  0, 0, 0, 0, 0 );
    }
    /*
     * Assign channel to mailbox.
     */
    status = SYS$ASSIGN ( &ctlmbx_name, &mbx_chan, 0, 0, 0 );
    if ( (status&1) == 0 ) exit ( status );
    /*
     * Initialize persona information.
     */
    original_persona = 1;			/* thats what the book says */
    persona_flags = IMP$M_ASSUME_SECURITY;	/* changes username */
    persona_flags |= IMP$M_ASSUME_ACCOUNT;	/* Override multi-job checks */
    persona_flags |= IMP$M_ASSUME_JOB_WIDE;	/* Override multi-job checks */
    pcache_waiters = pcache_sequence = 0;
    for ( i = 0; i < PCACHE_SIZE; i++ ) {
	pcache[i].username[0] = '\0';
	pcache[i].persona = 0;
	pcache[i].last_access = 0;
    }
    /*
     * Main loop, process requests from mailbox.
     */
    while ( status & 1 ) {
	/*
	 * Read message and verify sender.
	 */
	status = SYS$QIOW ( 8, mbx_chan, IO$_READVBLK, &iosb, 0, 0,
		&request, sizeof(request), 0, 0, 0, 0 );
        if ( (status&1) == 1 ) status = iosb.status;
	if ( status == SS$_ENDOFFILE ) continue;	/* ignore EOF */
        if ( (status&1) == 0 ) break;

	if ( (iosb.pid != pid) && (pid != 0) ) {
	    if ( initial_pid != 0 ) {
		/*
		 * Replace iosb.pid with initial thread pid.
		 */
#ifdef JPI$_INITIAL_THREAD_PID
		long sender_initial_pid = 0;
		code = JPI$_INITIAL_THREAD_PID;
		status = LIB$GETJPI ( &code, &pid, 0, &sender_initial_pid, 0 );
		if ( (status&1) && sender_initial_pid ) 
			iosb.pid = sender_initial_pid;
#endif
	    }
	    if ( iosb.pid != pid ) {
	        fprintf(stderr,"Initiator request written by wrong process: %x\n",iosb.pid);
	        continue;
	    }
	}
        /*
	 * Do sanity checks on message and fill in string descriptors for
	 * input and output devices.
	 */
	if ( request.ulen < 0 || request.out_len < 0 || request.inp_len < 0 ) {
	    reply.status = SS$_BADPARAM;
	} else if ( request.ulen + request.out_len + request.inp_len >
		sizeof(request.str) ) {
	    reply.status = SS$_BADPARAM;
	} else {
	    input.dsc$w_length = request.inp_len;
	    input.dsc$a_pointer = &request.str[request.ulen];
	    output.dsc$w_length = request.out_len;
	    output.dsc$a_pointer = &request.str[request.ulen+request.inp_len];
	    if ( request.out_len == 0 ) output = input;
	    reply.status = 1;
	}
	/*
	 * Change ownership of input/output devices so target user can
	 * read them.
	 */
	if ( request.is_shell && reply.status & 1 ) {
	    /*
	     * Verify that the input device is a terminal.
	     */
	    int inp_chan, chan_valid;
	    inp_chan = 0;
	    chan_valid =0;
	    status = SYS$ASSIGN ( &input, &inp_chan, 0, 0, 0 );
	    if ( status&1 ) {
		/* Make getdvi call */
		chan_valid = 1;
		item[0].length = 8;
	    }
	    if ( status&1 ) {
	    item[0].length = sizeof(request.uic);
	    item[0].code = OSS$_OWNER;
	    item[0].buffer = &request.uic;
	    item[0].retlen = (int *) 0;
	    item[1].code = item[1].length = 0;
	    status = SYS$SET_SECURITY ( &classname, &input, 0, 0, item, 0, 0 );
	    if ( (input.dsc$w_length != output.dsc$w_length) ||
	        strncmp ( input.dsc$a_pointer, output.dsc$a_pointer,
		    input.dsc$w_length ) != 0 ) {
	        status = SYS$SET_SECURITY (&classname, &output, 
			0, 0, item, 0, 0);
	    }
	    }
	    if ( chan_valid ) SYS$DASSGN ( inp_chan );
	    reply.status = status;
	}
	/*
	 * Assume persona of requested username.
	 */
	if ( reply.status&1 ) {
	     reply.status = become_persona ( request.str, request.ulen );
	}
	/*
	 * Create process, set flags to skip username/password prompt.
	 */
	if ( reply.status & 1 ) {
	    flags = /* PRC$M_DETACH | */ PRC$M_HIBER;
	    if ( request.is_shell != 2 ) flags |= PRC$M_NOPASSWORD;
	    if ( request.is_shell ) flags |= PRC$M_INTER;
            reply.status = SYS$CREPRC ( &reply.pid, &loginout, &input, &output, 
		&output, 0, 0, 0, 4, request.uic, 0, flags, 0, 0 );
	}
	/*
	 * Revert to original username and send reply.
	 */
	revert_persona();
	reply.pindex = 0;
	code = JPI$_PROC_INDEX;
        LIB$GETJPI ( &code, &reply.pid, 0, &reply.pindex, 0 );
	status = SYS$QIOW ( 0, mbx_chan, IO$_WRITEVBLK, &iosb, 0, 0,
		&reply, sizeof(reply), 0, 0, 0, 0 );
        if ( (status&1) == 1 ) status = iosb.status;
    }
    return status;
}
