/*
 * This modules handles making remote requests to the initiator sub-process.
 * It also handles creation of the special logical names used to make
 * current connection information globally available.  The logical names
 * are stored in 2 special logical name tables set by parameter file 
 * variables login_table (default SSH_LOGIN_MAP) and login_info_table
 * (SSH_LOGIN_INFO):
 *
 *   param.login_table		Holds remote connection information - host
 *				port and encryption type.
 *
 *   param.login_info_table	Holds sysuaf data needed to emulate a
 *				login: newmail count, last login, etc.
 *
 * The logical names in the table are of the form 'nn' where nn is the
 * the process index number in decimal and are multi-valued names.  The
 * first name index (0) is the PID, and the second (1) is the input device.  The
 * remaining equivalence strings depend upon the table:
 *
 *   Login_table:
 *	2	Remote connection info (terminal accpornam).
 *
 *   login_into_table:
 *	2	Flags from UAF file, in decimal.
 *	3	newmail count from UAF file.
 *      4	X11 info, of form 'transport:server-num' (e.g. TCPIP:1)
 *
 * Author:	David Jone
 * Date:	25-JUL-1998
 * Revised:	15-SEP-1998		Add x11 info.
 */
#include <descrip.h>
#include <lnmdef.h>
#include <ssdef.h>

#include <stdio.h>
#include "pthread_1c_np.h"
#include "tutil.h"
#include "parameters.h"
#include "user_info.h"
#include "initiator.h"
#include "initiator_client.h"
#include "helper.h"

static struct helper_context initiator_ctx;
int SYS$WAKE(), SYS$CRELNM();
static $DESCRIPTOR(login_table,"");
static $DESCRIPTOR(login_info_table,"");
/*****************************************************************************/
/* Initialize module, set global logical names from parameter file info.
 */
int init_initiator ( char *command, char errmsg[256] )
{
    int status, msgsize, length;
    struct initiator_reply reply;

    login_table.dsc$w_length = tu_strlen ( param.login_table );
    login_table.dsc$a_pointer = param.login_table;
    login_info_table.dsc$w_length = tu_strlen ( param.login_info_table );
    login_info_table.dsc$a_pointer = param.login_info_table;
    msgsize = sizeof(struct initiator_request);
    if ( sizeof(struct initiator_reply) > msgsize ) 
	msgsize = sizeof(struct initiator_reply);
    status = helper_create ( SSH_INITIATOR_MAILBOX_NAME, msgsize, command,
	&initiator_ctx );
    errmsg[0] = '\0';
    return status;
}
static char hex[16] = { '0', '1', '2', '3', '4', '5', '6', '7',
	'8', '9', 'A', 'B', 'C', 'D', 'E', 'F' };
/*****************************************************************************/
/* Log process creation in logical name tables.  Process is assumed to be
 * hibernating.
 */
static int prepare_process ( struct user_account_info *uai, char *terminal,
	char *accpornam, long pid, int pindex, char *x11_info )
{
    struct dsc$descriptor_s logical_name;
    struct { short int length, code; char *buffer; int *retlen; } item[6];
    char lognam[16], pidstr[8];
    int i, status, nibble;

    if ( *param.login_table ) {
	/*
	 * Create logical name to store remote connection information.
	 * The logical name is the process index.
	 */
	logical_name = login_table;
	logical_name.dsc$a_pointer = lognam;
	tu_strint ( pindex, lognam );
	logical_name.dsc$w_length = tu_strlen ( lognam );
	for ( i = 0; i < 8; i++ ) {
	    pidstr[7-i] = hex[pid&15];
	    pid = pid >> 4;
	}
	item[0].length = 8;
	item[0].code = LNM$_STRING;
	item[0].buffer = pidstr;
	item[0].retlen = 0;

	item[1].length = tu_strlen ( terminal );
	item[1].code = LNM$_STRING;
	item[1].buffer = terminal;
	item[1].retlen = 0;

	item[2].length = item[2].code = 0;
	if ( *accpornam ) {
	    item[2].length = tu_strlen ( accpornam );
	    item[2].code = LNM$_STRING;
	    item[2].buffer = accpornam;
	    item[2].retlen = 0;
	    item[3].length = item[3].code = 0;
	}

	status = SYS$CRELNM ( 0, &login_table, &logical_name, 0, item );
    } else status = SS$_IVLOGTAB;

    if ( (status&1) && param.login_info_table ) {
	/*
	 * Create the second logical name, this should be a protected table.
	 */
	char flags_str[16], newmail_count[16];
	tu_strint ( uai->flags, flags_str );
	tu_strint ( uai->newmail, newmail_count );
	item[2].length = tu_strlen (flags_str );
	item[2].code = LNM$_STRING;
	item[2].buffer = flags_str;
	item[2].retlen = 0;
	item[3].length = tu_strlen ( newmail_count );
	item[3].code = LNM$_STRING;
	item[3].buffer = newmail_count;
	item[3].retlen = 0;
	if ( x11_info ) {
	    /* If x11_info is zero length, don't add to logical name */
	    item[4].length = tu_strlen ( x11_info );
	    item[4].code = (item[4].length>0) ? LNM$_STRING : 0;
	    item[4].buffer = x11_info;
	    item[4].retlen = 0;
	    item[5].length = item[5].code = 0;
	} else item[4].length = item[4].code = 0;

	status = SYS$CRELNM ( 0, &login_info_table, &logical_name, 0, item );
    };
    return status;
}
/*****************************************************************************/
/* Send request to initiator.
 */
int initiate_user_process ( struct user_account_info *uai, int is_shell, 
	char *input_device, char *output_device, char *accpornam,
	char *x11_info, char errmsg[256] )
{
    int status, length;
    struct initiator_request request;
    struct initiator_reply reply;
    /*
     * Build message to send.
     */
    request.uic = uai->uic;
    request.ulen = tu_strlen ( uai->username );
    request.inp_len = tu_strlen ( input_device );
    request.out_len = tu_strlen ( output_device );
    request.is_shell = is_shell;
    if ( request.ulen + request.inp_len + request.out_len >
	sizeof(request.str ) ) {
	tu_strcpy ( errmsg, "Input parameters too large" );
	return 20;
    }
    tu_strncpy ( request.str, uai->username, request.ulen );
    tu_strncpy ( &request.str[request.ulen], input_device, request.inp_len );
    tu_strncpy ( &request.str[request.ulen+request.inp_len],
	output_device, request.out_len );
    /*
     * Acquire exclusive access to mailbox channel, send message, and read
     * response.
     */
    helper_lock ( &initiator_ctx );
    status = helper_write ( &initiator_ctx, &request,
	sizeof(request) - sizeof(request.str) +
	request.ulen + request.inp_len + request.out_len );

    if ( status&1 ) {
	status = helper_read ( &initiator_ctx, &reply, 
		sizeof(reply), &length );
	if ( status&1 ) {
	    status = reply.status;
	    if ( status&1 ) {
		/*
		 * Process creation successful, process is created in
		 * hibernate state to give us a chance to do additional
		 * setup before it runs.
		 */
		errmsg[0] = '\0';
		prepare_process ( uai, input_device, 
			accpornam, reply.pid, reply.pindex, x11_info );
		status = SYS$WAKE ( &reply.pid, 0 );
	    } else  tu_strcpy ( errmsg, "Error returned by initiator" );
	}
	else tu_strcpy ( errmsg, "Error reading initiator reply" );
    } else tu_strcpy ( errmsg, "Error sending request to initiator" );
    helper_unlock ( &initiator_ctx );
    /*
     * Return status code and message returned by initiator.
     */
    return status;
}
