/*
 * This program is run by a process spawned by the SSH server, it periodically
 * generates new server keys in background and notifies the server process
 * each time a new key file is produced.  The default interval between
 * refreshes if 60 minutes, but can be overridden by specifying the number
 * of minutes as a command line argument.
 *
 * Usage:
 *    (1) mcr 'disk_dir'key_generator [update-time [initial-update]]
 *    (2) mcr 'disk_dir'key_generator -new bits key-file [e]
 *
 * Usage 1 applies when program is a helper, usage 2 is when run interactively
 * to produce an original key.
 *
 * The process communicates with the parent via a mailbox with the name
 * SSH_KEYGEN_MAILBOX.  The ssh server initially writes the name of
 * the server key file (or a logical pointing to it) to this mailbox and
 * the reads responses written by this process.
 *
 * Author:	David Jones
 * Date:	19-JUN-1998
 * Revised:	17-SEP-1998	Add initial delay option.
 * Revised:	11-NOV-1998	Add -new support.
 * Revised:	23-MAY-1999	Use common helper comm. library.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <ssdef.h>		/* VMS system service return status values */

#include "sshrsa.h"
#include "sshrsa_rpc.h"

#include "helper_com.h"

void SYS$EXIT();
int SYS$CLREF(), SYS$SETPRI(), SYS$SETIMR(), SYS$WAITFR();
int LIB$CVT_TO_INTERNAL_TIME(), LIB$GET_EF();

#define PROCESS_NAME "RSA keygen"
#define CTLMBX_NAME "SSH_KEYGEN_MAILBOX"
#define DEFAULT_PUB_EXPONENT 0x010001

#ifdef DEBUG
static void dump_buffer ( char *heading, char *buffer, int bufsize )
{
    int i, j;
    printf ( "%s (%d bytes):", heading, bufsize );
    j = 15;
    for ( i = 0; i < bufsize; i++ ) {
	j++;
	if ( j > 15 ) {
	    printf("\n   %04x:", i ); j = 0; }
	printf(" %02x", 255&buffer[i] );
    }
    printf("\n");
}
#else
#define dump_buffer(a,b,c) 1
#endif
/***********************************************************************/
void callback ( int type, int sequence )
{
    /* printf("Generator progress: %d %d\n", type, sequence ); */
}
static time_t showtime;
void show_progress ( int type, int sequence )
{
    if ( type == 0 ) {
	time_t now;
	now = time ( &now );
	if ( now >= showtime ) {
	    printf ( "produced %d candidate primes...\n", sequence );
	    showtime += 30;
	}
    } else if ( type == 2 ) {
	printf("prime failed, restart....\n");
    } else if ( type == 3 ) {
	if ( sequence == 0 ) printf ( "first prime generated...\n" );
	else printf ( "second prime generated\n" );
    }
}
/***********************************************************************/
/* Create a new RSA key from scratch.
 */
static int create_new_key ( int bits, char *outfile, int pub_exp )
{
    char errmsg[256];
    sshrsa key;
    int status, length, off;
    FILE *tmp;

    printf("Generating new RSA private key (%d bits)...\n", bits);
    sshrsa_init ("");			/* init random number generator */
    time ( &showtime );			/* set milemarker */
    showtime += 10;
    key = sshrsa_generate_key ( bits, pub_exp, show_progress );
    if ( key ) {
	/*
	 * write_keyfile expects file to exist, so create it.
	 */
	tmp = fopen ( outfile, "w" );
	if ( tmp ) {
	    fclose ( tmp );
	    status = sshrsa_write_keyfile ( outfile, key, errmsg );
	} else {
	    strcpy ( errmsg, "Error creating outfile: " );
	    off = strlen ( errmsg );
	    length = strlen ( outfile );
	    if ( length+off >= sizeof(errmsg) ) length = sizeof(errmsg)-off-1;
	    strncpy ( &errmsg[off], outfile, length );
	    errmsg[length+off] = '\0';
	}
	if ( (status&1) == 0 ) printf ( "%s\n", errmsg );
	else printf("Keyfile written: %s\n", outfile );
	
    } else {
	printf ( "Error generating key\n" );
	status = SS$_ABORT;
    }
    return status;
}
/***********************************************************************/

int main ( int argc, char **argv )
{
    int status, bits, e_bits, m_bits, e, delay, initial_delay, ef, length; 
    long pid, code, lksb[2], delta[2], initial_delta[2];
    char *m_data, *e_data;
    char skey_file[256], errmsg[256];
    sshrsa new, old;
    /*
     * Check for alternate format: key_generator -new bits filename [e]
     */
    if ( argc > 3 && strcmp ( argv[1], "-new" ) == 0 ) {
	return create_new_key ( atoi ( argv[2] ), argv[3],
		argc > 4 ? atoi(argv[4]) : DEFAULT_PUB_EXPONENT );
    }
    /*
     * Initialize communication with parent process, lower our priority.
     */
    status = helper_setup ( PROCESS_NAME, CTLMBX_NAME, SYS$EXIT,
		(void *) SS$_ABORT );
    status = SYS$SETPRI ( 0, 0, 4, 0, 0, 0 );
    /*
     * Read server key file name.  We do this step as soon as possible
     * so that server startup may procede.
     */
    status = helper_recv_message ( skey_file, sizeof(skey_file)-1, &length );
    if ( (status&1) == 0 ) {
	printf("error reading mailbox: %d\n", status);
	return status;
    }
    skey_file[length] = '\0';
    /*
     * Determine delay time to use and wait initial_delay if specified.
     * The initial delay is intended to defer the initial key generation
     * a couple of minutes to give the rest of system boot time to finish.
     */
    ef = 0;
    LIB$GET_EF ( &ef );
    delay = 60;
    if ( argc > 1 ) delay = atoi ( argv[1] );
    code = 24;
    status = LIB$CVT_TO_INTERNAL_TIME ( &code, &delay, delta );
    if ( status&1 == 0 ) {
	printf ( "Error calculating delta time: %d\n", status );
	exit ( status );
    }
    initial_delay = 0;
    if ( argc > 2 ) initial_delay = atoi ( argv[2] );
    if ( initial_delay > 0 ) {
	/*
	 * Convert minutes to internal time and wait for VMS timer.
	 */
        status = LIB$CVT_TO_INTERNAL_TIME ( &code, &initial_delay, 
		initial_delta );
	if ( (status&1) == 0 ) exit ( status );
	SYS$CLREF ( ef );
	SYS$SETIMR ( ef, initial_delta, 0, initial_delta, 0 );
	SYS$WAITFR ( ef );
    }
    /*
     * Initialize random number generator and setup RSA calls for local use.
     */
    sshrsa_init("");
    /*
     * Read file to get the number of bits in the modulus and the exponent.
     */
    errmsg[0] = '\0';
    old = sshrsa_read_keyfile ( skey_file, &bits, errmsg );
    if ( !old ) {
	fprintf(stderr,"Error reading keyfile (%s)\n", errmsg );
	return 20;
    }
    status = sshrsa_extract_public_key ( old, &e_bits, &e_data,
	&m_bits, &m_data );
    sshrsa_destroy ( old );
    /*
     * Enter main loop: generate key and notify parent.
     */
    while ( status&1 ) {
	SYS$CLREF ( ef );
	SYS$SETIMR ( ef, delta, 0, delta, 0 );
	/*
	 * Generate new key.
	 */
	new = sshrsa_generate_key ( bits, 3, callback );
	if ( new ) {
	   /*
	    * Re-write key file.
	    */
	    status = sshrsa_write_keyfile ( skey_file, new, errmsg );
	    if ( (status&1) == 0 ) fprintf( stderr,
		"Error updating server key file: %s", errmsg );
	    sshrsa_destroy ( new );
	} else status = 0;
	/*
	 * Notify parent that keyfile has been updated.
	 */
	if ( status&1 ) {
	    status = helper_send_message ( skey_file, strlen(skey_file) );
	}
	/*
	 * Sleep for 1 hour and generate new key.
	 */
	SYS$WAITFR ( ef );
    }
    return status;
}
