/*
 * Author:	David Jones
 * Date:	10-SEP-1998
 */
#include "pthread_1c_np.h"
#ifndef PTHREAD_USE_D4
	    typedef void * (*startroutine_t)(void *);
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <jpidef.h>		/* VMS getjpi codes */
#include <lckdef.h>
#include <descrip.h>
#include <ssdef.h>
int LIB$GETJPI(), SYS$DASSGN();
#define A(v) (lval=v,&lval)

#define KEYGEN_MAILBOX_NAME "SSH_KEYGEN_MAILBOX"

#define PORT_COUNT_LIMIT 32

#include "parameters.h"		/* Global parameter settings */
#include "completion_port.h"
#include "cport_sshsess.h"	/* SSH session managment */
#include "cport_sshlogin.h"	/* login authentication */
#include "cport_ucx.h"		/* TCP/IP access via completion port */
#include "cport_admin.h"	/* Administrator interface */
#include "event_report.h"	/* Event logging routines */
#include "helper.h"		/* Helper process creation */
#include "tmemory.h"		/* per-thread heaps */
#include "tutil.h"
#include "monitor.h"
/*
 * The cipher_prefs and auth_prefs arrays provide hints to
 * sshsess_new_session about the ciphers and auth types to advertise.
 * (In client mode the prefs actuall assign the selection priority).
 */
static char ssh_cipher_prefs[] = { SSH_CIPHER_3DES, SSH_CIPHER_IDEA, 
	SSH_CIPHER_RC4, SSH_CIPHER_DES, SSH_CIPHER_BLOWFISH, 0 };
static char ssh_auth_prefs[] = { SSH_AUTH_PASSWORD, 0 };

/***********************************************************************/
/* Define data structures used to manage client threads:
 *    client_db	
 *    client_instance
 */
typedef struct client_instance
{
    struct client_instance *next;
    struct client_db *parent;
    pthread_t id;
    tm_destructor_handle mem_lock;
    int index;				/* uniqueu number */
    int server_idx;			/* server key context index */
    int state;
    int remote_port;
    char remote_version[128];		/* client's SSH version */

    void *io_ctx;			/* ucx context */
    sshpad pad;				/* hook to packet driver */
    sshsess_session session;
    unsigned char remote_host[16];
} *client;

struct client_db {
    pthread_mutex_t lock;		/* use to synch access to this struct */
    pthread_cond_t idle;
    pthread_key_t instance;


    char *ssh_version;			/* version ID string */
    char *cipher_prefs;
    char *auth_prefs;
    sshsess_server *server;		/* array of server key contexts */
    int *server_refs;
    int cur_server;
    int state;
    int active_count;			/* size of active list */
    client active;			/* list of current active threads */
    client free;			/* lookaside list for client blocks */

    char ctlmsg[256];
};
static struct client_db client_info;
/*
 * Forward and external references.
 */
static void *ssh_client_shell ( client info );
static void ssh_client_rundown ( void *ctx );

int ssh_do_session ( sshsess_session, char * );

/**************************************************************************/
/* Update_server_context handles refreshing the server key.  Caller must
 * acquire info->lock mutex prior to calling this function.
 */
static void update_server_context (struct client_db *info, sshsess_server ctx,
	int client_limit )
{
    int server_idx;
    /*
     * Find free slot starting at current position.
     */
    server_idx = info->cur_server;
    if ( info->server_refs[server_idx] == 1 ) {
	/* destroy old, preserv host key  howver. */
	evr_event ( EVR_CONFIG, "Deleting stale server key, slot: %d at %t",
		server_idx, 0 );
	info->server[server_idx]->host_key = (sshrsa) 0;
	sshsess_destroy_server_context ( info->server[server_idx] );
	info->server_refs[server_idx] = 0;
    } else if ( info->server_refs[server_idx] <= 0 ) {
	/* Reuse this slot. */
    } else {
	--info->server_refs[server_idx];
	for ( server_idx = 0; server_idx < client_limit; server_idx++ ) {
	    if ( info->server_refs[server_idx] <= 0 ) break;
	}
	if ( server_idx >= client_limit ) {
	    evr_event ( EVR_EXCEPTION, "No free server context slots" );
	    return;	/* no free */
	}
    }
    info->cur_server = server_idx;
    info->server[info->cur_server] = ctx;
    info->server_refs[info->cur_server] = 1;		/* main thread */
    evr_event ( EVR_CONFIG, "New current server context, slot %d at %t",
	server_idx, 0 );
}
/****************************************************************************/
/* Initialize the client context database.
 */
static void init_client_info ( char *version, int client_limit )
{
    int i;
    /*
     * Initialize pthreads objects.
     */
    INITIALIZE_MUTEX(&client_info.lock);
    INITIALIZE_CONDITION(&client_info.idle);
    CREATE_KEY(&client_info.instance, ssh_client_rundown);
    /*
     * Initialize client_db data structure, shared among all client
     * threads using the same server context.
     */
    client_info.server = malloc ( sizeof(sshsess_server) * client_limit );
    client_info.server_refs = malloc(sizeof(int)*client_limit);
    for ( i = 0; i < client_limit; i++ ) client_info.server_refs[i] = 0;
    client_info.cur_server = 0;

    client_info.ssh_version = version;
    client_info.cipher_prefs = ssh_cipher_prefs;
    client_info.auth_prefs = ssh_auth_prefs;

    client_info.state = 1;		/* normal operation */
    client_info.active_count = 0;
    client_info.active = (client) 0;
    client_info.free = (client) malloc (
	sizeof(struct client_instance)*client_limit );
    for ( i = 0; i < client_limit; i++ ) {
	client_info.free[i].next = &client_info.free[i+1];
	client_info.free[i].parent = (&client_info);
	client_info.free[i].index = i+1;
    }
    client_info.free[client_limit-1].next = (client) 0;
}
/************************************************************************/
/*  The monitor function is responsible for initializing the client_db
 *  structure and accepting new connections on the listener isb.
 *  When a connection is recieved, a new client thread is created
 *  to handle that connection.
 */
int ssh_monitor ( char *version,
	sshsess_server server_ctx,
	cport_isb *listener,		/* terminated by null */
	cport_isb key_generator,
	helper_ctx generator_ctx,
	int stacksize,
	int client_limit,
	char errmsg[256] )
{
    pthread_attr_t client_attr;
    int i, status, lcount, lndx;
    client client_block;
    cport_port port;
    cport_isb isb, admin_pipe, *tmp;
    struct cportucx_accept_result *remote_info;
    char admin_command[512];
    /*
     * Initialize.
     */
    INITIALIZE_THREAD_ATTR ( &client_attr );
    pthread_attr_setinheritsched ( &client_attr, PTHREAD_EXPLICIT_SCHED );
    pthread_attr_setstacksize ( &client_attr, stacksize );
    init_client_info ( version, client_limit );
    /*
     * Count number of listen ports and allocate contexts to
     * receive connect information and temporary data.
     */
    update_server_context ( &client_info, server_ctx, client_limit );

    for ( lcount = 0; listener[lcount]; lcount++ );
    if ( lcount == 0 ) {
	evr_event ( EVR_EXCEPTION, "Listening on no ports" );
	return 20;
    }
    remote_info = (struct cportucx_accept_result *) malloc (
	sizeof(struct cportucx_accept_result) * lcount );
    tmp = (cport_isb *) malloc ( sizeof(cport_isb) * (lcount+1) );
    for ( i = 0; i <= lcount; i++ ) tmp[i] = (cport_isb) 0;
    /*
     * Initialize the administrator interface, which returns pipe to read
     * commands from.
     */
    port = listener[0]->port;
    admin_pipe = execadmin_init ( port );
    if ( admin_pipe ) status = cport_start_io ( admin_pipe, CPORT_READ,
		admin_command, sizeof(admin_command)-1 );
    /*
     * Start initial reads on listen channel(s) and key_generator mailbox
     */
    if ( key_generator ) status = cport_start_io ( key_generator, 
	CPORT_READ, client_info.ctlmsg, sizeof(client_info.ctlmsg) );
    for ( i = 0; i < lcount; i++ ) {
        status = cport_start_io ( listener[i], CPORTUCX_ACCEPT,
	    (char *) &remote_info[i], sizeof ( remote_info ) );
	if ( (status&1) == 0 ) break;
    }
    /*
     * Main loop, process connections.
     */
    for ( isb = (cport_isb) 0; (status & 1) && (client_info.state == 1); ) {
	/*
	 * Get next completed I/O Synchronization Block (isb) and identify
         * which listen port or temp stream completed.
	 */
	isb = cport_next_completion ( port );
	if ( !isb ) break;	/* fatal error */
	for ( lndx = 0; listener[lndx]; lndx++ ) {
	    if ( (isb == listener[lndx]) || (isb == tmp[lndx]) ) break;
	}
        /*
         * Dispatch based upon the I/O that completed.
         */
	if ( isb == listener[lndx] ) {
	    /*
	     * New connection, details are in remote_info structure:
	     *  remote_info.io_ctx          ptr to io_ctx for new connection.
	     *  remote_info.remote_port     Remote TCP/IP port number
	     *  remote_info.remote_address  IP address
	     *
	     * Allocate client instance and copy remote info.
	     */
	    pthread_mutex_lock ( &client_info.lock );
	    client_block = client_info.free;
	    if ( client_block ) {
		client_info.free = client_block->next;
		client_block->next = client_info.active;
		client_info.active = client_block;
		client_block->state = 0;
		memcpy ( client_block->remote_host, 
			remote_info[lndx].remote_address, 8);
		client_block->remote_port = remote_info[lndx].remote_port;
		client_block->io_ctx = remote_info[lndx].io_ctx;
		client_block->session = (sshsess_session) 0;
		pthread_mutex_unlock ( &client_info.lock );
	    } else {
		/*
		 * Exceeded connection limit.  Assign temorary stream
		 * and issue rundown request.
		 */
		tmp[lndx] = cport_assign_stream ( 
			port, &cportucx_driver,	remote_info[lndx].io_ctx, 0 );
		if ( !tmp[lndx] ) fprintf(stderr,"Fatal assign failure\n");
		cport_start_io ( tmp[lndx], CPORT_WRITE, 
			"Server full\r\n", 11 );
		continue;
	    }
	    /*
	     * Begin another listen.
	     */
if ( listener[lndx] < (cport_isb) 60 ) printf("corruption of listener list[%d] = %d\n", 
lndx, listener[lndx] );
	    status = cport_start_io ( listener[lndx], CPORTUCX_ACCEPT,
		(char *) &remote_info[lndx], sizeof ( remote_info ) );
	    /*
	     * Create client thread.
	     */
#ifdef PTHREAD_USE_D4
	    status = pthread_create ( &client_block->id, client_attr,
		(pthread_startroutine_t) ssh_client_shell, client_block );
#else
	    status = pthread_create ( &client_block->id, &client_attr,
		(startroutine_t) ssh_client_shell, client_block );
#endif
	    if ( status != 0 ) {
		fprintf(stderr,"Client thread create failure\n");
		/* Failure to create a thread is a fatal error. */
		return 0;
	    }
	    status = 1;
	
	} else if ( isb == tmp[lndx] ) {
	    /*
	     * Write of the error message completed, kill the connection.
	     */
	    cport_deassign (tmp[lndx] );
	    tmp[lndx] = (cport_isb) 0;
	    /*
	     * Begin another listen.
	     */
	    status = cport_start_io ( listener[lndx], CPORTUCX_ACCEPT,
		(char *) &remote_info[lndx], sizeof ( remote_info ) );
	} else if ( isb == key_generator ) {
	    struct mbx_iosb { short status, length; long pid; } *iosb;
	    /*
	     * Control message from key generator sub-process received,
	     * verify it came from the process we spawned.
	     */
	    iosb = (struct mbx_iosb *) key_generator->iosb;
	    if ( iosb->pid != generator_ctx->pid ) {
		evr_event ( EVR_EXCEPTION, 
			"Wrong PID in key generator message: %x at %t", 
			iosb->pid, 0 );
	    } else if ( iosb->status & 1 ) {
		/*
		 * Message from key generator, update the server key.
		 */
		sshsess_server new_ctx;

		pthread_mutex_lock ( &client_info.lock );
		client_info.ctlmsg[iosb->length] = '\0';
		new_ctx = sshsess_server_copy ( client_info.ctlmsg,
			server_ctx, errmsg );
		if ( new_ctx ) {
		    update_server_context(&client_info, new_ctx, client_limit);
		     server_ctx = new_ctx;
	 	} else {
		    evr_event ( EVR_EXCEPTION, 
			"Error making new incarnation of server context: %s at %t",
			errmsg, 0 );
		}
		pthread_mutex_unlock ( &client_info.lock );
	    } else {
		/*
		 * Error status in IOSB.
		 */
	 	evr_event ( EVR_EXCEPTION, 
		    "Error on key generator mailbox: %d at %t", iosb->status,0);
	    }
	    /*
	     * Leave next message read pending.
	     */
	    status = cport_start_io ( key_generator, CPORT_READ, 
		client_info.ctlmsg, sizeof(client_info.ctlmsg)-1 );

	} else if ( isb == admin_pipe ) {
	    /*
	     * Administrator command sent by administrator.
	     */
	    if ( (admin_pipe->default_iosb[0]&1) == 1 ) {
		admin_command[admin_pipe->default_iosb[1]] = '\0';
		pthread_mutex_lock ( &client_info.lock );
		evr_event ( EVR_CONFIG, 
		    "Monitor received admin command '%s'", admin_command );
		    
		if ( tu_strncmp ( admin_command, "SHUTDOWN", 9 ) == 0 ) {
		    client_info.state = 0;
		} else if ( tu_strncmp ( admin_command, "CLOSE", 8 ) == 0 ) {
		    client_info.state = 2;
		} else if ( tu_strncmp ( admin_command, "SHUTDOWN_RESTART", 
				16 ) == 0 ) {
		    client_info.state = 3;
		} else if (tu_strncmp(admin_command, "CLOSE_RESTART", 16)==0) {
		    client_info.state = 4;
		}
		pthread_mutex_unlock ( &client_info.lock );
	    } else evr_event ( EVR_EXCEPTION, "Admin pipe error: %d\n",
		admin_pipe->default_iosb[0] );
	    execadmin_done ( 1, "" );
	    status = cport_start_io ( admin_pipe, CPORT_READ,
		admin_command, sizeof(admin_command)-1 );
	}
    }
    /*
     * Cleanup.
     */
    for ( i = 0; i < lcount; i++ ) cport_deassign ( listener[i] );
    if ( client_info.state > 2 ) {
	/*
	 * restart requested.
	 */
	pthread_mutex_lock ( &client_info.lock );
	client_info.state = client_info.state - 2;
	pthread_mutex_unlock ( &client_info.lock );
	helper_restart();
    }

    if ( client_info.state == 2 ) {
	/*
	 * client is in closed state, stop listening but stall until all
	 * close.
	 */
	pthread_mutex_lock ( &client_info.lock );
	while ( client_info.active ) pthread_cond_wait ( 
		&client_info.idle, &client_info.lock );
	pthread_mutex_unlock ( &client_info.lock );
    }
    if ( admin_pipe ) cport_deassign ( admin_pipe );
    if ( key_generator ) cport_deassign ( key_generator );
    for ( i = 0; i < lcount; i++ ) if ( tmp[i] ) cport_deassign ( tmp[i] );
    free ( tmp );
    pthread_mutex_destroy ( &client_info.lock );
    pthread_cond_destroy ( &client_info.idle );
    free ( client_info.server_refs );
    free ( client_info.server );
    free ( client_info.free );
    return 1;
}
/*****************************************************************************/
/* Scan active sessions.
 */
int ssh_scan_sessions ( 
	int (*callback)(int, sshsess_session, void *), void *arg)
{
    client cp;
    sshsess_session session;
    struct sshsess_session_st fallback;
    int i, status;
    /*
     * Scan the active connections.
     */
    pthread_mutex_lock ( &client_info.lock );
    for ( cp = client_info.active; cp; cp = cp->next ) {
	session = cp->session;
	if ( !session ) {
	    /*
	     * Connection has no session yet, make a fake one.
	     */
	    fallback.pad = (sshpad) 0;
	    fallback.remote_port = cp->remote_port;
	    memcpy ( fallback.remote_host, cp->remote_host, 
		sizeof(fallback.remote_host) );
	    fallback.hostname = "";
	    fallback.server_ctx = (struct sshsess_server_st *) 0;
	    fallback.locus = (sshmsg_locus) 0;
	    fallback.auth_type = 0;
	    fallback.username[0] = '\0';
	    fallback.ext_username = fallback.username;
	    session = &fallback;
        }
	status = callback ( cp->index, session, arg );
	if ( (status&1) == 0 ) break;
	i++;
    }
    pthread_mutex_unlock ( &client_info.lock );
    return i;
}
/*****************************************************************************/
/* Destructor routine for client_db.instance context key.  Ensures that
 * client thread exit will be noticed by the monitor thread.
 */
static void ssh_client_rundown ( void *ctx )
{
    client self, cp;
    struct client_db *info;
    /*
     * Restore state from context key.
     */
    self = (client) ctx;
    info = self->parent;
    if ( !info ) return;
    pthread_mutex_lock ( &info->lock );
    /*
     * De-reference server key context, if last one then run it down.
     */
    --info->server_refs[self->server_idx];
    if ( info->server_refs[self->server_idx] <= 0 ) {
	evr_event ( EVR_CONFIG, "Deleting server key, slot: %d at %t",
		self->server_idx, 0 );
	info->server[self->server_idx]->host_key = (sshrsa) 0;
	sshsess_destroy_server_context ( info->server[self->server_idx] );
	info->server_refs[self->server_idx] = 0;
    }
    /*
     * Move block to free list, signal if are shutting down.
     */
    if ( info->active == self ) {
	info->active = self->next;
	--info->active_count;
    } else {
	for ( cp = info->active; cp; cp = cp->next ) {
	    if ( cp->next == self ) {
		cp->next = self->next;
		--info->active_count;
		break;
	    }
	}
	if ( !cp ) {
	    evr_event ( EVR_EXCEPTION, 
		"Bugcheck, self block not found in client_db, %t", 0 );
	}
    }
    self->next = info->free;
    tm_unlock_destructor ( self, self->mem_lock );
    info->free = self;
    if ( (info->state != 1) && !info->active ) {
	pthread_cond_signal ( &info->idle );
    }
    pthread_mutex_unlock ( &info->lock );
}
#ifdef CHECK_STACK
int dirty_poison(char *ptr, int kbyte_count);
char *poison_stack(int kbyte_count );
#endif
/***********************************************************************/
/* ssh_client_shell is the start routine for client threads created
 * by the monitor thread (main thread).  It is responsible for:
 *    1. Creating a completion port for serializing async I/O related
 *       to the connection.
 *    2. Creating a session context for servicing the connection.
 *    3. Calling the service routine to process the connection.
 *    4. Destroying the completion port.
 */
static void *ssh_client_shell ( client self )
{
    int status; 
    cport_port port;
    char errmsg[256];
    sshsess_server server;
#ifdef CHECK_STACK
    char *pptr;
    int dcount;
    pptr = poison_stack(50);
#endif
    /*
     * Establish context key value to ensure clean rundown.
     */
    self->id = pthread_self();
    DETACH_THREAD ( self->id );
    pthread_setspecific ( self->parent->instance, self );
    self->mem_lock = tm_lock_destructor ( self );
    evr_open_report ( &self->index );
    evr_event ( EVR_NEW_CONNECTION, "New connection at %t from %a:%d",
	0, self->remote_host, self->remote_port );
    /*
     * Determine server key context to use and bump it's reference count.
     */
    pthread_mutex_lock ( &self->parent->lock );
    self->server_idx = self->parent->cur_server;
    server = self->parent->server[self->server_idx];
    self->parent->server_refs[self->server_idx]++;
    pthread_mutex_unlock ( &self->parent->lock );
    /*
     * Create completion port queue object.
     */
    port = cport_create_port ( 0 );
    /*
     * Now process the incoming connection.  Make service call inside
     * of TRY block so execeptions are less likely to kill the server.
     */
#ifndef DONT_TRY
    TRY {
#endif
	/*
	 * Create packet assembler object and attempt to start new
	 * session.
	 */
	self->pad = sshpad_create ( port, self->io_ctx, 1, 
		self->parent->ssh_version, self->remote_version, errmsg );
	if ( self->pad ) {
	    sshsess_session new_session;
	    /*
	     * Get session key and negotiate cipher.
	     */
	    new_session = sshsess_new_session ( 
		server,				/* server context */
		port,				/* completion queue */
		self->pad,			/* packet layer context */
		self->parent->cipher_prefs,
		self->parent->auth_prefs, errmsg );
	    if ( new_session ) {
		/*
		 * Let another module handle actual processing of session.
		 */
		evr_event ( EVR_NEW_SESSION,
			"New session established, cipher: %d, server ctx: %d",
			new_session->cipher, self->server_idx );
		new_session->remote_port = self->remote_port;
		memcpy ( new_session->remote_host, self->remote_host,
			sizeof(new_session->remote_host) );
		pthread_mutex_lock ( &self->parent->lock );
		self->session = new_session;
		pthread_mutex_unlock ( &self->parent->lock );
		status = ssh_do_session ( self->session, errmsg );
		evr_event ( EVR_EXIT, 
		    "Final session status: %d '%s' at %t", status, errmsg, 0 );
		/*
	 	 * Clean up.
		 */
		pthread_mutex_lock ( &self->parent->lock );
		self->session = (sshsess_session) 0;
		pthread_mutex_unlock ( &self->parent->lock );
		sshsess_rundown ( new_session );
	    } else {
		/*
		 * Session aborted or failed.
	         */
	    	evr_event ( EVR_EXIT, 
			"Session setup failed '%s' at %t", errmsg, 0 );
	    }
	} else {
	    /*
	     * Packet layer failed to start.
	     */
	    evr_event ( EVR_EXIT, 
			"Session PAD layer failure '%s' at %t", errmsg, 0 );
	}
#ifndef DONT_TRY
    }
#ifdef PTHREAD_USE_D4
    CATCH(cma_e_exit_thread) { /* Normal event, mst_exit called */ }
    CATCH_ALL {
	evr_event ( EVR_EXCEPTION, "Shell caught exception at %t", 0 );
	exc_report ( THIS_CATCH );
    }
#else
    CATCH(pthread_exit_e) { /* Normal event, mst_exit called */ }
    CATCH_ALL {
	evr_event ( EVR_EXCEPTION, "Shell caught exception at %t", 0 );
    }
#endif
    ENDTRY
#endif

    status = cport_destroy_port ( port );
#ifdef CHECK_STACK
    dcount = dirty_poison(pptr,50);
    evr_event ( EVR_DEBUG, "Dirty stack: %dK\n", dcount);
#endif
    evr_close_report ( );
    return (void *) 0;
}
