/*
 * This module sets the default value for the param struct in parameters.h
 * and reads a configuration file to override these defaults.
 *
 * Syntax of configuration file:
 *     # comment line
 *     parameter-name value
 *     parameter-name 'envvar'
 *     paraemter-name key=value
 *
 * A parameter with the 'list' type allows multiple instances of the
 * definition, you need a separate line for each instance.
 *
 * A parameter with an 'mlist' type builds a 2-level heirarchy of lists
 * of values, each sub-list distinguished by the key value.  If no 'key=' is
 * present (value only), the key is null string.
 *
 * If the value string is surrounded by single quotes ('), the string is
 * assumed to be an environment variable and is translated.
 *
 * Author:  David Jones
 * Date:    27-JUN-1998
 * Revised:  7-AUG-1998			support env var substitution.
 *					Change param.port to list type.
 * Revised: 26-SEP-1998			Added MLIST type and tt type support,
 *					added .include statements.
 * Revised: 19-OCT-1998			Use LIB$GET_SYMBOL in place of
 *					getenv() for quoted substitution.
 * Revised: 4-NOV-1998			Fix value length for quoted subst.
 * Revised: 23-MAY-1999			Added pty_accpornam support
 */
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <ttdef.h>		/* VMS terminal characteristics defs */
#include <tt2def.h>
#include <descrip.h>		/* VMS string descriptors */

#include "parameters.h"

#define PDEF_END 0		/* sentinel for end of list */
#define PDEF_INT 1
#define PDEF_STRING 2
#define PDEF_LIST 3
#define PDEF_MLIST 4		/* list with sublist */
#define NEST_LIMIT 10
/*
 * Define table to bind each config file keyword to a param structure field.
 */
static struct definition {
    char *name;			/* parameter name */
    int type;			/* data type */
    void *data;
    char *defvalue;		/* default value */
} pdef[] = {
    { "port", PDEF_LIST, &param.ports, "22" },
    { "login_timeout", PDEF_INT, &param.login_timeout, "60" },
    { "max_clients", PDEF_INT, &param.max_client, "16" },
    { "decnet_cmd", PDEF_INT, &param.decnet_cmd, "1" },
    { "host_key", PDEF_STRING, &param.hkey_file, "hkey.pem" },
    { "server_key", PDEF_STRING, &param.skey_file, "skey.pem" },
    { "event_formatter", PDEF_STRING, &param.event_fmt, "??" },
    { "event_log", PDEF_STRING, &param.event_log, "??" },
    { "allow_cipher", PDEF_LIST, &param.ciphers, "all" },
    { "allow_authtype", PDEF_LIST, &param.auth_types, "password" },
    { "user_keyfile", PDEF_STRING, &param.user_keyfile,
	"ssh/authorized_keys.dat" },
    { "rsa_engine", PDEF_STRING, &param.rsa_engine, "" },
    { "key_generator", PDEF_STRING, &param.keygen, "" },
    { "initiator", PDEF_STRING, &param.initiator, "" },
    { "cmd_task", PDEF_STRING, &param.cmd_task, "0=SSHEXEC" },
    { "login_table", PDEF_STRING, &param.login_table, "SSH_LOGIN_MAP" },
    { "login_info_table", PDEF_STRING, &param.login_info_table, 
	"SSH_LOGIN_INFO" },
    { "pty_map_table", PDEF_STRING, &param.pty_map_table, "SSH_PTY_MAP" },
    { "multi-user", PDEF_LIST, &param.multi_user, "127.0.0.1" },
    { "tt_char", PDEF_MLIST, &param.tt_char, "nomodem" },
    { "administrator", PDEF_LIST, &param.administrator, "SYSTEM" },
    { "x11_servers", PDEF_INT, &param.x11_servers, "0" },
    { "x11_server_number", PDEF_INT, &param.x11_server_number, "1" },
    { "x11_table_size", PDEF_INT, &param.x11_table_size, "5" },
    { "tt_type", PDEF_MLIST, &param.tt_type, "vt100" },
    { "pty_mode", PDEF_INT, &param.pty_mode, "1" },
    { "required_identifier", PDEF_LIST, &param.required_id, "" },
    { "port_forward", PDEF_INT, &param.port_forward, "3" },
    { "x11_decnet_node", PDEF_LIST, &param.x11_decnet_node, "" },
    { "fwd_map_table", PDEF_STRING, &param.fwd_map_table, "SSH_FWD_MAP" },
    { "fwd_map_enable", PDEF_LIST, &param.fwd_map_enable, "" },
    { "pty_accpornam", PDEF_INT, &param.pty_accpornam, "0" },
    { "pty_accpornam_image", PDEF_STRING, &param.pty_accpornam_image,
		"SYS$DISK:[].EXE" },
    { "", PDEF_END, (void *) 0, "" }
};
/*
 * define table for tt type codes, only major types defined, use numeric
 * codes directly for others.
 */
static char tt_type_sentinel[64];
static struct tttypedef {
    char *name;			/* terminal type name (e.g. vt100) */
    int type;			/* type code */
} tt_type_def[] = {
    { "VT100", TT$_VT100 },
    { "VT200_SERIES", TT$_VT200_SERIES },
    { "VT300_SERIES", TT$_VT300_SERIES },
    { "VT400_SERIES", TT$_VT400_SERIES },
    { "VT500_SERIES", TT$_VT500_SERIES },
    { "PRO_SERIES", TT$_PRO_SERIES },
    { "VT101", TT$_VT101 },
    { "VT102", TT$_VT102 },
    { "UNKNOWN", TT$_UNKNOWN },
    { tt_type_sentinel, TT$_UNKNOWN }	/* marks end of list */
};
/*
 * Define table for tt characteristics.
 */
static char tt_char_sentinel[64];
static struct ttchardef {
    char *name;			/* term characteristic name */
   int tt_char;
   int ext_char;
} tt_char_def[] = {
    { "crfill", TT$M_CRFILL,  0 },    { "eightbit", TT$M_EIGHTBIT, 0 },
    { "escape", TT$M_ESCAPE, 0 },     { "halfdup", TT$M_HALFDUP, 0 },
    { "hostsync", TT$M_HOSTSYNC, 0 }, { "lffill", TT$M_LFFILL, 0 },
    { "lower", TT$M_LOWER, 0 },       { "mbxdsabl", TT$M_MBXDSABL, 0 },
    { "mechform", TT$M_MECHFORM,0 },  { "mechtab", TT$M_MECHTAB, 0 },
    { "form", TT$M_MECHFORM,0 },      { "tab", TT$M_MECHTAB, 0 },
    { "modem", TT$M_MODEM, 0 },       { "nobrdcst", TT$M_NOBRDCST, 0 },
    { "noecho", TT$M_NOECHO, 0 },     { "notypeahd", TT$M_NOTYPEAHD, 0 },
    { "readsync", TT$M_READSYNC, 0 }, { "remote", TT$M_REMOTE, 0 },
    { "scope", TT$M_SCOPE, 0 },       { "ttsync", TT$M_TTSYNC, 0 },
    { "wrap", TT$M_WRAP, 0 },
    { "altypeahd", 0, TT2$M_ALTYPEAHD }, { "ansicrt", 0, TT2$M_ANSICRT },
    { "app_keypad", 0, TT2$M_APP_KEYPAD }, { "autobaud", 0, TT2$M_AUTOBAUD },
    { "avo", 0, TT2$M_AVO }, { "block", 0, TT2$M_BLOCK },
    { "brdcstmbx", 0, TT2$M_BRDCSTMBX }, { "commsync", 0, TT2$M_COMMSYNC },
    { "deccrt", 0, TT2$M_DECCRT }, { "deccrt2", 0, TT2$M_DECCRT2 },
    { "deccrt3", 0, TT2$M_DECCRT3 }, { "deccrt4", 0, TT2$M_DECCRT4 },
    { "dialup", 0, TT2$M_DIALUP }, { "disconnect", 0, TT2$M_DISCONNECT },
    { "dma", 0, TT2$M_DMA }, { "drcs", 0, TT2$M_DRCS },
    { "edit", 0, TT2$M_EDIT }, { "editing", 0, TT2$M_EDITING },
    { "fallback", 0, TT2$M_FALLBACK }, { "hangup", 0, TT2$M_HANGUP },
    { "insert", 0, TT2$M_INSERT }, { "localecho", 0, TT2$M_LOCALECHO },
    { "modhangup", 0, TT2$M_MODHANGUP }, { "pasthru", 0, TT2$M_PASTHRU },
    { "printer", 0, TT2$M_PRINTER }, { "regis", 0, TT2$M_REGIS },
    { "sixel", 0, TT2$M_SIXEL }, { "secure", 0, TT2$M_SECURE },
    { "setspeed", 0, TT2$M_SETSPEED }, { "syspwd", 0, TT2$M_SYSPWD },
    { "xon", 0, TT2$M_XON },
    { tt_char_sentinel, 0, 0 }
};
/*
 * Set_mvalue function inserts into keyword list.
 */
struct parameter_mlist_elem *set_mvalue ( struct parameter_mlist_elem *mlist, 
	char *value )
{
    char key[128];
    struct parameter_mlist_elem *p;
    struct parameter_list_elem *v;
    int i, keylen;
    /*
     * Parse value into key and string.  If no equals, key is null.
     */
    for ( keylen = i = 0; value[i]; i++ ) if ( value[i] == '=' ) {
	keylen = i;
	if ( keylen >= sizeof(key) ) keylen = sizeof(key) -1;
	strncpy ( key, value, keylen );
	value = &value[i+1];
	break;
    }
    key[keylen] = '\0';
    /*
     * scan for matching key in mlist, inserting new entry if needed.
     */
    for ( p = mlist; p; p = p->next ) if ( strcmp(p->value,key) == 0 ) break;
    if ( !p ) {
	/*
	 * Add new block.
	 */
	p = (struct parameter_mlist_elem *) malloc ( keylen +
		sizeof(struct parameter_mlist_elem) );
	if ( !p ) return mlist;
	p->next = mlist;
	mlist = p;
	p->child = (struct parameter_list_elem *) 0;
	strcpy ( p->value, key );
    }
    /*
     * Append sub-value.
     */
    v = (struct parameter_list_elem *) malloc ( strlen ( value ) +
	sizeof (struct parameter_list_elem) );
    strcpy ( v->value, value );
    v->next = p->child;
    p->child = v;

    return mlist;
}
/*
 * set_value function interprets value string according to type in
 * defintion struct and set value.
 */
static void set_value ( struct definition *def, char *value )
{
    int number, length, status, LIB$GET_SYMBOL(), tbl_ind;
    char *string;
    struct parameter_list_elem *element;

    length = strlen ( value );
    if ( *value == '\'' && length > 1 ) if ( value[length-1] == '\'' ) {
	/*
	 * String is surrounded by quotes, copy to varname and translate
	 * via LIB$GET_SYMBOL().
	 */
	static char sym_value[1023], fill;
	static $DESCRIPTOR(varname_dx, "");
	static $DESCRIPTOR(sym_value_dx, sym_value);

	varname_dx.dsc$a_pointer = &value[1];
	varname_dx.dsc$w_length = length-2;
	length = tbl_ind = 0;

	status = LIB$GET_SYMBOL ( &varname_dx, &sym_value_dx, &length,
		&tbl_ind );
	if ( (status&1) == 1 ) {
	    value = sym_value;
	    sym_value[length] = '\0';
	} else {
	    fprintf(stderr,"Symbol '%s' is undefined (code=%d)\n", value,status );
	    value = "**UNDEFINED**";
	    length = 13;
	}
    }

    switch ( def->type ) {
	case PDEF_INT:
	    number = atoi ( value );
	    *((int *) def->data) = number;
	    break;
	case PDEF_STRING:
	    if ( length == 0 ) *((char **) def->data) = "";
	    else {
		string = malloc ( length + 1 );
		strcpy ( string, value );
		*((char **) def->data) = string;
	    }
	    break;
	case PDEF_LIST:
	    length = length + sizeof(struct parameter_list_elem);
	    element = (struct parameter_list_elem *) malloc ( length );
	    strcpy ( element->value, value );
	    element->next = *((struct parameter_list_elem **) def->data);
	    *((struct parameter_list_elem **) def->data) = element;
	    break;

	case PDEF_MLIST:
	    *((struct parameter_mlist_elem **) def->data) = set_mvalue ( 
		*((struct parameter_mlist_elem **) def->data), value );
	    break;

	case PDEF_END:
	    fprintf(stderr,"Unknown parameter name: %s\n", def->name);
	    break;
    }
}
/***************************************************************************/
/* Initialize param struct to default values and read specified configuration
 * file to override the defaults.
 */
int init_parameters ( char *config_file )
{
    FILE *cf[NEST_LIMIT];
    int i, j, length, count, lvl;
    struct parameter_list_elem *list, *x, **listhead;
    char *value;
    char line[2048];
    /*
     * Set default values, list type entries start out empty.
     */
    for ( i = 0; pdef[i].type != PDEF_END; i++ ) {
	if ( pdef[i].type == PDEF_LIST ) {
	    *((void **) pdef[i].data) = (void *) 0;
	} else if ( pdef[i].type == PDEF_MLIST ) {
	    *((void **) pdef[i].data) = (void *) 0;
	} else set_value ( &pdef[i], pdef[i].defvalue );
    }
    count = i;		/* number of keywords */
    /*
     * open configuration file and read.
     */
    if ( config_file[0] == '\0' ) config_file = "_NL:"; /* no parameter file */
    cf[0] = fopen ( config_file, "r" );
    if ( !cf[0] ) return 0;
    for ( lvl = 0; lvl >= 0; --lvl ) {
      while ( fgets ( line, sizeof(line), cf[lvl] ) ) {
	char *tok, *keyword;
	/*
	 * Convert LF in line to NULL and parse initial word.
	 */
	tok = strchr ( line, '\n' ) ;
	if ( tok ) *tok = '\0';		/* convert LF to null */
	length = strlen ( line );
	keyword = strtok ( line, " \t" );
	if ( !keyword ) continue;		/* null line */
	if ( *keyword == '#' || *keyword == '!' ) continue;	/* comment */
        /*
         * force keyword to lowercase for comparison and search
         */
	for ( i=0; keyword[i]; i++ ) keyword[i] = tolower(keyword[i]);
	pdef[count].name = keyword;		/* set sentinel */
	for ( i = 0; 0 != strcmp ( pdef[i].name, keyword ); i++ );
	if ( pdef[i].type == PDEF_END ) {
	    /*
	     * keyword not a parameter name, see if special code.
	     */
	    if ( strcmp ( keyword, ".include" ) == 0 ) {
		if ( lvl < (NEST_LIMIT-1) ) {
		    if ( length <= strlen ( keyword ) ) continue;  /* no value */
		    for (j=strlen(keyword)+1; line[j] == ' ' || 
			line[j] == '\t'; j++);
		   cf[lvl+1] = fopen ( &line[j], "r" );
		   if ( cf[lvl+1] ) lvl++;
		   else fprintf ( stderr, "Open failure on include file '%s'\n",
			&line[j] );
		} else {
		    fprintf ( stderr, 
			"Nesting level too deep on include file\n");
		}
	    } else {
	        fprintf(stderr,"Unknown parameter name: '%s'\n", keyword );
	    }
	    continue;
	}
	/*
	 * Parse value from command line and dequote.
	 */
	if ( length <= strlen ( keyword ) ) continue;  /* no value */
	for (j=strlen(keyword)+1; line[j] == ' ' || line[j] == '\t'; j++);
	value = &line[j];
	if ( !*value ) continue;
	if ( *value == '"' ) {
	    value++;
	    for ( j = 0; value[j] && (value[j] != '"'); j++ );
	    value[j] = '\0';
	} else {
	    value = strtok ( value, " \t" );
	}
	set_value ( &pdef[i], value );
      }
      fclose ( cf[lvl] );
    }
    /*
     * File scan builds list in reverse order, invert them.
     * Note that member next is assumed to be at the same offset (0) for
     * both the parameter_list_elem and parameter_mlist_elem structs.
     */
    for ( i = 0; pdef[i].type != PDEF_END; i++ ) {
	if (pdef[i].type != PDEF_LIST && pdef[i].type != PDEF_MLIST ) continue;

	listhead = (struct parameter_list_elem **) pdef[i].data;
	if ( !*listhead ) {
	    /*  Set default value for empty list */
	    set_value ( &pdef[i], pdef[i].defvalue );
	    listhead = (struct parameter_list_elem **) pdef[i].data;
	}
	list = (struct parameter_list_elem *) 0;
	while ( *listhead ) {
	    x = *listhead;
	    *listhead = x->next;
	    x->next = list;
	    list = x;
	}
	*listhead = list;
    }
    return 1;
}
/****************************************************************************/
/* Scan TT list and set paramters.  defchar and defchar2 are not initialized.
 */
int convert_tt_parameters ( struct parameter_list_elem *tt_char,
	int *defchar, int *defchar2 )
{
    struct parameter_list_elem *item;
    int i, negate;
    char *teststr;
    char keyword[sizeof(tt_char_sentinel)];

    for ( item = tt_char; item; item = item->next ) {
	/*
	 * Make lowercase form of keyword for caseless compares.
         */
	for ( i = 0; i < sizeof(keyword)-1 && item->value[i]; i++ ) {
	    keyword[i] = tolower ( item->value[i] );
	}
	keyword[i] = '\0';
	/*
	 * Determine if prefix present.
	 */
	negate = 0;
	teststr = keyword;
	if ( keyword[0] == '-' ) { negate = 1; teststr++; }
	if ( keyword[0] == '+' ) teststr++;
	strcpy ( tt_char_sentinel, teststr );
	/*
   	 * Scan keyword table, and mask in/out bits if match found.
	 */
	for ( i = 0; 0 != strcmp ( teststr, tt_char_def[i].name ); i++ );
	if ( tt_char_def[i].name == tt_char_sentinel ) return 0;
	if ( negate ) {
	    *defchar &= (~tt_char_def[i].tt_char);
	    *defchar2 &= (~tt_char_def[i].ext_char);
	} else {
	    *defchar |= tt_char_def[i].tt_char;
	    *defchar2 |= tt_char_def[i].ext_char;
	}
    }
    return 1;
}
/****************************************************************************/
/* Match terminal name in term type table.
 */
int convert_tt_type_parameters ( char *type_key, char *type_value,
	struct parameter_mlist_elem *tt_char,
	unsigned char *type_code, int *defchar, int *defchar2 )
{
    int i;
    struct parameter_mlist_elem *clist;
    char keyword[sizeof(tt_type_sentinel)];
    /*
     * Scan for type key, case sensitive.
     */
    for ( clist = tt_char; clist; clist = clist->next ) {
	if ( strcmp ( type_key, clist->value ) == 0 ) break;
    }
    /*
     * type_value is either numeric code or standard name in tt_type_def.
     */
    if ( type_value ) {
        if ( type_value[0] >= '0' && type_value[0] <= '9' ) {
	    *type_code = atoi ( type_value );
	} else {
	    /*
	     * upcase type name and set sentinel for table search.
	     */
	    for ( i = 0; i < sizeof(tt_type_sentinel)-1 && 
		type_value[i]; i++ ) keyword[i] = toupper(type_value[i]);
	    keyword[i] = '\0';
	    strcpy ( tt_type_sentinel, keyword );
	    /*
	     * Search the terminal type name table and return matching
	     * type (sentinel's type is unknown).
	     */
	    for ( i = 0; strcmp ( keyword, tt_type_def[i].name ); i++ );
	    *type_code = tt_type_def[i].type;
	}
    }
    /*
     * clist_value is list of terminal characteristics specific to this
     * terminal.
     */
    if ( clist ) if ( clist->child ) {
	return convert_tt_parameters ( clist->child, defchar, defchar2 );
    }
    return 0;
}
