/*
 * main module for handling SSH port forwarding operations.  When a SSH client
 * requests port forwarding (e.g. X11_REQUEST_FORWARDING), a separate thread
 * (the forwarder thread) is created to oversee this activity.  The forwarder
 * thread communicates with the client thread via a pipe.
 *
 * Author:	David Jones
 * Date:	13-SEP-1998
 * Revised:	26-NOV-1998		Enhance X11 support.
 * Revised:	29-JAN-1999		Support port forward table, modify
 *					pf_create_forwarder interface.
 * Revised:	31-JAN-1999		Ensuer forward table cleanup.
 */
#include <stdlib.h>
#include <stdio.h>
#include <ssdef.h>			/* VMS sys service return codes */
#include <lnmdef.h>			/* VMS logical name codes */
#include <descrip.h>			/* VMS string descriptors */
#include "parameters.h"
#include "pthread_1c_np.h"
#include "tmemory.h"
#include "tutil.h"
#include "event_report.h"
#include "completion_port.h"
#include "cport_pipe.h"
#include "cport_sshpad.h"
#include "cport_sshmsg.h"
#ifndef PTHREAD_USE_D4
typedef void * (*startroutine_t)(void *);
#endif

#include "port_forwarder.h"
#ifdef CHECK_STACK
int dirty_poison(char *ptr, int kbyte_count);
char *poison_stack(int kbyte_count );
#endif

int SYS$CRELNM(), SYS$DELLNM(), SYS$CRELNT();
static $DESCRIPTOR(system_directory,"LNM$SYSTEM_DIRECTORY");
static $DESCRIPTOR(fwd_map_table,"");
static int *fwd_map_enable;		/* terminate with 0 */
/***************************************************************************/
/*
 * Prototypes for forward references.
 */
static void *pf_forwarder_shell ( struct pf_control_block * );
static void pf_fwdr_rundown(void *ctx);
static void pf_crtr_rundown(void *ctx);
static int pf_main ( struct pf_control_block *ctl );
static pthread_once_t setup = PTHREAD_ONCE_INIT;
/******************************************************************************/
/*
 * One-time initialization routine.
 */
static void pf_once ()
{
    INITIALIZE_MUTEX ( &pf.lock );
    INITIALIZE_THREAD_ATTR ( &pf.ctl_attr );
    pthread_attr_setinheritsched ( &pf.ctl_attr, PTHREAD_EXPLICIT_SCHED );
    pthread_attr_setstacksize ( &pf.ctl_attr, 62000 );
    pf.ctl_index = 0;			/* global thread number assignment */
    CREATE_KEY ( &pf.creator_key, pf_crtr_rundown );
    INITIALIZE_CONDITION ( &pf.ready );
    pf.free_ctl = (pf_ctlblk) 0;

    pf.x11_server_top = 0;		/* initialize to top of stack. */
    if ( param.x11_servers > 0 ) {
	int i;
	pf.x11_server_stack = (int *) malloc (sizeof(int) * param.x11_servers);
	if ( !pf.x11_server_stack ) param.x11_servers = 0;
	for ( i = 0; i < param.x11_servers; i++ ) pf.x11_server_stack[i] =
		param.x11_server_number + i;
	/*
	 * If x11_decnet_node is non-null, set flag to use alternate
	 * forwarding for X11 support instead of TCP/IP.
	 */
	pf.decnet_x11 = 0;
	if ( param.x11_decnet_node ) 
	    if ( param.x11_decnet_node->value[0] ) pf.decnet_x11 = 1;
    }
    /*
     * See if port forwarding map table is defined and enabled for any
     * target ports.
     */
    fwd_map_enable = (int *) 0;
    if ( param.fwd_map_table[0] && param.fwd_map_enable->value[0] ) {
	int status, attr, protection, mode, size, i;
	struct parameter_list_elem *elem;
	/*
	 * Delete existing logical name table and create new one.
	 */
	mode = 2;		/* supervisor mode */
	fwd_map_table.dsc$w_length = tu_strlen ( param.fwd_map_table );
	fwd_map_table.dsc$a_pointer = param.fwd_map_table;
	tu_strupcase ( param.fwd_map_table, param.fwd_map_table );
	status = SYS$DELLNM ( &fwd_map_table, 0, &mode );
	attr = LNM$M_CREATE_IF;
	protection = 0x0f000;	/* no world access */
	status = SYS$CRELNT ( &attr, 0, 0, 0, &protection, &fwd_map_table,
		&system_directory, &mode );
	/*
	 * Convert port enable list to integer array.
	 */
	for ( size = 0, elem = param.fwd_map_enable; elem;
		elem = elem->next ) size++;
	fwd_map_enable = (int *) malloc ( (size+1)*sizeof(int) );
	if ( fwd_map_enable ) {
	    for ( i=0, elem=param.fwd_map_enable; elem; elem=elem->next ) {
		fwd_map_enable[i] = atoi ( elem->value );
		if ( fwd_map_enable[i] ) i++;
	    }
	    fwd_map_enable[i] = 0;	/* mark end of list */
	}
    }
}
static pf_ctlblk allocate_ctl()
{
    pf_ctlblk ctl;

    pthread_mutex_lock ( &pf.lock );
    ctl = pf.free_ctl;
    if ( pf.free_ctl ) pf.free_ctl = ctl->next; 
    if ( !ctl ) {
        pthread_mutex_unlock ( &pf.lock );
	ctl = (pf_ctlblk) malloc (sizeof(struct pf_control_block) );
	pthread_mutex_lock ( &pf.lock );
	if ( ctl ) {
	    --pf.ctl_index;
	    ctl->index = pf.ctl_index;
	}
    }
    if ( ctl ) {
	/* Initialize the allocated structure */
	int i;
	ctl->status = 0;
	ctl->ref_count = 1;
	ctl->x11_server_number = 0;
        ctl->pipe_in = ctl->pipe_out = (cport_isb) 0;
	ctl->mask_high = 0;
	for ( i = 0; i < 256; i++ ) ctl->mask[i] = 0;
	ctl->high_chan = 0;
	ctl->spoof_packet = (char *) 0;
	GET_SPECIFIC(pf.creator_key,ctl->next);	/* make linked list */
	pthread_setspecific ( pf.creator_key, ctl );
    }
    pthread_mutex_unlock ( &pf.lock );
    return ctl;
}
static int deallocate_ctl ( pf_ctlblk ctl )
{
    pthread_mutex_lock ( &pf.lock );
    --ctl->ref_count;
    if ( ctl->ref_count <= 0 ) {
        ctl->next = pf.free_ctl;
        pf.free_ctl = ctl;
        pthread_mutex_unlock ( &pf.lock );
	return 1;
    } else {
        pthread_mutex_unlock ( &pf.lock );
    }
    return 0;
}
/* Rundown thread.
 */
static void pf_crtr_rundown(void *ctx) 
{
    pf_ctlblk ctl, next_ctl;
    int ref_count, is_creator;
    /*
     * Scan list adn rundown each open forwarder.
     */
    pthread_mutex_lock ( &pf.lock );
    for ( ctl = (pf_ctlblk) ctx; ctl; ctl = next_ctl ) {
	next_ctl = ctl->next;
	if ( ctl->ref_count > 1 )  {
	    --ctl->ref_count;
	    /* if ( ctl->pipe_in ) cport_cancel ( ctl->pipe_in );
	    if ( ctl->pipe_out ) cport_cancel ( ctl->pipe_out ); */
	} else {
	    ctl->ref_count = 0;
	    ctl->next = pf.free_ctl;	/* return block to free list */
	    pf.free_ctl = ctl;
	}
        tm_unlock_destructor ( ctl, ctl->dstr_hndl );
    }
    pthread_mutex_unlock ( &pf.lock );
}

static void pf_fwdr_rundown(void *ctx)
{
    pf_ctlblk ctl;
    pf_chan tcp;
    int i;

    ctl = (pf_ctlblk) ctx;
    /*
     * Rundown TCP stuff.
     */
    for ( tcp = ctl->tcp, i = 1; i <= ctl->high_chan; i++ ) {
	switch ( tcp[i].type ) {
	    case 1:			/* listener streams */
		if ( tcp[i].in ) {
		    cport_cancel ( tcp[i].in );
		    cport_deassign ( tcp[i].in ); 
		    tcp[i].in = (cport_isb) 0;
		}
		break;
	    case 2:			/* connections to port */
		/* if ( tcp[i].in ) {
		    cport_cancel ( tcp[i].in );
		    cport_deassign ( tcp[i].in ); tcp[i].in = (cport_isb) 0;
		}
		if ( tcp[i].out ) {
		    cport_cancel ( tcp[i].out );
		    cport_deassign ( tcp[i].out ); tcp[i].out = (cport_isb) 0;
		} */
                if ( fwd_map_enable && (tcp[i].port_num > 0) ) {
                    struct pf_map_arg map_arg;
                    evr_event ( EVR_EXCEPTION, 
                        "Dangling port connection, port %d", tcp[i].port_num);
                    map_arg.tcp = &tcp[i];
                    map_arg.username = ctl->username;
                    map_arg.accpornam = "";
                    map_arg.target_port = 0;
                    pf_map_socket ( 0, tcp[i].port_num, &map_arg );
                }
 		break;
	    default:
		break;			/* free(0) and pipe(3) are ignored */
        }
    }
    /*
     * Return X11 server number to pool.
     */
    pthread_mutex_lock ( &pf.lock );
    if ( ctl->x11_server_number > 0 && !pf.decnet_x11 ) {
	pf.x11_server_stack[--pf.x11_server_top] = ctl->x11_server_number;
    }
    /*
     * rundown pipe streams and deallocate control block, hold lock while 
     * doing so to synchronize deassigns and deallocate with cancels done by 
     * creator on its rundown.
     */
    if ( ctl->pipe_in ) cport_deassign ( ctl->pipe_in );
    if ( ctl->pipe_out ) cport_deassign ( ctl->pipe_out );
    ctl->pipe_in = ctl->pipe_out = (cport_isb) 0;
    if ( ctl->port ) cport_destroy_port ( ctl->port );
    ctl->port = (cport_port) 0;
    --ctl->ref_count;
    if ( ctl->ref_count <= 0 ) {
	ctl->next = pf.free_ctl;		/* put on free list */
	pf.free_ctl = ctl;
    }
    pthread_mutex_unlock ( &pf.lock );
}
/***************************************************************************/
/* Socket map routine is upcall from open_tcp_socket that globally associates
 * a driver-allocated port number with the authenticated username of the
 * socket's 'owner'.  Code: 1-add mapping, 0-delete mapping.
 */
int pf_map_socket ( int code, int port_num, void *map_arg_vp )
{
    struct pf_map_arg *map_arg;
    struct dsc$descriptor_s logical_name;
    struct { short int length, code; char *buffer; int *retlen; } item[6];
    char name[32];
    int i, status;
    map_arg = (struct pf_map_arg *) map_arg_vp;

    if ( !fwd_map_enable ) return 1;	/* not enabled */
    if ( code == 1 ) {
        /*
         * see if target port is on enable list.
         */
    	for ( i = 0; fwd_map_enable[i]; i++ )
	    if ( fwd_map_enable[i] == map_arg->target_port ) break;
	if ( !fwd_map_enable[i] ) return 1;	/* not on enabled list */
    }
    /*
     * construct logical name equal to local port.
     */
    tu_strint ( port_num, name );
    logical_name = fwd_map_table;
    logical_name.dsc$w_length = tu_strlen ( name );
    logical_name.dsc$a_pointer = name;
    if ( code == 1 ) {
	/* Create logical name, translation is username. */
	item[0].length = tu_strlen ( map_arg->username );
	item[0].code = LNM$_STRING;
	item[0].buffer = map_arg->username;
	item[0].retlen = 0;
	item[1].length = item[1].code = 0;	/* terminate item list */
	if ( map_arg->accpornam ) {
	    /* Check length of accpornam, don't add if zero */
	    item[1].length = tu_strlen ( map_arg->accpornam );
	    item[1].code = (item[1].length>0) ? LNM$_STRING : 0;
	    item[1].buffer = map_arg->accpornam;
	    item[1].retlen = 0;
	    item[2].length = item[2].code = 0;	/* new EOL */
	}
	status = SYS$CRELNM ( 0, &fwd_map_table, &logical_name, 0, item );
	map_arg->tcp->port_num = port_num;
    } else {
	/* delete logical name */
        status = SYS$DELLNM ( &fwd_map_table, &logical_name, 0 );
	map_arg->tcp->port_num = 0;
    }
    return status;
}
/***************************************************************************/
/* Global routine to initialize pf data structures.
 */
void pf_initialize ( )
{
    pthread_once ( &setup, pf_once );
}
/***************************************************************************/
int pf_create_forwarder ( int protocol_flags,
	void *pipe_ctx,
	char *username,		/* Needed for DECnet-based X11 forwarding */
	char *accpornam,	/* Remote host and port */
	char **buffer, 		/* output buffer (allocate by this function ) */
	int *bufsize )		/* size of buffer allocate */
{
    pf_ctlblk ctl;
    int status;
    pthread_t tid;
    pf_chan dummy;

    if ( param.x11_servers <= 0 ) return 0;	/* not enabled */
    pthread_once ( &setup, pf_once );
    /*
     * Allocate control block and copy caller's parameters.  accpornam
     * is truncated to 63 chars.
     */
    ctl = allocate_ctl();
    ctl->pipe_ctx = pipe_ctx;
    ctl->protocol_flags = protocol_flags;
    ctl->username = username;
    tu_strnzcpy ( ctl->accpornam, accpornam, sizeof(ctl->accpornam)-1 );
    /*
     * Allocate buffer that SSH thread will use to receive our output.
     */
    ctl->dstr_hndl = tm_lock_destructor ( ctl );
    *bufsize = sizeof(dummy->packet);
    *buffer = malloc ( *bufsize + 1000 );
    /*
     * create the forwarder thread.
     */
#ifdef PTHREAD_USE_D4
    status = pthread_create ( &ctl->thread_id, pf.ctl_attr,
		(pthread_startroutine_t) pf_forwarder_shell, ctl );
#else
    status = pthread_create ( &ctl->thread_id, &pf.ctl_attr,
		(startroutine_t) pf_forwarder_shell, ctl );
#endif
    evr_event ( EVR_DEBUG, "Create forwarder thread status: %d, flags: %d", 
		status, protocol_flags );
    if ( status != 0 ) {
	evr_event ( EVR_EXCEPTION, "Client thread create failure");
	deallocate_ctl ( ctl );
	return 0;
    }
    /*
     * Wait for forwarder thread to initialize.
     */
    pthread_mutex_lock ( &pf.lock );
    while ( ctl->status == 0 ) pthread_cond_wait ( &pf.ready, &pf.lock );
    status = ctl->status;
    pthread_mutex_unlock ( &pf.lock );
    return status;
}
/****************************************************************************/
/* Thread start routine for port forwarder thread.
 */
void *pf_forwarder_shell ( pf_ctlblk ctl )
{
    int status;
#ifdef CHECK_STACK
    char *pptr;
    int dcount;
    pptr = poison_stack(50);
#endif
    /*
     * Ensure deallocation on rundown.
     */
    evr_open_report ( &ctl->index );
    pthread_mutex_lock ( &pf.lock );
    ctl->ref_count++;
    pthread_mutex_unlock ( &pf.lock );
    evr_event ( EVR_DEBUG, "Forwarder thread started" );
    /*
     * Create completion port.
     */
    ctl->port = cport_create_port ( 0 );
    if ( ctl->port ) {
	/*
         * Assign streams to pipe context.
         */
	ctl->pipe_in = cport_assign_stream ( ctl->port, &cportpipe_driver, 
		ctl->pipe_ctx, 1 );
	ctl->pipe_out = cport_assign_stream ( ctl->port, &cportpipe_driver, 
		ctl->pipe_ctx, 0 );
	status = 1;
    } else {
	status = 44;
    }
    /*
     * Allocate channel table and make channel 0 the client pipe.
     */
    ctl->tcp = (pf_chan) tm_malloc ( param.x11_table_size *
	sizeof(struct pf_tcp_channel) + 1000);
    if ( ctl->tcp ) {
	ctl->tcp[0].flink = ctl->tcp[0].blink = (pf_chan) 0;
	ctl->tcp[0].type = 3;
	ctl->tcp[0].index = 0;
	ctl->tcp[0].port_num = 0;
	ctl->tcp[0].ctx = ctl->pipe_ctx;
	ctl->tcp[0].in = ctl->pipe_in;
	ctl->tcp[0].out = ctl->pipe_out;
	ctl->tcp[0].packet.channel = 0;
    }
    /*
     * Allocate initial buffer into which client thread writes messages.
     */
    ctl->in_bufsize = 4096;
    ctl->in_buffer = tm_malloc ( ctl->in_bufsize +2000);
    if ( !ctl->in_buffer || !ctl->tcp ) {
	pf_fwdr_rundown ( ctl );
	return (void *) 0;
    }
    /*
     * Notify creating thread that pipes are ready.
     */
    pthread_mutex_lock ( &pf.lock );
    ctl->status = status;
    pthread_cond_broadcast ( &pf.ready );
    pthread_mutex_unlock ( &pf.lock );
    /*
     * call dispatcher, trapping errors.
     */
#ifndef DONT_TRY
    TRY {
#endif
	status = pf_main ( ctl );
#ifndef DONT_TRY
    }
#ifdef PTHREAD_USE_D4
    CATCH(cma_e_exit_thread) { /* Normal event, mst_exit called */ }
    CATCH_ALL {
	evr_event ( EVR_EXCEPTION, 
		"Forwarder shell caught exception in at %t", 0 );
	exc_report ( THIS_CATCH );
    }
#else
    CATCH(pthread_exit_e) { /* Normal event, mst_exit called */ }
    CATCH_ALL {
	evr_event ( EVR_EXCEPTION, 
		"Forwarder shell caught exception at %t", 0 );
    }
#endif
    ENDTRY
#endif
    /*
     * Cleanup, rundown connection.
     */
    evr_event ( EVR_EXIT, "Forwarder exit with status: %d at %t", status, 0 );
    pf_fwdr_rundown ( ctl );
#ifdef CHECK_STACK
    dcount = dirty_poison(pptr,50);
    evr_event ( EVR_DEBUG, "Dirty stack: %dK\n", dcount);
#endif
    evr_close_report ( );
    return (void *) 0;
}
/*****************************************************************************/
/* Main routine for handling connections.
 */
static int pf_main ( pf_ctlblk ctl )
{
    int status, xfer, i;
    unsigned short *iosb;
    cport_isb isb;
    pf_chan tcp;
    /*
     * Initialize queues and channel table.
     */
    ctl->out_queue_head = (pf_chan) 0;
    ctl->out_queue_tail = (pf_chan) 0;
    ctl->tcp[0].in->user_val = 0;		/* table index */
    ctl->tcp[0].out->user_val = 0;
    /*
     * Initiate read on input pipe.
     */
    status = cport_start_io ( ctl->pipe_in, CPORT_READ, ctl->in_buffer,
	ctl->in_bufsize );
    ctl->in_idle = 0;
    /*
     * Main loop.
     */
    while ( status&1 ) {
	/*
	 * Wait for next I/O completion.
	 */
	isb = cport_next_completion ( ctl->port );
	if ( !isb ) break;
	iosb = (unsigned short *) isb->iosb;
	if ( isb == ctl->pipe_in ) {
/*printf("pf thread received message, status: %d %d %d %d\n", iosb[0],
iosb[1], iosb[2], iosb[3] ); */
	    /*
	     * Check completion status.
	     */
	    ctl->in_idle = 0;
	    if ( iosb[0] == SS$_DATAOVERUN ) {
		/*
		 * Buffer was too small, expand it and redo read.
		 */
		if ( ctl->in_bufsize >= (iosb[1]+4096) ) {
		    evr_event ( EVR_EXCEPTION, "Invalid dataoverun condition");
		    status = SS$_DATAOVERUN;
		} else 	status = 1;
		ctl->in_bufsize = iosb[1] + 4096;
		ctl->in_buffer = tm_realloc ( ctl->in_buffer, ctl->in_bufsize);
		ctl->in_idle = 1;
	    } else if ( iosb[0]&1 ) {
		/*
		 * Message OK, dispatch.
		 */
		status = pf_dispatch_pipe_in ( ctl );
	    } else {
		evr_event ( EVR_EXCEPTION, "Error reading pipe: %d", iosb[0]);
		status = iosb[0];
	    }

	} else if ( isb == ctl->pipe_out ) {
	    /*
	     * Dequeue channel from the output queue.
	     */
	    pf_chan chan;
	    chan = ctl->out_queue_head;
	    if ( !chan ) continue;
	    ctl->out_queue_head = chan->flink;
	    if ( ctl->out_queue_head ) ctl->out_queue_head->blink = (pf_chan) 0;
	    status = pf_dispatch_pipe_out ( ctl, chan );
	    /*
	     * Start next write.
	     */
	    if ( ctl->out_queue_head ) {
		chan = ctl->out_queue_head;
		cport_pipe_set_send_type ( ctl->pipe_out, chan->msg_type );
		cport_start_io ( ctl->pipe_out, CPORT_WRITE,
		    &chan->packet, chan->out_size );
	    }
	} else if ( isb->user_val <= ctl->high_chan ) {
	    /*
	     * Completion is on one of dynamically allocated blocks.
	     */
	    tcp = &ctl->tcp[isb->user_val];
	    if ( tcp->type == 0 ) continue;
	    if ( isb == tcp->in ) status = pf_dispatch_tcp_in ( ctl, tcp );
	    else if ( isb == tcp->out ) status = pf_dispatch_tcp_out ( ctl, tcp );
            else {
		evr_event ( EVR_EXCEPTION, "User_val mistmatch on tcpchannel");
	    }
	} else {
	    /*
	     * Bugcheck.
	     */
	    evr_event ( EVR_EXCEPTION, 
		"forwarder had I/O completion on uknown ISB" );
	    status = SS$_ABORT;
	    break;
	}
        /*
         * Start new read from client pipe.
         */
        if ( (status&1) == 1 && ctl->in_idle ) {
	    ctl->in_idle = 0;
	    status = cport_start_io ( ctl->pipe_in, CPORT_READ,
		ctl->in_buffer, ctl->in_bufsize );
        }
    }
    return status;
}
