/*
 * This module handles the connection state for a port forwarder thread.
 * The four main functions work in a coordinated fashion:
 *
 *   pf_dispatch_pipe_in is the initial point of processing for SSH
 *	messages sent by the remote SSH client.  The message to be
 *      processed in always in ctl->in_buffer.
 *
 *   pf_dispatch_pipe_out performs the post processing after an message
 *      is sent to the remote SSH client.  The message sent is always
 *      in the ctl->tcp[i].packet structure, where i is the channel
 *      number.  If the write succeeded, a new read on the channel's
 *      input stream is initiated.
 *
 *   pf_dispatch_tcp_in handles I/O completed on a channel's 'in' stream.
 *      Action to take depends upon the channel type (listener or connection),
 *      either queueing an SSH_SMSG_X11_OPEN_CHANNEL or SSH_MSG_CHANNEL_DATA
 *      message to the client.
 *
 *   pf_dispatch_tcp_out handles I/O completed on a channel's 'out' stream.
 *      The written data in always in ctl->in_buffer, being relayed from
 *      a SSH_MSG_CHANNEL_DATA message recieved from the SSH client.  Verify
 *      the write completed and mark in_buffer idle so the next message
 *      can be read from the client.
 *
 * Author:	David Jones
 * Date:	16-SEP-1998
 * Revsied:	25-SEP-1998		Deallocate to stack bottom on failure.
 * Revised:	13-OCT-1998		Fix allocation bug.
 * Revised:	27-OCT-1998		Add port_open support.
 * Revsied:	1-NOV-1998		Add port_fwd_request support.
 * Revised:	3-NOV-1998		Fix failure detection in dspt_pipe_out.
 * Revised:	4-NOV-1998		Fix accvio in dispatch_pipe_out.
 * Revised:	29-JAN-1999		New interface to open_tcp_socket.
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>			/* block copies */

#include "parameters.h"
#include "pthread_1c_np.h"
#include "tutil.h"
#include "tmemory.h"
#include "event_report.h"
#include "completion_port.h"
#include "cport_pipe.h"
#include "cport_sshpad.h"
#include "cport_sshmsg.h"
#include "cport_ucx.h"
#include "cport_x11.h"

#define X11_SERVER_PORT_BASE 6000
#include "port_forwarder.h"
/****************************************************************************/
/* Extract client channel number from packet in ctl->in_buffer and return
 * corresponding tcp channel.
 */
static pf_chan client_match ( pf_ctlblk ctl )
{
    pf_chan tcp;
    int i, number;
    unsigned char *bf;

    if ( ctl->pipe_in->default_iosb[1] < 4 ) {
	evr_event(EVR_EXCEPTION,"match client has runt packet: %d %d",
	ctl->pipe_in->default_iosb[0], ctl->pipe_in->default_iosb[1] );
	return (pf_chan) 0;
    }
    bf = (unsigned char *) ctl->in_buffer;
    number = (bf[3] << 24) | (bf[2]<<16) | (bf[1]<<8) | bf[0];
    if ( number > 0 && number <= ctl->high_chan ) {
	if ( ctl->tcp[number].type == 2 ) return &ctl->tcp[number];
    }
    return (pf_chan) 0;			/* lookup failure */
}
/*
 * Locate server channel in message and set packet.chan, return value
 * is table postion (must be > 1) or 0 if failure. Note: 'rfc' states
 * client sends it's local channel number first, but this is contradicted
 * by the code.
 */
static int map_client_channel ( pf_ctlblk ctl )
{
    int i, c_number, s_number;
    unsigned char *bf;
    if ( ctl->pipe_in->default_iosb[1] < 8 ) return 0;	/* invalid packet */
    bf = (unsigned char *) ctl->in_buffer;
    s_number = (bf[3] << 24) | (bf[2]<<16) | (bf[1]<<8) | bf[0];
    c_number = (bf[7] << 24) | (bf[6]<<16) | (bf[5]<<8) | bf[4];
    /*
     */
    if ( (s_number < 1) || (s_number > ctl->high_chan) ) return 0;
    if ( ctl->tcp[s_number].type != 2 ) return 0;
    /* printf("setting  tcp[%d].remote_chan to %d\n", s_number, c_number ); */
    ctl->tcp[s_number].remote_channel = c_number;
    ctl->tcp[s_number].packet.channel = c_number;
    return s_number;
}
/*****************************************************************************/
/* Place outbound tcp channel in output queue, starting new write if
 * none in progress.  dsize is size of data 1 channel number.
 */
static int queue_to_client ( pf_ctlblk ctl, int channel, 
	int msg_type, int dsize )
{
    int status;
    pf_chan chan;
    chan = &ctl->tcp[channel];
    chan->flink = (pf_chan) 0;	/* end of line, not foward */
    chan->out_size = dsize + sizeof(chan->packet.channel);
    chan->packet.channel = chan->remote_channel;
    chan->msg_type = msg_type;
    if ( ctl->out_queue_head ) {
	/* Queue not empty */
	chan->blink = ctl->out_queue_tail;
	ctl->out_queue_tail->flink = chan;
	ctl->out_queue_tail = chan;
	status = 1;
    } else {
	/*
	 * Queue empty, place at head and start writing data.
	 */
	chan->blink = (pf_chan) 0;
	ctl->out_queue_head = chan;
        ctl->out_queue_tail = chan;
	cport_pipe_set_send_type ( ctl->pipe_out, chan->msg_type );
	status = cport_start_io ( ctl->pipe_out, CPORT_WRITE,
		&chan->packet, chan->out_size );
    }
    return status;
}

static void generate_spoof_packet ( char *msg, int msglen,
	char **spoof_packet, int *spoof_packet_size )
{
    int prot_len, auth_len, i, j;
    unsigned char *umsg;
    struct init_packet {
	char bo, pad;
	unsigned short major_ver, minor_ver;
	unsigned short prot_len, auth_len;
	unsigned short pad2;
	char var[1];
    } *packet;
    /*
     * interpret SSH packet fields, big-endian, to get sizes of the
     * subfields.
     */
    umsg = (unsigned char *) msg;
    if ( msglen < 8 ) return;			/* runt message */
    prot_len = umsg[3];
    prot_len = prot_len + (umsg[0] * 0x01000000) + (umsg[1]*0x10000) +
	(umsg[2]*0x0100);
    if ( msglen < 4 + prot_len ) return;

    auth_len = umsg[prot_len+7];
    auth_len = auth_len + (umsg[prot_len+4] * 0x01000000) + 
	(umsg[prot_len+5]*0x10000) + (umsg[prot_len+6]*0x0100);
    if ( msglen < (prot_len+auth_len+8) ) return;
    /*
     * Make x11 packet: 12 byte header, followed by 2 longword-aligned strings.
     */
    packet = (struct init_packet *) tm_malloc ( sizeof(struct init_packet) +
	prot_len + (auth_len/2) + 8 );
    if ( packet ) {
	char *cp;
	/* header fields */
	packet->bo = 'l';
	packet->pad = 'B';
	packet->major_ver = 11;
	packet->minor_ver = 0;
	packet->prot_len = prot_len;
	packet->auth_len = auth_len/2;
	packet->pad2 = 0;
	/* protocol name string */
	memcpy ( packet->var, &umsg[4], prot_len );
	i = (prot_len + 3) & (~3);	/* align to next higher longword */
	/* authenticate key, convert packet string from hex to binary */
	for ( cp = &msg[prot_len+8], j = 0; j < auth_len; j+= 2 ) {
	    int nibble1, nibble2, octet;
	    char c;
	    nibble1 = nibble2 = 0;
	    c = *cp;
	    if ( (c >= '0') && (c <= '9') ) nibble1 = c - 48;
	    else if ( (c >= 'a') && (c <= 'f') ) nibble1 = (c - 97) + 10;
	    else if ( (c >= 'A') && (c <= 'F') ) nibble1 = (c - 65) + 10;
	    else nibble1 = 0;
	    cp++;
	    c = *cp;
	    if ( (c >= '0') && (c <= '9') ) nibble2 = c - 48;
	    else if ( (c >= 'a') && (c <= 'f') ) nibble2 = (c - 97)+10;
	    else if ( (c >= 'A') && (c <= 'F') ) nibble2 = (c - 65) +10;
	    else nibble2 = 0;
	    cp++;

	    octet = (nibble1*16) | (nibble2&15);
	    packet->var[i] = octet;

	    i++;
	}
	/* memcpy ( &packet->var[i], &umsg[prot_len+8], auth_len ); */

        *spoof_packet_size = 12 + ((i+3)&(~3));
    }
    *spoof_packet = (char *) packet;
}
/*****************************************************************************/
/* Open channel to host and port specified in SSH_MSG_PORT_OPEN message
 * stored in ctl->in_buffer.  If successful, set tcp->type to 2 (connection),
 * otherwise mark free (0).    ctl->in_buffer is destroyed.
 */
static int open_port ( pf_ctlblk ctl, int in_buf_len, pf_chan tcp )
{
    int host_len, port_num, i, status;
    unsigned char *bf;
    struct pf_map_arg map_arg;
    char errmsg[256];
    /*
     * Extract the host name and port.  Format of received message is:
     *    int local_chan, string hostname, int port_num [, string orig_info]
     *
     * Terminate host string with NUL, which will clobber port number.
     */
    memcpy ( &tcp->remote_channel, ctl->in_buffer, 4 );
    bf = (unsigned char *) &ctl->in_buffer[4];
    host_len = bf[0];
    host_len = (host_len<<24) | (bf[1]<<16) | (bf[2]<<8) | (bf[3]);
    if ( (host_len < 0) || (host_len > (in_buf_len+12)) ) return 20;

    i = host_len + 4;           /* skip string count, host */
    port_num = bf[i];		/* save bf[i]; */
    bf[i] = '\0';		/* terminate host string */
    port_num = (port_num<<24) | (bf[i+1]<<16) | (bf[i+2]<<8) | bf[i+3];
    /*
     * Attempt to open channel to port.  If open fails, return success but
     * with type 0.
     */
    errmsg[0] = '\0';
    map_arg.tcp = tcp;
    map_arg.username = ctl->username;
    map_arg.accpornam = ctl->accpornam;
    map_arg.target_port = port_num;
    status = cportucx_open_tcp_socket ( &ctl->in_buffer[8], port_num,
        0, (char **) 0, &tcp->ctx, pf_map_socket, &map_arg, errmsg );
    evr_event ( EVR_DEBUG, 
	"TCP open of '%s:%d' status: %d, rem/lcl chan: %d/%d",
	&ctl->in_buffer[8], port_num, status, tcp->remote_channel, tcp->index);
    if ( (status&1) == 0 ) return 1;       /* open failed. */
    /*
     * Assign streams, errors in assign are fatal.
     */
    tcp->in = cport_assign_stream ( ctl->port, &cportucx_driver, tcp->ctx, 1 );
    tcp->out = cport_assign_stream ( ctl->port, &cportucx_driver, tcp->ctx, 0);
    if ( tcp->in && tcp->out ) {
        tcp->in->user_val = tcp->index;
        tcp->out->user_val = tcp->index;
    } else {
        evr_event ( EVR_EXCEPTION, "Assign stream failure on OPEN_PORT" );
        return 44;
    }
    /*
     * Connection open, save remote channel number and mark channel active.
     */
    tcp->type = 2;
    tcp->init_packet_size = 0;          /* not initial packet */
    return status;
}
/*
 * Initialize the mask of message types we are interested in and send to
 * client threads.  Message type for message is x11 server number we
 * allocated (or zero if no X11 server enabled).
 */
static void send_msg_mask ( pf_ctlblk ctl )
{
    int xfer;
    /*
     * Set mask entries to true for the message types we want routed to
     * this thread.
     */
    ctl->mask[SSH_MSG_CHANNEL_OPEN_CONFIRMATION] = 1;
    ctl->mask[SSH_MSG_CHANNEL_OPEN_FAILURE] = 1;
    ctl->mask[SSH_MSG_CHANNEL_DATA] = 1;
    ctl->mask[SSH_MSG_CHANNEL_CLOSE] = 1;
    ctl->mask[SSH_MSG_CHANNEL_CLOSE_CONFIRMATION] = 1;
    ctl->mask[SSH_MSG_PORT_OPEN] = 1;
    ctl->mask_high = 34;

    cport_pipe_set_send_type ( ctl->pipe_out, ctl->x11_server_number );
    cport_do_io ( ctl->pipe_out, CPORT_WRITE, ctl->mask,
    (ctl->mask_high+1)*sizeof(ctl->mask[0]), &xfer );
}
/*
 * Handle setup of listen ports.
 */
static int port_forward_setup ( pf_ctlblk ctl, 
	int is_x11_port, int in_buf_len, pf_chan tcp )
{
    int status, retval, hlen;
    unsigned char *bf;
    char errmsg[256];
    cport_stream_handler *driver;
    /*
     * Decode client message in ctl->in_buffer to extract local port,
     * forward host, and forward port.
     */
    if ( is_x11_port ) {	/* X11 server */
	tcp->port_num = ctl->x11_server_number + X11_SERVER_PORT_BASE;
	tcp->forward_port = 0;
	tcp->forward_host = (char *) 0;
    } else {
	/*
	 * In_buffer contains local-port, hostname, remote-port.
	 */
	if ( in_buf_len < 13 ) return 20;	/* message not big enough */
        bf = (unsigned char *) ctl->in_buffer;
	tcp->port_num = (bf[0]<<24) | (bf[1]<<16) | (bf[2]<<8) | bf[3];
	hlen = (bf[4]<<24) | (bf[5]<<16) | (bf[6]<<8) | bf[7];
	if ( hlen < 0 ) return 20;
	if ( hlen+12 > in_buf_len ) return 20;	/* invalid length */
	bf = &bf[hlen + 8];
	tcp->forward_port = (bf[0]<<24) | (bf[1]<<16) | (bf[2]<<8) | bf[3];
	tcp->forward_host = tm_malloc ( hlen + 1 );
	if ( !tcp->forward_host ) return 44;
	tu_strnzcpy ( tcp->forward_host, &ctl->in_buffer[8], hlen );
	/*
	 * Validate the requested host and port.
	 */
	if (tcp->forward_port <= 0 || tcp->forward_port > 0x0ffff) return 20;
	if ( tcp->port_num <= 1024 || tcp->port_num > 0x0ffff ) return 20;
    }
    /*
     * Begin listening on requested channel.
     */
    tcp->type = 1;		/* mark as listen channel. */
    if ( is_x11_port && pf.decnet_x11 ) {
	driver = &cportx11_driver;
        retval = cportx11_create_x11_listener ( 
		ctl->username,		/* Local username */
		&ctl->x11_server_number, /* X11 server number */
		&tcp->ctx,		/* returned handle */
		errmsg );		/* diagnostic message */
        evr_event ( EVR_DEBUG, 
		"Created listen channel for tcp[%d] X11, user '%s', sts: %d",
		tcp->index, ctl->username, retval );
    } else {
	driver = &cportucx_driver;
        retval = cportucx_create_tcp_listener ( 
		tcp->port_num,		/* TCP port number */
		5,			/* backlog */
		0, (char **) 0,		/* socket options */
		&tcp->ctx,		/* returned handle */
		errmsg );		/* diagnostic message */
        evr_event ( EVR_DEBUG, 
		"Created listen channel for tcp[%d] port: %d, sts: %d",
		tcp->index, tcp->port_num, retval );
    }
    if ( retval&1 ) {
	tcp->in = cport_assign_stream ( ctl->port, driver, tcp->ctx, 0 );
	if ( tcp->in ) tcp->in->user_val = tcp->index;
	else {
	    retval = 44;
	    evr_event ( EVR_EXCEPTION, 
		    "Listener asign failure (%d), port: %d", retval,
		    tcp->port_num );
	}
    } else {
	tcp->type = 0;
	evr_event ( EVR_EXCEPTION, 
		"Listener create failure (%d), port: %d", retval,
		tcp->port_num );
    }
    return retval;
}
/*****************************************************************************/
/*
 * Pipe input dispatcher is called when the read on ctl->pipe_in completes
 * successfully.
 */
int pf_dispatch_pipe_in ( pf_ctlblk ctl )
{
    unsigned short *iosb;
    int type, i, status, number, client_channel, remaining;
    int retval, xfer;
    unsigned char *bf;
    pf_chan tcp;
    char errmsg[256];
    /*
     * The iosb in the pipe_in ISB contains the length and mesage type.
     */
    iosb = (unsigned short *) ctl->pipe_in->iosb;
    type = iosb[3];
    status = iosb[0];
#ifdef DEBUG
bf = (unsigned char *) ctl->in_buffer;
printf("message type received: %d, status: %d, len: %d, chan: %02x%02x%02x%02x\n", type, 
status,iosb[1], bf[0], bf[1], bf[2], bf[3] );
#endif
    if ( (status&1) == 0 ) {
        /* Error in pipe in */
    } else if ( type == SSH_MSG_CHANNEL_DATA ) {/* optimize for common case */
	/*
	 * Message is channel data to be sent to corresponding out stream
	 * in channel table.  We use the same buffer, the next read on
	 * pipe_in will be started after write completion.
	 */
	tcp = client_match ( ctl );
	if ( !tcp ) {		/* lookup failure, abort */
	    evr_event ( EVR_EXCEPTION, "Data Failed to match client channel: %d",
		ctl->in_buffer[0] );
	    ctl->in_idle = 1;
	    return 44;
	}
	ctl->in_pos = 8;		/* skip channel number and length */
	bf = (unsigned char *) ctl->in_buffer;
	i = (bf[4]<<24) | (bf[5]<<16) | (bf[6]<<8) | bf[7];
	if ( (i+8) != iosb[1] ) {
	   evr_event ( EVR_EXCEPTION, "Wrong length in received packet: %d",
		i );
	}
	ctl->in_endpos = iosb[1];	
	remaining = ctl->in_endpos - ctl->in_pos;
	if ( remaining > 4096 ) remaining = 4096;
	status = cport_start_io ( tcp->out, CPORT_WRITE,
	    &ctl->in_buffer[ctl->in_pos], remaining );
    }
    /*
     * Handle other message types via switch statement.
     */
    else switch ( type ) {
	case SSH_CMSG_EXEC_CMD:			/* signal start of inter. mode */
	    /*
	     * Start accept of port, return message type mask (controls
	     * which messages will be handed off to the port forwarder thread).
	     */
	    if ( ctl->mask_high == 0 ) send_msg_mask ( ctl );
	    ctl->in_idle = 1;
	    /*
	     * Begin accepting connections on listen ports.
	     */
	    for ( tcp = ctl->tcp, i = 1; i <= ctl->high_chan; i++ ) {
		if ( tcp[i].type == 1 ) {
	    	    if (ctl->tcp[i].in == (cport_isb) 0) evr_event ( 
			EVR_EXCEPTION,
			"ctl->tcp[%d].in is NULL(after exec_cmd", i );
		    status = cport_start_io ( tcp[i].in, CPORTUCX_ACCEPT,
			tcp[i].packet.data, 
			sizeof(struct cportucx_accept_result) );
		}
	    }
	    break;
	case SSH_MSG_CHANNEL_OPEN_CONFIRMATION:
	    /*
	     * client is confirming our request to open a channel,
	     * Match server channel and determine X11 or generic port forward.
	     * begin read of channel's input stream.
	     */
	    ctl->in_idle = 1;
	    i = map_client_channel ( ctl );
	    if ( i > 0 ) {
		ctl->tcp[i].init_packet_pos = 0;
		ctl->tcp[i].init_packet_size = 
			ctl->tcp[i].forward_port > 0 ? 0 : 12;
		status = cport_start_io ( ctl->tcp[i].in, CPORT_READ,
			ctl->tcp[i].packet.data,
			ctl->tcp[i].init_packet_size > 0 ? 12 : 
			sizeof(ctl->tcp[i].packet.data) );
	    } else {
		status = 44;		/* Bugcheck */
	    }
	    break;
	case SSH_MSG_CHANNEL_OPEN_FAILURE:
	    /*
	     * Client failed to connect to remote X server.  Rundown
	     * connection.
	     */
	    ctl->in_idle = 1;
	    tcp = client_match ( ctl );
	    if ( !tcp ) {		/* lookup failure, abort */
	        evr_event ( EVR_EXCEPTION, "open fail failed to match client channel: %d",
		ctl->in_buffer[0] );
	        return 44;
	    }
	    cport_deassign ( tcp->in );
	    cport_deassign ( tcp->out );
	    tcp->type = 0;
	    break;

	case SSH_MSG_CHANNEL_CLOSE:
	    ctl->in_idle = 1;
	    tcp = client_match ( ctl );
	    if ( tcp ) {
		if ( (tcp->type == 2) && (tcp->port_num > 0) ) {
		    struct pf_map_arg map_arg;
		    map_arg.tcp = tcp;
		    map_arg.username = ctl->username;
		    map_arg.accpornam = ctl->accpornam;
		    map_arg.target_port = 0;   /* unknown */
		    pf_map_socket ( 0, tcp->port_num, &map_arg );
		}
		cport_deassign ( tcp->in );
		cport_deassign ( tcp->out );
		tcp->type = 0;
	    } else evr_event ( EVR_EXCEPTION, 
		"Close failed to match client channel: %d", ctl->in_buffer[0]);
	    status = 1;
	    break;
	case SSH_MSG_CHANNEL_CLOSE_CONFIRMATION:
	    /*
	     * Client confirmed close of connection, rundown channel.
	     */
	    tcp = client_match ( ctl );
	    if ( !tcp ) {		/* lookup failure, abort */
	        evr_event ( EVR_EXCEPTION, 
			"Close confirm failed to match client channel: %d",
			ctl->in_buffer[0] );
	        ctl->in_idle = 1;
	        return 1;
	    }
	    if ( (tcp->type == 2) && (tcp->port_num > 0) ) {
		/*
		 * Remove entry from port map database.
		 */
		struct pf_map_arg map_arg;
		map_arg.tcp = tcp;
		map_arg.username = ctl->username;
		map_arg.accpornam = ctl->accpornam;
		map_arg.target_port = 0;   /* unknown */
		pf_map_socket ( 0, tcp->port_num, &map_arg );
	    }
	    cport_deassign ( tcp->in );
	    status = 1;
	    cport_deassign ( tcp->out );
	    tcp->type = 0;

	    ctl->in_idle = 1;		/* continue reading from pipe */
	    break;
	/*
         * Handle requests to do x11 forwarding.
	 */
	case SSH_CMSG_X11_FWD_WITH_AUTH_SPOOFING:
	    /*
	     * Auth with spoofing request has 2 or three arguments included
	     * in the message:
	     *   Authentication protocol.
	     *   hex authentication string.
	     *   screen number (in bit 1 in protocol flags set).
	     *
	     * Construct default spoof packet assuming client library
	     * is little-endian, then fall through to non-spoofed request
	     * processing.
	     */
	    generate_spoof_packet ( ctl->in_buffer, iosb[1],
		&ctl->spoof_packet, &ctl->spoof_packet_size );
	    
	case SSH_CMSG_X11_REQUEST_FORWARDING:
	    /*
	     * Allocate a channel slot and x11 server number.
	     */
	    ctl->in_idle = 1;	/* force new read */
	    retval = 1;
	    pthread_mutex_lock ( &pf.lock );		/* stack is shared */
	    for ( i = 1; i < param.x11_table_size; i++ ) {
		if ( i > ctl->high_chan ) break;
		if ( ctl->tcp[i].type == 0 ) break;
	    }
	    if ( i < param.x11_table_size &&
		pf.x11_server_top < param.x11_servers ) {
		/*
		 * Sucessfully allocated server number and channel slot.
		 * Initialize structures and start listening on TCP port.
		 */
		if ( i > ctl->high_chan ) ctl->high_chan = i;
		ctl->tcp[i].index = i;
		if ( pf.decnet_x11 ) {
		    ctl->x11_server_number = param.x11_server_number;
		} else {
		    ctl->x11_server_number = 
			pf.x11_server_stack[pf.x11_server_top++];
		}

		retval = port_forward_setup ( ctl, 1, iosb[1], &ctl->tcp[i] );

		if ( (retval&1) == 0 && !pf.decnet_x11 ) {
		    /*
		     * return failed port number to bottom of stack.
		     */
		    int j;
		    for ( j = pf.x11_server_top; j < param.x11_servers; j++ ) 
			pf.x11_server_stack[j-1] = pf.x11_server_stack[j];
		    pf.x11_server_stack[param.x11_servers-1] =
				ctl->x11_server_number;
		    --pf.x11_server_top;
		    ctl->x11_server_number = 0;
		}
		if ( (retval&1) == 0 ) ctl->x11_server_number = 0;
	    } else {
		evr_event ( EVR_EXCEPTION,  "No X11 server ports available" );
		retval = 44;
	    }
	    pthread_mutex_unlock ( &pf.lock );

	    status = cport_do_io ( ctl->pipe_out, CPORT_WRITE, &retval, 4, &xfer );
	    break;
	    /*
	     * Generic port forwarding.
	     */
	case SSH_CMSG_PORT_FORWARD_REQUEST:
	    /*
	     * Client wants the server to listen on a chosen TCP port and
	     * initiate a port open request for any connections that occur.
	     * This message only valid during the setup phase, allocate
	     * a channel a channel and begin listening.
	     */
	    ctl->in_idle = 1;		/* force new read */
	    retval = 1;
            for ( i = 1; i < param.x11_table_size; i++ ) {
                if ( i > ctl->high_chan ) break;
                if ( ctl->tcp[i].type == 0 ) break;
            }
            if ( i < param.x11_table_size ) {
		/*
		 * Found a free channel number, initialize for listens.
		 */
		if ( i > ctl->high_chan ) ctl->high_chan = i;
		ctl->tcp[i].index = i;
		retval = port_forward_setup ( ctl, 0, iosb[1], &ctl->tcp[i] );

	    } else {
		/*  Allocation failure */
		retval = 44;
	    }
	    status = cport_do_io ( ctl->pipe_out, CPORT_WRITE, &retval, 4, &xfer );
	    break;
        case SSH_MSG_PORT_OPEN:
            /*
             * Client wants us to open a TCP connection on a host/port
             * supplied by him,  send message mask if this is first call.
	     * Send result back as open_confirm or open_fialure using the
	     * buffer for channel zero.  Suspend reading from in_pipe until
	     * response completely sent (prevent more that 1 pending use of
	     * tcp[0]).
             */
	    if ( ctl->mask_high == 0 ) send_msg_mask ( ctl );
	    /*
	     * Allocate tcp channel.
	     */
            for ( i = 1; i < param.x11_table_size; i++ ) {
                if ( i > ctl->high_chan ) break;
                if ( ctl->tcp[i].type == 0 ) break;
            }
            if ( i < param.x11_table_size ) {
                /*
                 * Attempt port open and return status based upon whether
                 * tcp[i] was marked in use.
                 */
                if ( i > ctl->high_chan ) ctl->high_chan = i;
                ctl->tcp[i].index = i;
                status = open_port ( ctl, iosb[1], &ctl->tcp[i] );
                ctl->tcp[0].remote_channel = ctl->tcp[i].remote_channel;
                if ( ctl->tcp[i].type == 0 ) {  /* failure */
                    queue_to_client (ctl, 0, SSH_MSG_CHANNEL_OPEN_FAILURE, 0);
                } else {
                    /* memcpy ( &ctl->tcp[0].packet, ctl->in_buffer, 4 ); */
                    memcpy ( ctl->tcp[0].packet.data, &ctl->tcp[i].index, 4 );
                    queue_to_client (ctl, 0, SSH_MSG_CHANNEL_OPEN_CONFIRMATION,
                        4 );
                }
            } else {
                /*
                 * Allocation failure.
                 */
		memcpy ( &ctl->tcp[0].remote_channel, ctl->in_buffer, 4 );
		evr_event ( EVR_DEBUG, 
			"Allocation failure for port_open, remote chann: %d",
			ctl->tcp[0].remote_channel );
                queue_to_client ( ctl, 0, SSH_MSG_CHANNEL_OPEN_FAILURE, 0 );
            }
            break;

	default:
	    evr_event ( EVR_EXCEPTION, "Unknown message type: %d", type );
	    ctl->in_idle = 1;
	    break;
    }
    return status;
}
/**************************************************************************/
/* pipe out dispatch is called when I/O queued to ctl->pipe out completes.
 * The packet is dequed from ctl->out_queue_xxx prior to calling this
 * function.
 */
int pf_dispatch_pipe_out ( pf_ctlblk ctl, pf_chan tcp)
{
    int status, i;
    /*
     * Look at message type that was sent:
     */
    if ( tcp->msg_type == SSH_SMSG_X11_OPEN ) {
	/*
	 * Message sent, leave tcp channel in limbo until client sends
	 * open_confirm or open_failure on pipe_in.
	 */
	status = 1;
    } else if ( tcp->msg_type == SSH_MSG_PORT_OPEN ) {
	/*
	 * Message sent, leave tcp channel in limbo until client sends
	 * open_confirm or open_failure on pipe_in.
	 */
	status = 1;
    } else if ( tcp->msg_type == SSH_MSG_CHANNEL_DATA ) {
	/*
	 * Start new read on input if connection still active.
	 */
	if ( tcp->type == 2 ) status = cport_start_io ( tcp->in, CPORT_READ,
		tcp->packet.data, sizeof(tcp->packet.data) );
	else status = 44;
    } else if ( tcp->msg_type == SSH_MSG_CHANNEL_CLOSE ) {
	status = 1;
    } else if ( tcp->msg_type == SSH_MSG_CHANNEL_OPEN_CONFIRMATION ) {
        /*
         * PORT open request from client succeeded.  Initiate read on
         * the channel.
         */
        for ( i = 1; i <= ctl->high_chan; i++ ) {
            if ( (ctl->tcp[i].remote_channel == ctl->tcp[0].remote_channel)
                && (ctl->tcp[i].type == 2) ) {
		/*
		 * work around bug in secureCRT, if channel number is zero,
		 * assume client is SecureCRT and change to our channel.
		 */
                if ((param.port_forward&4) && (ctl->tcp[i].remote_channel==0)) 
			ctl->tcp[i].remote_channel = i;
                break;
            }
        }
        if ( (i <= ctl->high_chan) && (ctl->tcp[i].type == 2) ) {
            ctl->tcp[i].init_packet_pos = 0;
            ctl->tcp[i].init_packet_size = 0;
	    if (ctl->tcp[i].in == (cport_isb) 0 ) evr_event ( EVR_EXCEPTION,
		"ctl->tcp[%d].in is NULL(after open_confirm)",i);
            status = cport_start_io ( ctl->tcp[i].in, CPORT_READ,
                        ctl->tcp[i].packet.data, sizeof(tcp->packet.data) );
        } else {
            status=44;        /* client channel not found */
        }
        ctl->in_idle = 1;       /* force new read */
    } else if ( tcp->msg_type == SSH_MSG_CHANNEL_OPEN_FAILURE ) {
	/*
	 * Open failure sent to client thread, set flag to resume reading
	 * in_buffer.
	 */
	ctl->in_idle = 1;
	status = 1;
    } else {
	evr_event(EVR_EXCEPTION, 
		"Unexpected message (%d) sent to client thread",
		tcp->msg_type );
	status = 1;
    }
    return status;
}
/*****************************************************************************/
/* Setup new connection.  Allocate channel and validate connection is from
 * local host (plus any other checks possible).
 */
pf_chan allocate_channel ( pf_ctlblk ctl, pf_chan listener )
{
    int i, access_allowed;
    pf_chan tcp;
    void *tmp_ctx, *io_ctx;
    cport_stream_handler *driver;
    /*
     * Cast data portion of listener buffer to data structure used
     * to complete connection.
     */
    access_allowed = 1;
    if ( listener->forward_port == 0 && pf.decnet_x11 ) {
	/*
	 * Connection is from special X11 driver.
	 */
        struct cportx11_accept_result *result;

        result = (struct cportx11_accept_result *) listener->packet.data;
	io_ctx = result->io_ctx;
	driver = &cportx11_driver;
	evr_event ( EVR_DEBUG, 
	    "Allocating PF channel for X11 connect from DECnet node '%s' at %t",
	    result->remote_address, 0 );
    } else {
	/*
    	 * Do access control, connection must be from 127.0.0.1 (localhost).
    	 */
        struct cportucx_accept_result *result;

        result = (struct cportucx_accept_result *) listener->packet.data;
	io_ctx = result->io_ctx;

	driver = &cportucx_driver;
	evr_event ( EVR_DEBUG, 
	    "Allocating PF channel for connection %d.%d.%d.%d:%d at %t",
	    result->remote_address[0],result->remote_address[1],
	    result->remote_address[2],result->remote_address[3],
	    result->remote_port, 0 );

	if ( result->remote_address[0] != 127 ||
	     result->remote_address[1] != 0 || result->remote_address[2] != 0 ||
	    result->remote_address[3] != 1 ) access_allowed = 0;
    }
    /*
     * Search for free entry in table.  Ensure 1 past high mark is marked free.
     */
    tcp = ctl->tcp;
    if ( (ctl->high_chan+1) < param.x11_table_size ) 
		tcp[ctl->high_chan+1].type = 0;
    if ( access_allowed ) for ( i = 1; 
		i < param.x11_table_size; i++ ) if ( tcp[i].type == 0 ) {
	/*
	 * Found match, mark table entry allocated.
	 */
	if ( i > ctl->high_chan ) ctl->high_chan = i;
	tcp[i].type = 2;	/* flag as tcp connection */
	tcp[i].index = i;
	tcp[i].port_num = listener->port_num;
	tcp[i].forward_port = listener->forward_port;
	tcp[i].in = cport_assign_stream ( ctl->port, driver, io_ctx, 1 );
	if ( tcp[i].in ) tcp[i].in->user_val = i;
	tcp[i].out = cport_assign_stream ( ctl->port, driver, io_ctx, 0 );
	if ( tcp[i].out ) tcp[i].out->user_val = i;
        evr_event ( EVR_DEBUG, "Connection assigned channel %d", i );


	return &tcp[i];
    }
    /*
     * falling through loop means allocation faliure.  Assign temporary
     * stream and deassign to force connection close.
     */
    evr_event ( EVR_DEBUG, "No channel available for connection" );
    tmp_ctx = cport_assign_stream ( ctl->port, driver, io_ctx, 0 );
    if ( tmp_ctx ) cport_deassign ( tmp_ctx );
    return (pf_chan) 0;
}
/*******************************************************************************/
/* tcp dispatch_in is called when I/O on the in ISB completes.  What the
 * I/O is depends upon the channel type:
 *    accept for listen channel.
 *    read from TCP connection.
 */
int pf_dispatch_tcp_in ( pf_ctlblk ctl, pf_chan tcp)
{
	int status;
	pf_chan new_chan;
	unsigned short *iosb;

	iosb = (unsigned short *) tcp->in->iosb;
	if ( tcp->type == 1 ) {
	    if ( (iosb[0]&1) == 0 ) {
		evr_event ( EVR_EXCEPTION, "Error on listen channel: %d",
			iosb[0] );
		return iosb[0];
	    }
	    /*
	     * Listen channel has new connection.  Allocate TCP channel
	     * and assign stream.  Allocate function will close connection
	     * on failure.
	     */
	    new_chan = allocate_channel ( ctl, tcp );
	    if ( new_chan && (tcp->forward_port == 0) ) {
	        /*
	         * X11 request, queue open channel request to client.  Resume 
		 * processing in pf_dispatch_pipe_out.
	         */
	        new_chan->packet.channel = new_chan->index;
		new_chan->remote_channel = new_chan->index;
	        queue_to_client ( ctl, new_chan->index, SSH_SMSG_X11_OPEN, 0 );
	    } if ( new_chan && (tcp->forward_port > 0) ) {
	        /*
	         * Construct PORT_OPEN message amd queue to client.  Resume 
		 * processing in pf_dispatch_pipe_out.
		 * message is: [channel,] tcp-port, hostname (will get
		 * re-constructed by main thread).
	         */
		union { int l; char b[4]; } tmp;
		int i;
	        new_chan->packet.channel = new_chan->index;
		tmp.l = tcp->forward_port;
		new_chan->packet.data[0] = tmp.b[3];	/* reverse bytes */
		new_chan->packet.data[1] = tmp.b[2];
		new_chan->packet.data[2] = tmp.b[1];
		new_chan->packet.data[3] = tmp.b[0];
		tu_strnzcpy ( &new_chan->packet.data[4], tcp->forward_host,
			sizeof(new_chan->packet.data)-5 );
		tmp.l = tu_strlen ( tcp->forward_host );
		i = tu_strlen ( &new_chan->packet.data[4] ) + 4;

		new_chan->remote_channel = new_chan->index;
	        queue_to_client ( ctl, new_chan->index, SSH_MSG_PORT_OPEN, i );
	    }
	    /*
	     * Begin accepting next request.  
	     * Assume CPORTUCX_ACCEPT == CPORTX11_ACCEPT
	     */
	    status = cport_start_io ( tcp->in, CPORTUCX_ACCEPT,
		tcp->packet.data, sizeof(struct cportucx_accept_result) );

	} else if ( tcp->type == 2 ) {
	    /*
	     * Data received on local connection to port being forwarded,
	     * Determine if initial packet or followon.
	     */
	    if ( (iosb[0]&1) == 0 ) {

		evr_event ( EVR_EXCEPTION, 
			"Error on client channel[%d] read: %d %d %x %x",
			tcp->index, iosb[0], iosb[1], iosb[2], iosb[3] );
	        status = queue_to_client ( ctl, tcp->index, 
			SSH_MSG_CHANNEL_CLOSE, 0 );
		return status;
	    }
	    if ( tcp->init_packet_pos < tcp->init_packet_size ) {
		/*
		 * We are reading inital packet.
		 */
		char byte_order;
		tcp->init_packet_pos += iosb[1];
		byte_order = tcp->packet.data[0];
		if ( tcp->init_packet_pos >= tcp->init_packet_size ) {
		    /*
		     * determine if packet has authentication data.
		     */
		    unsigned char *bf;
		    int p_len, a_len, remaining;
		    bf = (unsigned char *) tcp->packet.data;
		    if ( byte_order == 'l' ) {
			p_len = (bf[7]<<8) | bf[6];
		        a_len = (bf[9]<<8) | bf[8];
		    } else if ( byte_order == 'B' ) {
			p_len = (bf[6]<<8) | bf[7];
		        a_len = (bf[8]<<8) | bf[9];
		    } else {
			/* bad packet */
			p_len = a_len = 0;
		    }
		    if ( (a_len + p_len) > 0 ) {
			/*
			 * Compute actual size of first packet.
			 */
			tcp->init_packet_size = 12 + ((p_len+3)&(~3)) +
				((a_len+3)&(~3));
			if (tcp->init_packet_size > sizeof(tcp->packet.data)) {
			    /* initial packet too large. */
			    tcp->init_packet_size = tcp->init_packet_pos;
			}
		    }
		    remaining = tcp->init_packet_size - tcp->init_packet_pos;
		    if ( remaining > 0 ) {
			/*
			 * Read remaining data.
			 */
			status = cport_start_io ( tcp->in, CPORT_READ,
			    &tcp->packet.data[tcp->init_packet_pos], a_len);
			return status;
		    }
		}
		if ( ctl->spoof_packet ) {
		    /*
		     * Substitute our own spoof packet, fixup byte order.
		     */
		    memcpy ( tcp->packet.data, ctl->spoof_packet,
			ctl->spoof_packet_size );
		    if ( byte_order != ctl->spoof_packet[0] ) {
			/* reverse bytes */
			tcp->packet.data[0] = byte_order;
			tcp->packet.data[6] = ctl->spoof_packet[7];
			tcp->packet.data[7] = ctl->spoof_packet[6];
			tcp->packet.data[8] = ctl->spoof_packet[9];
			tcp->packet.data[9] = ctl->spoof_packet[8];
		    }
		    iosb[1] = ctl->spoof_packet_size;
		}
	    }
	    /*
	     * Send received TCP data as channel_data message to client.
	     */
#ifdef DEBUG
printf("local channel: %d, remote channel: %d\n", tcp->index,
tcp->remote_channel );
#endif
	    status = queue_to_client ( ctl, tcp->index, 
		SSH_MSG_CHANNEL_DATA, iosb[1] );

	} else {
	    evr_event ( EVR_EXCEPTION, 
		"BUGCHECK, unexpected channel type in dispatch_tcp_in: %d",
		tcp->type );
	     status = 44;
	}
    return status;
}
/*******************************************************************************/
/* tcp dispatch_out is called when I/O on the out ISB completes.  Verify the
 * write succeeded and begin new read on the input pipe when finished.
 */
int pf_dispatch_tcp_out( pf_ctlblk ctl, pf_chan tcp)
{
    unsigned short *iosb;
    int status, remaining;

    if ( ctl->in_idle ) {
	evr_event(EVR_EXCEPTION, "BUGCHECK, tcp_out called when pipe in idle");
	return 0;
    }
    iosb = (unsigned short *) tcp->out->iosb;
    if ( (iosb[0]&1) == 0 ) {	/* error in I/O */
evr_event ( EVR_EXCEPTION, "Error on client channel write: %d",
			iosb[0] );
	ctl->in_idle = 1;
	status = 1;	/* iosb[0]; */
	return status;
    }
    ctl->in_pos += iosb[1];	/* update postion */
    remaining = ctl->in_endpos - ctl->in_pos;
    if ( remaining > 0 ) {
	/* Continue write, limit I/O size. */
	remaining = ctl->in_endpos - ctl->in_pos;
	if ( remaining > 4096 ) remaining = 4096;
	status = cport_start_io ( tcp->out, CPORT_WRITE,
		&ctl->in_buffer[ctl->in_pos], remaining );
    } else {
	/*
	 * Write completed, enable next read from ctl->pipe_in.
	 */
        ctl->in_idle = 1;
	status = 1;
    }
    return status;
}
